' This version of NucProp allows for an extended table of isotopes.
' Note: In this version of NucProp, the MAT numbers from JEFF are not preserved!
' The MAT numbers are the consecutive numbers of states in NUBASE!

' Data extracted from NUBASE 2016 (March 2018), KHS
' Entries with "xm", "xn" and other with "x" are included, (December 2018), KHS

Dim Shared As Integer N_MAT_MAX  ' Number of lines in table NuclideData
#DEFINE ISOSOURCE "NUBASE-2016"
'Print "Spins and energies for calculation of isomeric yields provided by "+ISOSOURCE+"."
'Print


Declare Function I_MAT_ENDF(IZ As Integer,IA As Integer) As Integer
Declare Function N_ISO_MAT(IMAT As Integer) As Integer
Declare Function R_AWR_ENDF(IZ As Integer,IA AS Integer) As Single
Declare Function ISO_for_MAT(IMAT As Integer) As Integer
Declare Function ISO_for_ZA(IZ As Integer,IA As Integer) As Integer
Declare Function NStates_for_ZA(IZ As Integer, IA As Integer) As Integer


Type NucProp
  I_Z As Integer    ' Atomic number
  I_A As Integer    ' Mass number
  I_ISO As Integer  ' Nmbr of isomeric state, 0 for ground state
  R_SPI As Single   ' Spin
  I_PAR As Integer  ' Parity
  R_AWR As Double   ' Atomic mass of nucleus in units of neutron mass
  R_EXC As Single   ' Excitation energy of isomeric state in MeV
  C_Lifetime As String  ' Life time of state
End Type


Dim Shared As Integer N_ISO_TOT  ' Total number of nuclides with isomers
N_ISO_TOT = 0

'Print "NucProp 2 **********"

Dim As Integer I_MAT_original
Dim As Single R_Dummy_NuclideData
Dim As String C_Dummy_NuclideData
Dim As Integer I_NuclideData

' Determine N_MAT_MAX (number of states in table NuclideData)
N_MAT_MAX = 0
Do 
  Read I_MAT_original
  If I_MAT_original = 9999 Then
    Exit Do
  End If
  N_MAT_MAX = N_MAT_MAX + 1
  For I_NuclideData = 1 To 7
    Read R_Dummy_NuclideData
  Next I_NuclideData
  Read C_Dummy_NUclideData
Loop 

Redim Shared NucTab(N_MAT_MAX) As NucProp
Redim Shared MAT_for_ISO_prov(N_MAT_MAX) As Integer

Scope
Dim As Integer I_Mat
Restore NuclideData
Do Until I_MAT = N_MAT_MAX 
  ' Read nuclide properties into UDT NucTab
  Read I_MAT_original
  I_MAT = I_MAT + 1
  If I_MAT > N_MAT_MAX Then Print "<E> NucPropx: N_MAT_MAX too small!"
  Read NucTab(I_MAT).I_Z, _
       NucTab(I_MAT).I_A, _
       NucTab(I_MAT).I_ISO, _
       NucTab(I_MAT).R_SPI, _
       NucTab(I_MAT).I_PAR, _
       NucTab(I_MAT).R_AWR, _
       NucTab(I_MAT).R_EXC, _
       NucTab(I_MAT).C_Lifetime 
  If NucTab(I_MAT).I_ISO < 0 or NucTab(I_MAT).I_ISO > 9 Then
    Print "<E> Error in NucProp"
    Print "GEF stopped."
    Sleep
  End If     
  If NucTab(I_MAT).R_SPI < 0 Or NucTab(I_MAT).R_SPI > 1.E3 Then _
          NucTab(I_MAT).R_SPI = 0     
  If NucTab(I_MAT).I_ISO = 1 Then  ' At least one isomeric state is in NucTab
    ' Establish table of MAT numbers of nuclides with isomers
    N_ISO_TOT = N_ISO_TOT + 1
    MAT_for_ISO_prov(N_ISO_TOT) = I_MAT - 1
  End If
'  PRINT I_MAT,NucTab(I_MAT).I_Z,NucTab(I_MAT).I_A     
'  If NucTab(I_MAT).I_Z = 111 Then Exit Do
Loop

If I_MAT < N_MAT_MAX Then 
  Print "<E> Nucprop: N_MAT_MAX too large, should be ";I_MAT
End If  
End Scope

Redim Shared MAT_for_ISO(N_ISO_TOT) As Integer  
   ' List of MAT numbers of nuclides with isomers
   ' MAT number refer to ground state !
   
'Scope  


   
Dim As Integer I1,J1
Dim As Single R1
For I1 = 1 To N_ISO_TOT
  MAT_for_ISO(I1) = MAT_for_ISO_prov(I1)
Next
' Erase MAT_for_ISO_prov



Type Isoprop
   I_MAT As Integer
   I_Z As Integer
   I_A As Integer
   N_STATES As Integer
   I_ISO(10) As Integer
   R_SPI(10) As Single
   I_PAR(10) As Integer
   R_EXC(10) As Single
   R_Lim(10) As Single
   R_Prob(10) As Single
   C_lifetime(10) As String
End Type


Redim Shared Isotab(N_ISO_TOT) As Isoprop



Scope

Dim Inmbr As Integer

For I1 = 1 To N_ISO_TOT
  Isotab(I1).I_MAT = MAT_for_ISO(I1)
  Isotab(I1).I_Z = NucTab(MAT_for_ISO(I1)).I_Z
  Isotab(I1).I_A = NucTab(MAT_for_ISO(I1)).I_A
  Isotab(I1).N_STATES = N_ISO_MAT(Isotab(I1).I_MAT) + 1  ' Number of states
  

  ' Sorting the spin in ascending order
  Inmbr = 0 
  For R1 = 0 To 50.0 Step 0.5
    For J1 = 1 To Isotab(I1).N_STATES
      If NucTab(MAT_for_ISO(I1)+J1-1).R_SPI = R1 Then
        Inmbr = Inmbr + 1
        Isotab(I1).I_ISO(Inmbr) = NucTab(MAT_for_ISO(I1)+J1-1).I_ISO
        Isotab(I1).R_SPI(Inmbr) = NucTab(MAT_for_ISO(I1)+J1-1).R_SPI
        Isotab(I1).I_PAR(Inmbr) = NucTab(MAT_for_ISO(I1)+J1-1).I_PAR
        Isotab(I1).R_EXC(Inmbr) = NucTab(MAT_for_ISO(I1)+J1-1).R_EXC 
        Isotab(I1).C_Lifetime(Inmbr) = NucTab(MAT_for_ISO(I1)+J1-1).C_Lifetime
  '     Print Inmbr,Mat_for_ISO(I1)+J1-1,NucTab(MAT_for_ISO(I1)+J1-1).R_SPI,NucTab(MAT_for_ISO(I1)+J1-1).R_EXC
      End If
    Next
  Next

  
  Dim As Single R_J_diff,R_E_diff,R_J_1
  ' Setting the limits of the angular-momentum distribution
  For J1 = 1 To Isotab(I1).N_STATES - 1
    R_J_diff = Isotab(I1).R_SPI(J1+1) - Isotab(I1).R_SPI(J1)
    R_E_diff = Isotab(I1).R_EXC(J1+1) - Isotab(I1).R_EXC(J1)
'   R_J_1 = R_J_diff * (0.5 + 0.5 * (R_E_diff / (Abs(R_E_diff) + 0.2 ) ) )
    R_J_1 = R_J_diff * (0.5 + 0.5 * (R_E_diff/R_J_diff / (Abs(R_E_diff/R_J_diff) + 0.05 ) ) )
    Isotab(I1).R_lim(J1) = Isotab(I1).R_SPI(J1) + R_J_1
    
    If Isotab(I1).R_SPI(J1+1) = Isotab(I1).R_SPI(J1) Then
      Isotab(I1).R_lim(J1) = 1.E3
    End If
  Next 
  Isotab(I1).R_lim(Isotab(I1).N_STATES) = 1.E3 
Next  

End Scope



Function I_MAT_ENDF(IZ As Integer,IA As Integer) As Integer
  ' Returns ENDF MAT number
  Dim As Integer I,IMAT
  Static As Integer I_message = 0
  For I = LBound(NucTab) To UBound(NucTab)
    IMAT = I
    If IZ = NucTab(IMAT).I_Z And IA = NucTab(IMAT).I_A Then Exit For
  Next
  If I > UBound(NucTab) Then
    If I_message = 0 Then
      Print "<W> NucProb.bas: Missing MAT number in NucTab for Z,A = ";IZ;",";IA
      Print "    Information on nuclear properties is not available."
      Print "    Please extend table!"
      Print "    (Note that this message is shown only once for the first case"
      Print "     encountered in one GEF session!)"
    End If  
    IMAT = 0
    I_message = 1
  End If
  I_MAT_ENDF = IMAT 
End Function

Function N_ISO_MAT(IMAT As Integer) As Integer
  ' Returns the number of isomers in a nuclide specified by MAT number
  ' (= number of states - 1 !)
  Dim As Integer I,I_first,I_last,IZ,IA
  I_first = IMAT
  IZ = NucTab(I_first).I_Z
  IA = NucTab(I_first).I_A
  I_last = I_first
  For I = I_first + 1 To N_MAT_MAX
    If NucTab(I).I_Z <> IZ Or NucTab(I).I_A <> IA Then Exit For
    I_last = I
  Next
  N_ISO_MAT = I_last - I_first
End Function

Function R_AWR_ENDF(IZ As Integer,IA AS Integer) As Single
  ' Returns ENDF AWR value
  Dim As Integer IMAT
  If IMAT > UBound(NucTab) Then IMAT = UBound(NucTab)
  IMAT = I_MAT_ENDF(IZ,IA)
  R_AWR_ENDF = NucTab(IMAT).R_AWR
End Function

Function ISO_for_MAT(IMAT As Integer) As Integer
  Dim As Integer I
  For I = 1 To N_ISO_TOT
    If IMAT = MAT_for_ISO(I) Then
      ISO_for_MAT = I
      Exit For
    End If
  Next
End Function

Function ISO_for_ZA(IZ As Integer,IA As Integer) As Integer
  ' Returns number of element in ISO array for nucleus Z,A
  Dim As Integer I
  Dim As Integer Ires = 0
  For I = 1 To N_ISO_TOT
    If Isotab(I).I_Z = IZ And Isotab(I).I_A = IA Then
      Ires = I
      Exit For
    End If    
  Next
  ISO_for_ZA = Ires
End Function

Function NStates_for_ZA(IZ As Integer, IA As Integer) As Integer
  ' Returns number of states
  Dim As Integer I
  Dim As Integer Ires = 1
  For I = 1 To N_ISO_TOT
    If Isotab(I).I_Z = IZ And Isotab(I).I_A = IA Then
      Ires = Isotab(I).N_STATES
      Exit For
    End If
  Next
  NStates_for_ZA = Ires
 End Function


NuclideData:
' Data extracted from NUBASE 2016.
' MAT no., Z, A, Isomer, Spin, Parity, AWR, EXC, t(1/2)
' In NucPropNUBASE, AWR is the mass excess!!!
' The MAT numbers are determined by counting. 
' They differ from the original MAT no. of JEFF 3.1.1!
' This is necessary, because the GEFY tables cover nuclides 
' beyond the range of MAT numbers of JEFF 3.1.1. 

' The following table can be modified just by changing the values for specific states or by
' adding new lines for additional states.
' No further adjustments of GEF are necessary.

Data _
 1, 0, 1, 0,0.5,             1,8071.317,   0            , "t=613.9s",_
 2, 1, 1, 0,0.5,             1,7288.970,   0            , "t=stbl",_
 3, 1, 2, 0,1,               1,13135.721,  0            , "t=stbl",_
 4, 1, 3, 0,0.5,             1,14949.809,  0            , "t=12.32y",_
 5, 2, 3, 0,0.5,             1,14931.217,  0            , "t=stbl",_
 6, 3, 3, 0,-77.777,         0,28670,      0            , "t=p-unst",_
 7, 1, 4, 0,2,              -1,24620,      0            , "t=139ys",_
 8, 2, 4, 0,0,               1,2424.915,   0            , "t=stbl",_
 9, 3, 4, 0,2,              -1,25320,      0            , "t=91ys",_
 10, 1, 5, 0,0.5,            1,32890,      0            , "t=&gt;910ys",_
 11, 2, 5, 0,1.5,           -1,11231,      0            , "t=700ys",_
 12, 3, 5, 0,1.5,           -1,11680,      0            , "t=370ys",_
 13, 4, 5, 0,0.5,            1,37140,      0            , "t=?",_
 14, 1, 6, 0,2,             -1,41880,      0            , "t=290ys",_
 15, 2, 6, 0,0,              1,17592.10,   0            , "t=806.92ms",_
 16, 3, 6, 0,1,              1,14086.878,  0            , "t=stbl",_
 17, 4, 6, 0,0,              1,18375,      0            , "t=5.0zs",_
 18, 5, 6, 0,2,             -1,47320,      0            , "t=p-unst#",_
 19, 1, 7, 0,0.5,            1,49140,      0            , "t=500#ys",_
 20, 2, 7, 0,1.5,           -1,26073,      0            , "t=2.51zs",_
 21, 3, 7, 0,1.5,           -1,14907.105,  0            , "t=stbl",_
 22, 4, 7, 0,1.5,           -1,15769.00,   0            , "t=53.22d",_
 23, 5, 7, 0,1.5,           -1,27677,      0            , "t=570ys",_
 24, 2, 8, 0,0,              1,31609.68,   0            , "t=119.1ms",_
 25, 3, 8, 0,2,              1,20945.80,   0            , "t=839.40ms",_
 26, 4, 8, 0,0,              1,4941.67,    0            , "t=81.9as",_
 27, 5, 8, 0,2,              1,22921.6,    0            , "t=770ms",_
 28, 6, 8, 0,0,              1,35064,      0            , "t=3.5zs",_
 29, 2, 9, 0,0.5,            1,40940,      0            , "t=2.5zs",_
 30, 3, 9, 0,1.5,           -1,24954.90,   0            , "t=178.3ms",_
 31, 4, 9, 0,1.5,           -1,11348.45,   0            , "t=stbl",_
 32, 5, 9, 0,1.5,           -1,12416.5,    0            , "t=800zs",_
 33, 6, 9, 0,1.5,           -1,28911.0,    0            , "t=126.5ms",_
 34, 2, 10, 0,0,             1,49200,      0            , "t=3.1zs",_
 35, 3, 10, 0,1,            -1,33053,      0            , "t=2.0zs",_
 36, 3, 10, 1,1,             1,33250,      0.2          , "t=3.7zs",_
 37, 3, 10, 2,2,             1,33530,      0.48         , "t=1.35zs",_
 38, 4, 10, 0,0,             1,12607.49,   0            , "t=1.51My",_
 39, 5, 10, 0,3,             1,12050.609,  0            , "t=stbl",_
 40, 6, 10, 0,0,             1,15698.67,   0            , "t=19.3009s",_
 41, 7, 10, 0,2,            -1,38800,      0            , "t=200ys",_
 42, 3, 11, 0,1.5,          -1,40728.3,    0            , "t=8.75ms",_
 43, 4, 11, 0,0.5,           1,20177.17,   0            , "t=13.76s",_
 44, 5, 11, 0,1.5,          -1,8667.707,   0            , "t=stbl",_
 45, 6, 11, 0,1.5,          -1,10649.40,   0            , "t=20.364m",_
 46, 7, 11, 0,0.5,           1,24300,      0            , "t=550ys",_
 47, 7, 11, 1,0.5,          -1,25040,      0.74         , "t=690ys",_
 48, 3, 12, 0,-77.777,       0,49010,      0            , "t=&lt;10ns",_
 49, 4, 12, 0,0,             1,25077.8,    0            , "t=21.50ms",_
 50, 4, 12, 1,0,             1,27328.8,    2.251        , "t=229ns",_
 51, 5, 12, 0,1,             1,13369.4,    0            , "t=20.20ms",_
 52, 6, 12, 0,0,             1,0.0,        0            , "t=stbl",_
 53, 7, 12, 0,1,             1,17338.1,    0            , "t=11.000ms",_
 54, 8, 12, 0,0,             1,31915,      0            , "t=&gt;6.3zs",_
 55, 3, 13, 0,1.5,          -1,56980,      0            , "t=3.3zs",_
 56, 4, 13, 0,0.5,          -1,33659,      0            , "t=1.0zs",_
 57, 5, 13, 0,1.5,          -1,16561.9,    0            , "t=17.33ms",_
 58, 6, 13, 0,0.5,          -1,3125.008,   0            , "t=stbl",_
 59, 7, 13, 0,0.5,          -1,5345.48,    0            , "t=9.965m",_
 60, 8, 13, 0,1.5,          -1,23115,      0            , "t=8.58ms",_
 61, 4, 14, 0,0,             1,39950,      0            , "t=4.35ms",_
 62, 5, 14, 0,2,            -1,23664,      0            , "t=12.5ms",_
 63, 6, 14, 0,0,             1,3019.893,   0            , "t=5.70ky",_
 64, 7, 14, 0,1,             1,2863.416,   0            , "t=stbl",_
 65, 8, 14, 0,0,             1,8007.781,   0            , "t=70.620s",_
 66, 9, 14, 0,2,            -1,31960,      0            , "t=500ys",_
 67, 4, 15, 0,2.5,           1,49830,      0            , "t=790ys",_
 68, 5, 15, 0,1.5,          -1,28958,      0            , "t=9.93ms",_
 69, 6, 15, 0,0.5,           1,9873.1,     0            , "t=2.449s",_
 70, 7, 15, 0,0.5,          -1,101.438,    0            , "t=stbl",_
 71, 8, 15, 0,0.5,          -1,2855.6,     0            , "t=122.24s",_
 72, 9, 15, 0,0.5,           1,16567,      0            , "t=1.1zs",_
 73, 10, 15, 0,1.5,         -1,40220,      0            , "t=770ys",_
 74, 4, 16, 0,0,             1,57450,      0            , "t=650ys",_
 75, 5, 16, 0,0,            -1,37113,      0            , "t=&gt;4.6zs",_
 76, 6, 16, 0,0,             1,13694,      0            , "t=747ms",_
 77, 7, 16, 0,2,            -1,5683.9,     0            , "t=7.13s",_
 78, 7, 16, 1,0,            -1,5804.3,     0.12042      , "t=5.25us",_
 79, 8, 16, 0,0,             1,-4737.001,  0            , "t=stbl",_
 80, 9, 16, 0,0,            -1,10680,      0            , "t=11zs",_
 81, 10, 16, 0,0,            1,23987,      0            , "t=&gt;5.7zs",_
 82, 5, 17, 0,1.5,          -1,43720,      0            , "t=5.08ms",_
 83, 6, 17, 0,1.5,           1,21032,      0            , "t=193ms",_
 84, 7, 17, 0,0.5,          -1,7870,       0            , "t=4.173s",_
 85, 8, 17, 0,2.5,           1,-808.763,   0            , "t=stbl",_
 86, 9, 17, 0,2.5,           1,1951.70,    0            , "t=64.370s",_
 87, 10, 17, 0,0.5,         -1,16500.4,    0            , "t=109.2ms",_
 88, 11, 17, 0,1.5,          1,35170,      0            , "t=?",_
 89, 5, 18, 0,2,            -1,51790,      0            , "t=&lt;26ns",_
 90, 6, 18, 0,0,             1,24920,      0            , "t=92ms",_
 91, 7, 18, 0,1,            -1,13113,      0            , "t=619.2ms",_
 92, 8, 18, 0,0,             1,-782.815,   0            , "t=stbl",_
 93, 9, 18, 0,1,             1,873.1,      0            , "t=109.739m",_
 94, 9, 18, 1,5,             1,1994.5,     1.12136      , "t=162ns",_
 95, 10, 18, 0,0,            1,5317.6,     0            , "t=1664.20ms",_
 96, 11, 18, 0,1,           -1,25040,      0            , "t=1.3zs",_
 97, 5, 19, 0,1.5,          -1,59770,      0            , "t=2.92ms",_
 98, 6, 19, 0,0.5,           1,32410,      0            , "t=46.2ms",_
 99, 7, 19, 0,0.5,          -1,15856,      0            , "t=336ms",_
 100, 8, 19, 0,2.5,          1,3332.9,     0            , "t=26.470s",_
 101, 9, 19, 0,0.5,          1,-1487.444,  0            , "t=stbl",_
 102, 10, 19, 0,0.5,         1,1752.05,    0            , "t=17.274s",_
 103, 11, 19, 0,2.5,         1,12929,      0            , "t=&gt;1as",_
 104, 12, 19, 0,0.5,        -1,31830,      0            , "t=5ps",_
 105, 5, 20, 0,-77.777,      0,68450,      0            , "t=?",_
 106, 6, 20, 0,0,            1,37500,      0            , "t=16ms",_
 107, 7, 20, 0,-77.777,      0,21770,      0            , "t=136ms",_
 108, 8, 20, 0,0,            1,3796.2,     0            , "t=13.51s",_
 109, 9, 20, 0,2,            1,-17.463,    0            , "t=11.163s",_
 110, 10, 20, 0,0,           1,-7041.930,  0            , "t=stbl",_
 111, 11, 20, 0,2,           1,6850.6,     0            , "t=447.9ms",_
 112, 12, 20, 0,0,           1,17477.7,    0            , "t=93ms",_
 113, 5, 21, 0,1.5,         -1,77330,      0            , "t=?",_
 114, 6, 21, 0,0.5,          1,45640,      0            , "t=?",_
 115, 7, 21, 0,0.5,         -1,25230,      0            , "t=84ms",_
 116, 8, 21, 0,2.5,          1,8062,       0            , "t=3.42s",_
 117, 9, 21, 0,2.5,          1,-47.6,      0            , "t=4.158s",_
 118, 10, 21, 0,1.5,         1,-5731.78,   0            , "t=stbl",_
 119, 11, 21, 0,1.5,         1,-2184.63,   0            , "t=22.422s",_
 120, 12, 21, 0,2.5,         1,10903.8,    0            , "t=118.6ms",_
 121, 13, 21, 0,2.5,         1,26990,      0            , "t=?",_
 122, 6, 22, 0,0,            1,53610,      0            , "t=6.2ms",_
 123, 7, 22, 0,0,           -1,31760,      0            , "t=23ms",_
 124, 8, 22, 0,0,            1,9280,       0            , "t=2.25s",_
 125, 9, 22, 0,4,            1,2793,       0            , "t=4.23s",_
 126, 10, 22, 0,0,           1,-8024.719,  0            , "t=stbl",_
 127, 11, 22, 0,3,           1,-5181.51,   0            , "t=2.6018y",_
 128, 11, 22, 1,1,           1,-4598.46,   0.58305      , "t=243ns",_
 129, 12, 22, 0,0,           1,-399.9,     0            , "t=3.8755s",_
 130, 13, 22, 0,4,           1,18200,      0            , "t=91.1ms",_
 131, 14, 22, 0,0,           1,33340,      0            , "t=29ms",_
 132, 6, 23, 0,1.5,          1,64170,      0            , "t=?",_
 133, 7, 23, 0,0.5,         -1,36720,      0            , "t=13.9ms",_
 134, 8, 23, 0,0.5,          1,14620,      0            , "t=97ms",_
 135, 9, 23, 0,2.5,          1,3290,       0            , "t=2.23s",_
 136, 10, 23, 0,2.5,         1,-5154.05,   0            , "t=37.140s",_
 137, 11, 23, 0,1.5,         1,-9529.852,  0            , "t=stbl",_
 138, 12, 23, 0,1.5,         1,-5473.51,   0            , "t=11.317s",_
 139, 13, 23, 0,2.5,         1,6748.1,     0            , "t=470ms",_
 140, 14, 23, 0,1.5,         1,23700,      0            , "t=42.3ms",_
 141, 7, 24, 0,-77.777,      0,46940,      0            , "t=?",_
 142, 8, 24, 0,0,            1,18500,      0            , "t=77.4ms",_
 143, 9, 24, 0,3,            1,7540,       0            , "t=384ms",_
 144, 10, 24, 0,0,           1,-5951.6,    0            , "t=3.38m",_
 145, 11, 24, 0,4,           1,-8417.901,  0            , "t=14.957h",_
 146, 11, 24, 1,1,           1,-7945.694,  0.4722074    , "t=20.18ms",_
 147, 12, 24, 0,0,           1,-13933.569,               0            , "t=stbl",_
 148, 13, 24, 0,4,           1,-48.86,     0            , "t=2.053s",_
 149, 13, 24, 1,1,           1,376.94,     0.4258       , "t=130ms",_
 150, 14, 24, 0,0,           1,10745,      0            , "t=140ms",_
 151, 15, 24, 0,1,           1,33320,      0            , "t=?",_
 152, 7, 25, 0,0.5,         -1,55980,      0            , "t=?",_
 153, 8, 25, 0,1.5,          1,27330,      0            , "t=5.18zs",_
 154, 9, 25, 0,2.5,          1,11330,      0            , "t=80ms",_
 155, 10, 25, 0,0.5,         1,-2036,      0            , "t=602ms",_
 156, 11, 25, 0,2.5,         1,-9357.8,    0            , "t=59.1s",_
 157, 12, 25, 0,2.5,         1,-13192.78,  0            , "t=stbl",_
 158, 13, 25, 0,2.5,         1,-8915.97,   0            , "t=7.183s",_
 159, 14, 25, 0,2.5,         1,3827,       0            , "t=220ms",_
 160, 15, 25, 0,0.5,         1,19740,      0            , "t=?",_
 161, 8, 26, 0,0,            1,34660,      0            , "t=4.2ps",_
 162, 9, 26, 0,1,            1,18650,      0            , "t=8.2ms",_
 163, 9, 26, 1,4,            1,19290,      0.6434       , "t=2.2ms",_
 164, 10, 26, 0,0,           1,481,        0            , "t=197ms",_
 165, 11, 26, 0,3,           1,-6861,      0            , "t=1071.28ms",_
 166, 11, 26, 1,1,           1,-6779,      0.0825       , "t=9us",_
 167, 12, 26, 0,0,           1,-16214.542,               0            , "t=stbl",_
 168, 13, 26, 0,5,           1,-12210.15,  0            , "t=717ky",_
 169, 13, 26, 1,0,           1,-11981.85,  0.228306     , "t=6346.0ms",_
 170, 14, 26, 0,0,           1,-7141.02,   0            , "t=2.2453s",_
 171, 15, 26, 0,3,           1,10970,      0            , "t=43.7ms",_
 172, 15, 26, 1,-77.777,     0,11130,      0.1644       , "t=120ns",_
 173, 16, 26, 0,0,           1,27080,      0            , "t=?",_
 174, 8, 27, 0,1.5,          1,44670,      0            , "t=?",_
 175, 9, 27, 0,2.5,          1,25450,      0            , "t=4.9ms",_
 176, 10, 27, 0,1.5,         1,7050,       0            , "t=31.5ms",_
 177, 11, 27, 0,2.5,         1,-5518,      0            , "t=301ms",_
 178, 12, 27, 0,0.5,         1,-14586.61,  0            , "t=9.435m",_
 179, 13, 27, 0,2.5,         1,-17196.86,  0            , "t=stbl",_
 180, 14, 27, 0,2.5,         1,-12384.50,  0            , "t=4.15s",_
 181, 15, 27, 0,0.5,         1,-722,       0            , "t=260ms",_
 182, 16, 27, 0,2.5,         1,17030,      0            , "t=15.5ms",_
 183, 8, 28, 0,0,            1,52080,      0            , "t=?",_
 184, 9, 28, 0,-77.777,      0,33740,      0            , "t=46zs",_
 185, 10, 28, 0,0,           1,11300,      0            , "t=20ms",_
 186, 11, 28, 0,1,           1,-988,       0            , "t=30.5ms",_
 187, 12, 28, 0,0,           1,-15018.8,   0            , "t=20.915h",_
 188, 13, 28, 0,3,           1,-16850.64,  0            , "t=2.245m",_
 189, 14, 28, 0,0,           1,-21492.794,               0            , "t=stbl",_
 190, 15, 28, 0,3,           1,-7147.7,    0            , "t=270.3ms",_
 191, 16, 28, 0,0,           1,4070,       0            , "t=125ms",_
 192, 17, 28, 0,1,           1,27520,      0            , "t=?",_
 193, 9, 29, 0,2.5,          1,40150,      0            , "t=2.5ms",_
 194, 10, 29, 0,1.5,        -1,18400,      0            , "t=14.7ms",_
 195, 11, 29, 0,1.5,         1,2680,       0            , "t=44.1ms",_
 196, 12, 29, 0,1.5,         1,-10603,     0            , "t=1.30s",_
 197, 13, 29, 0,2.5,         1,-18207.8,   0            , "t=6.56m",_
 198, 14, 29, 0,0.5,         1,-21895.078,               0            , "t=stbl",_
 199, 15, 29, 0,0.5,         1,-16952.8,   0            , "t=4.142s",_
 200, 16, 29, 0,2.5,         1,-3160,      0            , "t=188ms",_
 201, 17, 29, 0,0.5,         1,13160,      0            , "t=?",_
 202, 9, 30, 0,-77.777,      0,48110,      0            , "t=?",_
 203, 10, 30, 0,0,           1,23280,      0            , "t=7.22ms",_
 204, 11, 30, 0,2,           1,8475,       0            , "t=48.4ms",_
 205, 12, 30, 0,0,           1,-8884,      0            , "t=313ms",_
 206, 13, 30, 0,3,           1,-15864.8,   0            , "t=3.62s",_
 207, 14, 30, 0,0,           1,-24432.960,               0            , "t=stbl",_
 208, 15, 30, 0,1,           1,-20200.85,  0            , "t=2.498m",_
 209, 16, 30, 0,0,           1,-14059.25,  0            , "t=1.1759s",_
 210, 17, 30, 0,3,           1,4440,       0            , "t=?",_
 211, 18, 30, 0,0,           1,20930,      0            , "t=?",_
 212, 9, 31, 0,2.5,          1,56140,      0            , "t=1#ms",_
 213, 10, 31, 0,1.5,        -1,31180,      0            , "t=3.4ms",_
 214, 11, 31, 0,1.5,         1,12246,      0            , "t=17.35ms",_
 215, 12, 31, 0,0.5,         1,-3122,      0            , "t=236ms",_
 216, 13, 31, 0,2.5,         1,-14950.7,   0            , "t=644ms",_
 217, 14, 31, 0,1.5,         1,-22949.04,  0            , "t=157.36m",_
 218, 15, 31, 0,0.5,         1,-24440.541,               0            , "t=stbl",_
 219, 16, 31, 0,0.5,         1,-19042.52,  0            , "t=2.5534s",_
 220, 17, 31, 0,1.5,         1,-7035,      0            , "t=190ms",_
 221, 18, 31, 0,2.5,         1,11330,      0            , "t=15.1ms",_
 222, 10, 32, 0,0,           1,37000,      0            , "t=3.5ms",_
 223, 11, 32, 0,3,          -1,18640,      0            , "t=12.9ms",_
 224, 12, 32, 0,0,           1,-829,       0            , "t=86ms",_
 225, 13, 32, 0,1,           1,-11099,     0            , "t=33.0ms",_
 226, 13, 32, 1,4,           1,-10142,     0.9566       , "t=200ns",_
 227, 14, 32, 0,0,           1,-24077.69,  0            , "t=153y",_
 228, 15, 32, 0,1,           1,-24304.87,  0            , "t=14.268d",_
 229, 16, 32, 0,0,           1,-26015.533,               0            , "t=stbl",_
 230, 17, 32, 0,1,           1,-13334.7,   0            , "t=298ms",_
 231, 18, 32, 0,0,           1,-2200.4,    0            , "t=98ms",_
 232, 19, 32, 0,1,           1,21100,      0            , "t=?",_
 233, 19, 32, 1,4,           1,22050,      0.95         , "t=?",_
 234, 10, 33, 0,3.5,        -1,46000,      0            , "t=?",_
 235, 11, 33, 0,1.5,         1,23780,      0            , "t=8.2ms",_
 236, 12, 33, 0,1.5,        -1,4962.3,     0            , "t=90.5ms",_
 237, 13, 33, 0,2.5,         1,-8497,      0            , "t=41.7ms",_
 238, 14, 33, 0,1.5,         1,-20514.3,   0            , "t=6.18s",_
 239, 15, 33, 0,0.5,         1,-26337.3,   0            , "t=25.35d",_
 240, 16, 33, 0,1.5,         1,-26585.854,               0            , "t=stbl",_
 241, 17, 33, 0,1.5,         1,-21003.3,   0            , "t=2.5038s",_
 242, 18, 33, 0,0.5,         1,-9384.3,    0            , "t=173.0ms",_
 243, 19, 33, 0,1.5,         1,7040,       0            , "t=?",_
 244, 10, 34, 0,0,           1,52840,      0            , "t=1#ms",_
 245, 11, 34, 0,1,           1,31680,      0            , "t=5.5ms",_
 246, 12, 34, 0,0,           1,8323,       0            , "t=20ms",_
 247, 13, 34, 0,4,          -1,-3000,      0            , "t=56.3ms",_
 248, 13, 34, 1,1,           1,-2450,      0.55         , "t=26ms",_
 249, 14, 34, 0,0,           1,-19957,     0            , "t=2.77s",_
 250, 14, 34, 1,3,          -1,-15701,     4.2561       , "t=&lt;210ns",_
 251, 15, 34, 0,1,           1,-24548.7,   0            , "t=12.43s",_
 252, 16, 34, 0,0,           1,-29931.69,  0            , "t=stbl",_
 253, 17, 34, 0,0,           1,-24440.08,  0            , "t=1.5266s",_
 254, 17, 34, 1,3,           1,-24293.72,  0.14636      , "t=31.99m",_
 255, 18, 34, 0,0,           1,-18378.29,  0            , "t=843.8ms",_
 256, 19, 34, 0,1,           1,-1220,      0            , "t=?",_
 257, 20, 34, 0,0,           1,13850,      0            , "t=?",_
 258, 11, 35, 0,1.5,         1,38230,      0            , "t=1.5ms",_
 259, 12, 35, 0,3.5,        -1,15640,      0            , "t=70ms",_
 260, 13, 35, 0,2.5,         1,-224,       0            , "t=37.2ms",_
 261, 14, 35, 0,3.5,        -1,-14390,     0            , "t=780ms",_
 262, 15, 35, 0,0.5,         1,-24857.8,   0            , "t=47.3s",_
 263, 16, 35, 0,1.5,         1,-28846.21,  0            , "t=87.37d",_
 264, 17, 35, 0,1.5,         1,-29013.53,  0            , "t=stbl",_
 265, 18, 35, 0,1.5,         1,-23047.3,   0            , "t=1.7756s",_
 266, 19, 35, 0,1.5,         1,-11172.9,   0            , "t=178ms",_
 267, 20, 35, 0,0.5,         1,4790,       0            , "t=25.7ms",_
 268, 11, 36, 0,-77.777,     0,46300,      0            , "t=?",_
 269, 12, 36, 0,0,           1,20380,      0            , "t=3.9ms",_
 270, 13, 36, 0,-77.777,     0,5950,       0            , "t=90ms",_
 271, 14, 36, 0,0,           1,-12440,     0            , "t=450ms",_
 272, 15, 36, 0,4,          -1,-20251,     0            , "t=5.6s",_
 273, 16, 36, 0,0,           1,-30664.13,  0            , "t=stbl",_
 274, 17, 36, 0,2,           1,-29522.01,  0            , "t=301.3ky",_
 275, 18, 36, 0,0,           1,-30231.540,               0            , "t=stbl",_
 276, 19, 36, 0,2,           1,-17417.1,   0            , "t=341ms",_
 277, 20, 36, 0,0,           1,-6450,      0            , "t=101.2ms",_
 278, 21, 36, 0,-77.777,     0,15350,      0            , "t=?",_
 279, 11, 37, 0,1.5,         1,53530,      0            , "t=1#ms",_
 280, 12, 37, 0,1.5,        -1,28210,      0            , "t=8ms",_
 281, 13, 37, 0,2.5,         1,9810,       0            , "t=11.5ms",_
 282, 14, 37, 0,3.5,        -1,-6570,      0            , "t=90ms",_
 283, 15, 37, 0,0.5,         1,-19000,     0            , "t=2.31s",_
 284, 16, 37, 0,3.5,        -1,-26896.42,  0            , "t=5.05m",_
 285, 17, 37, 0,1.5,         1,-31761.54,  0            , "t=stbl",_
 286, 18, 37, 0,1.5,         1,-30947.66,  0            , "t=35.011d",_
 287, 19, 37, 0,1.5,         1,-24800.20,  0            , "t=1.2365s",_
 288, 20, 37, 0,1.5,         1,-13136.1,   0            , "t=181.1ms",_
 289, 21, 37, 0,3.5,        -1,3520,       0            , "t=?",_
 290, 12, 38, 0,0,           1,34070,      0            , "t=1#ms",_
 291, 13, 38, 0,-77.777,     0,16210,      0            , "t=9.0ms",_
 292, 14, 38, 0,0,           1,-4170,      0            , "t=90#ms",_
 293, 15, 38, 0,2,          -1,-14620,     0            , "t=640ms",_
 294, 16, 38, 0,0,           1,-26861,     0            , "t=170.3m",_
 295, 17, 38, 0,2,          -1,-29798.10,  0            , "t=37.24m",_
 296, 17, 38, 1,5,          -1,-29126.73,  0.671365     , "t=715ms",_
 297, 18, 38, 0,0,           1,-34714.82,  0            , "t=stbl",_
 298, 19, 38, 0,3,           1,-28800.75,  0            , "t=7.636m",_
 299, 19, 38, 1,0,           1,-28670.61,  0.13015      , "t=924.46ms",_
 300, 19, 38, 2,7,           1,-25342.61,  3.45814      , "t=21.95us",_
 301, 20, 38, 0,0,           1,-22058.50,  0            , "t=443.70ms",_
 302, 21, 38, 0,2,          -1,-4250,      0            , "t=?",_
 303, 21, 38, 1,5,          -1,-3580,      0.67         , "t=?",_
 304, 22, 38, 0,0,           1,10870,      0            , "t=?",_
 305, 12, 39, 0,3.5,        -1,42280,      0            , "t=?",_
 306, 13, 39, 0,2.5,         1,20650,      0            , "t=7.6ms",_
 307, 14, 39, 0,2.5,        -1,2320,       0            , "t=47.5ms",_
 308, 15, 39, 0,0.5,         1,-12770,     0            , "t=282ms",_
 309, 16, 39, 0,3.5,        -1,-23160,     0            , "t=11.5s",_
 310, 17, 39, 0,1.5,         1,-29800.2,   0            , "t=56.2m",_
 311, 18, 39, 0,3.5,        -1,-33242,     0            , "t=269y",_
 312, 19, 39, 0,1.5,         1,-33807.190,               0            , "t=stbl",_
 313, 20, 39, 0,1.5,         1,-27282.7,   0            , "t=860.3ms",_
 314, 21, 39, 0,3.5,        -1,-14173,     0            , "t=&lt;300ns",_
 315, 22, 39, 0,1.5,         1,2200,       0            , "t=28.5ms",_
 316, 12, 40, 0,0,           1,48350,      0            , "t=1#ms",_
 317, 13, 40, 0,-77.777,     0,27590,      0            , "t=10#ms",_
 318, 14, 40, 0,0,           1,5430,       0            , "t=33.0ms",_
 319, 15, 40, 0,2,          -1,-8110,      0            , "t=150ms",_
 320, 16, 40, 0,0,           1,-22838,     0            , "t=8.8s",_
 321, 17, 40, 0,2,          -1,-27560,     0            , "t=1.35m",_
 322, 18, 40, 0,0,           1,-35039.894,               0            , "t=stbl",_
 323, 19, 40, 0,4,          -1,-33535.49,  0            , "t=1.248Gy",_
 324, 19, 40, 1,0,           1,-31891.85,  1.643639     , "t=336ns",_
 325, 20, 40, 0,0,           1,-34846.384,               0            , "t=stbl",_
 326, 21, 40, 0,4,          -1,-20523.3,   0            , "t=182.3ms",_
 327, 22, 40, 0,0,           1,-8850,      0            , "t=52.4ms",_
 328, 23, 40, 0,2,          -1,12170,      0            , "t=?",_
 329, 13, 41, 0,2.5,         1,33420,      0            , "t=2#ms",_
 330, 14, 41, 0,3.5,        -1,12120,      0            , "t=20.0ms",_
 331, 15, 41, 0,0.5,         1,-4980,      0            , "t=101ms",_
 332, 16, 41, 0,3.5,        -1,-19009,     0            , "t=1.99s",_
 333, 17, 41, 0,0.5,         1,-27310,     0            , "t=38.4s",_
 334, 18, 41, 0,3.5,        -1,-33067.5,   0            , "t=109.61m",_
 335, 19, 41, 0,1.5,         1,-35559.543,               0            , "t=stbl",_
 336, 20, 41, 0,3.5,        -1,-35137.89,  0            , "t=99.4ky",_
 337, 21, 41, 0,3.5,        -1,-28642.41,  0            , "t=596.3ms",_
 338, 22, 41, 0,1.5,         1,-15698,     0            , "t=81.9ms",_
 339, 23, 41, 0,3.5,        -1,320,        0            , "t=?",_
 340, 13, 42, 0,-77.777,     0,40100,      0            , "t=1#ms",_
 341, 14, 42, 0,0,           1,16470,      0            , "t=12.5ms",_
 342, 15, 42, 0,-77.777,     0,1010,       0            , "t=48.5ms",_
 343, 16, 42, 0,0,           1,-17637.7,   0            , "t=1.016s",_
 344, 17, 42, 0,2,          -1,-24830,     0            , "t=6.8s",_
 345, 18, 42, 0,0,           1,-34423,     0            , "t=32.9y",_
 346, 19, 42, 0,2,          -1,-35022.03,  0            , "t=12.355h",_
 347, 20, 42, 0,0,           1,-38547.24,  0            , "t=stbl",_
 348, 21, 42, 0,0,           1,-32121.15,  0            , "t=680.79ms",_
 349, 21, 42, 1,7,           1,-31504.83,  0.61632      , "t=61.7s",_
 350, 22, 42, 0,0,           1,-25104.67,  0            , "t=208.65ms",_
 351, 23, 42, 0,2,          -1,-7620,      0            , "t=?",_
 352, 24, 42, 0,0,           1,6730,       0            , "t=13.3ms",_
 353, 13, 43, 0,2.5,         1,47020,      0            , "t=1#ms",_
 354, 14, 43, 0,1.5,        -1,23100,      0            , "t=15#ms",_
 355, 15, 43, 0,0.5,         1,4680,       0            , "t=35.8ms",_
 356, 16, 43, 0,1.5,        -1,-12195,     0            , "t=265ms",_
 357, 16, 43, 1,3.5,        -1,-11874,     0.3207       , "t=415.0ns",_
 358, 17, 43, 0,1.5,         1,-24160,     0            , "t=3.13s",_
 359, 18, 43, 0,2.5,        -1,-32010,     0            , "t=5.37m",_
 360, 19, 43, 0,1.5,         1,-36575.4,   0            , "t=22.3h",_
 361, 19, 43, 1,3.5,        -1,-35837.1,   0.7383       , "t=200ns",_
 362, 20, 43, 0,3.5,        -1,-38408.82,  0            , "t=stbl",_
 363, 21, 43, 0,3.5,        -1,-36188.1,   0            , "t=3.891h",_
 364, 21, 43, 1,1.5,         1,-36036.3,   0.15179      , "t=438us",_
 365, 21, 43, 2,9.5,        -1,-33064.4,   3.12373      , "t=472ns",_
 366, 22, 43, 0,3.5,        -1,-29321,     0            , "t=509ms",_
 367, 22, 43, 1,1.5,         1,-29008,     0.313        , "t=11.9us",_
 368, 22, 43, 2,9.5,        -1,-26255,     3.0664       , "t=556ns",_
 369, 23, 43, 0,3.5,        -1,-17920,     0            , "t=79.3ms",_
 370, 24, 43, 0,1.5,         1,-1970,      0            , "t=21.1ms",_
 371, 14, 44, 0,0,           1,28510,      0            , "t=10#ms",_
 372, 15, 44, 0,-77.777,     0,10450,      0            , "t=18.5ms",_
 373, 16, 44, 0,0,           1,-9204,      0            , "t=100ms",_
 374, 16, 44, 1,0,           1,-7839,      1.365        , "t=2.619us",_
 375, 17, 44, 0,2,          -1,-20380,     0            , "t=560ms",_
 376, 18, 44, 0,0,           1,-32673.3,   0            , "t=11.87m",_
 377, 19, 44, 0,2,          -1,-35781.5,   0            , "t=22.13m",_
 378, 20, 44, 0,0,           1,-41468.7,   0            , "t=stbl",_
 379, 21, 44, 0,2,           1,-37816.0,   0            , "t=4.0420h",_
 380, 21, 44, 1,1,          -1,-37748.1,   0.0678679    , "t=154.8ns",_
 381, 21, 44, 2,0,          -1,-37669.8,   0.1461914    , "t=51.0us",_
 382, 22, 44, 0,0,           1,-37548.6,   0            , "t=59.1y",_
 383, 23, 44, 0,2,           1,-24120,     0            , "t=111ms",_
 384, 23, 44, 1,6,           1,-23850,     0.27         , "t=150ms",_
 385, 23, 44, 2,0,          -1,-23970,     0.15         , "t=?",_
 386, 24, 44, 0,0,           1,-13360,     0            , "t=42.8ms",_
 387, 25, 44, 0,2,          -1,7030,       0            , "t=?",_
 388, 14, 45, 0,1.5,        -1,37490,      0            , "t=1#ms",_
 389, 15, 45, 0,0.5,         1,15600,      0            , "t=8#ms",_
 390, 16, 45, 0,1.5,        -1,-3990,      0            , "t=68ms",_
 391, 17, 45, 0,1.5,         1,-18260,     0            , "t=413ms",_
 392, 18, 45, 0,2.5,        -1,-29770.8,   0            , "t=21.48s",_
 393, 19, 45, 0,1.5,         1,-36615.6,   0            , "t=17.8m",_
 394, 20, 45, 0,3.5,        -1,-40812.2,   0            , "t=162.61d",_
 395, 21, 45, 0,3.5,        -1,-41071.9,   0            , "t=stbl",_
 396, 21, 45, 1,1.5,         1,-41059.5,   0.0124       , "t=318ms",_
 397, 22, 45, 0,3.5,        -1,-39009.8,   0            , "t=184.8m",_
 398, 22, 45, 1,1.5,        -1,-38973.3,   0.03653      , "t=3.0us",_
 399, 23, 45, 0,3.5,        -1,-31886.0,   0            , "t=547ms",_
 400, 23, 45, 1,1.5,        -1,-31829.2,   0.0568       , "t=512ns",_
 401, 24, 45, 0,3.5,        -1,-19510,     0            , "t=60.9ms",_
 402, 24, 45, 1,1.5,         0,-19400,     0.107        , "t=&gt;80us",_
 403, 25, 45, 0,3.5,        -1,-5250,      0            , "t=?",_
 404, 26, 45, 0,1.5,         1,13760,      0            , "t=2.2ms",_
 405, 15, 46, 0,-77.777,     0,22970,      0            , "t=4#ms",_
 406, 16, 46, 0,0,           1,340,        0            , "t=50ms",_
 407, 17, 46, 0,2,          -1,-13860,     0            , "t=232ms",_
 408, 18, 46, 0,0,           1,-29772.9,   0            , "t=8.4s",_
 409, 19, 46, 0,2,          -1,-35413.9,   0            , "t=105s",_
 410, 20, 46, 0,0,           1,-43139.4,   0            , "t=stbl",_
 411, 21, 46, 0,4,           1,-41761.2,   0            , "t=83.80d",_
 412, 21, 46, 1,6,           1,-41709.2,   0.052011     , "t=9.4us",_
 413, 21, 46, 2,1,          -1,-41618.7,   0.142528     , "t=18.75s",_
 414, 22, 46, 0,0,           1,-44127.80,  0            , "t=stbl",_
 415, 23, 46, 0,0,           1,-37075.35,  0            , "t=422.64ms",_
 416, 23, 46, 1,3,           1,-36273.89,  0.80146      , "t=1.02ms",_
 417, 24, 46, 0,0,           1,-29472,     0            , "t=224.3ms",_
 418, 25, 46, 0,4,           1,-12570,     0            , "t=36.2ms",_
 419, 25, 46, 1,1,          -1,-12420,     0.15         , "t=1#ms",_
 420, 26, 46, 0,0,           1,910,        0            , "t=13.0ms",_
 421, 15, 47, 0,0.5,         1,29710,      0            , "t=2#ms",_
 422, 16, 47, 0,1.5,        -1,7370,       0            , "t=20#ms",_
 423, 17, 47, 0,1.5,         1,-9780,      0            , "t=101ms",_
 424, 18, 47, 0,1.5,        -1,-25366.3,   0            , "t=1.23s",_
 425, 19, 47, 0,0.5,         1,-35712.0,   0            , "t=17.50s",_
 426, 20, 47, 0,3.5,        -1,-42344.4,   0            , "t=4.536d",_
 427, 21, 47, 0,3.5,        -1,-44336.6,   0            , "t=3.3492d",_
 428, 21, 47, 1,1.5,         1,-43569.8,   0.76683      , "t=272ns",_
 429, 22, 47, 0,2.5,        -1,-44937.36,  0            , "t=stbl",_
 430, 23, 47, 0,1.5,        -1,-42006.62,  0            , "t=32.6m",_
 431, 24, 47, 0,1.5,        -1,-34563,     0            , "t=500ms",_
 432, 25, 47, 0,2.5,        -1,-22570,     0            , "t=88.0ms",_
 433, 26, 47, 0,3.5,        -1,-6870,      0            , "t=21.9ms",_
 434, 26, 47, 1,1.5,         1,-6100,      0.77         , "t=?",_
 435, 27, 47, 0,3.5,        -1,10370,      0            , "t=?",_
 436, 16, 48, 0,0,           1,12760,      0            , "t=10#ms",_
 437, 17, 48, 0,-77.777,     0,-4280,      0            , "t=100#ms",_
 438, 18, 48, 0,0,           1,-22280,     0            , "t=415ms",_
 439, 19, 48, 0,1,          -1,-32284.5,   0            , "t=6.8s",_
 440, 20, 48, 0,0,           1,-44224.63,  0            , "t=45Ey",_
 441, 21, 48, 0,6,           1,-44504,     0            , "t=43.67h",_
 442, 22, 48, 0,0,           1,-48492.71,  0            , "t=stbl",_
 443, 23, 48, 0,4,           1,-44477.7,   0            , "t=15.9735d",_
 444, 24, 48, 0,0,           1,-42822,     0            , "t=21.56h",_
 445, 25, 48, 0,4,           1,-29296,     0            , "t=158.1ms",_
 446, 26, 48, 0,0,           1,-18000,     0            , "t=45.3ms",_
 447, 27, 48, 0,6,           1,1500,       0            , "t=?",_
 448, 28, 48, 0,0,           1,16790,      0            , "t=2.8ms",_
 449, 16, 49, 0,1.5,        -1,21090,      0            , "t=?",_
 450, 17, 49, 0,1.5,         1,940,        0            , "t=50#ms",_
 451, 18, 49, 0,1.5,        -1,-17190,     0            , "t=236ms",_
 452, 19, 49, 0,0.5,         1,-29611.5,   0            , "t=1.26s",_
 453, 20, 49, 0,1.5,        -1,-41299.77,  0            , "t=8.718m",_
 454, 21, 49, 0,3.5,        -1,-46561.3,   0            , "t=57.18m",_
 455, 22, 49, 0,3.5,        -1,-48563.79,  0            , "t=stbl",_
 456, 23, 49, 0,3.5,        -1,-47961.9,   0            , "t=330d",_
 457, 24, 49, 0,2.5,        -1,-45333.1,   0            , "t=42.3m",_
 458, 25, 49, 0,2.5,        -1,-37620.6,   0            , "t=382ms",_
 459, 26, 49, 0,3.5,        -1,-24751,     0            , "t=64.7ms",_
 460, 27, 49, 0,3.5,        -1,-9880,      0            , "t=?",_
 461, 28, 49, 0,3.5,        -1,8200,       0            , "t=7.5ms",_
 462, 17, 50, 0,-77.777,     0,7740,       0            , "t=20#ms",_
 463, 18, 50, 0,0,           1,-13330,     0            , "t=106ms",_
 464, 19, 50, 0,0,          -1,-25728,     0            , "t=472ms",_
 465, 19, 50, 1,2,          -1,-25557,     0.1714       , "t=125ns",_
 466, 20, 50, 0,0,           1,-39589.2,   0            , "t=13.9s",_
 467, 21, 50, 0,5,           1,-44547,     0            , "t=102.5s",_
 468, 21, 50, 1,2,           1,-44290,     0.256895     , "t=350ms",_
 469, 22, 50, 0,0,           1,-51431.66,  0            , "t=stbl",_
 470, 23, 50, 0,6,           1,-49224.0,   0            , "t=150Py",_
 471, 24, 50, 0,0,           1,-50262.1,   0            , "t=stbl",_
 472, 25, 50, 0,0,           1,-42627.6,   0            , "t=283.19ms",_
 473, 25, 50, 1,5,           1,-42402.3,   0.22531      , "t=1.75m",_
 474, 26, 50, 0,0,           1,-34476,     0            , "t=152.1ms",_
 475, 27, 50, 0,6,           1,-17630,     0            , "t=38.8ms",_
 476, 28, 50, 0,0,           1,-4120,      0            , "t=18.5ms",_
 477, 17, 51, 0,1.5,         1,14290,      0            , "t=2#ms",_
 478, 18, 51, 0,1.5,        -1,-6690,      0            , "t=60#ms",_
 479, 19, 51, 0,1.5,         1,-22516,     0            , "t=365ms",_
 480, 20, 51, 0,1.5,        -1,-36332.3,   0            , "t=10.0s",_
 481, 21, 51, 0,3.5,        -1,-43229,     0            , "t=12.4s",_
 482, 22, 51, 0,1.5,        -1,-49732.8,   0            , "t=5.76m",_
 483, 23, 51, 0,3.5,        -1,-52203.8,   0            , "t=stbl",_
 484, 24, 51, 0,3.5,        -1,-51451.4,   0            , "t=27.7010d",_
 485, 25, 51, 0,2.5,        -1,-48243.9,   0            , "t=46.2m",_
 486, 26, 51, 0,2.5,        -1,-40203,     0            , "t=305.4ms",_
 487, 27, 51, 0,3.5,        -1,-27340,     0            , "t=68.8ms",_
 488, 28, 51, 0,3.5,        -1,-11900,     0            , "t=23.8ms",_
 489, 18, 52, 0,0,           1,-1280,      0            , "t=10#ms",_
 490, 19, 52, 0,2,          -1,-17140,     0            , "t=110ms",_
 491, 20, 52, 0,0,           1,-34266.3,   0            , "t=4.6s",_
 492, 21, 52, 0,3,           1,-40440,     0            , "t=8.2s",_
 493, 22, 52, 0,0,           1,-49470,     0            , "t=1.7m",_
 494, 23, 52, 0,3,           1,-51443.8,   0            , "t=3.743m",_
 495, 24, 52, 0,0,           1,-55419.2,   0            , "t=stbl",_
 496, 25, 52, 0,6,           1,-50707.3,   0            , "t=5.591d",_
 497, 25, 52, 1,2,           1,-50329.6,   0.377749     , "t=21.1m",_
 498, 26, 52, 0,0,           1,-48330,     0            , "t=8.275h",_
 499, 26, 52, 1,12,          1,-41372,     6.958        , "t=45.9s",_
 500, 27, 52, 0,6,           1,-34361,     0            , "t=111.1ms",_
 501, 27, 52, 1,2,           1,-33974,     0.387        , "t=102ms",_
 502, 28, 52, 0,0,           1,-22330,     0            , "t=41.8ms",_
 503, 29, 52, 0,3,           1,-2280,      0            , "t=?",_
 504, 18, 53, 0,2.5,        -1,6790,       0            , "t=3#ms",_
 505, 19, 53, 0,1.5,         1,-12300,     0            , "t=30ms",_
 506, 20, 53, 0,1.5,        -1,-29390,     0            , "t=461ms",_
 507, 21, 53, 0,3.5,        -1,-38910,     0            , "t=2.4s",_
 508, 22, 53, 0,1.5,        -1,-46830,     0            , "t=32.7s",_
 509, 23, 53, 0,3.5,        -1,-51851,     0            , "t=1.543m",_
 510, 24, 53, 0,1.5,        -1,-55287.0,   0            , "t=stbl",_
 511, 25, 53, 0,3.5,        -1,-54690.1,   0            , "t=3.7My",_
 512, 26, 53, 0,3.5,        -1,-50947.5,   0            , "t=8.51m",_
 513, 26, 53, 1,9.5,        -1,-47907.1,   3.0404       , "t=2.54m",_
 514, 27, 53, 0,3.5,        -1,-42659.4,   0            , "t=242ms",_
 515, 27, 53, 1,9.5,        -1,-39485.2,   3.1742       , "t=247ms",_
 516, 28, 53, 0,3.5,        -1,-29631,     0            , "t=55.2ms",_
 517, 29, 53, 0,1.5,        -1,-13270,     0            , "t=?",_
 518, 19, 54, 0,2,          -1,-5000,      0            , "t=10ms",_
 519, 20, 54, 0,0,           1,-25160,     0            , "t=90ms",_
 520, 21, 54, 0,3,           1,-33890,     0            , "t=526ms",_
 521, 21, 54, 1,5,           1,-33780,     0.1105       , "t=2.77us",_
 522, 22, 54, 0,0,           1,-45620,     0            , "t=2.1s",_
 523, 23, 54, 0,3,           1,-49893,     0            , "t=49.8s",_
 524, 23, 54, 1,5,           1,-49785,     0.108        , "t=900ns",_
 525, 24, 54, 0,0,           1,-56934.8,   0            , "t=stbl",_
 526, 25, 54, 0,3,           1,-55557.6,   0            , "t=312.20d",_
 527, 26, 54, 0,0,           1,-56254.5,   0            , "t=stbl",_
 528, 26, 54, 1,10,          1,-49727.4,   6.5271       , "t=364ns",_
 529, 27, 54, 0,0,           1,-48010.0,   0            , "t=193.28ms",_
 530, 27, 54, 1,7,           1,-47812.4,   0.19757      , "t=1.48m",_
 531, 28, 54, 0,0,           1,-39278,     0            , "t=114.2ms",_
 532, 28, 54, 1,10,          1,-32821,     6.4574       , "t=152ns",_
 533, 29, 54, 0,3,           1,-21410,     0            , "t=?",_
 534, 30, 54, 0,0,           1,-6270,      0            , "t=1.8ms",_
 535, 19, 55, 0,1.5,         1,710,        0            , "t=3#ms",_
 536, 20, 55, 0,2.5,        -1,-18350,     0            , "t=22ms",_
 537, 21, 55, 0,3.5,        -1,-30160,     0            , "t=96ms",_
 538, 22, 55, 0,0.5,        -1,-41670,     0            , "t=1.3s",_
 539, 23, 55, 0,3.5,        -1,-49140,     0            , "t=6.54s",_
 540, 24, 55, 0,1.5,        -1,-55109.7,   0            , "t=3.497m",_
 541, 25, 55, 0,2.5,        -1,-57712.4,   0            , "t=stbl",_
 542, 26, 55, 0,1.5,        -1,-57481.3,   0            , "t=2.744y",_
 543, 27, 55, 0,3.5,        -1,-54029.9,   0            , "t=17.53h",_
 544, 28, 55, 0,3.5,        -1,-45335.8,   0            , "t=204.7ms",_
 545, 29, 55, 0,1.5,        -1,-31640,     0            , "t=57ms",_
 546, 30, 55, 0,2.5,        -1,-14570,     0            , "t=19.8ms",_
 547, 19, 56, 0,2,          -1,7930,       0            , "t=1#ms",_
 548, 20, 56, 0,0,           1,-13900,     0            , "t=11ms",_
 549, 21, 56, 0,1,           1,-24850,     0            , "t=26ms",_
 550, 21, 56, 1,6,           1,-24850,     0            , "t=75ms",_
 551, 21, 56, 2,4,           1,-24080,     0.7749       , "t=290ns",_
 552, 22, 56, 0,0,           1,-39320,     0            , "t=200ms",_
 553, 23, 56, 0,1,           1,-46150,     0            , "t=216ms",_
 554, 24, 56, 0,0,           1,-55285.0,   0            , "t=5.94m",_
 555, 25, 56, 0,3,           1,-56911.5,   0            , "t=2.5789h",_
 556, 26, 56, 0,0,           1,-60607.1,   0            , "t=stbl",_
 557, 27, 56, 0,4,           1,-56040.4,   0            , "t=77.236d",_
 558, 28, 56, 0,0,           1,-53907.5,   0            , "t=6.075d",_
 559, 29, 56, 0,4,           1,-38643,     0            , "t=93ms",_
 560, 30, 56, 0,0,           1,-25390,     0            , "t=32.9ms",_
 561, 31, 56, 0,3,           1,-3390,      0            , "t=?",_
 562, 20, 57, 0,2.5,        -1,-6870,      0            , "t=5#ms",_
 563, 21, 57, 0,3.5,        -1,-21000,     0            , "t=22ms",_
 564, 22, 57, 0,2.5,        -1,-33920,     0            , "t=95ms",_
 565, 23, 57, 0,3.5,        -1,-44410,     0            , "t=350ms",_
 566, 24, 57, 0,1.5,        -1,-52524.7,   0            , "t=21.1s",_
 567, 25, 57, 0,2.5,        -1,-57486.3,   0            , "t=85.4s",_
 568, 26, 57, 0,0.5,        -1,-60181.8,   0            , "t=stbl",_
 569, 27, 57, 0,3.5,        -1,-59345.6,   0            , "t=271.70d",_
 570, 28, 57, 0,1.5,        -1,-56083.8,   0            , "t=35.60h",_
 571, 29, 57, 0,1.5,        -1,-47308.9,   0            , "t=196.3ms",_
 572, 30, 57, 0,3.5,        -1,-32550,     0            , "t=38ms",_
 573, 31, 57, 0,0.5,        -1,-15010,     0            , "t=?",_
 574, 20, 58, 0,0,           1,-1920,      0            , "t=3#ms",_
 575, 21, 58, 0,3,           1,-14880,     0            , "t=12ms",_
 576, 22, 58, 0,0,           1,-31110,     0            , "t=55ms",_
 577, 23, 58, 0,1,           1,-40400,     0            , "t=191ms",_
 578, 24, 58, 0,0,           1,-51991.8,   0            , "t=7.0s",_
 579, 25, 58, 0,1,           1,-55827.6,   0            , "t=3.0s",_
 580, 25, 58, 1,4,           1,-55755.8,   0.07177      , "t=65.4s",_
 581, 26, 58, 0,0,           1,-62155.1,   0            , "t=stbl",_
 582, 27, 58, 0,2,           1,-59847.2,   0            , "t=70.86d",_
 583, 27, 58, 1,5,           1,-59822.3,   0.02495      , "t=9.10h",_
 584, 27, 58, 2,4,           1,-59794.1,   0.05315      , "t=10.5us",_
 585, 28, 58, 0,0,           1,-60228.7,   0            , "t=stbl",_
 586, 29, 58, 0,1,           1,-51667.7,   0            , "t=3.204s",_
 587, 30, 58, 0,0,           1,-42300,     0            , "t=86.7ms",_
 588, 31, 58, 0,2,           1,-23540,     0            , "t=?",_
 589, 31, 58, 1,5,           1,-23510,     0.03         , "t=?",_
 590, 32, 58, 0,0,           1,-7080,      0            , "t=?",_
 591, 21, 59, 0,3.5,        -1,-10300,     0            , "t=10#ms",_
 592, 22, 59, 0,2.5,        -1,-25510,     0            , "t=28.5ms",_
 593, 22, 59, 1,0.5,        -1,-25400,     0.109        , "t=590ns",_
 594, 23, 59, 0,2.5,        -1,-37830,     0            , "t=95ms",_
 595, 24, 59, 0,0.5,        -1,-48090,     0            , "t=1050ms",_
 596, 24, 59, 1,4.5,         1,-47590,     0.503        , "t=96us",_
 597, 25, 59, 0,2.5,        -1,-55525.3,   0            , "t=4.59s",_
 598, 26, 59, 0,1.5,        -1,-60664.8,   0            , "t=44.495d",_
 599, 27, 59, 0,3.5,        -1,-62229.7,   0            , "t=stbl",_
 600, 28, 59, 0,1.5,        -1,-61156.7,   0            , "t=81ky",_
 601, 29, 59, 0,1.5,        -1,-56358.3,   0            , "t=81.5s",_
 602, 30, 59, 0,1.5,        -1,-47215.6,   0            , "t=182.0ms",_
 603, 31, 59, 0,1.5,        -1,-33760,     0            , "t=?",_
 604, 32, 59, 0,3.5,        -1,-15870,     0            , "t=8#ms",_
 605, 21, 60, 0,3,           1,-4050,      0            , "t=3#ms",_
 606, 22, 60, 0,0,           1,-22330,     0            , "t=22.2ms",_
 607, 23, 60, 0,3,           1,-33240,     0            , "t=122ms",_
 608, 23, 60, 1,1,           1,-33240,     0            , "t=40ms",_
 609, 23, 60, 2,4,           1,-33040,     0.2037       , "t=230ns",_
 610, 24, 60, 0,0,           1,-46670,     0            , "t=490ms",_
 611, 25, 60, 0,1,           1,-52967.9,   0            , "t=280ms",_
 612, 25, 60, 1,4,           1,-52696.0,   0.2719       , "t=1.77s",_
 613, 26, 60, 0,0,           1,-61413,     0            , "t=2.62My",_
 614, 27, 60, 0,5,           1,-61650.3,   0            , "t=5.2712y",_
 615, 27, 60, 1,2,           1,-61591.7,   0.05859      , "t=10.467m",_
 616, 28, 60, 0,0,           1,-64473.1,   0            , "t=stbl",_
 617, 29, 60, 0,2,           1,-58345.1,   0            , "t=23.7m",_
 618, 30, 60, 0,0,           1,-54174.3,   0            , "t=2.38m",_
 619, 31, 60, 0,2,           1,-39590,     0            , "t=70ms",_
 620, 32, 60, 0,0,           1,-27090,     0            , "t=30#ms",_
 621, 33, 60, 0,5,           1,-5470,      0            , "t=?",_
 622, 33, 60, 1,2,           1,-5410,      0.06         , "t=?",_
 623, 21, 61, 0,3.5,        -1,930,        0            , "t=2#ms",_
 624, 22, 61, 0,0.5,        -1,-16350,     0            , "t=15ms",_
 625, 23, 61, 0,1.5,        -1,-30510,     0            , "t=48.2ms",_
 626, 24, 61, 0,2.5,        -1,-42480,     0            , "t=243ms",_
 627, 25, 61, 0,2.5,        -1,-51742.1,   0            , "t=709ms",_
 628, 26, 61, 0,1.5,        -1,-58920.5,   0            , "t=5.98m",_
 629, 26, 61, 1,4.5,         1,-58058.8,   0.86167      , "t=238ns",_
 630, 27, 61, 0,3.5,        -1,-62898.1,   0            , "t=1.649h",_
 631, 28, 61, 0,1.5,        -1,-64221.9,   0            , "t=stbl",_
 632, 29, 61, 0,1.5,        -1,-61984.1,   0            , "t=3.339h",_
 633, 30, 61, 0,1.5,        -1,-56349,     0            , "t=89.1s",_
 634, 31, 61, 0,1.5,        -1,-47130,     0            , "t=167ms",_
 635, 31, 61, 1,0.5,        -1,-47040,     0.09         , "t=?",_
 636, 32, 61, 0,1.5,        -1,-33360,     0            , "t=44ms",_
 637, 33, 61, 0,1.5,        -1,-16900,     0            , "t=?",_
 638, 22, 62, 0,0,           1,-12500,     0            , "t=10#ms",_
 639, 23, 62, 0,3,           1,-25480,     0            , "t=33.6ms",_
 640, 24, 62, 0,0,           1,-40890,     0            , "t=206ms",_
 641, 25, 62, 0,1,           1,-48524,     0            , "t=92ms",_
 642, 25, 62, 1,4,           1,-48181.0,   0.343        , "t=671ms",_
 643, 26, 62, 0,0,           1,-58878.0,   0            , "t=68s",_
 644, 27, 62, 0,2,           1,-61424,     0            , "t=1.54m",_
 645, 27, 62, 1,5,           1,-61402,     0.022        , "t=13.86m",_
 646, 28, 62, 0,0,           1,-66746.3,   0            , "t=stbl",_
 647, 29, 62, 0,1,           1,-62787.4,   0            , "t=9.67m",_
 648, 30, 62, 0,0,           1,-61168.0,   0            , "t=9.193h",_
 649, 31, 62, 0,0,           1,-51986.9,   0            , "t=116.121ms",_
 650, 32, 62, 0,0,           1,-41740,     0            , "t=129ms",_
 651, 33, 62, 0,1,           1,-24320,     0            , "t=?",_
 652, 22, 63, 0,0.5,        -1,-5750,      0            , "t=3#ms",_
 653, 23, 63, 0,1.5,        -1,-21890,     0            , "t=19.6ms",_
 654, 24, 63, 0,0.5,        -1,-36010,     0            , "t=129ms",_
 655, 25, 63, 0,2.5,        -1,-46887,     0            , "t=275ms",_
 656, 26, 63, 0,2.5,        -1,-55636,     0            , "t=6.1s",_
 657, 27, 63, 0,3.5,        -1,-61851,     0            , "t=26.9s",_
 658, 28, 63, 0,0.5,        -1,-65512.8,   0            , "t=101.2y",_
 659, 28, 63, 1,2.5,        -1,-65425.7,   0.08715      , "t=1.67us",_
 660, 29, 63, 0,1.5,        -1,-65579.8,   0            , "t=stbl",_
 661, 30, 63, 0,1.5,        -1,-62213.4,   0            , "t=38.47m",_
 662, 31, 63, 0,1.5,        -1,-56547.1,   0            , "t=32.4s",_
 663, 32, 63, 0,1.5,        -1,-46920,     0            , "t=142ms",_
 664, 33, 63, 0,1.5,        -1,-33500,     0            , "t=?",_
 665, 22, 64, 0,0,           1,-1030,      0            , "t=4#ms",_
 666, 23, 64, 0,1,           0,-16320,     0            , "t=15ms",_
 667, 23, 64, 1,-77.777,     0,-16240,     0.081        , "t=&lt;1us",_
 668, 24, 64, 0,0,           1,-33480,     0            , "t=43ms",_
 669, 25, 64, 0,1,           1,-42989,     0            , "t=88.8ms",_
 670, 25, 64, 1,4,           1,-42815,     0.1741       , "t=439us",_
 671, 26, 64, 0,0,           1,-54970,     0            , "t=2.0s",_
 672, 27, 64, 0,1,           1,-59792,     0            , "t=300ms",_
 673, 27, 64, 1,5,           1,-59686,     0.107        , "t=300#ms",_
 674, 28, 64, 0,0,           1,-67098.9,   0            , "t=stbl",_
 675, 29, 64, 0,1,           1,-65424.5,   0            , "t=12.7004h",_
 676, 30, 64, 0,0,           1,-66004.0,   0            , "t=stbl",_
 677, 31, 64, 0,0,           1,-58832.8,   0            , "t=2.627m",_
 678, 31, 64, 1,2,           1,-58790.0,   0.04285      , "t=21.9us",_
 679, 32, 64, 0,0,           1,-54315,     0            , "t=63.7s",_
 680, 33, 64, 0,0,           1,-39530,     0            , "t=40ms",_
 681, 34, 64, 0,0,           1,-26700,     0            , "t=30#ms",_
 682, 23, 65, 0,2.5,        -1,-11780,     0            , "t=10#ms",_
 683, 24, 65, 0,0.5,        -1,-28220,     0            , "t=27.5ms",_
 684, 25, 65, 0,2.5,        -1,-40967,     0            , "t=91.9ms",_
 685, 26, 65, 0,0.5,        -1,-51218,     0            , "t=810ms",_
 686, 26, 65, 1,4.5,         1,-50824,     0.3937       , "t=1.12s",_
 687, 26, 65, 2,2.5,         1,-50820,     0.3976       , "t=420ns",_
 688, 27, 65, 0,3.5,        -1,-59185.2,   0            , "t=1.16s",_
 689, 28, 65, 0,2.5,        -1,-65125.7,   0            , "t=2.5175h",_
 690, 28, 65, 1,0.5,        -1,-65062.3,   0.06337      , "t=69us",_
 691, 29, 65, 0,1.5,        -1,-67263.7,   0            , "t=stbl",_
 692, 30, 65, 0,2.5,        -1,-65912.0,   0            , "t=243.93d",_
 693, 30, 65, 1,0.5,        -1,-65858.1,   0.053928     , "t=1.6us",_
 694, 31, 65, 0,1.5,        -1,-62657.5,   0            , "t=15.2m",_
 695, 32, 65, 0,1.5,        -1,-56478.2,   0            , "t=30.9s",_
 696, 33, 65, 0,1.5,        -1,-46940,     0            , "t=170ms",_
 697, 34, 65, 0,1.5,        -1,-33020,     0            , "t=33ms",_
 698, 23, 66, 0,-77.777,     0,-5610,      0            , "t=5#ms",_
 699, 24, 66, 0,0,           1,-24720,     0            , "t=23.8ms",_
 700, 25, 66, 0,1,           1,-36750,     0            , "t=64.2ms",_
 701, 25, 66, 1,5,          -1,-36286,     0.4645       , "t=780us",_
 702, 26, 66, 0,0,           1,-50068,     0            , "t=351ms",_
 703, 27, 66, 0,1,           1,-56409,     0            , "t=194ms",_
 704, 27, 66, 1,3,           1,-56234,     0.1751       , "t=1.21us",_
 705, 27, 66, 2,8,          -1,-55767,     0.642        , "t=&gt;100us",_
 706, 28, 66, 0,0,           1,-66006.3,   0            , "t=54.6h",_
 707, 29, 66, 0,1,           1,-66258.3,   0            , "t=5.120m",_
 708, 29, 66, 1,6,          -1,-65104.1,   1.1542       , "t=600ns",_
 709, 30, 66, 0,0,           1,-68899.2,   0            , "t=stbl",_
 710, 31, 66, 0,0,           1,-63723.7,   0            , "t=9.304h",_
 711, 32, 66, 0,0,           1,-61607.0,   0            , "t=2.26h",_
 712, 33, 66, 0,0,           1,-52025,     0            , "t=95.77ms",_
 713, 33, 66, 1,5,           1,-50668,     1.35663      , "t=1.14us",_
 714, 33, 66, 2,9,           1,-49001,     3.0238       , "t=7.98us",_
 715, 34, 66, 0,0,           1,-41660,     0            , "t=33ms",_
 716, 23, 67, 0,2.5,        -1,-650,       0            , "t=2#ms",_
 717, 24, 67, 0,0.5,        -1,-18680,     0            , "t=10#ms",_
 718, 25, 67, 0,2.5,        -1,-33460,     0            , "t=46.7ms",_
 719, 26, 67, 0,0.5,        -1,-45610,     0            , "t=394ms",_
 720, 26, 67, 1,2.5,         1,-45210,     0.402        , "t=64us",_
 721, 26, 67, 2,4.5,         1,-45160,     0.45         , "t=75us",_
 722, 27, 67, 0,3.5,        -1,-55322,     0            , "t=329ms",_
 723, 27, 67, 1,0.5,        -1,-54830,     0.4916       , "t=496ms",_
 724, 28, 67, 0,0.5,        -1,-63742.7,   0            , "t=21s",_
 725, 28, 67, 1,4.5,         1,-62736.1,   1.0066       , "t=13.34us",_
 726, 29, 67, 0,1.5,        -1,-67319.5,   0            , "t=61.83h",_
 727, 30, 67, 0,2.5,        -1,-67880.3,   0            , "t=stbl",_
 728, 30, 67, 1,0.5,        -1,-67787.0,   0.093312     , "t=9.19us",_
 729, 30, 67, 2,4.5,         1,-67275.8,   0.60448      , "t=333ns",_
 730, 31, 67, 0,1.5,        -1,-66879.0,   0            , "t=3.2617d",_
 731, 32, 67, 0,0.5,        -1,-62658,     0            , "t=18.9m",_
 732, 32, 67, 1,2.5,        -1,-62640,     0.0182       , "t=13.7us",_
 733, 32, 67, 2,4.5,         1,-61906,     0.7517       , "t=109.1ns",_
 734, 33, 67, 0,2.5,        -1,-56587.2,   0            , "t=42.5s",_
 735, 34, 67, 0,2.5,        -1,-46580,     0            , "t=133ms",_
 736, 35, 67, 0,0.5,        -1,-32790,     0            , "t=?",_
 737, 24, 68, 0,0,           1,-14800,     0            , "t=5#ms",_
 738, 25, 68, 0,0,           0,-28380,     0            , "t=33.7ms",_
 739, 26, 68, 0,0,           1,-43490,     0            , "t=188ms",_
 740, 27, 68, 0,7,          -1,-51930,     0            , "t=200ms",_
 741, 27, 68, 1,1,           1,-51780,     0.15         , "t=1.6s",_
 742, 27, 68, 2,0,           0,-51740,     0.195        , "t=101ns",_
 743, 28, 68, 0,0,           1,-63463.8,   0            , "t=29s",_
 744, 28, 68, 1,0,           1,-61860,     1.60352      , "t=270ns",_
 745, 28, 68, 2,5,          -1,-60615,     2.8491       , "t=850us",_
 746, 29, 68, 0,1,           1,-65567.0,   0            , "t=30.9s",_
 747, 29, 68, 1,6,          -1,-64845.7,   0.72126      , "t=3.75m",_
 748, 30, 68, 0,0,           1,-70007.1,   0            , "t=stbl",_
 749, 31, 68, 0,1,           1,-67086.0,   0            , "t=67.845m",_
 750, 32, 68, 0,0,           1,-66978.8,   0            , "t=270.93d",_
 751, 33, 68, 0,3,           1,-58894.5,   0            , "t=151.6s",_
 752, 33, 68, 1,1,           1,-58469.4,   0.4251       , "t=111ns",_
 753, 34, 68, 0,0,           1,-54189.4,   0            , "t=35.5s",_
 754, 35, 68, 0,3,           1,-38790,     0            , "t=?",_
 755, 24, 69, 0,3.5,         1,-8580,      0            , "t=2#ms",_
 756, 25, 69, 0,2.5,        -1,-24770,     0            , "t=22.1ms",_
 757, 26, 69, 0,0.5,        -1,-39030,     0            , "t=108.2ms",_
 758, 27, 69, 0,3.5,        -1,-50280,     0            , "t=180ms",_
 759, 27, 69, 1,0.5,        -1,-49780,     0.5          , "t=750ms",_
 760, 28, 69, 0,4.5,         1,-59979,     0            , "t=11.4s",_
 761, 28, 69, 1,0.5,        -1,-59658,     0.321        , "t=3.5s",_
 762, 28, 69, 2,8.5,        -1,-57279,     2.7          , "t=439ns",_
 763, 29, 69, 0,1.5,        -1,-65736.2,   0            , "t=2.85m",_
 764, 29, 69, 1,6.5,         1,-62994.2,   2.742        , "t=357ns",_
 765, 30, 69, 0,0.5,        -1,-68417.8,   0            , "t=56.4m",_
 766, 30, 69, 1,4.5,         1,-67979.2,   0.438636     , "t=13.756h",_
 767, 31, 69, 0,1.5,        -1,-69327.8,   0            , "t=stbl",_
 768, 32, 69, 0,2.5,        -1,-67100.7,   0            , "t=39.05h",_
 769, 32, 69, 1,0.5,        -1,-67013.9,   0.08676      , "t=5.1us",_
 770, 32, 69, 2,4.5,         1,-66702.8,   0.39794      , "t=2.81us",_
 771, 33, 69, 0,2.5,        -1,-63110,     0            , "t=15.2m",_
 772, 34, 69, 0,0.5,        -1,-56434.7,   0            , "t=27.4s",_
 773, 34, 69, 1,2.5,        -1,-56395.9,   0.03885      , "t=2.0us",_
 774, 34, 69, 2,4.5,         1,-55860.7,   0.574        , "t=955ns",_
 775, 35, 69, 0,2.5,        -1,-46260,     0            , "t=&lt;24ns",_
 776, 35, 69, 1,2.5,        -1,-46220,     0.04         , "t=?",_
 777, 35, 69, 2,4.5,         1,-45690,     0.57         , "t=?",_
 778, 36, 69, 0,2.5,        -1,-32440,     0            , "t=28ms",_
 779, 24, 70, 0,0,           1,-4480,      0            , "t=1#ms",_
 780, 25, 70, 0,-77.777,     0,-19500,     0            , "t=19.9ms",_
 781, 26, 70, 0,0,           1,-36510,     0            , "t=63.0ms",_
 782, 27, 70, 0,6,          -1,-46630,     0            , "t=112ms",_
 783, 27, 70, 1,3,           1,-46430,     0.2          , "t=470ms",_
 784, 28, 70, 0,0,           1,-59213.9,   0            , "t=6.0s",_
 785, 28, 70, 1,8,           1,-56353.9,   2.86         , "t=232ns",_
 786, 29, 70, 0,6,          -1,-62976.4,   0            , "t=44.5s",_
 787, 29, 70, 1,3,          -1,-62875.3,   0.1011       , "t=33s",_
 788, 29, 70, 2,1,           1,-62733.8,   0.2426       , "t=6.6s",_
 789, 30, 70, 0,0,           1,-69564.7,   0            , "t=stbl",_
 790, 31, 70, 0,1,           1,-68910.1,   0            , "t=21.14m",_
 791, 32, 70, 0,0,           1,-70561.9,   0            , "t=stbl",_
 792, 33, 70, 0,4,           1,-64340,     0            , "t=52.6m",_
 793, 33, 70, 1,2,           1,-64310,     0.032008     , "t=96us",_
 794, 34, 70, 0,0,           1,-61929.9,   0            , "t=41.1m",_
 795, 35, 70, 0,0,           1,-51426,     0            , "t=79.1ms",_
 796, 35, 70, 1,9,           1,-49134,     2.2923       , "t=2.2s",_
 797, 36, 70, 0,0,           1,-41100,     0            , "t=52ms",_
 798, 25, 71, 0,2.5,        -1,-15570,     0            , "t=5#ms",_
 799, 26, 71, 0,3.5,         1,-31430,     0            , "t=33.7ms",_
 800, 27, 71, 0,3.5,        -1,-44370,     0            , "t=80ms",_
 801, 28, 71, 0,4.5,         1,-55406.2,   0            , "t=2.56s",_
 802, 28, 71, 1,0.5,        -1,-55406.0,   0.499        , "t=2.3s",_
 803, 29, 71, 0,1.5,        -1,-62711.1,   0            , "t=19.4s",_
 804, 29, 71, 1,9.5,        -1,-59955.4,   2.7557       , "t=271ns",_
 805, 30, 71, 0,0.5,        -1,-67328.8,   0            , "t=2.45m",_
 806, 30, 71, 1,4.5,         1,-67171.1,   0.1577       , "t=4.125h",_
 807, 31, 71, 0,1.5,        -1,-70139.1,   0            , "t=stbl",_
 808, 32, 71, 0,0.5,        -1,-69906.5,   0            , "t=11.43d",_
 809, 32, 71, 1,4.5,         1,-69708.1,   0.198354     , "t=20.41ms",_
 810, 33, 71, 0,2.5,        -1,-67893,     0            , "t=65.30h",_
 811, 34, 71, 0,2.5,        -1,-63146.5,   0            , "t=4.74m",_
 812, 34, 71, 1,0.5,        -1,-63097.7,   0.04879      , "t=5.6us",_
 813, 34, 71, 2,4.5,         1,-62886.0,   0.26048      , "t=19.0us",_
 814, 35, 71, 0,2.5,        -1,-56502,     0            , "t=21.4s",_
 815, 36, 71, 0,2.5,        -1,-46330,     0            , "t=100ms",_
 816, 37, 71, 0,2.5,        -1,-32060,     0            , "t=?",_
 817, 37, 71, 1,0.5,        -1,-32010,     0.05         , "t=?",_
 818, 37, 71, 2,4.5,         1,-31800,     0.26         , "t=?",_
 819, 25, 72, 0,-77.777,     0,-9900,      0            , "t=2#ms",_
 820, 26, 72, 0,0,           1,-28430,     0            , "t=19ms",_
 821, 27, 72, 0,6,          -1,-40200,     0            , "t=52.5ms",_
 822, 27, 72, 1,0,           1,-40000,     0.2          , "t=47.8ms",_
 823, 28, 72, 0,0,           1,-54226.1,   0            , "t=1.57s",_
 824, 29, 72, 0,2,          -1,-59783.0,   0            , "t=6.63s",_
 825, 29, 72, 1,6,          -1,-59512.7,   0.2703       , "t=1.76us",_
 826, 30, 72, 0,0,           1,-68145.5,   0            , "t=46.5h",_
 827, 31, 72, 0,3,          -1,-68588.3,   0            , "t=14.025h",_
 828, 31, 72, 1,0,           1,-68468.6,   0.11966      , "t=39.68ms",_
 829, 32, 72, 0,0,           1,-72585.90,  0            , "t=stbl",_
 830, 32, 72, 1,0,           1,-71894.47,  0.69143      , "t=444.2ns",_
 831, 33, 72, 0,2,          -1,-68230,     0            , "t=26.0h",_
 832, 34, 72, 0,0,           1,-67868.2,   0            , "t=8.40d",_
 833, 35, 72, 0,1,           1,-59061.7,   0            , "t=78.6s",_
 834, 35, 72, 1,3,          -1,-58960.9,   0.10076      , "t=10.6s",_
 835, 36, 72, 0,0,           1,-53941,     0            , "t=17.16s",_
 836, 37, 72, 0,1,           1,-38330,     0            , "t=?",_
 837, 37, 72, 1,3,          -1,-38230,     0.1          , "t=?",_
 838, 26, 73, 0,3.5,         1,-22900,     0            , "t=12.9ms",_
 839, 27, 73, 0,3.5,        -1,-37420,     0            , "t=40.7ms",_
 840, 28, 73, 0,4.5,         1,-50108.2,   0            , "t=840ms",_
 841, 29, 73, 0,1.5,        -1,-58987.4,   0            , "t=4.2s",_
 842, 30, 73, 0,0.5,        -1,-65593.4,   0            , "t=23.5s",_
 843, 30, 73, 1,2.5,         1,-65397.9,   0.1955       , "t=13.0ms",_
 844, 30, 73, 2,4.5,         1,-65355.8,   0.2376       , "t=5.8s",_
 845, 31, 73, 0,0.5,        -1,-69699.3,   0            , "t=4.86h",_
 846, 32, 73, 0,4.5,         1,-71297.52,  0            , "t=stbl",_
 847, 32, 73, 1,2.5,         1,-71284.24,  0.0132845    , "t=2.92us",_
 848, 32, 73, 2,0.5,        -1,-71230.79,  0.066726     , "t=499ms",_
 849, 33, 73, 0,1.5,        -1,-70953,     0            , "t=80.30d",_
 850, 33, 73, 1,4.5,         1,-70525,     0.427906     , "t=5.7us",_
 851, 34, 73, 0,4.5,         1,-68227,     0            , "t=7.15h",_
 852, 34, 73, 1,1.5,        -1,-68201,     0.02571      , "t=39.8m",_
 853, 35, 73, 0,0.5,        -1,-63647,     0            , "t=3.4m",_
 854, 36, 73, 0,1.5,        -1,-56552,     0            , "t=27.3s",_
 855, 36, 73, 1,4.5,         1,-56118,     0.43366      , "t=107ns",_
 856, 37, 73, 0,1.5,        -1,-46080,     0            , "t=?",_
 857, 37, 73, 1,4.5,         1,-45650,     0.43         , "t=?",_
 858, 38, 73, 0,0.5,        -1,-31950,     0            , "t=&gt;25ms",_
 859, 26, 74, 0,0,           1,-19590,     0            , "t=2#ms",_
 860, 27, 74, 0,-77.777,     0,-32820,     0            , "t=31.3ms",_
 861, 28, 74, 0,0,           1,-48460,     0            , "t=507.7ms",_
 862, 29, 74, 0,2,          -1,-56006,     0            , "t=1.63s",_
 863, 30, 74, 0,0,           1,-65756.7,   0            , "t=95.6s",_
 864, 31, 74, 0,3,          -1,-68049.6,   0            , "t=8.12m",_
 865, 31, 74, 1,0,           1,-67990,     0.059571     , "t=9.5s",_
 866, 32, 74, 0,0,           1,-73422.442,               0            , "t=stbl",_
 867, 33, 74, 0,2,          -1,-70860.1,   0            , "t=17.77d",_
 868, 34, 74, 0,0,           1,-72213.201,               0            , "t=stbl",_
 869, 35, 74, 0,0,          -1,-65288,     0            , "t=25.4m",_
 870, 35, 74, 1,4,           1,-65274,     0.01358      , "t=46m",_
 871, 36, 74, 0,0,           1,-62331.8,   0            , "t=11.50m",_
 872, 37, 74, 0,0,           1,-51916,     0            , "t=64.776ms",_
 873, 38, 74, 0,0,           1,-40830,     0            , "t=27ms",_
 874, 26, 75, 0,4.5,         1,-13640,     0            , "t=2#ms",_
 875, 27, 75, 0,3.5,        -1,-29650,     0            , "t=26.5ms",_
 876, 28, 75, 0,3.5,         1,-44030,     0            , "t=331.6ms",_
 877, 29, 75, 0,2.5,        -1,-54471.3,   0            , "t=1.224s",_
 878, 29, 75, 1,0.5,        -1,-54409.6,   0.0617       , "t=310ns",_
 879, 29, 75, 2,1.5,        -1,-54405.1,   0.0662       , "t=149ns",_
 880, 30, 75, 0,3.5,         1,-62558.9,   0            , "t=10.2s",_
 881, 30, 75, 1,0.5,        -1,-62432.0,   0.12694      , "t=5#s",_
 882, 31, 75, 0,1.5,        -1,-68464.6,   0            , "t=126s",_
 883, 32, 75, 0,0.5,        -1,-71856.96,  0            , "t=82.78m",_
 884, 32, 75, 1,3.5,         1,-71717.27,  0.13969      , "t=47.7s",_
 885, 32, 75, 2,2.5,         1,-71664.77,  0.19219      , "t=216ns",_
 886, 33, 75, 0,1.5,        -1,-73034.2,   0            , "t=stbl",_
 887, 33, 75, 1,4.5,         1,-72730.3,   0.3039243    , "t=17.62ms",_
 888, 34, 75, 0,2.5,         1,-72169.48,  0            , "t=119.78d",_
 889, 35, 75, 0,1.5,        -1,-69107,     0            , "t=96.7m",_
 890, 36, 75, 0,2.5,         1,-64324,     0            , "t=4.60m",_
 891, 37, 75, 0,1.5,        -1,-57218.7,   0            , "t=19.0s",_
 892, 38, 75, 0,1.5,        -1,-46620,     0            , "t=88ms",_
 893, 39, 75, 0,2.5,         1,-31820,     0            , "t=100#us",_
 894, 27, 76, 0,8,          -1,-24510,     0            , "t=23ms",_
 895, 27, 76, 1,1,          -1,-24410,     0.1          , "t=16ms",_
 896, 27, 76, 2,3,           1,-23770,     0.74         , "t=2.99us",_
 897, 28, 76, 0,0,           1,-41630,     0            , "t=234.6ms",_
 898, 28, 76, 1,8,           1,-39210,     2.4187       , "t=547.8ns",_
 899, 29, 76, 0,3,           0,-50976,     0            , "t=637.7ms",_
 900, 29, 76, 1,1,           0,-50980,     0            , "t=1.27s",_
 901, 30, 76, 0,0,           1,-62303.0,   0            , "t=5.7s",_
 902, 31, 76, 0,2,          -1,-66296.6,   0            , "t=32.6s",_
 903, 32, 76, 0,0,           1,-73212.889,               0            , "t=1.66Zy",_
 904, 33, 76, 0,2,          -1,-72291.4,   0            , "t=1.0778d",_
 905, 33, 76, 1,1,           1,-72247.0,   0.044425     , "t=1.84us",_
 906, 34, 76, 0,0,           1,-75251.950,               0            , "t=stbl",_
 907, 35, 76, 0,1,          -1,-70289,     0            , "t=16.2h",_
 908, 35, 76, 1,4,           1,-70186,     0.10258      , "t=1.31s",_
 909, 36, 76, 0,0,           1,-69014,     0            , "t=14.8h",_
 910, 37, 76, 0,1,          -1,-60479.1,   0            , "t=36.5s",_
 911, 37, 76, 1,4,           1,-60162.2,   0.31693      , "t=3.050us",_
 912, 38, 76, 0,0,           1,-54250,     0            , "t=7.89s",_
 913, 39, 76, 0,1,          -1,-38480,     0            , "t=120#us",_
 914, 27, 77, 0,3.5,        -1,-21020,     0            , "t=15ms",_
 915, 28, 77, 0,4.5,         1,-36800,     0            , "t=158.9ms",_
 916, 29, 77, 0,2.5,        -1,-48620,     0            , "t=467.9ms",_
 917, 30, 77, 0,3.5,         1,-58789.2,   0            , "t=2.08s",_
 918, 30, 77, 1,0.5,        -1,-58016.8,   0.77244      , "t=1.05s",_
 919, 31, 77, 0,1.5,        -1,-65992.3,   0            , "t=13.2s",_
 920, 32, 77, 0,3.5,         1,-71212.86,  0            , "t=11.211h",_
 921, 32, 77, 1,0.5,        -1,-71053.15,  0.15971      , "t=53.7s",_
 922, 33, 77, 0,1.5,        -1,-73916.3,   0            , "t=38.79h",_
 923, 33, 77, 1,4.5,         1,-73440.8,   0.47548      , "t=114.0us",_
 924, 34, 77, 0,0.5,        -1,-74599.49,  0            , "t=stbl",_
 925, 34, 77, 1,3.5,         1,-74437.57,  0.1619223    , "t=17.36s",_
 926, 35, 77, 0,1.5,        -1,-73234.8,   0            , "t=57.04h",_
 927, 35, 77, 1,4.5,         1,-73128.9,   0.10586      , "t=4.28m",_
 928, 36, 77, 0,2.5,         1,-70169.4,   0            , "t=74.4m",_
 929, 36, 77, 1,1.5,        -1,-70102.9,   0.0665       , "t=118ns",_
 930, 37, 77, 0,1.5,        -1,-64830.5,   0            , "t=3.78m",_
 931, 38, 77, 0,2.5,         1,-57803,     0            , "t=9.0s",_
 932, 39, 77, 0,2.5,         1,-46440,     0            , "t=63ms",_
 933, 40, 77, 0,1.5,        -1,-32040,     0            , "t=100#us",_
 934, 28, 78, 0,0,           1,-33890,     0            , "t=122.2ms",_
 935, 29, 78, 0,5,          -1,-44500,     0            , "t=330.7ms",_
 936, 30, 78, 0,0,           1,-57483.2,   0            , "t=1.47s",_
 937, 30, 78, 1,8,           1,-54807.9,   2.6753       , "t=320ns",_
 938, 31, 78, 0,2,          -1,-63706.0,   0            , "t=5.09s",_
 939, 31, 78, 1,-77.777,     0,-63207.1,   0.4989       , "t=110ns",_
 940, 32, 78, 0,0,           1,-71862,     0            , "t=88.0m",_
 941, 33, 78, 0,2,          -1,-72817,     0            , "t=90.7m",_
 942, 34, 78, 0,0,           1,-77025.94,  0            , "t=stbl",_
 943, 35, 78, 0,1,           1,-73452,     0            , "t=6.45m",_
 944, 35, 78, 1,4,           1,-73271,     0.18089      , "t=119.4us",_
 945, 36, 78, 0,0,           1,-74178.3,   0            , "t=stbl",_
 946, 37, 78, 0,0,           1,-66935,     0            , "t=17.66m",_
 947, 37, 78, 1,1,          -1,-66888,     0.04684      , "t=910ns",_
 948, 37, 78, 2,4,          -1,-66824,     0.11119      , "t=5.74m",_
 949, 37, 78, 0,0,           0,-66861,     0.074        , "t=R=2.0~0.5",_
 950, 38, 78, 0,0,           1,-63174,     0            , "t=156.1s",_
 951, 39, 78, 0,0,           1,-52170,     0            , "t=54ms",_
 952, 39, 78, 1,5,           1,-52170,     0            , "t=5.8s",_
 953, 40, 78, 0,0,           1,-40850,     0            , "t=50#ms",_
 954, 28, 79, 0,2.5,         1,-27570,     0            , "t=44ms",_
 955, 29, 79, 0,2.5,        -1,-41740,     0            , "t=241.0ms",_
 956, 30, 79, 0,4.5,         1,-53432.3,   0            , "t=746ms",_
 957, 30, 79, 1,0.5,         1,-52330,     1.1          , "t=&gt;200ms",_
 958, 31, 79, 0,1.5,        -1,-62547.7,   0            , "t=2.848s",_
 959, 32, 79, 0,0.5,        -1,-69530,     0            , "t=18.98s",_
 960, 32, 79, 1,3.5,         1,-69340,     0.18595      , "t=39.0s",_
 961, 33, 79, 0,1.5,        -1,-73636,     0            , "t=9.01m",_
 962, 33, 79, 1,4.5,         1,-72863,     0.77281      , "t=1.21us",_
 963, 34, 79, 0,3.5,         1,-75917.46,  0            , "t=327ky",_
 964, 34, 79, 1,0.5,        -1,-75821.69,  0.09577      , "t=3.92m",_
 965, 35, 79, 0,1.5,        -1,-76068.0,   0            , "t=stbl",_
 966, 35, 79, 1,4.5,         1,-75860.4,   0.20761      , "t=4.85s",_
 967, 36, 79, 0,0.5,        -1,-74442,     0            , "t=35.04h",_
 968, 36, 79, 1,3.5,         1,-74312,     0.12977      , "t=50s",_
 969, 37, 79, 0,2.5,         1,-70803.0,   0            , "t=22.9m",_
 970, 38, 79, 0,1.5,        -1,-65477,     0            , "t=2.25m",_
 971, 39, 79, 0,2.5,         1,-57820,     0            , "t=14.8s",_
 972, 40, 79, 0,2.5,         1,-46770,     0            , "t=56ms",_
 973, 41, 79, 0,4.5,         1,-31650,     0            , "t=?",_
 974, 28, 80, 0,0,           1,-22630,     0            , "t=30ms",_
 975, 29, 80, 0,-77.777,     0,-36200,     0            , "t=113.3ms",_
 976, 30, 80, 0,0,           1,-51648.6,   0            , "t=562.2ms",_
 977, 31, 80, 0,6,          -1,-59223.7,   0            , "t=1.9s",_
 978, 31, 80, 1,3,          -1,-59201.3,   0.02245      , "t=1.3s",_
 979, 32, 80, 0,0,           1,-69535.3,   0            , "t=29.5s",_
 980, 33, 80, 0,1,           1,-72214,     0            , "t=15.2s",_
 981, 34, 80, 0,0,           1,-77759.5,   0            , "t=stbl",_
 982, 35, 80, 0,1,           1,-75889.0,   0            , "t=17.68m",_
 983, 35, 80, 1,5,          -1,-75803.2,   0.085843     , "t=4.4205h",_
 984, 36, 80, 0,0,           1,-77893.3,   0            , "t=stbl",_
 985, 37, 80, 0,1,           1,-72175.5,   0            , "t=33.4s",_
 986, 37, 80, 1,6,           1,-71681.6,   0.4939       , "t=1.63us",_
 987, 38, 80, 0,0,           1,-70311,     0            , "t=106.3m",_
 988, 39, 80, 0,4,          -1,-61148,     0            , "t=30.1s",_
 989, 39, 80, 1,1,          -1,-60920,     0.2285       , "t=4.8s",_
 990, 39, 80, 2,2,           1,-60835,     0.3126       , "t=4.7us",_
 991, 40, 80, 0,0,           1,-54360,     0            , "t=4.6s",_
 992, 41, 80, 0,4,          -1,-38420,     0            , "t=?",_
 993, 29, 81, 0,2.5,        -1,-31420,     0            , "t=73.2ms",_
 994, 30, 81, 0,2.5,         1,-46200,     0            , "t=303.2ms",_
 995, 31, 81, 0,2.5,        -1,-57628,     0            , "t=1.217s",_
 996, 32, 81, 0,4.5,         1,-66291.7,   0            , "t=8s",_
 997, 32, 81, 1,0.5,         1,-65612.6,   0.67914      , "t=8s",_
 998, 33, 81, 0,1.5,        -1,-72533.3,   0            , "t=33.3s",_
 999, 34, 81, 0,0.5,        -1,-76389.0,   0            , "t=18.45m",_
 1000, 34, 81, 1,3.5,        1,-76286.0,   0.103        , "t=57.28m",_
 1001, 35, 81, 0,1.5,       -1,-77977.0,   0            , "t=stbl",_
 1002, 35, 81, 1,4.5,        1,-77440.8,   0.5362       , "t=34.6us",_
 1003, 36, 81, 0,3.5,        1,-77696.2,   0            , "t=229ky",_
 1004, 36, 81, 1,0.5,       -1,-77505.6,   0.19064      , "t=13.10s",_
 1005, 37, 81, 0,1.5,       -1,-75457,     0            , "t=4.572h",_
 1006, 37, 81, 1,4.5,        1,-75371,     0.08631      , "t=30.5m",_
 1007, 38, 81, 0,0.5,       -1,-71528,     0            , "t=22.3m",_
 1008, 38, 81, 1,2.5,       -1,-71449,     0.07923      , "t=390ns",_
 1009, 38, 81, 2,3.5,        1,-71439,     0.08905      , "t=6.4us",_
 1010, 39, 81, 0,2.5,        1,-65713,     0            , "t=70.4s",_
 1011, 40, 81, 0,1.5,       -1,-57460,     0            , "t=5.5s",_
 1012, 41, 81, 0,4.5,        1,-46360,     0            , "t=&lt;44ns",_
 1013, 42, 81, 0,2.5,        1,-31750,     0            , "t=1#ms",_
 1014, 29, 82, 0,-77.777,    0,-25320,     0            , "t=50#ms",_
 1015, 30, 82, 0,0,          1,-42314,     0            , "t=177.9ms",_
 1016, 31, 82, 0,2,          0,-52930.7,   0            , "t=599ms",_
 1017, 31, 82, 1,4,         -1,-52789.7,   0.141        , "t=93.5ns",_
 1018, 32, 82, 0,0,          1,-65415.1,   0            , "t=4.56s",_
 1019, 33, 82, 0,2,         -1,-70105,     0            , "t=19.1s",_
 1020, 33, 82, 1,5,         -1,-69973,     0.1321       , "t=13.6s",_
 1021, 34, 82, 0,0,          1,-77593.9,   0            , "t=92Ey",_
 1022, 35, 82, 0,5,         -1,-77498.7,   0            , "t=35.282h",_
 1023, 35, 82, 1,2,         -1,-77452.8,   0.0459492    , "t=6.13m",_
 1024, 36, 82, 0,0,          1,-80591.785,               0            , "t=stbl",_
 1025, 37, 82, 0,1,          1,-76188,     0            , "t=1.273m",_
 1026, 37, 82, 1,5,         -1,-76118.8,   0.069        , "t=6.472h",_
 1027, 38, 82, 0,0,          1,-76010,     0            , "t=25.36d",_
 1028, 39, 82, 0,1,          1,-68064,     0            , "t=8.30s",_
 1029, 39, 82, 1,4,         -1,-67661,     0.40263      , "t=258ns",_
 1030, 39, 82, 2,6,          1,-67557,     0.5075       , "t=147ns",_
 1031, 40, 82, 0,0,          1,-63631,     0            , "t=32s",_
 1032, 41, 82, 0,0,          1,-52090,     0            , "t=50ms",_
 1033, 41, 82, 1,5,          1,-50910,     1.18         , "t=92ns",_
 1034, 42, 82, 0,0,          1,-40370,     0            , "t=30#ms",_
 1035, 30, 83, 0,1.5,        1,-36290,     0            , "t=119ms",_
 1036, 31, 83, 0,2.5,       -1,-49257.1,   0            , "t=308.1ms",_
 1037, 31, 83, 1,-77.777,    0,-49059.8,   0.1973       , "t=120ns",_
 1038, 32, 83, 0,2.5,        1,-60976.4,   0            , "t=1.85s",_
 1039, 33, 83, 0,2.5,       -1,-69669.3,   0            , "t=13.4s",_
 1040, 34, 83, 0,4.5,        1,-75341,     0            , "t=22.25m",_
 1041, 34, 83, 1,0.5,       -1,-75112,     0.22892      , "t=70.1s",_
 1042, 35, 83, 0,1.5,       -1,-79014,     0            , "t=2.374h",_
 1043, 35, 83, 1,9.5,       -1,-75945,     3.0692       , "t=729ns",_
 1044, 36, 83, 0,4.5,        1,-79990.633,               0            , "t=stbl",_
 1045, 36, 83, 1,3.5,        1,-79981.228,               0.0094053    , "t=156.8ns",_
 1046, 36, 83, 2,0.5,       -1,-79949.076,               0.0415575    , "t=1.830h",_
 1047, 37, 83, 0,2.5,       -1,-79070.6,   0            , "t=86.2d",_
 1048, 37, 83, 1,4.5,        1,-79028.5,   0.042078     , "t=7.8ms",_
 1049, 38, 83, 0,3.5,        1,-76798,     0            , "t=32.41h",_
 1050, 38, 83, 1,0.5,       -1,-76539,     0.25915      , "t=4.95s",_
 1051, 39, 83, 0,4.5,        1,-72206,     0            , "t=7.08m",_
 1052, 39, 83, 1,1.5,       -1,-72144,     0.06204      , "t=2.85m",_
 1053, 40, 83, 0,0.5,       -1,-65912,     0            , "t=42s",_
 1054, 40, 83, 1,2.5,       -1,-65859,     0.05272      , "t=530ns",_
 1055, 40, 83, 2,3.5,        1,-65835,     0.07704      , "t=1.8us",_
 1056, 41, 83, 0,4.5,        1,-57560,     0            , "t=3.9s",_
 1057, 42, 83, 0,1.5,       -1,-46340,     0            , "t=23ms",_
 1058, 43, 83, 0,0.5,       -1,-31320,     0            , "t=?",_
 1059, 30, 84, 0,0,          1,-31930,     0            , "t=50#ms",_
 1060, 31, 84, 0,0,         -1,-44090,     0            , "t=85ms",_
 1061, 32, 84, 0,0,          1,-58148,     0            , "t=951ms",_
 1062, 33, 84, 0,3,         -1,-65854,     0            , "t=4.02s",_
 1063, 33, 84, 1,-77.777,    0,-65850,     0            , "t=650ms",_
 1064, 34, 84, 0,0,          1,-75947.7,   0            , "t=3.26m",_
 1065, 35, 84, 0,2,         -1,-77783,     0            , "t=31.76m",_
 1066, 35, 84, 1,6,         -1,-77470,     0.31         , "t=6.0m",_
 1067, 35, 84, 2,1,          1,-77375,     0.4082       , "t=&lt;140ns",_
 1068, 36, 84, 0,0,          1,-82439.335,               0            , "t=stbl",_
 1069, 36, 84, 1,8,          1,-79203.27,  3.23607      , "t=1.83us",_
 1070, 37, 84, 0,2,         -1,-79759.0,   0            , "t=32.82d",_
 1071, 37, 84, 1,6,         -1,-79295.4,   0.46359      , "t=20.26m",_
 1072, 38, 84, 0,0,          1,-80649.6,   0            , "t=stbl",_
 1073, 39, 84, 0,6,          1,-73894,     0            , "t=39.5m",_
 1074, 39, 84, 1,1,          1,-73827,     0.067        , "t=4.6s",_
 1075, 39, 84, 2,4,         -1,-73684,     0.21042      , "t=292ns",_
 1076, 40, 84, 0,0,          1,-71422,     0            , "t=25.8m",_
 1077, 41, 84, 0,1,          1,-61219,     0            , "t=9.8s",_
 1078, 41, 84, 1,3,          1,-61171,     0.048        , "t=176ns",_
 1079, 41, 84, 2,5,         -1,-60881,     0.3377       , "t=92ns",_
 1080, 42, 84, 0,0,          1,-54170,     0            , "t=2.3s",_
 1081, 43, 84, 0,1,          1,-37700,     0            , "t=?",_
 1082, 30, 85, 0,2.5,        1,-25230,     0            , "t=50#ms",_
 1083, 31, 85, 0,2.5,       -1,-39850,     0            , "t=92.2ms",_
 1084, 32, 85, 0,1.5,        1,-53123,     0            , "t=494ms",_
 1085, 33, 85, 0,2.5,       -1,-63189,     0            , "t=2.021s",_
 1086, 34, 85, 0,2.5,        1,-72413.6,   0            , "t=32.9s",_
 1087, 35, 85, 0,1.5,       -1,-78575,     0            , "t=2.90m",_
 1088, 36, 85, 0,4.5,        1,-81480.3,   0            , "t=10.739y",_
 1089, 36, 85, 1,0.5,       -1,-81175.4,   0.304871     , "t=4.480h",_
 1090, 36, 85, 2,8.5,        1,-79488.5,   1.9918       , "t=1.82us",_
 1091, 37, 85, 0,2.5,       -1,-82167.331,               0            , "t=stbl",_
 1092, 37, 85, 1,4.5,        1,-81653.325,               0.5140065    , "t=1.015us",_
 1093, 38, 85, 0,4.5,        1,-81103.3,   0            , "t=64.849d",_
 1094, 38, 85, 1,0.5,       -1,-80864.5,   0.23879      , "t=67.63m",_
 1095, 39, 85, 0,0.5,       -1,-77842,     0            , "t=2.68h",_
 1096, 39, 85, 1,4.5,        1,-77822,     0.01968      , "t=4.86h",_
 1097, 39, 85, 2,2.5,       -1,-77576,     0.26618      , "t=178ns",_
 1098, 40, 85, 0,3.5,        1,-73175,     0            , "t=7.86m",_
 1099, 40, 85, 1,0.5,       -1,-72883,     0.2922       , "t=10.9s",_
 1100, 41, 85, 0,4.5,        1,-66280,     0            , "t=20.5s",_
 1101, 41, 85, 1,0.5,       -1,-66130,     0.15         , "t=3.3s",_
 1102, 42, 85, 0,0.5,        1,-57510,     0            , "t=3.2s",_
 1103, 43, 85, 0,0.5,       -1,-45850,     0            , "t=?",_
 1104, 44, 85, 0,1.5,       -1,-30950,     0            , "t=1#ms",_
 1105, 31, 86, 0,-77.777,    0,-34080,     0            , "t=47ms",_
 1106, 32, 86, 0,0,          1,-49400,     0            , "t=226ms",_
 1107, 33, 86, 0,1,         -1,-58962,     0            , "t=945ms",_
 1108, 34, 86, 0,0,          1,-70503.2,   0            , "t=14.3s",_
 1109, 34, 86, 1,-77.777,    0,-68131.2,   2.372        , "t=620ms",_
 1110, 35, 86, 0,1,         -1,-75632,     0            , "t=55.1s",_
 1111, 36, 86, 0,0,          1,-83265.666,               0            , "t=stbl",_
 1112, 37, 86, 0,2,         -1,-82746.99,  0            , "t=18.642d",_
 1113, 37, 86, 1,6,         -1,-82190.94,  0.55605      , "t=1.017m",_
 1114, 38, 86, 0,0,          1,-84523.089,               0            , "t=stbl",_
 1115, 38, 86, 1,8,          1,-81567.00,  2.95609      , "t=455ns",_
 1116, 39, 86, 0,4,         -1,-79283,     0            , "t=14.74h",_
 1117, 39, 86, 1,8,          1,-79065,     0.21821      , "t=47.4m",_
 1118, 39, 86, 2,6,          1,-78981,     0.30218      , "t=125.3ns",_
 1119, 40, 86, 0,0,          1,-77969,     0            , "t=16.5h",_
 1120, 41, 86, 0,6,          1,-69134,     0            , "t=88s",_
 1121, 41, 86, 1,0,          0,-68880,     0.25         , "t=56.3s",_
 1122, 42, 86, 0,0,          1,-64110,     0            , "t=19.1s",_
 1123, 43, 86, 0,0,          1,-51570,     0            , "t=55ms",_
 1124, 43, 86, 1,6,          1,-50050,     1.524        , "t=1.10us",_
 1125, 44, 86, 0,0,          1,-39770,     0            , "t=50#ms",_
 1126, 31, 87, 0,2.5,       -1,-29250,     0            , "t=10#ms",_
 1127, 32, 87, 0,2.5,        1,-44080,     0            , "t=150#ms",_
 1128, 33, 87, 0,2.5,       -1,-55617.9,   0            , "t=492ms",_
 1129, 34, 87, 0,1.5,        1,-66426.1,   0            , "t=5.50s",_
 1130, 35, 87, 0,2.5,       -1,-73892,     0            , "t=55.65s",_
 1131, 36, 87, 0,2.5,        1,-80709.52,  0            , "t=76.3m",_
 1132, 37, 87, 0,1.5,       -1,-84597.791,               0            , "t=49.7Gy",_
 1133, 38, 87, 0,4.5,        1,-84880.066,               0            , "t=stbl",_
 1134, 38, 87, 1,0.5,       -1,-84491.537,               0.3885287    , "t=2.815h",_
 1135, 39, 87, 0,0.5,       -1,-83018.4,   0            , "t=79.8h",_
 1136, 39, 87, 1,4.5,        1,-82637.6,   0.38082      , "t=13.37h",_
 1137, 40, 87, 0,4.5,        1,-79347,     0            , "t=1.68h",_
 1138, 40, 87, 1,0.5,       -1,-79011,     0.33584      , "t=14.0s",_
 1139, 41, 87, 0,0.5,       -1,-73874,     0            , "t=3.7m",_
 1140, 41, 87, 1,4.5,        1,-73870,     0.0039       , "t=2.6m",_
 1141, 42, 87, 0,3.5,        1,-66884.8,   0            , "t=14.1s",_
 1142, 43, 87, 0,4.5,        1,-57690,     0            , "t=2.2s",_
 1143, 43, 87, 1,0.5,       -1,-57683,     0.007        , "t=2#s",_
 1144, 43, 87, 2,3.5,        1,-57619,     0.071        , "t=647ns",_
 1145, 44, 87, 0,0.5,       -1,-45520,     0            , "t=50#ms",_
 1146, 32, 88, 0,0,          1,-40140,     0            , "t=100#ms",_
 1147, 33, 88, 0,-77.777,    0,-50720,     0            , "t=270ms",_
 1148, 34, 88, 0,0,          1,-63884,     0            , "t=1.53s",_
 1149, 35, 88, 0,1,         -1,-70716,     0            , "t=16.34s",_
 1150, 35, 88, 1,4,         -1,-70446,     0.2701       , "t=5.51us",_
 1151, 36, 88, 0,0,          1,-79691.3,   0            , "t=2.825h",_
 1152, 37, 88, 0,2,         -1,-82608.99,  0            , "t=17.773m",_
 1153, 37, 88, 1,7,          1,-81235.2,   1.3738       , "t=123ns",_
 1154, 38, 88, 0,0,          1,-87921.618,               0            , "t=stbl",_
 1155, 39, 88, 0,4,         -1,-84299.0,   0            , "t=106.626d",_
 1156, 39, 88, 1,1,          1,-83906.1,   0.39286      , "t=301us",_
 1157, 39, 88, 2,8,          1,-83624.5,   0.67455      , "t=13.98ms",_
 1158, 40, 88, 0,0,          1,-83629,     0            , "t=83.4d",_
 1159, 40, 88, 1,8,          1,-80741,     2.88779      , "t=1.320us",_
 1160, 41, 88, 0,8,          1,-76170,     0            , "t=14.50m",_
 1161, 41, 88, 1,4,         -1,-76040,     0.13         , "t=7.7m",_
 1162, 42, 88, 0,0,          1,-72687,     0            , "t=8.0m",_
 1163, 43, 88, 0,5,          1,-61680,     0            , "t=6.4s",_
 1164, 43, 88, 1,2,          1,-61680,     0            , "t=5.8s",_
 1165, 43, 88, 2,4,          1,-61580,     0.1          , "t=146ns",_
 1166, 44, 88, 0,0,          1,-54340,     0            , "t=1.3s",_
 1167, 45, 88, 0,-77.777,    0,-36860,     0            , "t=1#ms",_
 1168, 32, 89, 0,1.5,        1,-33730,     0            , "t=50#ms",_
 1169, 33, 89, 0,2.5,       -1,-46800,     0            , "t=200#ms",_
 1170, 34, 89, 0,2.5,        1,-58992,     0            , "t=430ms",_
 1171, 35, 89, 0,1.5,       -1,-68274,     0            , "t=4.357s",_
 1172, 36, 89, 0,1.5,        1,-76535.8,   0            , "t=3.15m",_
 1173, 37, 89, 0,1.5,       -1,-81712,     0            , "t=15.32m",_
 1174, 38, 89, 0,2.5,        1,-86209.02,  0            , "t=50.563d",_
 1175, 39, 89, 0,0.5,       -1,-87708.4,   0            , "t=stbl",_
 1176, 39, 89, 1,4.5,        1,-86799.4,   0.90897      , "t=15.663s",_
 1177, 40, 89, 0,4.5,        1,-84876,     0            , "t=78.41h",_
 1178, 40, 89, 1,0.5,       -1,-84288,     0.58782      , "t=4.161m",_
 1179, 41, 89, 0,4.5,        1,-80625,     0            , "t=2.03h",_
 1180, 41, 89, 1,0.5,       -1,-80630,     0            , "t=1.10h",_
 1181, 42, 89, 0,4.5,        1,-75015,     0            , "t=2.11m",_
 1182, 42, 89, 1,0.5,       -1,-74628,     0.3875       , "t=190ms",_
 1183, 43, 89, 0,4.5,        1,-67395,     0            , "t=12.8s",_
 1184, 43, 89, 1,0.5,       -1,-67332,     0.0626       , "t=12.9s",_
 1185, 44, 89, 0,4.5,        1,-58260,     0            , "t=1.5s",_
 1186, 45, 89, 0,3.5,        1,-45860,     0            , "t=10#ms",_
 1187, 32, 90, 0,0,          1,-29220,     0            , "t=50#ms",_
 1188, 33, 90, 0,-77.777,    0,-41330,     0            , "t=80#ms",_
 1189, 33, 90, 1,-77.777,    0,-41210,     0.1245       , "t=220ns",_
 1190, 34, 90, 0,0,          1,-55800,     0            , "t=210ms",_
 1191, 35, 90, 0,-77.777,    0,-64000,     0            , "t=1.910s",_
 1192, 36, 90, 0,0,          1,-74959.2,   0            , "t=32.32s",_
 1193, 37, 90, 0,0,         -1,-79364,     0            , "t=158s",_
 1194, 37, 90, 1,3,         -1,-79257,     0.1069       , "t=258s",_
 1195, 37, 90, 0,0,          0,-79293,     0.071        , "t=R=2 ~ ~ 1",_
 1196, 38, 90, 0,0,          1,-85948.1,   0            , "t=28.79y",_
 1197, 39, 90, 0,2,         -1,-86494.1,   0            , "t=64.00h",_
 1198, 39, 90, 1,7,          1,-85812.4,   0.68167      , "t=3.19h",_
 1199, 40, 90, 0,0,          1,-88772.54,  0            , "t=stbl",_
 1200, 40, 90, 1,5,         -1,-86453.54,  2.319        , "t=809.2ms",_
 1201, 40, 90, 2,8,          1,-85183.12,  3.589419     , "t=131ns",_
 1202, 41, 90, 0,8,          1,-82662,     0            , "t=14.60h",_
 1203, 41, 90, 1,6,          1,-82540,     0.12237      , "t=63us",_
 1204, 41, 90, 2,4,         -1,-82537,     0.12467      , "t=18.81s",_
 1205, 42, 90, 0,0,          1,-80173,     0            , "t=5.56h",_
 1206, 42, 90, 1,8,          1,-77298,     2.87473      , "t=1.12us",_
 1207, 43, 90, 0,8,          1,-70724.7,   0            , "t=49.2s",_
 1208, 43, 90, 1,1,          1,-70580.7,   0.144        , "t=8.7s",_
 1209, 44, 90, 0,0,          1,-64884,     0            , "t=11s",_
 1210, 45, 90, 0,0,          1,-51700,     0            , "t=15ms",_
 1211, 45, 90, 1,9,          1,-51700,     0            , "t=1.1s",_
 1212, 46, 90, 0,0,          1,-39710,     0            , "t=10#ms",_
 1213, 33, 91, 0,2.5,       -1,-36900,     0            , "t=50#ms",_
 1214, 34, 91, 0,0.5,        1,-50580,     0            , "t=270ms",_
 1215, 35, 91, 0,2.5,       -1,-61107,     0            , "t=543ms",_
 1216, 36, 91, 0,2.5,        1,-70974.0,   0            , "t=8.57s",_
 1217, 37, 91, 0,1.5,       -1,-77745,     0            , "t=58.2s",_
 1218, 38, 91, 0,2.5,        1,-83652,     0            , "t=9.65h",_
 1219, 39, 91, 0,0.5,       -1,-86351.3,   0            , "t=58.51d",_
 1220, 39, 91, 1,4.5,        1,-85795.7,   0.55558      , "t=49.71m",_
 1221, 40, 91, 0,2.5,        1,-87895.57,  0            , "t=stbl",_
 1222, 40, 91, 1,10.5,       1,-84728.3,   3.1673       , "t=4.35us",_
 1223, 41, 91, 0,4.5,        1,-86638.0,   0            , "t=680y",_
 1224, 41, 91, 1,0.5,       -1,-86533.4,   0.1046       , "t=60.86d",_
 1225, 41, 91, 2,8.5,       -1,-84603.6,   2.03442      , "t=3.76us",_
 1226, 42, 91, 0,4.5,        1,-82209,     0            , "t=15.49m",_
 1227, 42, 91, 1,0.5,       -1,-81556,     0.65301      , "t=64.6s",_
 1228, 43, 91, 0,4.5,        1,-75986.6,   0            , "t=3.14m",_
 1229, 43, 91, 1,0.5,       -1,-75847.3,   0.1393       , "t=3.3m",_
 1230, 44, 91, 0,4.5,        1,-68239.8,   0            , "t=8.0s",_
 1231, 44, 91, 1,0.5,       -1,-68580,    -0.34         , "t=7.6s",_
 1232, 45, 91, 0,3.5,        1,-58570,     0            , "t=1.60s",_
 1233, 45, 91, 1,0.5,       -1,-58400,     0.1729       , "t=1.46s",_
 1234, 46, 91, 0,3.5,        1,-45930,     0            , "t=10#ms",_
 1235, 33, 92, 0,-77.777,    0,-30980,     0            , "t=30#ms",_
 1236, 34, 92, 0,0,          1,-46720,     0            , "t=100#ms",_
 1237, 34, 92, 1,-77.777,    0,-44780,     1.94         , "t=12us",_
 1238, 35, 92, 0,2,         -1,-56233,     0            , "t=0.314s",_
 1239, 35, 92, 1,-77.777,    0,-55571,     0.662        , "t=88ns",_
 1240, 35, 92, 2,-77.777,    0,-55095,     1.138        , "t=85ns",_
 1241, 36, 92, 0,0,          1,-68769.3,   0            , "t=1.840s",_
 1242, 37, 92, 0,0,         -1,-74772,     0            , "t=4.48s",_
 1243, 38, 92, 0,0,          1,-82867,     0            , "t=2.611h",_
 1244, 39, 92, 0,2,         -1,-84816,     0            , "t=3.54h",_
 1245, 39, 92, 1,7,          1,-84010,     0.807        , "t=3.7us",_
 1246, 40, 92, 0,0,          1,-88459.03,  0            , "t=stbl",_
 1247, 41, 92, 0,7,          1,-86453.3,   0            , "t=34.7My",_
 1248, 41, 92, 1,2,          1,-86317.8,   0.1355       , "t=10.15d",_
 1249, 41, 92, 2,2,         -1,-86227.5,   0.2258       , "t=5.9us",_
 1250, 42, 92, 0,0,          1,-86808.58,  0            , "t=stbl",_
 1251, 42, 92, 1,8,          1,-84048.06,  2.76052      , "t=190ns",_
 1252, 43, 92, 0,8,          1,-78926,     0            , "t=4.25m",_
 1253, 43, 92, 1,4,          1,-78656,     0.27009      , "t=1.03us",_
 1254, 43, 92, 2,3,          1,-78397,     0.52942      , "t=&lt;0.1us",_
 1255, 44, 92, 0,0,          1,-74301.2,   0            , "t=3.65m",_
 1256, 45, 92, 0,6,          1,-62999,     0            , "t=4.66s",_
 1257, 45, 92, 1,2,          1,-62950,     0.05         , "t=0.53s",_
 1258, 46, 92, 0,0,          1,-54580,     0            , "t=1.1s",_
 1259, 47, 92, 0,-77.777,    0,-37130,     0            , "t=1#ms",_
 1260, 34, 93, 0,0.5,        1,-40720,     0            , "t=50#ms",_
 1261, 34, 93, 1,-77.777,    0,-40040,     0.6782       , "t=420ns",_
 1262, 35, 93, 0,2.5,       -1,-52890,     0            , "t=152ms",_
 1263, 36, 93, 0,0.5,        1,-64136.0,   0            , "t=1.286s",_
 1264, 37, 93, 0,2.5,       -1,-72620,     0            , "t=5.84s",_
 1265, 37, 93, 1,13.5,      -1,-68197,     4.4231       , "t=111ns",_
 1266, 37, 93, 0,1.5,       -1,-72367,     0.25339      , "t=&lt;0.5ns",_
 1267, 38, 93, 0,2.5,        1,-80086,     0            , "t=7.43m",_
 1268, 39, 93, 0,0.5,       -1,-84227,     0            , "t=10.18h",_
 1269, 39, 93, 1,4.5,        1,-83468,     0.758719     , "t=820ms",_
 1270, 40, 93, 0,2.5,        1,-87122.0,   0            , "t=1.61My",_
 1271, 41, 93, 0,4.5,        1,-87212.8,   0            , "t=stbl",_
 1272, 41, 93, 1,0.5,       -1,-87182.0,   0.03077      , "t=16.12y",_
 1273, 41, 93, 2,-77.777,    0,-79753,     7.46         , "t=1.5us",_
 1274, 42, 93, 0,2.5,        1,-86807.07,  0            , "t=4.0ky",_
 1275, 42, 93, 1,10.5,       1,-84382.12,  2.42495      , "t=6.85h",_
 1276, 42, 93, 2,19.5,      -1,-77112,     9.695        , "t=1.8us",_
 1277, 43, 93, 0,4.5,        1,-83606.1,   0            , "t=2.75h",_
 1278, 43, 93, 1,0.5,       -1,-83214.3,   0.39184      , "t=43.5m",_
 1279, 43, 93, 2,8.5,       -1,-81420.9,   2.18516      , "t=10.2us",_
 1280, 44, 93, 0,4.5,        1,-77216.7,   0            , "t=59.7s",_
 1281, 44, 93, 1,0.5,       -1,-76482.3,   0.7344       , "t=10.8s",_
 1282, 44, 93, 2,10.5,       1,-75134.2,   2.0825       , "t=2.49us",_
 1283, 45, 93, 0,4.5,        1,-69011.8,   0            , "t=13.9s",_
 1284, 46, 93, 0,4.5,        1,-59000,     0            , "t=1.15s",_
 1285, 47, 93, 0,4.5,        1,-46270,     0            , "t=20#ms",_
 1286, 34, 94, 0,0,          1,-36800,     0            , "t=20#ms",_
 1287, 35, 94, 0,2,         -1,-47400,     0            , "t=70ms",_
 1288, 35, 94, 1,-77.777,    0,-47110,     0.2946       , "t=530ns",_
 1289, 36, 94, 0,0,          1,-61348,     0            , "t=212ms",_
 1290, 37, 94, 0,3,         -1,-68562.8,   0            , "t=2.702s",_
 1291, 37, 94, 1,10,        -1,-66487.9,   2.0749       , "t=107ns",_
 1292, 38, 94, 0,0,          1,-78845.7,   0            , "t=75.3s",_
 1293, 39, 94, 0,2,         -1,-82351,     0            , "t=18.7m",_
 1294, 39, 94, 1,5,          1,-81149,     1.2023       , "t=1.295us",_
 1295, 40, 94, 0,0,          1,-87269.32,  0            , "t=stbl",_
 1296, 41, 94, 0,6,          1,-86369.1,   0            , "t=20.4ky",_
 1297, 41, 94, 1,3,          1,-86328.2,   0.040892     , "t=6.263m",_
 1298, 42, 94, 0,0,          1,-88414.06,  0            , "t=stbl",_
 1299, 43, 94, 0,7,          1,-84158,     0            , "t=293m",_
 1300, 43, 94, 1,2,          1,-84082,     0.076        , "t=52.0m",_
 1301, 44, 94, 0,0,          1,-82584,     0            , "t=51.8m",_
 1302, 44, 94, 1,8,          1,-79940,     2.6441       , "t=71us",_
 1303, 45, 94, 0,4,          1,-72908,     0            , "t=70.6s",_
 1304, 45, 94, 1,2,          1,-72853,     0.0546       , "t=480ns",_
 1305, 45, 94, 2,8,          1,-72610,     0.3          , "t=25.8s",_
 1306, 46, 94, 0,0,          1,-66102,     0            , "t=9.0s",_
 1307, 46, 94, 1,14,         1,-61219,     4.8831       , "t=511.0ns",_
 1308, 46, 94, 2,19,        -1,-58893,     7.2091       , "t=197ns",_
 1309, 47, 94, 0,0,          1,-52410,     0            , "t=37ms",_
 1310, 47, 94, 1,7,          1,-51060,     1.35         , "t=550ms",_
 1311, 47, 94, 2,21,         1,-45920,     6.49         , "t=400ms",_
 1312, 48, 94, 0,0,          1,-40140,     0            , "t=80#ms",_
 1313, 34, 95, 0,1.5,        1,-30460,     0            , "t=10#ms",_
 1314, 35, 95, 0,2.5,       -1,-43770,     0            , "t=50#ms",_
 1315, 35, 95, 1,-77.777,    0,-43230,     0.5379       , "t=6.8us",_
 1316, 36, 95, 0,0.5,        1,-56159,     0            , "t=114ms",_
 1317, 36, 95, 1,3.5,        1,-55964,     0.1955       , "t=1.582us",_
 1318, 37, 95, 0,2.5,       -1,-65891,     0            , "t=377.7ms",_
 1319, 37, 95, 1,4.5,        1,-65056,     0.835        , "t=&lt;500ns",_
 1320, 38, 95, 0,0.5,        1,-75120,     0            , "t=23.90s",_
 1321, 39, 95, 0,0.5,       -1,-81209,     0            , "t=10.3m",_
 1322, 39, 95, 1,4.5,        1,-80121,     1.0876       , "t=48.6us",_
 1323, 40, 95, 0,2.5,        1,-85659.9,   0            , "t=64.032d",_
 1324, 41, 95, 0,4.5,        1,-86786.3,   0            , "t=34.991d",_
 1325, 41, 95, 1,0.5,       -1,-86550.6,   0.23569      , "t=3.61d",_
 1326, 42, 95, 0,2.5,        1,-87711.86,  0            , "t=stbl",_
 1327, 43, 95, 0,4.5,        1,-86021,     0            , "t=20.0h",_
 1328, 43, 95, 1,0.5,       -1,-85982,     0.03891      , "t=61d",_
 1329, 44, 95, 0,2.5,        1,-83458,     0            , "t=1.643h",_
 1330, 45, 95, 0,4.5,        1,-78341,     0            , "t=5.02m",_
 1331, 45, 95, 1,0.5,       -1,-77798,     0.5433       , "t=1.96m",_
 1332, 46, 95, 0,4.5,        1,-69966,     0            , "t=7.5s",_
 1333, 46, 95, 1,10.5,       1,-68091,     1.87513      , "t=13.3s",_
 1334, 47, 95, 0,4.5,        1,-59600,     0            , "t=1.76s",_
 1335, 47, 95, 1,0.5,       -1,-59260,     0.3442       , "t=&lt;500ms",_
 1336, 47, 95, 2,11.5,       1,-57070,     2.5313       , "t=&lt;16ms",_
 1337, 48, 95, 0,4.5,        1,-46630,     0            , "t=90ms",_
 1338, 35, 96, 0,-77.777,    0,-38160,     0            , "t=20#ms",_
 1339, 35, 96, 1,-77.777,    0,-37850,     0.3115       , "t=3.0us",_
 1340, 36, 96, 0,0,          1,-53080,     0            , "t=80ms",_
 1341, 37, 96, 0,2,         -1,-61354,     0            , "t=201ms",_
 1342, 37, 96, 1,1,          1,-61350,     0            , "t=200#ms",_
 1343, 37, 96, 2,10,        -1,-60219,     1.1346       , "t=1.80us",_
 1344, 38, 96, 0,0,          1,-72924,     0            , "t=1.07s",_
 1345, 39, 96, 0,0,         -1,-78336,     0            , "t=5.34s",_
 1346, 39, 96, 1,8,          1,-76796,     1.54         , "t=9.6s",_
 1347, 40, 96, 0,0,          1,-85438.85,  0            , "t=23Ey",_
 1348, 41, 96, 0,6,          1,-85602.82,  0            , "t=23.35h",_
 1349, 42, 96, 0,0,          1,-88794.88,  0            , "t=stbl",_
 1350, 43, 96, 0,7,          1,-85822,     0            , "t=4.28d",_
 1351, 43, 96, 1,4,          1,-85788,     0.03423      , "t=51.5m",_
 1352, 44, 96, 0,0,          1,-86080.37,  0            , "t=stbl",_
 1353, 45, 96, 0,6,          1,-79688,     0            , "t=9.90m",_
 1354, 45, 96, 1,3,          1,-79636,     0.05198      , "t=1.51m",_
 1355, 46, 96, 0,0,          1,-76183,     0            , "t=122s",_
 1356, 46, 96, 1,8,          1,-73652,     2.53057      , "t=1.81us",_
 1357, 47, 96, 0,8,          1,-64510,     0            , "t=4.44s",_
 1358, 47, 96, 1,2,          1,-64510,     0            , "t=6.9s",_
 1359, 47, 96, 2,13,        -1,-62050,     2.4614       , "t=100us",_
 1360, 48, 96, 0,0,          1,-55570,     0            , "t=880ms",_
 1361, 48, 96, 1,16,         1,-50270,     5.3          , "t=300ms",_
 1362, 49, 96, 0,-77.777,    0,-37890,     0            , "t=1#ms",_
 1363, 35, 97, 0,2.5,       -1,-34060,     0            , "t=10#ms",_
 1364, 36, 97, 0,1.5,        1,-47420,     0            , "t=62.2ms",_
 1365, 37, 97, 0,1.5,        1,-58519.1,   0            , "t=169.1ms",_
 1366, 37, 97, 1,0.5,        0,-58442.5,   0.0766       , "t=5.7us",_
 1367, 38, 97, 0,0.5,        1,-68581,     0            , "t=429ms",_
 1368, 38, 97, 1,3.5,        1,-68273,     0.30813      , "t=165ns",_
 1369, 38, 97, 2,4.5,        1,-67750,     0.83083      , "t=515ns",_
 1370, 39, 97, 0,0.5,       -1,-76121,     0            , "t=3.75s",_
 1371, 39, 97, 1,4.5,        1,-75453,     0.66752      , "t=1.17s",_
 1372, 39, 97, 2,13.5,      -1,-72598,     3.5226       , "t=142ms",_
 1373, 40, 97, 0,0.5,        1,-82942.7,   0            , "t=16.749h",_
 1374, 40, 97, 1,3.5,        1,-81678.3,   1.26435      , "t=104.8ns",_
 1375, 41, 97, 0,4.5,        1,-85606,     0            , "t=72.1m",_
 1376, 41, 97, 1,0.5,       -1,-84863,     0.74335      , "t=58.7s",_
 1377, 42, 97, 0,2.5,        1,-87544.69,  0            , "t=stbl",_
 1378, 43, 97, 0,4.5,        1,-87224,     0            , "t=4.21My",_
 1379, 43, 97, 1,0.5,       -1,-87127,     0.09657      , "t=91.0d",_
 1380, 44, 97, 0,2.5,        1,-86120.6,   0            , "t=2.8370d",_
 1381, 45, 97, 0,4.5,        1,-82600,     0            , "t=30.7m",_
 1382, 45, 97, 1,0.5,       -1,-82340,     0.25876      , "t=46.2m",_
 1383, 46, 97, 0,2.5,        1,-77806,     0            , "t=3.10m",_
 1384, 47, 97, 0,4.5,        1,-70830,     0            , "t=25.5s",_
 1385, 47, 97, 1,0.5,       -1,-70430,     0.4          , "t=100#ms",_
 1386, 48, 97, 0,4.5,        1,-60450,     0            , "t=1.10s",_
 1387, 48, 97, 1,12.5,       1,-58950,     1.5          , "t=3.8s",_
 1388, 49, 97, 0,4.5,        1,-47190,     0            , "t=50ms",_
 1389, 35, 98, 0,-77.777,    0,-28250,     0            , "t=5#ms",_
 1390, 36, 98, 0,0,          1,-44310,     0            , "t=42.8ms",_
 1391, 37, 98, 0,0,         -1,-54369,     0            , "t=114ms",_
 1392, 37, 98, 1,3,          0,-54296,     0.073        , "t=96ms",_
 1393, 37, 98, 2,-77.777,    0,-54191,     0.1783       , "t=358ns",_
 1394, 38, 98, 0,0,          1,-66423,     0            , "t=653ms",_
 1395, 39, 98, 0,0,         -1,-72295,     0            , "t=548ms",_
 1396, 39, 98, 1,5,          1,-72054,     0.241        , "t=2.0s",_
 1397, 39, 98, 2,2,         -1,-72124,     0.17074      , "t=610ns",_
 1398, 40, 98, 0,0,          1,-81287,     0            , "t=30.7s",_
 1399, 40, 98, 1,17,        -1,-74683,     6.6037       , "t=1.9us",_
 1400, 41, 98, 0,1,          1,-83525,     0            , "t=2.86s",_
 1401, 41, 98, 1,5,          1,-83441,     0.084        , "t=51.3m",_
 1402, 42, 98, 0,0,          1,-88115.97,  0            , "t=stbl",_
 1403, 43, 98, 0,6,          1,-86432,     0            , "t=4.2My",_
 1404, 43, 98, 1,2,         -1,-86341,     0.09076      , "t=14.7us",_
 1405, 44, 98, 0,0,          1,-88225,     0            , "t=stbl",_
 1406, 45, 98, 0,2,          1,-83175,     0            , "t=8.72m",_
 1407, 45, 98, 1,5,          1,-83120,     0.06         , "t=3.6m",_
 1408, 46, 98, 0,0,          1,-81321,     0            , "t=17.7m",_
 1409, 47, 98, 0,5,          0,-73070,     0            , "t=47.5s",_
 1410, 47, 98, 1,3,          1,-72900,     0.16783      , "t=220ns",_
 1411, 48, 98, 0,0,          1,-67640,     0            , "t=9.2s",_
 1412, 48, 98, 1,8,          1,-65210,     2.4275       , "t=189ns",_
 1413, 48, 98, 2,12,         1,-61010,     6.635        , "t=240ns",_
 1414, 49, 98, 0,0,          1,-53900,     0            , "t=37ms",_
 1415, 49, 98, 1,-77.777,    0,-53900,     0            , "t=1.01s",_
 1416, 36, 99, 0,2.5,       -1,-38760,     0            , "t=40ms",_
 1417, 37, 99, 0,1.5,        1,-51121,     0            , "t=56.4ms",_
 1418, 38, 99, 0,1.5,        1,-62521,     0            , "t=269ms",_
 1419, 39, 99, 0,2.5,        1,-70650,     0            , "t=1.484s",_
 1420, 39, 99, 1,8.5,        1,-68508,     2.14165      , "t=8.6us",_
 1421, 40, 99, 0,0.5,        1,-77621,     0            , "t=2.1s",_
 1422, 40, 99, 1,3.5,        1,-77369,     0.25196      , "t=293ns",_
 1423, 41, 99, 0,4.5,        1,-82335,     0            , "t=15.0s",_
 1424, 41, 99, 1,0.5,       -1,-81970,     0.36527      , "t=2.5m",_
 1425, 42, 99, 0,0.5,        1,-85970.10,  0            , "t=65.976h",_
 1426, 42, 99, 1,2.5,        1,-85872.32,  0.097785     , "t=15.5us",_
 1427, 42, 99, 2,5.5,       -1,-85286.00,  0.6841       , "t=760ns",_
 1428, 43, 99, 0,4.5,        1,-87327.9,   0            , "t=211.1ky",_
 1429, 43, 99, 1,0.5,       -1,-87185.2,   0.1426832    , "t=6.0067h",_
 1430, 44, 99, 0,2.5,        1,-87625.4,   0            , "t=stbl",_
 1431, 45, 99, 0,0.5,       -1,-85581,     0            , "t=16.1d",_
 1432, 45, 99, 1,4.5,        1,-85516,     0.0646       , "t=4.7h",_
 1433, 46, 99, 0,2.5,        1,-82183,     0            , "t=21.4m",_
 1434, 47, 99, 0,4.5,        1,-76712,     0            , "t=2.07m",_
 1435, 47, 99, 1,0.5,       -1,-76206,     0.5061       , "t=10.5s",_
 1436, 48, 99, 0,2.5,        1,-69931.1,   0            , "t=16s",_
 1437, 49, 99, 0,4.5,        1,-61380,     0            , "t=3.1s",_
 1438, 49, 99, 1,0.5,       -1,-60980,     0.4          , "t=1#s",_
 1439, 50, 99, 0,4.5,        1,-47940,     0            , "t=5#ms",_
 1440, 50, 99, 1,0.5,       -1,-47540,     0.4          , "t=?",_
 1441, 36, 100, 0,0,         1,-35050,     0            , "t=12ms",_
 1442, 37, 100, 0,3,         1,-46247,     0            , "t=48ms",_
 1443, 38, 100, 0,0,         1,-59821,     0            , "t=202ms",_
 1444, 38, 100, 1,4,        -1,-58202,     1.61872      , "t=122ns",_
 1445, 39, 100, 0,1,        -1,-67327,     0            , "t=735ms",_
 1446, 39, 100, 1,4,         1,-67183,     0.144        , "t=940ms",_
 1447, 40, 100, 0,0,         1,-76377,     0            , "t=7.1s",_
 1448, 41, 100, 0,1,         1,-79797,     0            , "t=1.5s",_
 1449, 41, 100, 1,5,         1,-79484.7,   0.313        , "t=2.99s",_
 1450, 41, 100, 2,4,        -1,-79450,     0.347        , "t=460ns",_
 1451, 42, 100, 0,0,         1,-86193.0,   0            , "t=7.1Ey",_
 1452, 43, 100, 0,1,         1,-86020.9,   0            , "t=15.46s",_
 1453, 43, 100, 1,4,         1,-85820.2,   0.20067      , "t=8.32us",_
 1454, 43, 100, 2,6,         1,-85777.0,   0.24395      , "t=3.2us",_
 1455, 44, 100, 0,0,         1,-89227.4,   0            , "t=stbl",_
 1456, 45, 100, 0,1,        -1,-85591,     0            , "t=20.8h",_
 1457, 45, 100, 1,2,         1,-85516,     0.074782     , "t=214.0ns",_
 1458, 45, 100, 2,5,         1,-85483,     0.1076       , "t=4.6m",_
 1459, 46, 100, 0,0,         1,-85213,     0            , "t=3.63d",_
 1460, 47, 100, 0,5,         1,-78138,     0            , "t=2.01m",_
 1461, 47, 100, 1,2,         1,-78122,     0.01552      , "t=2.24m",_
 1462, 48, 100, 0,0,         1,-74194.6,   0            , "t=49.1s",_
 1463, 49, 100, 0,6,         1,-64310,     0            , "t=5.83s",_
 1464, 50, 100, 0,0,         1,-57280,     0            , "t=1.16s",_
 1465, 50, 100, 1,6,         1,-52780,     4.5          , "t=100#ns",_
 1466, 36, 101, 0,2.5,       1,-29130,     0            , "t=5#ms",_
 1467, 37, 101, 0,1.5,       1,-42850,     0            , "t=31.8ms",_
 1468, 38, 101, 0,2.5,      -1,-55325,     0            , "t=113.8ms",_
 1469, 39, 101, 0,2.5,       1,-65061,     0            , "t=426ms",_
 1470, 39, 101, 1,-77.777,   0,-64730,     0.3315       , "t=190ns",_
 1471, 39, 101, 2,-77.777,   0,-63854,     1.207        , "t=870ns",_
 1472, 40, 101, 0,1.5,       1,-73166,     0            , "t=2.3s",_
 1473, 41, 101, 0,2.5,       1,-78891,     0            , "t=7.1s",_
 1474, 42, 101, 0,0.5,       1,-83519.9,   0            , "t=14.61m",_
 1475, 42, 101, 1,1.5,       1,-83506.4,   0.013497     , "t=226ns",_
 1476, 42, 101, 2,2.5,       1,-83462.9,   0.057015     , "t=133ns",_
 1477, 43, 101, 0,4.5,       1,-86345,     0            , "t=14.22m",_
 1478, 43, 101, 1,0.5,      -1,-86137,     0.207526     , "t=636us",_
 1479, 44, 101, 0,2.5,       1,-87958.1,   0            , "t=stbl",_
 1480, 44, 101, 1,5.5,      -1,-87430.5,   0.52756      , "t=17.5us",_
 1481, 45, 101, 0,0.5,      -1,-87412,     0            , "t=3.3y",_
 1482, 45, 101, 1,4.5,       1,-87255,     0.15732      , "t=4.34d",_
 1483, 46, 101, 0,2.5,       1,-85432,     0            , "t=8.47h",_
 1484, 47, 101, 0,4.5,       1,-81334,     0            , "t=11.1m",_
 1485, 47, 101, 1,0.5,      -1,-81060,     0.2741       , "t=3.10s",_
 1486, 48, 101, 0,2.5,       1,-75836.5,   0            , "t=1.36m",_
 1487, 49, 101, 0,4.5,       1,-68610,     0            , "t=15.1s",_
 1488, 49, 101, 1,0.5,      -1,-68060,     0.55         , "t=10#s",_
 1489, 50, 101, 0,3.5,       1,-60310,     0            , "t=1.97s",_
 1490, 37, 102, 0,4,         1,-37710,     0            , "t=37ms",_
 1491, 38, 102, 0,0,         1,-52160,     0            , "t=69ms",_
 1492, 39, 102, 0,2,        -1,-61173,     0            , "t=298ms",_
 1493, 39, 102, 1,0,         0,-60970,     0.2          , "t=360ms",_
 1494, 40, 102, 0,0,         1,-71588,     0            , "t=2.9s",_
 1495, 41, 102, 0,4,         1,-76304.5,   0            , "t=4.3s",_
 1496, 41, 102, 1,1,         1,-76210,     0.094        , "t=1.3s",_
 1497, 42, 102, 0,0,         1,-83566,     0            , "t=11.3m",_
 1498, 43, 102, 0,1,         1,-84573,     0            , "t=5.28s",_
 1499, 43, 102, 1,4,         0,-84553,     0.02         , "t=4.35m",_
 1500, 44, 102, 0,0,         1,-89106.4,   0            , "t=stbl",_
 1501, 45, 102, 0,1,        -1,-86783,     0            , "t=207.0d",_
 1502, 45, 102, 1,6,         1,-86642,     0.14073      , "t=3.742y",_
 1503, 46, 102, 0,0,         1,-87903.2,   0            , "t=stbl",_
 1504, 47, 102, 0,5,         1,-82247,     0            , "t=12.9m",_
 1505, 47, 102, 1,2,         1,-82238,     0.0094       , "t=7.7m",_
 1506, 48, 102, 0,0,         1,-79659.7,   0            , "t=5.5m",_
 1507, 49, 102, 0,6,         1,-70695,     0            , "t=23.3s",_
 1508, 50, 102, 0,0,         1,-64930,     0            , "t=3.8s",_
 1509, 50, 102, 1,6,         1,-62910,     2.017        , "t=367ns",_
 1510, 37, 103, 0,1.5,       1,-33610,     0            , "t=26ms",_
 1511, 38, 103, 0,2.5,       1,-47420,     0            , "t=53ms",_
 1512, 39, 103, 0,2.5,       1,-58458,     0            , "t=239ms",_
 1513, 40, 103, 0,2.5,      -1,-67815,     0            , "t=1.38s",_
 1514, 41, 103, 0,2.5,       1,-75029,     0            , "t=1.5s",_
 1515, 42, 103, 0,1.5,       1,-80961,     0            , "t=67.5s",_
 1516, 43, 103, 0,2.5,       1,-84604,     0            , "t=54.2s",_
 1517, 44, 103, 0,1.5,       1,-87267.2,   0            , "t=39.247d",_
 1518, 44, 103, 1,5.5,      -1,-87029.0,   0.2382       , "t=1.69ms",_
 1519, 45, 103, 0,0.5,      -1,-88031.7,   0            , "t=stbl",_
 1520, 45, 103, 1,3.5,       1,-87991.9,   0.039753     , "t=56.114m",_
 1521, 46, 103, 0,2.5,       1,-87457.2,   0            , "t=16.991d",_
 1522, 47, 103, 0,3.5,       1,-84803,     0            , "t=65.7m",_
 1523, 47, 103, 1,0.5,      -1,-84669,     0.13445      , "t=5.7s",_
 1524, 48, 103, 0,2.5,       1,-80651.6,   0            , "t=7.3m",_
 1525, 49, 103, 0,4.5,       1,-74633,     0            , "t=60s",_
 1526, 49, 103, 1,0.5,      -1,-74001,     0.6317       , "t=34s",_
 1527, 50, 103, 0,2.5,       1,-66970,     0            , "t=7.0s",_
 1528, 51, 103, 0,2.5,       1,-56180,     0            , "t=?",_
 1529, 38, 104, 0,0,         1,-44110,     0            , "t=50.6ms",_
 1530, 39, 104, 0,-77.777,   0,-54060,     0            , "t=197ms",_
 1531, 40, 104, 0,0,         1,-65724,     0            , "t=920ms",_
 1532, 41, 104, 0,1,         1,-71819.0,   0            , "t=4.9s",_
 1533, 41, 104, 1,0,         0,-71600,     0.21         , "t=940ms",_
 1534, 42, 104, 0,0,         1,-80350,     0            , "t=60s",_
 1535, 43, 104, 0,3,         1,-82503,     0            , "t=18.3m",_
 1536, 43, 104, 1,5,         1,-82433,     0.0697       , "t=3.5us",_
 1537, 43, 104, 2,-77.777,   1,-82397,     0.1061       , "t=400ns",_
 1538, 44, 104, 0,0,         1,-88095.7,   0            , "t=stbl",_
 1539, 45, 104, 0,1,         1,-86959.3,   0            , "t=42.3s",_
 1540, 45, 104, 1,5,         1,-86830.3,   0.1289679    , "t=4.34m",_
 1541, 46, 104, 0,0,         1,-89395.1,   0            , "t=stbl",_
 1542, 47, 104, 0,5,         1,-85116,     0            , "t=69.2m",_
 1543, 47, 104, 1,2,         1,-85109,     0.0069       , "t=33.5m",_
 1544, 48, 104, 0,0,         1,-83968.4,   0            , "t=57.7m",_
 1545, 49, 104, 0,5,         1,-76183,     0            , "t=1.80m",_
 1546, 49, 104, 1,3,         1,-76090,     0.09348      , "t=15.7s",_
 1547, 50, 104, 0,0,         1,-71627,     0            , "t=20.8s",_
 1548, 51, 104, 0,-77.777,   0,-59170,     0            , "t=470ms",_
 1549, 38, 105, 0,2.5,       1,-38610,     0            , "t=39ms",_
 1550, 39, 105, 0,2.5,       1,-51270,     0            , "t=95ms",_
 1551, 40, 105, 0,2.5,       1,-61465,     0            , "t=670ms",_
 1552, 41, 105, 0,2.5,       1,-69916,     0            , "t=2.95s",_
 1553, 42, 105, 0,2.5,      -1,-77337,     0            , "t=35.6s",_
 1554, 43, 105, 0,1.5,      -1,-82290,     0            , "t=7.6m",_
 1555, 44, 105, 0,1.5,       1,-85934.5,   0            , "t=4.44h",_
 1556, 44, 105, 1,2.5,       1,-85913.9,   0.02061      , "t=340ns",_
 1557, 45, 105, 0,3.5,       1,-87851.2,   0            , "t=35.357h",_
 1558, 45, 105, 1,0.5,      -1,-87721.4,   0.129782     , "t=42.9s",_
 1559, 46, 105, 0,2.5,       1,-88417.9,   0            , "t=stbl",_
 1560, 46, 105, 1,5.5,      -1,-87928.8,   0.48914      , "t=36.1us",_
 1561, 47, 105, 0,0.5,      -1,-87071,     0            , "t=41.29d",_
 1562, 47, 105, 1,3.5,       1,-87046,     0.025479     , "t=7.23m",_
 1563, 48, 105, 0,2.5,       1,-84333.8,   0            , "t=55.5m",_
 1564, 49, 105, 0,4.5,       1,-79641,     0            , "t=5.07m",_
 1565, 49, 105, 1,0.5,      -1,-78967,     0.67408      , "t=48s",_
 1566, 50, 105, 0,2.5,       1,-73338,     0            , "t=34s",_
 1567, 51, 105, 0,2.5,       1,-64015,     0            , "t=1.12s",_
 1568, 52, 105, 0,3.5,       1,-52810,     0            , "t=633ns",_
 1569, 38, 106, 0,0,         1,-34790,     0            , "t=21ms",_
 1570, 39, 106, 0,2,         1,-46050,     0            , "t=74ms",_
 1571, 40, 106, 0,0,         1,-58550,     0            , "t=178.6ms",_
 1572, 41, 106, 0,1,        -1,-66203,     0            , "t=1050ms",_
 1573, 41, 106, 1,3,         1,-65998,     0.2048       , "t=800ns",_
 1574, 41, 106, 2,3,         1,-65998,     0.2048       , "t=849ns",_
 1575, 42, 106, 0,0,         1,-76135,     0            , "t=8.73s",_
 1576, 43, 106, 0,1,         0,-79776,     0            , "t=35.6s",_
 1577, 44, 106, 0,0,         1,-86323,     0            , "t=371.8d",_
 1578, 45, 106, 0,1,         1,-86363,     0            , "t=30.07s",_
 1579, 45, 106, 1,6,         1,-86231,     0.132        , "t=131m",_
 1580, 46, 106, 0,0,         1,-89907.5,   0            , "t=stbl",_
 1581, 47, 106, 0,1,         1,-86942,     0            , "t=23.96m",_
 1582, 47, 106, 1,6,         1,-86852,     0.08966      , "t=8.28d",_
 1583, 48, 106, 0,0,         1,-87132.1,   0            , "t=stbl",_
 1584, 49, 106, 0,7,         1,-80608,     0            , "t=6.2m",_
 1585, 49, 106, 1,2,         1,-80579,     0.0286       , "t=5.2m",_
 1586, 50, 106, 0,0,         1,-77354,     0            , "t=1.92m",_
 1587, 51, 106, 0,2,         1,-66473,     0            , "t=600ms",_
 1588, 51, 106, 1,4,         1,-66370,     0.1035       , "t=226ns",_
 1589, 52, 106, 0,0,         1,-58220,     0            , "t=78us",_
 1590, 38, 107, 0,0.5,       1,-28900,     0            , "t=10#ms",_
 1591, 39, 107, 0,2.5,       1,-42360,     0            , "t=33.5ms",_
 1592, 40, 107, 0,2.5,       1,-54380,     0            , "t=145.7ms",_
 1593, 41, 107, 0,2.5,       1,-63724,     0            , "t=289ms",_
 1594, 42, 107, 0,2.5,       1,-72552,     0            , "t=3.5s",_
 1595, 42, 107, 1,0.5,       1,-72487,     0.0654       , "t=420ns",_
 1596, 43, 107, 0,1.5,      -1,-78750,     0            , "t=21.2s",_
 1597, 43, 107, 1,0.5,       1,-78720,     0.0301       , "t=3.85us",_
 1598, 43, 107, 2,2.5,       1,-78684,     0.06572      , "t=184ns",_
 1599, 44, 107, 0,2.5,       1,-83863,     0            , "t=3.75m",_
 1600, 45, 107, 0,3.5,       1,-86864,     0            , "t=21.7m",_
 1601, 45, 107, 1,0.5,      -1,-86596,     0.26836      , "t=&gt;10us",_
 1602, 46, 107, 0,2.5,       1,-88372.6,   0            , "t=6.5My",_
 1603, 46, 107, 1,0.5,       1,-88256.9,   0.11574      , "t=850ns",_
 1604, 46, 107, 2,5.5,      -1,-88158.0,   0.2146       , "t=21.3s",_
 1605, 47, 107, 0,0.5,      -1,-88406.7,   0            , "t=stbl",_
 1606, 47, 107, 1,3.5,       1,-88313.6,   0.093125     , "t=44.3s",_
 1607, 48, 107, 0,2.5,       1,-86990.3,   0            , "t=6.50h",_
 1608, 49, 107, 0,4.5,       1,-83564,     0            , "t=32.4m",_
 1609, 49, 107, 1,0.5,      -1,-82886,     0.6785       , "t=50.4s",_
 1610, 50, 107, 0,2.5,       1,-78512,     0            , "t=2.90m",_
 1611, 51, 107, 0,2.5,       1,-70653,     0            , "t=4.0s",_
 1612, 52, 107, 0,2.5,       1,-60540,     0            , "t=3.1ms",_
 1613, 53, 107, 0,2.5,       1,-49430,     0            , "t=20#us",_
 1614, 39, 108, 0,-77.777,   0,-37300,     0            , "t=30ms",_
 1615, 40, 108, 0,0,         1,-51350,     0            , "t=78.5ms",_
 1616, 40, 108, 1,6,         1,-49280,     2.0745       , "t=540ns",_
 1617, 41, 108, 0,2,         1,-59546,     0            , "t=198ms",_
 1618, 41, 108, 1,4,        -1,-59379,     0.1666       , "t=109ns",_
 1619, 42, 108, 0,0,         1,-70756,     0            , "t=1.105s",_
 1620, 43, 108, 0,2,         1,-75923,     0            , "t=5.17s",_
 1621, 44, 108, 0,0,         1,-83661,     0            , "t=4.55m",_
 1622, 45, 108, 0,1,         1,-85032,     0            , "t=16.8s",_
 1623, 45, 108, 1,5,         1,-84917,     0.115        , "t=6.0m",_
 1624, 46, 108, 0,0,         1,-89524.2,   0            , "t=stbl",_
 1625, 47, 108, 0,1,         1,-87606.8,   0            , "t=2.382m",_
 1626, 47, 108, 1,6,         1,-87497.3,   0.109466     , "t=438y",_
 1627, 48, 108, 0,0,         1,-89252.4,   0            , "t=stbl",_
 1628, 49, 108, 0,7,         1,-84120,     0            , "t=58.0m",_
 1629, 49, 108, 1,2,         1,-84090,     0.02975      , "t=39.6m",_
 1630, 50, 108, 0,0,         1,-82070,     0            , "t=10.30m",_
 1631, 51, 108, 0,4,         1,-72445,     0            , "t=7.4s",_
 1632, 52, 108, 0,0,         1,-65782,     0            , "t=2.1s",_
 1633, 53, 108, 0,1,         1,-52650,     0            , "t=36ms",_
 1634, 39, 109, 0,2.5,       1,-33200,     0            , "t=25ms",_
 1635, 40, 109, 0,0.5,       1,-46190,     0            , "t=56ms",_
 1636, 41, 109, 0,2.5,       1,-56690,     0            , "t=106.9ms",_
 1637, 41, 109, 1,-77.777,   0,-56380,     0.3122       , "t=115ns",_
 1638, 42, 109, 0,2.5,       1,-66666,     0            , "t=700ms",_
 1639, 42, 109, 1,0.5,       1,-66596,     0.0697       , "t=210ns",_
 1640, 43, 109, 0,2.5,       1,-74283,     0            , "t=1.14s",_
 1641, 44, 109, 0,2.5,       1,-80738,     0            , "t=34.5s",_
 1642, 44, 109, 1,2.5,      -1,-80642,     0.0962       , "t=680ns",_
 1643, 45, 109, 0,3.5,       1,-84999,     0            , "t=80s",_
 1644, 45, 109, 1,1.5,       1,-84773,     0.225974     , "t=1.66us",_
 1645, 46, 109, 0,2.5,       1,-87606.5,   0            , "t=13.7012h",_
 1646, 46, 109, 1,0.5,       1,-87493.1,   0.1134       , "t=380ns",_
 1647, 46, 109, 2,5.5,      -1,-87417.5,   0.18899      , "t=4.696m",_
 1648, 47, 109, 0,0.5,      -1,-88719.4,   0            , "t=stbl",_
 1649, 47, 109, 1,3.5,       1,-88631.4,   0.0880341    , "t=39.6s",_
 1650, 48, 109, 0,2.5,       1,-88504.3,   0            , "t=461.6d",_
 1651, 48, 109, 1,0.5,       1,-88444.8,   0.05949      , "t=12us",_
 1652, 48, 109, 2,5.5,      -1,-88040.8,   0.4635       , "t=10.9us",_
 1653, 49, 109, 0,4.5,       1,-86490,     0            , "t=4.167h",_
 1654, 49, 109, 1,0.5,      -1,-85840,     0.6501       , "t=1.34m",_
 1655, 49, 109, 2,9.5,       1,-84388,     2.1018       , "t=209ms",_
 1656, 50, 109, 0,2.5,       1,-82630,     0            , "t=18.0m",_
 1657, 51, 109, 0,2.5,       1,-76251,     0            , "t=17.0s",_
 1658, 52, 109, 0,2.5,       1,-67715,     0            , "t=4.6s",_
 1659, 53, 109, 0,0.5,       1,-57672,     0            , "t=103us",_
 1660, 54, 109, 0,3.5,       1,-46170,     0            , "t=13ms",_
 1661, 40, 110, 0,0,         1,-42890,     0            , "t=37.5ms",_
 1662, 41, 110, 0,5,         1,-52310,     0            , "t=82ms",_
 1663, 42, 110, 0,0,         1,-64543,     0            , "t=292ms",_
 1664, 43, 110, 0,2,         1,-71035,     0            , "t=900ms",_
 1665, 44, 110, 0,0,         1,-80073,     0            , "t=12.04s",_
 1666, 45, 110, 0,1,         1,-82829,     0            , "t=3.35s",_
 1667, 45, 110, 1,6,         1,-82610,     0.22         , "t=28.5s",_
 1668, 46, 110, 0,0,         1,-88330.9,   0            , "t=stbl",_
 1669, 47, 110, 0,1,         1,-87457.3,   0            , "t=24.56s",_
 1670, 47, 110, 1,2,        -1,-87456.2,   0.001112     , "t=660ns",_
 1671, 47, 110, 2,6,         1,-87339.7,   0.11759      , "t=249.83d",_
 1672, 48, 110, 0,0,         1,-90348.0,   0            , "t=stbl",_
 1673, 49, 110, 0,7,         1,-86470,     0            , "t=4.92h",_
 1674, 49, 110, 1,2,         1,-86408,     0.06208      , "t=69.1m",_
 1675, 50, 110, 0,0,         1,-85842,     0            , "t=4.154h",_
 1676, 51, 110, 0,3,         1,-77450,     0            , "t=23.6s",_
 1677, 52, 110, 0,0,         1,-72230,     0            , "t=18.6s",_
 1678, 53, 110, 0,1,         1,-60460,     0            , "t=664ms",_
 1679, 54, 110, 0,0,         1,-51920,     0            , "t=93ms",_
 1680, 40, 111, 0,1.5,       1,-37560,     0            , "t=24.0ms",_
 1681, 41, 111, 0,2.5,       1,-48880,     0            , "t=54ms",_
 1682, 42, 111, 0,0.5,       1,-59940,     0            , "t=193.6ms",_
 1683, 42, 111, 1,3.5,      -1,-59840,     0.1          , "t=~200ms",_
 1684, 43, 111, 0,2.5,       1,-69025,     0            , "t=350ms",_
 1685, 44, 111, 0,2.5,       1,-76785,     0            , "t=2.12s",_
 1686, 45, 111, 0,3.5,       1,-82304,     0            , "t=11s",_
 1687, 46, 111, 0,2.5,       1,-85985.9,   0            , "t=23.4m",_
 1688, 46, 111, 1,5.5,      -1,-85813.7,   0.17218      , "t=5.5h",_
 1689, 47, 111, 0,0.5,      -1,-88215.4,   0            , "t=7.433d",_
 1690, 47, 111, 1,3.5,       1,-88155.6,   0.05982      , "t=64.8s",_
 1691, 48, 111, 0,0.5,       1,-89252.2,   0            , "t=stbl",_
 1692, 48, 111, 1,5.5,      -1,-88856.0,   0.396214     , "t=48.50m",_
 1693, 49, 111, 0,4.5,       1,-88392,     0            , "t=2.8063d",_
 1694, 49, 111, 1,0.5,      -1,-87855,     0.53699      , "t=7.7m",_
 1695, 50, 111, 0,3.5,       1,-85939,     0            , "t=35.3m",_
 1696, 50, 111, 1,0.5,       1,-85684,     0.25471      , "t=12.5us",_
 1697, 51, 111, 0,2.5,       1,-80837,     0            , "t=75s",_
 1698, 52, 111, 0,2.5,       1,-73587,     0            , "t=26.2s",_
 1699, 53, 111, 0,2.5,       1,-64954,     0            , "t=2.5s",_
 1700, 54, 111, 0,2.5,       1,-54400,     0            , "t=740ms",_
 1701, 55, 111, 0,1.5,       1,-42820,     0            , "t=1#us",_
 1702, 40, 112, 0,0,         1,-33810,     0            , "t=43ms",_
 1703, 41, 112, 0,2,         1,-44270,     0            , "t=38ms",_
 1704, 42, 112, 0,0,         1,-57460,     0            , "t=125ms",_
 1705, 43, 112, 0,2,         1,-65259,     0            , "t=323ms",_
 1706, 43, 112, 1,-77.777,   0,-64907,     0.3523       , "t=150ns",_
 1707, 44, 112, 0,0,         1,-75631,     0            , "t=1.75s",_
 1708, 45, 112, 0,1,         1,-79730,     0            , "t=3.4s",_
 1709, 45, 112, 1,6,         1,-79390,     0.34         , "t=6.73s",_
 1710, 46, 112, 0,0,         1,-86322,     0            , "t=21.04h",_
 1711, 47, 112, 0,2,        -1,-86583.7,   0            , "t=3.130h",_
 1712, 48, 112, 0,0,         1,-90574.86,  0            , "t=stbl",_
 1713, 49, 112, 0,1,         1,-87990,     0            , "t=14.88m",_
 1714, 49, 112, 1,4,         1,-87833,     0.156592     , "t=20.67m",_
 1715, 49, 112, 2,7,         1,-87639,     0.3508       , "t=690ns",_
 1716, 50, 112, 0,0,         1,-88655.06,  0            , "t=stbl",_
 1717, 51, 112, 0,3,         1,-81599,     0            , "t=53.5s",_
 1718, 51, 112, 1,8,        -1,-80773,     0.8259       , "t=536ns",_
 1719, 52, 112, 0,0,         1,-77568,     0            , "t=2.0m",_
 1720, 53, 112, 0,1,         1,-67063,     0            , "t=3.34s",_
 1721, 54, 112, 0,0,         1,-60026,     0            , "t=2.7s",_
 1722, 55, 112, 0,1,         1,-46290,     0            , "t=490us",_
 1723, 41, 113, 0,2.5,       1,-40510,     0            , "t=32ms",_
 1724, 42, 113, 0,1.5,       1,-52490,     0            , "t=80ms",_
 1725, 43, 113, 0,2.5,       1,-62812,     0            , "t=152ms",_
 1726, 43, 113, 1,2.5,      -1,-62698,     0.1144       , "t=527ns",_
 1727, 44, 113, 0,0.5,       1,-71870,     0            , "t=800ms",_
 1728, 44, 113, 1,3.5,      -1,-71740,     0.13         , "t=510ms",_
 1729, 45, 113, 0,3.5,       1,-78768,     0            , "t=2.80s",_
 1730, 46, 113, 0,2.5,       1,-83591,     0            , "t=93s",_
 1731, 46, 113, 1,4.5,      -1,-83510,     0.0811       , "t=300ms",_
 1732, 47, 113, 0,0.5,      -1,-87027,     0            , "t=5.37h",_
 1733, 47, 113, 1,3.5,       1,-86984,     0.0435       , "t=68.7s",_
 1734, 48, 113, 0,0.5,       1,-89043.28,  0            , "t=8.04Py",_
 1735, 48, 113, 1,5.5,      -1,-88779.74,  0.26354      , "t=13.89y",_
 1736, 49, 113, 0,4.5,       1,-89367.12,  0            , "t=stbl",_
 1737, 49, 113, 1,0.5,      -1,-88975.42,  0.391699     , "t=1.6579h",_
 1738, 50, 113, 0,0.5,       1,-88328.1,   0            , "t=115.09d",_
 1739, 50, 113, 1,3.5,       1,-88250.7,   0.077389     , "t=21.4m",_
 1740, 51, 113, 0,2.5,       1,-84417,     0            , "t=6.67m",_
 1741, 52, 113, 0,3.5,       1,-78347,     0            , "t=1.7m",_
 1742, 53, 113, 0,2.5,       1,-71120,     0            , "t=6.6s",_
 1743, 54, 113, 0,2.5,       1,-62204,     0            , "t=2.74s",_
 1744, 54, 113, 1,5.5,      -1,-61800,     0.4036       , "t=6.9us",_
 1745, 55, 113, 0,1.5,       1,-51765,     0            , "t=17.7us",_
 1746, 56, 113, 0,2.5,       1,-39780,     0            , "t=100#ms",_
 1747, 41, 114, 0,-77.777,   0,-35390,     0            , "t=17ms",_
 1748, 42, 114, 0,0,         1,-49810,     0            , "t=58ms",_
 1749, 43, 114, 0,1,         1,-58600,     0            , "t=90ms",_
 1750, 43, 114, 1,4,         0,-58437,     0.16         , "t=100ms",_
 1751, 44, 114, 0,0,         1,-70222,     0            , "t=540ms",_
 1752, 45, 114, 0,1,         1,-75710,     0            , "t=1.85s",_
 1753, 45, 114, 1,7,        -1,-75510,     0.2          , "t=1.85s",_
 1754, 46, 114, 0,0,         1,-83491,     0            , "t=2.42m",_
 1755, 47, 114, 0,1,         1,-84931,     0            , "t=4.6s",_
 1756, 47, 114, 1,0,         1,-84732,     0.199        , "t=1.50ms",_
 1757, 48, 114, 0,0,         1,-90014.93,  0            , "t=stbl",_
 1758, 49, 114, 0,1,         1,-88569.8,   0            , "t=71.9s",_
 1759, 49, 114, 1,5,         1,-88379.5,   0.1902682    , "t=49.51d",_
 1760, 49, 114, 2,8,        -1,-88067.9,   0.501948     , "t=43.1ms",_
 1761, 50, 114, 0,0,         1,-90559.723,               0            , "t=stbl",_
 1762, 50, 114, 1,7,        -1,-87472.35,  3.08737      , "t=733ns",_
 1763, 51, 114, 0,3,         1,-84497,     0            , "t=3.49m",_
 1764, 51, 114, 1,8,        -1,-84002,     0.4955       , "t=219us",_
 1765, 52, 114, 0,0,         1,-81889,     0            , "t=15.2m",_
 1766, 53, 114, 0,1,         1,-72800,     0            , "t=2.1s",_
 1767, 53, 114, 1,7,         0,-72530,     0.2659       , "t=6.2s",_
 1768, 54, 114, 0,0,         1,-67086,     0            , "t=10.0s",_
 1769, 55, 114, 0,1,         1,-54680,     0            , "t=570ms",_
 1770, 56, 114, 0,0,         1,-45910,     0            , "t=460ms",_
 1771, 41, 115, 0,2.5,       1,-31350,     0            , "t=23ms",_
 1772, 42, 115, 0,1.5,       1,-44750,     0            , "t=45.5ms",_
 1773, 43, 115, 0,2.5,       1,-56320,     0            , "t=78ms",_
 1774, 44, 115, 0,1.5,       1,-66190,     0            , "t=318ms",_
 1775, 44, 115, 1,4.5,      -1,-65940,     0.25         , "t=76ms",_
 1776, 45, 115, 0,3.5,       1,-74230,     0            , "t=990ms",_
 1777, 46, 115, 0,0.5,       1,-80426,     0            , "t=25s",_
 1778, 46, 115, 1,3.5,      -1,-80337,     0.08921      , "t=50s",_
 1779, 47, 115, 0,0.5,      -1,-84983,     0            , "t=20.0m",_
 1780, 47, 115, 1,3.5,       1,-84942,     0.04116      , "t=18.0s",_
 1781, 48, 115, 0,0.5,       1,-88084.5,   0            , "t=53.46h",_
 1782, 48, 115, 1,5.5,      -1,-87903.5,   0.181        , "t=44.56d",_
 1783, 49, 115, 0,4.5,       1,-89536.346,               0            , "t=441Ty",_
 1784, 49, 115, 1,0.5,      -1,-89200.102,               0.336244     , "t=4.486h",_
 1785, 50, 115, 0,0.5,       1,-90033.835,               0            , "t=stbl",_
 1786, 50, 115, 1,3.5,       1,-89421.03,  0.61281      , "t=3.26us",_
 1787, 50, 115, 2,5.5,      -1,-89320.20,  0.71364      , "t=159us",_
 1788, 51, 115, 0,2.5,       1,-87003,     0            , "t=32.1m",_
 1789, 51, 115, 1,9.5,      -1,-84207,     2.79626      , "t=159ns",_
 1790, 52, 115, 0,3.5,       1,-82063,     0            , "t=5.8m",_
 1791, 52, 115, 1,0.5,       1,-82053,     0.01         , "t=6.7m",_
 1792, 52, 115, 2,5.5,      -1,-81783,     0.28005      , "t=7.5us",_
 1793, 53, 115, 0,2.5,       1,-76338,     0            , "t=1.3m",_
 1794, 54, 115, 0,2.5,       1,-68657,     0            , "t=18s",_
 1795, 55, 115, 0,4.5,       1,-59700,     0            , "t=1.4s",_
 1796, 56, 115, 0,2.5,       1,-49020,     0            , "t=450ms",_
 1797, 42, 116, 0,0,         1,-41500,     0            , "t=32ms",_
 1798, 43, 116, 0,2,         1,-51460,     0            , "t=57ms",_
 1799, 44, 116, 0,0,         1,-64069,     0            , "t=204ms",_
 1800, 45, 116, 0,1,         1,-70740,     0            , "t=685ms",_
 1801, 45, 116, 1,6,        -1,-70540,     0.2          , "t=570ms",_
 1802, 46, 116, 0,0,         1,-79832,     0            , "t=11.8s",_
 1803, 47, 116, 0,0,        -1,-82543,     0            , "t=3.83m",_
 1804, 47, 116, 1,3,         1,-82495,     0.0479       , "t=20s",_
 1805, 47, 116, 2,6,        -1,-82413,     0.1298       , "t=9.3s",_
 1806, 48, 116, 0,0,         1,-88712.48,  0            , "t=28.7Ey",_
 1807, 49, 116, 0,1,         1,-88249.75,  0            , "t=14.10s",_
 1808, 49, 116, 1,5,         1,-88122.48,  0.127267     , "t=54.29m",_
 1809, 49, 116, 2,8,        -1,-87960.09,  0.28966      , "t=2.18s",_
 1810, 50, 116, 0,0,         1,-91525.97,  0            , "t=stbl",_
 1811, 50, 116, 1,5,        -1,-89160.00,  2.365975     , "t=348ns",_
 1812, 50, 116, 2,10,        1,-87978.81,  3.54716      , "t=833ns",_
 1813, 51, 116, 0,3,         1,-86822,     0            , "t=15.8m",_
 1814, 51, 116, 1,1,         1,-86728,     0.09399      , "t=194ns",_
 1815, 51, 116, 2,8,        -1,-86440,     0.39         , "t=60.3m",_
 1816, 52, 116, 0,0,         1,-85269,     0            , "t=2.49h",_
 1817, 53, 116, 0,1,         1,-77490,     0            , "t=2.91s",_
 1818, 53, 116, 1,7,        -1,-77060,     0.4304       , "t=3.27us",_
 1819, 54, 116, 0,0,         1,-73047,     0            , "t=59s",_
 1820, 55, 116, 0,1,         1,-62040,     0            , "t=700ms",_
 1821, 55, 116, 1,4,         1,-61940,     0.1          , "t=3.85s",_
 1822, 56, 116, 0,0,         1,-54580,     0            , "t=1.3s",_
 1823, 57, 116, 0,-77.777,   0,-40650,     0            , "t=10#ms",_
 1824, 42, 117, 0,1.5,       1,-36170,     0            , "t=22ms",_
 1825, 43, 117, 0,2.5,       1,-48380,     0            , "t=44.5ms",_
 1826, 44, 117, 0,1.5,       1,-59490,     0            , "t=151ms",_
 1827, 44, 117, 1,-77.777,   0,-59310,     0.185        , "t=2.49us",_
 1828, 45, 117, 0,3.5,       1,-68897,     0            , "t=421ms",_
 1829, 46, 117, 0,1.5,       1,-76424,     0            , "t=4.3s",_
 1830, 46, 117, 1,4.5,      -1,-76221,     0.2033       , "t=19.1ms",_
 1831, 47, 117, 0,0.5,      -1,-82182,     0            , "t=73.6s",_
 1832, 47, 117, 1,3.5,       1,-82153,     0.0286       , "t=5.34s",_
 1833, 48, 117, 0,0.5,       1,-86418.4,   0            , "t=2.49h",_
 1834, 48, 117, 1,5.5,      -1,-86282.0,   0.1364       , "t=3.36h",_
 1835, 49, 117, 0,4.5,       1,-88943,     0            , "t=43.2m",_
 1836, 49, 117, 1,0.5,      -1,-88628,     0.315303     , "t=116.2m",_
 1837, 50, 117, 0,0.5,       1,-90397.8,   0            , "t=stbl",_
 1838, 50, 117, 1,5.5,      -1,-90083.2,   0.31458      , "t=14.00d",_
 1839, 50, 117, 2,9.5,       1,-87991.4,   2.4064       , "t=1.75us",_
 1840, 51, 117, 0,2.5,       1,-88640,     0            , "t=2.80h",_
 1841, 51, 117, 1,12.5,      1,-85509,     3.13076      , "t=355us",_
 1842, 51, 117, 2,11.5,     -1,-85409,     3.2307       , "t=290ns",_
 1843, 52, 117, 0,0.5,       1,-85095,     0            , "t=62m",_
 1844, 52, 117, 1,5.5,      -1,-84799,     0.2961       , "t=103ms",_
 1845, 53, 117, 0,2.5,       1,-80436,     0            , "t=2.22m",_
 1846, 54, 117, 0,2.5,       1,-74185,     0            , "t=61s",_
 1847, 55, 117, 0,4.5,       1,-66490,     0            , "t=8.4s",_
 1848, 55, 117, 1,1.5,       1,-66340,     0.15         , "t=6.5s",_
 1849, 55, 117, 0,0,         0,-66440,     0.05         , "t=R= ?",_
 1850, 56, 117, 0,1.5,       1,-57460,     0            , "t=1.75s",_
 1851, 57, 117, 0,1.5,       1,-46470,     0            , "t=21.7ms",_
 1852, 42, 118, 0,0,         1,-32630,     0            , "t=21ms",_
 1853, 43, 118, 0,2,         1,-43790,     0            , "t=30ms",_
 1854, 44, 118, 0,0,         1,-57260,     0            , "t=99ms",_
 1855, 45, 118, 0,4,        -1,-64887,     0            , "t=284ms",_
 1856, 46, 118, 0,0,         1,-75388.7,   0            , "t=1.9s",_
 1857, 47, 118, 0,1,        -1,-79553.8,   0            , "t=3.76s",_
 1858, 47, 118, 1,0,        -1,-79508.0,   0.04579      , "t=~0.1us",_
 1859, 47, 118, 2,4,         1,-79426.2,   0.12763      , "t=2.0s",_
 1860, 48, 118, 0,0,         1,-86702,     0            , "t=50.3m",_
 1861, 49, 118, 0,1,         1,-87228,     0            , "t=5.0s",_
 1862, 49, 118, 1,5,         1,-87130,     0.1          , "t=4.364m",_
 1863, 49, 118, 2,8,        -1,-86990,     0.24         , "t=8.5s",_
 1864, 50, 118, 0,0,         1,-91652.9,   0            , "t=stbl",_
 1865, 50, 118, 1,7,        -1,-89078.0,   2.57491      , "t=230ns",_
 1866, 50, 118, 2,10,        1,-88544.8,   3.10806      , "t=2.52us",_
 1867, 51, 118, 0,1,         1,-87996,     0            , "t=3.6m",_
 1868, 51, 118, 1,3,         1,-87945,     0.050814     , "t=20.6us",_
 1869, 51, 118, 2,8,        -1,-87746,     0.25         , "t=5.00h",_
 1870, 52, 118, 0,0,         1,-87697,     0            , "t=6.00d",_
 1871, 53, 118, 0,2,        -1,-80971,     0            , "t=13.7m",_
 1872, 53, 118, 1,7,        -1,-80782,     0.1888       , "t=8.5m",_
 1873, 54, 118, 0,0,         1,-78079,     0            , "t=3.8m",_
 1874, 55, 118, 0,2,         0,-68409,     0            , "t=14s",_
 1875, 55, 118, 1,7,        -1,-68310,     0.1          , "t=17s",_
 1876, 55, 118, 0,0,         0,-68404,     0.005        , "t=R&lt;0.1",_
 1877, 56, 118, 0,0,         1,-62350,     0            , "t=5.2s",_
 1878, 57, 118, 0,-77.777,   0,-49560,     0            , "t=200#ms",_
 1879, 43, 119, 0,2.5,       1,-40370,     0            , "t=22ms",_
 1880, 44, 119, 0,1.5,       1,-52560,     0            , "t=69.5ms",_
 1881, 44, 119, 1,-77.777,   0,-52330,     0.2271       , "t=384ns",_
 1882, 45, 119, 0,3.5,       1,-62823,     0            , "t=190ms",_
 1883, 46, 119, 0,1.5,       1,-71408,     0            , "t=920ms",_
 1884, 46, 119, 1,5.5,      -1,-71110,     0.3          , "t=3#ms",_
 1885, 47, 119, 0,0.5,      -1,-78646,     0            , "t=6.0s",_
 1886, 47, 119, 1,3.5,       1,-78626,     0.02         , "t=2.1s",_
 1887, 48, 119, 0,0.5,       1,-83980,     0            , "t=2.69m",_
 1888, 48, 119, 1,5.5,      -1,-83830,     0.14654      , "t=2.20m",_
 1889, 49, 119, 0,4.5,       1,-87699,     0            , "t=2.4m",_
 1890, 49, 119, 1,0.5,      -1,-87388,     0.31137      , "t=18.0m",_
 1891, 49, 119, 2,1.5,       1,-87045,     0.65427      , "t=130ns",_
 1892, 50, 119, 0,0.5,       1,-90065.0,   0            , "t=stbl",_
 1893, 50, 119, 1,5.5,      -1,-89975.5,   0.089531     , "t=293.1d",_
 1894, 50, 119, 2,9.5,       1,-87938.0,   2.127        , "t=9.6us",_
 1895, 51, 119, 0,2.5,       1,-89474,     0            , "t=38.19h",_
 1896, 51, 119, 1,9.5,      -1,-86920,     2.5536       , "t=130ns",_
 1897, 51, 119, 2,13.5,      1,-86622,     2.852        , "t=850ms",_
 1898, 52, 119, 0,0.5,       1,-87181,     0            , "t=16.05h",_
 1899, 52, 119, 1,5.5,      -1,-86920,     0.26096      , "t=4.70d",_
 1900, 53, 119, 0,2.5,       1,-83766,     0            , "t=19.1m",_
 1901, 54, 119, 0,2.5,       1,-78794,     0            , "t=5.8m",_
 1902, 55, 119, 0,4.5,       1,-72305,     0            , "t=43.0s",_
 1903, 55, 119, 1,1.5,       1,-72260,     0.05         , "t=30.4s",_
 1904, 55, 119, 0,0,         0,-72289,     0.016        , "t=R=.5~~.25",_
 1905, 56, 119, 0,2.5,       1,-64590,     0            , "t=5.4s",_
 1906, 57, 119, 0,5.5,      -1,-54790,     0            , "t=1#s",_
 1907, 58, 119, 0,2.5,       1,-43940,     0            , "t=200#ms",_
 1908, 43, 120, 0,-77.777,   0,-35520,     0            , "t=21ms",_
 1909, 44, 120, 0,0,         1,-50010,     0            , "t=45ms",_
 1910, 45, 120, 0,-77.777,   0,-58820,     0            , "t=129.6ms",_
 1911, 45, 120, 1,-77.777,   0,-58660,     0.1572       , "t=295ns",_
 1912, 46, 120, 0,0,         1,-70280.1,   0            , "t=492ms",_
 1913, 47, 120, 0,4,         1,-75652,     0            , "t=1.52s",_
 1914, 47, 120, 1,0,        -1,-75650,     0            , "t=940ms",_
 1915, 47, 120, 2,7,        -1,-75449,     0.203        , "t=384ms",_
 1916, 48, 120, 0,0,         1,-83957,     0            , "t=50.80s",_
 1917, 49, 120, 0,1,         1,-85730,     0            , "t=3.08s",_
 1918, 49, 120, 1,5,         1,-85680,     0.05         , "t=46.2s",_
 1919, 49, 120, 2,8,        -1,-85430,     0.3          , "t=47.3s",_
 1920, 50, 120, 0,0,         1,-91098.4,   0            , "t=stbl",_
 1921, 50, 120, 1,7,        -1,-88616.8,   2.48163      , "t=11.8us",_
 1922, 50, 120, 2,10,        1,-88196.2,   2.90222      , "t=6.26us",_
 1923, 51, 120, 0,1,         1,-88418,     0            , "t=15.89m",_
 1924, 51, 120, 1,8,        -1,-88420,     0            , "t=5.76d",_
 1925, 51, 120, 2,3,         1,-88340,     0.07816      , "t=246ns",_
 1926, 52, 120, 0,0,         1,-89368,     0            , "t=stbl",_
 1927, 53, 120, 0,2,        -1,-83753,     0            , "t=81.67m",_
 1928, 53, 120, 1,3,         1,-83680,     0.07261      , "t=242ns",_
 1929, 53, 120, 2,7,        -1,-83433,     0.32         , "t=53m",_
 1930, 54, 120, 0,0,         1,-82172,     0            , "t=46.0m",_
 1931, 55, 120, 0,2,         1,-73889,     0            , "t=60.4s",_
 1932, 55, 120, 1,7,        -1,-73790,     0.1          , "t=57s",_
 1933, 55, 120, 0,0,         0,-73884,     0.005        , "t=R&lt;0.1",_
 1934, 56, 120, 0,0,         1,-68890,     0            , "t=24s",_
 1935, 57, 120, 0,-77.777,   0,-57570,     0            , "t=2.8s",_
 1936, 58, 120, 0,0,         1,-49600,     0            , "t=250#ms",_
 1937, 43, 121, 0,2.5,       1,-31780,     0            , "t=22ms",_
 1938, 44, 121, 0,1.5,       1,-45050,     0            , "t=29ms",_
 1939, 45, 121, 0,3.5,       1,-56250,     0            , "t=76ms",_
 1940, 46, 121, 0,1.5,       1,-66182,     0            , "t=290ms",_
 1941, 46, 121, 1,5.5,      -1,-66047,     0.1355       , "t=460ns",_
 1942, 46, 121, 2,-77.777,   0,-66022,     0.16         , "t=460ns",_
 1943, 47, 121, 0,3.5,       1,-74403,     0            , "t=780ms",_
 1944, 47, 121, 1,0.5,      -1,-74383,     0.02         , "t=200#ms",_
 1945, 48, 121, 0,1.5,       1,-81073.8,   0            , "t=13.5s",_
 1946, 48, 121, 1,5.5,      -1,-80858.9,   0.21486      , "t=8.3s",_
 1947, 49, 121, 0,4.5,       1,-85836,     0            , "t=23.1s",_
 1948, 49, 121, 1,0.5,      -1,-85522,     0.31368      , "t=3.88m",_
 1949, 49, 121, 2,10.5,     -1,-83388,     2.448        , "t=17us",_
 1950, 50, 121, 0,1.5,       1,-89197.3,   0            , "t=27.03h",_
 1951, 50, 121, 1,5.5,      -1,-89191.0,   0.00631      , "t=43.9y",_
 1952, 50, 121, 2,9.5,       1,-87198.5,   1.9988       , "t=5.3us",_
 1953, 51, 121, 0,2.5,       1,-89600.3,   0            , "t=stbl",_
 1954, 51, 121, 1,12.5,      1,-86859,     2.741        , "t=179us",_
 1955, 52, 121, 0,0.5,       1,-88546,     0            , "t=19.17d",_
 1956, 52, 121, 1,5.5,      -1,-88252,     0.293974     , "t=164.2d",_
 1957, 53, 121, 0,2.5,       1,-86251,     0            , "t=2.12h",_
 1958, 53, 121, 1,-77.777,   0,-83874,     2.3769       , "t=9.0us",_
 1959, 54, 121, 0,2.5,       1,-82481,     0            , "t=40.1m",_
 1960, 55, 121, 0,1.5,       1,-77102,     0            , "t=155s",_
 1961, 55, 121, 1,4.5,       1,-77034,     0.0685       , "t=122s",_
 1962, 55, 121, 0,0,         0,-77056,     0.046        , "t=R=2     1",_
 1963, 56, 121, 0,2.5,       1,-70740,     0            , "t=29.7s",_
 1964, 57, 121, 0,5.5,      -1,-62190,     0            , "t=5.3s",_
 1965, 58, 121, 0,2.5,       1,-52690,     0            , "t=1.1s",_
 1966, 59, 121, 0,1.5,       0,-41420,     0            , "t=12ms",_
 1967, 44, 122, 0,0,         1,-42150,     0            , "t=25ms",_
 1968, 45, 122, 0,-77.777,   0,-52080,     0            , "t=51ms",_
 1969, 45, 122, 1,-77.777,   0,-51810,     0.271        , "t=830ns",_
 1970, 46, 122, 0,0,         1,-64616,     0            , "t=195ms",_
 1971, 47, 122, 0,3,         1,-71110,     0            , "t=529ms",_
 1972, 47, 122, 1,1,        -1,-71030,     0.08         , "t=550ms",_
 1973, 47, 122, 2,9,        -1,-71030,     0.08         , "t=200ms",_
 1974, 48, 122, 0,0,         1,-80612.4,   0            , "t=5.24s",_
 1975, 49, 122, 0,1,         1,-83570,     0            , "t=1.5s",_
 1976, 49, 122, 1,5,         1,-83530,     0.04         , "t=10.3s",_
 1977, 49, 122, 2,8,        -1,-83290,     0.29         , "t=10.8s",_
 1978, 50, 122, 0,0,         1,-89941.3,   0            , "t=stbl",_
 1979, 50, 122, 1,7,        -1,-87532.3,   2.40903      , "t=7.5us",_
 1980, 50, 122, 2,10,        1,-87175.7,   2.7656       , "t=62us",_
 1981, 51, 122, 0,2,        -1,-88335.4,   0            , "t=2.7238d",_
 1982, 51, 122, 1,3,         1,-88274.0,   0.0614131    , "t=1.86us",_
 1983, 51, 122, 2,5,         1,-88197.9,   0.1374726    , "t=530us",_
 1984, 52, 122, 0,0,         1,-90314.5,   0            , "t=stbl",_
 1985, 53, 122, 0,1,         1,-86080,     0            , "t=3.63m",_
 1986, 53, 122, 1,7,        -1,-85765,     0.3149       , "t=190ns",_
 1987, 53, 122, 2,7,        -1,-85701,     0.3794       , "t=81us",_
 1988, 54, 122, 0,0,         1,-85355,     0            , "t=20.1h",_
 1989, 55, 122, 0,1,         1,-78140,     0            , "t=21.18s",_
 1990, 55, 122, 1,3,         1,-78090,     0.04587      , "t=&gt;1us",_
 1991, 55, 122, 2,8,        -1,-78005,     0.14         , "t=3.70m",_
 1992, 55, 122, 0,0,         0,-78130,     0.014        , "t=R=0.1 .05",_
 1993, 56, 122, 0,0,         1,-74609,     0            , "t=1.95m",_
 1994, 57, 122, 0,-77.777,   0,-64540,     0            , "t=8.6s",_
 1995, 58, 122, 0,0,         1,-57870,     0            , "t=2#s",_
 1996, 59, 122, 0,-77.777,   0,-44780,     0            , "t=500#ms",_
 1997, 44, 123, 0,1.5,       1,-37080,     0            , "t=19ms",_
 1998, 45, 123, 0,3.5,       1,-49360,     0            , "t=42ms",_
 1999, 46, 123, 0,1.5,       1,-60430,     0            , "t=108ms",_
 2000, 47, 123, 0,3.5,       1,-69550,     0            , "t=300ms",_
 2001, 47, 123, 1,0.5,      -1,-69530,     0.02         , "t=100#ms",_
 2002, 47, 123, 2,-77.777,   0,-68150,     1.4          , "t=202ns",_
 2003, 48, 123, 0,1.5,       1,-77414.2,   0            , "t=2.10s",_
 2004, 48, 123, 1,5.5,      -1,-77271,     0.143        , "t=1.82s",_
 2005, 49, 123, 0,4.5,       1,-83430,     0            , "t=6.17s",_
 2006, 49, 123, 1,0.5,      -1,-83103,     0.32721      , "t=47.4s",_
 2007, 49, 123, 2,8.5,      -1,-81352,     2.0781       , "t=1.4us",_
 2008, 50, 123, 0,5.5,      -1,-87816.2,   0            , "t=129.2d",_
 2009, 50, 123, 1,1.5,       1,-87791.6,   0.0246       , "t=40.06m",_
 2010, 50, 123, 2,9.5,       1,-85871.2,   1.945        , "t=7.4us",_
 2011, 51, 123, 0,3.5,       1,-89224.1,   0            , "t=stbl",_
 2012, 51, 123, 1,9.5,      -1,-86986.3,   2.2378       , "t=214ns",_
 2013, 51, 123, 2,11.5,      1,-86610.7,   2.6134       , "t=65us",_
 2014, 52, 123, 0,0.5,       1,-89172.2,   0            , "t=stbl",_
 2015, 52, 123, 1,5.5,      -1,-88924.7,   0.24747      , "t=119.2d",_
 2016, 53, 123, 0,2.5,       1,-87944,     0            , "t=13.2235h",_
 2017, 54, 123, 0,0.5,       1,-85249,     0            , "t=2.08h",_
 2018, 54, 123, 1,3.5,      -1,-85064,     0.18518      , "t=5.49us",_
 2019, 55, 123, 0,0.5,       1,-81044,     0            , "t=5.88m",_
 2020, 55, 123, 1,5.5,      -1,-80888,     0.15627      , "t=1.64s",_
 2021, 55, 123, 2,4.5,       1,-80792,     0.252        , "t=114ns",_
 2022, 55, 123, 0,0,         0,-81037,     0.007        , "t=R&lt;0.1",_
 2023, 56, 123, 0,2.5,       1,-75655,     0            , "t=2.7m",_
 2024, 56, 123, 1,0.5,       1,-75534,     0.12095      , "t=830ns",_
 2025, 57, 123, 0,5.5,      -1,-68650,     0            , "t=17s",_
 2026, 58, 123, 0,2.5,       1,-60290,     0            , "t=3.8s",_
 2027, 59, 123, 0,1.5,       1,-50230,     0            , "t=800#ms",_
 2028, 44, 124, 0,0,         1,-33960,     0            , "t=15ms",_
 2029, 45, 124, 0,-77.777,   0,-44890,     0            , "t=30ms",_
 2030, 46, 124, 0,0,         1,-58390,     0            , "t=88ms",_
 2031, 46, 124, 1,-77.777,   0,-58330,     0.0622       , "t=&gt;20us",_
 2032, 47, 124, 0,2,        -1,-66200,     0            , "t=177.9ms",_
 2033, 47, 124, 1,8,        -1,-66200,     0            , "t=144ms",_
 2034, 47, 124, 2,-77.777,   0,-65970,     0.2311       , "t=1.7us",_
 2035, 48, 124, 0,0,         1,-76701.7,   0            , "t=1.25s",_
 2036, 49, 124, 0,3,         1,-80870,     0            , "t=3.12s",_
 2037, 49, 124, 1,8,        -1,-80890,    -0.02         , "t=3.7s",_
 2038, 50, 124, 0,0,         1,-88234.2,   0            , "t=stbl",_
 2039, 50, 124, 1,5,        -1,-86029.6,   2.20462      , "t=270ns",_
 2040, 50, 124, 2,7,        -1,-85909.2,   2.32501      , "t=3.1us",_
 2041, 51, 124, 0,3,        -1,-87620.2,   0            , "t=60.20d",_
 2042, 51, 124, 1,5,         1,-87609.3,   0.0108627    , "t=93s",_
 2043, 51, 124, 2,8,        -1,-87583.4,   0.036844     , "t=20.2m",_
 2044, 52, 124, 0,0,         1,-90525.3,   0            , "t=stbl",_
 2045, 53, 124, 0,2,        -1,-87365.7,   0            , "t=4.1760d",_
 2046, 54, 124, 0,0,         1,-87661.4,   0            , "t=stbl",_
 2047, 55, 124, 0,1,         1,-81731,     0            , "t=30.9s",_
 2048, 55, 124, 1,7,         1,-81268,     0.46263      , "t=6.3s",_
 2049, 55, 124, 0,0,         0,-81701,     0.03         , "t=R= ?",_
 2050, 56, 124, 0,0,         1,-79090,     0            , "t=11.0m",_
 2051, 57, 124, 0,7,        -1,-70260,     0            , "t=29.21s",_
 2052, 57, 124, 1,0,         1,-70160,     0.1          , "t=21s",_
 2053, 58, 124, 0,0,         1,-64920,     0            , "t=9.1s",_
 2054, 59, 124, 0,-77.777,   0,-53150,     0            , "t=1.2s",_
 2055, 60, 124, 0,0,         1,-44530,     0            , "t=500#ms",_
 2056, 45, 125, 0,3.5,       1,-42000,     0            , "t=26.5ms",_
 2057, 46, 125, 0,1.5,       1,-54120,     0            , "t=57ms",_
 2058, 47, 125, 0,3.5,       1,-64520,     0            , "t=159ms",_
 2059, 47, 125, 1,0.5,      -1,-64500,     0.02         , "t=50#ms",_
 2060, 47, 125, 2,-77.777,   0,-63640,     0.88         , "t=80ns",_
 2061, 48, 125, 0,1.5,       1,-73348.1,   0            , "t=680ms",_
 2062, 48, 125, 1,5.5,      -1,-73162,     0.186        , "t=480ms",_
 2063, 48, 125, 2,9.5,       1,-71840,     1.512        , "t=19us",_
 2064, 49, 125, 0,4.5,       1,-80477,     0            , "t=2.36s",_
 2065, 49, 125, 1,0.5,      -1,-80117,     0.36012      , "t=12.2s",_
 2066, 49, 125, 2,9.5,       1,-78468,     2.0094       , "t=9.4us",_
 2067, 50, 125, 0,5.5,      -1,-85896.4,   0            , "t=9.64d",_
 2068, 50, 125, 1,1.5,       1,-85868.9,   0.0275       , "t=9.52m",_
 2069, 50, 125, 2,9.5,       1,-84003.6,   1.8928       , "t=6.2us",_
 2070, 51, 125, 0,3.5,       1,-88256.3,   0            , "t=2.7586y",_
 2071, 51, 125, 1,7.5,      -1,-86285.1,   1.97125      , "t=4.1us",_
 2072, 51, 125, 2,9.5,      -1,-86144.2,   2.1121       , "t=28.0us",_
 2073, 52, 125, 0,0.5,       1,-89023.0,   0            , "t=stbl",_
 2074, 52, 125, 1,5.5,      -1,-88878.2,   0.144775     , "t=57.40d",_
 2075, 53, 125, 0,2.5,       1,-88837.2,   0            , "t=59.407d",_
 2076, 54, 125, 0,0.5,       1,-87193.4,   0            , "t=16.9h",_
 2077, 54, 125, 1,4.5,      -1,-86940.8,   0.25261      , "t=56.9s",_
 2078, 54, 125, 2,3.5,       1,-86897.5,   0.29589      , "t=140ns",_
 2079, 55, 125, 0,0.5,       1,-84088,     0            , "t=46.7m",_
 2080, 55, 125, 1,5.5,      -1,-83822,     0.2661       , "t=900us",_
 2081, 56, 125, 0,0.5,       1,-79669,     0            , "t=3.3m",_
 2082, 56, 125, 1,3.5,      -1,-79559,     0.11         , "t=2.76us",_
 2083, 57, 125, 0,5.5,      -1,-73759,     0            , "t=64.8s",_
 2084, 57, 125, 1,1.5,       1,-73652,     0.107        , "t=390ms",_
 2085, 58, 125, 0,3.5,      -1,-66660,     0            , "t=9.7s",_
 2086, 58, 125, 1,0.5,       1,-66570,     0.0936       , "t=13s",_
 2087, 59, 125, 0,1.5,       1,-57940,     0            , "t=3.3s",_
 2088, 60, 125, 0,2.5,       1,-47600,     0            , "t=650ms",_
 2089, 45, 126, 0,-77.777,   0,-37300,     0            , "t=19ms",_
 2090, 46, 126, 0,0,         1,-51860,     0            , "t=48.6ms",_
 2091, 46, 126, 1,5,        -1,-49840,     2.0235       , "t=330ns",_
 2092, 46, 126, 2,7,        -1,-49750,     2.1097       , "t=440ns",_
 2093, 47, 126, 0,2,         1,-60680,     0            , "t=99.3ms",_
 2094, 47, 126, 1,8,        -1,-60580,     0.1          , "t=92ms",_
 2095, 47, 126, 2,1,        -1,-60430,     0.2548       , "t=27us",_
 2096, 48, 126, 0,0,         1,-72256.8,   0            , "t=513ms",_
 2097, 49, 126, 0,3,         1,-77773,     0            , "t=1.53s",_
 2098, 49, 126, 1,8,        -1,-77710,     0.07         , "t=1.64s",_
 2099, 49, 126, 2,1,        -1,-77530,     0.2433       , "t=22us",_
 2100, 50, 126, 0,0,         1,-86015,     0            , "t=230ky",_
 2101, 50, 126, 1,7,        -1,-83796,     2.21899      , "t=5.8us",_
 2102, 50, 126, 2,10,        1,-83451,     2.5645       , "t=7.6us",_
 2103, 51, 126, 0,8,        -1,-86390,     0            , "t=12.35d",_
 2104, 51, 126, 1,5,         1,-86370,     0.0177       , "t=19.15m",_
 2105, 51, 126, 2,3,        -1,-86350,     0.0404       , "t=~11s",_
 2106, 52, 126, 0,0,         1,-90065.3,   0            , "t=stbl",_
 2107, 53, 126, 0,2,        -1,-87911,     0            , "t=12.93d",_
 2108, 53, 126, 1,3,         1,-87800,     0.111        , "t=128ns",_
 2109, 54, 126, 0,0,         1,-89147,     0            , "t=stbl",_
 2110, 55, 126, 0,1,         1,-84351,     0            , "t=1.64m",_
 2111, 55, 126, 1,-77.777,   0,-84078,     0.273        , "t=&gt;1us",_
 2112, 55, 126, 2,-77.777,   0,-83755,     0.5961       , "t=171us",_
 2113, 56, 126, 0,0,         1,-82670,     0            , "t=100m",_
 2114, 57, 126, 0,5,         1,-74970,     0            , "t=54s",_
 2115, 57, 126, 1,0,        -1,-74760,     0.21         , "t=20s",_
 2116, 58, 126, 0,0,         1,-70821,     0            , "t=51.0s",_
 2117, 59, 126, 0,4,         0,-60320,     0            , "t=3.12s",_
 2118, 60, 126, 0,0,         1,-52990,     0            , "t=1#s",_
 2119, 61, 126, 0,-77.777,   0,-39350,     0            , "t=500#ms",_
 2120, 45, 127, 0,3.5,       1,-34030,     0            , "t=28ms",_
 2121, 46, 127, 0,1.5,       1,-47180,     0            , "t=38ms",_
 2122, 47, 127, 0,3.5,       1,-58440,     0            , "t=89ms",_
 2123, 47, 127, 1,0.5,      -1,-58420,     0.02         , "t=20#ms",_
 2124, 48, 127, 0,1.5,       1,-68747,     0            , "t=330ms",_
 2125, 48, 127, 1,5.5,      -1,-68472,     0.276        , "t=200#ms",_
 2126, 48, 127, 2,9.5,       1,-66930,     1.813        , "t=17.5us",_
 2127, 49, 127, 0,4.5,       1,-76896,     0            , "t=1.09s",_
 2128, 49, 127, 1,0.5,      -1,-76487,     0.4089       , "t=3.67s",_
 2129, 49, 127, 2,10.5,     -1,-75030,     1.87         , "t=1.04s",_
 2130, 50, 127, 0,5.5,      -1,-83471,     0            , "t=2.10h",_
 2131, 50, 127, 1,1.5,       1,-83466,     0.00507      , "t=4.13m",_
 2132, 50, 127, 2,9.5,       1,-81644,     1.82667      , "t=4.52us",_
 2133, 51, 127, 0,3.5,       1,-86699,     0            , "t=3.85d",_
 2134, 51, 127, 1,7.5,      -1,-84779,     1.92019      , "t=11us",_
 2135, 51, 127, 2,11.5,      1,-84374,     2.3247       , "t=234ns",_
 2136, 52, 127, 0,1.5,       1,-88281.7,   0            , "t=9.35h",_
 2137, 52, 127, 1,5.5,      -1,-88193.5,   0.08823      , "t=106.1d",_
 2138, 53, 127, 0,2.5,       1,-88984,     0            , "t=stbl",_
 2139, 54, 127, 0,0.5,       1,-88322,     0            , "t=36.346d",_
 2140, 54, 127, 1,4.5,      -1,-88025,     0.2971       , "t=69.2s",_
 2141, 55, 127, 0,0.5,       1,-86240,     0            , "t=6.25h",_
 2142, 55, 127, 1,5.5,      -1,-85788,     0.45223      , "t=55us",_
 2143, 56, 127, 0,0.5,       1,-82818,     0            , "t=12.7m",_
 2144, 56, 127, 1,3.5,      -1,-82738,     0.08032      , "t=1.93s",_
 2145, 57, 127, 0,5.5,      -1,-77896,     0            , "t=5.1m",_
 2146, 57, 127, 1,1.5,       1,-77882,     0.0142       , "t=3.7m",_
 2147, 58, 127, 0,0.5,       1,-71979,     0            , "t=34s",_
 2148, 58, 127, 1,2.5,       1,-71972,     0.0073       , "t=28.6s",_
 2149, 58, 127, 2,3.5,      -1,-71942,     0.0368       , "t=&gt;10us",_
 2150, 59, 127, 0,1.5,       1,-64540,     0            , "t=4.2s",_
 2151, 59, 127, 1,5.5,      -1,-63940,     0.6          , "t=50#ms",_
 2152, 60, 127, 0,2.5,       1,-55540,     0            , "t=1.8s",_
 2153, 61, 127, 0,2.5,       1,-44790,     0            , "t=1#s",_
 2154, 46, 128, 0,0,         1,-44490,     0            , "t=35ms",_
 2155, 46, 128, 1,8,         1,-42340,     2.151        , "t=5.8us",_
 2156, 47, 128, 0,-77.777,   0,-54620,     0            , "t=59ms",_
 2157, 48, 128, 0,0,         1,-67242,     0            , "t=246ms",_
 2158, 48, 128, 1,5,        -1,-65372,     1.8705       , "t=270ns",_
 2159, 48, 128, 2,10,        1,-64527,     2.7146       , "t=3.56us",_
 2160, 49, 128, 0,3,         1,-74150,     0            , "t=816ms",_
 2161, 49, 128, 1,8,        -1,-74060,     0.08         , "t=720ms",_
 2162, 49, 128, 2,1,        -1,-73900,     0.24787      , "t=23us",_
 2163, 50, 128, 0,0,         1,-83362,     0            , "t=59.07m",_
 2164, 50, 128, 1,7,        -1,-81271,     2.0915       , "t=6.5s",_
 2165, 50, 128, 2,10,        1,-80870,     2.49191      , "t=2.91us",_
 2166, 51, 128, 0,8,        -1,-84630,     0            , "t=9.05h",_
 2167, 51, 128, 1,5,         1,-84620,     0.01         , "t=10.41m",_
 2168, 52, 128, 0,0,         1,-88993.7,   0            , "t=2.0Yy",_
 2169, 52, 128, 1,10,        1,-86202.9,   2.7908       , "t=363ns",_
 2170, 53, 128, 0,1,         1,-87739,     0            , "t=24.99m",_
 2171, 53, 128, 1,4,        -1,-87601,     0.137851     , "t=845ns",_
 2172, 53, 128, 2,6,        -1,-87572,     0.167368     , "t=175ns",_
 2173, 54, 128, 0,0,         1,-89860.3,   0            , "t=stbl",_
 2174, 54, 128, 1,8,        -1,-87073.1,   2.7872       , "t=83ns",_
 2175, 55, 128, 0,1,         1,-85932,     0            , "t=3.640m",_
 2176, 56, 128, 0,0,         1,-85378,     0            , "t=2.43d",_
 2177, 57, 128, 0,5,         1,-78630,     0            , "t=5.18m",_
 2178, 57, 128, 1,1,         1,-78530,     0.1          , "t=&lt;1.4m",_
 2179, 58, 128, 0,0,         1,-75534,     0            , "t=3.93m",_
 2180, 59, 128, 0,3,         1,-66331,     0            , "t=2.85s",_
 2181, 60, 128, 0,0,         1,-60310,     0            , "t=5#s",_
 2182, 61, 128, 0,5,         0,-47790,     0            , "t=1.0s",_
 2183, 62, 128, 0,0,         1,-38670,     0            , "t=500#ms",_
 2184, 46, 129, 0,3.5,      -1,-37610,     0            , "t=31ms",_
 2185, 47, 129, 0,3.5,       1,-51980,     0            , "t=49.9ms",_
 2186, 47, 129, 1,0.5,      -1,-51960,     0.02         , "t=10#ms",_
 2187, 48, 129, 0,1.5,       1,-63058,     0            , "t=151.5ms",_
 2188, 48, 129, 1,5.5,      -1,-62910,     0.15         , "t=147ms",_
 2189, 48, 129, 2,10.5,      1,-60970,     2.09         , "t=3.6ms",_
 2190, 49, 129, 0,4.5,       1,-72837.7,   0            , "t=570ms",_
 2191, 49, 129, 1,0.5,      -1,-72380,     0.458        , "t=1.23s",_
 2192, 49, 129, 2,8.5,      -1,-71149.7,   1.68797      , "t=11.2us",_
 2193, 50, 129, 0,1.5,       1,-80591,     0            , "t=2.23m",_
 2194, 50, 129, 1,5.5,      -1,-80556,     0.03515      , "t=6.9m",_
 2195, 50, 129, 2,9.5,       1,-78829,     1.7616       , "t=3.49us",_
 2196, 51, 129, 0,3.5,       1,-84629,     0            , "t=4.366h",_
 2197, 51, 129, 1,9.5,      -1,-82778,     1.85131      , "t=17.7m",_
 2198, 51, 129, 2,7.5,      -1,-82768,     1.86106      , "t=2.2us",_
 2199, 52, 129, 0,1.5,       1,-87004.8,   0            , "t=69.6m",_
 2200, 52, 129, 1,5.5,      -1,-86899.3,   0.10551      , "t=33.6d",_
 2201, 53, 129, 0,3.5,       1,-88507,     0            , "t=15.7My",_
 2202, 54, 129, 0,0.5,       1,-88696.059,               0            , "t=stbl",_
 2203, 54, 129, 1,5.5,      -1,-88459.92,  0.23614      , "t=8.88d",_
 2204, 55, 129, 0,0.5,       1,-87499,     0            , "t=32.06h",_
 2205, 55, 129, 1,5.5,      -1,-86924,     0.5754       , "t=718ns",_
 2206, 56, 129, 0,0.5,       1,-85063,     0            , "t=2.23h",_
 2207, 56, 129, 1,3.5,       1,-85055,     0.00842      , "t=2.135h",_
 2208, 57, 129, 0,1.5,       1,-81325,     0            , "t=11.6m",_
 2209, 57, 129, 1,5.5,      -1,-81153,     0.17233      , "t=560ms",_
 2210, 58, 129, 0,2.5,       1,-76287,     0            , "t=3.5m",_
 2211, 59, 129, 0,1.5,       1,-69774,     0            , "t=30s",_
 2212, 59, 129, 1,5.5,      -1,-69390,     0.38257      , "t=1#ms",_
 2213, 60, 129, 0,2.5,       1,-62320,     0            , "t=6.8s",_
 2214, 60, 129, 1,0.5,       1,-62270,     0.05         , "t=2.6s",_
 2215, 61, 129, 0,2.5,      -1,-52880,     0            , "t=2.4s",_
 2216, 62, 129, 0,1.5,       1,-42000,     0            , "t=550ms",_
 2217, 47, 130, 0,-77.777,   0,-45700,     0            , "t=40.6ms",_
 2218, 48, 130, 0,0,         1,-61118,     0            , "t=126.8ms",_
 2219, 48, 130, 1,8,         1,-58988,     2.1296       , "t=240ns",_
 2220, 49, 130, 0,1,        -1,-69880,     0            , "t=284ms",_
 2221, 49, 130, 1,8,        -1,-69830,     0.05         , "t=540ms",_
 2222, 49, 130, 2,5,         1,-69480,     0.4          , "t=540ms",_
 2223, 50, 130, 0,0,         1,-80132.2,   0            , "t=3.72m",_
 2224, 50, 130, 1,7,        -1,-78185.3,   1.94688      , "t=1.7m",_
 2225, 50, 130, 2,10,        1,-77697.4,   2.43479      , "t=1.501us",_
 2226, 51, 130, 0,8,        -1,-82286,     0            , "t=39.5m",_
 2227, 51, 130, 1,4,         0,-82281,     0.0048       , "t=6.3m",_
 2228, 51, 130, 2,6,        -1,-82201,     0.08467      , "t=800ns",_
 2229, 52, 130, 0,0,         1,-87352.949,               0            , "t=690Ey",_
 2230, 52, 130, 1,7,        -1,-85206.54,  2.14641      , "t=186ns",_
 2231, 52, 130, 2,10,        1,-84685.7,   2.6672       , "t=1.90us",_
 2232, 53, 130, 0,5,         1,-86936,     0            , "t=12.36h",_
 2233, 53, 130, 1,2,         1,-86896,     0.0399525    , "t=8.84m",_
 2234, 53, 130, 2,6,        -1,-86866,     0.0695865    , "t=133ns",_
 2235, 54, 130, 0,0,         1,-89880.463,               0            , "t=stbl",_
 2236, 55, 130, 0,1,         1,-86900,     0            , "t=29.21m",_
 2237, 55, 130, 1,5,        -1,-86737,     0.16325      , "t=3.46m",_
 2238, 55, 130, 0,0,         0,-86873,     0.027        , "t=R=.2~~~.1",_
 2239, 56, 130, 0,0,         1,-87261.5,   0            , "t=stbl",_
 2240, 56, 130, 1,8,        -1,-84786.4,   2.47512      , "t=9.54ms",_
 2241, 57, 130, 0,3,         1,-81627,     0            , "t=8.7m",_
 2242, 57, 130, 1,5,         1,-81413,     0.214        , "t=760ns",_
 2243, 57, 130, 2,6,         1,-81308,     0.3191       , "t=33ns",_
 2244, 58, 130, 0,0,         1,-79423,     0            , "t=22.9m",_
 2245, 58, 130, 1,7,        -1,-76969,     2.4536       , "t=100ns",_
 2246, 59, 130, 0,6,         0,-71180,     0            , "t=40.0s",_
 2247, 59, 130, 1,2,         1,-71080,     0.1          , "t=10#s",_
 2248, 60, 130, 0,0,         1,-66596,     0            , "t=21s",_
 2249, 61, 130, 0,5,         1,-55400,     0            , "t=2.6s",_
 2250, 62, 130, 0,0,         1,-47510,     0            , "t=1#s",_
 2251, 63, 130, 0,1,         1,-33680,     0            , "t=1.0ms",_
 2252, 47, 131, 0,3.5,       1,-40380,     0            , "t=35ms",_
 2253, 48, 131, 0,3.5,      -1,-55220,     0            , "t=98ms",_
 2254, 49, 131, 0,4.5,       1,-68025.0,   0            , "t=261ms",_
 2255, 49, 131, 1,0.5,      -1,-67660,     0.365        , "t=350ms",_
 2256, 49, 131, 2,10.5,      1,-64280,     3.75         , "t=320ms",_
 2257, 50, 131, 0,1.5,       1,-77265,     0            , "t=56.0s",_
 2258, 50, 131, 1,5.5,      -1,-77200,     0.0651       , "t=58.4s",_
 2259, 50, 131, 2,11.5,     -1,-72595,     4.67         , "t=304ns",_
 2260, 51, 131, 0,3.5,       1,-81981.4,   0            , "t=23.03m",_
 2261, 51, 131, 1,7.5,      -1,-80305.3,   1.67606      , "t=91us",_
 2262, 51, 131, 2,9.5,      -1,-80294.2,   1.6872       , "t=4.3us",_
 2263, 52, 131, 0,1.5,       1,-85211.01,  0            , "t=25.0m",_
 2264, 52, 131, 1,5.5,      -1,-85028.75,  0.182258     , "t=32.48h",_
 2265, 52, 131, 2,11.5,      1,-83271.0,   1.94         , "t=93ms",_
 2266, 53, 131, 0,3.5,       1,-87442.7,   0            , "t=8.0252d",_
 2267, 53, 131, 1,9.5,      -1,-85524.3,   1.9184       , "t=24us",_
 2268, 54, 131, 0,1.5,       1,-88413.558,               0            , "t=stbl",_
 2269, 54, 131, 1,5.5,      -1,-88249.628,               0.16393      , "t=11.84d",_
 2270, 55, 131, 0,2.5,       1,-88059,     0            , "t=9.689d",_
 2271, 56, 131, 0,0.5,       1,-86683.7,   0            , "t=11.52d",_
 2272, 56, 131, 1,4.5,      -1,-86495.7,   0.187995     , "t=14.26m",_
 2273, 57, 131, 0,1.5,       1,-83769,     0            , "t=59m",_
 2274, 57, 131, 1,5.5,      -1,-83464,     0.3046       , "t=170us",_
 2275, 58, 131, 0,3.5,       1,-79710,     0            , "t=10.3m",_
 2276, 58, 131, 1,0.5,       1,-79650,     0.06309      , "t=5.4m",_
 2277, 59, 131, 0,1.5,       1,-74300,     0            , "t=1.50m",_
 2278, 59, 131, 1,5.5,      -1,-74150,     0.1524       , "t=5.73s",_
 2279, 60, 131, 0,2.5,       1,-67768,     0            , "t=25.4s",_
 2280, 61, 131, 0,5.5,      -1,-59660,     0            , "t=6.3s",_
 2281, 62, 131, 0,2.5,       1,-50130,     0            , "t=1.2s",_
 2282, 63, 131, 0,1.5,       1,-39270,     0            , "t=17.8ms",_
 2283, 47, 132, 0,-77.777,   0,-33790,     0            , "t=30ms",_
 2284, 48, 132, 0,0,         1,-50260,     0            , "t=82ms",_
 2285, 49, 132, 0,7,        -1,-62410,     0            , "t=198ms",_
 2286, 50, 132, 0,0,         1,-76546.5,   0            , "t=39.7s",_
 2287, 50, 132, 1,8,         1,-71698.0,   4.84852      , "t=2.079us",_
 2288, 51, 132, 0,4,         1,-79635.3,   0            , "t=2.79m",_
 2289, 51, 132, 1,8,        -1,-79440,     0.2          , "t=4.10m",_
 2290, 51, 132, 2,6,        -1,-79380.8,   0.2545       , "t=102ns",_
 2291, 52, 132, 0,0,         1,-85188,     0            , "t=3.204d",_
 2292, 52, 132, 1,6,         1,-83413,     1.7748       , "t=145ns",_
 2293, 52, 132, 2,7,        -1,-83263,     1.92547      , "t=28.1us",_
 2294, 53, 132, 0,4,         1,-85703,     0            , "t=2.295h",_
 2295, 53, 132, 1,8,        -1,-85594,     0.11         , "t=1.387h",_
 2296, 54, 132, 0,0,         1,-89278.962,               0            , "t=stbl",_
 2297, 54, 132, 1,10,        1,-86526.75,  2.75221      , "t=8.39ms",_
 2298, 55, 132, 0,2,         1,-87152.7,   0            , "t=6.480d",_
 2299, 56, 132, 0,0,         1,-88435.0,   0            , "t=stbl",_
 2300, 57, 132, 0,2,        -1,-83720,     0            , "t=4.8h",_
 2301, 57, 132, 1,6,        -1,-83530,     0.1882       , "t=24.3m",_
 2302, 58, 132, 0,0,         1,-82471,     0            , "t=3.51h",_
 2303, 58, 132, 1,8,        -1,-80130,     2.34115      , "t=9.4ms",_
 2304, 59, 132, 0,2,         1,-75227,     0            , "t=1.49m",_
 2305, 59, 132, 1,5,         1,-75200,     0.03         , "t=1#s",_
 2306, 59, 132, 2,8,         1,-74980,     0.25         , "t=2.46us",_
 2307, 60, 132, 0,0,         1,-71426,     0            , "t=1.56m",_
 2308, 61, 132, 0,3,         1,-61630,     0            , "t=6.2s",_
 2309, 62, 132, 0,0,         1,-55080,     0            , "t=4.0s",_
 2310, 63, 132, 0,-77.777,   0,-42200,     0            , "t=100#ms",_
 2311, 48, 133, 0,3.5,      -1,-43920,     0            , "t=61ms",_
 2312, 49, 133, 0,4.5,       1,-57460,     0            , "t=165ms",_
 2313, 49, 133, 1,0.5,      -1,-57130,     0.33         , "t=180#ms",_
 2314, 50, 133, 0,3.5,      -1,-70873.9,   0            , "t=1.46s",_
 2315, 51, 133, 0,3.5,       1,-78924,     0            , "t=2.34m",_
 2316, 51, 133, 1,10.5,      1,-74360,     4.56         , "t=16.54us",_
 2317, 52, 133, 0,1.5,       1,-82937.1,   0            , "t=12.5m",_
 2318, 52, 133, 1,5.5,      -1,-82602.8,   0.33426      , "t=55.4m",_
 2319, 52, 133, 2,9.5,      -1,-81326.7,   1.6104       , "t=100ns",_
 2320, 53, 133, 0,3.5,       1,-85858,     0            , "t=20.83h",_
 2321, 53, 133, 1,9.5,      -1,-84224,     1.634148     , "t=9s",_
 2322, 53, 133, 2,7.5,      -1,-84129,     1.729137     , "t=~170ns",_
 2323, 54, 133, 0,1.5,       1,-87643.6,   0            , "t=5.2475d",_
 2324, 54, 133, 1,5.5,      -1,-87410.4,   0.233221     , "t=2.198d",_
 2325, 55, 133, 0,3.5,       1,-88070.931,               0            , "t=stbl",_
 2326, 56, 133, 0,0.5,       1,-87553.6,   0            , "t=10.551y",_
 2327, 56, 133, 1,5.5,      -1,-87265.3,   0.288252     , "t=38.90h",_
 2328, 57, 133, 0,2.5,       1,-85494,     0            , "t=3.912h",_
 2329, 58, 133, 0,0.5,       1,-82418,     0            , "t=97m",_
 2330, 58, 133, 1,4.5,      -1,-82381,     0.0372       , "t=5.1h",_
 2331, 59, 133, 0,1.5,       1,-77938,     0            , "t=6.5m",_
 2332, 59, 133, 1,5.5,      -1,-77746,     0.19212      , "t=1.1s",_
 2333, 60, 133, 0,3.5,       1,-72330,     0            , "t=70s",_
 2334, 60, 133, 1,0.5,       1,-72200,     0.12797      , "t=~70s",_
 2335, 60, 133, 2,4.5,      -1,-72150,     0.1761       , "t=301ns",_
 2336, 61, 133, 0,1.5,       1,-65410,     0            , "t=13.5s",_
 2337, 61, 133, 1,5.5,      -1,-65280,     0.1297       , "t=8#s",_
 2338, 62, 133, 0,2.5,       1,-57230,     0            , "t=2.89s",_
 2339, 62, 133, 1,0.5,      -1,-57110,     0.12         , "t=3.5s",_
 2340, 63, 133, 0,5.5,      -1,-47240,     0            , "t=200#ms",_
 2341, 64, 133, 0,2.5,       1,-35860,     0            , "t=10#ms",_
 2342, 48, 134, 0,0,         1,-38920,     0            , "t=65ms",_
 2343, 49, 134, 0,0,         0,-51660,     0            , "t=140ms",_
 2344, 50, 134, 0,0,         1,-66434,     0            , "t=890ms",_
 2345, 50, 134, 1,6,         1,-65187,     1.2474       , "t=87ns",_
 2346, 51, 134, 0,0,        -1,-74020.5,   0            , "t=780ms",_
 2347, 51, 134, 1,7,        -1,-73741.5,   0.279        , "t=10.07s",_
 2348, 52, 134, 0,0,         1,-82533.7,   0            , "t=41.8m",_
 2349, 52, 134, 1,6,         1,-80842.4,   1.69134      , "t=164.1ns",_
 2350, 53, 134, 0,4,         1,-84043,     0            , "t=52.5m",_
 2351, 53, 134, 1,8,        -1,-83727,     0.31649      , "t=3.52m",_
 2352, 54, 134, 0,0,         1,-88125.822,               0            , "t=stbl",_
 2353, 54, 134, 1,7,        -1,-86160.3,   1.9655       , "t=290ms",_
 2354, 54, 134, 2,10,        1,-85100.6,   3.0252       , "t=5us",_
 2355, 55, 134, 0,4,         1,-86891.154,               0            , "t=2.0652y",_
 2356, 55, 134, 1,8,        -1,-86752.410,               0.1387441    , "t=2.912h",_
 2357, 56, 134, 0,0,         1,-88949.9,   0            , "t=stbl",_
 2358, 56, 134, 1,10,        1,-85992.7,   2.9572       , "t=2.63us",_
 2359, 57, 134, 0,1,         1,-85219,     0            , "t=6.45m",_
 2360, 57, 134, 1,-77.777,   0,-84780,     0.44         , "t=29us",_
 2361, 58, 134, 0,0,         1,-84833,     0            , "t=3.16d",_
 2362, 58, 134, 1,10,        1,-81624,     3.2086       , "t=308ns",_
 2363, 59, 134, 0,2,        -1,-78528,     0            , "t=17m",_
 2364, 59, 134, 1,6,        -1,-78460,     0.068        , "t=~11m",_
 2365, 60, 134, 0,0,         1,-75646,     0            , "t=8.5m",_
 2366, 60, 134, 1,8,        -1,-73353,     2.293        , "t=410us",_
 2367, 61, 134, 0,5,         1,-66740,     0            , "t=22s",_
 2368, 61, 134, 1,2,         1,-66740,     0            , "t=~5s",_
 2369, 61, 134, 2,7,        -1,-66620,     0.12         , "t=20us",_
 2370, 62, 134, 0,0,         1,-61380,     0            , "t=9.5s",_
 2371, 63, 134, 0,-77.777,   0,-49930,     0            , "t=500ms",_
 2372, 64, 134, 0,0,         1,-41300,     0            , "t=400#ms",_
 2373, 49, 135, 0,4.5,       1,-46530,     0            , "t=101ms",_
 2374, 50, 135, 0,3.5,      -1,-60632,     0            , "t=515ms",_
 2375, 51, 135, 0,3.5,       1,-69690.3,   0            , "t=1.679s",_
 2376, 52, 135, 0,3.5,      -1,-77728.8,   0            , "t=19.0s",_
 2377, 52, 135, 1,9.5,      -1,-76173.9,   1.55489      , "t=511ns",_
 2378, 53, 135, 0,3.5,       1,-83779.1,   0            , "t=6.58h",_
 2379, 54, 135, 0,1.5,       1,-86413,     0            , "t=9.14h",_
 2380, 54, 135, 1,5.5,      -1,-85886,     0.526551     , "t=15.29m",_
 2381, 55, 135, 0,3.5,       1,-87581.6,   0            , "t=1.33My",_
 2382, 55, 135, 1,9.5,      -1,-85948.7,   1.6329       , "t=53m",_
 2383, 56, 135, 0,1.5,       1,-87850.5,   0            , "t=stbl",_
 2384, 56, 135, 1,5.5,      -1,-87582.3,   0.268218     , "t=28.11h",_
 2385, 57, 135, 0,2.5,       1,-86643,     0            , "t=19.5h",_
 2386, 58, 135, 0,0.5,       1,-84616,     0            , "t=17.7h",_
 2387, 58, 135, 1,5.5,      -1,-84170,     0.44581      , "t=20s",_
 2388, 59, 135, 0,1.5,       1,-80936,     0            , "t=24m",_
 2389, 59, 135, 1,5.5,      -1,-80578,     0.35806      , "t=105us",_
 2390, 60, 135, 0,4.5,      -1,-76214,     0            , "t=12.4m",_
 2391, 60, 135, 1,0.5,       1,-76149,     0.06495      , "t=5.5m",_
 2392, 61, 135, 0,2.5,       1,-70050,     0            , "t=49s",_
 2393, 61, 135, 1,5.5,      -1,-69830,     0.22         , "t=40s",_
 2394, 62, 135, 0,3.5,       1,-62860,     0            , "t=10.3s",_
 2395, 62, 135, 1,1.5,       1,-62860,     0            , "t=2.4s",_
 2396, 63, 135, 0,5.5,      -1,-54150,     0            , "t=1.5s",_
 2397, 64, 135, 0,2.5,       1,-44390,     0            , "t=1.1s",_
 2398, 65, 135, 0,3.5,      -1,-32830,     0            , "t=1.01ms",_
 2399, 49, 136, 0,-77.777,   0,-40510,     0            , "t=86ms",_
 2400, 50, 136, 0,0,         1,-55900,     0            , "t=350ms",_
 2401, 51, 136, 0,1,        -1,-64507,     0            , "t=923ms",_
 2402, 51, 136, 1,6,        -1,-64230,     0.277        , "t=570ns",_
 2403, 52, 136, 0,0,         1,-74425.3,   0            , "t=17.63s",_
 2404, 53, 136, 0,1,        -1,-79545,     0            , "t=83.4s",_
 2405, 53, 136, 1,6,        -1,-79339,     0.206        , "t=46.9s",_
 2406, 54, 136, 0,0,         1,-86429.159,               0            , "t=2.19Zy",_
 2407, 54, 136, 1,6,         1,-84537.456,               1.891703     , "t=2.95us",_
 2408, 55, 136, 0,5,         1,-86338.7,   0            , "t=13.16d",_
 2409, 55, 136, 1,8,        -1,-85820.8,   0.5179       , "t=17.5s",_
 2410, 56, 136, 0,0,         1,-88886.9,   0            , "t=stbl",_
 2411, 56, 136, 1,7,        -1,-86856.4,   2.030466     , "t=308.4ms",_
 2412, 56, 136, 2,10,        1,-85529.5,   3.3574       , "t=91ns",_
 2413, 57, 136, 0,1,         1,-86040,     0            , "t=9.87m",_
 2414, 57, 136, 1,7,        -1,-85780,     0.2593       , "t=114ms",_
 2415, 58, 136, 0,0,         1,-86508.4,   0            , "t=stbl",_
 2416, 58, 136, 1,10,        1,-83412.9,   3.0955       , "t=1.96us",_
 2417, 59, 136, 0,2,         1,-81340,     0            , "t=13.1m",_
 2418, 60, 136, 0,0,         1,-79199,     0            , "t=50.7m",_
 2419, 61, 136, 0,5,         1,-71170,     0            , "t=107s",_
 2420, 61, 136, 1,2,         1,-71070,     0.1          , "t=300s",_
 2421, 61, 136, 2,8,         1,-71100,     0.068        , "t=1.5us",_
 2422, 62, 136, 0,0,         1,-66811,     0            , "t=47s",_
 2423, 62, 136, 1,8,        -1,-64546,     2.2647       , "t=15us",_
 2424, 63, 136, 0,7,         1,-56240,     0            , "t=3.3s",_
 2425, 63, 136, 1,3,         1,-56240,     0            , "t=3.8s",_
 2426, 64, 136, 0,0,         1,-49090,     0            , "t=1#s",_
 2427, 65, 136, 0,-77.777,   0,-36130,     0            , "t=200#ms",_
 2428, 49, 137, 0,4.5,       1,-35040,     0            , "t=70ms",_
 2429, 50, 137, 0,2.5,      -1,-49790,     0            , "t=273ms",_
 2430, 51, 137, 0,3.5,       1,-60060,     0            , "t=484ms",_
 2431, 52, 137, 0,1.5,      -1,-69303.8,   0            , "t=2.49s",_
 2432, 53, 137, 0,3.5,       1,-76356,     0            , "t=24.13s",_
 2433, 54, 137, 0,3.5,      -1,-82383.40,  0            , "t=3.818m",_
 2434, 55, 137, 0,3.5,       1,-86545.6,   0            , "t=30.08y",_
 2435, 56, 137, 0,1.5,       1,-87721.2,   0            , "t=stbl",_
 2436, 56, 137, 1,5.5,      -1,-87059.5,   0.661659     , "t=2.552m",_
 2437, 56, 137, 2,8.5,      -1,-85372.1,   2.3491       , "t=590ns",_
 2438, 57, 137, 0,3.5,       1,-87140.7,   0            , "t=60ky",_
 2439, 57, 137, 1,9.5,      -1,-85271.2,   1.8695       , "t=342ns",_
 2440, 58, 137, 0,1.5,       1,-85918.6,   0            , "t=9.0h",_
 2441, 58, 137, 1,5.5,      -1,-85664.3,   0.25429      , "t=34.4h",_
 2442, 59, 137, 0,2.5,       1,-83202,     0            , "t=1.28h",_
 2443, 59, 137, 1,5.5,      -1,-82641,     0.56122      , "t=2.66us",_
 2444, 60, 137, 0,0.5,       1,-79585,     0            , "t=38.5m",_
 2445, 60, 137, 1,5.5,      -1,-79066,     0.51943      , "t=1.60s",_
 2446, 61, 137, 0,2.5,       1,-74073,     0            , "t=2#m",_
 2447, 61, 137, 1,5.5,      -1,-73930,     0.15         , "t=2.4m",_
 2448, 62, 137, 0,4.5,      -1,-68030,     0            , "t=45s",_
 2449, 62, 137, 1,0.5,       1,-67850,     0.18         , "t=20#s",_
 2450, 63, 137, 0,5.5,      -1,-60146,     0            , "t=8.4s",_
 2451, 64, 137, 0,3.5,       1,-51210,     0            , "t=2.2s",_
 2452, 65, 137, 0,5.5,      -1,-40970,     0            , "t=600#ms",_
 2453, 50, 138, 0,0,         1,-44860,     0            , "t=150ms",_
 2454, 50, 138, 1,6,         1,-43520,     1.344        , "t=210ns",_
 2455, 51, 138, 0,0,        -1,-54220,     0            , "t=348ms",_
 2456, 52, 138, 0,0,         1,-65696,     0            , "t=1.4s",_
 2457, 53, 138, 0,1,        -1,-71980,     0            , "t=6.23s",_
 2458, 53, 138, 1,3,        -1,-71912,     0.0679       , "t=1.26us",_
 2459, 54, 138, 0,0,         1,-79972.2,   0            , "t=14.14m",_
 2460, 55, 138, 0,3,        -1,-82887,     0            , "t=33.41m",_
 2461, 55, 138, 1,6,        -1,-82807,     0.0799       , "t=2.91m",_
 2462, 55, 138, 0,0,         0,-82847,     0.04         , "t=R= ?",_
 2463, 56, 138, 0,0,         1,-88261.6,   0            , "t=stbl",_
 2464, 56, 138, 1,6,         1,-86171.1,   2.09054      , "t=800ns",_
 2465, 57, 138, 0,5,         1,-86519,     0            , "t=102Gy",_
 2466, 57, 138, 1,3,         1,-86446,     0.07257      , "t=116ns",_
 2467, 57, 138, 2,7,        -1,-85780,     0.7392       , "t=2.0us",_
 2468, 58, 138, 0,0,         1,-87571,     0            , "t=stbl",_
 2469, 58, 138, 1,7,        -1,-85442,     2.12917      , "t=8.65ms",_
 2470, 59, 138, 0,1,         1,-83134,     0            , "t=1.45m",_
 2471, 59, 138, 1,7,        -1,-82782,     0.352        , "t=2.12h",_
 2472, 60, 138, 0,0,         1,-82018,     0            , "t=5.04h",_
 2473, 60, 138, 1,10,        1,-78843,     3.1749       , "t=370ns",_
 2474, 61, 138, 0,1,         1,-74940,     0            , "t=10s",_
 2475, 61, 138, 1,5,        -1,-74911,     0.03         , "t=3.24m",_
 2476, 62, 138, 0,0,         1,-71498,     0            , "t=3.1m",_
 2477, 63, 138, 0,6,        -1,-61750,     0            , "t=12.1s",_
 2478, 64, 138, 0,0,         1,-55800,     0            , "t=4.7s",_
 2479, 64, 138, 1,8,        -1,-53570,     2.2331       , "t=6.2us",_
 2480, 65, 138, 0,-77.777,   0,-43670,     0            , "t=800#ms",_
 2481, 66, 138, 0,0,         1,-34930,     0            , "t=200#ms",_
 2482, 50, 139, 0,2.5,      -1,-38440,     0            , "t=130ms",_
 2483, 51, 139, 0,3.5,       1,-49790,     0            , "t=93ms",_
 2484, 52, 139, 0,2.5,      -1,-60205,     0            , "t=500#ms",_
 2485, 53, 139, 0,3.5,       1,-68471,     0            , "t=2.282s",_
 2486, 54, 139, 0,1.5,      -1,-75644.6,   0            , "t=39.68s",_
 2487, 55, 139, 0,3.5,       1,-80701,     0            , "t=9.27m",_
 2488, 56, 139, 0,3.5,      -1,-84913.8,   0            , "t=83.13m",_
 2489, 57, 139, 0,3.5,       1,-87226.2,   0            , "t=stbl",_
 2490, 57, 139, 1,8.5,       1,-85426.3,   1.7999       , "t=315ns",_
 2491, 58, 139, 0,1.5,       1,-86948,     0            , "t=137.641d",_
 2492, 58, 139, 1,5.5,      -1,-86194,     0.75424      , "t=56.54s",_
 2493, 59, 139, 0,2.5,       1,-84819,     0            , "t=4.41h",_
 2494, 60, 139, 0,1.5,       1,-82014,     0            , "t=29.7m",_
 2495, 60, 139, 1,5.5,      -1,-81783,     0.23115      , "t=5.50h",_
 2496, 60, 139, 2,11.5,      1,-79398,     2.616        , "t=276.8ns",_
 2497, 61, 139, 0,2.5,       1,-77500,     0            , "t=4.15m",_
 2498, 61, 139, 1,5.5,      -1,-77311,     0.1887       , "t=180ms",_
 2499, 62, 139, 0,0.5,       1,-72380,     0            , "t=2.57m",_
 2500, 62, 139, 1,5.5,      -1,-71923,     0.4574       , "t=10.7s",_
 2501, 63, 139, 0,5.5,      -1,-65398,     0            , "t=17.9s",_
 2502, 63, 139, 1,3.5,       1,-65250,     0.1482       , "t=10us",_
 2503, 64, 139, 0,4.5,      -1,-57630,     0            , "t=5.7s",_
 2504, 64, 139, 1,0.5,       1,-57380,     0.25         , "t=4.8s",_
 2505, 65, 139, 0,5.5,      -1,-48130,     0            , "t=1.6s",_
 2506, 66, 139, 0,3.5,       1,-37640,     0            , "t=600ms",_
 2507, 51, 140, 0,4,        -1,-43940,     0            , "t=100#ms",_
 2508, 51, 140, 1,6,        -1,-43610,     0.33         , "t=41us",_
 2509, 52, 140, 0,0,         1,-56580,     0            , "t=300#ms",_
 2510, 53, 140, 0,4,        -1,-63606,     0            , "t=860ms",_
 2511, 54, 140, 0,0,         1,-72986.5,   0            , "t=13.60s",_
 2512, 55, 140, 0,1,        -1,-77050,     0            , "t=63.7s",_
 2513, 55, 140, 1,2,        -1,-77036,     0.013931     , "t=471ns",_
 2514, 56, 140, 0,0,         1,-83269,     0            , "t=12.7527d",_
 2515, 57, 140, 0,3,        -1,-84315.9,   0            , "t=40.285h",_
 2516, 58, 140, 0,0,         1,-88076.1,   0            , "t=stbl",_
 2517, 58, 140, 1,6,         1,-85968.2,   2.10786      , "t=7.3us",_
 2518, 59, 140, 0,1,         1,-84688,     0            , "t=3.39m",_
 2519, 59, 140, 1,5,         1,-84560,     0.1278       , "t=350ns",_
 2520, 59, 140, 2,7,        -1,-83924,     0.7637       , "t=3.05us",_
 2521, 60, 140, 0,0,         1,-84259,     0            , "t=3.37d",_
 2522, 60, 140, 1,7,        -1,-82038,     2.2214       , "t=600us",_
 2523, 60, 140, 2,20,        1,-76829,     7.4296       , "t=1.22us",_
 2524, 61, 140, 0,1,         1,-78214,     0            , "t=9.2s",_
 2525, 61, 140, 1,8,        -1,-77783,     0.431        , "t=5.95m",_
 2526, 62, 140, 0,0,         1,-75456,     0            , "t=14.82m",_
 2527, 63, 140, 0,1,         1,-66990,     0            , "t=1.51s",_
 2528, 63, 140, 1,5,        -1,-66780,     0.21         , "t=125ms",_
 2529, 63, 140, 2,8,         1,-66320,     0.669        , "t=299.8ns",_
 2530, 64, 140, 0,0,         1,-61782,     0            , "t=15.8s",_
 2531, 65, 140, 0,7,         1,-50480,     0            , "t=2.32s",_
 2532, 66, 140, 0,0,         1,-42830,     0            , "t=700#ms",_
 2533, 66, 140, 1,8,        -1,-40660,     2.1661       , "t=7.0us",_
 2534, 67, 140, 0,8,         1,-29260,     0            , "t=6ms",_
 2535, 51, 141, 0,3.5,       1,-39110,     0            , "t=100#ms",_
 2536, 52, 141, 0,2.5,      -1,-50490,     0            , "t=150#ms",_
 2537, 53, 141, 0,3.5,       1,-59927,     0            , "t=430ms",_
 2538, 54, 141, 0,2.5,      -1,-68197.3,   0            , "t=1.73s",_
 2539, 55, 141, 0,3.5,       1,-74478,     0            , "t=24.84s",_
 2540, 56, 141, 0,1.5,      -1,-79733,     0            , "t=18.27m",_
 2541, 57, 141, 0,3.5,       1,-82932,     0            , "t=3.92h",_
 2542, 58, 141, 0,3.5,      -1,-85432.9,   0            , "t=32.511d",_
 2543, 59, 141, 0,2.5,       1,-86015.6,   0            , "t=stbl",_
 2544, 60, 141, 0,1.5,       1,-84193,     0            , "t=2.49h",_
 2545, 60, 141, 1,5.5,      -1,-83436,     0.75651      , "t=62.0s",_
 2546, 61, 141, 0,2.5,       1,-80523,     0            , "t=20.90m",_
 2547, 61, 141, 1,5.5,      -1,-79894,     0.62862      , "t=630ns",_
 2548, 61, 141, 2,-77.777,   0,-77992,     2.53075      , "t=&gt;2us",_
 2549, 62, 141, 0,0.5,       1,-75934,     0            , "t=10.2m",_
 2550, 62, 141, 1,5.5,      -1,-75758,     0.1759       , "t=22.6m",_
 2551, 63, 141, 0,2.5,       1,-69926,     0            , "t=40.7s",_
 2552, 63, 141, 1,5.5,      -1,-69830,     0.09645      , "t=2.7s",_
 2553, 64, 141, 0,0.5,       1,-63224,     0            , "t=14s",_
 2554, 64, 141, 1,5.5,      -1,-62846,     0.37776      , "t=24.5s",_
 2555, 65, 141, 0,2.5,      -1,-54540,     0            , "t=3.5s",_
 2556, 65, 141, 1,5.5,      -1,-54540,     0            , "t=7.9s",_
 2557, 66, 141, 0,4.5,      -1,-45380,     0            , "t=900ms",_
 2558, 67, 141, 0,3.5,      -1,-34360,     0            , "t=4.1ms",_
 2559, 67, 141, 1,0.5,       1,-34290,     0.066        , "t=7.3us",_
 2560, 52, 142, 0,0,         1,-46370,     0            , "t=100#ms",_
 2561, 53, 142, 0,2,        -1,-54770,     0            , "t=222ms",_
 2562, 54, 142, 0,0,         1,-65229.6,   0            , "t=1.23s",_
 2563, 55, 142, 0,0,        -1,-70515,     0            , "t=1.684s",_
 2564, 56, 142, 0,0,         1,-77842,     0            , "t=10.6m",_
 2565, 57, 142, 0,2,        -1,-80024,     0            , "t=91.1m",_
 2566, 57, 142, 1,4,        -1,-79878,     0.14582      , "t=870ns",_
 2567, 58, 142, 0,0,         1,-84533.2,   0            , "t=stbl",_
 2568, 59, 142, 0,2,        -1,-83787.5,   0            , "t=19.12h",_
 2569, 59, 142, 1,5,        -1,-83783.8,   0.003694     , "t=14.6m",_
 2570, 60, 142, 0,0,         1,-85950.0,   0            , "t=stbl",_
 2571, 60, 142, 1,6,         1,-83740.7,   2.209303     , "t=16.5us",_
 2572, 61, 142, 0,1,         1,-81142,     0            , "t=40.5s",_
 2573, 61, 142, 1,8,        -1,-80259,     0.88317      , "t=2.0ms",_
 2574, 61, 142, 2,13,       -1,-78313,     2.8287       , "t=67us",_
 2575, 62, 142, 0,0,         1,-78986,     0            , "t=72.49m",_
 2576, 62, 142, 1,7,        -1,-76614,     2.3721       , "t=170ns",_
 2577, 62, 142, 2,10,        1,-75324,     3.6622       , "t=480ns",_
 2578, 63, 142, 0,1,         1,-71310,     0            , "t=2.36s",_
 2579, 63, 142, 1,8,        -1,-70856,     0.46         , "t=1.223m",_
 2580, 64, 142, 0,0,         1,-66960,     0            , "t=70.2s",_
 2581, 65, 142, 0,1,         1,-56560,     0            , "t=597ms",_
 2582, 65, 142, 1,5,        -1,-56280,     0.2797       , "t=303ms",_
 2583, 65, 142, 2,8,         1,-55910,     0.6521       , "t=26us",_
 2584, 66, 142, 0,0,         1,-50120,     0            , "t=2.3s",_
 2585, 67, 142, 0,7,        -1,-37250,     0            , "t=400ms",_
 2586, 68, 142, 0,0,         1,-28030,     0            , "t=10#us",_
 2587, 52, 143, 0,3.5,       1,-40280,     0            , "t=100#ms",_
 2588, 53, 143, 0,3.5,       1,-50630,     0            , "t=130ms",_
 2589, 54, 143, 0,2.5,      -1,-60203,     0            , "t=511ms",_
 2590, 55, 143, 0,1.5,       1,-67676,     0            , "t=1.791s",_
 2591, 56, 143, 0,2.5,      -1,-73937,     0            , "t=14.5s",_
 2592, 57, 143, 0,3.5,       1,-78172,     0            , "t=14.2m",_
 2593, 58, 143, 0,1.5,      -1,-81606.7,   0            , "t=33.039h",_
 2594, 59, 143, 0,3.5,       1,-83068.2,   0            , "t=13.57d",_
 2595, 60, 143, 0,3.5,      -1,-84002.2,   0            , "t=stbl",_
 2596, 61, 143, 0,2.5,       1,-82960.7,   0            , "t=265d",_
 2597, 62, 143, 0,1.5,       1,-79517.2,   0            , "t=8.75m",_
 2598, 62, 143, 1,5.5,      -1,-78763.2,   0.75399      , "t=66s",_
 2599, 62, 143, 2,11.5,     -1,-76723,     2.7938       , "t=30ms",_
 2600, 63, 143, 0,2.5,       1,-74241,     0            , "t=2.59m",_
 2601, 63, 143, 1,5.5,      -1,-73851,     0.38951      , "t=50.0us",_
 2602, 64, 143, 0,0.5,       1,-68230,     0            , "t=39s",_
 2603, 64, 143, 1,5.5,      -1,-68080,     0.1526       , "t=110.0s",_
 2604, 65, 143, 0,5.5,      -1,-60420,     0            , "t=12s",_
 2605, 65, 143, 1,2.5,       1,-60420,     0            , "t=?s",_
 2606, 66, 143, 0,0.5,       1,-52169,     0            , "t=5.6s",_
 2607, 66, 143, 1,5.5,      -1,-51858,     0.3107       , "t=3.0s",_
 2608, 66, 143, 2,-77.777,   0,-51763,     0.4063       , "t=1.2us",_
 2609, 67, 143, 0,5.5,      -1,-42050,     0            , "t=300#ms",_
 2610, 68, 143, 0,4.5,      -1,-31260,     0            , "t=200#ms",_
 2611, 53, 144, 0,1,        -1,-45280,     0            , "t=100#ms",_
 2612, 54, 144, 0,0,         1,-56872,     0            , "t=388ms",_
 2613, 55, 144, 0,1,        -1,-63271,     0            , "t=994ms",_
 2614, 55, 144, 1,4,        -1,-63179,     0.0922       , "t=1.1us",_
 2615, 55, 144, 2,0,         0,-62970,     0.3          , "t=&lt;1s",_
 2616, 56, 144, 0,0,         1,-71767,     0            , "t=11.5s",_
 2617, 57, 144, 0,3,        -1,-74850,     0            , "t=40.8s",_
 2618, 58, 144, 0,0,         1,-80431.9,   0            , "t=284.91d",_
 2619, 59, 144, 0,0,        -1,-80750.5,   0            , "t=17.28m",_
 2620, 59, 144, 1,3,        -1,-80691.5,   0.05903      , "t=7.2m",_
 2621, 60, 144, 0,0,         1,-83748.0,   0            , "t=2.29Py",_
 2622, 61, 144, 0,5,        -1,-81416.1,   0            , "t=363d",_
 2623, 61, 144, 1,9,         1,-80575,     0.8409       , "t=780ns",_
 2624, 61, 144, 2,27,        1,-72820,     8.5958       , "t=~2.7us",_
 2625, 62, 144, 0,0,         1,-81965.5,   0            , "t=stbl",_
 2626, 62, 144, 1,6,         1,-79641.9,   2.3236       , "t=880ns",_
 2627, 63, 144, 0,1,         1,-75619,     0            , "t=10.2s",_
 2628, 63, 144, 1,8,        -1,-74491,     1.1276       , "t=1.0us",_
 2629, 64, 144, 0,0,         1,-71760,     0            , "t=4.47m",_
 2630, 64, 144, 1,10,        1,-68327,     3.4331       , "t=145ns",_
 2631, 65, 144, 0,1,         1,-62368,     0            , "t=~1s",_
 2632, 65, 144, 1,6,        -1,-61971,     0.3969       , "t=4.25s",_
 2633, 65, 144, 2,8,        -1,-61892,     0.4762       , "t=2.8us",_
 2634, 66, 144, 0,0,         1,-56570,     0            , "t=9.1s",_
 2635, 67, 144, 0,5,        -1,-44610,     0            , "t=700ms",_
 2636, 67, 144, 1,8,         1,-44345,     0.2653       , "t=519ns",_
 2637, 68, 144, 0,0,         1,-36610,     0            , "t=400#ms",_
 2638, 69, 144, 0,10,        1,-22260,     0            , "t=2.3us",_
 2639, 53, 145, 0,3.5,       1,-40940,     0            , "t=100#ms",_
 2640, 54, 145, 0,1.5,      -1,-51493,     0            , "t=188ms",_
 2641, 55, 145, 0,1.5,       1,-60054,     0            , "t=582ms",_
 2642, 55, 145, 1,9.5,      -1,-59291,     0.7629       , "t=500ns",_
 2643, 56, 145, 0,2.5,      -1,-67516,     0            , "t=4.31s",_
 2644, 57, 145, 0,2.5,       1,-72835,     0            , "t=24.8s",_
 2645, 58, 145, 0,2.5,      -1,-77070,     0            , "t=3.01m",_
 2646, 59, 145, 0,3.5,       1,-79626,     0            , "t=5.984h",_
 2647, 60, 145, 0,3.5,      -1,-81432.0,   0            , "t=stbl",_
 2648, 61, 145, 0,2.5,       1,-81267.5,   0            , "t=17.7y",_
 2649, 62, 145, 0,3.5,      -1,-80651.3,   0            , "t=340d",_
 2650, 62, 145, 1,24.5,      1,-71865.1,   8.7862       , "t=990ns",_
 2651, 63, 145, 0,2.5,       1,-77992,     0            , "t=5.93d",_
 2652, 63, 145, 1,5.5,      -1,-77276,     0.716        , "t=490ns",_
 2653, 64, 145, 0,0.5,       1,-72926,     0            , "t=23.0m",_
 2654, 64, 145, 1,5.5,      -1,-72177,     0.7491       , "t=85s",_
 2655, 65, 145, 0,5.5,      -1,-66390,     0            , "t=30.9s",_
 2656, 65, 145, 1,1.5,       1,-65540,     0.85         , "t=?",_
 2657, 66, 145, 0,0.5,       1,-58243,     0            , "t=9.5s",_
 2658, 66, 145, 1,5.5,      -1,-58125,     0.1182       , "t=14.1s",_
 2659, 67, 145, 0,5.5,      -1,-49120,     0            , "t=2.4s",_
 2660, 67, 145, 1,2.5,       1,-49020,     0.1          , "t=100#ms",_
 2661, 68, 145, 0,0.5,       1,-39240,     0            , "t=900ms",_
 2662, 68, 145, 1,5.5,      -1,-39040,     0.205        , "t=1.0s",_
 2663, 69, 145, 0,5.5,      -1,-27580,     0            , "t=3.17us",_
 2664, 54, 146, 0,0,         1,-47955,     0            , "t=146ms",_
 2665, 55, 146, 0,1,        -1,-55310.4,   0            , "t=323ms",_
 2666, 55, 146, 1,4,        -1,-55263.7,   0.0467       , "t=1.25us",_
 2667, 56, 146, 0,0,         1,-64947,     0            , "t=2.22s",_
 2668, 57, 146, 0,2,        -1,-69050,     0            , "t=6.27s",_
 2669, 57, 146, 1,6,        -1,-68920,     0.13         , "t=10.0s",_
 2670, 58, 146, 0,0,         1,-75635,     0            , "t=13.52m",_
 2671, 59, 146, 0,2,        -1,-76680,     0            , "t=24.15m",_
 2672, 60, 146, 0,0,         1,-80925.9,   0            , "t=stbl",_
 2673, 61, 146, 0,3,        -1,-79454,     0            , "t=5.53y",_
 2674, 62, 146, 0,0,         1,-80996,     0            , "t=68My",_
 2675, 63, 146, 0,4,        -1,-77118,     0            , "t=4.61d",_
 2676, 63, 146, 1,9,         1,-76452,     0.66637      , "t=235us",_
 2677, 64, 146, 0,0,         1,-76086,     0            , "t=48.27d",_
 2678, 65, 146, 0,1,         1,-67760,     0            , "t=8s",_
 2679, 65, 146, 1,5,        -1,-67610,     0.15         , "t=24.1s",_
 2680, 65, 146, 2,10,        1,-66830,     0.93         , "t=1.18ms",_
 2681, 66, 146, 0,0,         1,-62555,     0            , "t=33.2s",_
 2682, 66, 146, 1,10,        1,-59619,     2.9357       , "t=150ms",_
 2683, 67, 146, 0,6,        -1,-51238,     0            , "t=2.8s",_
 2684, 68, 146, 0,0,         1,-44322,     0            , "t=1.7s",_
 2685, 69, 146, 0,1,         1,-31060,     0            , "t=155ms",_
 2686, 69, 146, 1,5,        -1,-30750,     0.304        , "t=75ms",_
 2687, 69, 146, 2,10,        1,-30620,     0.437        , "t=200ms",_
 2688, 54, 147, 0,1.5,      -1,-42360,     0            , "t=130ms",_
 2689, 55, 147, 0,1.5,       1,-51920,     0            , "t=230ms",_
 2690, 55, 147, 1,9.5,      -1,-51219,     0.7014       , "t=190ns",_
 2691, 56, 147, 0,2.5,      -1,-60264,     0            , "t=894ms",_
 2692, 57, 147, 0,2.5,       1,-66678,     0            , "t=4.06s",_
 2693, 58, 147, 0,2.5,      -1,-72014,     0            , "t=56.4s",_
 2694, 59, 147, 0,1.5,       1,-75444,     0            , "t=13.4m",_
 2695, 60, 147, 0,2.5,      -1,-78146.7,   0            , "t=10.98d",_
 2696, 61, 147, 0,3.5,       1,-79042.3,   0            , "t=2.6234y",_
 2697, 62, 147, 0,3.5,      -1,-79266.4,   0            , "t=106.6Gy",_
 2698, 63, 147, 0,2.5,       1,-77544.8,   0            , "t=24.1d",_
 2699, 63, 147, 1,5.5,      -1,-76919.5,   0.62527      , "t=765ns",_
 2700, 64, 147, 0,3.5,      -1,-75356.9,   0            , "t=38.06h",_
 2701, 64, 147, 1,24.5,      1,-66769.1,   8.5878       , "t=510ns",_
 2702, 65, 147, 0,0.5,       1,-70743,     0            , "t=1.64h",_
 2703, 65, 147, 1,5.5,      -1,-70692,     0.0506       , "t=1.87m",_
 2704, 66, 147, 0,0.5,       1,-64196,     0            , "t=67s",_
 2705, 66, 147, 1,5.5,      -1,-63446,     0.7505       , "t=55.2s",_
 2706, 66, 147, 2,13.5,     -1,-60789,     3.4072       , "t=400ns",_
 2707, 67, 147, 0,5.5,      -1,-55757,     0            , "t=5.8s",_
 2708, 67, 147, 1,13.5,     -1,-53070,     2.6871       , "t=315ns",_
 2709, 68, 147, 0,0.5,       1,-46610,     0            , "t=3.2s",_
 2710, 68, 147, 1,5.5,      -1,-46510,     0.1          , "t=1.6s",_
 2711, 69, 147, 0,5.5,      -1,-35974,     0            , "t=580ms",_
 2712, 69, 147, 1,1.5,       1,-35913,     0.062        , "t=360us",_
 2713, 54, 148, 0,0,         1,-38600,     0            , "t=100#ms",_
 2714, 55, 148, 0,-77.777,   0,-46911,     0            , "t=145ms",_
 2715, 55, 148, 1,4,        -1,-46866,     0.0452       , "t=4.8us",_
 2716, 56, 148, 0,0,         1,-57590,     0            , "t=620ms",_
 2717, 57, 148, 0,2,        -1,-62709,     0            , "t=1.35s",_
 2718, 58, 148, 0,0,         1,-70398,     0            , "t=56.8s",_
 2719, 59, 148, 0,1,        -1,-72535,     0            , "t=2.29m",_
 2720, 59, 148, 1,4,         0,-72458,     0.0768       , "t=2.01m",_
 2721, 60, 148, 0,0,         1,-77408.0,   0            , "t=stbl",_
 2722, 61, 148, 0,1,        -1,-76866,     0            , "t=5.368d",_
 2723, 61, 148, 1,5,        -1,-76728,     0.1379       , "t=41.29d",_
 2724, 62, 148, 0,0,         1,-79336.3,   0            , "t=6.3Py",_
 2725, 63, 148, 0,5,        -1,-76299,     0            , "t=54.5d",_
 2726, 63, 148, 1,9,         1,-75579,     0.7204       , "t=162ns",_
 2727, 64, 148, 0,0,         1,-76269.3,   0            , "t=70.9y",_
 2728, 65, 148, 0,2,        -1,-70537,     0            , "t=60m",_
 2729, 65, 148, 1,9,         1,-70447,     0.0901       , "t=2.20m",_
 2730, 65, 148, 2,27,        1,-61918,     8.6186       , "t=1.310us",_
 2731, 66, 148, 0,0,         1,-67860,     0            , "t=3.3m",_
 2732, 66, 148, 1,10,        1,-64941,     2.9191       , "t=471ns",_
 2733, 67, 148, 0,1,         1,-57990,     0            , "t=2.2s",_
 2734, 67, 148, 1,5,        -1,-57740,     0.25         , "t=9.49s",_
 2735, 67, 148, 2,10,        1,-57050,     0.94         , "t=2.36ms",_
 2736, 68, 148, 0,0,         1,-51479,     0            , "t=4.6s",_
 2737, 68, 148, 1,10,        1,-48566,     2.9132       , "t=13us",_
 2738, 69, 148, 0,10,        1,-38765,     0            , "t=700ms",_
 2739, 70, 148, 0,0,         1,-30330,     0            , "t=250#ms",_
 2740, 55, 149, 0,1.5,       1,-43250,     0            , "t=113ms",_
 2741, 56, 149, 0,1.5,      -1,-53120,     0            , "t=348ms",_
 2742, 57, 149, 0,1.5,      -1,-60220,     0            , "t=1.07s",_
 2743, 58, 149, 0,1.5,      -1,-66670,     0            , "t=4.94s",_
 2744, 59, 149, 0,2.5,       1,-71039,     0            , "t=2.26m",_
 2745, 60, 149, 0,2.5,      -1,-74375.5,   0            , "t=1.728h",_
 2746, 61, 149, 0,3.5,       1,-76064.3,   0            , "t=53.08h",_
 2747, 61, 149, 1,5.5,      -1,-75824.1,   0.240214     , "t=35us",_
 2748, 62, 149, 0,3.5,      -1,-77135.7,   0            , "t=stbl",_
 2749, 63, 149, 0,2.5,       1,-76441,     0            , "t=93.1d",_
 2750, 63, 149, 1,5.5,      -1,-75945,     0.496386     , "t=2.45us",_
 2751, 64, 149, 0,3.5,      -1,-75127,     0            , "t=9.28d",_
 2752, 65, 149, 0,0.5,       1,-71489,     0            , "t=4.118h",_
 2753, 65, 149, 1,5.5,      -1,-71453,     0.03578      , "t=4.16m",_
 2754, 66, 149, 0,3.5,      -1,-67696,     0            , "t=4.20m",_
 2755, 66, 149, 1,13.5,     -1,-65035,     2.6611       , "t=490ms",_
 2756, 67, 149, 0,5.5,      -1,-61647,     0            , "t=21.1s",_
 2757, 67, 149, 1,0.5,       1,-61598,     0.0488       , "t=56s",_
 2758, 68, 149, 0,0.5,       1,-53742,     0            , "t=4s",_
 2759, 68, 149, 1,5.5,      -1,-53000,     0.7418       , "t=8.9s",_
 2760, 68, 149, 2,9.5,       1,-51131,     2.6111       , "t=610ns",_
 2761, 69, 149, 0,5.5,      -1,-43880,     0            , "t=900ms",_
 2762, 70, 149, 0,0.5,       1,-33200,     0            , "t=700ms",_
 2763, 55, 150, 0,-77.777,   0,-38170,     0            , "t=84.4ms",_
 2764, 56, 150, 0,0,         1,-49900,     0            , "t=259ms",_
 2765, 57, 150, 0,3,         1,-56130,     0            , "t=504ms",_
 2766, 58, 150, 0,0,         1,-64847,     0            , "t=6.05s",_
 2767, 59, 150, 0,1,        -1,-68300,     0            , "t=6.19s",_
 2768, 60, 150, 0,0,         1,-73679.8,   0            , "t=8.2Ey",_
 2769, 61, 150, 0,1,        -1,-73597,     0            , "t=2.698h",_
 2770, 62, 150, 0,0,         1,-77051.1,   0            , "t=stbl",_
 2771, 63, 150, 0,5,        -1,-74792,     0            , "t=36.9y",_
 2772, 63, 150, 1,0,        -1,-74750,     0.0417       , "t=12.8h",_
 2773, 64, 150, 0,0,         1,-75764,     0            , "t=1.79My",_
 2774, 65, 150, 0,2,        -1,-71106,     0            , "t=3.48h",_
 2775, 65, 150, 1,9,         1,-70645,     0.461        , "t=5.8m",_
 2776, 66, 150, 0,0,         1,-69310,     0            , "t=7.17m",_
 2777, 67, 150, 0,2,        -1,-61946,     0            , "t=76.8s",_
 2778, 67, 150, 1,9,         1,-61950,    -0            , "t=23.3s",_
 2779, 67, 150, 2,28,       -1,-54050,     7.9          , "t=787ns",_
 2780, 68, 150, 0,0,         1,-57831,     0            , "t=18.5s",_
 2781, 68, 150, 1,10,        1,-55035,     2.7965       , "t=2.55us",_
 2782, 69, 150, 0,1,         1,-46490,     0            , "t=3#s",_
 2783, 69, 150, 1,6,        -1,-46350,     0.14         , "t=2.20s",_
 2784, 69, 150, 2,10,        1,-45680,     0.81         , "t=5.2ms",_
 2785, 70, 150, 0,0,         1,-38640,     0            , "t=700#ms",_
 2786, 71, 150, 0,5,        -1,-24640,     0            , "t=45ms",_
 2787, 71, 150, 1,1,         1,-24620,     0.022        , "t=40us",_
 2788, 55, 151, 0,1.5,       1,-34230,     0            , "t=69ms",_
 2789, 56, 151, 0,1.5,      -1,-44940,     0            , "t=167ms",_
 2790, 57, 151, 0,2.5,       1,-53310,     0            , "t=465ms",_
 2791, 58, 151, 0,1.5,      -1,-61225,     0            , "t=1.76s",_
 2792, 59, 151, 0,1.5,      -1,-66780,     0            , "t=18.90s",_
 2793, 59, 151, 1,3.5,       1,-66745,     0.0351       , "t=50us",_
 2794, 60, 151, 0,1.5,       1,-70943.0,   0            , "t=12.44m",_
 2795, 61, 151, 0,2.5,       1,-73386,     0            , "t=28.40h",_
 2796, 62, 151, 0,2.5,      -1,-74576.3,   0            , "t=90y",_
 2797, 62, 151, 1,5.5,      -1,-74315.2,   0.26113      , "t=1.4us",_
 2798, 63, 151, 0,2.5,       1,-74652.9,   0            , "t=4.6Ey",_
 2799, 63, 151, 1,5.5,      -1,-74456.7,   0.196245     , "t=58.9us",_
 2800, 64, 151, 0,3.5,      -1,-74189,     0            , "t=123.9d",_
 2801, 65, 151, 0,0.5,       1,-71624,     0            , "t=17.609h",_
 2802, 65, 151, 1,5.5,      -1,-71524,     0.09953      , "t=25s",_
 2803, 66, 151, 0,3.5,      -1,-68752,     0            , "t=17.9m",_
 2804, 67, 151, 0,5.5,      -1,-63623,     0            , "t=35.2s",_
 2805, 67, 151, 1,0.5,       1,-63582,     0.041        , "t=47.2s",_
 2806, 68, 151, 0,3.5,      -1,-58266,     0            , "t=23.5s",_
 2807, 68, 151, 1,13.5,     -1,-55680,     2.586        , "t=580ms",_
 2808, 68, 151, 2,32.5,     -1,-47979,     10.2866      , "t=420ns",_
 2809, 69, 151, 0,5.5,      -1,-50773,     0            , "t=4.17s",_
 2810, 69, 151, 1,0.5,       1,-50679,     0.094        , "t=6.6s",_
 2811, 69, 151, 2,13.5,     -1,-48117,     2.65567      , "t=451ns",_
 2812, 70, 151, 0,0.5,       1,-41540,     0            , "t=1.6s",_
 2813, 70, 151, 1,5.5,      -1,-40790,     0.75         , "t=1.6s",_
 2814, 70, 151, 2,9.5,      -1,-39000,     2.54         , "t=2.6us",_
 2815, 71, 151, 0,5.5,      -1,-30110,     0            , "t=78.4ms",_
 2816, 71, 151, 1,1.5,       1,-30060,     0.053        , "t=16.5us",_
 2817, 55, 152, 0,-77.777,   0,-28930,     0            , "t=30#ms",_
 2818, 56, 152, 0,0,         1,-41710,     0            , "t=139ms",_
 2819, 57, 152, 0,-77.777,   0,-49290,     0            , "t=287ms",_
 2820, 58, 152, 0,0,         1,-58980,     0            , "t=1.42s",_
 2821, 59, 152, 0,4,         1,-63758,     0            , "t=3.57s",_
 2822, 59, 152, 1,3,         1,-63643,     0.1148       , "t=4.1us",_
 2823, 60, 152, 0,0,         1,-70149,     0            , "t=11.4m",_
 2824, 61, 152, 0,1,         1,-71254,     0            , "t=4.12m",_
 2825, 61, 152, 1,4,        -1,-71110,     0.14         , "t=7.52m",_
 2826, 61, 152, 2,8,         0,-71000,     0.25         , "t=13.8m",_
 2827, 62, 152, 0,0,         1,-74762.6,   0            , "t=stbl",_
 2828, 63, 152, 0,3,        -1,-72888.3,   0            , "t=13.517y",_
 2829, 63, 152, 1,0,        -1,-72842.7,   0.0455998    , "t=9.3116h",_
 2830, 63, 152, 2,1,        -1,-72823.0,   0.0652969    , "t=940ns",_
 2831, 64, 152, 0,0,         1,-74706.9,   0            , "t=108Ty",_
 2832, 65, 152, 0,2,        -1,-70720,     0            , "t=17.5h",_
 2833, 65, 152, 1,5,        -1,-70380,     0.34215      , "t=960ns",_
 2834, 65, 152, 2,8,         1,-70220,     0.50174      , "t=4.2m",_
 2835, 66, 152, 0,0,         1,-70118,     0            , "t=2.38h",_
 2836, 67, 152, 0,2,        -1,-63605,     0            , "t=161.8s",_
 2837, 67, 152, 1,9,         1,-63445,     0.16         , "t=49.8s",_
 2838, 67, 152, 2,19,       -1,-60585,     3.01959      , "t=8.4us",_
 2839, 68, 152, 0,0,         1,-60500,     0            , "t=10.3s",_
 2840, 69, 152, 0,2,        -1,-51720,     0            , "t=8.0s",_
 2841, 69, 152, 1,9,         1,-51820,    -0.1          , "t=5.2s",_
 2842, 69, 152, 2,17,        1,-49060,     2.665        , "t=294ns",_
 2843, 70, 152, 0,0,         1,-46270,     0            , "t=3.03s",_
 2844, 70, 152, 1,10,        1,-43530,     2.7445       , "t=30us",_
 2845, 71, 152, 0,4,        -1,-33420,     0            , "t=650ms",_
 2846, 56, 153, 0,2.5,      -1,-36470,     0            , "t=116ms",_
 2847, 57, 153, 0,2.5,       1,-46060,     0            , "t=245ms",_
 2848, 58, 153, 0,1.5,      -1,-54910,     0            , "t=865ms",_
 2849, 59, 153, 0,2.5,      -1,-61568,     0            , "t=4.28s",_
 2850, 60, 153, 0,1.5,      -1,-67330.3,   0            , "t=31.6s",_
 2851, 60, 153, 1,2.5,       1,-67138.6,   0.1917       , "t=1.10us",_
 2852, 61, 153, 0,2.5,      -1,-70648,     0            , "t=5.25m",_
 2853, 62, 153, 0,1.5,       1,-72559.7,   0            , "t=46.284h",_
 2854, 62, 153, 1,5.5,      -1,-72461.3,   0.09837      , "t=10.6ms",_
 2855, 63, 153, 0,2.5,       1,-73367.2,   0            , "t=stbl",_
 2856, 63, 153, 1,9.5,      -1,-71596.2,   1.771        , "t=475ns",_
 2857, 64, 153, 0,1.5,      -1,-72882.6,   0            , "t=240.4d",_
 2858, 64, 153, 1,4.5,       1,-72787.4,   0.0951736    , "t=3.5us",_
 2859, 64, 153, 2,5.5,      -1,-72711.4,   0.171188     , "t=76.0us",_
 2860, 65, 153, 0,2.5,       1,-71313,     0            , "t=2.34d",_
 2861, 65, 153, 1,5.5,      -1,-71150,     0.163175     , "t=186us",_
 2862, 66, 153, 0,3.5,      -1,-69143,     0            , "t=6.4h",_
 2863, 67, 153, 0,5.5,      -1,-65012,     0            , "t=2.01m",_
 2864, 67, 153, 1,0.5,       1,-64943,     0.0687       , "t=9.3m",_
 2865, 67, 153, 2,15.5,      1,-62240,     2.772        , "t=229ns",_
 2866, 68, 153, 0,3.5,      -1,-60469,     0            , "t=37.1s",_
 2867, 68, 153, 1,13.5,     -1,-57671,     2.7982       , "t=373ns",_
 2868, 68, 153, 2,20.5,     -1,-55221,     5.2481       , "t=248ns",_
 2869, 69, 153, 0,5.5,      -1,-53973,     0            , "t=1.48s",_
 2870, 69, 153, 1,0.5,       1,-53930,     0.0432       , "t=2.5s",_
 2871, 70, 153, 0,3.5,      -1,-47210,     0            , "t=4.2s",_
 2872, 70, 153, 1,13.5,     -1,-44510,     2.7          , "t=15us",_
 2873, 71, 153, 0,5.5,      -1,-38370,     0            , "t=900ms",_
 2874, 71, 153, 1,0.5,       1,-38290,     0.08         , "t=1#s",_
 2875, 71, 153, 2,11.5,     -1,-35870,     2.5025       , "t=&gt;100ns",_
 2876, 72, 153, 0,0.5,       1,-27300,     0            , "t=400#ms",_
 2877, 72, 153, 1,5.5,      -1,-26550,     0.75         , "t=500#ms",_
 2878, 56, 154, 0,0,         1,-32820,     0            , "t=53ms",_
 2879, 57, 154, 0,-77.777,   0,-41530,     0            , "t=161ms",_
 2880, 58, 154, 0,0,         1,-52220,     0            , "t=722ms",_
 2881, 59, 154, 0,3,         1,-58100,     0            , "t=2.3s",_
 2882, 60, 154, 0,0,         1,-65820,     0            , "t=25.9s",_
 2883, 60, 154, 1,4,        -1,-64520,     1.2979       , "t=3.2us",_
 2884, 61, 154, 0,4,         1,-68510,     0            , "t=2.68m",_
 2885, 61, 154, 1,1,        -1,-68490,     0.02         , "t=1.73m",_
 2886, 62, 154, 0,0,         1,-72455.2,   0            , "t=stbl",_
 2887, 63, 154, 0,3,        -1,-71738.1,   0            , "t=8.601y",_
 2888, 63, 154, 1,2,         1,-71669.9,   0.0681702    , "t=2.2us",_
 2889, 63, 154, 2,8,        -1,-71592.8,   0.1453       , "t=46.3m",_
 2890, 64, 154, 0,0,         1,-73706.0,   0            , "t=stbl",_
 2891, 65, 154, 0,0,         1,-70160,     0            , "t=21.5h",_
 2892, 65, 154, 1,3,        -1,-70150,     0.012        , "t=9.994h",_
 2893, 65, 154, 2,7,        -1,-69960,     0.2          , "t=22.7h",_
 2894, 66, 154, 0,0,         1,-70394,     0            , "t=3.0My",_
 2895, 67, 154, 0,2,        -1,-64639,     0            , "t=11.76m",_
 2896, 67, 154, 1,8,         1,-64397,     0.243        , "t=3.10m",_
 2897, 68, 154, 0,0,         1,-62605,     0            , "t=3.73m",_
 2898, 69, 154, 0,2,        -1,-54427,     0            , "t=8.1s",_
 2899, 69, 154, 1,9,         1,-54350,     0.07         , "t=3.30s",_
 2900, 70, 154, 0,0,         1,-49932,     0            , "t=409ms",_
 2901, 71, 154, 0,2,        -1,-39720,     0            , "t=1#s",_
 2902, 71, 154, 1,9,         1,-39660,     0.06         , "t=1.12s",_
 2903, 71, 154, 2,17,        1,-37000,     2.72         , "t=35us",_
 2904, 72, 154, 0,0,         1,-32670,     0            , "t=2s",_
 2905, 72, 154, 1,10,        1,-29960,     2.71         , "t=9us",_
 2906, 57, 155, 0,2.5,       1,-37930,     0            , "t=101ms",_
 2907, 58, 155, 0,2.5,      -1,-47780,     0            , "t=313ms",_
 2908, 59, 155, 0,2.5,      -1,-55415,     0            , "t=1.47s",_
 2909, 60, 155, 0,1.5,      -1,-62284,     0            , "t=8.9s",_
 2910, 61, 155, 0,2.5,      -1,-66940,     0            , "t=41.5s",_
 2911, 62, 155, 0,1.5,      -1,-70190.8,   0            , "t=22.3m",_
 2912, 62, 155, 1,2.5,       1,-70174.3,   0.0165       , "t=2.8us",_
 2913, 62, 155, 2,5.5,      -1,-69652.2,   0.5386       , "t=1.00us",_
 2914, 63, 155, 0,2.5,       1,-71818.1,   0            , "t=4.741y",_
 2915, 64, 155, 0,1.5,      -1,-72069.9,   0            , "t=stbl",_
 2916, 64, 155, 1,5.5,      -1,-71948.8,   0.12105      , "t=31.97ms",_
 2917, 65, 155, 0,1.5,       1,-71250,     0            , "t=5.32d",_
 2918, 66, 155, 0,1.5,      -1,-69156,     0            , "t=9.9h",_
 2919, 66, 155, 1,5.5,      -1,-68922,     0.23433      , "t=6us",_
 2920, 67, 155, 0,2.5,       1,-66040,     0            , "t=48m",_
 2921, 67, 155, 1,5.5,      -1,-65898,     0.14197      , "t=880us",_
 2922, 68, 155, 0,3.5,      -1,-62209,     0            , "t=5.3m",_
 2923, 69, 155, 0,5.5,      -1,-56626,     0            , "t=21.6s",_
 2924, 69, 155, 1,0.5,       1,-56585,     0.041        , "t=45s",_
 2925, 70, 155, 0,3.5,      -1,-50503,     0            , "t=1.793s",_
 2926, 71, 155, 0,5.5,      -1,-42545,     0            , "t=68.6ms",_
 2927, 71, 155, 1,0.5,       1,-42524,     0.021        , "t=138ms",_
 2928, 71, 155, 2,12.5,     -1,-40764,     1.781        , "t=2.69ms",_
 2929, 72, 155, 0,3.5,      -1,-34170,     0            , "t=840ms",_
 2930, 73, 155, 0,5.5,      -1,-23930,     0            , "t=3.2ms",_
 2931, 57, 156, 0,-77.777,   0,-33050,     0            , "t=84ms",_
 2932, 58, 156, 0,0,         1,-44820,     0            , "t=233ms",_
 2933, 59, 156, 0,-77.777,   0,-51570,     0            , "t=444ms",_
 2934, 60, 156, 0,0,         1,-60470,     0            , "t=5.06s",_
 2935, 60, 156, 1,5,        -1,-59040,     1.4313       , "t=365ns",_
 2936, 61, 156, 0,4,         1,-64164,     0            , "t=27.2s",_
 2937, 61, 156, 1,1,         1,-64014,     0.1503       , "t=5.6s",_
 2938, 62, 156, 0,0,         1,-69360,     0            , "t=9.4h",_
 2939, 62, 156, 1,5,        -1,-67962,     1.39755      , "t=185ns",_
 2940, 63, 156, 0,0,         1,-70083,     0            , "t=15.19d",_
 2941, 64, 156, 0,0,         1,-72534.9,   0            , "t=stbl",_
 2942, 64, 156, 1,7,        -1,-70397.3,   2.1376       , "t=1.3us",_
 2943, 65, 156, 0,3,        -1,-70091,     0            , "t=5.35d",_
 2944, 65, 156, 1,7,        -1,-70037,     0.054        , "t=24.4h",_
 2945, 65, 156, 2,0,         1,-70003,     0.0884       , "t=5.3h",_
 2946, 66, 156, 0,0,         1,-70529.0,   0            , "t=stbl",_
 2947, 67, 156, 0,4,        -1,-65480,     0            , "t=56m",_
 2948, 67, 156, 1,1,        -1,-65430,     0.05237      , "t=9.5s",_
 2949, 67, 156, 2,9,         1,-65304,     0.17         , "t=7.6m",_
 2950, 68, 156, 0,0,         1,-64212,     0            , "t=19.5m",_
 2951, 69, 156, 0,2,        -1,-56835,     0            , "t=83.8s",_
 2952, 69, 156, 1,11,       -1,-56440,     0.4          , "t=~400ns",_
 2953, 70, 156, 0,0,         1,-53266,     0            , "t=26.1s",_
 2954, 71, 156, 0,2,        -1,-43700,     0            , "t=494ms",_
 2955, 71, 156, 1,9,         1,-43680,     0.02         , "t=198ms",_
 2956, 72, 156, 0,0,         1,-37820,     0            , "t=23ms",_
 2957, 72, 156, 1,8,         1,-35860,     1.959        , "t=480us",_
 2958, 73, 156, 0,2,        -1,-25860,     0            , "t=106ms",_
 2959, 73, 156, 1,9,         1,-25770,     0.094        , "t=360ms",_
 2960, 58, 157, 0,3.5,       1,-39930,     0            , "t=175ms",_
 2961, 59, 157, 0,2.5,      -1,-48540,     0            , "t=307ms",_
 2962, 60, 157, 0,2.5,      -1,-56462,     0            , "t=1.15s",_
 2963, 61, 157, 0,2.5,      -1,-62297,     0            , "t=10.56s",_
 2964, 62, 157, 0,1.5,      -1,-66678,     0            , "t=8.03m",_
 2965, 63, 157, 0,2.5,       1,-69459,     0            , "t=15.18h",_
 2966, 64, 157, 0,1.5,      -1,-70823.5,   0            , "t=stbl",_
 2967, 64, 157, 1,2.5,       1,-70759.6,   0.063916     , "t=460ns",_
 2968, 64, 157, 2,5.5,      -1,-70397.0,   0.426539     , "t=18.5us",_
 2969, 65, 157, 0,1.5,       1,-70763.4,   0            , "t=71y",_
 2970, 66, 157, 0,1.5,      -1,-69425,     0            , "t=8.14h",_
 2971, 66, 157, 1,4.5,       1,-69263,     0.16199      , "t=1.3us",_
 2972, 66, 157, 2,5.5,      -1,-69226,     0.19938      , "t=21.6ms",_
 2973, 67, 157, 0,3.5,      -1,-66833,     0            , "t=12.6m",_
 2974, 68, 157, 0,1.5,      -1,-63414,     0            , "t=18.65m",_
 2975, 68, 157, 1,4.5,       1,-63259,     0.1554       , "t=76ms",_
 2976, 69, 157, 0,0.5,       1,-58709,     0            , "t=3.63m",_
 2977, 70, 157, 0,3.5,      -1,-53422,     0            , "t=38.6s",_
 2978, 71, 157, 0,0.5,       1,-46441,     0            , "t=6.8s",_
 2979, 71, 157, 1,5.5,      -1,-46420,     0.0209       , "t=4.79s",_
 2980, 72, 157, 0,3.5,      -1,-38900,     0            , "t=115ms",_
 2981, 73, 157, 0,0.5,       1,-29590,     0            , "t=10.1ms",_
 2982, 73, 157, 1,5.5,      -1,-29570,     0.022        , "t=4.3ms",_
 2983, 73, 157, 2,12.5,     -1,-28000,     1.593        , "t=1.7ms",_
 2984, 74, 157, 0,3.5,      -1,-19470,     0            , "t=275ms",_
 2985, 58, 158, 0,0,         1,-36660,     0            , "t=99ms",_
 2986, 59, 158, 0,-77.777,   0,-44330,     0            , "t=181ms",_
 2987, 60, 158, 0,0,         1,-54060,     0            , "t=810ms",_
 2988, 61, 158, 0,-77.777,   0,-59089,     0            , "t=4.8s",_
 2989, 61, 158, 1,-77.777,   0,-58940,     0.15         , "t=&gt;16us",_
 2990, 62, 158, 0,0,         1,-65250,     0            , "t=5.30m",_
 2991, 62, 158, 1,5,        -1,-63971,     1.2791       , "t=115ns",_
 2992, 63, 158, 0,1,        -1,-67255,     0            , "t=45.9m",_
 2993, 64, 158, 0,0,         1,-70689.5,   0            , "t=stbl",_
 2994, 65, 158, 0,3,        -1,-69470.7,   0            , "t=180y",_
 2995, 65, 158, 1,0,        -1,-69360.4,   0.1103       , "t=10.70s",_
 2996, 65, 158, 2,7,        -1,-69082.3,   0.3884       , "t=400us",_
 2997, 66, 158, 0,0,         1,-70407.3,   0            , "t=stbl",_
 2998, 67, 158, 0,5,         1,-66188,     0            , "t=11.3m",_
 2999, 67, 158, 1,2,        -1,-66121,     0.067199     , "t=28m",_
 3000, 67, 158, 2,9,         1,-66010,     0.18         , "t=21.3m",_
 3001, 68, 158, 0,0,         1,-65304,     0            , "t=2.29h",_
 3002, 69, 158, 0,2,        -1,-58703,     0            , "t=3.98m",_
 3003, 69, 158, 1,5,         1,-58650,     0.05         , "t=~20ns",_
 3004, 70, 158, 0,0,         1,-56010,     0            , "t=1.49m",_
 3005, 71, 158, 0,2,        -1,-47212,     0            , "t=10.6s",_
 3006, 72, 158, 0,0,         1,-42102,     0            , "t=0.99s",_
 3007, 73, 158, 0,2,        -1,-31170,     0            , "t=49ms",_
 3008, 73, 158, 1,9,         1,-31030,     0.141        , "t=36.0ms",_
 3009, 73, 158, 2,19,       -1,-28360,     2.805        , "t=6.1us",_
 3010, 74, 158, 0,0,         1,-23630,     0            , "t=1.25ms",_
 3011, 74, 158, 1,8,         1,-21740,     1.889        , "t=143us",_
 3012, 59, 159, 0,2.5,      -1,-41090,     0            , "t=134ms",_
 3013, 60, 159, 0,3.5,       1,-49810,     0            , "t=500ms",_
 3014, 61, 159, 0,2.5,      -1,-56554,     0            , "t=1.49s",_
 3015, 61, 159, 1,-77.777,   0,-55089,     1.465        , "t=4.42us",_
 3016, 62, 159, 0,2.5,      -1,-62208,     0            , "t=11.37s",_
 3017, 62, 159, 1,5.5,      -1,-60931,     1.2768       , "t=116ns",_
 3018, 63, 159, 0,2.5,       1,-66043,     0            , "t=18.1m",_
 3019, 64, 159, 0,1.5,      -1,-68561.4,   0            , "t=18.479h",_
 3020, 65, 159, 0,1.5,       1,-69532.4,   0            , "t=stbl",_
 3021, 66, 159, 0,1.5,      -1,-69167.1,   0            , "t=144.4d",_
 3022, 66, 159, 1,5.5,      -1,-68814.3,   0.35277      , "t=122us",_
 3023, 67, 159, 0,3.5,      -1,-67330,     0            , "t=33.05m",_
 3024, 67, 159, 1,0.5,       1,-67124,     0.20591      , "t=8.30s",_
 3025, 68, 159, 0,1.5,      -1,-64561,     0            , "t=36m",_
 3026, 68, 159, 1,4.5,       1,-64378,     0.182602     , "t=337ns",_
 3027, 68, 159, 2,5.5,      -1,-64132,     0.42905      , "t=590ns",_
 3028, 69, 159, 0,2.5,       1,-60570,     0            , "t=9.13m",_
 3029, 70, 159, 0,2.5,      -1,-55839,     0            , "t=1.67m",_
 3030, 71, 159, 0,0.5,       1,-49710,     0            , "t=12.1s",_
 3031, 71, 159, 1,5.5,      -1,-49610,     0.1          , "t=10#s",_
 3032, 72, 159, 0,3.5,      -1,-42853,     0            , "t=5.20s",_
 3033, 73, 159, 0,0.5,       1,-34439,     0            , "t=1.04s",_
 3034, 73, 159, 1,5.5,      -1,-34375,     0.064        , "t=560ms",_
 3035, 74, 159, 0,3.5,      -1,-25300,     0            , "t=8.2ms",_
 3036, 75, 159, 0,0.5,       1,-14750,     0            , "t=40#us",_
 3037, 75, 159, 1,5.5,      -1,-14540,     0.21         , "t=21.6us",_
 3038, 59, 160, 0,-77.777,   0,-36520,     0            , "t=170ms",_
 3039, 60, 160, 0,0,         1,-47130,     0            , "t=439ms",_
 3040, 61, 160, 0,0,        -1,-53000,     0            , "t=725ms",_
 3041, 62, 160, 0,0,         1,-60235,     0            , "t=9.6s",_
 3042, 62, 160, 1,5,        -1,-58874,     1.3613       , "t=120ns",_
 3043, 62, 160, 2,11,        1,-57478,     2.7573       , "t=1.8us",_
 3044, 63, 160, 0,5,        -1,-63480,     0            , "t=42.4s",_
 3045, 63, 160, 1,1,        -1,-63400,     0.08         , "t=29.9s",_
 3046, 64, 160, 0,0,         1,-67941.7,   0            , "t=stbl",_
 3047, 65, 160, 0,3,        -1,-67836.3,   0            , "t=72.3d",_
 3048, 66, 160, 0,0,         1,-69672.7,   0            , "t=stbl",_
 3049, 67, 160, 0,5,         1,-66383,     0            , "t=25.6m",_
 3050, 67, 160, 1,2,        -1,-66323,     0.05998      , "t=5.02h",_
 3051, 67, 160, 2,9,         1,-66186,     0.197        , "t=3s",_
 3052, 68, 160, 0,0,         1,-66064,     0            , "t=28.58h",_
 3053, 69, 160, 0,1,        -1,-60300,     0            , "t=9.4m",_
 3054, 69, 160, 1,5,         1,-60230,     0.07         , "t=74.5s",_
 3055, 69, 160, 2,8,         0,-60200,     0.1          , "t=~200ns",_
 3056, 70, 160, 0,0,         1,-58163,     0            , "t=4.8m",_
 3057, 71, 160, 0,2,        -1,-50270,     0            , "t=36.1s",_
 3058, 71, 160, 1,-77.777,   0,-50270,     0            , "t=40s",_
 3059, 72, 160, 0,0,         1,-45939,     0            , "t=13.6s",_
 3060, 73, 160, 0,2,        -1,-35820,     0            , "t=1.70s",_
 3061, 73, 160, 1,9,         1,-35710,     0.11         , "t=1.55s",_
 3062, 74, 160, 0,0,         1,-29330,     0            , "t=90ms",_
 3063, 75, 160, 0,4,        -1,-16740,     0            , "t=611us",_
 3064, 75, 160, 1,9,         1,-16560,     0.182        , "t=2.8us",_
 3065, 60, 161, 0,0.5,      -1,-42590,     0            , "t=215ms",_
 3066, 61, 161, 0,2.5,      -1,-50240,     0            , "t=1.05s",_
 3067, 61, 161, 1,6.5,       1,-49270,     0.966        , "t=0.89us",_
 3068, 62, 161, 0,3.5,       1,-56672,     0            , "t=4.8s",_
 3069, 63, 161, 0,2.5,       1,-61792,     0            , "t=26.2s",_
 3070, 64, 161, 0,2.5,      -1,-65505.8,   0            , "t=3.646m",_
 3071, 65, 161, 0,1.5,       1,-67461.6,   0            , "t=6.89d",_
 3072, 66, 161, 0,2.5,       1,-68055.8,   0            , "t=stbl",_
 3073, 66, 161, 1,5.5,      -1,-67570.2,   0.48556      , "t=760ns",_
 3074, 67, 161, 0,3.5,      -1,-67197.3,   0            , "t=2.48h",_
 3075, 67, 161, 1,0.5,       1,-66986.2,   0.21115      , "t=6.76s",_
 3076, 68, 161, 0,1.5,      -1,-65202,     0            , "t=3.21h",_
 3077, 68, 161, 1,5.5,      -1,-64806,     0.39644      , "t=7.5us",_
 3078, 69, 161, 0,3.5,       1,-61899,     0            , "t=30.2m",_
 3079, 69, 161, 1,0.5,       1,-61891,     0.00751      , "t=5#m",_
 3080, 69, 161, 2,3.5,      -1,-61821,     0.0782       , "t=110ns",_
 3081, 70, 161, 0,1.5,      -1,-57839,     0            , "t=4.2m",_
 3082, 71, 161, 0,0.5,       1,-52562,     0            , "t=77s",_
 3083, 71, 161, 1,4.5,      -1,-52388,     0.174        , "t=7.3ms",_
 3084, 72, 161, 0,3.5,      -1,-46315,     0            , "t=18.4s",_
 3085, 72, 161, 1,6.5,       1,-45986,     0.329        , "t=4.8us",_
 3086, 73, 161, 0,0.5,       1,-38779,     0            , "t=3#s",_
 3087, 73, 161, 1,5.5,      -1,-38718,     0.061        , "t=3.08s",_
 3088, 74, 161, 0,3.5,      -1,-30560,     0            , "t=409ms",_
 3089, 75, 161, 0,0.5,       1,-20840,     0            , "t=440us",_
 3090, 75, 161, 1,5.5,      -1,-20720,     0.1237       , "t=14.7ms",_
 3091, 76, 161, 0,3.5,      -1,-9980,      0            , "t=640us",_
 3092, 60, 162, 0,0,         1,-39550,     0            , "t=310ms",_
 3093, 61, 162, 0,6,        -1,-46370,     0            , "t=630ms",_
 3094, 62, 162, 0,0,         1,-54530,     0            , "t=2.7s",_
 3095, 63, 162, 0,1,         1,-58700,     0            , "t=~11s",_
 3096, 63, 162, 1,6,         1,-58540,     0.157        , "t=7.5s",_
 3097, 64, 162, 0,0,         1,-64280,     0            , "t=8.4m",_
 3098, 65, 162, 0,1,        -1,-65680,     0            , "t=7.60m",_
 3099, 66, 162, 0,0,         1,-68181.5,   0            , "t=stbl",_
 3100, 66, 162, 1,8,         1,-65993.4,   2.1881       , "t=8.3us",_
 3101, 67, 162, 0,1,         1,-66042,     0            , "t=15.0m",_
 3102, 67, 162, 1,6,        -1,-65936,     0.10587      , "t=67.0m",_
 3103, 68, 162, 0,0,         1,-66334.5,   0            , "t=stbl",_
 3104, 68, 162, 1,7,        -1,-64308.5,   2.02601      , "t=88ns",_
 3105, 69, 162, 0,1,        -1,-61478,     0            , "t=21.70m",_
 3106, 69, 162, 1,5,         1,-61350,     0.13         , "t=24.3s",_
 3107, 70, 162, 0,0,         1,-59826,     0            , "t=18.87m",_
 3108, 71, 162, 0,1,        -1,-52830,     0            , "t=1.37m",_
 3109, 71, 162, 1,4,        -1,-52710,     0.12         , "t=1.5m",_
 3110, 71, 162, 2,-77.777,   0,-52530,     0.3          , "t=1.9m",_
 3111, 72, 162, 0,0,         1,-49169,     0            , "t=39.4s",_
 3112, 73, 162, 0,7,         1,-39780,     0            , "t=3.57s",_
 3113, 74, 162, 0,0,         1,-33999,     0            , "t=1.19s",_
 3114, 75, 162, 0,2,        -1,-22500,     0            , "t=107ms",_
 3115, 75, 162, 1,9,         1,-22330,     0.175        , "t=77ms",_
 3116, 76, 162, 0,0,         1,-14440,     0            , "t=2.1ms",_
 3117, 61, 163, 0,2.5,      -1,-43250,     0            , "t=430ms",_
 3118, 62, 163, 0,0.5,      -1,-50720,     0            , "t=1.3s",_
 3119, 63, 163, 0,2.5,       1,-56480,     0            , "t=7.7s",_
 3120, 64, 163, 0,3.5,       1,-61314,     0            , "t=68s",_
 3121, 64, 163, 1,0.5,      -1,-61176,     0.1378       , "t=23.5s",_
 3122, 65, 163, 0,1.5,       1,-64596,     0            , "t=19.5m",_
 3123, 66, 163, 0,2.5,      -1,-66381.2,   0            , "t=stbl",_
 3124, 67, 163, 0,3.5,      -1,-66378.3,   0            , "t=4.570ky",_
 3125, 67, 163, 1,0.5,       1,-66080.4,   0.29788      , "t=1.09s",_
 3126, 67, 163, 2,11.5,      1,-64268.9,   2.1094       , "t=800ns",_
 3127, 68, 163, 0,2.5,      -1,-65168,     0            , "t=75.0m",_
 3128, 68, 163, 1,5.5,      -1,-64723,     0.4455       , "t=580ns",_
 3129, 69, 163, 0,0.5,       1,-62729,     0            , "t=1.810h",_
 3130, 69, 163, 1,3.5,      -1,-62642,     0.08692      , "t=380ns",_
 3131, 70, 163, 0,1.5,      -1,-59299,     0            , "t=11.05m",_
 3132, 71, 163, 0,0.5,       1,-54791,     0            , "t=3.97m",_
 3133, 72, 163, 0,2.5,      -1,-49264,     0            , "t=40.0s",_
 3134, 73, 163, 0,0.5,       1,-42530,     0            , "t=10.6s",_
 3135, 73, 163, 1,4.5,      -1,-42390,     0.14         , "t=10#s",_
 3136, 74, 163, 0,3.5,      -1,-34910,     0            , "t=2.63s",_
 3137, 74, 163, 1,6.5,       1,-34430,     0.4803       , "t=154ns",_
 3138, 75, 163, 0,0.5,       1,-26002,     0            , "t=390ms",_
 3139, 75, 163, 1,5.5,      -1,-25882,     0.12         , "t=214ms",_
 3140, 76, 163, 0,3.5,      -1,-16190,     0            , "t=5.5ms",_
 3141, 61, 164, 0,-77.777,   0,-38870,     0            , "t=200#ms",_
 3142, 62, 164, 0,0,         1,-48100,     0            , "t=1.43s",_
 3143, 62, 164, 1,6,        -1,-46620,     1.4855       , "t=600ns",_
 3144, 63, 164, 0,0,        -1,-53380,     0            , "t=4.15s",_
 3145, 64, 164, 0,0,         1,-59770,     0            , "t=45s",_
 3146, 65, 164, 0,5,         1,-62080,     0            , "t=3.0m",_
 3147, 66, 164, 0,0,         1,-65968.0,   0            , "t=stbl",_
 3148, 67, 164, 0,1,         1,-64981.5,   0            , "t=29m",_
 3149, 67, 164, 1,6,        -1,-64841.7,   0.13977      , "t=36.4m",_
 3150, 68, 164, 0,0,         1,-65942.9,   0            , "t=stbl",_
 3151, 68, 164, 1,12,        1,-62566.8,   3.3761       , "t=68ns",_
 3152, 69, 164, 0,1,         1,-61904,     0            , "t=2.0m",_
 3153, 69, 164, 1,6,        -1,-61894,     0.01         , "t=5.1m",_
 3154, 70, 164, 0,0,         1,-61017,     0            , "t=75.8m",_
 3155, 71, 164, 0,1,        -1,-54642,     0            , "t=3.14m",_
 3156, 72, 164, 0,0,         1,-51819,     0            , "t=111s",_
 3157, 73, 164, 0,3,         1,-43283,     0            , "t=14.2s",_
 3158, 74, 164, 0,0,         1,-38236,     0            , "t=6.3s",_
 3159, 75, 164, 0,2,        -1,-27470,     0            , "t=719ms",_
 3160, 75, 164, 1,9,         1,-27520,    -0.05         , "t=890ms",_
 3161, 76, 164, 0,0,         1,-20420,     0            , "t=21ms",_
 3162, 77, 164, 0,2,        -1,-7340,      0            , "t=1#ms",_
 3163, 77, 164, 1,9,         1,-7080,      0.26         , "t=70us",_
 3164, 62, 165, 0,2.5,      -1,-43810,     0            , "t=980ms",_
 3165, 63, 165, 0,2.5,       1,-50720,     0            , "t=2.53s",_
 3166, 64, 165, 0,0.5,      -1,-56450,     0            , "t=11.0s",_
 3167, 65, 165, 0,1.5,       1,-60570,     0            , "t=2.11m",_
 3168, 66, 165, 0,3.5,       1,-63612.6,   0            , "t=2.334h",_
 3169, 66, 165, 1,0.5,      -1,-63504.4,   0.1081552    , "t=1.257m",_
 3170, 67, 165, 0,3.5,      -1,-64899.0,   0            , "t=stbl",_
 3171, 67, 165, 1,1.5,       1,-64537.3,   0.361675     , "t=1.512us",_
 3172, 67, 165, 2,3.5,       1,-64183.7,   0.71533      , "t=&lt;100ns",_
 3173, 68, 165, 0,2.5,      -1,-64521.6,   0            , "t=10.36h",_
 3174, 68, 165, 1,5.5,      -1,-63970.3,   0.5513       , "t=250ns",_
 3175, 68, 165, 2,9.5,       0,-62698.6,   1.823        , "t=370ns",_
 3176, 69, 165, 0,0.5,       1,-62929.6,   0            , "t=30.06h",_
 3177, 69, 165, 1,3.5,       1,-62849.2,   0.08037      , "t=80us",_
 3178, 69, 165, 2,3.5,      -1,-62769.1,   0.16047      , "t=9.0us",_
 3179, 70, 165, 0,2.5,      -1,-60295,     0            , "t=9.9m",_
 3180, 70, 165, 1,4.5,       1,-60168,     0.1268       , "t=300ns",_
 3181, 71, 165, 0,0.5,       1,-56442,     0            , "t=10.74m",_
 3182, 72, 165, 0,2.5,      -1,-51636,     0            , "t=76s",_
 3183, 73, 165, 0,0.5,       1,-45848,     0            , "t=31.0s",_
 3184, 73, 165, 1,4.5,      -1,-45823,     0.024        , "t=30#s",_
 3185, 74, 165, 0,2.5,      -1,-38861,     0            , "t=5.1s",_
 3186, 75, 165, 0,0.5,       1,-30660,     0            , "t=2.62s",_
 3187, 75, 165, 1,5.5,      -1,-30632,     0.027        , "t=1.74s",_
 3188, 76, 165, 0,3.5,      -1,-21800,     0            , "t=71ms",_
 3189, 77, 165, 0,0.5,       1,-11590,     0            , "t=50#ns",_
 3190, 77, 165, 1,5.5,      -1,-11410,     0.18         , "t=325us",_
 3191, 62, 166, 0,0,         1,-40730,     0            , "t=800ms",_
 3192, 63, 166, 0,6,        -1,-47210,     0            , "t=1.24s",_
 3193, 64, 166, 0,0,         1,-54530,     0            , "t=5.1s",_
 3194, 64, 166, 1,6,        -1,-52930,     1.6015       , "t=950ns",_
 3195, 65, 166, 0,2,        -1,-57880,     0            , "t=27.1s",_
 3196, 66, 166, 0,0,         1,-62584.8,   0            , "t=81.6h",_
 3197, 67, 166, 0,0,        -1,-63071.3,   0            , "t=26.824h",_
 3198, 67, 166, 1,7,        -1,-63065.3,   0.005969     , "t=1.133ky",_
 3199, 67, 166, 2,3,         1,-62880.4,   0.1909021    , "t=185us",_
 3200, 68, 166, 0,0,         1,-64926.0,   0            , "t=stbl",_
 3201, 69, 166, 0,2,         1,-61888,     0            , "t=7.70h",_
 3202, 69, 166, 1,6,        -1,-61771,     0.117        , "t=348ms",_
 3203, 69, 166, 2,6,        -1,-61649,     0.239        , "t=2us",_
 3204, 70, 166, 0,0,         1,-61596,     0            , "t=56.7h",_
 3205, 71, 166, 0,6,        -1,-56021,     0            , "t=2.65m",_
 3206, 71, 166, 1,3,        -1,-55990,     0.03437      , "t=1.41m",_
 3207, 71, 166, 2,0,        -1,-55980,     0.043        , "t=2.12m",_
 3208, 72, 166, 0,0,         1,-53859,     0            , "t=6.77m",_
 3209, 73, 166, 0,2,         1,-46098,     0            , "t=34.4s",_
 3210, 74, 166, 0,0,         1,-41888,     0            , "t=19.2s",_
 3211, 75, 166, 0,7,         1,-31890,     0            , "t=2.25s",_
 3212, 76, 166, 0,0,         1,-25432,     0            , "t=213ms",_
 3213, 77, 166, 0,2,        -1,-13350,     0            , "t=10.5ms",_
 3214, 77, 166, 1,9,         1,-13180,     0.171        , "t=15.1ms",_
 3215, 78, 166, 0,0,         1,-4730,      0            , "t=300us",_
 3216, 63, 167, 0,2.5,       1,-44010,     0            , "t=1.33s",_
 3217, 64, 167, 0,2.5,      -1,-50810,     0            , "t=4.2s",_
 3218, 65, 167, 0,1.5,       1,-55930,     0            , "t=18.9s",_
 3219, 66, 167, 0,0.5,      -1,-59930,     0            , "t=6.20m",_
 3220, 67, 167, 0,3.5,      -1,-62281,     0            , "t=3.1h",_
 3221, 67, 167, 1,1.5,       1,-62022,     0.25934      , "t=6.0us",_
 3222, 68, 167, 0,3.5,       1,-63291.2,   0            , "t=stbl",_
 3223, 68, 167, 1,0.5,      -1,-63083.4,   0.207801     , "t=2.269s",_
 3224, 69, 167, 0,0.5,       1,-62543.6,   0            , "t=9.25d",_
 3225, 69, 167, 1,3.5,       1,-62364.1,   0.17948      , "t=1.16us",_
 3226, 69, 167, 2,3.5,      -1,-62250.8,   0.29282      , "t=0.9us",_
 3227, 70, 167, 0,2.5,      -1,-60591,     0            , "t=17.5m",_
 3228, 70, 167, 1,5.5,      -1,-60019,     0.571548     , "t=~180ns",_
 3229, 71, 167, 0,3.5,       1,-57500,     0            , "t=51.5m",_
 3230, 71, 167, 1,0.5,      -1,-57500,     0            , "t=&gt;1m",_
 3231, 72, 167, 0,2.5,      -1,-53468,     0            , "t=2.05m",_
 3232, 73, 167, 0,1.5,       1,-48351,     0            , "t=1.33m",_
 3233, 74, 167, 0,1.5,      -1,-42098,     0            , "t=19.9s",_
 3234, 75, 167, 0,4.5,      -1,-34830,     0            , "t=3.4s",_
 3235, 75, 167, 1,0.5,       1,-34700,     0.128        , "t=5.9s",_
 3236, 76, 167, 0,3.5,      -1,-26500,     0            , "t=839ms",_
 3237, 76, 167, 1,6.5,       1,-26060,     0.4351       , "t=672ns",_
 3238, 77, 167, 0,0.5,       1,-17072,     0            , "t=29.3ms",_
 3239, 77, 167, 1,5.5,      -1,-16897,     0.1755       , "t=25.7ms",_
 3240, 78, 167, 0,3.5,      -1,-6610,      0            , "t=800us",_
 3241, 63, 168, 0,2,         1,-39740,     0            , "t=200ms",_
 3242, 64, 168, 0,0,         1,-48360,     0            , "t=3.03s",_
 3243, 65, 168, 0,4,        -1,-52720,     0            , "t=9.4s",_
 3244, 65, 168, 1,6,         1,-52510,     0.211        , "t=0.71us",_
 3245, 66, 168, 0,0,         1,-58560,     0            , "t=8.7m",_
 3246, 67, 168, 0,3,         1,-60060,     0            , "t=2.99m",_
 3247, 67, 168, 1,6,         1,-60000,     0.059        , "t=132s",_
 3248, 67, 168, 2,1,        -1,-59920,     0.14343      , "t=&gt;4us",_
 3249, 68, 168, 0,0,         1,-62991.2,   0            , "t=stbl",_
 3250, 68, 168, 1,4,        -1,-61897.2,   1.094038     , "t=109.0ns",_
 3251, 69, 168, 0,3,         1,-61312.9,   0            , "t=93.1d",_
 3252, 70, 168, 0,0,         1,-61581.9,   0            , "t=stbl",_
 3253, 71, 168, 0,6,        -1,-57070,     0            , "t=5.5m",_
 3254, 71, 168, 1,3,         1,-56870,     0.20281      , "t=6.7m",_
 3255, 72, 168, 0,0,         1,-55361,     0            , "t=25.95m",_
 3256, 73, 168, 0,2,        -1,-48394,     0            , "t=2.0m",_
 3257, 74, 168, 0,0,         1,-44893,     0            , "t=50.9s",_
 3258, 75, 168, 0,7,         1,-35790,     0            , "t=4.4s",_
 3259, 76, 168, 0,0,         1,-29995,     0            , "t=2.1s",_
 3260, 77, 168, 0,2,        -1,-18670,     0            , "t=230ms",_
 3261, 77, 168, 1,9,         1,-18620,     0.05         , "t=163ms",_
 3262, 78, 168, 0,0,         1,-11010,     0            , "t=2.02ms",_
 3263, 64, 169, 0,3.5,      -1,-44150,     0            , "t=750ms",_
 3264, 65, 169, 0,1.5,       1,-50330,     0            , "t=5.13s",_
 3265, 66, 169, 0,2.5,      -1,-55600,     0            , "t=39s",_
 3266, 67, 169, 0,3.5,      -1,-58797,     0            , "t=4.72m",_
 3267, 67, 169, 1,9.5,       1,-57411,     1.3862       , "t=118us",_
 3268, 68, 169, 0,0.5,      -1,-60923.1,   0            , "t=9.392d",_
 3269, 68, 169, 1,2.5,      -1,-60831.1,   0.09205      , "t=285ns",_
 3270, 68, 169, 2,3.5,       1,-60679.4,   0.24369      , "t=200ns",_
 3271, 69, 169, 0,0.5,       1,-61275.2,   0            , "t=stbl",_
 3272, 69, 169, 1,3.5,       1,-60959.1,   0.3161463    , "t=659.9ns",_
 3273, 70, 169, 0,3.5,       1,-60377.6,   0            , "t=32.018d",_
 3274, 70, 169, 1,0.5,      -1,-60353.4,   0.0241999    , "t=46s",_
 3275, 71, 169, 0,3.5,       1,-58085,     0            , "t=34.06h",_
 3276, 71, 169, 1,0.5,      -1,-58056,     0.029        , "t=160s",_
 3277, 72, 169, 0,2.5,      -1,-54717,     0            , "t=3.24m",_
 3278, 73, 169, 0,2.5,       1,-50290,     0            , "t=4.9m",_
 3279, 74, 169, 0,2.5,      -1,-44918,     0            , "t=74s",_
 3280, 75, 169, 0,4.5,      -1,-38409,     0            , "t=8.1s",_
 3281, 75, 169, 1,0.5,       1,-38234,     0.175        , "t=15.1s",_
 3282, 76, 169, 0,2.5,      -1,-30723,     0            , "t=3.46s",_
 3283, 77, 169, 0,0.5,       1,-22094,     0            , "t=353ms",_
 3284, 77, 169, 1,5.5,      -1,-21941,     0.153        , "t=280ms",_
 3285, 78, 169, 0,3.5,      -1,-12510,     0            , "t=6.99ms",_
 3286, 79, 169, 0,0.5,       1,-1790,      0            , "t=150#us",_
 3287, 64, 170, 0,0,         1,-41380,     0            , "t=420ms",_
 3288, 65, 170, 0,2,        -1,-46720,     0            , "t=960ms",_
 3289, 66, 170, 0,0,         1,-53660,     0            , "t=54.9s",_
 3290, 66, 170, 1,6,         1,-52020,     1.64392      , "t=0.94us",_
 3291, 67, 170, 0,6,         1,-56240,     0            , "t=2.76m",_
 3292, 67, 170, 1,1,         1,-56140,     0.1          , "t=43s",_
 3293, 68, 170, 0,0,         1,-60108.7,   0            , "t=stbl",_
 3294, 69, 170, 0,1,        -1,-59795.9,   0            , "t=128.6d",_
 3295, 69, 170, 1,3,         1,-59612.7,   0.183197     , "t=4.12us",_
 3296, 70, 170, 0,0,         1,-60763.919,               0            , "t=stbl",_
 3297, 70, 170, 1,4,        -1,-59505.46,  1.25846      , "t=370ns",_
 3298, 71, 170, 0,0,         1,-57306,     0            , "t=2.012d",_
 3299, 71, 170, 1,4,        -1,-57213,     0.09291      , "t=670ms",_
 3300, 72, 170, 0,0,         1,-56254,     0            , "t=16.01h",_
 3301, 73, 170, 0,3,         1,-50138,     0            , "t=6.76m",_
 3302, 74, 170, 0,0,         1,-47291,     0            , "t=2.42m",_
 3303, 75, 170, 0,5,         1,-38913,     0            , "t=9.2s",_
 3304, 76, 170, 0,0,         1,-33926,     0            , "t=7.37s",_
 3305, 77, 170, 0,3,        -1,-23360,     0            , "t=910ms",_
 3306, 77, 170, 1,8,         1,-23200,     0.16         , "t=811ms",_
 3307, 78, 170, 0,0,         1,-16299,     0            , "t=13.93ms",_
 3308, 79, 170, 0,2,        -1,-3750,      0            , "t=290us",_
 3309, 79, 170, 1,9,         1,-3470,      0.28         , "t=620us",_
 3310, 65, 171, 0,1.5,       1,-44030,     0            , "t=1.23s",_
 3311, 66, 171, 0,3.5,      -1,-50190,     0            , "t=4.07s",_
 3312, 67, 171, 0,3.5,      -1,-54520,     0            , "t=53s",_
 3313, 68, 171, 0,2.5,      -1,-57719.0,   0            , "t=7.516h",_
 3314, 68, 171, 1,0.5,      -1,-57520.4,   0.1986       , "t=210ns",_
 3315, 69, 171, 0,0.5,       1,-59210.3,   0            , "t=1.92y",_
 3316, 69, 171, 1,3.5,      -1,-58785.3,   0.424956     , "t=2.60us",_
 3317, 69, 171, 2,9.5,       1,-57535.8,   1.6745       , "t=1.7us",_
 3318, 70, 171, 0,0.5,      -1,-59306.810,               0            , "t=stbl",_
 3319, 70, 171, 1,3.5,       1,-59211.528,               0.095282     , "t=5.25ms",_
 3320, 70, 171, 2,2.5,      -1,-59184.394,               0.122416     , "t=265ns",_
 3321, 71, 171, 0,3.5,       1,-57828.4,   0            , "t=8.24d",_
 3322, 71, 171, 1,0.5,      -1,-57757.3,   0.07113      , "t=79s",_
 3323, 72, 171, 0,3.5,       1,-55431,     0            , "t=12.1h",_
 3324, 72, 171, 1,0.5,      -1,-55409,     0.02193      , "t=29.5s",_
 3325, 73, 171, 0,2.5,      -1,-51720,     0            , "t=23.3m",_
 3326, 74, 171, 0,2.5,      -1,-47086,     0            , "t=2.38m",_
 3327, 75, 171, 0,4.5,      -1,-41250,     0            , "t=15.2s",_
 3328, 76, 171, 0,2.5,      -1,-34302,     0            , "t=8.3s",_
 3329, 77, 171, 0,0.5,       1,-26410,     0            , "t=3.1s",_
 3330, 77, 171, 1,5.5,      -1,-26250,     0.167        , "t=1.47s",_
 3331, 78, 171, 0,3.5,      -1,-17470,     0            , "t=45.5ms",_
 3332, 78, 171, 1,6.5,       1,-17060,     0.4126       , "t=901ns",_
 3333, 79, 171, 0,0.5,       1,-7562,      0            , "t=22.3us",_
 3334, 79, 171, 1,5.5,      -1,-7308,      0.255        , "t=1.036ms",_
 3335, 80, 171, 0,1.5,      -1,3480,       0            , "t=70us",_
 3336, 65, 172, 0,6,         1,-39850,     0            , "t=760ms",_
 3337, 66, 172, 0,0,         1,-48010,     0            , "t=3.4s",_
 3338, 66, 172, 1,8,        -1,-46730,     1.278        , "t=710ms",_
 3339, 67, 172, 0,0,         1,-51480,     0            , "t=25s",_
 3340, 68, 172, 0,0,         1,-56484,     0            , "t=49.3h",_
 3341, 68, 172, 1,6,         1,-54983,     1.5009       , "t=579ns",_
 3342, 69, 172, 0,2,        -1,-57374,     0            , "t=63.6h",_
 3343, 69, 172, 1,6,         1,-56898,     0.4762       , "t=132us",_
 3344, 70, 172, 0,0,         1,-59255.446,               0            , "t=stbl",_
 3345, 70, 172, 1,6,        -1,-57705.02,  1.55043      , "t=3.6us",_
 3346, 71, 172, 0,4,        -1,-56736.0,   0            , "t=6.70d",_
 3347, 71, 172, 1,1,        -1,-56694.1,   0.04186      , "t=3.7m",_
 3348, 71, 172, 2,1,         1,-56670.2,   0.06579      , "t=332ns",_
 3349, 72, 172, 0,0,         1,-56402,     0            , "t=1.87y",_
 3350, 72, 172, 1,8,        -1,-54396,     2.00584      , "t=163ns",_
 3351, 73, 172, 0,3,         1,-51330,     0            , "t=36.8m",_
 3352, 74, 172, 0,0,         1,-49097,     0            , "t=6.6m",_
 3353, 75, 172, 0,5,         1,-41540,     0            , "t=15s",_
 3354, 75, 172, 1,2,         0,-41540,     0            , "t=55s",_
 3355, 76, 172, 0,0,         1,-37244,     0            , "t=19.2s",_
 3356, 77, 172, 0,3,        -1,-27380,     0            , "t=4.4s",_
 3357, 77, 172, 1,7,         1,-27240,     0.139        , "t=2.19s",_
 3358, 78, 172, 0,0,         1,-21107,     0            , "t=97.6ms",_
 3359, 79, 172, 0,2,        -1,-9320,      0            , "t=28ms",_
 3360, 79, 172, 1,9,         1,-9160,      0.16         , "t=11.0ms",_
 3361, 80, 172, 0,0,         1,-1060,      0            , "t=231us",_
 3362, 66, 173, 0,4.5,       1,-43940,     0            , "t=1.43s",_
 3363, 67, 173, 0,3.5,      -1,-49350,     0            , "t=6.90s",_
 3364, 68, 173, 0,3.5,      -1,-53650,     0            , "t=1.434m",_
 3365, 69, 173, 0,0.5,       1,-56256,     0            , "t=8.24h",_
 3366, 69, 173, 1,3.5,      -1,-55938,     0.31773      , "t=10.7us",_
 3367, 69, 173, 2,9.5,      -1,-54350,     1.9057       , "t=250ns",_
 3368, 70, 173, 0,2.5,      -1,-57551.225,               0            , "t=stbl",_
 3369, 70, 173, 1,0.5,      -1,-57152.3,   0.3989       , "t=2.9us",_
 3370, 71, 173, 0,3.5,       1,-56880.9,   0            , "t=1.37y",_
 3371, 71, 173, 1,2.5,      -1,-56757.2,   0.123672     , "t=74.2us",_
 3372, 72, 173, 0,0.5,      -1,-55412,     0            , "t=23.6h",_
 3373, 72, 173, 1,2.5,      -1,-55305,     0.10716      , "t=180ns",_
 3374, 72, 173, 2,3.5,       1,-55215,     0.19747      , "t=160ns",_
 3375, 73, 173, 0,2.5,      -1,-52397,     0            , "t=3.14h",_
 3376, 73, 173, 1,4.5,      -1,-52224,     0.1731       , "t=225ns",_
 3377, 73, 173, 2,10.5,     -1,-50678,     1.7194       , "t=132ns",_
 3378, 74, 173, 0,2.5,      -1,-48727,     0            , "t=7.6m",_
 3379, 75, 173, 0,2.5,      -1,-43554,     0            , "t=2.0m",_
 3380, 76, 173, 0,2.5,      -1,-37438,     0            , "t=22.4s",_
 3381, 77, 173, 0,0.5,       1,-30268,     0            , "t=9.0s",_
 3382, 77, 173, 1,5.5,      -1,-30042,     0.226        , "t=2.20s",_
 3383, 78, 173, 0,2.5,      -1,-21940,     0            , "t=382ms",_
 3384, 79, 173, 0,0.5,       1,-12832,     0            , "t=25.5ms",_
 3385, 79, 173, 1,5.5,      -1,-12619,     0.214        , "t=12.2ms",_
 3386, 80, 173, 0,1.5,      -1,-2710,      0            , "t=800us",_
 3387, 66, 174, 0,0,         1,-41370,     0            , "t=1#s",_
 3388, 67, 174, 0,8,        -1,-45690,     0            , "t=3.2s",_
 3389, 68, 174, 0,0,         1,-51950,     0            , "t=3.2m",_
 3390, 68, 174, 1,8,        -1,-50840,     1.1116       , "t=3.9s",_
 3391, 69, 174, 0,4,        -1,-53860,     0            , "t=5.4m",_
 3392, 69, 174, 1,0,         1,-53610,     0.2524       , "t=2.29s",_
 3393, 70, 174, 0,0,         1,-56944.512,               0            , "t=stbl",_
 3394, 70, 174, 1,6,         1,-55426.364,               1.518148     , "t=830us",_
 3395, 70, 174, 2,7,        -1,-55179.3,   1.7652       , "t=256ns",_
 3396, 71, 174, 0,1,        -1,-55570.2,   0            , "t=3.31y",_
 3397, 71, 174, 1,6,        -1,-55399.4,   0.17083      , "t=142d",_
 3398, 71, 174, 2,3,         1,-55329.4,   0.240818     , "t=395ns",_
 3399, 72, 174, 0,0,         1,-55844.5,   0            , "t=2.0Py",_
 3400, 72, 174, 1,6,         1,-54295.2,   1.5493       , "t=138ns",_
 3401, 72, 174, 2,8,        -1,-54047.0,   1.7975       , "t=2.39us",_
 3402, 73, 174, 0,3,         1,-51741,     0            , "t=1.14h",_
 3403, 74, 174, 0,0,         1,-50227,     0            , "t=33.2m",_
 3404, 74, 174, 1,-77.777,   0,-48555,     1.672        , "t=&gt;187ns",_
 3405, 74, 174, 2,-77.777,   0,-48307,     1.9197       , "t=187ns",_
 3406, 75, 174, 0,3,         1,-43673,     0            , "t=2.40m",_
 3407, 75, 174, 1,7,         1,-43570,     0.1          , "t=1#m",_
 3408, 76, 174, 0,0,         1,-39995,     0            , "t=44s",_
 3409, 77, 174, 0,3,         1,-30863,     0            , "t=7.9s",_
 3410, 77, 174, 1,7,         1,-30671,     0.192        , "t=4.9s",_
 3411, 78, 174, 0,0,         1,-25318,     0            , "t=889ms",_
 3412, 79, 174, 0,0,         0,-14240,     0            , "t=139ms",_
 3413, 79, 174, 1,0,         0,-13990,     0.25         , "t=171ms",_
 3414, 80, 174, 0,0,         1,-6641,      0            , "t=2.0ms",_
 3415, 67, 175, 0,3.5,      -1,-43200,     0            , "t=1.88s",_
 3416, 68, 175, 0,4.5,       1,-48650,     0            , "t=1.2m",_
 3417, 69, 175, 0,0.5,       1,-52310,     0            , "t=15.2m",_
 3418, 69, 175, 1,3.5,      -1,-51870,     0.44         , "t=319ns",_
 3419, 69, 175, 2,11.5,      1,-50790,     1.5177       , "t=21us",_
 3420, 70, 175, 0,3.5,      -1,-54695.55,  0            , "t=4.185d",_
 3421, 70, 175, 1,0.5,      -1,-54180.68,  0.514866     , "t=68.2ms",_
 3422, 71, 175, 0,3.5,       1,-55165.6,   0            , "t=stbl",_
 3423, 71, 175, 1,2.5,      -1,-54812.1,   0.35348      , "t=1.49us",_
 3424, 71, 175, 2,9.5,       1,-53773.4,   1.3922       , "t=984us",_
 3425, 72, 175, 0,2.5,      -1,-54481.7,   0            , "t=70.65d",_
 3426, 72, 175, 1,0.5,      -1,-54355.8,   0.12589      , "t=53.7us",_
 3427, 72, 175, 2,9.5,       1,-53048.3,   1.43341      , "t=1.10us",_
 3428, 73, 175, 0,3.5,       1,-52409,     0            , "t=10.5h",_
 3429, 73, 175, 1,4.5,      -1,-52278,     0.13141      , "t=222ns",_
 3430, 73, 175, 2,0.5,       1,-52070,     0.3392       , "t=170ns",_
 3431, 74, 175, 0,0.5,      -1,-49633,     0            , "t=35.2m",_
 3432, 74, 175, 1,3.5,       1,-49398,     0.23496      , "t=216ns",_
 3433, 75, 175, 0,2.5,      -1,-45288,     0            , "t=5.89m",_
 3434, 76, 175, 0,2.5,      -1,-40105,     0            , "t=1.4m",_
 3435, 77, 175, 0,2.5,      -1,-33395,     0            , "t=9s",_
 3436, 78, 175, 0,3.5,      -1,-25713,     0            , "t=2.43s",_
 3437, 79, 175, 0,0.5,       1,-17400,     0            , "t=202ms",_
 3438, 79, 175, 1,5.5,      -1,-17240,     0.167        , "t=134ms",_
 3439, 80, 175, 0,3.5,      -1,-7970,      0            , "t=10.6ms",_
 3440, 80, 175, 1,6.5,       1,-7480,      0.494        , "t=340ns",_
 3441, 67, 176, 0,-77.777,   0,-39290,     0            , "t=2#s",_
 3442, 68, 176, 0,0,         1,-46630,     0            , "t=20#s",_
 3443, 69, 176, 0,4,         1,-49370,     0            , "t=1.85m",_
 3444, 70, 176, 0,0,         1,-53491.314,               0            , "t=stbl",_
 3445, 70, 176, 1,8,        -1,-52441.5,   1.0498       , "t=11.4s",_
 3446, 71, 176, 0,7,        -1,-53382.2,   0            , "t=36.84Gy",_
 3447, 71, 176, 1,1,        -1,-53259.4,   0.122845     , "t=3.664h",_
 3448, 71, 176, 2,12,        1,-51867.7,   1.5145       , "t=312ns",_
 3449, 72, 176, 0,0,         1,-54576.3,   0            , "t=stbl",_
 3450, 72, 176, 1,6,         1,-53243.2,   1.33307      , "t=9.6us",_
 3451, 72, 176, 2,8,        -1,-53017.0,   1.55931      , "t=9.9us",_
 3452, 73, 176, 0,1,        -1,-51370,     0            , "t=8.09h",_
 3453, 73, 176, 1,7,         1,-51270,     0.103        , "t=1.08ms",_
 3454, 73, 176, 2,14,       -1,-49900,     1.474        , "t=3.8us",_
 3455, 74, 176, 0,0,         1,-50642,     0            , "t=2.5h",_
 3456, 75, 176, 0,3,         1,-45063,     0            , "t=5.3m",_
 3457, 76, 176, 0,0,         1,-42098,     0            , "t=3.6m",_
 3458, 77, 176, 0,-77.777,   0,-33878,     0            , "t=8.7s",_
 3459, 78, 176, 0,0,         1,-28934,     0            , "t=6.33s",_
 3460, 79, 176, 0,3,        -1,-18520,     0            , "t=1.05s",_
 3461, 79, 176, 1,7,         1,-18380,     0.139        , "t=860ms",_
 3462, 80, 176, 0,0,         1,-11785,     0            , "t=20.3ms",_
 3463, 81, 176, 0,3,        -1,580,        0            , "t=6.2ms",_
 3464, 68, 177, 0,0.5,      -1,-42860,     0            , "t=3#s",_
 3465, 69, 177, 0,3.5,      -1,-47470,     0            , "t=90s",_
 3466, 70, 177, 0,4.5,       1,-50986.40,  0            , "t=1.911h",_
 3467, 70, 177, 1,0.5,      -1,-50654.9,   0.3315       , "t=6.41s",_
 3468, 71, 177, 0,3.5,       1,-52383.8,   0            , "t=6.6457d",_
 3469, 71, 177, 1,4.5,      -1,-52233.4,   0.1503967    , "t=130ns",_
 3470, 71, 177, 2,0.5,       1,-51814.1,   0.5697068    , "t=155us",_
 3471, 72, 177, 0,3.5,      -1,-52880.6,   0            , "t=stbl",_
 3472, 72, 177, 1,11.5,      1,-51565.1,   1.31545      , "t=1.09s",_
 3473, 72, 177, 2,9.5,      -1,-51538.2,   1.34238      , "t=55.9us",_
 3474, 73, 177, 0,3.5,       1,-51715,     0            , "t=56.56h",_
 3475, 73, 177, 1,4.5,      -1,-51642,     0.07336      , "t=410ns",_
 3476, 73, 177, 2,2.5,      -1,-51529,     0.18615      , "t=3.62us",_
 3477, 74, 177, 0,0.5,      -1,-49702,     0            , "t=132m",_
 3478, 75, 177, 0,2.5,      -1,-46269,     0            , "t=14m",_
 3479, 75, 177, 1,2.5,       1,-46184,     0.08471      , "t=50us",_
 3480, 76, 177, 0,0.5,      -1,-41956,     0            , "t=3.0m",_
 3481, 77, 177, 0,2.5,      -1,-36047,     0            , "t=30s",_
 3482, 78, 177, 0,2.5,      -1,-29370,     0            , "t=10.6s",_
 3483, 78, 177, 1,0.5,      -1,-29223,     0.1474       , "t=2.2us",_
 3484, 79, 177, 0,0.5,       1,-21545,     0            , "t=1.46s",_
 3485, 79, 177, 1,5.5,      -1,-21356,     0.189        , "t=1.180s",_
 3486, 80, 177, 0,3.5,      -1,-12780,     0            , "t=127.3ms",_
 3487, 80, 177, 1,6.5,       1,-12460,     0.323        , "t=1.50us",_
 3488, 81, 177, 0,0.5,       1,-3341,      0            , "t=18ms",_
 3489, 81, 177, 1,5.5,      -1,-2534,      0.807        , "t=180us",_
 3490, 68, 178, 0,0,         1,-40260,     0            , "t=1#s",_
 3491, 69, 178, 0,-77.777,   0,-44120,     0            , "t=30#s",_
 3492, 70, 178, 0,0,         1,-49695,     0            , "t=74m",_
 3493, 71, 178, 0,1,         1,-50337.8,   0            , "t=28.4m",_
 3494, 71, 178, 1,9,        -1,-50214,     0.1238       , "t=23.1m",_
 3495, 72, 178, 0,0,         1,-52435.2,   0            , "t=stbl",_
 3496, 72, 178, 1,8,        -1,-51287.8,   1.147416     , "t=4.0s",_
 3497, 72, 178, 2,16,        1,-49989.1,   2.44609      , "t=31y",_
 3498, 73, 178, 0,7,        -1,-50600,     0            , "t=2.36h",_
 3499, 73, 178, 1,1,         1,-50498,     0.1          , "t=9.31m",_
 3500, 73, 178, 2,15,       -1,-49130,     1.46782      , "t=59ms",_
 3501, 74, 178, 0,0,         1,-50407,     0            , "t=21.6d",_
 3502, 74, 178, 1,25,        1,-43834,     6.5727       , "t=220ns",_
 3503, 75, 178, 0,3,         1,-45653,     0            , "t=13.2m",_
 3504, 76, 178, 0,0,         1,-43544,     0            , "t=5.0m",_
 3505, 77, 178, 0,-77.777,   0,-36252,     0            , "t=12s",_
 3506, 78, 178, 0,0,         1,-31998,     0            , "t=20.7s",_
 3507, 79, 178, 0,-77.777,   0,-22304,     0            , "t=2.6s",_
 3508, 79, 178, 1,-77.777,   0,-22115,     0.189        , "t=&gt;1s",_
 3509, 80, 178, 0,0,         1,-16316,     0            , "t=266.5ms",_
 3510, 81, 178, 0,4,        -1,-4790,      0            , "t=255ms",_
 3511, 82, 178, 0,0,         1,3574,       0            , "t=230us",_
 3512, 69, 179, 0,0.5,       1,-41600,     0            , "t=20#s",_
 3513, 70, 179, 0,0.5,      -1,-46540,     0            , "t=8.0m",_
 3514, 71, 179, 0,3.5,       1,-49059,     0            , "t=4.59h",_
 3515, 71, 179, 1,0.5,       1,-48467,     0.5924       , "t=3.1ms",_
 3516, 72, 179, 0,4.5,       1,-50462.9,   0            , "t=stbl",_
 3517, 72, 179, 1,0.5,      -1,-50087.9,   0.3750352    , "t=18.67s",_
 3518, 72, 179, 2,12.5,     -1,-49357.2,   1.10574      , "t=25.05d",_
 3519, 73, 179, 0,3.5,       1,-50357.3,   0            , "t=1.82y",_
 3520, 73, 179, 1,4.5,      -1,-50326.6,   0.0307       , "t=1.42us",_
 3521, 73, 179, 2,0.5,       1,-49837.1,   0.52023      , "t=280ns",_
 3522, 73, 179, 0,18.5,      1,-47718.0,   2.6393       , "t=54.1ms",_
 3523, 74, 179, 0,3.5,      -1,-49295,     0            , "t=37.05m",_
 3524, 74, 179, 1,0.5,      -1,-49073,     0.22191      , "t=6.40m",_
 3525, 74, 179, 2,10.5,      1,-47663,     1.6319       , "t=390ns",_
 3526, 75, 179, 0,2.5,       1,-46584,     0            , "t=19.5m",_
 3527, 75, 179, 1,2.5,      -1,-46519,     0.06535      , "t=95us",_
 3528, 75, 179, 2,11.5,      1,-44760,     1.822        , "t=408ns",_
 3529, 76, 179, 0,0.5,      -1,-43019,     0            , "t=6.5m",_
 3530, 76, 179, 1,3.5,      -1,-42874,     0.14541      , "t=500ns",_
 3531, 76, 179, 2,4.5,       1,-42776,     0.243        , "t=783ns",_
 3532, 77, 179, 0,2.5,      -1,-38082,     0            , "t=79s",_
 3533, 78, 179, 0,0.5,      -1,-32268,     0            , "t=21.2s",_
 3534, 79, 179, 0,0.5,       1,-24989,     0            , "t=7.1s",_
 3535, 79, 179, 1,1.5,      -1,-24900,     0.0895       , "t=328ns",_
 3536, 80, 179, 0,3.5,      -1,-16928,     0            , "t=1.05s",_
 3537, 80, 179, 1,6.5,       1,-16757,     0.1714       , "t=6.4us",_
 3538, 81, 179, 0,0.5,       1,-8270,      0            , "t=265ms",_
 3539, 81, 179, 1,5.5,      -1,-7440,      0.825        , "t=1.41ms",_
 3540, 82, 179, 0,4.5,      -1,2050,       0            , "t=3.9ms",_
 3541, 69, 180, 0,-77.777,   0,-37920,     0            , "t=5#s",_
 3542, 70, 180, 0,0,         1,-44600,     0            , "t=2.4m",_
 3543, 71, 180, 0,5,         1,-46680,     0            , "t=5.7m",_
 3544, 71, 180, 1,3,        -1,-46670,     0.0139       , "t=~1s",_
 3545, 71, 180, 2,9,        -1,-46060,     0.624        , "t=&gt;1ms",_
 3546, 72, 180, 0,0,         1,-49779.3,   0            , "t=stbl",_
 3547, 72, 180, 1,8,        -1,-48637.7,   1.141552     , "t=5.53h",_
 3548, 72, 180, 2,4,        -1,-48404.9,   1.37436      , "t=570us",_
 3549, 73, 180, 0,1,         1,-48932.9,   0            , "t=8.154h",_
 3550, 73, 180, 1,9,        -1,-48857.5,   0.0753       , "t=stbl",_
 3551, 73, 180, 2,15,       -1,-47480.5,   1.45239      , "t=31.2us",_
 3552, 74, 180, 0,0,         1,-49636.1,   0            , "t=1.8Ey",_
 3553, 74, 180, 1,8,        -1,-48107.0,   1.52905      , "t=5.47ms",_
 3554, 74, 180, 2,14,       -1,-46371.2,   3.2649       , "t=2.3us",_
 3555, 75, 180, 0,1,        -1,-45837,     0            , "t=2.46m",_
 3556, 75, 180, 1,4,         1,-45750,     0.09         , "t=&gt;1us",_
 3557, 75, 180, 2,21,       -1,-42280,     3.561        , "t=9.0us",_
 3558, 76, 180, 0,0,         1,-44358,     0            , "t=21.5m",_
 3559, 77, 180, 0,5,         1,-37978,     0            , "t=1.5m",_
 3560, 78, 180, 0,0,         1,-34436,     0            , "t=56s",_
 3561, 79, 180, 0,-77.777,   0,-25626,     0            , "t=8.4s",_
 3562, 80, 180, 0,0,         1,-20250,     0            , "t=2.59s",_
 3563, 81, 180, 0,4,        -1,-9390,      0            , "t=1.09s",_
 3564, 82, 180, 0,0,         1,-1941,      0            , "t=4.1ms",_
 3565, 69, 181, 0,0.5,       1,-35170,     0            , "t=5#s",_
 3566, 70, 181, 0,1.5,      -1,-41090,     0            , "t=1#m",_
 3567, 71, 181, 0,3.5,       1,-44800,     0            , "t=3.5m",_
 3568, 72, 181, 0,0.5,      -1,-47402.8,   0            , "t=42.39d",_
 3569, 72, 181, 1,4.5,       1,-46807.5,   0.59527      , "t=80us",_
 3570, 72, 181, 2,8.5,       1,-46359.3,   1.0435       , "t=~100us",_
 3571, 73, 181, 0,3.5,       1,-48438.3,   0            , "t=stbl",_
 3572, 73, 181, 1,4.5,      -1,-48432.1,   0.006237     , "t=6.05us",_
 3573, 73, 181, 2,0.5,       1,-47823.1,   0.61519      , "t=18us",_
 3574, 74, 181, 0,4.5,       1,-48233.8,   0            , "t=121.2d",_
 3575, 74, 181, 1,2.5,      -1,-47868.3,   0.36555      , "t=14.59us",_
 3576, 74, 181, 2,10.5,      1,-46580.7,   1.6531       , "t=140ns",_
 3577, 75, 181, 0,2.5,       1,-46517,     0            , "t=19.9h",_
 3578, 75, 181, 1,4.5,      -1,-46254,     0.26291      , "t=156.7ns",_
 3579, 75, 181, 2,10.5,     -1,-44861,     1.65637      , "t=250ns",_
 3580, 76, 181, 0,0.5,      -1,-43550,     0            , "t=105m",_
 3581, 76, 181, 1,3.5,      -1,-43501,     0.0492       , "t=2.7m",_
 3582, 76, 181, 2,4.5,       1,-43393,     0.15691      , "t=262ns",_
 3583, 77, 181, 0,2.5,      -1,-39463,     0            , "t=4.90m",_
 3584, 77, 181, 1,2.5,       1,-39174,     0.28933      , "t=298ns",_
 3585, 77, 181, 2,4.5,      -1,-39097,     0.3663       , "t=126ns",_
 3586, 78, 181, 0,0.5,      -1,-34382,     0            , "t=52.0s",_
 3587, 78, 181, 1,3.5,      -1,-34265,     0.11665      , "t=&gt;300ns",_
 3588, 79, 181, 0,1.5,      -1,-27871,     0            , "t=13.7s",_
 3589, 80, 181, 0,0.5,      -1,-20661,     0            , "t=3.6s",_
 3590, 80, 181, 1,6.5,       1,-20450,     0.21         , "t=480us",_
 3591, 81, 181, 0,0.5,       1,-12799,     0            , "t=3.2s",_
 3592, 81, 181, 1,4.5,      -1,-11963,     0.8359       , "t=1.40ms",_
 3593, 82, 181, 0,4.5,      -1,-3120,      0            , "t=39.0ms",_
 3594, 70, 182, 0,0,         1,-38820,     0            , "t=10#s",_
 3595, 71, 182, 0,1,        -1,-41880,     0            , "t=2.0m",_
 3596, 72, 182, 0,0,         1,-46050,     0            , "t=8.90My",_
 3597, 72, 182, 1,8,        -1,-44877,     1.17287      , "t=61.5m",_
 3598, 72, 182, 2,13,        1,-43479,     2.5713       , "t=40us",_
 3599, 73, 182, 0,3,        -1,-46429.9,   0            , "t=114.74d",_
 3600, 73, 182, 1,5,         1,-46413.6,   0.016273     , "t=283ms",_
 3601, 73, 182, 2,10,       -1,-45910.3,   0.519577     , "t=15.84m",_
 3602, 74, 182, 0,0,         1,-48246.1,   0            , "t=stbl",_
 3603, 74, 182, 1,10,        1,-46015.5,   2.23065      , "t=1.3us",_
 3604, 75, 182, 0,7,         1,-45450,     0            , "t=64.2h",_
 3605, 75, 182, 1,2,         1,-45386,     0.06         , "t=14.14h",_
 3606, 75, 182, 2,2,        -1,-45150,     0.3          , "t=585ns",_
 3607, 76, 182, 0,0,         1,-44609,     0            , "t=21.84h",_
 3608, 76, 182, 1,8,        -1,-42778,     1.8314       , "t=780us",_
 3609, 76, 182, 2,25,        1,-37560,     7.0495       , "t=150ns",_
 3610, 77, 182, 0,3,         1,-39052,     0            , "t=15.0m",_
 3611, 77, 182, 1,5,         1,-38981,     0.07102      , "t=170ns",_
 3612, 77, 182, 2,6,        -1,-38876,     0.1764       , "t=130ns",_
 3613, 78, 182, 0,0,         1,-36168,     0            , "t=2.67m",_
 3614, 79, 182, 0,2,         1,-28301,     0            , "t=15.5s",_
 3615, 80, 182, 0,0,         1,-23577,     0            , "t=10.83s",_
 3616, 81, 182, 0,2,        -1,-13328,     0            , "t=1.9s",_
 3617, 81, 182, 1,7,         1,-13280,     0.05         , "t=3#s",_
 3618, 82, 182, 0,0,         1,-6825,      0            , "t=55ms",_
 3619, 70, 183, 0,1.5,      -1,-35100,     0            , "t=3#s",_
 3620, 71, 183, 0,3.5,       1,-39720,     0            , "t=58s",_
 3621, 72, 183, 0,1.5,      -1,-43280,     0            , "t=1.018h",_
 3622, 72, 183, 1,13.5,     -1,-41820,     1.464        , "t=40s",_
 3623, 73, 183, 0,3.5,       1,-45292.8,   0            , "t=5.1d",_
 3624, 73, 183, 1,4.5,      -1,-45219.6,   0.073164     , "t=106ns",_
 3625, 73, 183, 2,9.5,       1,-43957,     1.336        , "t=900ns",_
 3626, 74, 183, 0,0.5,      -1,-46365.6,   0            , "t=stbl",_
 3627, 74, 183, 1,5.5,       1,-46056.1,   0.309492     , "t=5.30s",_
 3628, 75, 183, 0,2.5,       1,-45810,     0            , "t=70.0d",_
 3629, 75, 183, 1,12.5,      1,-43903,     1.90721      , "t=1.04ms",_
 3630, 76, 183, 0,4.5,       1,-43660,     0            , "t=13.0h",_
 3631, 76, 183, 1,0.5,      -1,-43490,     0.17073      , "t=9.9h",_
 3632, 77, 183, 0,2.5,      -1,-40203,     0            , "t=58m",_
 3633, 78, 183, 0,0.5,      -1,-35772,     0            , "t=6.5m",_
 3634, 78, 183, 1,3.5,      -1,-35737,     0.03474      , "t=43s",_
 3635, 78, 183, 2,4.5,       1,-35576,     0.1959       , "t=&gt;150ns",_
 3636, 79, 183, 0,2.5,      -1,-30191,     0            , "t=42.8s",_
 3637, 79, 183, 1,0.5,       1,-30118,     0.0733       , "t=&gt;1us",_
 3638, 80, 183, 0,0.5,      -1,-23805,     0            , "t=9.4s",_
 3639, 80, 183, 1,6.5,       1,-23601,     0.204        , "t=&gt;8#us",_
 3640, 81, 183, 0,0.5,       1,-16587,     0            , "t=6.9s",_
 3641, 81, 183, 1,4.5,      -1,-15959,     0.6287       , "t=53.3ms",_
 3642, 81, 183, 2,6.5,       1,-15612,     0.97501      , "t=1.48us",_
 3643, 82, 183, 0,1.5,      -1,-7575,      0            , "t=535ms",_
 3644, 82, 183, 1,6.5,       1,-7481,      0.094        , "t=415ms",_
 3645, 70, 184, 0,0,         1,-32540,     0            , "t=1#s",_
 3646, 71, 184, 0,3,         1,-36410,     0            , "t=20s",_
 3647, 72, 184, 0,0,         1,-41500,     0            , "t=4.12h",_
 3648, 72, 184, 1,8,        -1,-40230,     1.2722       , "t=48s",_
 3649, 72, 184, 2,15,        1,-39020,     2.477        , "t=16m",_
 3650, 73, 184, 0,5,        -1,-42839,     0            , "t=8.7h",_
 3651, 74, 184, 0,0,         1,-45705.4,   0            , "t=stbl",_
 3652, 74, 184, 1,5,        -1,-44420.4,   1.284997     , "t=8.33us",_
 3653, 74, 184, 2,14,       -1,-41842.2,   3.8632       , "t=188ns",_
 3654, 75, 184, 0,3,        -1,-44220,     0            , "t=35.4d",_
 3655, 75, 184, 1,8,         1,-44032,     0.1880463    , "t=169d",_
 3656, 76, 184, 0,0,         1,-44252.5,   0            , "t=stbl",_
 3657, 77, 184, 0,5,        -1,-39611,     0            , "t=3.09h",_
 3658, 77, 184, 1,3,         1,-39385,     0.22565      , "t=470us",_
 3659, 77, 184, 2,7,         1,-39283,     0.3284       , "t=350ns",_
 3660, 78, 184, 0,0,         1,-37334,     0            , "t=17.3m",_
 3661, 78, 184, 1,8,        -1,-35494,     1.8403       , "t=1.01ms",_
 3662, 79, 184, 0,5,         1,-30319,     0            , "t=20.6s",_
 3663, 79, 184, 1,2,         1,-30251,     0.06846      , "t=47.6s",_
 3664, 80, 184, 0,0,         1,-26349,     0            , "t=30.87s",_
 3665, 81, 184, 0,2,        -1,-16883,     0            , "t=9.5s",_
 3666, 81, 184, 1,7,         1,-16930,    -0.05         , "t=10#s",_
 3667, 81, 184, 2,10,       -1,-16430,     0.45         , "t=47.1ms",_
 3668, 82, 184, 0,0,         1,-11052,     0            , "t=490ms",_
 3669, 83, 184, 0,3,         1,1060,       0            , "t=6.6ms",_
 3670, 83, 184, 1,10,       -1,1210,       0.15         , "t=13ms",_
 3671, 70, 185, 0,1.5,      -1,-28500,     0            , "t=300#ms",_
 3672, 71, 185, 0,3.5,       1,-33890,     0            , "t=6#s",_
 3673, 72, 185, 0,1.5,      -1,-38320,     0            , "t=3.5m",_
 3674, 73, 185, 0,3.5,       1,-41394,     0            , "t=49.4m",_
 3675, 73, 185, 1,1.5,       1,-40988,     0.406        , "t=900ns",_
 3676, 73, 185, 2,10.5,     -1,-40121,     1.2734       , "t=11.8ms",_
 3677, 74, 185, 0,1.5,      -1,-43387.8,   0            , "t=75.1d",_
 3678, 74, 185, 1,5.5,       1,-43190.4,   0.197383     , "t=1.597m",_
 3679, 75, 185, 0,2.5,       1,-43819.0,   0            , "t=stbl",_
 3680, 75, 185, 1,10.5,      0,-41695.2,   2.1238       , "t=121ns",_
 3681, 76, 185, 0,0.5,      -1,-42805.9,   0            , "t=92.95d",_
 3682, 76, 185, 1,3.5,      -1,-42703.5,   0.10237      , "t=3.0us",_
 3683, 76, 185, 2,5.5,       1,-42530.4,   0.27553      , "t=780ns",_
 3684, 77, 185, 0,2.5,      -1,-40336,     0            , "t=14.4h",_
 3685, 77, 185, 1,-77.777,   0,-38140,     2.197        , "t=120ns",_
 3686, 78, 185, 0,4.5,       1,-36688,     0            , "t=70.9m",_
 3687, 78, 185, 1,0.5,      -1,-36585,     0.10341      , "t=33.0m",_
 3688, 78, 185, 2,2.5,      -1,-36487,     0.20089      , "t=728ns",_
 3689, 79, 185, 0,2.5,      -1,-31858.1,   0            , "t=4.25m",_
 3690, 79, 185, 1,0.5,       1,-31760,     0.1          , "t=6.8m",_
 3691, 80, 185, 0,0.5,      -1,-26184,     0            , "t=49.1s",_
 3692, 80, 185, 1,6.5,       1,-26080,     0.1037       , "t=21.6s",_
 3693, 81, 185, 0,0.5,       1,-19758,     0            , "t=19.5s",_
 3694, 81, 185, 1,4.5,      -1,-19303,     0.4548       , "t=1.93s",_
 3695, 82, 185, 0,1.5,      -1,-11541,     0            , "t=6.3s",_
 3696, 82, 185, 1,6.5,       1,-11470,     0.07         , "t=4.07s",_
 3697, 83, 185, 0,4.5,      -1,-2240,      0            , "t=2#ms",_
 3698, 83, 185, 1,0.5,       1,-2156,      0.08         , "t=58us",_
 3699, 83, 185, 2,6.5,       1,-2060,      0.18         , "t=50us",_
 3700, 71, 186, 0,-77.777,   0,-30210,     0            , "t=2#s",_
 3701, 72, 186, 0,0,         1,-36420,     0            , "t=2.6m",_
 3702, 72, 186, 1,17,        1,-33450,     2.968        , "t=&gt;20s",_
 3703, 73, 186, 0,2,        -1,-38610,     0            , "t=10.5m",_
 3704, 73, 186, 1,9,         1,-38270,     0.336        , "t=1.54m",_
 3705, 74, 186, 0,0,         1,-42508.5,   0            , "t=stbl",_
 3706, 74, 186, 1,7,        -1,-40991.3,   1.5172       , "t=18us",_
 3707, 74, 186, 2,16,        1,-38965.7,   3.5428       , "t=2.0s",_
 3708, 75, 186, 0,1,        -1,-41927.1,   0            , "t=3.7183d",_
 3709, 75, 186, 1,8,         1,-41778.9,   0.1482       , "t=~200ky",_
 3710, 76, 186, 0,0,         1,-42999.9,   0            , "t=2.0Py",_
 3711, 77, 186, 0,5,         1,-39172,     0            , "t=16.64h",_
 3712, 77, 186, 1,2,        -1,-39171,     0.0008       , "t=1.92h",_
 3713, 78, 186, 0,0,         1,-37864,     0            , "t=2.08h",_
 3714, 79, 186, 0,3,        -1,-31715,     0            , "t=10.7m",_
 3715, 79, 186, 1,2,         1,-31487,     0.22777      , "t=110ns",_
 3716, 80, 186, 0,0,         1,-28539,     0            , "t=1.38m",_
 3717, 80, 186, 1,8,        -1,-26322,     2.2173       , "t=82us",_
 3718, 81, 186, 0,2,        -1,-19887,     0            , "t=40#s",_
 3719, 81, 186, 1,7,         1,-19860,     0.02         , "t=27.5s",_
 3720, 81, 186, 2,10,       -1,-19490,     0.4          , "t=2.9s",_
 3721, 82, 186, 0,0,         1,-14682,     0            , "t=4.82s",_
 3722, 83, 186, 0,3,         1,-3146,      0            , "t=14.8ms",_
 3723, 83, 186, 1,10,       -1,-2980,      0.17         , "t=9.8ms",_
 3724, 84, 186, 0,0,         1,4101,       0            , "t=34us",_
 3725, 71, 187, 0,3.5,       1,-27580,     0            , "t=1#s",_
 3726, 72, 187, 0,1.5,      -1,-32820,     0            , "t=30#s",_
 3727, 72, 187, 1,4.5,      -1,-32320,     0.5          , "t=270ns",_
 3728, 73, 187, 0,3.5,       1,-36900,     0            , "t=2.3m",_
 3729, 73, 187, 1,13.5,     -1,-35110,     1.789        , "t=22s",_
 3730, 73, 187, 2,20.5,      1,-33970,     2.935        , "t=&gt;5m",_
 3731, 74, 187, 0,1.5,      -1,-39904.0,   0            , "t=24.000h",_
 3732, 74, 187, 1,5.5,       1,-39493.9,   0.41006      , "t=1.38us",_
 3733, 75, 187, 0,2.5,       1,-41216.5,   0            , "t=43.3Gy",_
 3734, 75, 187, 1,4.5,      -1,-41010.3,   0.2062473    , "t=555.3ns",_
 3735, 75, 187, 2,10.5,      1,-39534.5,   1.682        , "t=354ns",_
 3736, 76, 187, 0,0.5,      -1,-41218.9,   0            , "t=stbl",_
 3737, 76, 187, 1,3.5,      -1,-41118.5,   0.10045      , "t=112ns",_
 3738, 76, 187, 2,5.5,       1,-40961.8,   0.2571       , "t=231us",_
 3739, 77, 187, 0,1.5,       1,-39549,     0            , "t=10.5h",_
 3740, 77, 187, 1,4.5,      -1,-39363,     0.18616      , "t=30.3ms",_
 3741, 77, 187, 2,5.5,      -1,-39115,     0.43375      , "t=152ns",_
 3742, 78, 187, 0,1.5,      -1,-36685,     0            , "t=2.35h",_
 3743, 78, 187, 1,5.5,       1,-36511,     0.17438      , "t=311us",_
 3744, 79, 187, 0,0.5,       1,-33028,     0            , "t=8.3m",_
 3745, 79, 187, 1,4.5,      -1,-32908,     0.12033      , "t=2.3s",_
 3746, 80, 187, 0,1.5,      -1,-28118,     0            , "t=1.9m",_
 3747, 80, 187, 1,6.5,       1,-28059,     0.059        , "t=2.4m",_
 3748, 81, 187, 0,0.5,       1,-22445,     0            , "t=~51s",_
 3749, 81, 187, 1,4.5,      -1,-22111,     0.334        , "t=15.60s",_
 3750, 81, 187, 2,-77.777,   0,-20970,     1.48         , "t=1.11us",_
 3751, 82, 187, 0,1.5,      -1,-14987,     0            , "t=15.2s",_
 3752, 82, 187, 1,6.5,       1,-14968,     0.019        , "t=18.3s",_
 3753, 83, 187, 0,4.5,      -1,-6383,      0            , "t=37ms",_
 3754, 83, 187, 1,0.5,       1,-6275,      0.108        , "t=370us",_
 3755, 83, 187, 2,6.5,       1,-6131,      0.252        , "t=7us",_
 3756, 84, 187, 0,0.5,      -1,2830,       0            , "t=1.40ms",_
 3757, 84, 187, 1,6.5,       1,2830,       0.004        , "t=0.5ms",_
 3758, 71, 188, 0,-77.777,   0,-23790,     0            , "t=300#ms",_
 3759, 72, 188, 0,0,         1,-30880,     0            , "t=20#s",_
 3760, 73, 188, 0,-77.777,   0,-33610,     0            , "t=19.6s",_
 3761, 73, 188, 1,-77.777,   0,-33320,     0.2924       , "t=3.6us",_
 3762, 74, 188, 0,0,         1,-38668,     0            , "t=69.78d",_
 3763, 74, 188, 1,8,        -1,-36739,     1.9293       , "t=109.5ns",_
 3764, 75, 188, 0,1,        -1,-39016.8,   0            , "t=17.0040h",_
 3765, 75, 188, 1,6,        -1,-38844.7,   0.172069     , "t=18.59m",_
 3766, 76, 188, 0,0,         1,-41137.2,   0            , "t=stbl",_
 3767, 77, 188, 0,1,        -1,-38345,     0            , "t=41.5h",_
 3768, 77, 188, 1,11,       -1,-37380,     0.97         , "t=4.2ms",_
 3769, 78, 188, 0,0,         1,-37821,     0            , "t=10.2d",_
 3770, 79, 188, 0,1,        -1,-32371.3,   0            , "t=8.84m",_
 3771, 80, 188, 0,0,         1,-30202,     0            , "t=3.25m",_
 3772, 80, 188, 1,12,        1,-27478,     2.7243       , "t=134ns",_
 3773, 81, 188, 0,2,        -1,-22336,     0            , "t=71s",_
 3774, 81, 188, 1,7,         1,-22308,     0.03         , "t=71s",_
 3775, 81, 188, 2,9,        -1,-22030,     0.31         , "t=41ms",_
 3776, 82, 188, 0,0,         1,-17815,     0            , "t=25.1s",_
 3777, 82, 188, 1,8,        -1,-15237,     2.5782       , "t=1.15us",_
 3778, 82, 188, 2,12,        1,-15105,     2.7097       , "t=94ns",_
 3779, 83, 188, 0,3,         1,-7195,      0            , "t=60ms",_
 3780, 83, 188, 1,7,         1,-7130,      0.066        , "t=&gt;5us",_
 3781, 83, 188, 2,10,       -1,-7040,      0.153        , "t=265ms",_
 3782, 84, 188, 0,0,         1,-544,       0            , "t=275us",_
 3783, 72, 189, 0,1.5,      -1,-27160,     0            , "t=2#s",_
 3784, 73, 189, 0,3.5,       1,-31830,     0            , "t=3#s",_
 3785, 73, 189, 1,-77.777,   0,-30230,     1.6          , "t=1.6us",_
 3786, 74, 189, 0,1.5,      -1,-35620,     0            , "t=10.7m",_
 3787, 75, 189, 0,2.5,       1,-37979,     0            , "t=24.3h",_
 3788, 75, 189, 1,14.5,      1,-36208,     1.7709       , "t=223us",_
 3789, 76, 189, 0,1.5,      -1,-38986.7,   0            , "t=stbl",_
 3790, 76, 189, 1,4.5,      -1,-38955.9,   0.030812     , "t=5.81h",_
 3791, 77, 189, 0,1.5,       1,-38450,     0            , "t=13.2d",_
 3792, 77, 189, 1,5.5,      -1,-38078,     0.37217      , "t=13.3ms",_
 3793, 77, 189, 2,12.5,      1,-36117,     2.3332       , "t=3.7ms",_
 3794, 78, 189, 0,1.5,      -1,-36469,     0            , "t=10.87h",_
 3795, 78, 189, 1,4.5,      -1,-36296,     0.1728       , "t=464ns",_
 3796, 78, 189, 2,6.5,       1,-36278,     0.1915       , "t=143us",_
 3797, 79, 189, 0,0.5,       1,-33582,     0            , "t=28.7m",_
 3798, 79, 189, 1,5.5,      -1,-33335,     0.24723      , "t=4.59m",_
 3799, 79, 189, 2,4.5,      -1,-33257,     0.32511      , "t=190ns",_
 3800, 80, 189, 0,1.5,      -1,-29630,     0            , "t=7.6m",_
 3801, 80, 189, 1,6.5,       1,-29548,     0.08         , "t=8.6m",_
 3802, 81, 189, 0,0.5,       1,-24616,     0            , "t=2.3m",_
 3803, 81, 189, 1,4.5,      -1,-24331,     0.285        , "t=1.4m",_
 3804, 82, 189, 0,1.5,      -1,-17844,     0            , "t=39s",_
 3805, 82, 189, 1,6.5,       1,-17804,     0.04         , "t=50.5s",_
 3806, 82, 189, 2,15.5,     -1,-15370,     2.475        , "t=26us",_
 3807, 83, 189, 0,4.5,      -1,-10065,     0            , "t=658ms",_
 3808, 83, 189, 1,0.5,       1,-9881,      0.184        , "t=4.9ms",_
 3809, 83, 189, 2,6.5,       1,-9707,      0.3576       , "t=880ns",_
 3810, 84, 189, 0,2.5,      -1,-1422,      0            , "t=3.8ms",_
 3811, 72, 190, 0,0,         1,-25030,     0            , "t=2#s",_
 3812, 73, 190, 0,3,         0,-28510,     0            , "t=5.3s",_
 3813, 73, 190, 1,-77.777,   0,-28310,     0.2          , "t=42ns",_
 3814, 74, 190, 0,0,         1,-34380,     0            , "t=30.0m",_
 3815, 74, 190, 1,8,         1,-32640,     1.742        , "t=111ns",_
 3816, 74, 190, 2,10,       -1,-32540,     1.839        , "t=166us",_
 3817, 75, 190, 0,2,        -1,-35640,     0            , "t=3.1m",_
 3818, 75, 190, 1,6,        -1,-35440,     0.204        , "t=3.2h",_
 3819, 76, 190, 0,0,         1,-38707.8,   0            , "t=stbl",_
 3820, 76, 190, 1,10,       -1,-37002.4,   1.7054       , "t=9.86m",_
 3821, 77, 190, 0,4,        -1,-36753.5,   0            , "t=11.78d",_
 3822, 77, 190, 1,1,        -1,-36727.4,   0.0261       , "t=1.120h",_
 3823, 77, 190, 2,4,         1,-36717.3,   0.036154     , "t=&gt;2us",_
 3824, 78, 190, 0,0,         1,-37306.5,   0            , "t=650Gy",_
 3825, 79, 190, 0,1,        -1,-32834,     0            , "t=42.8m",_
 3826, 79, 190, 1,11,       -1,-32630,     0.2          , "t=125ms",_
 3827, 80, 190, 0,0,         1,-31371,     0            , "t=20.0m",_
 3828, 81, 190, 0,2,        -1,-24372,     0            , "t=2.6m",_
 3829, 81, 190, 1,7,         1,-24289,     0.083        , "t=3.7m",_
 3830, 81, 190, 2,8,        -1,-24080,     0.29         , "t=750us",_
 3831, 82, 190, 0,0,         1,-20417,     0            , "t=71s",_
 3832, 82, 190, 1,10,        1,-17802,     2.6148       , "t=150ns",_
 3833, 82, 190, 2,12,        1,-17799,     2.618        , "t=24.3us",_
 3834, 83, 190, 0,3,         1,-10600,     0            , "t=6.3s",_
 3835, 83, 190, 1,10,       -1,-10470,     0.13         , "t=6.2s",_
 3836, 83, 190, 2,5,        -1,-10479,     0.121        , "t=175ns",_
 3837, 84, 190, 0,0,         1,-4564,      0            , "t=2.46ms",_
 3838, 73, 191, 0,3.5,       1,-26490,     0            , "t=3#s",_
 3839, 74, 191, 0,1.5,      -1,-31180,     0            , "t=45#s",_
 3840, 74, 191, 1,-77.777,   0,-30950,     0.235        , "t=340ns",_
 3841, 75, 191, 0,1.5,       1,-34350,     0            , "t=9.8m",_
 3842, 75, 191, 1,12.5,     -1,-32749,     1.6015       , "t=51us",_
 3843, 76, 191, 0,4.5,      -1,-36395.2,   0            , "t=14.99d",_
 3844, 76, 191, 1,1.5,      -1,-36320.8,   0.074382     , "t=13.10h",_
 3845, 77, 191, 0,1.5,       1,-36708.8,   0            , "t=stbl",_
 3846, 77, 191, 1,5.5,      -1,-36537.5,   0.17129      , "t=4.899s",_
 3847, 77, 191, 2,15.5,      1,-34607.8,   2.101        , "t=5.7s",_
 3848, 78, 191, 0,1.5,      -1,-35698,     0            , "t=2.83d",_
 3849, 78, 191, 1,4.5,      -1,-35597,     0.100663     , "t=&gt;1us",_
 3850, 78, 191, 2,6.5,       1,-35549,     0.149035     , "t=95us",_
 3851, 79, 191, 0,1.5,       1,-33798,     0            , "t=3.18h",_
 3852, 79, 191, 1,5.5,      -1,-33532,     0.2662       , "t=920ms",_
 3853, 79, 191, 2,15.5,      1,-31308,     2.4896       , "t=402ns",_
 3854, 80, 191, 0,1.5,      -1,-30592,     0            , "t=49m",_
 3855, 80, 191, 1,6.5,       1,-30460,     0.128        , "t=50.8m",_
 3856, 81, 191, 0,0.5,       1,-26283,     0            , "t=20#m",_
 3857, 81, 191, 1,4.5,      -1,-25986,     0.297        , "t=5.22m",_
 3858, 82, 191, 0,1.5,      -1,-20230,     0            , "t=1.33m",_
 3859, 82, 191, 1,6.5,       1,-20231,     0            , "t=2.18m",_
 3860, 82, 191, 2,16.5,      1,-17610,     2.62         , "t=180ns",_
 3861, 83, 191, 0,4.5,      -1,-13239,     0            , "t=11.7s",_
 3862, 83, 191, 1,0.5,       1,-12997,     0.242        , "t=114ms",_
 3863, 83, 191, 2,6.5,       1,-12809,     0.4297       , "t=562ns",_
 3864, 84, 191, 0,1.5,      -1,-5069,      0            , "t=22ms",_
 3865, 84, 191, 1,6.5,       1,-5008,      0.061        , "t=93ms",_
 3866, 85, 191, 0,0.5,       1,3864,       0            , "t=2.1ms",_
 3867, 85, 191, 1,3.5,      -1,3922,       0.058        , "t=2.2ms",_
 3868, 73, 192, 0,2,         0,-23060,     0            , "t=2.2s",_
 3869, 74, 192, 0,0,         1,-29650,     0            , "t=1#m",_
 3870, 75, 192, 0,-77.777,   0,-31590,     0            , "t=16.0s",_
 3871, 75, 192, 1,-77.777,   0,-31430,     0.159        , "t=88us",_
 3872, 75, 192, 2,-77.777,   0,-31320,     0.267        , "t=70s",_
 3873, 76, 192, 0,0,         1,-35882.2,   0            , "t=stbl",_
 3874, 76, 192, 1,10,       -1,-33866.8,   2.0154       , "t=5.9s",_
 3875, 76, 192, 2,20,        1,-31301.9,   4.5803       , "t=204ns",_
 3876, 77, 192, 0,4,         1,-34835.6,   0            , "t=73.830d",_
 3877, 77, 192, 1,1,        -1,-34778.9,   0.05672      , "t=1.45m",_
 3878, 77, 192, 2,11,       -1,-34667.5,   0.16814      , "t=241y",_
 3879, 78, 192, 0,0,         1,-36288.5,   0            , "t=stbl",_
 3880, 78, 192, 1,10,       -1,-34116.1,   2.17237      , "t=272ns",_
 3881, 79, 192, 0,1,        -1,-32772,     0            , "t=4.94h",_
 3882, 79, 192, 1,5,         1,-32637,     0.13541      , "t=29ms",_
 3883, 79, 192, 2,11,       -1,-32340,     0.4316       , "t=160ms",_
 3884, 80, 192, 0,0,         1,-32012,     0            , "t=4.85h",_
 3885, 81, 192, 0,2,        -1,-25870,     0            , "t=9.6m",_
 3886, 81, 192, 1,7,         1,-25730,     0.138        , "t=10.8m",_
 3887, 81, 192, 2,8,        -1,-25480,     0.388        , "t=296ns",_
 3888, 82, 192, 0,0,         1,-22556,     0            , "t=3.5m",_
 3889, 82, 192, 1,10,        1,-19975,     2.5811       , "t=166ns",_
 3890, 82, 192, 2,12,        1,-19931,     2.6251       , "t=1.09us",_
 3891, 83, 192, 0,3,         1,-13530,     0            , "t=34.6s",_
 3892, 83, 192, 1,10,       -1,-13398,     0.14         , "t=39.6s",_
 3893, 84, 192, 0,0,         1,-8071,      0            , "t=32.2ms",_
 3894, 84, 192, 1,11,       -1,-5776,      2.2946       , "t=580ns",_
 3895, 85, 192, 0,3,         1,2926,       0            , "t=11.5ms",_
 3896, 85, 192, 1,9,        -1,2926,       0            , "t=88ms",_
 3897, 73, 193, 0,3.5,       1,-20870,     0            , "t=500#ms",_
 3898, 74, 193, 0,1.5,      -1,-26290,     0            , "t=3#s",_
 3899, 75, 193, 0,2.5,       1,-30230,     0            , "t=20#s",_
 3900, 75, 193, 1,4.5,      -1,-30080,     0.146        , "t=69us",_
 3901, 76, 193, 0,1.5,      -1,-33394.3,   0            , "t=29.830h",_
 3902, 76, 193, 1,-77.777,   0,-33152.3,   0.242        , "t=132ns",_
 3903, 77, 193, 0,1.5,       1,-34536.2,   0            , "t=stbl",_
 3904, 77, 193, 1,5.5,      -1,-34456.0,   0.080239     , "t=10.53d",_
 3905, 77, 193, 2,15.5,      1,-32258.7,   2.2775       , "t=124.8us",_
 3906, 78, 193, 0,0.5,      -1,-34479.6,   0            , "t=50y",_
 3907, 78, 193, 1,6.5,       1,-34329.8,   0.14978      , "t=4.33d",_
 3908, 79, 193, 0,1.5,       1,-33405,     0            , "t=17.65h",_
 3909, 79, 193, 1,5.5,      -1,-33115,     0.29019      , "t=3.9s",_
 3910, 79, 193, 2,15.5,      1,-30919,     2.4865       , "t=150ns",_
 3911, 80, 193, 0,1.5,      -1,-31062,     0            , "t=3.80h",_
 3912, 80, 193, 1,6.5,       1,-30921,     0.14076      , "t=11.8h",_
 3913, 81, 193, 0,0.5,       1,-27477,     0            , "t=21.6m",_
 3914, 81, 193, 1,4.5,      -1,-27105,     0.372        , "t=2.11m",_
 3915, 82, 193, 0,1.5,      -1,-22190,     0            , "t=5#m",_
 3916, 82, 193, 1,6.5,       1,-22060,     0.13         , "t=5.8m",_
 3917, 82, 193, 2,16.5,      1,-19450,     2.742        , "t=180ns",_
 3918, 83, 193, 0,4.5,      -1,-15885,     0            , "t=63.6s",_
 3919, 83, 193, 1,0.5,       1,-15580,     0.305        , "t=3.07s",_
 3920, 83, 193, 2,6.5,       1,-15280,     0.6055       , "t=153ns",_
 3921, 84, 193, 0,1.5,      -1,-8325,      0            , "t=388ms",_
 3922, 84, 193, 1,6.5,       1,-8225,      0.1          , "t=245ms",_
 3923, 85, 193, 0,0.5,       1,-67,        0            , "t=29ms",_
 3924, 85, 193, 1,3.5,      -1,-59,        0.008        , "t=21ms",_
 3925, 85, 193, 2,6.5,       1,-25,        0.042        , "t=28ms",_
 3926, 86, 193, 0,1.5,      -1,9043,       0            , "t=1.15ms",_
 3927, 73, 194, 0,-77.777,   0,-17300,     0            , "t=300#ms",_
 3928, 74, 194, 0,0,         1,-24530,     0            , "t=5#s",_
 3929, 75, 194, 0,0,         1,-27240,     0            , "t=5s",_
 3930, 75, 194, 1,11,       -1,-26960,     0.285        , "t=25s",_
 3931, 75, 194, 2,-77.777,   0,-26410,     0.833        , "t=100s",_
 3932, 76, 194, 0,0,         1,-32435.1,   0            , "t=6.0y",_
 3933, 77, 194, 0,1,        -1,-32531.7,   0            , "t=19.28h",_
 3934, 77, 194, 1,4,         1,-32384.6,   0.147072     , "t=31.85ms",_
 3935, 77, 194, 2,10,        0,-32160,     0.37         , "t=171d",_
 3936, 78, 194, 0,0,         1,-34760.1,   0            , "t=stbl",_
 3937, 79, 194, 0,1,        -1,-32211.9,   0            , "t=38.02h",_
 3938, 79, 194, 1,5,         1,-32104.5,   0.1074       , "t=600ms",_
 3939, 79, 194, 2,11,       -1,-31736.1,   0.4758       , "t=420ms",_
 3940, 80, 194, 0,0,         1,-32183.9,   0            , "t=447y",_
 3941, 81, 194, 0,2,        -1,-26937,     0            , "t=33.0m",_
 3942, 81, 194, 1,7,         1,-26677,     0.26         , "t=32.8m",_
 3943, 82, 194, 0,0,         1,-24208,     0            , "t=10.7m",_
 3944, 82, 194, 1,12,        1,-21580,     2.6281       , "t=370ns",_
 3945, 82, 194, 2,11,       -1,-21275,     2.933        , "t=133ns",_
 3946, 83, 194, 0,3,         1,-16029,     0            , "t=95s",_
 3947, 83, 194, 1,6,         1,-15880,     0.15         , "t=125s",_
 3948, 83, 194, 2,10,       -1,-15849,     0.18         , "t=115s",_
 3949, 84, 194, 0,0,         1,-11005,     0            , "t=392ms",_
 3950, 84, 194, 1,11,       -1,-8480,      2.5252       , "t=15us",_
 3951, 85, 194, 0,4,        -1,-720,       0            , "t=286ms",_
 3952, 85, 194, 1,9,        -1,-740,      -0.02         , "t=323ms",_
 3953, 86, 194, 0,0,         1,5723,       0            , "t=780us",_
 3954, 74, 195, 0,2.5,      -1,-21010,     0            , "t=3#s",_
 3955, 75, 195, 0,2.5,       1,-25580,     0            , "t=6s",_
 3956, 76, 195, 0,1.5,      -1,-29510,     0            , "t=6.5m",_
 3957, 76, 195, 1,6.5,       1,-29060,     0.454        , "t=2.0h",_
 3958, 77, 195, 0,1.5,       1,-31692.3,   0            , "t=2.29h",_
 3959, 77, 195, 1,5.5,      -1,-31592,     0.1          , "t=3.67h",_
 3960, 77, 195, 2,13.5,      1,-29338,     2.354        , "t=4.4us",_
 3961, 78, 195, 0,0.5,      -1,-32793.8,   0            , "t=stbl",_
 3962, 78, 195, 1,6.5,       1,-32534.7,   0.259077     , "t=4.010d",_
 3963, 79, 195, 0,1.5,       1,-32567.0,   0            , "t=186.01d",_
 3964, 79, 195, 1,5.5,      -1,-32248.4,   0.31858      , "t=30.5s",_
 3965, 79, 195, 2,15.5,     -1,-30067,     2.5          , "t=12.89us",_
 3966, 80, 195, 0,0.5,      -1,-31013,     0            , "t=10.69h",_
 3967, 80, 195, 1,6.5,       1,-30837,     0.17607      , "t=41.60h",_
 3968, 81, 195, 0,0.5,       1,-28155,     0            , "t=1.16h",_
 3969, 81, 195, 1,4.5,      -1,-27672,     0.48263      , "t=3.6s",_
 3970, 82, 195, 0,1.5,      -1,-23708,     0            , "t=~15m",_
 3971, 82, 195, 1,6.5,       1,-23505,     0.2029       , "t=15.0m",_
 3972, 82, 195, 2,10.5,     -1,-21949,     1.759        , "t=10.0us",_
 3973, 83, 195, 0,4.5,      -1,-18026,     0            , "t=183s",_
 3974, 83, 195, 1,0.5,       1,-17626,     0.399        , "t=87s",_
 3975, 83, 195, 2,14.5,     -1,-15631,     2.3955       , "t=750ns",_
 3976, 84, 195, 0,1.5,      -1,-11060,     0            , "t=4.64s",_
 3977, 84, 195, 1,6.5,       1,-10965,     0.09         , "t=1.92s",_
 3978, 85, 195, 0,0.5,       1,-3470,      0            , "t=290ms",_
 3979, 85, 195, 1,3.5,      -1,-3441,      0.029        , "t=143ms",_
 3980, 86, 195, 0,1.5,      -1,5050,       0            , "t=7ms",_
 3981, 86, 195, 1,6.5,       1,5131,       0.08         , "t=6ms",_
 3982, 74, 196, 0,0,         1,-18880,     0            , "t=3#s",_
 3983, 75, 196, 0,-77.777,   0,-22540,     0            , "t=2.4s",_
 3984, 75, 196, 1,-77.777,   0,-22420,     0.12         , "t=3.6us",_
 3985, 76, 196, 0,0,         1,-28280,     0            , "t=34.9m",_
 3986, 77, 196, 0,0,        -1,-29440,     0            , "t=52s",_
 3987, 77, 196, 1,10,        0,-29227,     0.21         , "t=1.40h",_
 3988, 78, 196, 0,0,         1,-32644.5,   0            , "t=stbl",_
 3989, 79, 196, 0,2,        -1,-31138.7,   0            , "t=6.1669d",_
 3990, 79, 196, 1,5,         1,-31054,     0.084656     , "t=8.1s",_
 3991, 79, 196, 2,12,       -1,-30543,     0.59566      , "t=9.6h",_
 3992, 80, 196, 0,0,         1,-31825.9,   0            , "t=stbl",_
 3993, 81, 196, 0,2,        -1,-27497,     0            , "t=1.84h",_
 3994, 81, 196, 1,7,         1,-27103,     0.3942       , "t=1.41h",_
 3995, 82, 196, 0,0,         1,-25348,     0            , "t=37m",_
 3996, 82, 196, 1,4,         1,-23610,     1.73827      , "t=&lt;1us",_
 3997, 82, 196, 2,5,        -1,-23550,     1.79751      , "t=140ns",_
 3998, 83, 196, 0,3,         1,-18009,     0            , "t=5.1m",_
 3999, 83, 196, 1,7,         1,-17843,     0.1664       , "t=0.6s",_
 4000, 83, 196, 2,10,       -1,-17737,     0.272        , "t=4.00m",_
 4001, 84, 196, 0,0,         1,-13473,     0            , "t=5.56s",_
 4002, 84, 196, 1,11,       -1,-10979,     2.4939       , "t=856ns",_
 4003, 85, 196, 0,3,         1,-3910,      0            , "t=388ms",_
 4004, 85, 196, 1,10,       -1,-3950,     -0.04         , "t=20#ms",_
 4005, 85, 196, 2,5,         1,-3750,      0.1579       , "t=11us",_
 4006, 86, 196, 0,0,         1,1971,       0            , "t=4.7ms",_
 4007, 74, 197, 0,2.5,      -1,-15140,     0            , "t=1#s",_
 4008, 75, 197, 0,2.5,       1,-20500,     0            , "t=300#ms",_
 4009, 76, 197, 0,2.5,      -1,-25310,     0            , "t=2.8m",_
 4010, 77, 197, 0,1.5,       1,-28264,     0            , "t=5.8m",_
 4011, 77, 197, 1,5.5,      -1,-28149,     0.115        , "t=8.9m",_
 4012, 77, 197, 2,-77.777,   0,-27860,     0.4          , "t=30us",_
 4013, 78, 197, 0,0.5,      -1,-30419.7,   0            , "t=19.8915h",_
 4014, 78, 197, 1,6.5,       1,-30020.1,   0.39959      , "t=95.41m",_
 4015, 79, 197, 0,1.5,       1,-31139.7,   0            , "t=stbl",_
 4016, 79, 197, 1,5.5,      -1,-30730.6,   0.40915      , "t=7.73s",_
 4017, 79, 197, 2,13.5,      1,-28607.2,   2.5325       , "t=150ns",_
 4018, 80, 197, 0,0.5,      -1,-30540,     0            , "t=64.94h",_
 4019, 80, 197, 1,6.5,       1,-30241,     0.29893      , "t=23.8h",_
 4020, 81, 197, 0,0.5,       1,-28342,     0            , "t=2.84h",_
 4021, 81, 197, 1,4.5,      -1,-27734,     0.60822      , "t=540ms",_
 4022, 82, 197, 0,1.5,      -1,-24745,     0            , "t=8.1m",_
 4023, 82, 197, 1,6.5,       1,-24426,     0.31931      , "t=42.9m",_
 4024, 82, 197, 2,10.5,     -1,-22831,     1.9141       , "t=1.15us",_
 4025, 83, 197, 0,4.5,      -1,-19687,     0            , "t=9.33m",_
 4026, 83, 197, 1,0.5,       1,-19155,     0.533        , "t=5.04m",_
 4027, 83, 197, 2,14.5,     -1,-17284,     2.403        , "t=263ns",_
 4028, 84, 197, 0,1.5,      -1,-13360,     0            , "t=53.6s",_
 4029, 84, 197, 1,6.5,       1,-13130,     0.23         , "t=25.8s",_
 4030, 85, 197, 0,4.5,      -1,-6355,      0            , "t=388.2ms",_
 4031, 85, 197, 1,0.5,       1,-6311,      0.045        , "t=2.0s",_
 4032, 85, 197, 2,6.5,       1,-6044,      0.3107       , "t=1.3us",_
 4033, 86, 197, 0,1.5,      -1,1510,       0            , "t=54ms",_
 4034, 86, 197, 1,6.5,       1,1709,       0.199        , "t=25.6ms",_
 4035, 87, 197, 0,3.5,      -1,10250,      0            , "t=2.33ms",_
 4036, 75, 198, 0,-77.777,   0,-17140,     0            , "t=300#ms",_
 4037, 76, 198, 0,0,         1,-23840,     0            , "t=1#m",_
 4038, 77, 198, 0,-77.777,   0,-25820,     0            , "t=8s",_
 4039, 78, 198, 0,0,         1,-29904.0,   0            , "t=stbl",_
 4040, 79, 198, 0,2,        -1,-29580.8,   0            , "t=2.6941d",_
 4041, 79, 198, 1,5,         1,-29268.6,   0.3122227    , "t=124ns",_
 4042, 79, 198, 2,12,       -1,-28768.9,   0.8119       , "t=2.272d",_
 4043, 80, 198, 0,0,         1,-30954.3,   0            , "t=stbl",_
 4044, 81, 198, 0,2,        -1,-27529,     0            , "t=5.3h",_
 4045, 81, 198, 1,7,         1,-26985,     0.5436       , "t=1.87h",_
 4046, 81, 198, 2,5,         0,-26842,     0.6868       , "t=150ns",_
 4047, 82, 198, 0,0,         1,-26067,     0            , "t=2.4h",_
 4048, 82, 198, 1,7,        -1,-23926,     2.1414       , "t=4.19us",_
 4049, 82, 198, 2,9,        -1,-23836,     2.2314       , "t=137ns",_
 4050, 83, 198, 0,3,         1,-19369,     0            , "t=10.3m",_
 4051, 83, 198, 1,7,         1,-19085,     0.28         , "t=11.6m",_
 4052, 83, 198, 2,10,       -1,-18837,     0.53         , "t=7.7s",_
 4053, 84, 198, 0,0,         1,-15473,     0            , "t=1.760m",_
 4054, 84, 198, 1,11,       -1,-12907,     2.56592      , "t=200ns",_
 4055, 84, 198, 2,12,        1,-12730,     2.74         , "t=750ns",_
 4056, 85, 198, 0,3,         1,-6715,      0            , "t=3.0s",_
 4057, 85, 198, 1,10,       -1,-6430,      0.284        , "t=1.21s",_
 4058, 86, 198, 0,0,         1,-1230,      0            , "t=65ms",_
 4059, 87, 198, 0,0,         0,9570,       0            , "t=15ms",_
 4060, 87, 198, 1,0,         0,9580,       0            , "t=1.1ms",_
 4061, 75, 199, 0,2.5,       1,-14860,     0            , "t=100#ms",_
 4062, 76, 199, 0,2.5,      -1,-20480,     0            , "t=6s",_
 4063, 77, 199, 0,1.5,       1,-24400,     0            , "t=7s",_
 4064, 77, 199, 1,5.5,      -1,-24270,     0.13         , "t=235ns",_
 4065, 78, 199, 0,2.5,      -1,-27388.7,   0            , "t=30.80m",_
 4066, 78, 199, 1,6.5,       1,-26964.7,   0.424        , "t=13.6s",_
 4067, 79, 199, 0,1.5,       1,-29093.7,   0            , "t=3.139d",_
 4068, 79, 199, 1,5.5,      -1,-28544.8,   0.5489405    , "t=440us",_
 4069, 80, 199, 0,0.5,      -1,-29546.1,   0            , "t=stbl",_
 4070, 80, 199, 1,6.5,       1,-29013.6,   0.53248      , "t=42.67m",_
 4071, 81, 199, 0,0.5,       1,-28059,     0            , "t=7.42h",_
 4072, 81, 199, 1,4.5,      -1,-27310,     0.74887      , "t=28.4ms",_
 4073, 82, 199, 0,1.5,      -1,-25232,     0            , "t=90m",_
 4074, 82, 199, 1,6.5,       1,-24803,     0.4295       , "t=12.2m",_
 4075, 82, 199, 2,14.5,     -1,-22668,     2.5638       , "t=10.1us",_
 4076, 83, 199, 0,4.5,      -1,-20798,     0            , "t=27m",_
 4077, 83, 199, 1,0.5,       1,-20131,     0.667        , "t=24.70m",_
 4078, 83, 199, 2,12.5,      1,-18851,     1.947        , "t=100ns",_
 4079, 84, 199, 0,1.5,      -1,-15208,     0            , "t=5.47m",_
 4080, 84, 199, 1,6.5,       1,-14897,     0.3119       , "t=4.17m",_
 4081, 85, 199, 0,4.5,      -1,-8823,      0            , "t=7.02s",_
 4082, 85, 199, 1,0.5,       1,-8579,      0.244        , "t=273ms",_
 4083, 85, 199, 2,6.5,       1,-8250,      0.5729       , "t=70ns",_
 4084, 86, 199, 0,1.5,      -1,-1500,      0            , "t=590ms",_
 4085, 86, 199, 1,6.5,       1,-1337,      0.16         , "t=310ms",_
 4086, 87, 199, 0,0.5,       1,6771,       0            , "t=6.6ms",_
 4087, 87, 199, 1,3.5,      -1,6817,       0.045        , "t=6.5ms",_
 4088, 87, 199, 2,-77.777,   0,7020,       0.25         , "t=2.2ms",_
 4089, 76, 200, 0,0,         1,-18780,     0            , "t=7s",_
 4090, 77, 200, 0,2,        -1,-21610,     0            , "t=43s",_
 4091, 78, 200, 0,0,         1,-26599,     0            , "t=12.6h",_
 4092, 79, 200, 0,1,        -1,-27240,     0            , "t=48.4m",_
 4093, 79, 200, 1,12,       -1,-26233,     1.01         , "t=18.7h",_
 4094, 80, 200, 0,0,         1,-29503.3,   0            , "t=stbl",_
 4095, 81, 200, 0,2,        -1,-27047,     0            , "t=26.1h",_
 4096, 81, 200, 1,7,         1,-26293,     0.7536       , "t=34.0ms",_
 4097, 81, 200, 2,5,         1,-26285,     0.762        , "t=330ns",_
 4098, 82, 200, 0,0,         1,-26251,     0            , "t=21.5h",_
 4099, 82, 200, 1,9,        -1,-24068,     2.1833       , "t=448ns",_
 4100, 82, 200, 2,12,        1,-23245,     3.0058       , "t=199ns",_
 4101, 83, 200, 0,7,         1,-20371,     0            , "t=36.4m",_
 4102, 83, 200, 1,2,         1,-20270,     0.1          , "t=31m",_
 4103, 83, 200, 2,10,       -1,-19943,     0.4282       , "t=400ms",_
 4104, 84, 200, 0,0,         1,-16942,     0            , "t=11.51m",_
 4105, 84, 200, 1,11,       -1,-14346,     2.5961       , "t=100ns",_
 4106, 84, 200, 2,12,        1,-14125,     2.817        , "t=268ns",_
 4107, 85, 200, 0,3,         1,-8988,      0            , "t=43.2s",_
 4108, 85, 200, 1,7,         1,-8875,      0.1129       , "t=47s",_
 4109, 85, 200, 2,10,       -1,-8644,      0.3438       , "t=8.0s",_
 4110, 86, 200, 0,0,         1,-4005,      0            , "t=1.09s",_
 4111, 86, 200, 1,-77.777,   0,-1685,      2.32         , "t=28us",_
 4112, 87, 200, 0,3,         1,6130,       0            , "t=47.5ms",_
 4113, 87, 200, 1,10,       -1,6180,       0.05         , "t=190ms",_
 4114, 87, 200, 2,-77.777,   0,6280,       0.15         , "t=790ns",_
 4115, 76, 201, 0,0.5,      -1,-15240,     0            , "t=1#s",_
 4116, 77, 201, 0,1.5,       1,-19900,     0            , "t=21s",_
 4117, 78, 201, 0,2.5,      -1,-23740,     0            , "t=2.5m",_
 4118, 79, 201, 0,1.5,       1,-26401,     0            , "t=26.0m",_
 4119, 79, 201, 1,5.5,      -1,-25807,     0.594        , "t=730us",_
 4120, 79, 201, 2,-77.777,   0,-24791,     1.61         , "t=5.6us",_
 4121, 80, 201, 0,1.5,      -1,-27662.5,   0            , "t=stbl",_
 4122, 80, 201, 1,6.5,       1,-26896.3,   0.76622      , "t=94.0us",_
 4123, 81, 201, 0,0.5,       1,-27181,     0            , "t=3.0442d",_
 4124, 81, 201, 1,4.5,      -1,-26262,     0.91916      , "t=2.01ms",_
 4125, 82, 201, 0,2.5,      -1,-25271,     0            , "t=9.33h",_
 4126, 82, 201, 1,6.5,       1,-24642,     0.6291       , "t=60.8s",_
 4127, 82, 201, 2,14.5,     -1,-22333,     2.938        , "t=508ns",_
 4128, 83, 201, 0,4.5,      -1,-21416,     0            , "t=103m",_
 4129, 83, 201, 1,0.5,       1,-20570,     0.84635      , "t=57.5m",_
 4130, 83, 201, 2,12.5,      1,-19443,     1.973        , "t=118ns",_
 4131, 84, 201, 0,1.5,      -1,-16521,     0            , "t=15.6m",_
 4132, 84, 201, 1,6.5,       1,-16097,     0.4238       , "t=8.96m",_
 4133, 85, 201, 0,4.5,      -1,-10789,     0            , "t=85.2s",_
 4134, 85, 201, 1,0.5,       1,-10330,     0.459        , "t=45ms",_
 4135, 85, 201, 2,14.5,      1,-8470,      2.319        , "t=3.39us",_
 4136, 86, 201, 0,1.5,      -1,-4070,      0            , "t=7.0s",_
 4137, 86, 201, 1,6.5,       1,-3790,      0.28         , "t=3.8s",_
 4138, 87, 201, 0,4.5,      -1,3589,       0            , "t=62.8ms",_
 4139, 87, 201, 1,0.5,       1,3715,       0.127        , "t=17ms",_
 4140, 87, 201, 2,6.5,       1,3790,       0.2          , "t=890ns",_
 4141, 88, 201, 0,1.5,      -1,11937,      0            , "t=20ms",_
 4142, 88, 201, 1,6.5,       1,12200,      0.263        , "t=6ms",_
 4143, 76, 202, 0,0,         1,-13090,     0            , "t=200#ms",_
 4144, 77, 202, 0,2,        -1,-16780,     0            , "t=11s",_
 4145, 77, 202, 1,-77.777,   0,-14780,     2            , "t=3.4us",_
 4146, 78, 202, 0,0,         1,-22692,     0            , "t=44h",_
 4147, 78, 202, 1,7,        -1,-20904,     1.7885       , "t=141us",_
 4148, 79, 202, 0,1,        -1,-24353,     0            , "t=28.4s",_
 4149, 80, 202, 0,0,         1,-27345.3,   0            , "t=stbl",_
 4150, 81, 202, 0,2,        -1,-25980.2,   0            , "t=12.31d",_
 4151, 81, 202, 1,7,         1,-25030.0,   0.95019      , "t=591us",_
 4152, 82, 202, 0,0,         1,-25941,     0            , "t=52.5ky",_
 4153, 82, 202, 1,9,        -1,-23771,     2.16985      , "t=3.54h",_
 4154, 82, 202, 2,16,        1,-21800,     4.14         , "t=110ns",_
 4155, 83, 202, 0,5,         1,-20741,     0            , "t=1.72h",_
 4156, 83, 202, 1,10,       -1,-20116,     0.625        , "t=3.04us",_
 4157, 83, 202, 2,17,        1,-18124,     2.617        , "t=310ns",_
 4158, 84, 202, 0,0,         1,-17942,     0            , "t=44.6m",_
 4159, 84, 202, 1,8,         1,-16230,     1.712        , "t=110ns",_
 4160, 85, 202, 0,3,         1,-10591,     0            , "t=184s",_
 4161, 85, 202, 1,7,         1,-10401,     0.19         , "t=182s",_
 4162, 85, 202, 2,10,       -1,-10010,     0.58         , "t=460ms",_
 4163, 86, 202, 0,0,         1,-6275,      0            , "t=9.7s",_
 4164, 86, 202, 1,11,       -1,-3970,      2.31         , "t=2.22us",_
 4165, 87, 202, 0,3,         1,3096,       0            , "t=372ms",_
 4166, 87, 202, 1,10,       -1,3370,       0.274        , "t=286ms",_
 4167, 88, 202, 0,0,         1,9075,       0            , "t=4.1ms",_
 4168, 76, 203, 0,4.5,       1,-7640,      0            , "t=100#ms",_
 4169, 77, 203, 0,1.5,       1,-14690,     0            , "t=6#s",_
 4170, 77, 203, 1,11.5,      1,-12550,     2.14         , "t=798ns",_
 4171, 78, 203, 0,0.5,      -1,-19630,     0            , "t=22s",_
 4172, 78, 203, 1,16.5,      1,-16530,     3.1          , "t=641ns",_
 4173, 79, 203, 0,1.5,       1,-23143,     0            , "t=60s",_
 4174, 79, 203, 1,5.5,      -1,-22502,     0.641        , "t=140us",_
 4175, 80, 203, 0,2.5,      -1,-25269.3,   0            , "t=46.613d",_
 4176, 80, 203, 1,6.5,       1,-24336.2,   0.93314      , "t=21.9us",_
 4177, 80, 203, 2,26.5,      1,-16988.3,   8.281        , "t=146ns",_
 4178, 81, 203, 0,0.5,       1,-25761.4,   0            , "t=stbl",_
 4179, 81, 203, 1,12.5,      1,-22200,     3.565        , "t=7.7us",_
 4180, 82, 203, 0,2.5,      -1,-24787,     0            , "t=51.916h",_
 4181, 82, 203, 1,6.5,       1,-23962,     0.8252       , "t=6.21s",_
 4182, 82, 203, 2,14.5,     -1,-21838,     2.9492       , "t=480ms",_
 4183, 83, 203, 0,4.5,      -1,-21525,     0            , "t=11.76h",_
 4184, 83, 203, 1,0.5,       1,-20427,     1.09812      , "t=305ms",_
 4185, 83, 203, 2,12.5,      1,-19484,     2.0415       , "t=194ns",_
 4186, 84, 203, 0,2.5,      -1,-17311,     0            , "t=36.7m",_
 4187, 84, 203, 1,6.5,       1,-16669,     0.64168      , "t=45s",_
 4188, 84, 203, 2,-77.777,   0,-15153,     2.1585       , "t=&gt;200ns",_
 4189, 85, 203, 0,4.5,      -1,-12163,     0            , "t=7.4m",_
 4190, 86, 203, 0,1.5,      -1,-6154,      0            , "t=44s",_
 4191, 86, 203, 1,6.5,       1,-5791,      0.363        , "t=26.9s",_
 4192, 87, 203, 0,4.5,      -1,876,        0            , "t=550ms",_
 4193, 87, 203, 1,0.5,       1,1237,       0.361        , "t=43ms",_
 4194, 87, 203, 2,6.5,       1,1300,       0.426        , "t=370ns",_
 4195, 88, 203, 0,1.5,      -1,8660,       0            , "t=36ms",_
 4196, 88, 203, 1,6.5,       1,8851,       0.19         , "t=25ms",_
 4197, 77, 204, 0,-77.777,   0,-9690,      0            , "t=1#s",_
 4198, 78, 204, 0,0,         1,-17920,     0            , "t=10.3s",_
 4199, 78, 204, 1,5,        -1,-15930,     1.9951       , "t=5.5us",_
 4200, 78, 204, 2,7,        -1,-15890,     2.035        , "t=55us",_
 4201, 79, 204, 0,2,        -1,-20650,     0            , "t=38.3s",_
 4202, 79, 204, 1,16,        1,-16830,     3.816        , "t=2.1us",_
 4203, 80, 204, 0,0,         1,-24690.1,   0            , "t=stbl",_
 4204, 80, 204, 1,4,         1,-20301.4,   4.3887       , "t=29ns",_
 4205, 80, 204, 2,2,         1,-17464.0,   7.2261       , "t=&gt;480ns",_
 4206, 81, 204, 0,2,        -1,-24346.1,   0            , "t=3.783y",_
 4207, 81, 204, 1,7,         1,-23242.0,   1.1041       , "t=61.7us",_
 4208, 81, 204, 2,12,       -1,-22027.1,   2.319        , "t=2.6us",_
 4209, 82, 204, 0,0,         1,-25109.9,   0            , "t=stbl",_
 4210, 82, 204, 1,4,         1,-23835.8,   1.27413      , "t=265ns",_
 4211, 82, 204, 2,9,        -1,-22924.0,   2.18588      , "t=66.93m",_
 4212, 83, 204, 0,6,         1,-20646,     0            , "t=11.22h",_
 4213, 83, 204, 1,10,       -1,-19841,     0.8055       , "t=13.0ms",_
 4214, 83, 204, 2,17,        1,-17813,     2.8334       , "t=1.07ms",_
 4215, 84, 204, 0,0,         1,-18341,     0            , "t=3.519h",_
 4216, 84, 204, 1,8,         1,-16702,     1.63903      , "t=158.6ns",_
 4217, 85, 204, 0,7,         1,-11875,     0            , "t=9.12m",_
 4218, 85, 204, 1,10,       -1,-11288,     0.5873       , "t=108ms",_
 4219, 86, 204, 0,0,         1,-7970,      0            , "t=1.242m",_
 4220, 87, 204, 0,3,         1,607,        0            , "t=1.75s",_
 4221, 87, 204, 1,7,         1,658,        0.05         , "t=2.41s",_
 4222, 87, 204, 2,10,       -1,934,        0.326        , "t=1.65s",_
 4223, 88, 204, 0,0,         1,6057,       0            , "t=60ms",_
 4224, 77, 205, 0,1.5,       1,-5960,      0            , "t=300#ms",_
 4225, 78, 205, 0,4.5,       1,-12970,     0            , "t=5#s",_
 4226, 79, 205, 0,1.5,       1,-18770,     0            , "t=32.5s",_
 4227, 79, 205, 1,5.5,      -1,-17860,     0.907        , "t=6s",_
 4228, 79, 205, 2,9.5,       1,-15920,     2.85         , "t=163ns",_
 4229, 80, 205, 0,0.5,      -1,-22288,     0            , "t=5.14m",_
 4230, 80, 205, 1,6.5,       1,-20732,     1.5564       , "t=1.09ms",_
 4231, 80, 205, 2,11.5,     -1,-18972,     3.3158       , "t=5.89us",_
 4232, 81, 205, 0,0.5,       1,-23820.9,   0            , "t=stbl",_
 4233, 81, 205, 1,12.5,      1,-20530.3,   3.2906       , "t=2.6us",_
 4234, 81, 205, 2,17.5,     -1,-18985.3,   4.8356       , "t=235ns",_
 4235, 82, 205, 0,2.5,      -1,-23770.2,   0            , "t=17.3My",_
 4236, 82, 205, 1,0.5,      -1,-23767.9,   0.002329     , "t=24.2us",_
 4237, 82, 205, 2,6.5,       1,-22756.4,   1.01385      , "t=5.55ms",_
 4238, 83, 205, 0,4.5,      -1,-21065,     0            , "t=15.31d",_
 4239, 83, 205, 1,0.5,       1,-19568,     1.49717      , "t=7.9us",_
 4240, 83, 205, 2,12.5,      1,-18926,     2.139        , "t=220ns",_
 4241, 84, 205, 0,2.5,      -1,-17521,     0            , "t=1.74h",_
 4242, 84, 205, 1,0.5,      -1,-17378,     0.143166     , "t=310ns",_
 4243, 84, 205, 2,6.5,       1,-16641,     0.88031      , "t=645us",_
 4244, 85, 205, 0,4.5,      -1,-12972,     0            , "t=33.8m",_
 4245, 85, 205, 1,14.5,      1,-10632,     2.33965      , "t=7.76us",_
 4246, 86, 205, 0,2.5,      -1,-7710,      0            , "t=2.83m",_
 4247, 86, 205, 1,6.5,       1,-7053,      0.6571       , "t=&gt;10s",_
 4248, 87, 205, 0,4.5,      -1,-1310,      0            , "t=3.82s",_
 4249, 87, 205, 1,6.5,       1,-766,       0.544        , "t=80ns",_
 4250, 87, 205, 2,0.5,       1,-701,       0.609        , "t=1.15ms",_
 4251, 88, 205, 0,1.5,      -1,5840,       0            , "t=220ms",_
 4252, 88, 205, 1,6.5,       1,6140,       0.3          , "t=180ms",_
 4253, 89, 205, 0,4.5,      -1,14110,      0            , "t=80ms",_
 4254, 78, 206, 0,0,         1,-9630,      0            , "t=5#s",_
 4255, 79, 206, 0,5,         1,-14220,     0            , "t=47s",_
 4256, 80, 206, 0,0,         1,-20946,     0            , "t=8.32m",_
 4257, 80, 206, 1,5,        -1,-18844,     2.1024       , "t=2.09us",_
 4258, 80, 206, 2,10,        1,-17224,     3.7223       , "t=106ns",_
 4259, 81, 206, 0,0,        -1,-22253.4,   0            , "t=4.202m",_
 4260, 81, 206, 1,12,       -1,-19610.3,   2.6431       , "t=3.74m",_
 4261, 82, 206, 0,0,         1,-23785.6,   0            , "t=stbl",_
 4262, 82, 206, 1,7,        -1,-21585.4,   2.20016      , "t=125us",_
 4263, 82, 206, 2,12,        1,-19758.3,   4.0273       , "t=202ns",_
 4264, 83, 206, 0,6,         1,-20028,     0            , "t=6.243d",_
 4265, 83, 206, 1,4,         1,-19968,     0.059897     , "t=7.7us",_
 4266, 83, 206, 2,10,       -1,-18983,     1.0448       , "t=890us",_
 4267, 84, 206, 0,0,         1,-18189,     0            , "t=8.8d",_
 4268, 84, 206, 1,8,         1,-16603,     1.5859       , "t=232ns",_
 4269, 84, 206, 2,9,        -1,-15927,     2.26209      , "t=1.05us",_
 4270, 85, 206, 0,5,         1,-12430,     0            , "t=30.6m",_
 4271, 85, 206, 1,10,       -1,-11620,     0.81         , "t=813ns",_
 4272, 86, 206, 0,0,         1,-9133,      0            , "t=5.67m",_
 4273, 87, 206, 0,3,         1,-1242,      0            , "t=~16s",_
 4274, 87, 206, 1,7,         1,-1048,      0.19         , "t=~16s",_
 4275, 87, 206, 2,10,       -1,-517,       0.73         , "t=700ms",_
 4276, 87, 206, 0,0,         0,-1140,      0.1          , "t=R= ?",_
 4277, 88, 206, 0,0,         1,3566,       0            , "t=240ms",_
 4278, 89, 206, 0,3,         1,13480,      0            , "t=25ms",_
 4279, 89, 206, 1,10,       -1,13700,      0.22         , "t=41ms",_
 4280, 78, 207, 0,4.5,       1,-4540,      0            , "t=1#s",_
 4281, 79, 207, 0,1.5,       1,-10810,     0            , "t=10#s",_
 4282, 80, 207, 0,4.5,       1,-16487,     0            , "t=2.9m",_
 4283, 81, 207, 0,0.5,       1,-21034,     0            , "t=4.77m",_
 4284, 81, 207, 1,5.5,      -1,-19686,     1.34818      , "t=1.33s",_
 4285, 82, 207, 0,0.5,      -1,-22452.0,   0            , "t=stbl",_
 4286, 82, 207, 1,6.5,       1,-20818.6,   1.633356     , "t=806ms",_
 4287, 83, 207, 0,4.5,      -1,-20054.6,   0            , "t=31.20y",_
 4288, 83, 207, 1,10.5,      1,-17953.0,   2.10161      , "t=182us",_
 4289, 84, 207, 0,2.5,      -1,-17146,     0            , "t=5.80h",_
 4290, 84, 207, 1,0.5,      -1,-17077,     0.068557     , "t=205ns",_
 4291, 84, 207, 2,6.5,       1,-16031,     1.115076     , "t=49us",_
 4292, 85, 207, 0,4.5,      -1,-13227,     0            , "t=1.81h",_
 4293, 85, 207, 1,12.5,      1,-11110,     2.1173       , "t=108ns",_
 4294, 86, 207, 0,2.5,      -1,-8635,      0            , "t=9.25m",_
 4295, 86, 207, 1,6.5,       1,-7736,      0.8991       , "t=184.5us",_
 4296, 87, 207, 0,4.5,      -1,-2844,      0            , "t=14.8s",_
 4297, 88, 207, 0,2.5,      -1,3540,       0            , "t=1.38s",_
 4298, 88, 207, 1,6.5,       1,4102,       0.56         , "t=57ms",_
 4299, 89, 207, 0,4.5,      -1,11150,      0            , "t=31ms",_
 4300, 78, 208, 0,0,         1,-990,       0            , "t=1#s",_
 4301, 79, 208, 0,-77.777,   0,-6100,      0            , "t=10#s",_
 4302, 80, 208, 0,0,         1,-13270,     0            , "t=42m",_
 4303, 80, 208, 1,8,         1,-11930,     1.338        , "t=99ns",_
 4304, 81, 208, 0,5,         1,-16750.1,   0            , "t=3.053m",_
 4305, 82, 208, 0,0,         1,-21748.6,   0            , "t=stbl",_
 4306, 82, 208, 1,10,        1,-16853.4,   4.89523      , "t=500ns",_
 4307, 83, 208, 0,5,         1,-18870.2,   0            , "t=368ky",_
 4308, 83, 208, 1,10,       -1,-17299.1,   1.5711       , "t=2.58ms",_
 4309, 84, 208, 0,0,         1,-17469.6,   0            , "t=2.898y",_
 4310, 84, 208, 1,8,         1,-15941.4,   1.52822      , "t=350ns",_
 4311, 85, 208, 0,6,         1,-12470,     0            , "t=1.63h",_
 4312, 85, 208, 1,16,       -1,-10194,     2.2764       , "t=1.5us",_
 4313, 86, 208, 0,0,         1,-9656,      0            , "t=24.35m",_
 4314, 86, 208, 1,8,         1,-7828,      1.8283       , "t=487ns",_
 4315, 87, 208, 0,7,         1,-2666,      0            , "t=59.1s",_
 4316, 87, 208, 1,10,       -1,-1839,      0.827        , "t=432ns",_
 4317, 88, 208, 0,0,         1,1728,       0            , "t=1.110s",_
 4318, 88, 208, 1,8,         1,3875,       2.1474       , "t=263ns",_
 4319, 89, 208, 0,3,         1,10750,      0            , "t=97ms",_
 4320, 89, 208, 1,10,       -1,11258,      0.5          , "t=28ms",_
 4321, 90, 208, 0,0,         1,16680,      0            , "t=2.4ms",_
 4322, 79, 209, 0,1.5,       1,-2540,      0            , "t=1#s",_
 4323, 80, 209, 0,4.5,       1,-8640,      0            , "t=38s",_
 4324, 81, 209, 0,0.5,       1,-13645,     0            , "t=2.162m",_
 4325, 82, 209, 0,4.5,       1,-17614.6,   0            , "t=3.234h",_
 4326, 83, 209, 0,4.5,      -1,-18258.7,   0            , "t=20.1Ey",_
 4327, 84, 209, 0,0.5,      -1,-16366.1,   0            , "t=124y",_
 4328, 84, 209, 1,15.5,     -1,-12100.7,   4.2654       , "t=119ns",_
 4329, 85, 209, 0,4.5,      -1,-12883,     0            , "t=5.42h",_
 4330, 85, 209, 1,14.5,      1,-10454,     2.42932      , "t=916ns",_
 4331, 86, 209, 0,2.5,      -1,-8941,      0            , "t=28.8m",_
 4332, 86, 209, 1,6.5,       1,-7767,      1.17401      , "t=13.4us",_
 4333, 86, 209, 2,17.5,      1,-5304,      3.63681      , "t=3.0us",_
 4334, 87, 209, 0,4.5,      -1,-3770,      0            , "t=50.5s",_
 4335, 87, 209, 1,22.5,     -1,890,        4.6598       , "t=420ns",_
 4336, 88, 209, 0,2.5,      -1,1858,       0            , "t=4.71s",_
 4337, 88, 209, 1,6.5,       1,2740,       0.8824       , "t=117us",_
 4338, 89, 209, 0,4.5,      -1,8840,       0            , "t=94ms",_
 4339, 90, 209, 0,2.5,      -1,16370,      0            , "t=60#ms",_
 4340, 90, 209, 1,6.5,       1,16840,      0.47         , "t=3.1ms",_
 4341, 79, 210, 0,-77.777,   0,2330,       0            , "t=1#s",_
 4342, 80, 210, 0,0,         1,-5370,      0            , "t=64s",_
 4343, 80, 210, 1,3,        -1,-4710,      0.663        , "t=2.1us",_
 4344, 80, 210, 2,8,         1,-3960,      1.406        , "t=2us",_
 4345, 81, 210, 0,5,         1,-9247,      0            , "t=1.30m",_
 4346, 82, 210, 0,0,         1,-14728.5,   0            , "t=22.20y",_
 4347, 82, 210, 1,8,         1,-13451,     1.278        , "t=201ns",_
 4348, 83, 210, 0,1,        -1,-14792.0,   0            , "t=5.012d",_
 4349, 83, 210, 1,9,        -1,-14520.7,   0.27131      , "t=3.04My",_
 4350, 84, 210, 0,0,         1,-15953.1,   0            , "t=138.376d",_
 4351, 84, 210, 1,8,         1,-14396.1,   1.55697      , "t=98.9ns",_
 4352, 84, 210, 2,16,        1,-10895.5,   5.05765      , "t=263ns",_
 4353, 85, 210, 0,5,         1,-11972,     0            , "t=8.1h",_
 4354, 85, 210, 1,15,       -1,-9422,      2.5496       , "t=482ns",_
 4355, 85, 210, 2,19,        1,-7944,      4.0277       , "t=5.66us",_
 4356, 86, 210, 0,0,         1,-9605,      0            , "t=2.4h",_
 4357, 86, 210, 1,8,         1,-7900,      1.71         , "t=644ns",_
 4358, 86, 210, 2,17,       -1,-5750,      3.857        , "t=1.06us",_
 4359, 87, 210, 0,6,         1,-3333,      0            , "t=3.18m",_
 4360, 88, 210, 0,0,         1,443,        0            , "t=4.0s",_
 4361, 88, 210, 1,8,         1,2494,       2.0509       , "t=2.29us",_
 4362, 89, 210, 0,7,         1,8790,       0            , "t=350ms",_
 4363, 90, 210, 0,0,         1,14059,      0            , "t=16.0ms",_
 4364, 80, 211, 0,4.5,       1,-620,       0            , "t=26s",_
 4365, 81, 211, 0,0.5,       1,-6080,      0            , "t=80s",_
 4366, 82, 211, 0,4.5,       1,-10492.9,   0            , "t=36.164m",_
 4367, 82, 211, 1,13.5,      1,-8789,      1.704        , "t=159ns",_
 4368, 83, 211, 0,4.5,      -1,-11859,     0            , "t=2.14m",_
 4369, 83, 211, 1,12.5,     -1,-10602,     1.257        , "t=1.4us",_
 4370, 84, 211, 0,4.5,       1,-12432.6,   0            , "t=516ms",_
 4371, 84, 211, 1,12.5,      1,-10970,     1.462        , "t=25.2s",_
 4372, 84, 211, 2,15.5,     -1,-10298,     2.135        , "t=243ns",_
 4373, 85, 211, 0,4.5,      -1,-11647.3,   0            , "t=7.214h",_
 4374, 85, 211, 1,19.5,     -1,-6832.8,    4.8145       , "t=4.23us",_
 4375, 86, 211, 0,0.5,      -1,-8755,      0            , "t=14.6h",_
 4376, 86, 211, 1,8.5,      -1,-7152,      1.603        , "t=596ns",_
 4377, 86, 211, 2,31.5,     -1,125,        8.88         , "t=201ns",_
 4378, 87, 211, 0,4.5,      -1,-4140,      0            , "t=3.10m",_
 4379, 87, 211, 1,14.5,      1,-1717,      2.42316      , "t=146ns",_
 4380, 87, 211, 2,22.5,     -1,517,        4.6573       , "t=123ns",_
 4381, 88, 211, 0,2.5,      -1,832,        0            , "t=13.2s",_
 4382, 88, 211, 1,6.5,       1,2030,       1.1981       , "t=9.5us",_
 4383, 89, 211, 0,4.5,      -1,7200,       0            , "t=213ms",_
 4384, 90, 211, 0,2.5,      -1,13910,      0            , "t=48ms",_
 4385, 91, 211, 0,4.5,      -1,22080,      0            , "t=3#ms",_
 4386, 80, 212, 0,0,         1,2760,       0            , "t=1#m",_
 4387, 81, 212, 0,5,         1,-1550,      0            , "t=31s",_
 4388, 82, 212, 0,0,         1,-7548.8,    0            , "t=10.64h",_
 4389, 82, 212, 1,8,         1,-6213.8,    1.335        , "t=6.0us",_
 4390, 83, 212, 0,1,        -1,-8118.0,    0            , "t=60.55m",_
 4391, 83, 212, 1,8,        -1,-7870,      0.25         , "t=25.0m",_
 4392, 83, 212, 2,0,         0,-6639,      1.479        , "t=7.0m",_
 4393, 84, 212, 0,0,         1,-10369.5,   0            , "t=294.7ns",_
 4394, 84, 212, 1,18,        1,-7446,      2.923        , "t=45.1s",_
 4395, 85, 212, 0,1,        -1,-8628.2,    0            , "t=314ms",_
 4396, 85, 212, 1,9,        -1,-8405.3,    0.2229       , "t=119ms",_
 4397, 85, 212, 2,25,       -1,-3856.6,    4.7716       , "t=152us",_
 4398, 86, 212, 0,0,         1,-8660,      0            , "t=23.9m",_
 4399, 86, 212, 1,6,         1,-7020,      1.6398       , "t=118ns",_
 4400, 86, 212, 2,8,         1,-6966,      1.694        , "t=910ns",_
 4401, 87, 212, 0,5,         1,-3516,      0            , "t=20.0m",_
 4402, 87, 212, 1,11,        1,-1965,      1.5514       , "t=31.9us",_
 4403, 87, 212, 2,15,       -1,-1024,      2.4922       , "t=604ns",_
 4404, 88, 212, 0,0,         1,-199,       0            , "t=13.0s",_
 4405, 88, 212, 1,8,         1,1759,       1.9584       , "t=8.1us",_
 4406, 88, 212, 2,11,       -1,2414,       2.6134       , "t=512ns",_
 4407, 89, 212, 0,6,         1,7280,       0            , "t=895ms",_
 4408, 90, 212, 0,0,         1,12111,      0            , "t=31.7ms",_
 4409, 91, 212, 0,7,         1,21590,      0            , "t=7.5ms",_
 4410, 80, 213, 0,2.5,       1,7670,       0            , "t=1#s",_
 4411, 81, 213, 0,0.5,       1,1784,       0            , "t=24s",_
 4412, 82, 213, 0,4.5,       1,-3204,      0            , "t=10.2m",_
 4413, 83, 213, 0,4.5,      -1,-5232,      0            , "t=45.61m",_
 4414, 83, 213, 1,12.5,     -1,-3930,      1.3          , "t=&gt;168s",_
 4415, 84, 213, 0,4.5,       1,-6654,      0            , "t=3.708us",_
 4416, 85, 213, 0,4.5,      -1,-6580,      0            , "t=125ns",_
 4417, 85, 213, 1,-77.777,   0,-5210,      1.37         , "t=110ns",_
 4418, 85, 213, 2,24.5,      1,-3600,      2.98         , "t=45us",_
 4419, 86, 213, 0,4.5,       1,-5696,      0            , "t=19.5ms",_
 4420, 86, 213, 1,12.5,      1,-3990,      1.71         , "t=1.00us",_
 4421, 86, 213, 2,15.5,     -1,-3460,      2.24         , "t=1.36us",_
 4422, 87, 213, 0,4.5,      -1,-3553,      0            , "t=34.14s",_
 4423, 87, 213, 1,10.5,     -1,-1963,      1.59041      , "t=505ns",_
 4424, 87, 213, 2,14.5,      1,-1015,      2.53762      , "t=238ns",_
 4425, 88, 213, 0,0.5,      -1,346,        0            , "t=2.73m",_
 4426, 88, 213, 1,8.5,      -1,2114,       1.768        , "t=2.20ms",_
 4427, 89, 213, 0,4.5,      -1,6155,       0            , "t=738ms",_
 4428, 90, 213, 0,2.5,      -1,12120,      0            , "t=144ms",_
 4429, 90, 213, 1,6.5,       1,13300,      1.18         , "t=1.4us",_
 4430, 91, 213, 0,4.5,      -1,19660,      0            , "t=7ms",_
 4431, 80, 214, 0,0,         1,11180,      0            , "t=1#s",_
 4432, 81, 214, 0,5,         1,6470,       0            , "t=11s",_
 4433, 82, 214, 0,0,         1,-182.8,     0            , "t=27.06m",_
 4434, 82, 214, 1,8,         1,1237,       1.42         , "t=6.2us",_
 4435, 83, 214, 0,1,        -1,-1201,      0            , "t=19.9m",_
 4436, 83, 214, 1,8,        -1,-1000,      0.2          , "t=&gt;93s",_
 4437, 84, 214, 0,0,         1,-4470.0,    0            , "t=163.72us",_
 4438, 85, 214, 0,1,        -1,-3380,      0            , "t=558ns",_
 4439, 85, 214, 1,-77.777,   0,-3321,      0.059        , "t=265ns",_
 4440, 85, 214, 2,9,        -1,-3146,      0.234        , "t=760ns",_
 4441, 86, 214, 0,0,         1,-4320,      0            , "t=270ns",_
 4442, 86, 214, 1,22,        1,275,        4.5954       , "t=245ns",_
 4443, 87, 214, 0,1,        -1,-959,       0            , "t=5.18ms",_
 4444, 87, 214, 1,8,        -1,-837,       0.122        , "t=3.35ms",_
 4445, 87, 214, 2,11,        1,-321,       0.638        , "t=103ns",_
 4446, 88, 214, 0,0,         1,93,         0            , "t=2.437s",_
 4447, 88, 214, 1,6,         1,1913,       1.8197       , "t=118ns",_
 4448, 88, 214, 2,8,         1,1958,       1.8652       , "t=67.3us",_
 4449, 88, 214, 0,25,       -1,6670,       6.577        , "t=128ns",_
 4450, 89, 214, 0,5,         1,6444,       0            , "t=8.2s",_
 4451, 90, 214, 0,0,         1,10695,      0            , "t=87ms",_
 4452, 90, 214, 1,8,         1,12876,      2.181        , "t=1.24us",_
 4453, 91, 214, 0,-77.777,   0,19490,      0            , "t=17ms",_
 4454, 80, 215, 0,1.5,       1,16210,      0            , "t=1#s",_
 4455, 81, 215, 0,0.5,       1,9910,       0            , "t=10s",_
 4456, 82, 215, 0,4.5,       1,4340,       0            , "t=2.34m",_
 4457, 83, 215, 0,4.5,      -1,1629,       0            , "t=7.6m",_
 4458, 83, 215, 1,12.5,     -1,2996,       1.367        , "t=36.9s",_
 4459, 84, 215, 0,4.5,       1,-541.7,     0            , "t=1.781ms",_
 4460, 85, 215, 0,4.5,      -1,-1256,      0            , "t=100us",_
 4461, 86, 215, 0,4.5,       1,-1169,      0            , "t=2.30us",_
 4462, 87, 215, 0,4.5,      -1,318,        0            , "t=86ns",_
 4463, 88, 215, 0,4.5,       1,2534,       0            , "t=1.67ms",_
 4464, 88, 215, 1,12.5,      1,4412,       1.8778       , "t=7.31us",_
 4465, 88, 215, 2,14.5,     -1,4781,       2.2469       , "t=1.39us",_
 4466, 89, 215, 0,4.5,      -1,6031,       0            , "t=170ms",_
 4467, 89, 215, 1,10.5,     -1,7827,       1.796        , "t=185ns",_
 4468, 89, 215, 2,14.5,      1,8520,       2.49         , "t=335ns",_
 4469, 90, 215, 0,0.5,      -1,10922,      0            , "t=1.2s",_
 4470, 90, 215, 1,4.5,       1,12390,      1.47         , "t=770ns",_
 4471, 91, 215, 0,4.5,      -1,17860,      0            , "t=14ms",_
 4472, 92, 215, 0,2.5,      -1,24920,      0            , "t=1.4ms",_
 4473, 80, 216, 0,0,         1,19860,      0            , "t=100#ms",_
 4474, 81, 216, 0,5,         1,14720,      0            , "t=6s",_
 4475, 82, 216, 0,0,         1,7480,       0            , "t=1.65m",_
 4476, 82, 216, 1,8,         1,8990,       1.514        , "t=400ns",_
 4477, 83, 216, 0,6,        -1,5874,       0            , "t=2.25m",_
 4478, 83, 216, 1,3,        -1,5898,       0.024        , "t=6.6m",_
 4479, 84, 216, 0,0,         1,1782.4,     0            , "t=145ms",_
 4480, 85, 216, 0,1,        -1,2257,       0            , "t=300us",_
 4481, 85, 216, 1,9,        -1,2417,       0.161        , "t=100#us",_
 4482, 86, 216, 0,0,         1,253,        0            , "t=45us",_
 4483, 87, 216, 0,1,        -1,2971,       0            , "t=700ns",_
 4484, 87, 216, 1,9,        -1,3190,       0.219        , "t=850ns",_
 4485, 88, 216, 0,0,         1,3291,       0            , "t=182ns",_
 4486, 89, 216, 0,1,        -1,8144,       0            , "t=440us",_
 4487, 89, 216, 1,9,        -1,8188,       0.044        , "t=441us",_
 4488, 89, 216, 2,-77.777,   0,8560,       0.42         , "t=~300ns",_
 4489, 90, 216, 0,0,         1,10298,      0            , "t=26.0ms",_
 4490, 90, 216, 1,8,         1,12342,      2.043        , "t=134us",_
 4491, 90, 216, 2,11,       -1,12945,      2.6468       , "t=580ns",_
 4492, 91, 216, 0,-77.777,   0,17800,      0            , "t=105ms",_
 4493, 92, 216, 0,0,         1,23066,      0            , "t=6.9ms",_
 4494, 92, 216, 1,8,         1,25320,      2.25         , "t=1.4ms",_
 4495, 81, 217, 0,0.5,       1,18310,      0            , "t=1#s",_
 4496, 82, 217, 0,4.5,       1,12240,      0            , "t=20s",_
 4497, 83, 217, 0,4.5,      -1,8730,       0            , "t=98.5s",_
 4498, 83, 217, 1,12.5,     -1,10210,      1.48         , "t=2.70us",_
 4499, 84, 217, 0,4.5,       1,5884,       0            , "t=1.514s",_
 4500, 85, 217, 0,4.5,      -1,4395,       0            , "t=32.62ms",_
 4501, 86, 217, 0,4.5,       1,3659,       0            , "t=540us",_
 4502, 87, 217, 0,4.5,      -1,4315,       0            , "t=16.8us",_
 4503, 88, 217, 0,4.5,       1,5890,       0            , "t=1.63us",_
 4504, 89, 217, 0,4.5,      -1,8704,       0            , "t=69ns",_
 4505, 89, 217, 1,14.5,      1,10716,      2.012        , "t=740ns",_
 4506, 90, 217, 0,4.5,       1,12206,      0            , "t=247us",_
 4507, 90, 217, 1,7.5,      -1,12880,      0.6738       , "t=141ns",_
 4508, 90, 217, 2,12.5,      1,14510,      2.307        , "t=71us",_
 4509, 91, 217, 0,4.5,      -1,17068,      0            , "t=3.48ms",_
 4510, 91, 217, 1,11.5,     -1,18929,      1.86         , "t=1.08ms",_
 4511, 92, 217, 0,0.5,      -1,22970,      0            , "t=800us",_
 4512, 81, 218, 0,5,         1,23180,      0            , "t=200#ms",_
 4513, 82, 218, 0,0,         1,15450,      0            , "t=15s",_
 4514, 83, 218, 0,6,        -1,13216,      0            , "t=33s",_
 4515, 84, 218, 0,0,         1,8356.9,     0            , "t=3.098m",_
 4516, 85, 218, 0,1,        -1,8098,       0            , "t=1.5s",_
 4517, 86, 218, 0,0,         1,5217.3,     0            , "t=33.75ms",_
 4518, 87, 218, 0,1,        -1,7059,       0            , "t=1.0ms",_
 4519, 87, 218, 1,8,        -1,7146,       0.086        , "t=22.0ms",_
 4520, 88, 218, 0,0,         1,6651,       0            , "t=25.2us",_
 4521, 89, 218, 0,1,        -1,10840,      0            , "t=1.00us",_
 4522, 89, 218, 1,9,        -1,10990,      0.15         , "t=32ns",_
 4523, 89, 218, 2,11,        1,11370,      0.53         , "t=103ns",_
 4524, 90, 218, 0,0,         1,12367,      0            , "t=117ns",_
 4525, 91, 218, 0,-77.777,   0,18684,      0            , "t=113us",_
 4526, 92, 218, 0,0,         1,21895,      0            , "t=550us",_
 4527, 92, 218, 1,8,         1,24004,      2.109        , "t=660us",_
 4528, 82, 219, 0,4.5,       1,20280,      0            , "t=10#s",_
 4529, 83, 219, 0,4.5,      -1,16280,      0            , "t=8.7s",_
 4530, 84, 219, 0,4.5,       1,12681,      0            , "t=10.3m",_
 4531, 85, 219, 0,4.5,      -1,10396,      0            , "t=56s",_
 4532, 86, 219, 0,2.5,       1,8829.4,     0            , "t=3.96s",_
 4533, 87, 219, 0,4.5,      -1,8618,       0            , "t=20ms",_
 4534, 88, 219, 0,3.5,       1,9394,       0            , "t=10ms",_
 4535, 89, 219, 0,4.5,      -1,11570,      0            , "t=11.8us",_
 4536, 90, 219, 0,4.5,       1,14470,      0            , "t=1.021us",_
 4537, 91, 219, 0,4.5,      -1,18540,      0            , "t=53ns",_
 4538, 92, 219, 0,4.5,       1,23290,      0            , "t=55us",_
 4539, 93, 219, 0,4.5,      -1,29460,      0            , "t=&lt;5us",_
 4540, 82, 220, 0,0,         1,23670,      0            , "t=30#s",_
 4541, 83, 220, 0,1,        -1,20820,      0            , "t=9.5s",_
 4542, 84, 220, 0,0,         1,15263,      0            , "t=40#s",_
 4543, 85, 220, 0,3,        -1,14376,      0            , "t=3.71m",_
 4544, 86, 220, 0,0,         1,10612.1,    0            , "t=55.6s",_
 4545, 87, 220, 0,1,         1,11482,      0            , "t=27.4s",_
 4546, 88, 220, 0,0,         1,10270,      0            , "t=17.9ms",_
 4547, 89, 220, 0,3,        -1,13744,      0            , "t=26.36ms",_
 4548, 90, 220, 0,0,         1,14669,      0            , "t=9.7us",_
 4549, 91, 220, 0,1,        -1,20220,      0            , "t=780ns",_
 4550, 92, 220, 0,0,         1,22930,      0            , "t=60#ns",_
 4551, 93, 220, 0,1,        -1,30310,      0            , "t=30#ns",_
 4552, 83, 221, 0,4.5,      -1,24100,      0            , "t=5#s",_
 4553, 84, 221, 0,4.5,       1,19774,      0            , "t=2.2m",_
 4554, 85, 221, 0,1.5,      -1,16783,      0            , "t=2.3m",_
 4555, 86, 221, 0,3.5,       1,14471,      0            , "t=25.7m",_
 4556, 87, 221, 0,2.5,      -1,13277,      0            , "t=4.801m",_
 4557, 88, 221, 0,2.5,       1,12964,      0            , "t=28s",_
 4558, 89, 221, 0,4.5,      -1,14520,      0            , "t=52ms",_
 4559, 90, 221, 0,3.5,       1,16940,      0            , "t=1.78ms",_
 4560, 91, 221, 0,4.5,      -1,20380,      0            , "t=5.9us",_
 4561, 92, 221, 0,4.5,       1,24520,      0            , "t=660ns",_
 4562, 93, 221, 0,4.5,      -1,29850,      0            , "t=30#ns",_
 4563, 83, 222, 0,1,        -1,28730,      0            , "t=2#s",_
 4564, 84, 222, 0,0,         1,22490,      0            , "t=9.1m",_
 4565, 85, 222, 0,-77.777,   0,20953,      0            , "t=54s",_
 4566, 86, 222, 0,0,         1,16372.2,    0            , "t=3.8215d",_
 4567, 87, 222, 0,2,        -1,16378,      0            , "t=14.2m",_
 4568, 88, 222, 0,0,         1,14320,      0            , "t=33.6s",_
 4569, 89, 222, 0,1,        -1,16621,      0            , "t=5.0s",_
 4570, 89, 222, 1,0,         0,16820,      0.2          , "t=1.05m",_
 4571, 90, 222, 0,0,         1,17203,      0            , "t=2.24ms",_
 4572, 91, 222, 0,-77.777,   0,22160,      0            , "t=3.2ms",_
 4573, 92, 222, 0,0,         1,24270,      0            , "t=4.7us",_
 4574, 93, 222, 0,1,        -1,31020,      0            , "t=700#ns",_
 4575, 83, 223, 0,4.5,      -1,32140,      0            , "t=1#s",_
 4576, 84, 223, 0,4.5,       1,27080,      0            , "t=1#m",_
 4577, 85, 223, 0,1.5,      -1,23428,      0            , "t=50s",_
 4578, 86, 223, 0,3.5,       1,20390,      0            , "t=24.3m",_
 4579, 87, 223, 0,1.5,      -1,18382.4,    0            , "t=22.00m",_
 4580, 88, 223, 0,1.5,       1,17233.3,    0            , "t=11.4377d",_
 4581, 89, 223, 0,2.5,      -1,17826,      0            , "t=2.10m",_
 4582, 90, 223, 0,2.5,       1,19386,      0            , "t=600ms",_
 4583, 91, 223, 0,4.5,      -1,22320,      0            , "t=5.1ms",_
 4584, 92, 223, 0,3.5,       1,25840,      0            , "t=21us",_
 4585, 93, 223, 0,4.5,      -1,30600,      0            , "t=1#us",_
 4586, 83, 224, 0,1,        -1,36830,      0            , "t=300#ms",_
 4587, 84, 224, 0,0,         1,29910,      0            , "t=1#m",_
 4588, 85, 224, 0,-77.777,   0,27711,      0            , "t=2.5m",_
 4589, 86, 224, 0,0,         1,22445,      0            , "t=107m",_
 4590, 87, 224, 0,1,        -1,21749,      0            , "t=3.33m",_
 4591, 87, 224, 0,-77.777,   0,21850,      0.1          , "t=contamntn",_
 4592, 88, 224, 0,0,         1,18825.9,    0            , "t=3.6319d",_
 4593, 89, 224, 0,0,        -1,20234,      0            , "t=2.78h",_
 4594, 90, 224, 0,0,         1,19994,      0            , "t=1.04s",_
 4595, 91, 224, 0,5,        -1,23862,      0            , "t=846ms",_
 4596, 92, 224, 0,0,         1,25722,      0            , "t=396us",_
 4597, 93, 224, 0,1,        -1,31880,      0            , "t=100#us",_
 4598, 84, 225, 0,4.5,       1,34530,      0            , "t=20#s",_
 4599, 85, 225, 0,0.5,       1,30400,      0            , "t=2#m",_
 4600, 86, 225, 0,3.5,      -1,26534,      0            , "t=4.66m",_
 4601, 87, 225, 0,1.5,      -1,23821,      0            , "t=3.95m",_
 4602, 88, 225, 0,0.5,       1,21993.1,    0            , "t=14.9d",_
 4603, 89, 225, 0,1.5,      -1,21637,      0            , "t=9.920d",_
 4604, 90, 225, 0,1.5,       1,22310,      0            , "t=8.75m",_
 4605, 91, 225, 0,2.5,      -1,24340,      0            , "t=1.7s",_
 4606, 92, 225, 0,2.5,       1,27380,      0            , "t=61ms",_
 4607, 93, 225, 0,4.5,      -1,31590,      0            , "t=6ms",_
 4608, 84, 226, 0,0,         1,37550,      0            , "t=20#s",_
 4609, 85, 226, 0,-77.777,   0,34610,      0            , "t=20#s",_
 4610, 86, 226, 0,0,         1,28747,      0            , "t=7.4m",_
 4611, 87, 226, 0,1,        -1,27521,      0            , "t=49s",_
 4612, 88, 226, 0,0,         1,23667.8,    0            , "t=1.600ky",_
 4613, 89, 226, 0,1,        -1,24309,      0            , "t=29.37h",_
 4614, 90, 226, 0,0,         1,23198,      0            , "t=30.70m",_
 4615, 91, 226, 0,-77.777,   0,26033,      0            , "t=1.8m",_
 4616, 92, 226, 0,0,         1,27329,      0            , "t=269ms",_
 4617, 93, 226, 0,-77.777,   0,32780,      0            , "t=35ms",_
 4618, 84, 227, 0,4.5,       1,42280,      0            , "t=5#s",_
 4619, 85, 227, 0,0.5,       1,37480,      0            , "t=20#s",_
 4620, 86, 227, 0,2.5,       1,32886,      0            , "t=20.2s",_
 4621, 87, 227, 0,0.5,       1,29682,      0            , "t=2.47m",_
 4622, 88, 227, 0,1.5,       1,27177.7,    0            , "t=42.2m",_
 4623, 89, 227, 0,1.5,      -1,25849.6,    0            , "t=21.772y",_
 4624, 90, 227, 0,0.5,       1,25804.8,    0            , "t=18.697d",_
 4625, 91, 227, 0,2.5,      -1,26831,      0            , "t=38.3m",_
 4626, 92, 227, 0,1.5,       1,29045,      0            , "t=1.1m",_
 4627, 93, 227, 0,2.5,      -1,32560,      0            , "t=510ms",_
 4628, 94, 227, 0,2.5,       1,36770,      0            , "t=20#ms",_
 4629, 85, 228, 0,-77.777,   0,41680,      0            , "t=5#s",_
 4630, 86, 228, 0,0,         1,35243,      0            , "t=65s",_
 4631, 87, 228, 0,2,        -1,33384,      0            , "t=38s",_
 4632, 88, 228, 0,0,         1,28940.3,    0            , "t=5.75y",_
 4633, 89, 228, 0,3,         1,28894.7,    0            , "t=6.15h",_
 4634, 90, 228, 0,0,         1,26771.0,    0            , "t=1.9124y",_
 4635, 91, 228, 0,3,         1,28924,      0            , "t=22h",_
 4636, 92, 228, 0,0,         1,29222,      0            , "t=9.1m",_
 4637, 93, 228, 0,-77.777,   0,33600,      0            , "t=61.4s",_
 4638, 94, 228, 0,0,         1,36087,      0            , "t=2.1s",_
 4639, 85, 229, 0,0.5,       1,44820,      0            , "t=5#s",_
 4640, 86, 229, 0,2.5,       1,39362,      0            , "t=11.9s",_
 4641, 87, 229, 0,0.5,       1,35668,      0            , "t=50.2s",_
 4642, 88, 229, 0,2.5,       1,32562,      0            , "t=4.0m",_
 4643, 89, 229, 0,1.5,       1,30690,      0            , "t=62.7m",_
 4644, 90, 229, 0,2.5,       1,29585.6,    0            , "t=7.920ky",_
 4645, 90, 229, 1,1.5,       1,29585.6,    7.6e-06      , "t=&lt;1s",_
 4646, 91, 229, 0,2.5,       1,29897,      0            , "t=1.50d",_
 4647, 91, 229, 1,1.5,      -1,29909,      0.0122       , "t=420ns",_
 4648, 92, 229, 0,1.5,       1,31211,      0            , "t=57.8m",_
 4649, 93, 229, 0,2.5,       1,33780,      0            , "t=4.00m",_
 4650, 94, 229, 0,1.5,       1,37400,      0            , "t=91s",_
 4651, 95, 229, 0,2.5,      -1,42150,      0            , "t=1.8s",_
 4652, 86, 230, 0,0,         1,42050,      0            , "t=10#s",_
 4653, 87, 230, 0,-77.777,   0,39487,      0            , "t=19.1s",_
 4654, 88, 230, 0,0,         1,34516,      0            , "t=93m",_
 4655, 89, 230, 0,1,         1,33838,      0            , "t=122s",_
 4656, 90, 230, 0,0,         1,30862.6,    0            , "t=75.4ky",_
 4657, 91, 230, 0,2,        -1,32174,      0            , "t=17.4d",_
 4658, 92, 230, 0,0,         1,31615,      0            , "t=20.23d",_
 4659, 93, 230, 0,-77.777,   0,35240,      0            , "t=4.6m",_
 4660, 94, 230, 0,0,         1,36934,      0            , "t=1.70m",_
 4661, 95, 230, 0,-77.777,   0,42930,      0            , "t=40s",_
 4662, 86, 231, 0,0.5,       1,46450,      0            , "t=300#ms",_
 4663, 87, 231, 0,0.5,       1,42081,      0            , "t=17.6s",_
 4664, 88, 231, 0,2.5,       1,38216,      0            , "t=104s",_
 4665, 88, 231, 1,0.5,       1,38282,      0.06621      , "t=~53us",_
 4666, 89, 231, 0,0.5,       1,35763,      0            , "t=7.5m",_
 4667, 90, 231, 0,2.5,       1,33815.9,    0            , "t=25.52h",_
 4668, 91, 231, 0,1.5,      -1,33424.4,    0            , "t=32.76ky",_
 4669, 92, 231, 0,2.5,       1,33806.0,    0            , "t=4.2d",_
 4670, 93, 231, 0,2.5,       1,35620,      0            , "t=48.8m",_
 4671, 94, 231, 0,1.5,       1,38309,      0            , "t=8.6m",_
 4672, 95, 231, 0,2.5,      -1,42410,      0            , "t=1#m",_
 4673, 96, 231, 0,1.5,       1,47270,      0            , "t=20#s",_
 4674, 87, 232, 0,5,         0,46073,      0            , "t=5.5s",_
 4675, 88, 232, 0,0,         1,40497,      0            , "t=4.0m",_
 4676, 89, 232, 0,1,         1,39154,      0            , "t=1.98m",_
 4677, 90, 232, 0,0,         1,35446.8,    0            , "t=14.0Gy",_
 4678, 91, 232, 0,2,        -1,35947,      0            , "t=1.32d",_
 4679, 92, 232, 0,0,         1,34609.5,    0            , "t=68.9y",_
 4680, 93, 232, 0,4,         1,37360,      0            , "t=14.7m",_
 4681, 94, 232, 0,0,         1,38363,      0            , "t=33.7m",_
 4682, 95, 232, 0,1,        -1,43340,      0            , "t=1.31m",_
 4683, 96, 232, 0,0,         1,46310,      0            , "t=10#s",_
 4684, 87, 233, 0,0.5,       1,48920,      0            , "t=900ms",_
 4685, 88, 233, 0,0.5,       1,44334,      0            , "t=30s",_
 4686, 89, 233, 0,0.5,       1,41308,      0            , "t=145s",_
 4687, 90, 233, 0,0.5,       1,38731.7,    0            , "t=21.83m",_
 4688, 91, 233, 0,1.5,      -1,37489.5,    0            , "t=26.975d",_
 4689, 92, 233, 0,2.5,       1,36919.2,    0            , "t=159.2ky",_
 4690, 93, 233, 0,2.5,       1,37950,      0            , "t=36.2m",_
 4691, 94, 233, 0,2.5,       1,40050,      0            , "t=20.9m",_
 4692, 95, 233, 0,2.5,      -1,43260,      0            , "t=3.2m",_
 4693, 96, 233, 0,1.5,       1,47290,      0            , "t=27s",_
 4694, 97, 233, 0,-77.777,   0,52860,      0            , "t=40s",_
 4695, 88, 234, 0,0,         1,46931,      0            , "t=30s",_
 4696, 89, 234, 0,1,         1,44841,      0            , "t=45s",_
 4697, 90, 234, 0,0,         1,40613.0,    0            , "t=24.10d",_
 4698, 91, 234, 0,4,         1,40339,      0            , "t=6.70h",_
 4699, 91, 234, 1,0,        -1,40417.9,    0.079        , "t=1.159m",_
 4700, 92, 234, 0,0,         1,38145.0,    0            , "t=245.5ky",_
 4701, 92, 234, 1,6,        -1,39566.3,    1.421257     , "t=33.5us",_
 4702, 93, 234, 0,0,         1,39955,      0            , "t=4.4d",_
 4703, 94, 234, 0,0,         1,40350,      0            , "t=8.8h",_
 4704, 95, 234, 0,-77.777,   0,44460,      0            , "t=2.32m",_
 4705, 96, 234, 0,0,         1,46725,      0            , "t=52s",_
 4706, 97, 234, 0,-77.777,   0,53460,      0            , "t=20s",_
 4707, 88, 235, 0,2.5,       1,51130,      0            , "t=3#s",_
 4708, 89, 235, 0,0.5,       1,47357,      0            , "t=62s",_
 4709, 90, 235, 0,0.5,       1,44018,      0            , "t=7.2m",_
 4710, 91, 235, 0,1.5,      -1,42289,      0            , "t=24.4m",_
 4711, 92, 235, 0,3.5,      -1,40918.8,    0            , "t=704My",_
 4712, 92, 235, 1,0.5,       1,40918.9,    7.6e-05      , "t=25.7m",_
 4713, 92, 235, 2,-77.777,   0,43420,      2.5          , "t=3.6ms",_
 4714, 93, 235, 0,2.5,       1,41043.1,    0            , "t=396.1d",_
 4715, 94, 235, 0,2.5,       1,42182,      0            , "t=25.3m",_
 4716, 95, 235, 0,2.5,      -1,44630,      0            , "t=10.3m",_
 4717, 96, 235, 0,2.5,       1,48030,      0            , "t=5#m",_
 4718, 97, 235, 0,-77.777,   0,52700,      0            , "t=1#m",_
 4719, 89, 236, 0,-77.777,   0,51220,      0            , "t=4.5m",_
 4720, 90, 236, 0,0,         1,46255,      0            , "t=37.3m",_
 4721, 91, 236, 0,1,        -1,45334,      0            , "t=9.1m",_
 4722, 92, 236, 0,0,         1,42444.6,    0            , "t=23.42My",_
 4723, 92, 236, 1,0,         1,45195,      2.75         , "t=120ns",_
 4724, 93, 236, 0,6,        -1,43380,      0            , "t=153ky",_
 4725, 93, 236, 1,1,         0,43438,      0.06         , "t=22.5h",_
 4726, 94, 236, 0,0,         1,42901.6,    0            , "t=2.858y",_
 4727, 94, 236, 1,5,        -1,44087.0,    1.18545      , "t=1.2us",_
 4728, 95, 236, 0,5,        -1,46040,      0            , "t=3.6m",_
 4729, 95, 236, 1,1,        -1,46090,      0.05         , "t=2.9m",_
 4730, 96, 236, 0,0,         1,47855,      0            , "t=6.8m",_
 4731, 97, 236, 0,-77.777,   0,53540,      0            , "t=2#m",_
 4732, 89, 237, 0,0.5,       1,54020,      0            , "t=4#m",_
 4733, 90, 237, 0,2.5,       1,49955,      0            , "t=4.8m",_
 4734, 91, 237, 0,0.5,       1,47528,      0            , "t=8.7m",_
 4735, 92, 237, 0,0.5,       1,45390.2,    0            , "t=6.752d",_
 4736, 92, 237, 1,3.5,      -1,45664.2,    0.274        , "t=155ns",_
 4737, 93, 237, 0,2.5,       1,44871.7,    0            , "t=2.144My",_
 4738, 93, 237, 1,5.5,       0,45816.9,    0.9452       , "t=710ns",_
 4739, 94, 237, 0,3.5,      -1,45091.7,    0            , "t=45.64d",_
 4740, 94, 237, 1,0.5,       1,45237.2,    0.145543     , "t=180ms",_
 4741, 94, 237, 2,-77.777,   0,47990,      2.9          , "t=1.1us",_
 4742, 95, 237, 0,2.5,      -1,46570,      0            , "t=73.6m",_
 4743, 96, 237, 0,2.5,       1,49250,      0            , "t=20#m",_
 4744, 97, 237, 0,1.5,      -1,53190,      0            , "t=2#m",_
 4745, 98, 237, 0,2.5,       1,57940,      0            , "t=0.8s",_
 4746, 90, 238, 0,0,         1,52530,      0            , "t=9.4m",_
 4747, 91, 238, 0,3,        -1,50894,      0            , "t=2.28m",_
 4748, 92, 238, 0,0,         1,47307.8,    0            , "t=4.468Gy",_
 4749, 92, 238, 1,0,         1,49865.7,    2.5579       , "t=280ns",_
 4750, 93, 238, 0,2,         1,47454.7,    0            , "t=2.099d",_
 4751, 93, 238, 1,-77.777,   0,49760,      2.3          , "t=112ns",_
 4752, 94, 238, 0,0,         1,46163.2,    0            , "t=87.7y",_
 4753, 95, 238, 0,1,         1,48420,      0            , "t=98m",_
 4754, 95, 238, 1,-77.777,   0,50920,      2.5          , "t=35us",_
 4755, 96, 238, 0,0,         1,49445,      0            , "t=2.2h",_
 4756, 97, 238, 0,-77.777,   0,54220,      0            , "t=2.40m",_
 4757, 98, 238, 0,0,         1,57280,      0            , "t=21.1ms",_
 4758, 90, 239, 0,3.5,       1,56450,      0            , "t=2#m",_
 4759, 91, 239, 0,1.5,      -1,53340,      0            , "t=1.8h",_
 4760, 92, 239, 0,2.5,       1,50572.7,    0            , "t=23.45m",_
 4761, 92, 239, 1,2.5,       1,50593,      0.02         , "t=&gt;250ns",_
 4762, 92, 239, 2,0.5,       1,50706.5,    0.1337991    , "t=780ns",_
 4763, 93, 239, 0,2.5,       1,49311.1,    0            , "t=2.356d",_
 4764, 94, 239, 0,0.5,       1,48588.3,    0            , "t=24.11ky",_
 4765, 94, 239, 1,3.5,      -1,48979.9,    0.391584     , "t=193ns",_
 4766, 94, 239, 2,2.5,       1,51690,      3.1          , "t=7.5us",_
 4767, 95, 239, 0,2.5,      -1,49390.4,    0            , "t=11.9h",_
 4768, 95, 239, 1,3.5,       1,51890,      2.5          , "t=163ns",_
 4769, 96, 239, 0,3.5,      -1,51150,      0            , "t=2.5h",_
 4770, 97, 239, 0,3.5,       1,54250,      0            , "t=4#m",_
 4771, 98, 239, 0,2.5,       1,58270,      0            , "t=60s",_
 4772, 99, 239, 0,-77.777,   0,63560,      0            , "t=1#s",_
 4773, 91, 240, 0,-77.777,   0,56910,      0            , "t=2#m",_
 4774, 92, 240, 0,0,         1,52715.5,    0            , "t=14.1h",_
 4775, 93, 240, 0,5,         1,52316,      0            , "t=61.9m",_
 4776, 93, 240, 1,1,         1,52334,      0.018        , "t=7.22m",_
 4777, 94, 240, 0,0,         1,50125.4,    0            , "t=6.561ky",_
 4778, 94, 240, 1,5,        -1,51434.1,    1.30874      , "t=165ns",_
 4779, 95, 240, 0,3,        -1,51510,      0            , "t=50.8h",_
 4780, 95, 240, 1,-77.777,   0,54510,      3            , "t=940us",_
 4781, 96, 240, 0,0,         1,51724.3,    0            , "t=27d",_
 4782, 97, 240, 0,-77.777,   0,55660,      0            , "t=4.8m",_
 4783, 98, 240, 0,0,         1,57991,      0            , "t=40.3s",_
 4784, 99, 240, 0,-77.777,   0,64200,      0            , "t=1#s",_
 4785, 91, 241, 0,1.5,      -1,59640,      0            , "t=2#m",_
 4786, 92, 241, 0,3.5,       1,56200,      0            , "t=5#m",_
 4787, 93, 241, 0,2.5,       1,54260,      0            , "t=13.9m",_
 4788, 94, 241, 0,2.5,       1,52955.2,    0            , "t=14.329y",_
 4789, 94, 241, 1,0.5,       1,53116.9,    0.1616853    , "t=880ns",_
 4790, 94, 241, 2,-77.777,   0,55160,      2.2          , "t=20.5us",_
 4791, 95, 241, 0,2.5,      -1,52934.4,    0            , "t=432.6y",_
 4792, 95, 241, 1,-77.777,   0,55130,      2.2          , "t=1.2us",_
 4793, 96, 241, 0,0.5,       1,53701.8,    0            , "t=32.8d",_
 4794, 97, 241, 0,3.5,       1,56030,      0            , "t=4.6m",_
 4795, 98, 241, 0,3.5,      -1,59330,      0            , "t=2.35m",_
 4796, 99, 241, 0,1.5,      -1,63860,      0            , "t=10s",_
 4797, 100, 241, 0,2.5,      1,69130,      0            , "t=730us",_
 4798, 92, 242, 0,0,         1,58620,      0            , "t=16.8m",_
 4799, 93, 242, 0,1,         1,57420,      0            , "t=2.2m",_
 4800, 93, 242, 1,6,         1,57420,      0            , "t=5.5m",_
 4801, 94, 242, 0,0,         1,54716.9,    0            , "t=375ky",_
 4802, 95, 242, 0,1,        -1,55468.1,    0            , "t=16.02h",_
 4803, 95, 242, 1,5,        -1,55516.7,    0.0486       , "t=141y",_
 4804, 95, 242, 2,2,         1,57670,      2.2          , "t=14.0ms",_
 4805, 96, 242, 0,0,         1,54803.8,    0            , "t=162.8d",_
 4806, 96, 242, 1,-77.777,   0,57600,      2.8          , "t=180ns",_
 4807, 97, 242, 0,2,        -1,57730,      0            , "t=7.0m",_
 4808, 97, 242, 1,-77.777,   0,57930,      0.2          , "t=600ns",_
 4809, 98, 242, 0,0,         1,59387,      0            , "t=3.49m",_
 4810, 99, 242, 0,-77.777,   0,64800,      0            , "t=17.8s",_
 4811, 100, 242, 0,0,        1,68400,      0            , "t=800us",_
 4812, 92, 243, 0,4.5,      -1,62360,      0            , "t=10#m",_
 4813, 93, 243, 0,2.5,       1,59880,      0            , "t=1.85m",_
 4814, 94, 243, 0,3.5,       1,57754.6,    0            , "t=4.956h",_
 4815, 94, 243, 1,0.5,       1,58138.2,    0.38364      , "t=330ns",_
 4816, 95, 243, 0,2.5,      -1,57175.0,    0            , "t=7.364ky",_
 4817, 95, 243, 1,-77.777,   0,59480,      2.3          , "t=5.5us",_
 4818, 96, 243, 0,2.5,       1,57182.0,    0            , "t=29.1y",_
 4819, 96, 243, 1,0.5,       1,57269.4,    0.0874       , "t=1.08us",_
 4820, 97, 243, 0,1.5,      -1,58690,      0            , "t=4.6h",_
 4821, 98, 243, 0,0.5,       1,60990,      0            , "t=10.7m",_
 4822, 99, 243, 0,3.5,       1,64750,      0            , "t=21.6s",_
 4823, 100, 243, 0,3.5,     -1,69390,      0            , "t=231ms",_
 4824, 93, 244, 0,7,        -1,63200,      0            , "t=2.29m",_
 4825, 94, 244, 0,0,         1,59806.0,    0            , "t=80.0My",_
 4826, 94, 244, 1,8,        -1,61022,      1.216        , "t=1.75s",_
 4827, 95, 244, 0,6,        -1,59879.2,    0            , "t=10.1h",_
 4828, 95, 244, 1,1,         1,59968.5,    0.0893       , "t=26m",_
 4829, 95, 244, 2,-77.777,   0,60080,      0.2          , "t=900us",_
 4830, 96, 244, 0,0,         1,58451.9,    0            , "t=18.10y",_
 4831, 96, 244, 1,6,         1,59492.1,    1.040188     , "t=34ms",_
 4832, 96, 244, 2,-77.777,   0,59550,      1.1          , "t=&gt;500ns",_
 4833, 97, 244, 0,4,        -1,60714,      0            , "t=5.02h",_
 4834, 97, 244, 1,-77.777,   0,61210,      0.5          , "t=820ns",_
 4835, 98, 244, 0,0,         1,61478.2,    0            , "t=19.4m",_
 4836, 99, 244, 0,-77.777,   0,66030,      0            , "t=37s",_
 4837, 100, 244, 0,0,        1,68970,      0            , "t=3.12ms",_
 4838, 93, 245, 0,2.5,       1,65890,      0            , "t=2#m",_
 4839, 94, 245, 0,4.5,      -1,63178,      0            , "t=10.5h",_
 4840, 94, 245, 1,2.5,       1,63443,      0.2645       , "t=330ns",_
 4841, 95, 245, 0,2.5,       1,61900.5,    0            , "t=2.05h",_
 4842, 95, 245, 1,-77.777,   0,64300,      2.4          , "t=640ns",_
 4843, 96, 245, 0,3.5,       1,61004.6,    0            , "t=8.25ky",_
 4844, 96, 245, 1,0.5,       1,61360.5,    0.35592      , "t=290ns",_
 4845, 97, 245, 0,1.5,      -1,61813.8,    0            , "t=4.95d",_
 4846, 98, 245, 0,0.5,       1,63385.2,    0            , "t=45.0m",_
 4847, 99, 245, 0,1.5,      -1,66370,      0            , "t=1.1m",_
 4848, 100, 245, 0,0.5,      1,70190,      0            , "t=4.2s",_
 4849, 101, 245, 0,3.5,     -1,75270,      0            , "t=400ms",_
 4850, 101, 245, 1,0.5,     -1,75370,      0.1          , "t=900us",_
 4851, 94, 246, 0,0,         1,65395,      0            , "t=10.84d",_
 4852, 95, 246, 0,7,        -1,64994,      0            , "t=39m",_
 4853, 95, 246, 1,2,        -1,65024,      0.03         , "t=25.0m",_
 4854, 95, 246, 2,-77.777,   0,66990,      2            , "t=73us",_
 4855, 96, 246, 0,0,         1,62617.0,    0            , "t=4.706ky",_
 4856, 96, 246, 1,8,        -1,63796.7,    1.17966      , "t=1.12s",_
 4857, 97, 246, 0,2,        -1,63970,      0            , "t=1.80d",_
 4858, 98, 246, 0,0,         1,64090.3,    0            , "t=35.7h",_
 4859, 99, 246, 0,4,        -1,67900,      0            , "t=7.5m",_
 4860, 100, 246, 0,0,        1,70189,      0            , "t=1.54s",_
 4861, 101, 246, 0,-77.777,  0,76120,      0            , "t=0.92s",_
 4862, 101, 246, 1,-77.777,  0,76170,      0.06         , "t=4.4s",_
 4863, 94, 247, 0,0.5,       1,69110,      0            , "t=2.27d",_
 4864, 95, 247, 0,2.5,       0,67150,      0            , "t=23.0m",_
 4865, 96, 247, 0,4.5,      -1,65533,      0            , "t=15.6My",_
 4866, 96, 247, 1,2.5,       1,65760,      0.22738      , "t=26.3us",_
 4867, 96, 247, 2,0.5,       1,65938,      0.4049       , "t=100.6ns",_
 4868, 97, 247, 0,1.5,      -1,65490,      0            , "t=1.38ky",_
 4869, 98, 247, 0,3.5,       1,66104,      0            , "t=3.11h",_
 4870, 99, 247, 0,3.5,       1,68578,      0            , "t=4.55m",_
 4871, 100, 247, 0,3.5,      1,71670,      0            , "t=31s",_
 4872, 100, 247, 1,0.5,      1,71720,      0.049        , "t=5.1s",_
 4873, 101, 247, 0,3.5,     -1,75940,      0            , "t=1.2s",_
 4874, 101, 247, 1,0.5,     -1,76200,      0.26         , "t=250ms",_
 4875, 95, 248, 0,-77.777,   0,70560,      0            , "t=3#m",_
 4876, 96, 248, 0,0,         1,67392.8,    0            , "t=348ky",_
 4877, 96, 248, 1,8,        -1,68850.9,    1.4581       , "t=146us",_
 4878, 97, 248, 0,6,         1,68080,      0            , "t=&gt;9y",_
 4879, 97, 248, 1,1,        -1,68108,      0.03         , "t=23.7h",_
 4880, 98, 248, 0,0,         1,67238,      0            , "t=333.5d",_
 4881, 99, 248, 0,2,        -1,70300,      0            , "t=24m",_
 4882, 100, 248, 0,0,        1,71898,      0            , "t=34.5s",_
 4883, 100, 248, 1,-77.777,  0,73100,      1.2          , "t=10.1ms",_
 4884, 101, 248, 0,-77.777,  0,77150,      0            , "t=7s",_
 4885, 102, 248, 0,0,        1,80620,      0            , "t=?",_
 4886, 95, 249, 0,-77.777,   0,73100,      0            , "t=1#m",_
 4887, 96, 249, 0,0.5,       1,70750.7,    0            , "t=64.15m",_
 4888, 96, 249, 1,3.5,       1,70799.5,    0.04876      , "t=23us",_
 4889, 97, 249, 0,3.5,       1,69846.4,    0            , "t=327.2d",_
 4890, 97, 249, 1,1.5,      -1,69855.2,    0.008777     , "t=300us",_
 4891, 98, 249, 0,4.5,      -1,69722.8,    0            , "t=351y",_
 4892, 98, 249, 1,2.5,       1,69867.8,    0.14498      , "t=45us",_
 4893, 99, 249, 0,3.5,       1,71180,      0            , "t=102.2m",_
 4894, 100, 249, 0,3.5,      1,73519,      0            , "t=1.6m",_
 4895, 101, 249, 0,3.5,     -1,77230,      0            , "t=23.4s",_
 4896, 101, 249, 1,0.5,     -1,77330,      0.1          , "t=1.9s",_
 4897, 102, 249, 0,2.5,      1,81780,      0            , "t=57us",_
 4898, 96, 250, 0,0,         1,72990,      0            , "t=8300#y",_
 4899, 97, 250, 0,2,        -1,72950,      0            , "t=3.212h",_
 4900, 97, 250, 1,4,         1,72986,      0.03559      , "t=29us",_
 4901, 97, 250, 2,7,         1,73034,      0.0841       , "t=213us",_
 4902, 98, 250, 0,0,         1,71170.4,    0            , "t=13.08y",_
 4903, 99, 250, 0,6,         1,73230,      0            , "t=8.6h",_
 4904, 99, 250, 1,1,        -1,73430,      0.2          , "t=2.22h",_
 4905, 100, 250, 0,0,        1,74072,      0            , "t=30.4m",_
 4906, 100, 250, 1,8,       -1,75271,      1.1992       , "t=1.92s",_
 4907, 101, 250, 0,-77.777,  0,78630,      0            , "t=52s",_
 4908, 102, 250, 0,0,        1,81560,      0            , "t=5.0us",_
 4909, 102, 250, 1,6,        1,82610,      1.05         , "t=51us",_
 4910, 96, 251, 0,0.5,       1,76648,      0            , "t=16.8m",_
 4911, 97, 251, 0,1.5,      -1,75228,      0            , "t=55.6m",_
 4912, 97, 251, 1,3.5,       1,75264,      0.0355       , "t=58us",_
 4913, 98, 251, 0,0.5,       1,74135,      0            , "t=900y",_
 4914, 98, 251, 1,5.5,      -1,74505,      0.37047      , "t=1.3us",_
 4915, 99, 251, 0,1.5,      -1,74512,      0            , "t=33h",_
 4916, 100, 251, 0,4.5,     -1,75954,      0            , "t=5.30h",_
 4917, 100, 251, 1,2.5,      1,76154,      0.2          , "t=21.1us",_
 4918, 101, 251, 0,3.5,     -1,78967,      0            , "t=4.21m",_
 4919, 102, 251, 0,3.5,      1,82850,      0            , "t=800ms",_
 4920, 102, 251, 1,0.5,      1,82960,      0.106        , "t=1.02s",_
 4921, 102, 251, 2,-77.777,  0,84600,      1.75         , "t=~2us",_
 4922, 103, 251, 0,-77.777,  0,87730,      0            , "t=150#us",_
 4923, 96, 252, 0,0,         1,79060,      0            , "t=1#m",_
 4924, 97, 252, 0,-77.777,   0,78540,      0            , "t=1.8m",_
 4925, 98, 252, 0,0,         1,76034.6,    0            , "t=2.645y",_
 4926, 99, 252, 0,4,         1,77290,      0            , "t=471.7d",_
 4927, 100, 252, 0,0,        1,76816,      0            , "t=25.39h",_
 4928, 101, 252, 0,-77.777,  0,80510,      0            , "t=2.3m",_
 4929, 102, 252, 0,0,        1,82871,      0            , "t=2.45s",_
 4930, 102, 252, 1,8,       -1,84126,      1.2545       , "t=109ms",_
 4931, 103, 252, 0,-77.777,  0,88740,      0            , "t=369ms",_
 4932, 97, 253, 0,-77.777,   0,80930,      0            , "t=10#m",_
 4933, 98, 253, 0,3.5,       1,79302,      0            , "t=17.81d",_
 4934, 99, 253, 0,3.5,       1,79010.5,    0            , "t=20.47d",_
 4935, 100, 253, 0,0.5,      1,79345.7,    0            , "t=3.00d",_
 4936, 100, 253, 1,5.5,     -1,79697,      0.351        , "t=560ns",_
 4937, 101, 253, 0,3.5,     -1,81170,      0            , "t=12m",_
 4938, 102, 253, 0,4.5,     -1,84359,      0            , "t=1.56m",_
 4939, 102, 253, 1,2.5,      1,84526,      0.16734      , "t=30.3us",_
 4940, 102, 253, 2,12.5,     1,85560,      1.2          , "t=706us",_
 4941, 103, 253, 0,3.5,     -1,88580,      0            , "t=632ms",_
 4942, 103, 253, 1,0.5,     -1,88610,      0.03         , "t=1.32s",_
 4943, 104, 253, 0,3.5,      1,93560,      0            , "t=13ms",_
 4944, 104, 253, 1,0.5,     -1,93760,      0.2          , "t=52us",_
 4945, 97, 254, 0,-77.777,   0,84390,      0            , "t=1#m",_
 4946, 98, 254, 0,0,         1,81341,      0            , "t=60.5d",_
 4947, 99, 254, 0,7,         1,81991,      0            , "t=275.7d",_
 4948, 99, 254, 1,2,         1,82075,      0.0842       , "t=39.3h",_
 4949, 100, 254, 0,0,        1,80902.8,    0            , "t=3.240h",_
 4950, 101, 254, 0,0,       -1,83450,      0            , "t=10m",_
 4951, 101, 254, 1,3,       -1,83500,      0.05         , "t=28m",_
 4952, 102, 254, 0,0,        1,84723,      0            , "t=51.2s",_
 4953, 102, 254, 1,8,       -1,86018,      1.295        , "t=264.9ms",_
 4954, 102, 254, 2,16,       1,87940,      3.22         , "t=183.8us",_
 4955, 103, 254, 0,-77.777,  0,89870,      0            , "t=17.1s",_
 4956, 104, 254, 0,0,        1,93200,      0            , "t=23.2us",_
 4957, 104, 254, 1,8,       -1,94500,      1.3          , "t=4.7us",_
 4958, 104, 254, 2,16,       1,95200,      2            , "t=247us",_
 4959, 98, 255, 0,3.5,       1,84810,      0            , "t=85m",_
 4960, 99, 255, 0,3.5,       1,84089,      0            , "t=39.8d",_
 4961, 100, 255, 0,3.5,      1,83800,      0            , "t=20.07h",_
 4962, 101, 255, 0,3.5,     -1,84843,      0            , "t=27m",_
 4963, 102, 255, 0,0.5,      1,86807,      0            , "t=3.52m",_
 4964, 102, 255, 1,5.5,     -1,87020,      0.21         , "t=1#s",_
 4965, 103, 255, 0,0.5,     -1,89947,      0            , "t=31.1s",_
 4966, 103, 255, 1,3.5,     -1,89988,      0.041        , "t=2.54s",_
 4967, 103, 255, 2,7.5,      1,90741,      0.794        , "t=&lt;1us",_
 4968, 104, 255, 0,4.5,     -1,94330,      0            , "t=1.66s",_
 4969, 104, 255, 1,2.5,      1,94480,      0.15         , "t=50us",_
 4970, 105, 255, 0,-77.777,  0,99590,      0            , "t=1.7s",_
 4971, 98, 256, 0,0,         1,87040,      0            , "t=12.3m",_
 4972, 99, 256, 0,1,         1,87190,      0            , "t=25.4m",_
 4973, 99, 256, 1,8,         1,87190,      0            , "t=7.6h",_
 4974, 100, 256, 0,0,        1,85487,      0            , "t=157.6m",_
 4975, 101, 256, 0,7,       -1,87460,      0            , "t=30#m",_
 4976, 101, 256, 1,1,       -1,87620,      0.16         , "t=77m",_
 4977, 102, 256, 0,0,        1,87822,      0            , "t=2.91s",_
 4978, 103, 256, 0,-77.777,  0,91750,      0            , "t=27s",_
 4979, 104, 256, 0,0,        1,94222,      0            , "t=6.67ms",_
 4980, 104, 256, 1,5,       -1,95340,      1.12         , "t=25us",_
 4981, 104, 256, 2,8,       -1,95620,      1.4          , "t=17us",_
 4982, 105, 256, 0,-77.777,  0,100500,     0            , "t=1.7s",_
 4983, 99, 257, 0,3.5,       1,89400,      0            , "t=7.7d",_
 4984, 100, 257, 0,4.5,      1,88590,      0            , "t=100.5d",_
 4985, 101, 257, 0,3.5,     -1,88993.1,    0            , "t=5.52h",_
 4986, 102, 257, 0,1.5,      1,90247,      0            , "t=24.5s",_
 4987, 103, 257, 0,0.5,     -1,92670,      0            , "t=6.0s",_
 4988, 104, 257, 0,0.5,      1,95866,      0            , "t=4.82s",_
 4989, 104, 257, 1,5.5,     -1,95940,      0.073        , "t=4.3s",_
 4990, 104, 257, 2,10.5,     1,97022,      1.155        , "t=106us",_
 4991, 105, 257, 0,4.5,      1,100210,     0            , "t=2.3s",_
 4992, 105, 257, 1,0.5,     -1,100350,     0.14         , "t=670ms",_
 4993, 99, 258, 0,-77.777,   0,92700,      0            , "t=3#m",_
 4994, 100, 258, 0,0,        1,90430,      0            , "t=370us",_
 4995, 101, 258, 0,8,       -1,91687,      0            , "t=51.5d",_
 4996, 101, 258, 1,1,       -1,91690,      0            , "t=57.0m",_
 4997, 102, 258, 0,0,        1,91480,      0            , "t=1.2ms",_
 4998, 103, 258, 0,-77.777,  0,94780,      0            , "t=3.6s",_
 4999, 104, 258, 0,0,        1,96340,      0            , "t=13.8ms",_
 5000, 105, 258, 0,-77.777,  0,101800,     0            , "t=4.5s",_
 5001, 105, 258, 1,-77.777,  0,101860,     0.06         , "t=1.9s",_
 5002, 106, 258, 0,0,        1,105240,     0            , "t=2.7ms",_
 5003, 100, 259, 0,1.5,      1,93700,      0            , "t=1.5s",_
 5004, 101, 259, 0,3.5,     -1,93620,      0            , "t=1.60h",_
 5005, 102, 259, 0,4.5,      1,94079,      0            , "t=58m",_
 5006, 103, 259, 0,0.5,     -1,95850,      0            , "t=6.2s",_
 5007, 104, 259, 0,3.5,      1,98360,      0            , "t=2.63s",_
 5008, 105, 259, 0,4.5,      1,101990,     0            , "t=510ms",_
 5009, 106, 259, 0,5.5,     -1,106520,     0            , "t=402ms",_
 5010, 106, 259, 1,0.5,      1,106610,     0.087        , "t=226ms",_
 5011, 100, 260, 0,0,        1,95770,      0            , "t=1#m",_
 5012, 101, 260, 0,-77.777,  0,96550,      0            , "t=27.8d",_
 5013, 102, 260, 0,0,        1,95610,      0            , "t=106ms",_
 5014, 103, 260, 0,-77.777,  0,98280,      0            , "t=3.0m",_
 5015, 104, 260, 0,0,        1,99150,      0            , "t=21ms",_
 5016, 105, 260, 0,-77.777,  0,103670,     0            , "t=1.52s",_
 5017, 106, 260, 0,0,        1,106548,     0            , "t=4.95ms",_
 5018, 107, 260, 0,-77.777,  0,113320,     0            , "t=41ms",_
 5019, 101, 261, 0,3.5,     -1,98580,      0            , "t=40#m",_
 5020, 102, 261, 0,1.5,      1,98460,      0            , "t=3#h",_
 5021, 103, 261, 0,-77.777,  0,99560,      0            , "t=39m",_
 5022, 104, 261, 0,1.5,      1,101320,     0            , "t=2.2s",_
 5023, 104, 261, 1,4.5,      1,101390,     0.07         , "t=20s",_
 5024, 105, 261, 0,-77.777,  0,104310,     0            , "t=4.7s",_
 5025, 106, 261, 0,1.5,      1,108005,     0            , "t=183ms",_
 5026, 106, 261, 1,5.5,     -1,108110,     0.1          , "t=9.3us",_
 5027, 107, 261, 0,2.5,     -1,113130,     0            , "t=12.8ms",_
 5028, 101, 262, 0,-77.777,  0,101630,     0            , "t=3#m",_
 5029, 102, 262, 0,0,        1,100100,     0            , "t=~5ms",_
 5030, 103, 262, 0,-77.777,  0,102100,     0            , "t=~4h",_
 5031, 104, 262, 0,0,        1,102390,     0            , "t=250ms",_
 5032, 104, 262, 1,0,        0,103390,     1            , "t=47ms",_
 5033, 105, 262, 0,-77.777,  0,106250,     0            , "t=34s",_
 5034, 106, 262, 0,0,        1,108370,     0            , "t=10.9ms",_
 5035, 107, 262, 0,-77.777,  0,114540,     0            , "t=84ms",_
 5036, 107, 262, 1,-77.777,  0,114760,     0.21         , "t=9.5ms",_
 5037, 102, 263, 0,-77.777,  0,103130,     0            , "t=20#m",_
 5038, 103, 263, 0,-77.777,  0,103730,     0            , "t=5#h",_
 5039, 104, 263, 0,1.5,      1,104760,     0            , "t=11m",_
 5040, 105, 263, 0,-77.777,  0,107110,     0            , "t=29s",_
 5041, 106, 263, 0,3.5,      1,110190,     0            , "t=940ms",_
 5042, 106, 263, 1,1.5,      1,110240,     0.051        , "t=420ms",_
 5043, 107, 263, 0,-77.777,  0,114500,     0            , "t=200#ms",_
 5044, 108, 263, 0,1.5,      1,119680,     0            , "t=760us",_
 5045, 108, 263, 1,5.5,     -1,120000,     0.32         , "t=760us",_
 5046, 102, 264, 0,0,        1,105010,     0            , "t=1#m",_
 5047, 103, 264, 0,-77.777,  0,106380,     0            , "t=10#h",_
 5048, 104, 264, 0,0,        1,106080,     0            , "t=1#h",_
 5049, 105, 264, 0,-77.777,  0,109360,     0            , "t=3#m",_
 5050, 106, 264, 0,0,        1,110780,     0            , "t=47ms",_
 5051, 107, 264, 0,-77.777,  0,116060,     0            , "t=1.07s",_
 5052, 108, 264, 0,0,        1,119563,     0            , "t=540us",_
 5053, 103, 265, 0,-77.777,  0,108230,     0            , "t=10#h",_
 5054, 104, 265, 0,1.5,      1,108690,     0            , "t=1.6m",_
 5055, 105, 265, 0,-77.777,  0,110480,     0            , "t=15#m",_
 5056, 106, 265, 0,4.5,      1,112790,     0            , "t=9.2s",_
 5057, 106, 265, 1,1.5,      1,112860,     0.06         , "t=16.4s",_
 5058, 107, 265, 0,-77.777,  0,116420,     0            , "t=1.19s",_
 5059, 108, 265, 0,1.5,      1,120900,     0            , "t=1.96ms",_
 5060, 108, 265, 1,4.5,      1,121130,     0.229        , "t=360us",_
 5061, 109, 265, 0,-77.777,  0,126680,     0            , "t=2#ms",_
 5062, 103, 266, 0,-77.777,  0,111620,     0            , "t=21h",_
 5063, 104, 266, 0,0,        1,110080,     0            , "t=4#h",_
 5064, 105, 266, 0,-77.777,  0,112740,     0            , "t=80m",_
 5065, 106, 266, 0,0,        1,113620,     0            , "t=390ms",_
 5066, 107, 266, 0,-77.777,  0,118100,     0            , "t=2.5s",_
 5067, 108, 266, 0,0,        1,121140,     0            , "t=3.02ms",_
 5068, 108, 266, 1,9,       -1,122240,     1.1          , "t=280ms",_
 5069, 109, 266, 0,-77.777,  0,127960,     0            , "t=1.2ms",_
 5070, 109, 266, 1,-77.777,  0,129100,     1.14         , "t=6ms",_
 5071, 104, 267, 0,-77.777,  0,113440,     0            , "t=2.5h",_
 5072, 105, 267, 0,-77.777,  0,114070,     0            , "t=100m",_
 5073, 106, 267, 0,-77.777,  0,115810,     0            , "t=1.8m",_
 5074, 107, 267, 0,-77.777,  0,118770,     0            , "t=22s",_
 5075, 108, 267, 0,2.5,      1,122650,     0            , "t=55ms",_
 5076, 108, 267, 1,-77.777,  0,122690,     0.039        , "t=990us",_
 5077, 109, 267, 0,-77.777,  0,127790,     0            , "t=10#ms",_
 5078, 110, 267, 0,1.5,      1,133880,     0            , "t=10us",_
 5079, 104, 268, 0,0,        1,115480,     0            , "t=1#h",_
 5080, 105, 268, 0,-77.777,  0,117060,     0            , "t=29h",_
 5081, 106, 268, 0,0,        1,116800,     0            , "t=2#m",_
 5082, 107, 268, 0,-77.777,  0,120810,     0            , "t=25#s",_
 5083, 108, 268, 0,0,        1,122830,     0            , "t=1.42s",_
 5084, 109, 268, 0,5,        1,129150,     0            , "t=27ms",_
 5085, 110, 268, 0,0,        1,133650,     0            , "t=100#us",_
 5086, 105, 269, 0,-77.777,  0,119150,     0            , "t=3#h",_
 5087, 106, 269, 0,-77.777,  0,119760,     0            , "t=5m",_
 5088, 107, 269, 0,-77.777,  0,121480,     0            , "t=1#m",_
 5089, 108, 269, 0,4.5,      1,124560,     0            , "t=16s",_
 5090, 109, 269, 0,-77.777,  0,129370,     0            , "t=100#ms",_
 5091, 110, 269, 0,4.5,      1,134830,     0            , "t=230us",_
 5092, 105, 270, 0,-77.777,  0,122310,     0            , "t=2.0h",_
 5093, 106, 270, 0,0,        1,121490,     0            , "t=3#m",_
 5094, 107, 270, 0,-77.777,  0,124230,     0            , "t=3.8m",_
 5095, 108, 270, 0,0,        1,125110,     0            , "t=9s",_
 5096, 109, 270, 0,-77.777,  0,130710,     0            , "t=6.3ms",_
 5097, 110, 270, 0,0,        1,134680,     0            , "t=205us",_
 5098, 110, 270, 1,10,      -1,136070,     1.39         , "t=10ms",_
 5099, 106, 271, 0,-77.777,  0,124760,     0            , "t=3.1m",_
 5100, 107, 271, 0,-77.777,  0,125920,     0            , "t=10m",_
 5101, 108, 271, 0,-77.777,  0,127740,     0            , "t=10#s",_
 5102, 109, 271, 0,-77.777,  0,131100,     0            , "t=400#ms",_
 5103, 110, 271, 0,6.5,     -1,135950,     0            , "t=90ms",_
 5104, 110, 271, 1,4.5,      1,136020,     0.068        , "t=1.7ms",_
 5105, 106, 272, 0,0,        1,126580,     0            , "t=4#m",_
 5106, 107, 272, 0,-77.777,  0,128790,     0            , "t=11.3s",_
 5107, 108, 272, 0,0,        1,129010,     0            , "t=10#s",_
 5108, 109, 272, 0,-77.777,  0,133580,     0            , "t=400#ms",_
 5109, 110, 272, 0,0,        1,136020,     0            , "t=200#ms",_
 5110, 111, 272, 0,5,        1,142770,     0            , "t=4.5ms",_
 5111, 106, 273, 0,-77.777,  0,130020,     0            , "t=5#m",_
 5112, 107, 273, 0,-77.777,  0,130630,     0            , "t=1#m",_
 5113, 108, 273, 0,1.5,      1,131890,     0            , "t=1060ms",_
 5114, 109, 273, 0,-77.777,  0,134710,     0            , "t=800#ms",_
 5115, 110, 273, 0,6.5,     -1,138360,     0            , "t=240us",_
 5116, 110, 273, 1,1.5,      1,138560,     0.198        , "t=120ms",_
 5117, 111, 273, 0,-77.777,  0,142700,     0            , "t=2#ms",_
 5118, 107, 274, 0,-77.777,  0,133680,     0            , "t=60s",_
 5119, 108, 274, 0,0,        1,133490,     0            , "t=500#ms",_
 5120, 109, 274, 0,-77.777,  0,137250,     0            , "t=850ms",_
 5121, 110, 274, 0,0,        1,139200,     0            , "t=10#ms",_
 5122, 111, 274, 0,-77.777,  0,144610,     0            , "t=29ms",_
 5123, 107, 275, 0,-77.777,  0,135690,     0            , "t=5#m",_
 5124, 108, 275, 0,-77.777,  0,136620,     0            , "t=290ms",_
 5125, 109, 275, 0,-77.777,  0,138830,     0            , "t=117ms",_
 5126, 110, 275, 0,-77.777,  0,141570,     0            , "t=10#ms",_
 5127, 111, 275, 0,-77.777,  0,145300,     0            , "t=5#ms",_
 5128, 108, 276, 0,0,        1,138290,     0            , "t=100#ms",_
 5129, 109, 276, 0,-77.777,  0,141320,     0            , "t=630ms",_
 5130, 109, 276, 1,-77.777,  0,141570,     0.25         , "t=10s",_
 5131, 110, 276, 0,0,        1,142540,     0            , "t=100#ms",_
 5132, 111, 276, 0,-77.777,  0,147490,     0            , "t=10#ms",_
 5133, 108, 277, 0,1.5,      1,141490,     0            , "t=11ms",_
 5134, 108, 277, 1,-77.777,  0,141590,     0.1          , "t=110s",_
 5135, 109, 277, 0,-77.777,  0,142970,     0            , "t=9s",_
 5136, 110, 277, 0,5.5,      1,145140,     0            , "t=6ms",_
 5137, 111, 277, 0,-77.777,  0,148340,     0            , "t=10#ms",_
 5138, 109, 278, 0,-77.777,  0,145740,     0            , "t=7s",_
 5139, 110, 278, 0,0,        1,146380,     0            , "t=270#ms",_
 5140, 111, 278, 0,-77.777,  0,150520,     0            , "t=8ms",_
 5141, 109, 279, 0,-77.777,  0,147500,     0            , "t=30#s",_
 5142, 110, 279, 0,-77.777,  0,149130,     0            , "t=210ms",_
 5143, 111, 279, 0,-77.777,  0,151780,     0            , "t=180ms",_
 5144, 110, 280, 0,0,        1,150520,     0            , "t=11s",_
 5145, 111, 280, 0,-77.777,  0,153890,     0            , "t=4.3s",_
 5146, 110, 281, 0,1.5,      1,153430,     0            , "t=14s",_
 5147, 110, 281, 1,-77.777,  0,153470,     0.04         , "t=0.9s",_
 5148, 111, 281, 0,-77.777,  0,155300,     0            , "t=24s",_
 5149, 111, 282, 0,-77.777,  0,157800,     0            , "t=1.6m",_
 5150, 111, 283, 0,-77.777,  0,159280,     0            , "t=30#s",_
 9999
