' Nuclear properties from JEFF-3.3

' Table of nuclear properties extracted from JEFF-3.3 decay table by Kilian Kern (March 2020).

Dim Shared As Integer N_MAT_MAX  ' Number of lines in table NuclideData
#DEFINE ISOSOURCE "JEFF-3.3"
' Print "Spins and energies for calculation of isomeric yields provided by "+ISOSOURCE+"."
' Print


Declare Function I_MAT_ENDF(IZ As Integer,IA As Integer) As Integer
Declare Function N_ISO_MAT(IMAT As Integer) As Integer
Declare Function R_AWR_ENDF(IZ As Integer,IA AS Integer) As Single
Declare Function ISO_for_MAT(IMAT As Integer) As Integer
Declare Function ISO_for_ZA(IZ As Integer,IA As Integer) As Integer
Declare Function NStates_for_ZA(IZ As Integer, IA As Integer) As Integer


Type NucProp
  I_Z As Integer    ' Atomic number
  I_A As Integer    ' Mass number
  I_ISO As Integer  ' Nmbr of isomeric state, 0 for ground state
  R_SPI As Single   ' Spin
  I_PAR As Integer  ' Parity
  R_AWR As Double   ' Atomic mass of nucleus in units of neutron mass
  R_EXC As Single   ' Excitation energy of isomeric state in MeV
End Type


Dim Shared As Integer N_ISO_TOT  ' Total number of nuclides with isomers
N_ISO_TOT = 0

'Print "NucProp 2 **********"

Dim As Integer I_MAT_original
Dim As Single R_Dummy_NuclideData
Dim As Integer I_NuclideData

' Determine N_MAT_MAX (number of states in table NuclideData)
N_MAT_MAX = 0
Do 
  Read I_MAT_original
  If I_MAT_original = 9999 Then
    Exit Do
  End If
  N_MAT_MAX = N_MAT_MAX + 1
  For I_NuclideData = 1 To 7
    Read R_Dummy_NuclideData
  Next I_NuclideData
Loop 

Redim Shared NucTab(N_MAT_MAX) As NucProp
Redim Shared MAT_for_ISO_prov(N_MAT_MAX) As Integer

Scope
Dim As Integer I_Mat
Restore NuclideData
Do Until I_MAT = N_MAT_MAX 
  ' Read nuclide properties into UDT NucTab
  Read I_MAT_original
  I_MAT = I_MAT + 1
  If I_MAT > N_MAT_MAX Then Print "<E> NucPropx: N_MAT_MAX too small!"
  Read NucTab(I_MAT).I_Z, _
       NucTab(I_MAT).I_A, _
       NucTab(I_MAT).I_ISO, _
       NucTab(I_MAT).R_SPI, _
       NucTab(I_MAT).I_PAR, _
       NucTab(I_MAT).R_AWR, _
       NucTab(I_MAT).R_EXC 
  If NucTab(I_MAT).I_ISO < 0 or NucTab(I_MAT).I_ISO > 9 Then
    Print "<E> Error in NucProp"
    Print "GEF stopped."
    Sleep
  End If     
  If NucTab(I_MAT).R_SPI < 0 Or NucTab(I_MAT).R_SPI > 1.E3 Then _
          NucTab(I_MAT).R_SPI = 0     
  If NucTab(I_MAT).I_ISO = 1 Then  ' At least one isomeric state is in NucTab
    ' Establish table of MAT numbers of nuclides with isomers
    N_ISO_TOT = N_ISO_TOT + 1
    MAT_for_ISO_prov(N_ISO_TOT) = I_MAT - 1
  End If
'  PRINT I_MAT,NucTab(I_MAT).I_Z,NucTab(I_MAT).I_A     
 ' If NucTab(I_MAT).I_Z = 111 Then Exit Do
Loop

If I_MAT < N_MAT_MAX Then 
  Print "<E> Nucprop: N_MAT_MAX too large, should be ";I_MAT
End If  
End Scope

Redim Shared MAT_for_ISO(N_ISO_TOT) As Integer  
   ' List of MAT numbers of nuclides with isomers
   ' MAT number refers to ground state !
   
'Scope  


   
Dim As Integer I1,J1
Dim As Single R1
For I1 = 1 To N_ISO_TOT
  MAT_for_ISO(I1) = MAT_for_ISO_prov(I1)
Next
' Erase MAT_for_ISO_prov



Type Isoprop
   I_MAT As Integer
   I_Z As Integer
   I_A As Integer
   N_STATES As Integer
   I_ISO(10) As Integer
   R_SPI(10) As Single
   I_PAR(10) As Integer
   R_EXC(10) As Single
   R_Lim(10) As Single
   R_Prob(10) As Single
End Type


Redim Shared Isotab(N_ISO_TOT) As Isoprop



Scope

Dim Inmbr As Integer

For I1 = 1 To N_ISO_TOT
  Isotab(I1).I_MAT = MAT_for_ISO(I1)
  Isotab(I1).I_Z = NucTab(MAT_for_ISO(I1)).I_Z
  Isotab(I1).I_A = NucTab(MAT_for_ISO(I1)).I_A
  Isotab(I1).N_STATES = N_ISO_MAT(Isotab(I1).I_MAT) + 1  ' Number of states
  

  ' Sorting the spin in ascending order
  Inmbr = 0 
  For R1 = 0 To 50.0 Step 0.5
    For J1 = 1 To Isotab(I1).N_STATES
      If NucTab(MAT_for_ISO(I1)+J1-1).R_SPI = R1 Then
        Inmbr = Inmbr + 1
        Isotab(I1).I_ISO(Inmbr) = NucTab(MAT_for_ISO(I1)+J1-1).I_ISO
        Isotab(I1).R_SPI(Inmbr) = NucTab(MAT_for_ISO(I1)+J1-1).R_SPI
        Isotab(I1).I_PAR(Inmbr) = NucTab(MAT_for_ISO(I1)+J1-1).I_PAR
        Isotab(I1).R_EXC(Inmbr) = NucTab(MAT_for_ISO(I1)+J1-1).R_EXC 
  '     Print Inmbr,Mat_for_ISO(I1)+J1-1,NucTab(MAT_for_ISO(I1)+J1-1).R_SPI,NucTab(MAT_for_ISO(I1)+J1-1).R_EXC
      End If
    Next
  Next

  
  Dim As Single R_J_diff,R_E_diff,R_J_1
  ' Setting the limits of the angular-momentum distribution
  For J1 = 1 To Isotab(I1).N_STATES - 1
    R_J_diff = Isotab(I1).R_SPI(J1+1) - Isotab(I1).R_SPI(J1)
    R_E_diff = Isotab(I1).R_EXC(J1+1) - Isotab(I1).R_EXC(J1)
'   R_J_1 = R_J_diff * (0.5 + 0.5 * (R_E_diff / (Abs(R_E_diff) + 0.2 ) ) )
    R_J_1 = R_J_diff * (0.5 + 0.5 * (R_E_diff/R_J_diff / (Abs(R_E_diff/R_J_diff) + 0.05 ) ) )
    Isotab(I1).R_lim(J1) = Isotab(I1).R_SPI(J1) + R_J_1
    
    If Isotab(I1).R_SPI(J1+1) = Isotab(I1).R_SPI(J1) Then
      Isotab(I1).R_lim(J1) = 1.E3
    End If
  Next 
  Isotab(I1).R_lim(Isotab(I1).N_STATES) = 1.E3 
Next  

End Scope



Function I_MAT_ENDF(IZ As Integer,IA As Integer) As Integer
  ' Returns ENDF MAT number of the g.s. of the nucleus, specified by Z and A.
  Dim As Integer I,IMAT
  Static As Integer I_message = 0
  For I = LBound(NucTab) To UBound(NucTab)
    IMAT = I
    If IZ = NucTab(IMAT).I_Z And IA = NucTab(IMAT).I_A Then Exit For
  Next
  If I > UBound(NucTab) Then
    If I_message = 0 Then
      Print "<W> NucProb.bas: Missing MAT number in NucTab for Z,A = ";IZ;",";IA
      Print "    Information on nuclear properties is not available."
      Print "    Please extend table!"
      Print "    (Note that this message is shown only once for the first case"
      Print "     encountered in one GEF session!)"
    End If  
    IMAT = 0
    I_message = 1
  End If
  I_MAT_ENDF = IMAT 
End Function

Function N_ISO_MAT(IMAT As Integer) As Integer
  ' Returns the number of isomers in a nuclide specified by MAT number
  ' (= number of states - 1 !)
  Dim As Integer I,I_first,I_last,IZ,IA
  I_first = IMAT
  IZ = NucTab(I_first).I_Z
  IA = NucTab(I_first).I_A
  I_last = I_first
  For I = I_first + 1 To N_MAT_MAX
    If NucTab(I).I_Z <> IZ Or NucTab(I).I_A <> IA Then Exit For
    I_last = I
  Next
  N_ISO_MAT = I_last - I_first
End Function

Function R_AWR_ENDF(IZ As Integer,IA AS Integer) As Single
  ' Returns ENDF AWR value
  Dim As Integer IMAT
  If IMAT > UBound(NucTab) Then IMAT = UBound(NucTab)
  IMAT = I_MAT_ENDF(IZ,IA)
  R_AWR_ENDF = NucTab(IMAT).R_AWR
End Function

Function ISO_for_MAT(IMAT As Integer) As Integer
  Dim As Integer I
  For I = 1 To N_ISO_TOT
    If IMAT = MAT_for_ISO(I) Then
      ISO_for_MAT = I
      Exit For
    End If
  Next
End Function

Function ISO_for_ZA(IZ As Integer,IA As Integer) As Integer
  ' Returns number of element in ISO array for nucleus Z,A
  Dim As Integer I
  Dim As Integer Ires = 0
  For I = 1 To N_ISO_TOT
    If Isotab(I).I_Z = IZ And Isotab(I).I_A = IA Then
      Ires = I
      Exit For
    End If    
  Next
  ISO_for_ZA = Ires
End Function

Function NStates_for_ZA(IZ As Integer, IA As Integer) As Integer
  ' Returns number of states
  Dim As Integer I
  Dim As Integer Ires = 1
  For I = 1 To N_ISO_TOT
    If Isotab(I).I_Z = IZ And Isotab(I).I_A = IA Then
      Ires = Isotab(I).N_STATES
      Exit For
    End If
  Next
  NStates_for_ZA = Ires
 End Function


NuclideData:
' Data extracted from JEFF 3.3
' MAT no., Z, A, Isomer, Spin, Parity, AWR, EXC
' The MAT no. are taken from JEFF 3.3. This causes a problem, because
' the GEFY tables cover nuclides beyond this range. 
' Therefore, the table is extended by repeating the last MAT no.
' The MAT no. in the GEFY tables are determined by counting. They differ
' from the original MAT no. of JEFF 3.3!

' The following table can be modified just by changing the values for specific states or by
' adding new lines for additional states.
' No further adjustments of GEF are necessary.

Data _
 1,    0,   1,   0, 0.5,   +1, 1.000000000000000e+00, 0,_
 2,    1,   1,   0, 0.5,   +1, 9.991673314231245e-01, 0,_
 3,    1,   2,   0, 1,     +1, 1.996799676999397e+00, 0,_
 4,    1,   3,   0, 0.5,   +1, 2.990139970852545e+00, 0,_
 5,    1,   4,   0, 2,     -1, 3.991842885431074e+00, 0,_
 6,    1,   5,   0, 0.5,   +1, 4.992055750156149e+00, 0,_
 7,    1,   6,   0, 2,     -1, 5.993026368014116e+00, 0,_
 8,    1,   7,   0, 0.5,   +1, 6.992162500533119e+00, 0,_
 9,    2,   3,   0, 0.5,   +1, 2.990120182972857e+00, 0,_
 10,   2,   4,   0, 0,     +1, 3.968218970792754e+00, 0,_
 11,   2,   5,   0, 1.5,   -1, 4.969001246489691e+00, 0,_
 12,   2,   6,   0, 0,     +1, 5.967180771928517e+00, 0,_
 13,   2,   7,   0, 1.5,   -1, 6.967616840610098e+00, 0,_
 14,   2,   8,   0, 0,     +1, 7.964919037031011e+00, 0,_
 15,   2,   9,   0, 0.5,   -1, 8.966254577862134e+00, 0,_
 16,   2,   10,  0, 0,     +1, 9.966456798814603e+00, 0,_
 17,   3,   4,   0, 2,     -1, 3.992590104837815e+00, 0,_
 18,   3,   5,   0, 1.5,   -1, 4.969477694111158e+00, 0,_
 19,   3,   6,   0, 1,     +1, 5.963450094355636e+00, 0,_
 20,   3,   7,   0, 1.5,   -1, 6.955732599221317e+00, 0,_
 21,   3,   8,   0, 2,     +1, 7.953569238083465e+00, 0,_
 22,   3,   9,   0, 1.5,   -1, 8.949245730096221e+00, 0,_
 23,   3,   10,  0,-77.777,+0, 9.949273832768927e+00, 0,_
 24,   3,   11,  0, 1.5,   -1, 1.094885269408021e+01, 0,_
 25,   3,   12,  0,-77.777,+0, 1.194907620158649e+01, 0,_
 26,   4,   5,   0, 0.5,   +1, 4.996575097392783e+00, 0,_
 27,   4,   6,   0, 0,     +1, 5.968014069475419e+00, 0,_
 28,   4,   7,   0, 1.5,   -1, 6.956649930959030e+00, 0,_
 29,   4,   8,   0, 0,     +1, 7.936535688357951e+00, 0,_
 30,   4,   9,   0, 1.5,   -1, 8.934764087311883e+00, 0,_
 31,   4,   10,  0, 0,     +1, 9.927513625136291e+00, 0,_
 32,   4,   11,  0, 0.5,   +1, 1.092697972154596e+01, 0,_
 33,   4,   12,  0, 0,     +1, 1.192360504897966e+01, 0,_
 34,   4,   13,  0, 0.5,   +1, 1.292414785380157e+01, 0,_
 35,   4,   14,  0, 0,     +1, 1.392225772875598e+01, 0,_
 36,   4,   15,  0,-77.777,+0, 1.492417350787122e+01, 0,_
 37,   4,   16,  0, 0,     +1, 1.592369456306763e+01, 0,_
 38,   5,   6,   0, 2,     -1, 5.998820723412360e+00, 0,_
 39,   5,   7,   0, 1.5,   -1, 6.969323399421655e+00, 0,_
 40,   5,   8,   0, 2,     +1, 7.955672087073981e+00, 0,_
 41,   5,   9,   0, 1.5,   -1, 8.935900820613513e+00, 0,_
 42,   5,   10,  0, 3,     +1, 9.926920927808741e+00, 0,_
 43,   5,   11,  0, 1.5,   -1, 1.091472994978706e+01, 0,_
 44,   5,   12,  0, 1,     +1, 1.191114358154597e+01, 0,_
 45,   5,   13,  0, 1.5,   -1, 1.290595100199071e+01, 0,_
 46,   5,   14,  0, 2,     -1, 1.390491905887097e+01, 0,_
 47,   5,   15,  0, 1.5,   -1, 1.490196369205564e+01, 0,_
 48,   5,   16,  0, 0,     -1, 1.590205197824326e+01, 0,_
 49,   5,   17,  0, 1.5,   -1, 1.690049007518180e+01, 0,_
 50,   5,   18,  0, 4,     -1, 1.790049539625515e+01, 0,_
 51,   5,   19,  0, 1.5,   -1, 1.890039566261871e+01, 0,_
 52,   6,   8,   0, 0,     +1, 7.968595829930051e+00, 0,_
 53,   6,   9,   0, 1.5,   -1, 8.953456262190072e+00, 0,_
 54,   6,   10,  0, 0,     +1, 9.930803640430717e+00, 0,_
 55,   6,   11,  0, 1.5,   -1, 1.091683910513353e+01, 0,_
 56,   6,   12,  0, 0,     +1, 1.189691423959614e+01, 0,_
 57,   6,   13,  0, 0.5,   -1, 1.289164977512760e+01, 0,_
 58,   6,   14,  0, 0,     +1, 1.388294741772195e+01, 0,_
 59,   6,   15,  0, 0.5,   +1, 1.488165100279813e+01, 0,_
 60,   6,   16,  0, 0,     +1, 1.587712728461538e+01, 0,_
 61,   6,   17,  0, 1.5,   +1, 1.687634654979686e+01, 0,_
 62,   6,   18,  0, 0,     +1, 1.787189347945650e+01, 0,_
 63,   6,   19,  0, 0.5,   +1, 1.887127954730690e+01, 0,_
 64,   6,   20,  0, 0,     +1, 1.986810626372203e+01, 0,_
 65,   6,   21,  0, 0.5,   +1, 2.086817898577159e+01, 0,_
 66,   6,   22,  0, 0,     +1, 2.186806901285764e+01, 0,_
 67,   7,   10,  0, 2,     -1, 9.955390918733977e+00, 0,_
 68,   7,   11,  0, 0.5,   +1, 1.093137153088771e+01, 0,_
 69,   7,   12,  0, 1,     +1, 1.191536752542780e+01, 0,_
 70,   7,   13,  0, 0.5,   -1, 1.289401307115645e+01, 0,_
 71,   7,   14,  0, 1,     +1, 1.388278087681489e+01, 0,_
 72,   7,   15,  0, 0.5,   -1, 1.487125076294100e+01, 0,_
 73,   7,   16,  0, 2,     -1, 1.586860182599664e+01, 0,_
 74,   7,   17,  0, 0.5,   -1, 1.686233813651869e+01, 0,_
 75,   7,   18,  0, 1,     -1, 1.785932799135315e+01, 0,_
 76,   7,   19,  0, 0.5,   -1, 1.885365706761001e+01, 0,_
 77,   7,   20,  0,-77.777,+0, 1.985135695804576e+01, 0,_
 78,   7,   21,  0, 0.5,   -1, 2.084645479703823e+01, 0,_
 79,   7,   22,  0,-77.777,+0, 2.184481741400438e+01, 0,_
 80,   7,   23,  0, 0.5,   -1, 2.284150131391253e+01, 0,_
 81,   7,   24,  0,-77.777,+0, 2.384378560490338e+01, 0,_
 82,   7,   25,  0, 0.5,   -1, 2.484482171130860e+01, 0,_
 83,   8,   12,  0, 0,     +1, 1.193088166174262e+01, 0,_
 84,   8,   13,  0, 1.5,   -1, 1.291292602004641e+01, 0,_
 85,   8,   14,  0, 0,     +1, 1.388825613569758e+01, 0,_
 86,   8,   15,  0, 0.5,   -1, 1.487418208236496e+01, 0,_
 87,   8,   16,  0, 0,     +1, 1.585751062492031e+01, 0,_
 88,   8,   17,  0, 2.5,   +1, 1.685310105469511e+01, 0,_
 89,   8,   18,  0, 0,     +1, 1.784453819156333e+01, 0,_
 90,   8,   19,  0, 2.5,   +1, 1.884032811288071e+01, 0,_
 91,   8,   20,  0, 0,     +1, 1.983223074804537e+01, 0,_
 92,   8,   21,  0, 2.5,   +1, 2.082818051911808e+01, 0,_
 93,   8,   22,  0, 0,     +1, 2.182088957471755e+01, 0,_
 94,   8,   23,  0, 0.5,   +1, 2.281798080315825e+01, 0,_
 95,   8,   24,  0, 0,     +1, 2.381351886366833e+01, 0,_
 96,   8,   25,  0, 1.5,   +1, 2.481432488276075e+01, 0,_
 97,   8,   26,  0, 0,     +1, 2.581353802103139e+01, 0,_
 98,   8,   27,  0, 1.5,   +1, 2.681560008262130e+01, 0,_
 99,   8,   28,  0, 0,     +1, 2.781489626531898e+01, 0,_
 100,  9,   14,  0, 2,     -1, 1.391375369449697e+01, 0,_
 101,  9,   15,  0, 0.5,   +1, 1.488877515561370e+01, 0,_
 102,  9,   16,  0, 0,     -1, 1.587391954639702e+01, 0,_
 103,  9,   17,  0, 2.5,   +1, 1.685603907832766e+01, 0,_
 104,  9,   18,  0, 1,     +1, 1.784630063232251e+01, 0,_
 105,  9,   19,  0, 0.5,   +1, 1.883519775983795e+01, 0,_
 106,  9,   20,  0, 2,     +1, 1.982817181238122e+01, 0,_
 107,  9,   21,  0, 2.5,   +1, 2.081954925231831e+01, 0,_
 108,  9,   22,  0,-77.777,+0, 2.181398248705075e+01, 0,_
 109,  9,   23,  0,-77.777,+0, 2.280591553568525e+01, 0,_
 110,  9,   24,  0,-77.777,+0, 2.380185827171601e+01, 0,_
 111,  9,   25,  0, 2.5,   +1, 2.479730119954278e+01, 0,_
 112,  9,   26,  0, 1,     +1, 2.579649592634723e+01, 0,_
 113,  9,   27,  0, 2.5,   +1, 2.679514432999584e+01, 0,_
 114,  9,   28,  0,-77.777,+0, 2.779537848028330e+01, 0,_
 115,  9,   29,  0, 2.5,   +1, 2.879360880356311e+01, 0,_
 116,  9,   30,  0,-77.777,+0, 2.979349190069661e+01, 0,_
 117,  9,   31,  0, 2.5,   +1, 3.079344935354410e+01, 0,_
 118,  10,  16,  0, 0,     +1, 1.588808197117848e+01, 0,_
 119,  10,  17,  0, 0.5,   -1, 1.687152362701676e+01, 0,_
 120,  10,  18,  0, 0,     +1, 1.785103101198097e+01, 0,_
 121,  10,  19,  0, 0.5,   +1, 1.883864562450089e+01, 0,_
 122,  10,  20,  0, 0,     +1, 1.982069551803240e+01, 0,_
 123,  10,  21,  0, 1.5,   +1, 2.081349946422289e+01, 0,_
 124,  10,  22,  0, 0,     +1, 2.180246855430872e+01, 0,_
 125,  10,  23,  0, 2.5,   +1, 2.279693339121101e+01, 0,_
 126,  10,  24,  0, 0,     +1, 2.378749401181884e+01, 0,_
 127,  10,  25,  0, 1.5,   +1, 2.478307157008419e+01, 0,_
 128,  10,  26,  0, 0,     +1, 2.577715957817639e+01, 0,_
 129,  10,  27,  0, 1.5,   +1, 2.677556147577913e+01, 0,_
 130,  10,  28,  0, 0,     +1, 2.777149311694595e+01, 0,_
 131,  10,  29,  0, 1.5,   +1, 2.877045939127190e+01, 0,_
 132,  10,  30,  0, 0,     +1, 2.976706313869458e+01, 0,_
 133,  10,  31,  0, 3.5,   -1, 3.076688237021821e+01, 0,_
 134,  10,  32,  0, 0,     +1, 3.176448342505610e+01, 0,_
 135,  10,  33,  0, 3.5,   -1, 3.276546996098532e+01, 0,_
 136,  10,  34,  0, 0,     +1, 3.376416435810438e+01, 0,_
 137,  11,  18,  0, 1,     -1, 1.787201983856546e+01, 0,_
 138,  11,  19,  0, 2.5,   +1, 1.885054191316108e+01, 0,_
 139,  11,  20,  0, 2,     +1, 1.983548164727718e+01, 0,_
 140,  11,  21,  0, 1.5,   +1, 2.081727476852888e+01, 0,_
 141,  11,  22,  0, 3,     +1, 2.180549464250920e+01, 0,_
 142,  11,  23,  0, 1.5,   +1, 2.279227612799473e+01, 0,_
 143,  11,  24,  0, 4,     +1, 2.378486912226585e+01, 0,_
 144,  11,  24,  1, 1,     +1, 2.378537180147231e+01, 0.4723,_
 145,  11,  25,  0, 2.5,   +1, 2.477527827235298e+01, 0,_
 146,  11,  26,  0, 3,     +1, 2.576934544002567e+01, 0,_
 147,  11,  27,  0, 2.5,   +1, 2.676218433359012e+01, 0,_
 148,  11,  28,  0, 1,     +1, 2.775841467355697e+01, 0,_
 149,  11,  29,  0, 1.5,   +1, 2.875372845542262e+01, 0,_
 150,  11,  30,  0, 2,     +1, 2.975130537538715e+01, 0,_
 151,  11,  31,  0, 1.5,   +1, 3.074672883887082e+01, 0,_
 152,  11,  32,  0,-77.777,+0, 3.174494376060373e+01, 0,_
 153,  11,  33,  0, 1.5,   +1, 3.274182385252460e+01, 0,_
 154,  11,  34,  0, 1,     +1, 3.374164151662978e+01, 0,_
 155,  11,  35,  0, 1.5,   +1, 3.474002361975005e+01, 0,_
 156,  11,  36,  0,-77.777,+0, 3.574002370320691e+01, 0,_
 157,  11,  37,  0, 1.5,   +1, 3.673912953527675e+01, 0,_
 158,  12,  20,  0, 0,     +1, 1.984679229050574e+01, 0,_
 159,  12,  21,  0, 2.5,   +1, 2.083120512516132e+01, 0,_
 160,  12,  22,  0, 0,     +1, 2.181058378154783e+01, 0,_
 161,  12,  23,  0, 1.5,   +1, 2.279659337839345e+01, 0,_
 162,  12,  24,  0, 0,     +1, 2.377899867519554e+01, 0,_
 163,  12,  25,  0, 2.5,   +1, 2.477119663043660e+01, 0,_
 164,  12,  26,  0, 0,     +1, 2.575939002486004e+01, 0,_
 165,  12,  27,  0, 0.5,   +1, 2.675253218861382e+01, 0,_
 166,  12,  28,  0, 0,     +1, 2.774348167275189e+01, 0,_
 167,  12,  29,  0, 1.5,   +1, 2.873959125408217e+01, 0,_
 168,  12,  30,  0, 0,     +1, 2.973283035389317e+01, 0,_
 169,  12,  31,  0, 1.5,   +1, 3.073037214425277e+01, 0,_
 170,  12,  32,  0, 0,     +1, 3.172422242225619e+01, 0,_
 171,  12,  33,  0, 3.5,   -1, 3.272179564029757e+01, 0,_
 172,  12,  34,  0, 0,     +1, 3.371678239978461e+01, 0,_
 173,  12,  35,  0, 3.5,   -1, 3.471597896466231e+01, 0,_
 174,  12,  36,  0, 0,     +1, 3.571243376767576e+01, 0,_
 175,  12,  37,  0, 3.5,   -1, 3.671217833019999e+01, 0,_
 176,  12,  38,  0, 0,     +1, 3.770982751896148e+01, 0,_
 177,  12,  39,  0, 3.5,   -1, 3.870996540834160e+01, 0,_
 178,  12,  40,  0, 0,     +1, 3.970784090119717e+01, 0,_
 179,  13,  21,  0, 0.5,   +1, 2.084832601013427e+01, 0,_
 180,  13,  22,  0, 3,     +1, 2.183038158127973e+01, 0,_
 181,  13,  23,  0, 2.5,   +1, 2.280960107777331e+01, 0,_
 182,  13,  24,  0, 4,     +1, 2.379377647084027e+01, 0,_
 183,  13,  24,  1, 1,     +1, 2.379422965908103e+01, 0.4258,_
 184,  13,  25,  0, 2.5,   +1, 2.477574853060482e+01, 0,_
 185,  13,  26,  0, 5,     +1, 2.576365198731414e+01, 0,_
 186,  13,  26,  1, 0,     +1, 2.576389512099596e+01, 0.22844,_
 187,  13,  27,  0, 2.5,   +1, 2.674975404102878e+01, 0,_
 188,  13,  28,  0, 3,     +1, 2.774153204709409e+01, 0,_
 189,  13,  29,  0, 2.5,   +1, 2.873149715972838e+01, 0,_
 190,  13,  30,  0, 3,     +1, 2.972540029671318e+01, 0,_
 191,  13,  31,  0,-77.777,+0, 3.071778275425731e+01, 0,_
 192,  13,  32,  0, 1,     +1, 3.171329133917095e+01, 0,_
 193,  13,  33,  0, 2.5,   +1, 3.270747021093707e+01, 0,_
 194,  13,  34,  0, 4,     -1, 3.370473040530226e+01, 0,_
 195,  13,  35,  0, 2.5,   +1, 3.469909507910933e+01, 0,_
 196,  13,  36,  0,-77.777,+0, 3.569707584280196e+01, 0,_
 197,  13,  37,  0, 1.5,   +1, 3.669259277240953e+01, 0,_
 198,  13,  38,  0,-77.777,+0, 3.769081426718756e+01, 0,_
 199,  13,  39,  0, 1.5,   +1, 3.868694983633559e+01, 0,_
 200,  13,  40,  0,-77.777,+0, 3.968574535722568e+01, 0,_
 201,  13,  41,  0, 1.5,   +1, 4.068336010937749e+01, 0,_
 202,  13,  42,  0,-77.777,+0, 4.168187902701153e+01, 0,_
 203,  14,  22,  0, 0,     +1, 2.184649198597918e+01, 0,_
 204,  14,  23,  0, 1.5,   +1, 2.282764041741388e+01, 0,_
 205,  14,  24,  0, 0,     +1, 2.380526482521669e+01, 0,_
 206,  14,  25,  0, 2.5,   +1, 2.478931150358567e+01, 0,_
 207,  14,  26,  0, 0,     +1, 2.576904717942864e+01, 0,_
 208,  14,  27,  0, 2.5,   +1, 2.675487594020359e+01, 0,_
 209,  14,  28,  0, 0,     +1, 2.773659130618814e+01, 0,_
 210,  14,  29,  0, 0.5,   +1, 2.872757266638286e+01, 0,_
 211,  14,  30,  0, 0,     +1, 2.971628106211332e+01, 0,_
 212,  14,  31,  0, 1.5,   +1, 3.070926995494673e+01, 0,_
 213,  14,  32,  0, 0,     +1, 3.169947822861116e+01, 0,_
 214,  14,  33,  0, 1.5,   +1, 3.269468031134042e+01, 0,_
 215,  14,  34,  0, 0,     +1, 3.368668316313641e+01, 0,_
 216,  14,  35,  0, 3.5,   -1, 3.468401605458747e+01, 0,_
 217,  14,  36,  0, 0,     +1, 3.567750667895933e+01, 0,_
 218,  14,  37,  0, 3.5,   -1, 3.667515803395062e+01, 0,_
 219,  14,  38,  0, 0,     +1, 3.766912321830021e+01, 0,_
 220,  14,  39,  0, 3.5,   -1, 3.866744087980169e+01, 0,_
 221,  14,  40,  0, 0,     +1, 3.966215972474568e+01, 0,_
 222,  14,  41,  0, 3.5,   -1, 4.066068954788442e+01, 0,_
 223,  14,  42,  0, 0,     +1, 4.165672895030902e+01, 0,_
 224,  14,  43,  0, 1.5,   -1, 4.265519631464799e+01, 0,_
 225,  14,  44,  0, 0,     +1, 4.365236592392535e+01, 0,_
 226,  15,  24,  0, 1,     +1, 2.382929119772147e+01, 0,_
 227,  15,  25,  0, 0.5,   +1, 2.480624596688671e+01, 0,_
 228,  15,  26,  0, 3,     +1, 2.578832632327017e+01, 0,_
 229,  15,  27,  0, 0.5,   +1, 2.676728811079031e+01, 0,_
 230,  15,  28,  0, 3,     +1, 2.775185906237601e+01, 0,_
 231,  15,  29,  0, 0.5,   +1, 2.873283279059932e+01, 0,_
 232,  15,  30,  0, 1,     +1, 2.972078538553010e+01, 0,_
 233,  15,  31,  0, 0.5,   +1, 3.070768251461358e+01, 0,_
 234,  15,  32,  0, 1,     +1, 3.169923642779488e+01, 0,_
 235,  15,  33,  0, 0.5,   +1, 3.268848274275054e+01, 0,_
 236,  15,  34,  0, 1,     +1, 3.368179596033726e+01, 0,_
 237,  15,  35,  0, 0.5,   +1, 3.467287649691696e+01, 0,_
 238,  15,  36,  0, 4,     -1, 3.566918909809732e+01, 0,_
 239,  15,  37,  0, 0.5,   +1, 3.666193438079207e+01, 0,_
 240,  15,  38,  0,-77.777,+0, 3.765799970758420e+01, 0,_
 241,  15,  39,  0, 0.5,   +1, 3.865137495469035e+01, 0,_
 242,  15,  40,  0,-77.777,+0, 3.964774449648509e+01, 0,_
 243,  15,  41,  0, 0.5,   +1, 4.064249024332640e+01, 0,_
 244,  15,  42,  0,-77.777,+0, 4.164027452650613e+01, 0,_
 245,  15,  43,  0, 0.5,   +1, 4.263559019998115e+01, 0,_
 246,  15,  44,  0,-77.777,+0, 4.363314150192367e+01, 0,_
 247,  15,  45,  0, 0.5,   +1, 4.463003153371640e+01, 0,_
 248,  15,  46,  0,-77.777,+0, 4.562928508580472e+01, 0,_
 249,  16,  26,  0, 0,     +1, 2.580546779387039e+01, 0,_
 250,  16,  27,  0, 2.5,   +1, 2.678618000511630e+01, 0,_
 251,  16,  28,  0, 0,     +1, 2.776380176089864e+01, 0,_
 252,  16,  29,  0, 2.5,   +1, 2.874751663631230e+01, 0,_
 253,  16,  30,  0, 0,     +1, 2.972732202608990e+01, 0,_
 254,  16,  31,  0, 0.5,   +1, 3.071342774107989e+01, 0,_
 255,  16,  32,  0, 0,     +1, 3.169741573537146e+01, 0,_
 256,  16,  33,  0, 1.5,   +1, 3.268821825040445e+01, 0,_
 257,  16,  34,  0, 0,     +1, 3.367606672864756e+01, 0,_
 258,  16,  35,  0, 1.5,   +1, 3.466863154804162e+01, 0,_
 259,  16,  36,  0, 0,     +1, 3.565810621038638e+01, 0,_
 260,  16,  37,  0, 3.5,   -1, 3.665352579151037e+01, 0,_
 261,  16,  38,  0, 0,     +1, 3.764497278973079e+01, 0,_
 262,  16,  39,  0, 3.5,   -1, 3.864031874283536e+01, 0,_
 263,  16,  40,  0, 0,     +1, 3.963207397721542e+01, 0,_
 264,  16,  41,  0, 3.5,   -1, 4.062755907167188e+01, 0,_
 265,  16,  42,  0, 0,     +1, 4.162042759846688e+01, 0,_
 266,  16,  43,  0, 1.5,   -1, 4.261762946325296e+01, 0,_
 267,  16,  44,  0, 0,     +1, 4.361222261035815e+01, 0,_
 268,  16,  45,  0, 1.5,   -1, 4.460918219151151e+01, 0,_
 269,  16,  46,  0, 0,     +1, 4.560519978292665e+01, 0,_
 270,  16,  47,  0, 1.5,   -1, 4.660409147054018e+01, 0,_
 271,  16,  48,  0, 0,     +1, 4.760123926880809e+01, 0,_
 272,  16,  49,  0, 1.5,   -1, 4.860151595552102e+01, 0,_
 273,  17,  28,  0, 1,     +1, 2.778875279627746e+01, 0,_
 274,  17,  29,  0, 1.5,   +1, 2.876488487201202e+01, 0,_
 275,  17,  30,  0, 3,     +1, 2.974701462240059e+01, 0,_
 276,  17,  31,  0, 1.5,   +1, 3.072620809142004e+01, 0,_
 277,  17,  32,  0, 1,     +1, 3.171091225176306e+01, 0,_
 278,  17,  33,  0, 1.5,   +1, 3.269415984612768e+01, 0,_
 279,  17,  34,  0, 0,     +1, 3.368191156265293e+01, 0,_
 280,  17,  34,  1, 3,     +1, 3.368206733679787e+01, 0.14636,_
 281,  17,  35,  0, 1.5,   +1, 3.466845346313237e+01, 0,_
 282,  17,  36,  0, 2,     +1, 3.565932180040123e+01, 0,_
 283,  17,  37,  0, 1.5,   +1, 3.664834773592611e+01, 0,_
 284,  17,  38,  0, 2,     -1, 3.764184698258656e+01, 0,_
 285,  17,  38,  1, 5,     -1, 3.764256146183978e+01, 0.6713,_
 286,  17,  39,  0, 1.5,   +1, 3.863325426593303e+01, 0,_
 287,  17,  40,  0, 2,     -1, 3.962705041297666e+01, 0,_
 288,  17,  41,  0,-77.777,+0, 4.061872667762281e+01, 0,_
 289,  17,  42,  0,-77.777,+0, 4.161277084360323e+01, 0,_
 290,  17,  43,  0, 1.5,   +1, 4.260489586382013e+01, 0,_
 291,  17,  44,  0,-77.777,+0, 4.360032318190400e+01, 0,_
 292,  17,  45,  0, 1.5,   +1, 4.459399117340462e+01, 0,_
 293,  17,  46,  0,-77.777,+0, 4.559008705642957e+01, 0,_
 294,  17,  47,  0, 1.5,   +1, 4.658583862986809e+01, 0,_
 295,  17,  48,  0,-77.777,+0, 4.758310143164031e+01, 0,_
 296,  17,  49,  0, 1.5,   +1, 4.858006681055655e+01, 0,_
 297,  17,  50,  0,-77.777,+0, 4.957871362001865e+01, 0,_
 298,  17,  51,  0, 1.5,   +1, 5.057709473172940e+01, 0,_
 299,  18,  30,  0, 0,     +1, 2.976456307751426e+01, 0,_
 300,  18,  31,  0, 2.5,   +1, 3.074574867590044e+01, 0,_
 301,  18,  32,  0, 0,     +1, 3.172276275713162e+01, 0,_
 302,  18,  33,  0, 0.5,   +1, 3.270652625027673e+01, 0,_
 303,  18,  34,  0, 0,     +1, 3.368836326122788e+01, 0,_
 304,  18,  35,  0, 1.5,   +1, 3.467480346787581e+01, 0,_
 305,  18,  36,  0, 0,     +1, 3.565856662691591e+01, 0,_
 306,  18,  37,  0, 1.5,   +1, 3.664921396016599e+01, 0,_
 307,  18,  38,  0, 0,     +1, 3.763661401183759e+01, 0,_
 308,  18,  39,  0, 3.5,   -1, 3.862959088581339e+01, 0,_
 309,  18,  40,  0, 0,     +1, 3.961908706945790e+01, 0,_
 310,  18,  41,  0, 3.5,   -1, 4.061259584675618e+01, 0,_
 311,  18,  42,  0, 0,     +1, 4.160256302945353e+01, 0,_
 312,  18,  43,  0, 2.5,   -1, 4.259654061633623e+01, 0,_
 313,  18,  44,  0, 0,     +1, 4.358724401577749e+01, 0,_
 314,  18,  45,  0,-77.777,+0, 4.458174268552106e+01, 0,_
 315,  18,  46,  0, 0,     +1, 4.557314993813382e+01, 0,_
 316,  18,  47,  0, 1.5,   -1, 4.656924948739117e+01, 0,_
 317,  18,  48,  0, 0,     +1, 4.756394244266696e+01, 0,_
 318,  18,  49,  0, 1.5,   -1, 4.856077100706945e+01, 0,_
 319,  18,  50,  0, 0,     +1, 4.955628892808653e+01, 0,_
 320,  18,  51,  0, 1.5,   -1, 5.055476521511120e+01, 0,_
 321,  18,  52,  0, 0,     +1, 5.155193284156950e+01, 0,_
 322,  18,  53,  0, 2.5,   -1, 5.255193193361684e+01, 0,_
 323,  19,  32,  0, 1,     +1, 3.174756006455027e+01, 0,_
 324,  19,  33,  0, 1.5,   +1, 3.272400921486032e+01, 0,_
 325,  19,  34,  0, 1,     +1, 3.370662493238458e+01, 0,_
 326,  19,  35,  0, 1.5,   +1, 3.468744164513375e+01, 0,_
 327,  19,  36,  0, 2,     +1, 3.567220535349819e+01, 0,_
 328,  19,  37,  0, 1.5,   +1, 3.665575684164872e+01, 0,_
 329,  19,  38,  0, 3,     +1, 3.764290848277677e+01, 0,_
 330,  19,  38,  1, 0,     +1, 3.764304727034510e+01, 0.1304,_
 331,  19,  39,  0, 1.5,   +1, 3.862898954399958e+01, 0,_
 332,  19,  40,  0, 4,     -1, 3.962068823768995e+01, 0,_
 333,  19,  41,  0, 1.5,   +1, 4.060994351593943e+01, 0,_
 334,  19,  42,  0, 2,     -1, 4.160192512682611e+01, 0,_
 335,  19,  43,  0, 1.5,   +1, 4.259168136930273e+01, 0,_
 336,  19,  44,  0, 2,     -1, 4.358393585074898e+01, 0,_
 337,  19,  45,  0, 1.5,   +1, 4.457445757043006e+01, 0,_
 338,  19,  46,  0, 2,     -1, 4.556714610087825e+01, 0,_
 339,  19,  47,  0, 0.5,   +1, 4.655823839755767e+01, 0,_
 340,  19,  48,  0, 2,     -1, 4.755329588023422e+01, 0,_
 341,  19,  49,  0, 1.5,   +1, 4.854755031822536e+01, 0,_
 342,  19,  50,  0,-77.777,+0, 4.954309328422974e+01, 0,_
 343,  19,  51,  0, 1.5,   +1, 5.053792103550950e+01, 0,_
 344,  19,  52,  0, 2,     -1, 5.153505508590160e+01, 0,_
 345,  19,  53,  0, 1.5,   +1, 5.253161795255273e+01, 0,_
 346,  19,  54,  0, 2,     -1, 5.353079020906041e+01, 0,_
 347,  19,  55,  0, 1.5,   +1, 5.452827706938415e+01, 0,_
 348,  20,  34,  0, 0,     +1, 3.372266593841763e+01, 0,_
 349,  20,  35,  0, 0.5,   +1, 3.470442904375470e+01, 0,_
 350,  20,  36,  0, 0,     +1, 3.568387661896539e+01, 0,_
 351,  20,  37,  0, 1.5,   +1, 3.666817123497559e+01, 0,_
 352,  20,  38,  0, 0,     +1, 3.765008441393734e+01, 0,_
 353,  20,  39,  0, 1.5,   +1, 3.863593369986358e+01, 0,_
 354,  20,  40,  0, 0,     +1, 3.961929302608110e+01, 0,_
 355,  20,  41,  0, 3.5,   -1, 4.061039229038662e+01, 0,_
 356,  20,  42,  0, 0,     +1, 4.159817315930881e+01, 0,_
 357,  20,  43,  0, 3.5,   -1, 4.258973000471263e+01, 0,_
 358,  20,  44,  0, 0,     +1, 4.357788285643516e+01, 0,_
 359,  20,  45,  0, 3.5,   -1, 4.456999110497722e+01, 0,_
 360,  20,  46,  0, 0,     +1, 4.555892374886627e+01, 0,_
 361,  20,  47,  0, 3.5,   -1, 4.655117934366542e+01, 0,_
 362,  20,  48,  0, 0,     +1, 4.754058771342980e+01, 0,_
 363,  20,  49,  0, 1.5,   -1, 4.853511023053797e+01, 0,_
 364,  20,  50,  0, 0,     +1, 4.952834031744502e+01, 0,_
 365,  20,  51,  0, 1.5,   -1, 5.052321624924463e+01, 0,_
 366,  20,  52,  0, 0,     +1, 5.151682469867230e+01, 0,_
 367,  20,  53,  0, 1.5,   -1, 5.251342657927086e+01, 0,_
 368,  20,  54,  0, 0,     +1, 5.350933511563881e+01, 0,_
 369,  20,  55,  0, 2.5,   -1, 5.450799283060564e+01, 0,_
 370,  20,  56,  0, 0,     +1, 5.550414128807742e+01, 0,_
 371,  20,  57,  0, 2.5,   -1, 5.650302603582431e+01, 0,_
 372,  21,  36,  0,-77.777,+0, 3.570708114767746e+01, 0,_
 373,  21,  37,  0, 3.5,   -1, 3.668589877533072e+01, 0,_
 374,  21,  38,  0, 2,     -1, 3.766903894849102e+01, 0,_
 375,  21,  39,  0, 3.5,   -1, 3.864988695309888e+01, 0,_
 376,  21,  40,  0, 4,     -1, 3.963453736219196e+01, 0,_
 377,  21,  41,  0, 3.5,   -1, 4.061730557039729e+01, 0,_
 378,  21,  42,  0, 0,     +1, 4.160501259021574e+01, 0,_
 379,  21,  42,  1, 7,     +1, 4.160566851048115e+01, 0.61628,_
 380,  21,  43,  0, 3.5,   -1, 4.259209356664743e+01, 0,_
 381,  21,  44,  0, 2,     +1, 4.358177049437964e+01, 0,_
 382,  21,  44,  1, 6,     +1, 4.358205903203128e+01, 0.2711,_
 383,  21,  45,  0, 3.5,   -1, 4.456971467720064e+01, 0,_
 384,  21,  45,  1, 1.5,   +1, 4.456972787479149e+01, 0.0124,_
 385,  21,  46,  0, 4,     +1, 4.556039053627683e+01, 0,_
 386,  21,  46,  1, 1,     -1, 4.556054223406911e+01, 0.14253,_
 387,  21,  47,  0, 3.5,   -1, 4.654905902603916e+01, 0,_
 388,  21,  48,  0, 6,     +1, 4.754029054140042e+01, 0,_
 389,  21,  49,  0, 3.5,   -1, 4.852951030145209e+01, 0,_
 390,  21,  50,  0, 5,     +1, 4.952306324086933e+01, 0,_
 391,  21,  50,  1, 2,     +1, 4.952333665982919e+01, 0.256895,_
 392,  21,  51,  0, 3.5,   -1, 5.051587627946490e+01, 0,_
 393,  21,  52,  0, 3,     +1, 5.151025036769678e+01, 0,_
 394,  21,  53,  0, 3.5,   -1, 5.250329518792402e+01, 0,_
 395,  21,  54,  0, 3,     +1, 5.350004324888208e+01, 0,_
 396,  21,  55,  0, 3.5,   -1, 5.449542433654862e+01, 0,_
 397,  21,  56,  0, 3,     +1, 5.549248231212262e+01, 0,_
 398,  21,  57,  0, 3.5,   -1, 5.648799626750162e+01, 0,_
 399,  21,  58,  0, 3,     +1, 5.748591934801414e+01, 0,_
 400,  21,  59,  0, 3.5,   -1, 5.848219668872353e+01, 0,_
 401,  21,  60,  0, 3,     +1, 5.948025856656884e+01, 0,_
 402,  22,  38,  0, 0,     +1, 3.768513051640959e+01, 0,_
 403,  22,  39,  0, 1.5,   +1, 3.866731298797361e+01, 0,_
 404,  22,  40,  0, 0,     +1, 3.964696114020134e+01, 0,_
 405,  22,  41,  0, 1.5,   +1, 4.063108308538967e+01, 0,_
 406,  22,  42,  0, 0,     +1, 4.161248038242488e+01, 0,_
 407,  22,  43,  0, 3.5,   -1, 4.259940228620769e+01, 0,_
 408,  22,  44,  0, 0,     +1, 4.358205511219027e+01, 0,_
 409,  22,  45,  0, 3.5,   -1, 4.457190936838626e+01, 0,_
 410,  22,  46,  0, 0,     +1, 4.555787173249954e+01, 0,_
 411,  22,  47,  0, 2.5,   -1, 4.654841961448643e+01, 0,_
 412,  22,  48,  0, 0,     +1, 4.753604510276878e+01, 0,_
 413,  22,  49,  0, 3.5,   -1, 4.852737897323171e+01, 0,_
 414,  22,  50,  0, 0,     +1, 4.951573615346489e+01, 0,_
 415,  22,  51,  0, 1.5,   -1, 5.050895376744878e+01, 0,_
 416,  22,  52,  0, 0,     +1, 5.150064322180719e+01, 0,_
 417,  22,  53,  0, 1.5,   -1, 5.249486123144693e+01, 0,_
 418,  22,  54,  0, 0,     +1, 5.348755760196160e+01, 0,_
 419,  22,  55,  0, 1.5,   -1, 5.448317533709774e+01, 0,_
 420,  22,  56,  0, 0,     +1, 5.547708392782631e+01, 0,_
 421,  22,  57,  0, 2.5,   -1, 5.647424548487554e+01, 0,_
 422,  22,  58,  0, 0,     +1, 5.746864106290018e+01, 0,_
 423,  22,  59,  0, 2.5,   -1, 5.846601093690055e+01, 0,_
 424,  22,  60,  0, 0,     +1, 5.946080512896806e+01, 0,_
 425,  22,  61,  0, 0.5,   -1, 6.045857949805259e+01, 0,_
 426,  22,  62,  0, 0,     +1, 6.145408651356496e+01, 0,_
 427,  22,  63,  0, 0.5,   -1, 6.245267978691298e+01, 0,_
 428,  23,  40,  0, 2,     -1, 3.966933356403216e+01, 0,_
 429,  23,  41,  0, 3.5,   -1, 4.064813136349501e+01, 0,_
 430,  23,  42,  0, 2,     -1, 4.163109010871316e+01, 0,_
 431,  23,  43,  0, 3.5,   -1, 4.261154058784580e+01, 0,_
 432,  23,  44,  0, 2,     +1, 4.359635128604725e+01, 0,_
 433,  23,  44,  1, 6,     +1, 4.359663865294487e+01, 0.27,_
 434,  23,  45,  0, 3.5,   -1, 4.457949141062849e+01, 0,_
 435,  23,  46,  0, 0,     +1, 4.556537780798729e+01, 0,_
 436,  23,  46,  1, 3,     +1, 4.556623043621573e+01, 0.8011,_
 437,  23,  47,  0, 1.5,   -1, 4.655153887237937e+01, 0,_
 438,  23,  48,  0, 4,     +1, 4.754031837224846e+01, 0,_
 439,  23,  49,  0, 3.5,   -1, 4.852801954175933e+01, 0,_
 440,  23,  50,  0, 6,     +1, 4.951808579997567e+01, 0,_
 441,  23,  51,  0, 3.5,   -1, 5.050632382269865e+01, 0,_
 442,  23,  52,  0, 3,     +1, 5.149854230507383e+01, 0,_
 443,  23,  53,  0, 3.5,   -1, 5.248951833519320e+01, 0,_
 444,  23,  54,  0, 3,     +1, 5.348301167800997e+01, 0,_
 445,  23,  55,  0, 3.5,   -1, 5.447521793630575e+01, 0,_
 446,  23,  56,  0, 1,     +1, 5.546980946443919e+01, 0,_
 447,  23,  57,  0, 1.5,   -1, 5.646307242747734e+01, 0,_
 448,  23,  58,  0, 3,     +1, 5.745875168552266e+01, 0,_
 449,  23,  59,  0, 3.5,   -1, 5.845289623369979e+01, 0,_
 450,  23,  60,  0, 3,     +1, 5.944919105395042e+01, 0,_
 451,  23,  61,  0, 3.5,   -1, 6.044351205616815e+01, 0,_
 452,  23,  62,  0, 3,     +1, 6.144027518754231e+01, 0,_
 453,  23,  63,  0, 3.5,   -1, 6.243550163416052e+01, 0,_
 454,  23,  64,  0,-77.777,+0, 6.343283978305627e+01, 0,_
 455,  23,  65,  0, 2.5,   -1, 6.442908143302293e+01, 0,_
 456,  24,  42,  0, 0,     +1, 4.164636277236824e+01, 0,_
 457,  24,  43,  0, 1.5,   +1, 4.262851252741810e+01, 0,_
 458,  24,  44,  0, 0,     +1, 4.360779909177430e+01, 0,_
 459,  24,  45,  0, 3.5,   -1, 4.459265836904223e+01, 0,_
 460,  24,  45,  1, 1.5,   +1, 4.459271158513437e+01, 0.05,_
 461,  24,  46,  0, 0,     +1, 4.557347068290736e+01, 0,_
 462,  24,  47,  0, 1.5,   -1, 4.655946172750664e+01, 0,_
 463,  24,  48,  0, 0,     +1, 4.754208054120281e+01, 0,_
 464,  24,  49,  0, 2.5,   -1, 4.853081750662927e+01, 0,_
 465,  24,  50,  0, 0,     +1, 4.951698097122380e+01, 0,_
 466,  24,  51,  0, 3.5,   -1, 5.050712466843774e+01, 0,_
 467,  24,  52,  0, 0,     +1, 5.149431112092827e+01, 0,_
 468,  24,  53,  0, 1.5,   -1, 5.248586139031275e+01, 0,_
 469,  24,  54,  0, 0,     +1, 5.347551715740016e+01, 0,_
 470,  24,  55,  0, 1.5,   -1, 5.446886912323656e+01, 0,_
 471,  24,  56,  0, 0,     +1, 5.546009207777661e+01, 0,_
 472,  24,  57,  0, 1.5,   -1, 5.645443944355630e+01, 0,_
 473,  24,  58,  0, 0,     +1, 5.744641614196915e+01, 0,_
 474,  24,  59,  0, 2.5,   -1, 5.844198294740684e+01, 0,_
 475,  24,  60,  0, 0,     +1, 5.943489974295714e+01, 0,_
 476,  24,  61,  0, 2.5,   -1, 6.043077339856394e+01, 0,_
 477,  24,  62,  0, 0,     +1, 6.142386483288399e+01, 0,_
 478,  24,  63,  0, 0.5,   -1, 6.242047621217717e+01, 0,_
 479,  24,  64,  0, 0,     +1, 6.341457603687945e+01, 0,_
 480,  24,  65,  0, 0.5,   -1, 6.441158404640504e+01, 0,_
 481,  24,  66,  0, 0,     +1, 6.540671829193789e+01, 0,_
 482,  24,  67,  0, 0.5,   -1, 6.640455611123170e+01, 0,_
 483,  25,  44,  0, 2,     -1, 4.362950104616637e+01, 0,_
 484,  25,  45,  0, 3.5,   -1, 4.460784081443099e+01, 0,_
 485,  25,  46,  0, 4,     +1, 4.559145983838944e+01, 0,_
 486,  25,  46,  1, 1,     -1, 4.559161948666589e+01, 0.15,_
 487,  25,  47,  0, 2.5,   -1, 4.657222955138751e+01, 0,_
 488,  25,  48,  0, 4,     +1, 4.755647622184190e+01, 0,_
 489,  25,  49,  0, 2.5,   -1, 4.853902601063307e+01, 0,_
 490,  25,  50,  0, 0,     +1, 4.952510651209615e+01, 0,_
 491,  25,  50,  1, 5,     +1, 4.952535024179820e+01, 0.229,_
 492,  25,  51,  0, 2.5,   -1, 5.051053850086712e+01, 0,_
 493,  25,  52,  0, 6,     +1, 5.149932616201937e+01, 0,_
 494,  25,  52,  1, 2,     +1, 5.149972819682371e+01, 0.377738,_
 495,  25,  53,  0, 3.5,   -1, 5.248649666669636e+01, 0,_
 496,  25,  54,  0, 3,     +1, 5.347698287408843e+01, 0,_
 497,  25,  55,  0, 2.5,   -1, 5.446609900904286e+01, 0,_
 498,  25,  56,  0, 3,     +1, 5.545836091812923e+01, 0,_
 499,  25,  57,  0, 2.5,   -1, 5.644915875924136e+01, 0,_
 500,  25,  58,  0, 3,     +1, 5.744233366032889e+01, 0,_
 501,  25,  58,  1, 0,     +1, 5.744241135582343e+01, 0.073,_
 502,  25,  59,  0, 2.5,   -1, 5.843406485996795e+01, 0,_
 503,  25,  60,  0, 0,     +1, 5.942819625808922e+01, 0,_
 504,  25,  60,  1, 3,     +1, 5.942848564719833e+01, 0.2719,_
 505,  25,  61,  0, 2.5,   -1, 6.042091044125873e+01, 0,_
 506,  25,  62,  0, 3,     +1, 6.141574512447385e+01, 0,_
 507,  25,  62,  1, 1,     +1, 6.141574618879569e+01, 0.001,_
 508,  25,  63,  0, 2.5,   -1, 6.240889683748414e+01, 0,_
 509,  25,  64,  0, 1,     +1, 6.340445510391163e+01, 0,_
 510,  25,  65,  0, 2.5,   -1, 6.439801635927192e+01, 0,_
 511,  25,  66,  0,-77.777,+0, 6.539391407341355e+01, 0,_
 512,  25,  67,  0, 2.5,   -1, 6.638882541637840e+01, 0,_
 513,  25,  68,  0,-77.777,+0, 6.738564208386664e+01, 0,_
 514,  25,  69,  0, 2.5,   -1, 6.838089331572286e+01, 0,_
 515,  26,  45,  0, 1.5,   +1, 4.462807548273351e+01, 0,_
 516,  26,  46,  0, 0,     +1, 4.560580652555287e+01, 0,_
 517,  26,  47,  0, 3.5,   -1, 4.658893579320846e+01, 0,_
 518,  26,  48,  0, 0,     +1, 4.756849896082073e+01, 0,_
 519,  26,  49,  0, 3.5,   -1, 4.855272373599588e+01, 0,_
 520,  26,  50,  0, 0,     +1, 4.953378194916426e+01, 0,_
 521,  26,  51,  0, 2.5,   -1, 5.051909705570986e+01, 0,_
 522,  26,  52,  0, 0,     +1, 5.150185597044235e+01, 0,_
 523,  26,  52,  1, 12,    +1, 5.150926152182604e+01, 6.958,_
 524,  26,  53,  0, 3.5,   -1, 5.249047998359079e+01, 0,_
 525,  26,  53,  1, 9.5,   -1, 5.249371594772229e+01, 3.0404,_
 526,  26,  54,  0, 0,     +1, 5.347624117782413e+01, 0,_
 527,  26,  55,  0, 1.5,   -1, 5.446634498865026e+01, 0,_
 528,  26,  56,  0, 0,     +1, 5.545442766939838e+01, 0,_
 529,  26,  57,  0, 0.5,   -1, 5.644628978466455e+01, 0,_
 530,  26,  58,  0, 0,     +1, 5.743559910666944e+01, 0,_
 531,  26,  59,  0, 1.5,   -1, 5.842859479363228e+01, 0,_
 532,  26,  60,  0, 0,     +1, 5.941920797579863e+01, 0,_
 533,  26,  61,  0,-77.777,+0, 6.041327034207501e+01, 0,_
 534,  26,  62,  0, 0,     +1, 6.140472503859036e+01, 0,_
 535,  26,  63,  0, 2.5,   -1, 6.239958554405785e+01, 0,_
 536,  26,  64,  0, 0,     +1, 6.339170398428977e+01, 0,_
 537,  26,  65,  0, 0.5,   -1, 6.438710646657375e+01, 0,_
 538,  26,  66,  0, 0,     +1, 6.537974001642419e+01, 0,_
 539,  26,  67,  0, 0.5,   -1, 6.637589394845935e+01, 0,_
 540,  26,  68,  0, 0,     +1, 6.736956328034564e+01, 0,_
 541,  26,  69,  0, 0.5,   -1, 6.836571681879121e+01, 0,_
 542,  26,  70,  0, 0,     +1, 6.935980810150906e+01, 0,_
 543,  26,  71,  0, 3.5,   +1, 7.035662476899731e+01, 0,_
 544,  26,  72,  0, 0,     +1, 7.135122662761795e+01, 0,_
 545,  27,  47,  0, 3.5,   -1, 4.660728480060400e+01, 0,_
 546,  27,  48,  0, 6,     +1, 4.758925312771170e+01, 0,_
 547,  27,  49,  0, 3.5,   -1, 4.856855059757262e+01, 0,_
 548,  27,  50,  0, 6,     +1, 4.955171159033284e+01, 0,_
 549,  27,  51,  0, 3.5,   -1, 5.053278467464402e+01, 0,_
 550,  27,  52,  0, 6,     +1, 5.151672392387742e+01, 0,_
 551,  27,  52,  1, 2,     +1, 5.151712836617777e+01, 0.38,_
 552,  27,  53,  0, 3.5,   -1, 5.249930119156625e+01, 0,_
 553,  27,  53,  1, 9.5,   -1, 5.250270169985468e+01, 3.195,_
 554,  27,  54,  0, 0,     +1, 5.348501603046467e+01, 0,_
 555,  27,  54,  1, 7,     +1, 5.348522783051143e+01, 0.199,_
 556,  27,  55,  0, 3.5,   -1, 5.447001840679130e+01, 0,_
 557,  27,  56,  0, 4,     +1, 5.545928808728653e+01, 0,_
 558,  27,  57,  0, 3.5,   -1, 5.644717985131198e+01, 0,_
 559,  27,  58,  0, 2,     +1, 5.743805551311437e+01, 0,_
 560,  27,  58,  1, 5,     +1, 5.743808200302072e+01, 0.024889,_
 561,  27,  59,  0, 3.5,   -1, 5.842692923257861e+01, 0,_
 562,  27,  60,  0, 5,     +1, 5.941895542017187e+01, 0,_
 563,  27,  60,  1, 2,     +1, 5.941901779262484e+01, 0.058603,_
 564,  27,  61,  0, 3.5,   -1, 6.040903692527522e+01, 0,_
 565,  27,  62,  0, 2,     +1, 6.140201502562459e+01, 0,_
 566,  27,  62,  1, 5,     +1, 6.140203844070513e+01, 0.022,_
 567,  27,  63,  0, 3.5,   -1, 6.239296991195313e+01, 0,_
 568,  27,  64,  0, 1,     +1, 6.338657098925961e+01, 0,_
 569,  27,  65,  0, 3.5,   -1, 6.437862669210570e+01, 0,_
 570,  27,  66,  0, 3,     +1, 6.537299147695063e+01, 0,_
 571,  27,  67,  0, 3.5,   -1, 6.636555765754997e+01, 0,_
 572,  27,  68,  0, 7,     -1, 6.736057641081362e+01, 0,_
 573,  27,  68,  1, 3,     +1, 6.736073605909007e+01, 0.15,_
 574,  27,  69,  0, 3.5,   -1, 6.835374366714235e+01, 0,_
 575,  27,  70,  0, 6,     -1, 6.934903742848415e+01, 0,_
 576,  27,  70,  1, 3,     +1, 6.934925029285276e+01, 0.2,_
 577,  27,  71,  0, 3.5,   -1, 7.034285203160741e+01, 0,_
 578,  27,  72,  0,-77.777,+0, 7.133870016833318e+01, 0,_
 579,  27,  73,  0, 3.5,   -1, 7.233307003712613e+01, 0,_
 580,  27,  74,  0,-77.777,+0, 7.332937315448304e+01, 0,_
 581,  27,  75,  0, 3.5,   -1, 7.432415743245534e+01, 0,_
 582,  28,  48,  0, 0,     +1, 4.760553008921051e+01, 0,_
 583,  28,  49,  0, 3.5,   -1, 4.858779385635517e+01, 0,_
 584,  28,  50,  0, 0,     +1, 4.956609099401043e+01, 0,_
 585,  28,  51,  0, 3.5,   -1, 5.054922026166602e+01, 0,_
 586,  28,  52,  0, 0,     +1, 5.152952896923730e+01, 0,_
 587,  28,  53,  0, 3.5,   -1, 5.251316782138615e+01, 0,_
 588,  28,  54,  0, 0,     +1, 5.349430931295421e+01, 0,_
 589,  28,  55,  0, 3.5,   -1, 5.447927165814724e+01, 0,_
 590,  28,  56,  0, 0,     +1, 5.546155813947541e+01, 0,_
 591,  28,  57,  0, 1.5,   -1, 5.645065138278400e+01, 0,_
 592,  28,  58,  0, 0,     +1, 5.743764938319593e+01, 0,_
 593,  28,  59,  0, 1.5,   -1, 5.842807125207580e+01, 0,_
 594,  28,  60,  0, 0,     +1, 5.941595104185706e+01, 0,_
 595,  28,  61,  0, 1.5,   -1, 6.040762793406544e+01, 0,_
 596,  28,  62,  0, 0,     +1, 6.139635066086837e+01, 0,_
 597,  28,  63,  0, 0.5,   -1, 6.238907307273690e+01, 0,_
 598,  28,  64,  0, 0,     +1, 6.337879442255547e+01, 0,_
 599,  28,  65,  0, 2.5,   -1, 6.437230410082690e+01, 0,_
 600,  28,  66,  0, 0,     +1, 6.536277637891521e+01, 0,_
 601,  28,  67,  0, 0.5,   -1, 6.635659510332783e+01, 0,_
 602,  28,  68,  0, 0,     +1, 6.734830142651923e+01, 0,_
 603,  28,  69,  0, 4.5,   +1, 6.834342028438492e+01, 0,_
 604,  28,  69,  1, 0.5,   -1, 6.834376193169652e+01, 0.321,_
 605,  28,  70,  0, 0,     +1, 6.933564378626650e+01, 0,_
 606,  28,  71,  0, 0.5,   -1, 7.033110585226264e+01, 0,_
 607,  28,  72,  0, 0,     +1, 7.132377145041721e+01, 0,_
 608,  28,  73,  0, 4.5,   +1, 7.231956375095881e+01, 0,_
 609,  28,  74,  0, 0,     +1, 7.331273135428088e+01, 0,_
 610,  28,  75,  0, 3.5,   +1, 7.430885205228610e+01, 0,_
 611,  28,  76,  0, 0,     +1, 7.530281544317589e+01, 0,_
 612,  28,  77,  0, 4.5,   +1, 7.629936641291276e+01, 0,_
 613,  28,  78,  0, 0,     +1, 7.729387309621949e+01, 0,_
 614,  29,  52,  0, 3,     +1, 5.155086806774506e+01, 0,_
 615,  29,  53,  0, 1.5,   -1, 5.253058093819484e+01, 0,_
 616,  29,  54,  0, 3,     +1, 5.351332653036620e+01, 0,_
 617,  29,  55,  0, 1.5,   -1, 5.449385334803187e+01, 0,_
 618,  29,  56,  0, 4,     +1, 5.547780449417951e+01, 0,_
 619,  29,  57,  0, 1.5,   -1, 5.645999075193649e+01, 0,_
 620,  29,  58,  0, 1,     +1, 5.744676106424664e+01, 0,_
 621,  29,  59,  0, 1.5,   -1, 5.843317827316794e+01, 0,_
 622,  29,  60,  0, 2,     +1, 5.942247318800969e+01, 0,_
 623,  29,  61,  0, 1.5,   -1, 6.041000972207286e+01, 0,_
 624,  29,  62,  0, 1,     +1, 6.140056420089871e+01, 0,_
 625,  29,  63,  0, 1.5,   -1, 6.238900178741819e+01, 0,_
 626,  29,  64,  0, 1,     +1, 6.338057649703016e+01, 0,_
 627,  29,  65,  0, 1.5,   -1, 6.437002860778257e+01, 0,_
 628,  29,  66,  0, 1,     +1, 6.536250818281187e+01, 0,_
 629,  29,  67,  0, 1.5,   -1, 6.635278820180902e+01, 0,_
 630,  29,  68,  0, 1,     +1, 6.734606292296409e+01, 0,_
 631,  29,  68,  1, 6,     -1, 6.734683093760601e+01, 0.7216,_
 632,  29,  69,  0, 1.5,   -1, 6.833729238214201e+01, 0,_
 633,  29,  70,  0, 6,     -1, 6.933163926113961e+01, 0,_
 634,  29,  70,  1, 3,     -1, 6.933174686407794e+01, 0.1011,_
 635,  29,  70,  2, 1,     +1, 6.933189746561872e+01, 0.2426,_
 636,  29,  71,  0, 1.5,   -1, 7.032333108794100e+01, 0,_
 637,  29,  72,  0, 2,     +0, 7.131785707894812e+01, 0,_
 638,  29,  73,  0, 1.5,   -1, 7.231011333303459e+01, 0,_
 639,  29,  74,  0, 1,     +1, 7.330469584330703e+01, 0,_
 640,  29,  75,  0, 1.5,   -1, 7.429773895236146e+01, 0,_
 641,  29,  76,  0,-77.777,+0, 7.529286865624729e+01, 0,_
 642,  29,  76,  1,-77.777,+0, 7.529286865624729e+01, 0,_
 643,  29,  77,  0, 1.5,   -1, 7.628678146046632e+01, 0,_
 644,  29,  78,  0,-77.777,+0, 7.728258292460612e+01, 0,_
 645,  29,  79,  0, 1.5,   -1, 7.827692701675156e+01, 0,_
 646,  29,  80,  0,-77.777,+0, 7.927423344054267e+01, 0,_
 647,  30,  54,  0, 0,     +1, 5.352943891788470e+01, 0,_
 648,  30,  55,  0, 2.5,   -1, 5.451201597043766e+01, 0,_
 649,  30,  56,  0, 0,     +1, 5.549191026882959e+01, 0,_
 650,  30,  57,  0, 3.5,   -1, 5.647569882381595e+01, 0,_
 651,  30,  58,  0, 0,     +1, 5.745673267606961e+01, 0,_
 652,  30,  59,  0, 1.5,   -1, 5.844290912912026e+01, 0,_
 653,  30,  60,  0, 0,     +1, 5.942691225784326e+01, 0,_
 654,  30,  61,  0, 1.5,   -1, 6.041600734219934e+01, 0,_
 655,  30,  61,  1, 0.5,   -1, 6.041610142825026e+01, 0.0884,_
 656,  30,  61,  2, 1.5,   -1, 6.041645233516191e+01, 0.4181,_
 657,  30,  61,  3, 2.5,   -1, 6.041681199079910e+01, 0.75602,_
 658,  30,  62,  0, 0,     +1, 6.140228782186810e+01, 0,_
 659,  30,  63,  0, 1.5,   -1, 6.239258467301610e+01, 0,_
 660,  30,  64,  0, 0,     +1, 6.337995975603893e+01, 0,_
 661,  30,  65,  0, 2.5,   -1, 6.437146718760947e+01, 0,_
 662,  30,  66,  0, 0,     +1, 6.535969742776773e+01, 0,_
 663,  30,  67,  0, 2.5,   -1, 6.635219132965599e+01, 0,_
 664,  30,  68,  0, 0,     +1, 6.734133727232904e+01, 0,_
 665,  30,  69,  0, 0.5,   -1, 6.833443826284545e+01, 0,_
 666,  30,  69,  1, 4.5,   +1, 6.833490511272139e+01, 0.438636,_
 667,  30,  70,  0, 0,     +1, 6.932462712272867e+01, 0,_
 668,  30,  71,  0, 0.5,   -1, 7.031841642111533e+01, 0,_
 669,  30,  71,  1, 4.5,   +1, 7.031858426466998e+01, 0.1577,_
 670,  30,  72,  0, 0,     +1, 7.130895669998262e+01, 0,_
 671,  30,  73,  0, 0.5,   -1, 7.230308245896653e+01, 0,_
 672,  30,  73,  1, 2.5,   +1, 7.230329053388684e+01, 0.1955,_
 673,  30,  73,  2, 3.5,   +1, 7.230333534183643e+01, 0.2376,_
 674,  30,  74,  0, 0,     +1, 7.329431816501584e+01, 0,_
 675,  30,  75,  0, 3.5,   +1, 7.428913117800167e+01, 0,_
 676,  30,  76,  0, 0,     +1, 7.528081304906866e+01, 0,_
 677,  30,  77,  0, 3.5,   +1, 7.627596240566541e+01, 0,_
 678,  30,  77,  1, 0.5,   -1, 7.627678447721374e+01, 0.77239,_
 679,  30,  78,  0, 0,     +1, 7.726876188772722e+01, 0,_
 680,  30,  79,  0, 4.5,   +1, 7.826448291187278e+01, 0,_
 681,  30,  80,  0, 0,     +1, 7.925779084425535e+01, 0,_
 682,  30,  81,  0, 2.5,   +1, 8.025499980158514e+01, 0,_
 683,  30,  82,  0, 0,     +1, 8.125054496653584e+01, 0,_
 684,  30,  83,  0, 2.5,   +1, 8.224836583371825e+01, 0,_
 685,  31,  56,  0, 3,     +1, 5.551532537887216e+01, 0,_
 686,  31,  57,  0, 0.5,   -1, 5.649436706507692e+01, 0,_
 687,  31,  58,  0, 2,     +1, 5.747669824806894e+01, 0,_
 688,  31,  59,  0, 1.5,   -1, 5.845723002278221e+01, 0,_
 689,  31,  60,  0, 2,     +1, 5.944243431056309e+01, 0,_
 690,  31,  61,  0, 1.5,   -1, 6.042581426503848e+01, 0,_
 691,  31,  62,  0, 0,     +1, 6.141205943169155e+01, 0,_
 692,  31,  63,  0, 1.5,   -1, 6.239861544488551e+01, 0,_
 693,  31,  64,  0, 0,     +1, 6.338759221442948e+01, 0,_
 694,  31,  65,  0, 1.5,   -1, 6.437493103664913e+01, 0,_
 695,  31,  66,  0, 0,     +1, 6.536520582596107e+01, 0,_
 696,  31,  67,  0, 1.5,   -1, 6.635325699773182e+01, 0,_
 697,  31,  68,  0, 1,     +1, 6.734444626318498e+01, 0,_
 698,  31,  69,  0, 1.5,   -1, 6.833346976777372e+01, 0,_
 699,  31,  70,  0, 1,     +1, 6.932532382188909e+01, 0,_
 700,  31,  71,  0, 1.5,   -1, 7.031542529497156e+01, 0,_
 701,  31,  72,  0, 3,     -1, 7.130848541066489e+01, 0,_
 702,  31,  72,  1, 0,     +1, 7.130861276741662e+01, 0.11966,_
 703,  31,  73,  0, 1.5,   -1, 7.229871242494347e+01, 0,_
 704,  31,  74,  0, 3,     -1, 7.329187777479170e+01, 0,_
 705,  31,  74,  1,-77.777,+0, 7.329194117750821e+01, 0.059571,_
 706,  31,  75,  0, 1.5,   -1, 7.428284564164510e+01, 0,_
 707,  31,  76,  0,-77.777,+0, 7.527656254730860e+01, 0,_
 708,  31,  77,  0, 1.5,   -1, 7.626829593597988e+01, 0,_
 709,  31,  78,  0, 3,     +1, 7.726213891522642e+01, 0,_
 710,  31,  79,  0, 1.5,   -1, 7.825478120930882e+01, 0,_
 711,  31,  80,  0, 3,     +0, 7.924972854737911e+01, 0,_
 712,  31,  81,  0, 2.5,   -1, 8.024283641926900e+01, 0,_
 713,  31,  82,  0,-77.777,+0, 8.123924531109900e+01, 0,_
 714,  31,  83,  0, 1.5,   -1, 8.223456472020507e+01, 0,_
 715,  31,  84,  0,-77.777,+0, 8.323147626459297e+01, 0,_
 716,  31,  85,  0, 1.5,   -1, 8.422739669787516e+01, 0,_
 717,  31,  86,  0,-77.777,+0, 8.522494700840818e+01, 0,_
 718,  32,  58,  0, 0,     +1, 5.749421645428674e+01, 0,_
 719,  32,  59,  0, 3.5,   -1, 5.847627103402268e+01, 0,_
 720,  32,  60,  0, 0,     +1, 5.945573902632103e+01, 0,_
 721,  32,  61,  0, 1.5,   -1, 6.044047536880849e+01, 0,_
 722,  32,  62,  0, 0,     +1, 6.142296517732370e+01, 0,_
 723,  32,  63,  0, 1.5,   -1, 6.240886047753998e+01, 0,_
 724,  32,  64,  0, 0,     +1, 6.339240010248421e+01, 0,_
 725,  32,  65,  0, 1.5,   -1, 6.438150779162093e+01, 0,_
 726,  32,  66,  0, 0,     +1, 6.536745859986483e+01, 0,_
 727,  32,  67,  0, 0.5,   -1, 6.635774930823943e+01, 0,_
 728,  32,  68,  0, 0,     +1, 6.734456036153240e+01, 0,_
 729,  32,  69,  0, 2.5,   -1, 6.833584016845138e+01, 0,_
 730,  32,  70,  0, 0,     +1, 6.932356584362277e+01, 0,_
 731,  32,  71,  0, 0.5,   -1, 7.031567289751636e+01, 0,_
 732,  32,  71,  1, 4.5,   +1, 7.031588402384740e+01, 0.198367,_
 733,  32,  72,  0, 0,     +1, 7.130423066963772e+01, 0,_
 734,  32,  73,  0, 4.5,   +1, 7.229701143785343e+01, 0,_
 735,  32,  73,  1, 0.5,   -1, 7.229708245579272e+01, 0.066726,_
 736,  32,  74,  0, 0,     +1, 7.328615936037127e+01, 0,_
 737,  32,  75,  0, 0.5,   -1, 7.427923505209957e+01, 0,_
 738,  32,  75,  1, 3.5,   +1, 7.427938372721782e+01, 0.13969,_
 739,  32,  76,  0, 0,     +1, 7.526920143175521e+01, 0,_
 740,  32,  77,  0, 3.5,   +1, 7.626273962494724e+01, 0,_
 741,  32,  77,  1, 0.5,   -1, 7.626290959714558e+01, 0.1597,_
 742,  32,  78,  0, 0,     +1, 7.725345819989403e+01, 0,_
 743,  32,  79,  0, 0.5,   -1, 7.824735339866280e+01, 0,_
 744,  32,  79,  1, 3.5,   +1, 7.824755138381204e+01, 0.18602,_
 745,  32,  80,  0, 0,     +1, 7.923875364399308e+01, 0,_
 746,  32,  81,  0, 4.5,   +1, 8.023361541846474e+01, 0,_
 747,  32,  81,  1, 0.5,   +1, 8.023433824200120e+01, 0.67914,_
 748,  32,  82,  0, 0,     +1, 8.122595794500765e+01, 0,_
 749,  32,  83,  0, 2.5,   +1, 8.222209159875247e+01, 0,_
 750,  32,  84,  0, 0,     +1, 8.321651102810735e+01, 0,_
 751,  32,  85,  0, 2.5,   +1, 8.421326877414499e+01, 0,_
 752,  32,  86,  0, 0,     +1, 8.520864129603329e+01, 0,_
 753,  32,  87,  0, 2.5,   +1, 8.620571473858720e+01, 0,_
 754,  32,  88,  0, 0,     +1, 8.720131792082300e+01, 0,_
 755,  32,  89,  0, 1.5,   +1, 8.819954833828672e+01, 0,_
 756,  33,  60,  0, 5,     +1, 5.947874964127946e+01, 0,_
 757,  33,  60,  1, 2,     +1, 5.947881350059004e+01, 0.06,_
 758,  33,  61,  0, 1.5,   -1, 6.045799357502629e+01, 0,_
 759,  33,  62,  0, 1,     +1, 6.144150552675659e+01, 0,_
 760,  33,  63,  0, 1.5,   -1, 6.242314470590366e+01, 0,_
 761,  33,  64,  0, 0,     +1, 6.340813385781870e+01, 0,_
 762,  33,  65,  0, 1.5,   -1, 6.439166266351084e+01, 0,_
 763,  33,  66,  0, 0,     +1, 6.537765688557762e+01, 0,_
 764,  33,  67,  0, 2.5,   -1, 6.636421081086314e+01, 0,_
 765,  33,  68,  0, 3,     +1, 6.735316462828530e+01, 0,_
 766,  33,  69,  0, 2.5,   -1, 6.834008520853639e+01, 0,_
 767,  33,  70,  0, 4,     +1, 6.933018592616482e+01, 0,_
 768,  33,  71,  0, 2.5,   -1, 7.031781580341712e+01, 0,_
 769,  33,  72,  0, 2,     -1, 7.130886696552415e+01, 0,_
 770,  33,  73,  0, 1.5,   -1, 7.229737839123327e+01, 0,_
 771,  33,  74,  0, 2,     -1, 7.328888656536954e+01, 0,_
 772,  33,  75,  0, 1.5,   -1, 7.427798209982554e+01, 0,_
 773,  33,  75,  1, 4.5,   +1, 7.427830556852007e+01, 0.30392,_
 774,  33,  76,  0, 2,     -1, 7.527018221733076e+01, 0,_
 775,  33,  77,  0, 1.5,   -1, 7.625986227692565e+01, 0,_
 776,  33,  78,  0, 2,     -1, 7.725244188914114e+01, 0,_
 777,  33,  79,  0, 1.5,   -1, 7.824297961314610e+01, 0,_
 778,  33,  80,  0, 1,     +1, 7.923590212615510e+01, 0,_
 779,  33,  81,  0, 1.5,   -1, 8.022697232828196e+01, 0,_
 780,  33,  82,  0, 1,     +1, 8.122096590065175e+01, 0,_
 781,  33,  82,  1, 5,     -1, 8.122122665950329e+01, 0.245,_
 782,  33,  83,  0, 1.5,   -1, 8.221283956683024e+01, 0,_
 783,  33,  84,  0, 3,     +1, 8.320831028684009e+01, 0,_
 784,  33,  84,  1,-77.777,+0, 8.320831028684009e+01, 0,_
 785,  33,  85,  0, 1.5,   -1, 8.420255560287224e+01, 0,_
 786,  33,  86,  0,-77.777,+0, 8.519846401432260e+01, 0,_
 787,  33,  87,  0, 1.5,   -1, 8.619343287787638e+01, 0,_
 788,  33,  88,  0,-77.777,+0, 8.719005550867618e+01, 0,_
 789,  33,  89,  0, 1.5,   -1, 8.818563886272159e+01, 0,_
 790,  33,  90,  0,-77.777,+0, 8.918286795657013e+01, 0,_
 791,  33,  91,  0, 1.5,   -1, 9.017899658585152e+01, 0,_
 792,  33,  92,  0,-77.777,+0, 9.117670155626966e+01, 0,_
 793,  34,  65,  0, 1.5,   -1, 6.440647531314865e+01, 0,_
 794,  34,  66,  0, 0,     +1, 6.538868851840779e+01, 0,_
 795,  34,  67,  0, 2.5,   -1, 6.637486141328968e+01, 0,_
 796,  34,  68,  0, 0,     +1, 6.735817234583429e+01, 0,_
 797,  34,  69,  0, 1.5,   -1, 6.834719218022498e+01, 0,_
 798,  34,  70,  0, 0,     +1, 6.933275305421635e+01, 0,_
 799,  34,  71,  0, 2.5,   -1, 7.032286770313137e+01, 0,_
 800,  34,  72,  0, 0,     +1, 7.130925184259672e+01, 0,_
 801,  34,  73,  0, 4.5,   +1, 7.230027905125833e+01, 0,_
 802,  34,  73,  1, 1.5,   -1, 7.230030641497291e+01, 0.02571,_
 803,  34,  74,  0, 0,     +1, 7.328744638144202e+01, 0,_
 804,  34,  75,  0, 2.5,   +1, 7.427890243420561e+01, 0,_
 805,  34,  76,  0, 0,     +1, 7.526703121450500e+01, 0,_
 806,  34,  77,  0, 0.5,   -1, 7.625913516330397e+01, 0,_
 807,  34,  77,  1, 3.5,   +1, 7.625930750042544e+01, 0.161922,_
 808,  34,  78,  0, 0,     +1, 7.724796215367189e+01, 0,_
 809,  34,  79,  0, 3.5,   +1, 7.824055145889827e+01, 0,_
 810,  34,  79,  1, 0.5,   -1, 7.824065338900117e+01, 0.09577,_
 811,  34,  80,  0, 0,     +1, 7.923000050024680e+01, 0,_
 812,  34,  81,  0, 0.5,   -1, 8.022286864039208e+01, 0,_
 813,  34,  81,  1, 3.5,   +1, 8.022297825489869e+01, 0.10299,_
 814,  34,  82,  0, 0,     +1, 8.121299576520448e+01, 0,_
 815,  34,  83,  0, 4.5,   +1, 8.220680357618110e+01, 0,_
 816,  34,  83,  1, 0.5,   -1, 8.220704677372223e+01, 0.2285,_
 817,  34,  84,  0, 0,     +1, 8.319756684882609e+01, 0,_
 818,  34,  85,  0, 2.5,   +1, 8.419273777353696e+01, 0,_
 819,  34,  86,  0, 0,     +1, 8.518618064865211e+01, 0,_
 820,  34,  87,  0, 2.5,   +1, 8.618192945407525e+01, 0,_
 821,  34,  88,  0, 0,     +1, 8.717604440471258e+01, 0,_
 822,  34,  89,  0, 2.5,   +1, 8.817266037918887e+01, 0,_
 823,  34,  90,  0, 0,     +1, 8.916746740108698e+01, 0,_
 824,  34,  91,  0, 0.5,   +1, 9.016443278000322e+01, 0,_
 825,  34,  92,  0, 0,     +1, 9.115994673538222e+01, 0,_
 826,  34,  93,  0, 0.5,   +1, 9.215775084675235e+01, 0,_
 827,  34,  94,  0, 0,     +1, 9.315332428670254e+01, 0,_
 828,  35,  67,  0, 0.5,   -1, 6.638953823982325e+01, 0,_
 829,  35,  68,  0, 3,     +1, 6.737456109966197e+01, 0,_
 830,  35,  69,  0, 0.5,   -1, 6.835802190754936e+01, 0,_
 831,  35,  70,  0, 0,     +1, 6.934393298109111e+01, 0,_
 832,  35,  70,  1, 9,     +1, 6.934637272605185e+01, 2.2923,_
 833,  35,  71,  0, 2.5,   -1, 7.032993915360825e+01, 0,_
 834,  35,  72,  0, 3,     +1, 7.131862472733943e+01, 0,_
 835,  35,  72,  1, 1,     -1, 7.131873233027775e+01, 0.1011,_
 836,  35,  73,  0, 0.5,   -1, 7.230515355211872e+01, 0,_
 837,  35,  74,  0, 0,     -1, 7.329481686284116e+01, 0,_
 838,  35,  74,  1, 4,     +1, 7.329483155048260e+01, 0.0138,_
 839,  35,  75,  0, 1.5,   -1, 7.428216189029300e+01, 0,_
 840,  35,  76,  0, 1,     -1, 7.527231331851837e+01, 0,_
 841,  35,  76,  1, 4,     +1, 7.527242249665302e+01, 0.10258,_
 842,  35,  77,  0, 1.5,   -1, 7.626058762187273e+01, 0,_
 843,  35,  77,  1, 4.5,   +1, 7.626070029098304e+01, 0.10586,_
 844,  35,  78,  0, 1,     +1, 7.725176581129874e+01, 0,_
 845,  35,  79,  0, 1.5,   -1, 7.824039119754937e+01, 0,_
 846,  35,  79,  1, 4.5,   +1, 7.824061172503525e+01, 0.2072,_
 847,  35,  80,  0, 1,     +1, 7.923199127534812e+01, 0,_
 848,  35,  80,  1, 5,     -1, 7.923208264737835e+01, 0.08585,_
 849,  35,  81,  0, 1.5,   -1, 8.022117844776888e+01, 0,_
 850,  35,  82,  0, 5,     -1, 8.121309711005983e+01, 0,_
 851,  35,  82,  1, 2,     -1, 8.121314601458420e+01, 0.045949,_
 852,  35,  83,  0, 1.5,   -1, 8.220289413119283e+01, 0,_
 853,  35,  84,  0, 2,     -1, 8.319561343201829e+01, 0,_
 854,  35,  84,  1, 6,     -1, 8.319595401500806e+01, 0.32,_
 855,  35,  85,  0, 1.5,   -1, 8.418617959956238e+01, 0,_
 856,  35,  86,  0, 2,     -1, 8.518072165041232e+01, 0,_
 857,  35,  87,  0, 1.5,   -1, 8.617398370333653e+01, 0,_
 858,  35,  88,  0, 2,     -1, 8.716877320901125e+01, 0,_
 859,  35,  89,  0, 1.5,   -1, 8.816278147902717e+01, 0,_
 860,  35,  90,  0, 1,     -1, 8.915873987437129e+01, 0,_
 861,  35,  91,  0, 1.5,   -1, 9.015322848229964e+01, 0,_
 862,  35,  92,  0, 2,     -1, 9.114982602746437e+01, 0,_
 863,  35,  93,  0, 1.5,   -1, 9.214479312432638e+01, 0,_
 864,  35,  94,  0,-77.777,+0, 9.314204601200342e+01, 0,_
 865,  35,  95,  0, 1.5,   -1, 9.413731806345956e+01, 0,_
 866,  35,  96,  0,-77.777,+0, 9.513469586873609e+01, 0,_
 867,  35,  97,  0, 1.5,   -1, 9.613047750468550e+01, 0,_
 868,  36,  69,  0, 2.5,   -1, 6.837273599819258e+01, 0,_
 869,  36,  70,  0, 0,     +1, 6.935492243539467e+01, 0,_
 870,  36,  71,  0, 2.5,   -1, 7.034076885515600e+01, 0,_
 871,  36,  72,  0, 0,     +1, 7.132407529859830e+01, 0,_
 872,  36,  73,  0, 1.5,   -1, 7.231270568750136e+01, 0,_
 873,  36,  74,  0, 0,     +1, 7.329796333492541e+01, 0,_
 874,  36,  75,  0, 2.5,   +1, 7.428725295266611e+01, 0,_
 875,  36,  76,  0, 0,     +1, 7.527367070587123e+01, 0,_
 876,  36,  77,  0, 2.5,   +1, 7.626385015826951e+01, 0,_
 877,  36,  78,  0, 0,     +1, 7.725099298973255e+01, 0,_
 878,  36,  79,  0, 0.5,   -1, 7.824212154820658e+01, 0,_
 879,  36,  79,  1, 3.5,   +1, 7.824225966525215e+01, 0.12977,_
 880,  36,  80,  0, 0,     +1, 7.922985799801663e+01, 0,_
 881,  36,  81,  0, 3.5,   +1, 8.022147736567042e+01, 0,_
 882,  36,  81,  1, 0.5,   -1, 8.022168015091117e+01, 0.19053,_
 883,  36,  82,  0, 0,     +1, 8.120980503084908e+01, 0,_
 884,  36,  83,  0, 4.5,   +1, 8.220185436963918e+01, 0,_
 885,  36,  83,  1, 0.5,   -1, 8.220189858476151e+01, 0.041543,_
 886,  36,  84,  0, 0,     +1, 8.319065768279810e+01, 0,_
 887,  36,  85,  0, 4.5,   +1, 8.418308789194860e+01, 0,_
 888,  36,  85,  1, 0.5,   -1, 8.418341237174889e+01, 0.30487,_
 889,  36,  86,  0, 0,     +1, 8.517259724100592e+01, 0,_
 890,  36,  87,  0, 2.5,   +1, 8.616672732028483e+01, 0,_
 891,  36,  88,  0, 0,     +1, 8.715922057180846e+01, 0,_
 892,  36,  89,  0, 1.5,   +1, 8.815398855993095e+01, 0,_
 893,  36,  90,  0, 0,     +1, 8.914707602068165e+01, 0,_
 894,  36,  91,  0, 2.5,   +1, 9.014272716731003e+01, 0,_
 895,  36,  92,  0, 0,     +1, 9.113648313946568e+01, 0,_
 896,  36,  93,  0, 0.5,   +1, 9.213282400969710e+01, 0,_
 897,  36,  94,  0, 0,     +1, 9.312720109595136e+01, 0,_
 898,  36,  95,  0, 0.5,   +1, 9.412413323191500e+01, 0,_
 899,  36,  96,  0, 0,     +1, 9.511882004937443e+01, 0,_
 900,  36,  97,  0, 1.5,   +1, 9.611624957251999e+01, 0,_
 901,  36,  98,  0, 0,     +1, 9.711097160583701e+01, 0,_
 902,  36,  99,  0, 1.5,   +1, 9.810828992654234e+01, 0,_
 903,  36,  100, 0, 0,     +1, 9.910364525639817e+01, 0,_
 904,  37,  71,  0, 2.5,   -1, 7.035595358475229e+01, 0,_
 905,  37,  72,  0, 3,     +1, 7.134068992723974e+01, 0,_
 906,  37,  73,  0, 1.5,   -1, 7.232384893718093e+01, 0,_
 907,  37,  74,  0, 0,     +1, 7.330904912845767e+01, 0,_
 908,  37,  75,  0, 1.5,   -1, 7.429481488416621e+01, 0,_
 909,  37,  76,  0, 1,     -1, 7.528275430326447e+01, 0,_
 910,  37,  77,  0, 1.5,   -1, 7.626953252107413e+01, 0,_
 911,  37,  78,  0, 0,     +1, 7.725870172122310e+01, 0,_
 912,  37,  78,  1, 4,     -1, 7.725882006316883e+01, 0.11119,_
 913,  37,  79,  0, 2.5,   +1, 7.824599490489692e+01, 0,_
 914,  37,  80,  0, 1,     +1, 7.923594366224835e+01, 0,_
 915,  37,  81,  0, 1.5,   -1, 8.022386092532666e+01, 0,_
 916,  37,  81,  1, 4.5,   +1, 8.022395266986953e+01, 0.0862,_
 917,  37,  82,  0, 1,     +1, 8.121449228498655e+01, 0,_
 918,  37,  82,  1, 5,     -1, 8.121456561676153e+01, 0.0689,_
 919,  37,  83,  0, 2.5,   -1, 8.220283354913130e+01, 0,_
 920,  37,  83,  1, 4.5,   +1, 8.220287836772411e+01, 0.04211,_
 921,  37,  84,  0, 2,     -1, 8.319351046009299e+01, 0,_
 922,  37,  84,  1, 6,     -1, 8.319400387969942e+01, 0.4636,_
 923,  37,  85,  0, 2.5,   -1, 8.418235670323723e+01, 0,_
 924,  37,  86,  0, 2,     -1, 8.517314927441291e+01, 0,_
 925,  37,  86,  1, 6,     -1, 8.517374103735763e+01, 0.556,_
 926,  37,  87,  0, 1.5,   -1, 8.616258895090712e+01, 0,_
 927,  37,  88,  0, 2,     -1, 8.715611518968318e+01, 0,_
 928,  37,  89,  0, 1.5,   -1, 8.814847898691731e+01, 0,_
 929,  37,  90,  0, 0,     +1, 8.914238751915272e+01, 0,_
 930,  37,  90,  1, 3,     -1, 8.914250129515774e+01, 0.1069,_
 931,  37,  91,  0, 1.5,   -1, 9.013552056689596e+01, 0,_
 932,  37,  92,  0, 0,     -1, 9.113009388878498e+01, 0,_
 933,  37,  93,  0, 2.5,   -1, 9.212379440149208e+01, 0,_
 934,  37,  94,  0, 3,     -1, 9.311952199868277e+01, 0,_
 935,  37,  95,  0, 2.5,   -1, 9.411377463330000e+01, 0,_
 936,  37,  96,  0, 2,     +1, 9.511001313554144e+01, 0,_
 937,  37,  96,  1, 1,     -1, 9.511001313554144e+01, 0,_
 938,  37,  97,  0, 1.5,   +1, 9.610444023344893e+01, 0,_
 939,  37,  98,  0,-77.777,+0, 9.710026668011723e+01, 0,_
 940,  37,  98,  1,-77.777,+0, 9.710068176563601e+01, 0.39,_
 941,  37,  99,  0, 2.5,   +1, 9.809513312115351e+01, 0,_
 942,  37,  100, 0, 3,     +1, 9.909173023009804e+01, 0,_
 943,  37,  101, 0, 1.5,   +1, 1.000867606443205e+02, 0,_
 944,  37,  102, 0,-77.777,+0, 1.010836387791989e+02, 0,_
 945,  38,  73,  0, 0.5,   -1, 7.233888961100834e+01, 0,_
 946,  38,  74,  0, 0,     +1, 7.332085099824940e+01, 0,_
 947,  38,  75,  0, 1.5,   -1, 7.430609669720592e+01, 0,_
 948,  38,  76,  0, 0,     +1, 7.528938656428107e+01, 0,_
 949,  38,  77,  0, 2.5,   +1, 7.627701157073838e+01, 0,_
 950,  38,  78,  0, 0,     +1, 7.726270514390259e+01, 0,_
 951,  38,  79,  0, 1.5,   -1, 7.825166358625019e+01, 0,_
 952,  38,  80,  0, 0,     +1, 7.923792756787311e+01, 0,_
 953,  38,  81,  0, 0.5,   -1, 8.022804216226061e+01, 0,_
 954,  38,  82,  0, 0,     +1, 8.121468146872537e+01, 0,_
 955,  38,  83,  0, 3.5,   +1, 8.220525277871064e+01, 0,_
 956,  38,  83,  1, 0.5,   -1, 8.220552859771627e+01, 0.25915,_
 957,  38,  84,  0, 0,     +1, 8.319256256849391e+01, 0,_
 958,  38,  85,  0, 4.5,   +1, 8.418348919568550e+01, 0,_
 959,  38,  85,  1, 0.5,   -1, 8.418374320673655e+01, 0.23866,_
 960,  38,  86,  0, 0,     +1, 8.517125893733456e+01, 0,_
 961,  38,  87,  0, 4.5,   +1, 8.616228851927195e+01, 0,_
 962,  38,  87,  1, 0.5,   -1, 8.616270204343058e+01, 0.388533,_
 963,  38,  88,  0, 0,     +1, 8.715046084868196e+01, 0,_
 964,  38,  89,  0, 2.5,   +1, 8.814369312699070e+01, 0,_
 965,  38,  90,  0, 0,     +1, 8.913538031796118e+01, 0,_
 966,  38,  91,  0, 2.5,   +1, 9.012923373476714e+01, 0,_
 967,  38,  92,  0, 0,     +1, 9.112147828526422e+01, 0,_
 968,  38,  93,  0, 2.5,   +1, 9.211584824030982e+01, 0,_
 969,  38,  94,  0, 0,     +1, 9.310857765549520e+01, 0,_
 970,  38,  95,  0, 0.5,   +1, 9.410395300884907e+01, 0,_
 971,  38,  96,  0, 0,     +1, 9.509769913432538e+01, 0,_
 972,  38,  97,  0, 0.5,   +1, 9.609373068875220e+01, 0,_
 973,  38,  98,  0, 0,     +1, 9.708743738785471e+01, 0,_
 974,  38,  99,  0, 1.5,   +1, 9.808299957629826e+01, 0,_
 975,  38,  100, 0, 0,     +1, 9.907728329556959e+01, 0,_
 976,  38,  101, 0, 2.5,   -1, 1.000734780032869e+02, 0,_
 977,  38,  102, 0, 0,     +1, 1.010682567432456e+02, 0,_
 978,  38,  103, 0,-77.777,+0, 1.020647078978720e+02, 0,_
 979,  38,  104, 0, 0,     +1, 1.030596468357295e+02, 0,_
 980,  38,  105, 0,-77.777,+0, 1.040569056718636e+02, 0,_
 981,  39,  76,  0,-77.777,+0, 7.530616839017242e+01, 0,_
 982,  39,  77,  0, 2.5,   +1, 7.628910730720017e+01, 0,_
 983,  39,  78,  0, 0,     +1, 7.727441371016160e+01, 0,_
 984,  39,  78,  1, 5,     +1, 7.727441371016160e+01, 0,_
 985,  39,  79,  0, 2.5,   +1, 7.825981528843694e+01, 0,_
 986,  39,  80,  0, 4,     -1, 7.924768027647407e+01, 0,_
 987,  39,  80,  1, 1,     -1, 7.924792347401520e+01, 0.2285,_
 988,  39,  81,  0, 2.5,   +1, 8.023423142184728e+01, 0,_
 989,  39,  82,  0, 1,     +1, 8.122313853000134e+01, 0,_
 990,  39,  83,  0, 4.5,   +1, 8.221014008164217e+01, 0,_
 991,  39,  83,  1, 1.5,   -1, 8.221020604831000e+01, 0.06198,_
 992,  39,  84,  0, 6,     +1, 8.319975221183954e+01, 0,_
 993,  39,  84,  1, 1,     +1, 8.319982352140302e+01, 0.067,_
 994,  39,  85,  0, 0.5,   -1, 8.418696011644926e+01, 0,_
 995,  39,  85,  1, 4.5,   +1, 8.418698119002175e+01, 0.0198,_
 996,  39,  86,  0, 4,     -1, 8.517683598438138e+01, 0,_
 997,  39,  86,  1, 8,     +1, 8.517706832583971e+01, 0.2183,_
 998,  39,  87,  0, 0.5,   -1, 8.616426995613830e+01, 0,_
 999,  39,  87,  1, 4.5,   +1, 8.616467527118256e+01, 0.38082,_
 1000, 39,  88,  0, 4,     -1, 8.715431646051995e+01, 0,_
 1001, 39,  88,  1, 8,     +1, 8.715503439881915e+01, 0.67455,_
 1002, 39,  89,  0, 0.5,   -1, 8.814209735125314e+01, 0,_
 1003, 39,  89,  1, 4.5,   +1, 8.814306477723557e+01, 0.90896,_
 1004, 39,  90,  0, 2,     -1, 8.913479926771267e+01, 0,_
 1005, 39,  90,  1, 7,     +1, 8.913552517778247e+01, 0.68204,_
 1006, 39,  91,  0, 0.5,   -1, 9.012636073705049e+01, 0,_
 1007, 39,  91,  1, 4.5,   +1, 9.012695204233681e+01, 0.55557,_
 1008, 39,  92,  0, 2,     -1, 9.111940378166329e+01, 0,_
 1009, 39,  93,  0, 0.5,   -1, 9.211144054363049e+01, 0,_
 1010, 39,  93,  1, 4.5,   +1, 9.211224808612134e+01, 0.758739,_
 1011, 39,  94,  0, 2,     -1, 9.310484640645404e+01, 0,_
 1012, 39,  95,  0, 0.5,   -1, 9.409747203692524e+01, 0,_
 1013, 39,  96,  0, 0,     -1, 9.509193930377226e+01, 0,_
 1014, 39,  96,  1, 8,     +1, 9.509364221872109e+01, 1.6,_
 1015, 39,  97,  0, 0.5,   -1, 9.608570573430693e+01, 0,_
 1016, 39,  97,  1, 4.5,   +1, 9.608641616913714e+01, 0.6675,_
 1017, 39,  97,  2, 13.5,  -1, 9.608945565945642e+01, 3.5233,_
 1018, 39,  98,  0, 0,     -1, 9.708118803794561e+01, 0,_
 1019, 39,  98,  1, 5,     +1, 9.708225235978863e+01, 1,_
 1020, 39,  99,  0, 2.5,   +1, 9.807434831574272e+01, 0,_
 1021, 39,  100, 0,-77.777,+0, 9.906929397039966e+01, 0,_
 1022, 39,  100, 1,-77.777,+0, 9.906950683476826e+01, 0.2,_
 1023, 39,  101, 0, 2.5,   +1, 1.000631156640838e+02, 0,_
 1024, 39,  102, 0,-77.777,+0, 1.010586630805255e+02, 0,_
 1025, 39,  102, 1,-77.777,+0, 1.010588759448941e+02, 0.2,_
 1026, 39,  103, 0, 2.5,   +1, 1.020529628755022e+02, 0,_
 1027, 39,  104, 0, 0,     -1, 1.030490486679610e+02, 0,_
 1028, 39,  105, 0, 2.5,   +1, 1.040434314250778e+02, 0,_
 1029, 39,  106, 0,-77.777,+0, 1.050403938297654e+02, 0,_
 1030, 39,  107, 0, 2.5,   +1, 1.060357293314308e+02, 0,_
 1031, 39,  108, 0,-77.777,+0, 1.070325321191858e+02, 0,_
 1032, 40,  78,  0, 0,     +1, 7.728646528428631e+01, 0,_
 1033, 40,  79,  0, 2.5,   +1, 7.827157340534373e+01, 0,_
 1034, 40,  80,  0, 0,     +1, 7.925490492054141e+01, 0,_
 1035, 40,  81,  0, 1.5,   -1, 8.024301502962530e+01, 0,_
 1036, 40,  82,  0, 0,     +1, 8.122785646152187e+01, 0,_
 1037, 40,  83,  0, 0.5,   -1, 8.221683893662762e+01, 0,_
 1038, 40,  84,  0, 0,     +1, 8.320238400854265e+01, 0,_
 1039, 40,  85,  0, 3.5,   +1, 8.419192723577841e+01, 0,_
 1040, 40,  85,  1, 0.5,   -1, 8.419223823062093e+01, 0.2922,_
 1041, 40,  86,  0, 0,     +1, 8.517823458264516e+01, 0,_
 1042, 40,  87,  0, 4.5,   +1, 8.616817733602053e+01, 0,_
 1043, 40,  87,  1, 0.5,   -1, 8.616853526745633e+01, 0.3363,_
 1044, 40,  88,  0, 0,     +1, 8.715502971324513e+01, 0,_
 1045, 40,  89,  0, 4.5,   +1, 8.814511235351237e+01, 0,_
 1046, 40,  89,  1, 0.5,   -1, 8.814573796189170e+01, 0.5878,_
 1047, 40,  90,  0, 0,     +1, 8.913237423838763e+01, 0,_
 1048, 40,  90,  1, 5,     -1, 8.913484240074158e+01, 2.319,_
 1049, 40,  91,  0, 2.5,   +1, 9.012471713571769e+01, 0,_
 1050, 40,  92,  0, 0,     +1, 9.111552695107400e+01, 0,_
 1051, 40,  93,  0, 2.5,   +1, 9.210835946491818e+01, 0,_
 1052, 40,  94,  0, 0,     +1, 9.309961222122841e+01, 0,_
 1053, 40,  95,  0, 2.5,   +1, 9.409273464205302e+01, 0,_
 1054, 40,  96,  0, 0,     +1, 9.508437947703455e+01, 0,_
 1055, 40,  97,  0, 0.5,   +1, 9.607844574153317e+01, 0,_
 1056, 40,  98,  0, 0,     +1, 9.707161772787671e+01, 0,_
 1057, 40,  99,  0, 0.5,   +1, 9.806692914919631e+01, 0,_
 1058, 40,  100, 0, 0,     +1, 9.905966181323069e+01, 0,_
 1059, 40,  101, 0, 1.5,   +1, 1.000544893820911e+02, 0,_
 1060, 40,  102, 0, 0,     +1, 1.010475786680267e+02, 0,_
 1061, 40,  103, 0, 2.5,   -1, 1.020430032071897e+02, 0,_
 1062, 40,  104, 0, 0,     +1, 1.030366385158841e+02, 0,_
 1063, 40,  105, 0, 2.5,   +1, 1.040325813302362e+02, 0,_
 1064, 40,  106, 0, 0,     +1, 1.050270930796456e+02, 0,_
 1065, 40,  107, 0, 2.5,   +1, 1.060229411400328e+02, 0,_
 1066, 40,  108, 0, 0,     +1, 1.070175717495295e+02, 0,_
 1067, 40,  109, 0,-77.777,+0, 1.080144736782365e+02, 0,_
 1068, 40,  110, 0, 0,     +1, 1.090094027019987e+02, 0,_
 1069, 41,  81,  0, 1.5,   -1, 8.025482866546523e+01, 0,_
 1070, 41,  82,  0, 0,     +1, 8.124014002547425e+01, 0,_
 1071, 41,  83,  0, 2.5,   +1, 8.222573195437742e+01, 0,_
 1072, 41,  84,  0, 3,     +1, 8.321324325211127e+01, 0,_
 1073, 41,  85,  0, 4.5,   +1, 8.419926628255588e+01, 0,_
 1074, 41,  85,  1, 0.5,   -1, 8.420007410283473e+01, 0.759,_
 1075, 41,  86,  0, 6,     +1, 8.518763782434738e+01, 0,_
 1076, 41,  86,  1,-77.777,+0, 8.518790390480812e+01, 0.25,_
 1077, 41,  87,  0, 0.5,   -1, 8.617400199880781e+01, 0,_
 1078, 41,  87,  1, 4.5,   +1, 8.617400608580368e+01, 0.00384,_
 1079, 41,  88,  0, 8,     +1, 8.716296453666813e+01, 0,_
 1080, 41,  88,  1, 4,     -1, 8.716337962218691e+01, 0.39,_
 1081, 41,  89,  0, 4.5,   +1, 8.814963609566740e+01, 0,_
 1082, 41,  89,  1, 0.5,   -1, 8.814963609566740e+01, 0,_
 1083, 41,  90,  0, 8,     +1, 8.913887832700726e+01, 0,_
 1084, 41,  90,  1, 4,     -1, 8.913901104794108e+01, 0.1247,_
 1085, 41,  90,  2, 1,     +1, 8.913928489795130e+01, 0.382,_
 1086, 41,  91,  0, 4.5,   +1, 9.012605558913152e+01, 0,_
 1087, 41,  91,  1, 0.5,   -1, 9.012616680012090e+01, 0.10449,_
 1088, 41,  92,  0, 7,     +1, 9.111766169965722e+01, 0,_
 1089, 41,  92,  1, 2,     +1, 9.111780591526694e+01, 0.1355,_
 1090, 41,  93,  0, 4.5,   +1, 9.210826281736112e+01, 0,_
 1091, 41,  93,  1, 0.5,   -1, 9.210829561976033e+01, 0.03082,_
 1092, 41,  94,  0, 6,     +1, 9.310057038680435e+01, 0,_
 1093, 41,  94,  1, 3,     +1, 9.310061397078383e+01, 0.04095,_
 1094, 41,  95,  0, 4.5,   +1, 9.409153587724913e+01, 0,_
 1095, 41,  95,  1, 0.5,   -1, 9.409178672726431e+01, 0.23569,_
 1096, 41,  96,  0, 6,     +1, 9.508420495921676e+01, 0,_
 1097, 41,  97,  0, 4.5,   +1, 9.607561132947505e+01, 0,_
 1098, 41,  97,  1, 0.5,   -1, 9.607640254633314e+01, 0.7434,_
 1099, 41,  98,  0, 1,     +1, 9.706923589228165e+01, 0,_
 1100, 41,  98,  1, 5,     +1, 9.706932529531646e+01, 0.084,_
 1101, 41,  99,  0, 4.5,   +1, 9.806191116560173e+01, 0,_
 1102, 41,  99,  1, 0.5,   -1, 9.806229995172777e+01, 0.36529,_
 1103, 41,  100, 0, 1,     +1, 9.905602187201491e+01, 0,_
 1104, 41,  100, 1, 4,     +1, 9.905653274649956e+01, 0.48,_
 1105, 41,  101, 0, 2.5,   +1, 1.000483955704559e+02, 0,_
 1106, 41,  102, 0, 1,     +1, 1.010425584339325e+02, 0,_
 1107, 41,  102, 1,-77.777,+0, 1.010426967957721e+02, 0.13,_
 1108, 41,  103, 0, 2.5,   +1, 1.020353258934668e+02, 0,_
 1109, 41,  104, 0, 1,     +1, 1.030301515251131e+02, 0,_
 1110, 41,  104, 1,-77.777,+0, 1.030303856759185e+02, 0.22,_
 1111, 41,  105, 0, 2.5,   +1, 1.040235869398716e+02, 0,_
 1112, 41,  106, 0, 2,     +1, 1.050189474290225e+02, 0,_
 1113, 41,  107, 0, 2.5,   +1, 1.060129959859557e+02, 0,_
 1114, 41,  108, 0, 2,     +1, 1.070088522909045e+02, 0,_
 1115, 41,  109, 0, 2.5,   +1, 1.080033014843560e+02, 0,_
 1116, 41,  110, 0, 2,     +1, 1.089993726118852e+02, 0,_
 1117, 41,  111, 0, 2.5,   +1, 1.099944374587517e+02, 0,_
 1118, 41,  112, 0, 2,     +1, 1.109907445417466e+02, 0,_
 1119, 41,  113, 0, 2.5,   +1, 1.119861593561736e+02, 0,_
 1120, 42,  83,  0, 1.5,   -1, 8.223766951640735e+01, 0,_
 1121, 42,  84,  0, 0,     +1, 8.322074524794886e+01, 0,_
 1122, 42,  85,  0, 0.5,   -1, 8.420860030404540e+01, 0,_
 1123, 42,  86,  0, 0,     +1, 8.519298477546604e+01, 0,_
 1124, 42,  87,  0, 3.5,   +1, 8.618144126717367e+01, 0,_
 1125, 42,  88,  0, 0,     +1, 8.716667587225767e+01, 0,_
 1126, 42,  89,  0, 4.5,   +1, 8.815560723425421e+01, 0,_
 1127, 42,  89,  1, 0.5,   -1, 8.815601965896838e+01, 0.3875,_
 1128, 42,  90,  0, 0,     +1, 8.914152744066045e+01, 0,_
 1129, 42,  91,  0, 4.5,   +1, 9.013076966307761e+01, 0,_
 1130, 42,  91,  1, 0.5,   -1, 9.013146455880891e+01, 0.6529,_
 1131, 42,  92,  0, 0,     +1, 9.111728356317798e+01, 0,_
 1132, 42,  93,  0, 2.5,   +1, 9.210869468724493e+01, 0,_
 1133, 42,  93,  1, 10.5,  +1, 9.211127559321172e+01, 2.42493,_
 1134, 42,  94,  0, 0,     +1, 9.309839384634421e+01, 0,_
 1135, 42,  95,  0, 2.5,   +1, 9.409055074037720e+01, 0,_
 1136, 42,  96,  0, 0,     +1, 9.508080758021977e+01, 0,_
 1137, 42,  97,  0, 2.5,   +1, 9.607354769965373e+01, 0,_
 1138, 42,  98,  0, 0,     +1, 9.706434919311853e+01, 0,_
 1139, 42,  99,  0, 0.5,   +1, 9.805804261328194e+01, 0,_
 1140, 42,  100, 0, 0,     +1, 9.904921486713914e+01, 0,_
 1141, 42,  101, 0, 0.5,   +1, 1.000434694003056e+02, 0,_
 1142, 42,  102, 0, 0,     +1, 1.010348298425265e+02, 0,_
 1143, 42,  103, 0, 1.5,   +1, 1.020290123369030e+02, 0,_
 1144, 42,  104, 0, 0,     +1, 1.030210718408132e+02, 0,_
 1145, 42,  105, 0, 2.5,   -1, 1.040156879787051e+02, 0,_
 1146, 42,  106, 0, 0,     +1, 1.050083774668549e+02, 0,_
 1147, 42,  107, 0, 3.5,   -1, 1.060036004177632e+02, 0,_
 1148, 42,  108, 0, 0,     +1, 1.069969210530759e+02, 0,_
 1149, 42,  109, 0, 2.5,   +1, 1.079926835934865e+02, 0,_
 1150, 42,  110, 0, 0,     +1, 1.089863531048180e+02, 0,_
 1151, 42,  111, 0, 2.5,   +1, 1.099826615123361e+02, 0,_
 1152, 42,  112, 0, 0,     +1, 1.109767061829439e+02, 0,_
 1153, 42,  113, 0,-77.777,+0, 1.119734098297469e+02, 0,_
 1154, 42,  114, 0, 0,     +1, 1.129676746091307e+02, 0,_
 1155, 42,  115, 0,-77.777,+0, 1.139644674827905e+02, 0,_
 1156, 43,  85,  0, 0.5,   -1, 8.422101003774753e+01, 0,_
 1157, 43,  86,  0, 0,     +1, 8.520633131185177e+01, 0,_
 1158, 43,  87,  0, 0.5,   -1, 8.619122745666552e+01, 0,_
 1159, 43,  87,  1, 4.5,   +1, 8.619124874310238e+01, 0.02,_
 1160, 43,  88,  0,-77.777,+0, 8.717838897917225e+01, 0,_
 1161, 43,  88,  1,-77.777,+0, 8.717838897917225e+01, 0,_
 1162, 43,  89,  0, 4.5,   +1, 8.816371745983228e+01, 0,_
 1163, 43,  89,  1, 0.5,   -1, 8.816378408637966e+01, 0.0626,_
 1164, 43,  90,  0, 1,     +1, 8.915158295943672e+01, 0,_
 1165, 43,  90,  1, 8,     +1, 8.915191289920806e+01, 0.31,_
 1166, 43,  91,  0, 4.5,   +1, 9.013739206056088e+01, 0,_
 1167, 43,  91,  1, 0.5,   -1, 9.013754032059362e+01, 0.1393,_
 1168, 43,  92,  0, 8,     +1, 9.112567348917548e+01, 0,_
 1169, 43,  93,  0, 4.5,   +1, 9.211210154312552e+01, 0,_
 1170, 43,  93,  1, 0.5,   -1, 9.211251857635327e+01, 0.39183,_
 1171, 43,  94,  0, 7,     +1, 9.310292333173460e+01, 0,_
 1172, 43,  94,  1, 2,     +1, 9.310300422019468e+01, 0.076,_
 1173, 43,  95,  0, 4.5,   +1, 9.409234999498746e+01, 0,_
 1174, 43,  95,  1, 0.5,   -1, 9.409239139710715e+01, 0.0389,_
 1175, 43,  96,  0, 7,     +1, 9.508397206720642e+01, 0,_
 1176, 43,  96,  1, 4,     +1, 9.508400855215919e+01, 0.03428,_
 1177, 43,  97,  0, 4.5,   +1, 9.607388856607490e+01, 0,_
 1178, 43,  97,  1, 0.5,   -1, 9.607399137956493e+01, 0.0966,_
 1179, 43,  98,  0, 6,     +1, 9.706614126199260e+01, 0,_
 1180, 43,  99,  0, 4.5,   +1, 9.805659751493744e+01, 0,_
 1181, 43,  99,  1, 0.5,   -1, 9.805674937557096e+01, 0.142683,_
 1182, 43,  100, 0, 1,     +1, 9.904939801517681e+01, 0,_
 1183, 43,  101, 0, 4.5,   +1, 1.000404630689140e+02, 0,_
 1184, 43,  102, 0, 1,     +1, 1.010337582646585e+02, 0,_
 1185, 43,  102, 1,-77.777,+0, 1.010337795510954e+02, 0.02,_
 1186, 43,  103, 0, 2.5,   +1, 1.020251348034043e+02, 0,_
 1187, 43,  104, 0, 3,     +1, 1.030187798497230e+02, 0,_
 1188, 43,  105, 0, 1.5,   -1, 1.040104164490657e+02, 0,_
 1189, 43,  106, 0,-77.777,+0, 1.050045015305170e+02, 0,_
 1190, 43,  107, 0, 1.5,   -1, 1.059970033735955e+02, 0,_
 1191, 43,  108, 0, 2,     +1, 1.069914218769739e+02, 0,_
 1192, 43,  109, 0, 1.5,   -1, 1.079845768874122e+02, 0,_
 1193, 43,  110, 0, 2,     +1, 1.089794436070346e+02, 0,_
 1194, 43,  111, 0, 1.5,   -1, 1.099729922952878e+02, 0,_
 1195, 43,  112, 0, 2,     +1, 1.109684097151390e+02, 0,_
 1196, 43,  113, 0, 2.5,   +1, 1.119624240535726e+02, 0,_
 1197, 43,  114, 0, 2,     +1, 1.129583157032622e+02, 0,_
 1198, 43,  115, 0, 2.5,   +1, 1.139521521937335e+02, 0,_
 1199, 43,  116, 0, 2,     +1, 1.149487388542131e+02, 0,_
 1200, 43,  117, 0, 1.5,   -1, 1.159434200255953e+02, 0,_
 1201, 43,  118, 0, 2,     +1, 1.169397171944951e+02, 0,_
 1202, 44,  87,  0, 0.5,   -1, 8.620418003665030e+01, 0,_
 1203, 44,  88,  0, 0,     +1, 8.718620289128160e+01, 0,_
 1204, 44,  89,  0, 3.5,   +1, 8.817343956857840e+01, 0,_
 1205, 44,  90,  0, 0,     +1, 8.915779955119248e+01, 0,_
 1206, 44,  91,  0, 4.5,   +1, 9.014563717434403e+01, 0,_
 1207, 44,  91,  1, 0.5,   -1, 9.014606290308123e+01, 0.4,_
 1208, 44,  92,  0, 0,     +1, 9.113059543691267e+01, 0,_
 1209, 44,  93,  0, 4.5,   +1, 9.211890191348878e+01, 0,_
 1210, 44,  93,  1, 0.5,   -1, 9.211968355145029e+01, 0.7344,_
 1211, 44,  94,  0, 0,     +1, 9.310459934720167e+01, 0,_
 1212, 44,  95,  0, 2.5,   +1, 9.409507848584387e+01, 0,_
 1213, 44,  96,  0, 0,     +1, 9.508369668635829e+01, 0,_
 1214, 44,  97,  0, 2.5,   +1, 9.607506344187499e+01, 0,_
 1215, 44,  98,  0, 0,     +1, 9.706423330230270e+01, 0,_
 1216, 44,  99,  0, 2.5,   +1, 9.805628085873676e+01, 0,_
 1217, 44,  100, 0, 0,     +1, 9.904598532683403e+01, 0,_
 1218, 44,  101, 0, 2.5,   +1, 1.000387457642147e+02, 0,_
 1219, 44,  102, 0, 0,     +1, 1.010289330993101e+02, 0,_
 1220, 44,  103, 0, 1.5,   +1, 1.020223001900901e+02, 0,_
 1221, 44,  103, 1, 5.5,   -1, 1.020225534986887e+02, 0.238,_
 1222, 44,  104, 0, 0,     +1, 1.030128278780565e+02, 0,_
 1223, 44,  105, 0, 1.5,   +1, 1.040065376316917e+02, 0,_
 1224, 44,  106, 0, 0,     +1, 1.049975334146544e+02, 0,_
 1225, 44,  107, 0, 2.5,   +1, 1.059915619233042e+02, 0,_
 1226, 44,  108, 0, 0,     +1, 1.069831855441049e+02, 0,_
 1227, 44,  109, 0, 2.5,   +1, 1.079777060228751e+02, 0,_
 1228, 44,  110, 0, 0,     +1, 1.089698241944350e+02, 0,_
 1229, 44,  111, 0, 2.5,   +1, 1.099647324660132e+02, 0,_
 1230, 44,  112, 0, 0,     +1, 1.109573706953165e+02, 0,_
 1231, 44,  113, 0, 2.5,   +1, 1.119527849386917e+02, 0,_
 1232, 44,  113, 1, 5.5,   -1, 1.119529233005313e+02, 0.13,_
 1233, 44,  114, 0, 0,     +1, 1.129459466599810e+02, 0,_
 1234, 44,  115, 0, 2.5,   +1, 1.139416476080838e+02, 0,_
 1235, 44,  116, 0, 0,     +1, 1.149353143692452e+02, 0,_
 1236, 44,  117, 0, 2.5,   +1, 1.159315974670697e+02, 0,_
 1237, 44,  118, 0, 0,     +1, 1.169253804214269e+02, 0,_
 1238, 44,  119, 0, 2.5,   +1, 1.179217876367833e+02, 0,_
 1239, 44,  120, 0, 0,     +1, 1.189159136188344e+02, 0,_
 1240, 45,  89,  0, 3.5,   +1, 8.818663721210820e+01, 0,_
 1241, 45,  90,  0, 0,     +1, 8.917183158579387e+01, 0,_
 1242, 45,  90,  1, 9,     +1, 8.917183158579387e+01, 0,_
 1243, 45,  91,  0, 3.5,   +1, 9.015592946055047e+01, 0,_
 1244, 45,  92,  0, 6,     +1, 9.114262452487806e+01, 0,_
 1245, 45,  93,  0, 4.5,   +1, 9.212763458264564e+01, 0,_
 1246, 45,  94,  0, 4,     +1, 9.311489770281716e+01, 0,_
 1247, 45,  94,  1, 8,     +1, 9.311521699937006e+01, 0.3,_
 1248, 45,  95,  0, 4.5,   +1, 9.410052476925657e+01, 0,_
 1249, 45,  95,  1, 0.5,   -1, 9.410110301531388e+01, 0.5433,_
 1250, 45,  96,  0, 6,     +1, 9.509050052764627e+01, 0,_
 1251, 45,  96,  1, 3,     +1, 9.509055587238211e+01, 0.052,_
 1252, 45,  97,  0, 4.5,   +1, 9.607881304884623e+01, 0,_
 1253, 45,  97,  1, 0.5,   -1, 9.607908845276633e+01, 0.25876,_
 1254, 45,  98,  0, 2,     +1, 9.706960776007850e+01, 0,_
 1255, 45,  98,  1, 5,     +1, 9.706967161938908e+01, 0.06,_
 1256, 45,  99,  0, 0.5,   -1, 9.805845641968429e+01, 0,_
 1257, 45,  99,  1, 4.5,   +1, 9.805852485557880e+01, 0.0643,_
 1258, 45,  100, 0, 1,     -1, 9.904985548127161e+01, 0,_
 1259, 45,  100, 1, 5,     +1, 9.904997000230192e+01, 0.1076,_
 1260, 45,  101, 0, 0.5,   -1, 1.000393265616538e+02, 0,_
 1261, 45,  101, 1, 4.5,   +1, 1.000394940007661e+02, 0.15732,_
 1262, 45,  102, 0, 6,     +1, 1.010314056449106e+02, 0,_
 1263, 45,  102, 1, 2,     -1, 1.010314801474396e+02, 0.07,_
 1264, 45,  103, 0, 0.5,   -1, 1.020214864758554e+02, 0,_
 1265, 45,  103, 1, 3.5,   +1, 1.020215287890346e+02, 0.039756,_
 1266, 45,  104, 0, 1,     +1, 1.030140373332292e+02, 0,_
 1267, 45,  104, 1, 5,     +1, 1.030141745839168e+02, 0.128956,_
 1268, 45,  105, 0, 3.5,   +1, 1.040044975894857e+02, 0,_
 1269, 45,  105, 1, 0.5,   -1, 1.040046354936669e+02, 0.12957,_
 1270, 45,  106, 0, 1,     +1, 1.049974914760489e+02, 0,_
 1271, 45,  106, 1, 6,     +1, 1.049976372881413e+02, 0.137,_
 1272, 45,  107, 0, 3.5,   +1, 1.059883676900664e+02, 0,_
 1273, 45,  108, 0, 1,     +1, 1.069817270290154e+02, 0,_
 1274, 45,  108, 1, 5,     +1, 1.069817270290154e+02, 0,_
 1275, 45,  109, 0, 3.5,   +1, 1.079731708894247e+02, 0,_
 1276, 45,  110, 0, 4,     +1, 1.089668908059989e+02, 0,_
 1277, 45,  110, 1, 1,     +1, 1.089669972381832e+02, 0.1,_
 1278, 45,  111, 0, 3.5,   +1, 1.099588582803376e+02, 0,_
 1279, 45,  112, 0, 1,     +1, 1.109530062459033e+02, 0,_
 1280, 45,  112, 1,-77.777,+0, 1.109533894017668e+02, 0.36,_
 1281, 45,  113, 0, 3.5,   +1, 1.119454417379083e+02, 0,_
 1282, 45,  114, 0, 1,     +1, 1.129401047952472e+02, 0,_
 1283, 45,  114, 1,-77.777,+0, 1.129403176596158e+02, 0.2,_
 1284, 45,  115, 0, 3.5,   +1, 1.139330903559532e+02, 0,_
 1285, 45,  116, 0, 1,     +1, 1.149282183080184e+02, 0,_
 1286, 45,  116, 1, 6,     -1, 1.149283779562949e+02, 0.15,_
 1287, 45,  117, 0, 3.5,   +1, 1.159215848485662e+02, 0,_
 1288, 45,  118, 0, 1,     +1, 1.169172622080623e+02, 0,_
 1289, 45,  119, 0, 3.5,   +1, 1.179108691961523e+02, 0,_
 1290, 45,  120, 0, 3.5,   +1, 1.189065447988707e+02, 0,_
 1291, 45,  121, 0, 3.5,   +1, 1.199006836692455e+02, 0,_
 1292, 45,  122, 0,-77.777,+0, 1.208965317296328e+02, 0,_
 1293, 46,  91,  0, 3.5,   +1, 9.016938189632690e+01, 0,_
 1294, 46,  92,  0, 0,     +1, 9.115158518749082e+01, 0,_
 1295, 46,  93,  0, 4.5,   +1, 9.213828947787542e+01, 0,_
 1296, 46,  94,  0, 0,     +1, 9.312214078115312e+01, 0,_
 1297, 46,  95,  0, 4.5,   +1, 9.410943815254073e+01, 0,_
 1298, 46,  95,  1, 10.5,  +1, 9.411142843438716e+01, 1.87,_
 1299, 46,  96,  0, 0,     +1, 9.509423024396831e+01, 0,_
 1300, 46,  97,  0, 2.5,   +1, 9.608391296946338e+01, 0,_
 1301, 46,  98,  0, 0,     +1, 9.707158125689470e+01, 0,_
 1302, 46,  99,  0, 2.5,   +1, 9.806207367645884e+01, 0,_
 1303, 46,  100, 0, 0,     +1, 9.905025816604197e+01, 0,_
 1304, 46,  101, 0, 2.5,   +1, 1.000414342219547e+02, 0,_
 1305, 46,  102, 0, 0,     +1, 1.010302137604887e+02, 0,_
 1306, 46,  103, 0, 2.5,   +1, 1.020220979494879e+02, 0,_
 1307, 46,  104, 0, 0,     +1, 1.030114449024239e+02, 0,_
 1308, 46,  105, 0, 2.5,   +1, 1.040038944962379e+02, 0,_
 1309, 46,  106, 0, 0,     +1, 1.049937185600484e+02, 0,_
 1310, 46,  107, 0, 2.5,   +1, 1.059867616958709e+02, 0,_
 1311, 46,  107, 1, 5.5,   -1, 1.059869904186349e+02, 0.2149,_
 1312, 46,  108, 0, 0,     +1, 1.069769455768955e+02, 0,_
 1313, 46,  109, 0, 2.5,   +1, 1.079703961800478e+02, 0,_
 1314, 46,  109, 1, 5.5,   -1, 1.079705972304440e+02, 0.1889,_
 1315, 46,  110, 0, 0,     +1, 1.089610346748820e+02, 0,_
 1316, 46,  111, 0, 2.5,   +1, 1.099549400474953e+02, 0,_
 1317, 46,  111, 1, 5.5,   -1, 1.099551233024302e+02, 0.17218,_
 1318, 46,  112, 0, 0,     +1, 1.109459923021356e+02, 0,_
 1319, 46,  113, 0, 2.5,   +1, 1.119403079219911e+02, 0,_
 1320, 46,  113, 1, 4.5,   -1, 1.119403942384925e+02, 0.0811,_
 1321, 46,  114, 0, 0,     +1, 1.129318240313691e+02, 0,_
 1322, 46,  115, 0, 2.5,   +1, 1.139264952271888e+02, 0,_
 1323, 46,  115, 1, 5.5,   -1, 1.139265901434107e+02, 0.08918,_
 1324, 46,  116, 0, 0,     +1, 1.149185377542023e+02, 0,_
 1325, 46,  117, 0, 2.5,   +1, 1.159135735864778e+02, 0,_
 1326, 46,  117, 1, 5.5,   -1, 1.159137898566763e+02, 0.2032,_
 1327, 46,  118, 0, 0,     +1, 1.169060854576636e+02, 0,_
 1328, 46,  119, 0,-77.777,+0, 1.179017318772514e+02, 0,_
 1329, 46,  120, 0, 0,     +1, 1.188943417948731e+02, 0,_
 1330, 46,  121, 0, 2.5,   +1, 1.198901126095876e+02, 0,_
 1331, 46,  122, 0, 0,     +1, 1.208831890369417e+02, 0,_
 1332, 46,  123, 0,-77.777,+0, 1.218790542546621e+02, 0,_
 1333, 46,  124, 0, 0,     +1, 1.228726349614772e+02, 0,_
 1334, 47,  93,  0, 4.5,   +1, 9.215184204601334e+01, 0,_
 1335, 47,  94,  0, 0,     +1, 9.313671421160504e+01, 0,_
 1336, 47,  94,  1, 7,     +1, 9.313815104609311e+01, 1.35,_
 1337, 47,  94,  2, 21,    +1, 9.314363230358462e+01, 6.5,_
 1338, 47,  95,  0, 4.5,   +1, 9.412047401571532e+01, 0,_
 1339, 47,  95,  1, 0.5,   -1, 9.412084035529368e+01, 0.3442,_
 1340, 47,  95,  2, 11.5,  +1, 9.412316781429999e+01, 2.531,_
 1341, 47,  95,  3, 18.5,  +1, 9.412564555555052e+01, 4.859,_
 1342, 47,  96,  0,-77.777,+0, 9.510665276586184e+01, 0,_
 1343, 47,  96,  1, 2,     +1, 9.510665276586184e+01, 0,_
 1344, 47,  97,  0, 4.5,   +1, 9.609134193708432e+01, 0,_
 1345, 47,  98,  0, 6,     +1, 9.708036676619618e+01, 0,_
 1346, 47,  99,  0, 4.5,   +1, 9.806789570705384e+01, 0,_
 1347, 47,  99,  1, 0.5,   -1, 9.806843436033859e+01, 0.5061,_
 1348, 47,  100, 0, 5,     +1, 9.905778805022936e+01, 0,_
 1349, 47,  100, 1, 2,     +1, 9.905780456850437e+01, 0.01552,_
 1350, 47,  101, 0, 4.5,   +1, 1.000457955563592e+02, 0,_
 1351, 47,  101, 1, 0.5,   -1, 1.000460872869764e+02, 0.2741,_
 1352, 47,  102, 0, 5,     +1, 1.010362340769534e+02, 0,_
 1353, 47,  102, 1, 2,     +1, 1.010362439751465e+02, 0.0093,_
 1354, 47,  103, 0, 3.5,   +1, 1.020249231890251e+02, 0,_
 1355, 47,  103, 1, 0.5,   -1, 1.020250662870969e+02, 0.13445,_
 1356, 47,  104, 0, 5,     +1, 1.030159987675658e+02, 0,_
 1357, 47,  104, 1, 2,     +1, 1.030160061113865e+02, 0.0069,_
 1358, 47,  105, 0, 0.5,   -1, 1.040053281933729e+02, 0,_
 1359, 47,  105, 1, 3.5,   +1, 1.040053552963287e+02, 0.025465,_
 1360, 47,  106, 0, 1,     +1, 1.049968744287121e+02, 0,_
 1361, 47,  106, 1, 6,     +1, 1.049969698558085e+02, 0.08966,_
 1362, 47,  107, 0, 0.5,   -1, 1.059867254767069e+02, 0,_
 1363, 47,  107, 1, 3.5,   +1, 1.059868245906142e+02, 0.093124,_
 1364, 47,  108, 0, 1,     +1, 1.069789863547274e+02, 0,_
 1365, 47,  108, 1, 6,     +1, 1.069791028341099e+02, 0.10944,_
 1366, 47,  109, 0, 0.5,   -1, 1.079692116429620e+02, 0,_
 1367, 47,  109, 1, 3.5,   +1, 1.079693053394711e+02, 0.088034,_
 1368, 47,  110, 0, 1,     +1, 1.089619644692917e+02, 0,_
 1369, 47,  110, 1, 6,     +1, 1.089620896228973e+02, 0.11759,_
 1370, 47,  111, 0, 0.5,   -1, 1.099525670780756e+02, 0,_
 1371, 47,  111, 1, 3.5,   +1, 1.099526307458082e+02, 0.05982,_
 1372, 47,  112, 0, 2,     -1, 1.109457132838060e+02, 0,_
 1373, 47,  113, 0, 0.5,   -1, 1.119366511981949e+02, 0,_
 1374, 47,  113, 1, 3.5,   +1, 1.119366971768986e+02, 0.0432,_
 1375, 47,  114, 0, 1,     +1, 1.129302915610950e+02, 0,_
 1376, 47,  114, 1, 4,     +1, 1.129305033611418e+02, 0.199,_
 1377, 47,  115, 0, 0.5,   -1, 1.139216458912762e+02, 0,_
 1378, 47,  115, 1, 3.5,   +1, 1.139216896349040e+02, 0.0411,_
 1379, 47,  116, 0, 0,     -1, 1.149156523579182e+02, 0,_
 1380, 47,  116, 1, 5,     +1, 1.149157395258772e+02, 0.0819,_
 1381, 47,  117, 0, 0.5,   -1, 1.159074457139772e+02, 0,_
 1382, 47,  117, 1, 3.5,   +1, 1.159074761535819e+02, 0.0286,_
 1383, 47,  118, 0, 1,     -1, 1.169016525087924e+02, 0,_
 1384, 47,  118, 1, 4,     +1, 1.169017881991842e+02, 0.12749,_
 1385, 47,  119, 0, 0.5,   -1, 1.178940284607073e+02, 0,_
 1386, 47,  119, 1, 3.5,   +1, 1.178940497471441e+02, 0.02,_
 1387, 47,  120, 0, 3,     +1, 1.188886248407989e+02, 0,_
 1388, 47,  120, 1, 6,     -1, 1.188888408981330e+02, 0.203,_
 1389, 47,  121, 0, 3.5,   +1, 1.198813633600979e+02, 0,_
 1390, 47,  122, 0, 3,     +1, 1.208762816429294e+02, 0,_
 1391, 47,  122, 1, 8,     -1, 1.208763667886768e+02, 0.08,_
 1392, 47,  123, 0, 3.5,   +1, 1.218693494083386e+02, 0,_
 1393, 47,  124, 0, 3,     +1, 1.228643222196851e+02, 0,_
 1394, 47,  124, 1, 8,     -1, 1.228643222196851e+02, 0,_
 1395, 47,  125, 0, 3.5,   +1, 1.238575200153927e+02, 0,_
 1396, 47,  126, 0, 3,     +1, 1.248530161254003e+02, 0,_
 1397, 47,  127, 0, 3.5,   +1, 1.258468099852622e+02, 0,_
 1398, 47,  128, 0,-77.777,+0, 1.268422852756699e+02, 0,_
 1399, 47,  129, 0, 3.5,   +1, 1.278365044502158e+02, 0,_
 1400, 47,  130, 0,-77.777,+0, 1.288346010273943e+02, 0,_
 1401, 48,  95,  0, 4.5,   +1, 9.413427443623326e+01, 0,_
 1402, 48,  96,  0, 0,     +1, 9.511616642480793e+01, 0,_
 1403, 48,  97,  0, 4.5,   +1, 9.610238095888965e+01, 0,_
 1404, 48,  98,  0, 0,     +1, 9.708614603432436e+01, 0,_
 1405, 48,  99,  0, 2.5,   +1, 9.807511324668053e+01, 0,_
 1406, 48,  100, 0, 0,     +1, 9.906198505850594e+01, 0,_
 1407, 48,  101, 0, 2.5,   +1, 1.000516471111297e+02, 0,_
 1408, 48,  102, 0, 0,     +1, 1.010389874779654e+02, 0,_
 1409, 48,  103, 0, 2.5,   +1, 1.020293412697278e+02, 0,_
 1410, 48,  104, 0, 0,     +1, 1.030172206847562e+02, 0,_
 1411, 48,  105, 0, 2.5,   +1, 1.040082412400685e+02, 0,_
 1412, 48,  106, 0, 0,     +1, 1.049966724686788e+02, 0,_
 1413, 48,  107, 0, 2.5,   +1, 1.059882329912205e+02, 0,_
 1414, 48,  108, 0, 0,     +1, 1.069772348523480e+02, 0,_
 1415, 48,  109, 0, 2.5,   +1, 1.079694405842054e+02, 0,_
 1416, 48,  110, 0, 0,     +1, 1.089588878687762e+02, 0,_
 1417, 48,  111, 0, 0.5,   +1, 1.099514635897094e+02, 0,_
 1418, 48,  111, 1, 5.5,   -1, 1.099518852953100e+02, 0.39622,_
 1419, 48,  112, 0, 0,     +1, 1.109414654241523e+02, 0,_
 1420, 48,  113, 0, 0.5,   +1, 1.119345050335311e+02, 0,_
 1421, 48,  113, 1, 5.5,   -1, 1.119347856952011e+02, 0.2637,_
 1422, 48,  114, 0, 0,     +1, 1.129248804072873e+02, 0,_
 1423, 48,  115, 0, 0.5,   +1, 1.139183445511109e+02, 0,_
 1424, 48,  115, 1, 5.5,   -1, 1.139185371933644e+02, 0.181,_
 1425, 48,  116, 0, 0,     +1, 1.149090856756672e+02, 0,_
 1426, 48,  117, 0, 0.5,   +1, 1.159029368469306e+02, 0,_
 1427, 48,  117, 1, 5.5,   -1, 1.159030820204300e+02, 0.1364,_
 1428, 48,  118, 0, 0,     +1, 1.168940448961158e+02, 0,_
 1429, 48,  119, 0, 0.5,   +1, 1.178883542373748e+02, 0,_
 1430, 48,  119, 1, 5.5,   -1, 1.178885102030977e+02, 0.14654,_
 1431, 48,  120, 0, 0,     +1, 1.188797847395322e+02, 0,_
 1432, 48,  121, 0, 1.5,   +1, 1.198742632628429e+02, 0,_
 1433, 48,  121, 1, 5.5,   -1, 1.198744919749638e+02, 0.21489,_
 1434, 48,  122, 0, 0,     +1, 1.208661639161800e+02, 0,_
 1435, 48,  123, 0, 1.5,   +1, 1.218609773425836e+02, 0,_
 1436, 48,  123, 1, 5.5,   -1, 1.218613142323766e+02, 0.31653,_
 1437, 48,  124, 0, 0,     +1, 1.228531452016058e+02, 0,_
 1438, 48,  125, 0, 1.5,   +1, 1.238481240080057e+02, 0,_
 1439, 48,  125, 1, 5.5,   -1, 1.238483198432248e+02, 0.184,_
 1440, 48,  126, 0, 0,     +1, 1.248406950137952e+02, 0,_
 1441, 48,  127, 0, 1.5,   +1, 1.258358396661537e+02, 0,_
 1442, 48,  128, 0, 0,     +1, 1.268288515349028e+02, 0,_
 1443, 48,  129, 0, 1.5,   +1, 1.278247140123673e+02, 0,_
 1444, 48,  129, 1, 5.5,   -1, 1.278247140123673e+02, 0,_
 1445, 48,  130, 0, 0,     +1, 1.288181888039291e+02, 0,_
 1446, 48,  131, 0, 3.5,   -1, 1.298158763592476e+02, 0,_
 1447, 48,  132, 0, 0,     +1, 1.308125601778601e+02, 0,_
 1448, 49,  97,  0, 4.5,   +1, 9.611649863045398e+01, 0,_
 1449, 49,  98,  0, 0,     +1, 9.710077000187654e+01, 0,_
 1450, 49,  98,  1,-77.777,+0, 9.710077000187654e+01, 0,_
 1451, 49,  99,  0, 4.5,   +1, 9.808421850339757e+01, 0,_
 1452, 49,  99,  1, 0.5,   -1, 9.808464423213476e+01, 0.4,_
 1453, 49,  100, 0, 6,     +1, 9.907250228760117e+01, 0,_
 1454, 49,  101, 0, 4.5,   +1, 1.000593342913601e+02, 0,_
 1455, 49,  101, 1, 0.5,   -1, 1.000599196683738e+02, 0.55,_
 1456, 49,  102, 0, 6,     +1, 1.010485289191805e+02, 0,_
 1457, 49,  103, 0, 4.5,   +1, 1.020357474507088e+02, 0,_
 1458, 49,  103, 1, 0.5,   -1, 1.020364197828170e+02, 0.6317,_
 1459, 49,  104, 0,-77.777,+0, 1.030255071928610e+02, 0,_
 1460, 49,  104, 1, 3,     +1, 1.030256066856669e+02, 0.09348,_
 1461, 49,  105, 0, 4.5,   +1, 1.040132363865448e+02, 0,_
 1462, 49,  105, 1, 0.5,   -1, 1.040139538246128e+02, 0.67408,_
 1463, 49,  106, 0, 7,     +1, 1.050036161086232e+02, 0,_
 1464, 49,  106, 1, 3,     +1, 1.050036465482279e+02, 0.0286,_
 1465, 49,  107, 0, 4.5,   +1, 1.059918793587528e+02, 0,_
 1466, 49,  107, 1, 0.5,   -1, 1.059926015011233e+02, 0.6785,_
 1467, 49,  108, 0, 7,     +1, 1.069826975862189e+02, 0,_
 1468, 49,  108, 1, 2,     +1, 1.069827292497937e+02, 0.02975,_
 1469, 49,  109, 0, 4.5,   +1, 1.079715849901087e+02, 0,_
 1470, 49,  109, 1, 0.5,   -1, 1.079722769057389e+02, 0.6501,_
 1471, 49,  109, 2, 9.5,   +1, 1.079738211303009e+02, 2.101,_
 1472, 49,  110, 0, 7,     +1, 1.089630153098468e+02, 0,_
 1473, 49,  110, 1, 2,     +1, 1.089630813829468e+02, 0.06208,_
 1474, 49,  111, 0, 4.5,   +1, 1.099523791236845e+02, 0,_
 1475, 49,  111, 1, 0.5,   -1, 1.099529508773786e+02, 0.5372,_
 1476, 49,  112, 0, 1,     +1, 1.109442164081079e+02, 0,_
 1477, 49,  112, 1, 4,     +1, 1.109443830596220e+02, 0.15658,_
 1478, 49,  113, 0, 4.5,   +1, 1.119341603710029e+02, 0,_
 1479, 49,  113, 1, 0.5,   -1, 1.119345772648045e+02, 0.391699,_
 1480, 49,  114, 0, 1,     +1, 1.129264184919135e+02, 0,_
 1481, 49,  114, 1, 5,     +1, 1.129266210749331e+02, 0.19034,_
 1482, 49,  114, 2, 8,     -1, 1.129269527601922e+02, 0.50198,_
 1483, 49,  115, 0, 4.5,   +1, 1.139167992966110e+02, 0,_
 1484, 49,  115, 1, 0.5,   -1, 1.139171571684448e+02, 0.336244,_
 1485, 49,  116, 0, 1,     +1, 1.149095781702431e+02, 0,_
 1486, 49,  116, 1, 5,     +1, 1.149097136232911e+02, 0.127267,_
 1487, 49,  116, 2, 8,     -1, 1.149098864617082e+02, 0.28966,_
 1488, 49,  117, 0, 4.5,   +1, 1.159002498039320e+02, 0,_
 1489, 49,  117, 1, 0.5,   -1, 1.159005853846091e+02, 0.3153,_
 1490, 49,  118, 0, 1,     +1, 1.168934844562799e+02, 0,_
 1491, 49,  118, 1, 5,     +1, 1.168935908884641e+02, 0.1,_
 1492, 49,  118, 2, 8,     -1, 1.168937398935222e+02, 0.24,_
 1493, 49,  119, 0, 4.5,   +1, 1.178843926055808e+02, 0,_
 1494, 49,  119, 1, 0.5,   -1, 1.178847240034731e+02, 0.31137,_
 1495, 49,  120, 0, 1,     +1, 1.188778998102855e+02, 0,_
 1496, 49,  120, 1, 5,     +1, 1.188779743128145e+02, 0.07,_
 1497, 49,  120, 2, 8,     -1, 1.188782191068384e+02, 0.3,_
 1498, 49,  121, 0, 4.5,   +1, 1.198691948036155e+02, 0,_
 1499, 49,  121, 1, 0.5,   -1, 1.198695285749454e+02, 0.3136,_
 1500, 49,  122, 0, 1,     +1, 1.208630131314643e+02, 0,_
 1501, 49,  122, 1, 8,     -1, 1.208632259958329e+02, 0.2,_
 1502, 49,  122, 2, 8,     -1, 1.208633217847988e+02, 0.29,_
 1503, 49,  123, 0, 4.5,   +1, 1.218545741982899e+02, 0,_
 1504, 49,  123, 1, 0.5,   -1, 1.218549224550401e+02, 0.32721,_
 1505, 49,  124, 0, 1,     +1, 1.228487085448630e+02, 0,_
 1506, 49,  124, 1, 8,     -1, 1.228487617609552e+02, 0.05,_
 1507, 49,  125, 0, 4.5,   +1, 1.238405367648291e+02, 0,_
 1508, 49,  125, 1, 0.5,   -1, 1.238409200484112e+02, 0.36012,_
 1509, 49,  126, 0, 3,     +1, 1.248348241017538e+02, 0,_
 1510, 49,  126, 1, 8,     -1, 1.248349305339381e+02, 0.1,_
 1511, 49,  127, 0, 4.5,   +1, 1.258271667383432e+02, 0,_
 1512, 49,  127, 1, 0.5,   -1, 1.258276563263909e+02, 0.46,_
 1513, 49,  128, 0, 3,     +1, 1.268215034018571e+02, 0,_
 1514, 49,  128, 1, 1,     -1, 1.268217672153123e+02, 0.24787,_
 1515, 49,  128, 2, 8,     -1, 1.268218439848468e+02, 0.32,_
 1516, 49,  129, 0, 4.5,   +1, 1.278143052900698e+02, 0,_
 1517, 49,  129, 1, 0.5,   -1, 1.278147097323701e+02, 0.38,_
 1518, 49,  130, 0, 1,     -1, 1.288088593647344e+02, 0,_
 1519, 49,  130, 1, 10,    -1, 1.288089125808265e+02, 0.05,_
 1520, 49,  130, 2, 5,     +1, 1.288092850934716e+02, 0.4,_
 1521, 49,  131, 0, 4.5,   +1, 1.298022465821190e+02, 0,_
 1522, 49,  131, 1, 0.5,   -1, 1.298026190947641e+02, 0.35,_
 1523, 49,  131, 2,-77.777,+0, 1.298066103016754e+02, 4.1,_
 1524, 49,  132, 0, 7,     -1, 1.307996306600436e+02, 0,_
 1525, 49,  133, 0, 4.5,   +1, 1.317963061022372e+02, 0,_
 1526, 49,  133, 1, 0.5,   -1, 1.317966573284453e+02, 0.33,_
 1527, 49,  134, 0,-77.777,+0, 1.327938921035129e+02, 0,_
 1528, 49,  135, 0, 4.5,   +1, 1.337907642899343e+02, 0,_
 1529, 50,  99,  0, 4.5,   +1, 9.809851462867549e+01, 0,_
 1530, 50,  100, 0, 0,     +1, 9.907998447111090e+01, 0,_
 1531, 50,  101, 0, 2.5,   +1, 1.000681769147726e+02, 0,_
 1532, 50,  102, 0, 0,     +1, 1.010546594129679e+02, 0,_
 1533, 50,  103, 0, 2.5,   +1, 1.020439001571934e+02, 0,_
 1534, 50,  104, 0, 0,     +1, 1.030303558367697e+02, 0,_
 1535, 50,  105, 0, 2.5,   +1, 1.040199443615065e+02, 0,_
 1536, 50,  106, 0, 0,     +1, 1.050070798882642e+02, 0,_
 1537, 50,  107, 0, 2.5,   +1, 1.059972563475971e+02, 0,_
 1538, 50,  108, 0, 0,     +1, 1.069848793186907e+02, 0,_
 1539, 50,  109, 0, 2.5,   +1, 1.079756925563926e+02, 0,_
 1540, 50,  110, 0, 0,     +1, 1.089636836884029e+02, 0,_
 1541, 50,  111, 0, 3.5,   +1, 1.099549903902793e+02, 0,_
 1542, 50,  112, 0, 0,     +1, 1.109435087132245e+02, 0,_
 1543, 50,  113, 0, 0.5,   +1, 1.119352661862072e+02, 0,_
 1544, 50,  113, 1, 3.5,   +1, 1.119353485540747e+02, 0.07739,_
 1545, 50,  114, 0, 0,     +1, 1.129243005734983e+02, 0,_
 1546, 50,  115, 0, 0.5,   +1, 1.139162698085802e+02, 0,_
 1547, 50,  116, 0, 0,     +1, 1.149060912164046e+02, 0,_
 1548, 50,  117, 0, 0.5,   +1, 1.158987015246417e+02, 0,_
 1549, 50,  117, 1, 5.5,   -1, 1.158990363390070e+02, 0.31458,_
 1550, 50,  118, 0, 0,     +1, 1.168887752114895e+02, 0,_
 1551, 50,  119, 0, 0.5,   +1, 1.178818746950635e+02, 0,_
 1552, 50,  119, 1, 5.5,   -1, 1.178819699837981e+02, 0.08953,_
 1553, 50,  120, 0, 0,     +1, 1.188721844018187e+02, 0,_
 1554, 50,  121, 0, 1.5,   +1, 1.198656173083111e+02, 0,_
 1555, 50,  121, 1, 5.5,   -1, 1.198656240135387e+02, 0.0063,_
 1556, 50,  122, 0, 0,     +1, 1.208562348992756e+02, 0,_
 1557, 50,  123, 0, 5.5,   -1, 1.218499062655343e+02, 0,_
 1558, 50,  123, 1, 1.5,   +1, 1.218499324478516e+02, 0.0246,_
 1559, 50,  124, 0, 0,     +1, 1.228408708865129e+02, 0,_
 1560, 50,  125, 0, 5.5,   -1, 1.238347685965220e+02, 0,_
 1561, 50,  125, 1, 1.5,   +1, 1.238347978653726e+02, 0.0275,_
 1562, 50,  126, 0, 0,     +1, 1.248260516066851e+02, 0,_
 1563, 50,  127, 0, 5.5,   -1, 1.258201692261968e+02, 0,_
 1564, 50,  127, 1, 1.5,   +1, 1.258201742285095e+02, 0.0047,_
 1565, 50,  128, 0, 0,     +1, 1.268116945388295e+02, 0,_
 1566, 50,  128, 1, 7,     -1, 1.268139205679641e+02, 2.0915,_
 1567, 50,  129, 0, 1.5,   +1, 1.278060534079338e+02, 0,_
 1568, 50,  129, 1, 5.5,   -1, 1.278060908720627e+02, 0.0352,_
 1569, 50,  130, 0, 0,     +1, 1.287979511286815e+02, 0,_
 1570, 50,  130, 1, 7,     -1, 1.288000232568777e+02, 1.9469,_
 1571, 50,  131, 0, 1.5,   +1, 1.297924127355716e+02, 0,_
 1572, 50,  131, 1, 5.5,   -1, 1.297924978813190e+02, 0.08,_
 1573, 50,  132, 0, 0,     +1, 1.307845864696866e+02, 0,_
 1574, 50,  133, 0, 3.5,   -1, 1.317820335288838e+02, 0,_
 1575, 50,  134, 0, 0,     +1, 1.327781687778066e+02, 0,_
 1576, 50,  135, 0, 3.5,   -1, 1.337757529667857e+02, 0,_
 1577, 50,  136, 0, 0,     +1, 1.347722002301297e+02, 0,_
 1578, 50,  137, 0, 2.5,   -1, 1.357701133965471e+02, 0,_
 1579, 51,  103, 0, 2.5,   +1, 1.020553886483844e+02, 0,_
 1580, 51,  104, 0,-77.777,+0, 1.030436102930221e+02, 0,_
 1581, 51,  105, 0, 2.5,   +1, 1.040298665128999e+02, 0,_
 1582, 51,  106, 0, 4,     +1, 1.050186601324233e+02, 0,_
 1583, 51,  107, 0, 2.5,   +1, 1.060056208429490e+02, 0,_
 1584, 51,  108, 0, 4,     +1, 1.069951230000540e+02, 0,_
 1585, 51,  109, 0, 2.5,   +1, 1.079824820866842e+02, 0,_
 1586, 51,  110, 0, 3,     +1, 1.089726157438940e+02, 0,_
 1587, 51,  111, 0, 2.5,   +1, 1.099604204026790e+02, 0,_
 1588, 51,  112, 0, 3,     +1, 1.109510186661148e+02, 0,_
 1589, 51,  113, 0, 2.5,   +1, 1.119394289234395e+02, 0,_
 1590, 51,  114, 0, 3,     +1, 1.129307537165569e+02, 0,_
 1591, 51,  115, 0, 2.5,   +1, 1.139194951641458e+02, 0,_
 1592, 51,  116, 0, 3,     +1, 1.149110975955507e+02, 0,_
 1593, 51,  116, 1, 8,     -1, 1.149115052308166e+02, 0.383,_
 1594, 51,  117, 0, 2.5,   +1, 1.159005728279560e+02, 0,_
 1595, 51,  118, 0, 1,     +1, 1.168926670540018e+02, 0,_
 1596, 51,  118, 1, 8,     -1, 1.168929331344625e+02, 0.25,_
 1597, 51,  119, 0, 2.5,   +1, 1.178825035421564e+02, 0,_
 1598, 51,  119, 1, 13.5,  +1, 1.178855389880526e+02, 2.852,_
 1599, 51,  120, 0, 1,     +1, 1.188750374315562e+02, 0,_
 1600, 51,  120, 1, 8,     -1, 1.188752502959248e+02, 0.2,_
 1601, 51,  121, 0, 2.5,   +1, 1.198651883264033e+02, 0,_
 1602, 51,  122, 0, 2,     -1, 1.208579441616609e+02, 0,_
 1603, 51,  122, 1, 8,     -1, 1.208581182421416e+02, 0.16356,_
 1604, 51,  123, 0, 3.5,   +1, 1.218484078194435e+02, 0,_
 1605, 51,  124, 0, 3,     -1, 1.228415243215533e+02, 0,_
 1606, 51,  124, 1, 5,     +1, 1.228415358800885e+02, 0.01086,_
 1607, 51,  124, 2, 8,     -1, 1.228415635418132e+02, 0.03685,_
 1608, 51,  125, 0, 3.5,   +1, 1.238322569051165e+02, 0,_
 1609, 51,  126, 0, 8,     -1, 1.248256492927019e+02, 0,_
 1610, 51,  126, 1, 5,     +1, 1.248256681311985e+02, 0.0177,_
 1611, 51,  126, 2, 3,     -1, 1.248256922913043e+02, 0.0404,_
 1612, 51,  127, 0, 3.5,   +1, 1.258167328778659e+02, 0,_
 1613, 51,  128, 0, 8,     -1, 1.268103446832148e+02, 0,_
 1614, 51,  128, 1, 5,     +1, 1.268103553264332e+02, 0.01,_
 1615, 51,  129, 0, 3.5,   +1, 1.278017552451525e+02, 0,_
 1616, 51,  129, 1, 9.5,   -1, 1.278037256241805e+02, 1.8513,_
 1617, 51,  130, 0, 8,     -1, 1.287956591435398e+02, 0,_
 1618, 51,  130, 1,-77.777,+0, 1.287956642522847e+02, 0.0048,_
 1619, 51,  131, 0, 3.5,   +1, 1.297873925104001e+02, 0,_
 1620, 51,  132, 0, 4,     +1, 1.307812990677477e+02, 0,_
 1621, 51,  132, 1, 8,     -1, 1.307815119321163e+02, 0.2,_
 1622, 51,  133, 0, 3.5,   +1, 1.317734661385994e+02, 0,_
 1623, 51,  134, 0, 0,     -1, 1.327700939871875e+02, 0,_
 1624, 51,  134, 1, 7,     -1, 1.327701791329350e+02, 0.08,_
 1625, 51,  135, 0, 3.5,   +1, 1.337661122547440e+02, 0,_
 1626, 51,  136, 0, 1,     -1, 1.347630386256612e+02, 0,_
 1627, 51,  137, 0, 3.5,   +1, 1.357591806498766e+02, 0,_
 1628, 51,  138, 0, 2,     -1, 1.367568057900174e+02, 0,_
 1629, 51,  139, 0, 3.5,   +1, 1.377529324364797e+02, 0,_
 1630, 52,  105, 0, 2.5,   +1, 1.040417911459582e+02, 0,_
 1631, 52,  106, 0, 0,     +1, 1.050274445600970e+02, 0,_
 1632, 52,  107, 0, 2.5,   +1, 1.060163853018190e+02, 0,_
 1633, 52,  108, 0, 0,     +1, 1.070022152830191e+02, 0,_
 1634, 52,  109, 0, 2.5,   +1, 1.079915666992808e+02, 0,_
 1635, 52,  110, 0, 0,     +1, 1.089781714224073e+02, 0,_
 1636, 52,  111, 0, 2.5,   +1, 1.099681359481272e+02, 0,_
 1637, 52,  112, 0, 0,     +1, 1.109553094339726e+02, 0,_
 1638, 52,  113, 0, 3.5,   +1, 1.119458892928821e+02, 0,_
 1639, 52,  114, 0, 0,     +1, 1.129335294738536e+02, 0,_
 1640, 52,  115, 0, 3.5,   +1, 1.139247536002397e+02, 0,_
 1641, 52,  115, 1, 0.5,   +1, 1.139247748866765e+02, 0.02,_
 1642, 52,  116, 0, 0,     +1, 1.149127506886383e+02, 0,_
 1643, 52,  117, 0, 0.5,   +1, 1.159043449210865e+02, 0,_
 1644, 52,  117, 1, 5.5,   -1, 1.159046600667842e+02, 0.2961,_
 1645, 52,  118, 0, 0,     +1, 1.168929859557450e+02, 0,_
 1646, 52,  119, 0, 0.5,   +1, 1.178849440315214e+02, 0,_
 1647, 52,  119, 1, 5.5,   -1, 1.178852217769496e+02, 0.26096,_
 1648, 52,  120, 0, 0,     +1, 1.188740260847908e+02, 0,_
 1649, 52,  121, 0, 0.5,   +1, 1.198663109935866e+02, 0,_
 1650, 52,  121, 1, 5.5,   -1, 1.198666238946296e+02, 0.293991,_
 1651, 52,  122, 0, 0,     +1, 1.208558377733384e+02, 0,_
 1652, 52,  123, 0, 0.5,   +1, 1.218484630716874e+02, 0,_
 1653, 52,  123, 1, 5.5,   -1, 1.218487264168411e+02, 0.24743,_
 1654, 52,  124, 0, 0,     +1, 1.228384323878981e+02, 0,_
 1655, 52,  125, 0, 0.5,   +1, 1.238314408888290e+02, 0,_
 1656, 52,  125, 1, 5.5,   -1, 1.238315949664449e+02, 0.144766,_
 1657, 52,  126, 0, 0,     +1, 1.248217409878346e+02, 0,_
 1658, 52,  127, 0, 1.5,   +1, 1.258150489063374e+02, 0,_
 1659, 52,  127, 1, 5.5,   -1, 1.258151428114536e+02, 0.08823,_
 1660, 52,  128, 0, 0,     +1, 1.268057005898925e+02, 0,_
 1661, 52,  129, 0, 1.5,   +1, 1.277992269486291e+02, 0,_
 1662, 52,  129, 1, 5.5,   -1, 1.277993392345835e+02, 0.1055,_
 1663, 52,  130, 0, 0,     +1, 1.287902659342443e+02, 0,_
 1664, 52,  131, 0, 1.5,   +1, 1.297839551647111e+02, 0,_
 1665, 52,  131, 1, 5.5,   -1, 1.297841492118695e+02, 0.18232,_
 1666, 52,  132, 0, 0,     +1, 1.307753889791677e+02, 0,_
 1667, 52,  133, 0, 1.5,   +1, 1.317691943552426e+02, 0,_
 1668, 52,  133, 1, 5.5,   -1, 1.317695501261051e+02, 0.33427,_
 1669, 52,  134, 0, 0,     +1, 1.327610332021273e+02, 0,_
 1670, 52,  135, 0, 3.5,   -1, 1.337575567484855e+02, 0,_
 1671, 52,  136, 0, 0,     +1, 1.347524822666236e+02, 0,_
 1672, 52,  137, 0, 1.5,   -1, 1.357493427296274e+02, 0,_
 1673, 52,  138, 0, 0,     +1, 1.367445920748313e+02, 0,_
 1674, 52,  139, 0, 2.5,   -1, 1.377418456951600e+02, 0,_
 1675, 52,  140, 0, 0,     +1, 1.387371174729871e+02, 0,_
 1676, 52,  141, 0, 2.5,   -1, 1.397350079192724e+02, 0,_
 1677, 52,  142, 0, 0,     +1, 1.407307994693170e+02, 0,_
 1678, 53,  108, 0, 1,     +1, 1.070161920257202e+02, 0,_
 1679, 53,  109, 0, 2.5,   +1, 1.080022555720976e+02, 0,_
 1680, 53,  110, 0, 1,     +1, 1.089906938456639e+02, 0,_
 1681, 53,  111, 0, 2.5,   +1, 1.099773249759744e+02, 0,_
 1682, 53,  112, 0, 1,     +1, 1.109664892617064e+02, 0,_
 1683, 53,  113, 0, 2.5,   +1, 1.119535817027978e+02, 0,_
 1684, 53,  114, 0, 1,     +1, 1.129432066221780e+02, 0,_
 1685, 53,  114, 1,-77.777,+0, 1.129434896253560e+02, 0.2659,_
 1686, 53,  115, 0, 2.5,   +1, 1.139308468031494e+02, 0,_
 1687, 53,  116, 0, 1,     +1, 1.149210276276584e+02, 0,_
 1688, 53,  117, 0, 2.5,   +1, 1.159093039524968e+02, 0,_
 1689, 53,  118, 0, 2,     -1, 1.169001440920961e+02, 0,_
 1690, 53,  118, 1, 7,     -1, 1.169002547815678e+02, 0.104,_
 1691, 53,  119, 0, 2.5,   +1, 1.178885793835025e+02, 0,_
 1692, 53,  120, 0, 2,     -1, 1.188800022527981e+02, 0,_
 1693, 53,  120, 1, 7,     -1, 1.188803428357879e+02, 0.32,_
 1694, 53,  121, 0, 2.5,   +1, 1.198687526042359e+02, 0,_
 1695, 53,  122, 0, 1,     +1, 1.208603441122908e+02, 0,_
 1696, 53,  123, 0, 2.5,   +1, 1.218497705157943e+02, 0,_
 1697, 53,  124, 0, 2,     -1, 1.228417952063592e+02, 0,_
 1698, 53,  125, 0, 2.5,   +1, 1.238316386086038e+02, 0,_
 1699, 53,  126, 0, 2,     -1, 1.248240335698048e+02, 0,_
 1700, 53,  127, 0, 2.5,   +1, 1.258143015064941e+02, 0,_
 1701, 53,  128, 0, 1,     +1, 1.268070363050332e+02, 0,_
 1702, 53,  129, 0, 3.5,   +1, 1.277976279983982e+02, 0,_
 1703, 53,  130, 0, 5,     +1, 1.287907095543138e+02, 0,_
 1704, 53,  130, 1, 2,     +1, 1.287907520766322e+02, 0.0399525,_
 1705, 53,  131, 0, 3.5,   +1, 1.297815799190151e+02, 0,_
 1706, 53,  132, 0, 4,     +1, 1.307748405294385e+02, 0,_
 1707, 53,  132, 1, 8,     -1, 1.307749682480597e+02, 0.12,_
 1708, 53,  133, 0, 3.5,   +1, 1.317660853237388e+02, 0,_
 1709, 53,  133, 1, 9.5,   -1, 1.317678245852786e+02, 1.63415,_
 1710, 53,  134, 0, 4,     +1, 1.327594264088558e+02, 0,_
 1711, 53,  134, 1, 8,     -1, 1.327597632667191e+02, 0.3165,_
 1712, 53,  135, 0, 3.5,   +1, 1.337511172105397e+02, 0,_
 1713, 53,  136, 0, 1,     -1, 1.347470329970855e+02, 0,_
 1714, 53,  136, 1, 6,     -1, 1.347477141630650e+02, 0.64,_
 1715, 53,  137, 0, 3.5,   +1, 1.357418365926723e+02, 0,_
 1716, 53,  138, 0, 2,     -1, 1.367379039667251e+02, 0,_
 1717, 53,  139, 0, 3.5,   +1, 1.377330481352758e+02, 0,_
 1718, 53,  140, 0, 3,     -1, 1.387296353053399e+02, 0,_
 1719, 53,  141, 0, 3.5,   +1, 1.397249610584755e+02, 0,_
 1720, 53,  142, 0, 2,     -1, 1.407218589382660e+02, 0,_
 1721, 53,  143, 0, 3.5,   +1, 1.417176742821391e+02, 0,_
 1722, 53,  144, 0, 1,     -1, 1.427147784583881e+02, 0,_
 1723, 54,  110, 0, 0,     +1, 1.089997848052643e+02, 0,_
 1724, 54,  111, 0, 2.5,   +1, 1.099885623550308e+02, 0,_
 1725, 54,  112, 0, 0,     +1, 1.109739788857446e+02, 0,_
 1726, 54,  113, 0, 2.5,   +1, 1.119630710801419e+02, 0,_
 1727, 54,  114, 0, 0,     +1, 1.129492842906919e+02, 0,_
 1728, 54,  115, 0, 2.5,   +1, 1.139390219115235e+02, 0,_
 1729, 54,  116, 0, 0,     +1, 1.149257590839876e+02, 0,_
 1730, 54,  117, 0, 2.5,   +1, 1.159159567525444e+02, 0,_
 1731, 54,  118, 0, 0,     +1, 1.169032221014046e+02, 0,_
 1732, 54,  119, 0, 2.5,   +1, 1.178938702515761e+02, 0,_
 1733, 54,  120, 0, 0,     +1, 1.188816844814286e+02, 0,_
 1734, 54,  121, 0, 2.5,   +1, 1.198727655910430e+02, 0,_
 1735, 54,  122, 0, 0,     +1, 1.208611162596985e+02, 0,_
 1736, 54,  123, 0, 0.5,   +1, 1.218526388915598e+02, 0,_
 1737, 54,  124, 0, 0,     +1, 1.228414805002611e+02, 0,_
 1738, 54,  125, 0, 0.5,   +1, 1.238333881658376e+02, 0,_
 1739, 54,  125, 1, 4.5,   -1, 1.238336570135352e+02, 0.2526,_
 1740, 54,  126, 0, 0,     +1, 1.248227184462398e+02, 0,_
 1741, 54,  127, 0, 0.5,   +1, 1.258150064591388e+02, 0,_
 1742, 54,  127, 1, 4.5,   -1, 1.258153226691584e+02, 0.2971,_
 1743, 54,  128, 0, 0,     +1, 1.268047782667449e+02, 0,_
 1744, 54,  129, 0, 0.5,   +1, 1.277974269127088e+02, 0,_
 1745, 54,  129, 1, 5.5,   -1, 1.277976782416688e+02, 0.23614,_
 1746, 54,  130, 0, 0,     +1, 1.287875758456357e+02, 0,_
 1747, 54,  131, 0, 1.5,   +1, 1.297805466244258e+02, 0,_
 1748, 54,  131, 1, 5.5,   -1, 1.297807210987055e+02, 0.16393,_
 1749, 54,  132, 0, 0,     +1, 1.307710350763392e+02, 0,_
 1750, 54,  132, 1, 10,    +1, 1.307739642603427e+02, 2.75216,_
 1751, 54,  133, 0, 1.5,   +1, 1.317641851773474e+02, 0,_
 1752, 54,  133, 1, 5.5,   -1, 1.317644333974233e+02, 0.233219,_
 1753, 54,  134, 0, 0,     +1, 1.327550814277513e+02, 0,_
 1754, 54,  134, 1, 7,     -1, 1.327571733523337e+02, 1.9655,_
 1755, 54,  135, 0, 1.5,   +1, 1.337483137810205e+02, 0,_
 1756, 54,  135, 1, 5.5,   -1, 1.337488742209734e+02, 0.52657,_
 1757, 54,  136, 0, 0,     +1, 1.347397062636143e+02, 0,_
 1758, 54,  137, 0, 3.5,   -1, 1.357354217695744e+02, 0,_
 1759, 54,  138, 0, 0,     +1, 1.367293975511004e+02, 0,_
 1760, 54,  139, 0, 1.5,   -1, 1.377254130922807e+02, 0,_
 1761, 54,  140, 0, 0,     +1, 1.387196517123329e+02, 0,_
 1762, 54,  141, 0, 2.5,   -1, 1.397161584324887e+02, 0,_
 1763, 54,  142, 0, 0,     +1, 1.407107264979244e+02, 0,_
 1764, 54,  143, 0, 2.5,   -1, 1.417074861147518e+02, 0,_
 1765, 54,  144, 0, 0,     +1, 1.427024404452335e+02, 0,_
 1766, 54,  145, 0, 1.5,   -1, 1.436995749140004e+02, 0,_
 1767, 54,  146, 0, 0,     +1, 1.446947504160490e+02, 0,_
 1768, 54,  147, 0, 1.5,   -1, 1.456921150871322e+02, 0,_
 1769, 55,  112, 0, 1,     +1, 1.109885984782066e+02, 0,_
 1770, 55,  113, 0, 2.5,   +1, 1.119741816301613e+02, 0,_
 1771, 55,  114, 0, 1,     +1, 1.129624857451999e+02, 0,_
 1772, 55,  115, 0, 4.5,   +1, 1.139485553599950e+02, 0,_
 1773, 55,  116, 0, 1,     +1, 1.149374714850187e+02, 0,_
 1774, 55,  116, 1,-77.777,+0, 1.149375779172030e+02, 0.1,_
 1775, 55,  117, 0, 4.5,   +1, 1.159241437786524e+02, 0,_
 1776, 55,  117, 1, 1.5,   +1, 1.159243034269288e+02, 0.15,_
 1777, 55,  118, 0,-77.777,+0, 1.169135137640144e+02, 0,_
 1778, 55,  118, 1, 7,     -1, 1.169136201961987e+02, 0.1,_
 1779, 55,  119, 0, 4.5,   +1, 1.179007770219918e+02, 0,_
 1780, 55,  119, 1, 1.5,   +1, 1.179008302380840e+02, 0.05,_
 1781, 55,  120, 0, 2,     +1, 1.188905010952124e+02, 0,_
 1782, 55,  120, 1, 7,     -1, 1.188906075273967e+02, 0.1,_
 1783, 55,  121, 0, 1.5,   +1, 1.198784902116870e+02, 0,_
 1784, 55,  121, 1, 4.5,   +1, 1.198785631177333e+02, 0.0685,_
 1785, 55,  122, 0, 1,     +1, 1.208687902521994e+02, 0,_
 1786, 55,  122, 1, 8,     -1, 1.208689243567517e+02, 0.126,_
 1787, 55,  122, 2, 5,     -1, 1.208689254210735e+02, 0.127,_
 1788, 55,  123, 0, 0.5,   +1, 1.218571144234527e+02, 0,_
 1789, 55,  123, 1, 5.5,   -1, 1.218572807450271e+02, 0.15627,_
 1790, 55,  124, 0, 1,     +1, 1.228477920214612e+02, 0,_
 1791, 55,  124, 1, 7,     +1, 1.228482843235296e+02, 0.46255,_
 1792, 55,  125, 0, 0.5,   +1, 1.238366933437492e+02, 0,_
 1793, 55,  125, 1, 5.5,   -1, 1.238369770919526e+02, 0.2666,_
 1794, 55,  126, 0, 1,     +1, 1.248278230760521e+02, 0,_
 1795, 55,  127, 0, 0.5,   +1, 1.258172217458658e+02, 0,_
 1796, 55,  128, 0, 1,     +1, 1.268089596870896e+02, 0,_
 1797, 55,  129, 0, 0.5,   +1, 1.277987007064730e+02, 0,_
 1798, 55,  130, 0, 1,     +1, 1.287907482906666e+02, 0,_
 1799, 55,  130, 1, 5,     -1, 1.287909220412074e+02, 0.16325,_
 1800, 55,  131, 0, 2.5,   +1, 1.297809242156298e+02, 0,_
 1801, 55,  132, 0, 2,     +1, 1.307732981232582e+02, 0,_
 1802, 55,  133, 0, 3.5,   +1, 1.317637303295651e+02, 0,_
 1803, 55,  134, 0, 4,     +1, 1.327563955113277e+02, 0,_
 1804, 55,  134, 1, 8,     -1, 1.327565431827905e+02, 0.138747,_
 1805, 55,  135, 0, 3.5,   +1, 1.337470701301506e+02, 0,_
 1806, 55,  135, 1, 9.5,   -1, 1.337488084870168e+02, 1.6333,_
 1807, 55,  136, 0, 5,     +1, 1.347398025433584e+02, 0,_
 1808, 55,  136, 1, 8,     -1, 1.347404411364642e+02, 0.6,_
 1809, 55,  137, 0, 3.5,   +1, 1.357309918454937e+02, 0,_
 1810, 55,  138, 0, 3,     -1, 1.367262953672622e+02, 0,_
 1811, 55,  138, 1, 6,     -1, 1.367263804065774e+02, 0.0799,_
 1812, 55,  139, 0, 3.5,   +1, 1.377200315122189e+02, 0,_
 1813, 55,  140, 0, 1,     -1, 1.387153266764051e+02, 0,_
 1814, 55,  141, 0, 3.5,   +1, 1.397094742523469e+02, 0,_
 1815, 55,  142, 0, 0,     -1, 1.407051016507517e+02, 0,_
 1816, 55,  143, 0, 1.5,   +1, 1.416995328243437e+02, 0,_
 1817, 55,  144, 0, 1,     -1, 1.426956297840394e+02, 0,_
 1818, 55,  144, 1,-77.777,+0, 1.426959490805923e+02, 0.3,_
 1819, 55,  145, 0, 1.5,   +1, 1.436904631625596e+02, 0,_
 1820, 55,  146, 0, 1,     -1, 1.446869218717266e+02, 0,_
 1821, 55,  147, 0, 1.5,   +1, 1.456819397703952e+02, 0,_
 1822, 55,  148, 0,-77.777,+0, 1.466786806089349e+02, 0,_
 1823, 55,  149, 0, 1.5,   +1, 1.476739863395638e+02, 0,_
 1824, 55,  150, 0,-77.777,+0, 1.486708030070521e+02, 0,_
 1825, 55,  151, 0, 1.5,   +1, 1.496664061892879e+02, 0,_
 1826, 56,  114, 0, 0,     +1, 1.129718271229481e+02, 0,_
 1827, 56,  115, 0, 2.5,   +1, 1.139599218701414e+02, 0,_
 1828, 56,  116, 0, 0,     +1, 1.149454136666831e+02, 0,_
 1829, 56,  117, 0, 1.5,   +1, 1.159337602874135e+02, 0,_
 1830, 56,  118, 0, 0,     +1, 1.169199584027621e+02, 0,_
 1831, 56,  119, 0, 2.5,   +1, 1.179089882285780e+02, 0,_
 1832, 56,  120, 0, 0,     +1, 1.188958227049123e+02, 0,_
 1833, 56,  121, 0, 2.5,   +1, 1.198852566322227e+02, 0,_
 1834, 56,  122, 0, 0,     +1, 1.208725534989829e+02, 0,_
 1835, 56,  123, 0, 2.5,   +1, 1.218628497275257e+02, 0,_
 1836, 56,  124, 0, 0,     +1, 1.228506034913116e+02, 0,_
 1837, 56,  125, 0, 0.5,   +1, 1.238413965667181e+02, 0,_
 1838, 56,  126, 0, 0,     +1, 1.248296121231100e+02, 0,_
 1839, 56,  127, 0, 0.5,   +1, 1.258208640793527e+02, 0,_
 1840, 56,  127, 1, 3.5,   -1, 1.258209495763264e+02, 0.08033,_
 1841, 56,  128, 0, 0,     +1, 1.268095483464062e+02, 0,_
 1842, 56,  129, 0, 0.5,   +1, 1.278012934465981e+02, 0,_
 1843, 56,  129, 1, 3.5,   +1, 1.278013024081880e+02, 0.00842,_
 1844, 56,  130, 0, 0,     +1, 1.287903632182863e+02, 0,_
 1845, 56,  130, 1, 8,     -1, 1.287929975425664e+02, 2.47512,_
 1846, 56,  131, 0, 0.5,   +1, 1.297823877165178e+02, 0,_
 1847, 56,  131, 1, 4.5,   -1, 1.297825872768634e+02, 0.1875,_
 1848, 56,  132, 0, 0,     +1, 1.307719333042996e+02, 0,_
 1849, 56,  133, 0, 0.5,   +1, 1.317642809227218e+02, 0,_
 1850, 56,  133, 1, 5.5,   -1, 1.317645877134930e+02, 0.28825,_
 1851, 56,  134, 0, 0,     +1, 1.327542043931820e+02, 0,_
 1852, 56,  135, 0, 1.5,   +1, 1.337467839816037e+02, 0,_
 1853, 56,  135, 1, 5.5,   -1, 1.337470694518798e+02, 0.268218,_
 1854, 56,  136, 0, 0,     +1, 1.347370904127418e+02, 0,_
 1855, 56,  136, 1, 7,     -1, 1.347392514863144e+02, 2.03047,_
 1856, 56,  137, 0, 1.5,   +1, 1.357297405984440e+02, 0,_
 1857, 56,  137, 1, 5.5,   -1, 1.357304448165704e+02, 0.661659,_
 1858, 56,  138, 0, 0,     +1, 1.367205749561430e+02, 0,_
 1859, 56,  139, 0, 3.5,   -1, 1.377155477060221e+02, 0,_
 1860, 56,  140, 0, 0,     +1, 1.387087073929392e+02, 0,_
 1861, 56,  141, 0, 1.5,   -1, 1.397038811302788e+02, 0,_
 1862, 56,  142, 0, 0,     +1, 1.406973026048281e+02, 0,_
 1863, 56,  143, 0, 2.5,   -1, 1.416928683732514e+02, 0,_
 1864, 56,  144, 0, 0,     +1, 1.426865875512256e+02, 0,_
 1865, 56,  145, 0, 2.5,   -1, 1.436825214468576e+02, 0,_
 1866, 56,  146, 0, 0,     +1, 1.446766653050138e+02, 0,_
 1867, 56,  147, 0, 1.5,   -1, 1.456730591056080e+02, 0,_
 1868, 56,  148, 0, 0,     +1, 1.466673106774342e+02, 0,_
 1869, 56,  149, 0, 1.5,   -1, 1.476634813642903e+02, 0,_
 1870, 56,  150, 0, 0,     +1, 1.486583181869671e+02, 0,_
 1871, 56,  151, 0, 1.5,   -1, 1.496550069626273e+02, 0,_
 1872, 56,  152, 0, 0,     +1, 1.506498536993994e+02, 0,_
 1873, 56,  153, 0, 2.5,   -1, 1.516468408893251e+02, 0,_
 1874, 57,  117, 0,-77.777,+0, 1.159454534065208e+02, 0,_
 1875, 57,  117, 1, 4.5,   +1, 1.159456002829351e+02, 0.138,_
 1876, 57,  118, 0,-77.777,+0, 1.169335754125189e+02, 0,_
 1877, 57,  119, 0, 5.5,   -1, 1.179194191594402e+02, 0,_
 1878, 57,  120, 0,-77.777,+0, 1.189078693219894e+02, 0,_
 1879, 57,  121, 0, 5.5,   -1, 1.198943614507367e+02, 0,_
 1880, 57,  122, 0,-77.777,+0, 1.208832666702556e+02, 0,_
 1881, 57,  123, 0, 5.5,   -1, 1.218703040742389e+02, 0,_
 1882, 57,  124, 0,-77.777,+0, 1.228600026940114e+02, 0,_
 1883, 57,  124, 1, 0,     +1, 1.228601091261957e+02, 0.1,_
 1884, 57,  125, 0, 5.5,   -1, 1.238476861569483e+02, 0,_
 1885, 57,  125, 1, 1.5,   +1, 1.238478000393855e+02, 0.107,_
 1886, 57,  126, 0, 5,     +1, 1.248378036075866e+02, 0,_
 1887, 57,  126, 1,-77.777,+0, 1.248379100397709e+02, 0.1,_
 1888, 57,  127, 0, 5.5,   -1, 1.258261024978970e+02, 0,_
 1889, 57,  127, 1, 1.5,   +1, 1.258261182498603e+02, 0.0148,_
 1890, 57,  128, 0, 5,     +1, 1.268167357828742e+02, 0,_
 1891, 57,  128, 1,-77.777,+0, 1.268168422150585e+02, 0.1,_
 1892, 57,  129, 0, 1.5,   +1, 1.278052725470278e+02, 0,_
 1893, 57,  129, 1, 5.5,   -1, 1.278054557168170e+02, 0.1721,_
 1894, 57,  130, 0, 3,     +1, 1.287963597974328e+02, 0,_
 1895, 57,  131, 0, 1.5,   +1, 1.297854896574606e+02, 0,_
 1896, 57,  132, 0, 2,     -1, 1.307769477158456e+02, 0,_
 1897, 57,  132, 1, 6,     -1, 1.307771479999301e+02, 0.18818,_
 1898, 57,  133, 0, 2.5,   +1, 1.317664726069623e+02, 0,_
 1899, 57,  134, 0, 1,     +1, 1.327581755950521e+02, 0,_
 1900, 57,  135, 0, 2.5,   +1, 1.337480688106680e+02, 0,_
 1901, 57,  136, 0, 1,     +1, 1.347401231374376e+02, 0,_
 1902, 57,  136, 1,-77.777,+0, 1.347403945395076e+02, 0.255,_
 1903, 57,  137, 0, 3.5,   +1, 1.357303584874875e+02, 0,_
 1904, 57,  138, 0, 5,     +1, 1.367224294917481e+02, 0,_
 1905, 57,  139, 0, 3.5,   +1, 1.377130865021465e+02, 0,_
 1906, 57,  140, 0, 3,     -1, 1.387075935621889e+02, 0,_
 1907, 57,  141, 0, 3.5,   +1, 1.397004763543754e+02, 0,_
 1908, 57,  142, 0, 2,     -1, 1.406949802934606e+02, 0,_
 1909, 57,  143, 0, 3.5,   +1, 1.416883616952370e+02, 0,_
 1910, 57,  144, 0, 3,     -1, 1.426833067471170e+02, 0,_
 1911, 57,  145, 0, 2.5,   +1, 1.436768601673678e+02, 0,_
 1912, 57,  146, 0, 2,     -1, 1.446722981827604e+02, 0,_
 1913, 57,  146, 1, 6,     -1, 1.446724365446000e+02, 0.13,_
 1914, 57,  147, 0, 1.5,   +1, 1.456662321605126e+02, 0,_
 1915, 57,  148, 0, 2,     -1, 1.466618666712892e+02, 0,_
 1916, 57,  149, 0, 2.5,   +1, 1.476559250986956e+02, 0,_
 1917, 57,  150, 0, 3,     +1, 1.486516876400976e+02, 0,_
 1918, 57,  151, 0, 2.5,   +1, 1.496460981566809e+02, 0,_
 1919, 57,  152, 0, 2,     -1, 1.506417866001354e+02, 0,_
 1920, 57,  153, 0, 2.5,   +1, 1.516366343283170e+02, 0,_
 1921, 57,  154, 0,-77.777,+0, 1.526328650727790e+02, 0,_
 1922, 57,  155, 0, 2.5,   +1, 1.536281053991304e+02, 0,_
 1923, 58,  119, 0, 2.5,   +1, 1.179309661061192e+02, 0,_
 1924, 58,  120, 0, 0,     +1, 1.189163518218422e+02, 0,_
 1925, 58,  121, 0, 2.5,   +1, 1.199044728364308e+02, 0,_
 1926, 58,  122, 0, 0,     +1, 1.208903651624186e+02, 0,_
 1927, 58,  123, 0, 2.5,   +1, 1.218792069317282e+02, 0,_
 1928, 58,  124, 0, 0,     +1, 1.228656891463803e+02, 0,_
 1929, 58,  125, 0, 3.5,   -1, 1.238552447305443e+02, 0,_
 1930, 58,  126, 0, 0,     +1, 1.248422236413659e+02, 0,_
 1931, 58,  127, 0, 2.5,   +1, 1.258323998478894e+02, 0,_
 1932, 58,  127, 1, 0.5,   +1, 1.258323998478894e+02, 0,_
 1933, 58,  128, 0, 0,     +1, 1.268200261491276e+02, 0,_
 1934, 58,  129, 0, 2.5,   +1, 1.278106336187923e+02, 0,_
 1935, 58,  130, 0, 0,     +1, 1.287987060543144e+02, 0,_
 1936, 58,  131, 0, 3.5,   +1, 1.297898116725636e+02, 0,_
 1937, 58,  131, 1, 0.5,   +1, 1.297898774476535e+02, 0.0618,_
 1938, 58,  132, 0, 0,     +1, 1.307782810496207e+02, 0,_
 1939, 58,  132, 1, 8,     -1, 1.307807724141909e+02, 2.3408,_
 1940, 58,  133, 0, 0.5,   +1, 1.317697466397439e+02, 0,_
 1941, 58,  133, 1, 4.5,   -1, 1.317697861260843e+02, 0.0371,_
 1942, 58,  134, 0, 0,     +1, 1.327585861684680e+02, 0,_
 1943, 58,  135, 0, 0.5,   +1, 1.337502263467990e+02, 0,_
 1944, 58,  135, 1, 5.5,   -1, 1.337507008214766e+02, 0.4458,_
 1945, 58,  136, 0, 0,     +1, 1.347396219561315e+02, 0,_
 1946, 58,  137, 0, 1.5,   +1, 1.357316591949667e+02, 0,_
 1947, 58,  137, 1, 5.5,   -1, 1.357319298413681e+02, 0.25429,_
 1948, 58,  138, 0, 0,     +1, 1.367213101001818e+02, 0,_
 1949, 58,  138, 1, 7,     -1, 1.367235762223203e+02, 2.12917,_
 1950, 58,  139, 0, 1.5,   +1, 1.377133827561307e+02, 0,_
 1951, 58,  139, 1, 5.5,   -1, 1.377141855102376e+02, 0.75424,_
 1952, 58,  140, 0, 0,     +1, 1.387035914798950e+02, 0,_
 1953, 58,  141, 0, 3.5,   -1, 1.396978141868331e+02, 0,_
 1954, 58,  142, 0, 0,     +1, 1.406901813062805e+02, 0,_
 1955, 58,  143, 0, 1.5,   -1, 1.416847055831406e+02, 0,_
 1956, 58,  144, 0, 0,     +1, 1.426773654687935e+02, 0,_
 1957, 58,  145, 0, 1.5,   -1, 1.436723562712486e+02, 0,_
 1958, 58,  146, 0, 0,     +1, 1.446652895093258e+02, 0,_
 1959, 58,  147, 0, 2.5,   -1, 1.456605534688974e+02, 0,_
 1960, 58,  148, 0, 0,     +1, 1.466536823834543e+02, 0,_
 1961, 58,  149, 0, 1.5,   -1, 1.476490602222719e+02, 0,_
 1962, 58,  150, 0, 0,     +1, 1.486424100645091e+02, 0,_
 1963, 58,  151, 0, 1.5,   -1, 1.496376743482716e+02, 0,_
 1964, 58,  152, 0, 0,     +1, 1.506314729668992e+02, 0,_
 1965, 58,  153, 0, 1.5,   -1, 1.516272149464678e+02, 0,_
 1966, 58,  154, 0, 0,     +1, 1.526214876571278e+02, 0,_
 1967, 58,  155, 0, 2.5,   -1, 1.536176222348663e+02, 0,_
 1968, 58,  156, 0, 0,     +1, 1.546121824542871e+02, 0,_
 1969, 58,  157, 0, 3.5,   +1, 1.556087958828238e+02, 0,_
 1970, 59,  121, 0, 1.5,   -1, 1.199164659173939e+02, 0,_
 1971, 59,  122, 0,-77.777,+0, 1.209043014060407e+02, 0,_
 1972, 59,  123, 0, 1.5,   +1, 1.218899101889058e+02, 0,_
 1973, 59,  124, 0,-77.777,+0, 1.228782106486175e+02, 0,_
 1974, 59,  125, 0, 1.5,   +1, 1.238645233322417e+02, 0,_
 1975, 59,  126, 0,-77.777,+0, 1.248533958352464e+02, 0,_
 1976, 59,  127, 0, 1.5,   +1, 1.258403142700873e+02, 0,_
 1977, 59,  127, 1, 5.5,   -1, 1.258409528631932e+02, 0.6,_
 1978, 59,  128, 0, 4,     -1, 1.268298212751849e+02, 0,_
 1979, 59,  129, 0, 1.5,   +1, 1.278175665455654e+02, 0,_
 1980, 59,  129, 1, 5.5,   -1, 1.278179738615347e+02, 0.3827,_
 1981, 59,  130, 0,-77.777,+0, 1.288074839942042e+02, 0,_
 1982, 59,  130, 1, 2,     +1, 1.288075904263885e+02, 0.1,_
 1983, 59,  131, 0, 1.5,   +1, 1.297955672955747e+02, 0,_
 1984, 59,  131, 1, 5.5,   -1, 1.297957294982236e+02, 0.1524,_
 1985, 59,  132, 0, 2,     +1, 1.307859904027250e+02, 0,_
 1986, 59,  132, 1, 5,     +1, 1.307859904027250e+02, 0,_
 1987, 59,  133, 0, 1.5,   +1, 1.317745154771690e+02, 0,_
 1988, 59,  134, 0, 6,     -1, 1.327652966100565e+02, 0,_
 1989, 59,  134, 1, 2,     -1, 1.327652966100565e+02, 0,_
 1990, 59,  135, 0, 1.5,   +1, 1.337541433810272e+02, 0,_
 1991, 59,  136, 0, 2,     +1, 1.347451223893408e+02, 0,_
 1992, 59,  137, 0, 2.5,   +1, 1.357345508470449e+02, 0,_
 1993, 59,  138, 0, 1,     +1, 1.367260324970108e+02, 0,_
 1994, 59,  138, 1, 7,     -1, 1.367264199101617e+02, 0.364,_
 1995, 59,  139, 0, 2.5,   +1, 1.377156487663430e+02, 0,_
 1996, 59,  140, 0, 1,     +1, 1.387071974028759e+02, 0,_
 1997, 59,  141, 0, 2.5,   +1, 1.396971939768999e+02, 0,_
 1998, 59,  142, 0, 2,     -1, 1.406909749851202e+02, 0,_
 1999, 59,  142, 1, 5,     -1, 1.406909789167251e+02, 0.003694,_
 2000, 59,  143, 0, 3.5,   +1, 1.416831499961707e+02, 0,_
 2001, 59,  144, 0, 0,     -1, 1.426770263274249e+02, 0,_
 2002, 59,  144, 1, 3,     -1, 1.426770891543433e+02, 0.05903,_
 2003, 59,  145, 0, 3.5,   +1, 1.436696327592508e+02, 0,_
 2004, 59,  146, 0, 2,     -1, 1.446641766362771e+02, 0,_
 2005, 59,  147, 0, 1.5,   +1, 1.456569026578676e+02, 0,_
 2006, 59,  148, 0, 1,     -1, 1.466514079105704e+02, 0,_
 2007, 59,  148, 1, 4,     +0, 1.466515036995363e+02, 0.09,_
 2008, 59,  149, 0, 2.5,   +1, 1.476444097184956e+02, 0,_
 2009, 59,  150, 0, 1,     -1, 1.486387342947447e+02, 0,_
 2010, 59,  151, 0, 1.5,   -1, 1.496317624880429e+02, 0,_
 2011, 59,  152, 0, 4,     -1, 1.506263879283304e+02, 0,_
 2012, 59,  153, 0, 2.5,   -1, 1.516201278872394e+02, 0,_
 2013, 59,  154, 0,-77.777,+0, 1.526152236720314e+02, 0,_
 2014, 59,  155, 0, 2.5,   -1, 1.536094959078062e+02, 0,_
 2015, 59,  156, 0, 2,     -1, 1.546050006837245e+02, 0,_
 2016, 59,  157, 0, 2.5,   -1, 1.555996322846307e+02, 0,_
 2017, 59,  158, 0,-77.777,+0, 1.565955229756273e+02, 0,_
 2018, 59,  159, 0, 2.5,   -1, 1.575903826007231e+02, 0,_
 2019, 60,  124, 0, 0,     +1, 1.228873911007724e+02, 0,_
 2020, 60,  125, 0, 2.5,   +1, 1.238755289693228e+02, 0,_
 2021, 60,  126, 0, 0,     +1, 1.248611982281686e+02, 0,_
 2022, 60,  127, 0, 2.5,   +1, 1.258499012001454e+02, 0,_
 2023, 60,  128, 0, 0,     +1, 1.268362247892743e+02, 0,_
 2024, 60,  129, 0, 2.5,   +1, 1.278255047615918e+02, 0,_
 2025, 60,  130, 0, 0,     +1, 1.288123577634044e+02, 0,_
 2026, 60,  131, 0, 2.5,   +1, 1.298025201100228e+02, 0,_
 2027, 60,  132, 0, 0,     +1, 1.307900365799401e+02, 0,_
 2028, 60,  133, 0, 3.5,   +1, 1.317804812234795e+02, 0,_
 2029, 60,  133, 1, 0.5,   +1, 1.317806174247457e+02, 0.12797,_
 2030, 60,  134, 0, 0,     +1, 1.327683635165698e+02, 0,_
 2031, 60,  135, 0, 4.5,   -1, 1.337591693771935e+02, 0,_
 2032, 60,  135, 1, 0.5,   +1, 1.337592385581133e+02, 0.065,_
 2033, 60,  136, 0, 0,     +1, 1.347474011758475e+02, 0,_
 2034, 60,  137, 0, 0.5,   +1, 1.357384006329738e+02, 0,_
 2035, 60,  137, 1, 5.5,   -1, 1.357389534736688e+02, 0.51943,_
 2036, 60,  138, 0, 0,     +1, 1.367272198665537e+02, 0,_
 2037, 60,  139, 0, 1.5,   +1, 1.377186340362307e+02, 0,_
 2038, 60,  139, 1, 5.5,   -1, 1.377188800542247e+02, 0.23115,_
 2039, 60,  140, 0, 0,     +1, 1.387076541848982e+02, 0,_
 2040, 60,  141, 0, 1.5,   +1, 1.396991342496003e+02, 0,_
 2041, 60,  141, 1, 5.5,   -1, 1.396999394197178e+02, 0.75651,_
 2042, 60,  142, 0, 0,     +1, 1.406886733833062e+02, 0,_
 2043, 60,  143, 0, 3.5,   -1, 1.416821559326475e+02, 0,_
 2044, 60,  144, 0, 0,     +1, 1.426738360856018e+02, 0,_
 2045, 60,  145, 0, 3.5,   -1, 1.436677105807655e+02, 0,_
 2046, 60,  146, 0, 0,     +1, 1.446596587374897e+02, 0,_
 2047, 60,  147, 0, 2.5,   -1, 1.456540261346987e+02, 0,_
 2048, 60,  148, 0, 0,     +1, 1.466462219247014e+02, 0,_
 2049, 60,  149, 0, 2.5,   -1, 1.476408590328875e+02, 0,_
 2050, 60,  150, 0, 0,     +1, 1.486330090138219e+02, 0,_
 2051, 60,  151, 0, 1.5,   +1, 1.496273313346143e+02, 0,_
 2052, 60,  152, 0, 0,     +1, 1.506195854799728e+02, 0,_
 2053, 60,  153, 0, 1.5,   -1, 1.516139954413667e+02, 0,_
 2054, 60,  154, 0, 0,     +1, 1.526070071068768e+02, 0,_
 2055, 60,  155, 0, 1.5,   -1, 1.536021856594925e+02, 0,_
 2056, 60,  156, 0, 0,     +1, 1.545955216864380e+02, 0,_
 2057, 60,  157, 0, 2.5,   -1, 1.555912013747551e+02, 0,_
 2058, 60,  158, 0, 0,     +1, 1.565851726602388e+02, 0,_
 2059, 60,  159, 0, 3.5,   +1, 1.575811030076162e+02, 0,_
 2060, 60,  160, 0, 0,     +1, 1.585753578729049e+02, 0,_
 2061, 60,  161, 0, 0.5,   -1, 1.595716054713287e+02, 0,_
 2062, 61,  126, 0,-77.777,+0, 1.248757184119980e+02, 0,_
 2063, 61,  127, 0, 2.5,   +1, 1.258613420660058e+02, 0,_
 2064, 61,  128, 0, 6,     +1, 1.268495592473179e+02, 0,_
 2065, 61,  129, 0, 2.5,   +1, 1.278355457572100e+02, 0,_
 2066, 61,  130, 0,-77.777,+0, 1.288242784714724e+02, 0,_
 2067, 61,  131, 0, 2.5,   +1, 1.298111493186564e+02, 0,_
 2068, 61,  132, 0, 3,     +1, 1.308004649817166e+02, 0,_
 2069, 61,  133, 0, 1.5,   +1, 1.317878513618509e+02, 0,_
 2070, 61,  133, 1, 5.5,   -1, 1.317879901494192e+02, 0.1304,_
 2071, 61,  134, 0, 2,     +1, 1.327778441576132e+02, 0,_
 2072, 61,  134, 1, 2,     +1, 1.327778441576132e+02, 0,_
 2073, 61,  135, 0,-77.777,+0, 1.337657272339170e+02, 0,_
 2074, 61,  135, 1, 5.5,   -1, 1.337657804500091e+02, 0.05,_
 2075, 61,  136, 0, 5,     -1, 1.347559470118975e+02, 0,_
 2076, 61,  136, 1, 2,     +1, 1.347562450220136e+02, 0.28,_
 2077, 61,  137, 0, 2.5,   +1, 1.357442668764678e+02, 0,_
 2078, 61,  137, 1, 5.5,   -1, 1.357444584543995e+02, 0.18,_
 2079, 61,  138, 0, 1,     +1, 1.367347529529938e+02, 0,_
 2080, 61,  138, 1, 5,     -1, 1.367347848826491e+02, 0.03,_
 2081, 61,  139, 0, 2.5,   +1, 1.377234378109273e+02, 0,_
 2082, 61,  139, 1, 5.5,   -1, 1.377236386484591e+02, 0.1887,_
 2083, 61,  140, 0, 1,     +1, 1.387140882244868e+02, 0,_
 2084, 61,  140, 1, 8,     -1, 1.387145458828793e+02, 0.43,_
 2085, 61,  141, 0, 2.5,   +1, 1.397030400025795e+02, 0,_
 2086, 61,  142, 0, 1,     +1, 1.406937905762600e+02, 0,_
 2087, 61,  142, 1, 8,     -1, 1.406947305533821e+02, 0.88317,_
 2088, 61,  143, 0, 2.5,   +1, 1.416832645128930e+02, 0,_
 2089, 61,  144, 0, 5,     -1, 1.426763179395463e+02, 0,_
 2090, 61,  145, 0, 2.5,   +1, 1.436678856379101e+02, 0,_
 2091, 61,  146, 0, 3,     -1, 1.446612249493954e+02, 0,_
 2092, 61,  147, 0, 3.5,   +1, 1.456530730213457e+02, 0,_
 2093, 61,  148, 0, 1,     -1, 1.466467990856504e+02, 0,_
 2094, 61,  148, 1, 6,     -1, 1.466469448977429e+02, 0.137,_
 2095, 61,  149, 0, 3.5,   +1, 1.476390616173419e+02, 0,_
 2096, 61,  150, 0, 1,     -1, 1.486330969439151e+02, 0,_
 2097, 61,  151, 0, 2.5,   +1, 1.496247311172786e+02, 0,_
 2098, 61,  152, 0, 1,     +1, 1.506184096405226e+02, 0,_
 2099, 61,  152, 1, 4,     -1, 1.506185692887991e+02, 0.15,_
 2100, 61,  152, 2, 8,     -1, 1.506187502235124e+02, 0.32,_
 2101, 61,  153, 0, 2.5,   -1, 1.516104645234730e+02, 0,_
 2102, 61,  154, 0,-77.777,+0, 1.526041472740872e+02, 0,_
 2103, 61,  154, 1,-77.777,+0, 1.526042004901793e+02, 0.05,_
 2104, 61,  155, 0, 2.5,   -1, 1.535972299562440e+02, 0,_
 2105, 61,  156, 0, 4,     -1, 1.545915943385766e+02, 0,_
 2106, 61,  157, 0, 2.5,   -1, 1.555849905242519e+02, 0,_
 2107, 61,  158, 0,-77.777,+0, 1.565798142117440e+02, 0,_
 2108, 61,  159, 0, 2.5,   -1, 1.575739217119388e+02, 0,_
 2109, 61,  160, 0, 1,     -1, 1.585691119929291e+02, 0,_
 2110, 61,  161, 0, 2.5,   -1, 1.595634659991698e+02, 0,_
 2111, 61,  162, 0,-77.777,+0, 1.605589898686439e+02, 0,_
 2112, 61,  163, 0, 2.5,   -1, 1.615537206105022e+02, 0,_
 2113, 62,  128, 0, 0,     +1, 1.268592611808803e+02, 0,_
 2114, 62,  129, 0, 2.5,   +1, 1.278471264118128e+02, 0,_
 2115, 62,  130, 0, 0,     +1, 1.288326757101066e+02, 0,_
 2116, 62,  131, 0, 2.5,   +1, 1.298212894552266e+02, 0,_
 2117, 62,  132, 0, 0,     +1, 1.308074345906419e+02, 0,_
 2118, 62,  133, 0, 2.5,   +1, 1.317965539546171e+02, 0,_
 2119, 62,  134, 0, 0,     +1, 1.327835517022197e+02, 0,_
 2120, 62,  135, 0, 3.5,   +1, 1.337733848810493e+02, 0,_
 2121, 62,  135, 1,-77.777,+0, 1.337733848810493e+02, 0,_
 2122, 62,  136, 0, 0,     +1, 1.347605864178356e+02, 0,_
 2123, 62,  137, 0, 4.5,   -1, 1.357507021107049e+02, 0,_
 2124, 62,  137, 1, 0.5,   +1, 1.357508936886366e+02, 0.18,_
 2125, 62,  138, 0, 0,     +1, 1.367384171163270e+02, 0,_
 2126, 62,  139, 0, 0.5,   +1, 1.377288874185361e+02, 0,_
 2127, 62,  139, 1, 5.5,   -1, 1.377293742393471e+02, 0.4574,_
 2128, 62,  140, 0, 0,     +1, 1.387170233865545e+02, 0,_
 2129, 62,  141, 0, 0.5,   +1, 1.397079241885195e+02, 0,_
 2130, 62,  141, 1, 5.5,   -1, 1.397081115091639e+02, 0.176,_
 2131, 62,  142, 0, 0,     +1, 1.406960848009958e+02, 0,_
 2132, 62,  143, 0, 1.5,   +1, 1.416869295040531e+02, 0,_
 2133, 62,  143, 1, 5.5,   -1, 1.416877319920795e+02, 0.75399,_
 2134, 62,  143, 2, 11.5,  -1, 1.416899030064181e+02, 2.7938,_
 2135, 62,  144, 0, 0,     +1, 1.426757331556495e+02, 0,_
 2136, 62,  145, 0, 3.5,   -1, 1.436685414265567e+02, 0,_
 2137, 62,  146, 0, 0,     +1, 1.446595837651190e+02, 0,_
 2138, 62,  147, 0, 3.5,   -1, 1.456528345139919e+02, 0,_
 2139, 62,  148, 0, 0,     +1, 1.466441696266909e+02, 0,_
 2140, 62,  149, 0, 3.5,   -1, 1.476379212158251e+02, 0,_
 2141, 62,  150, 0, 0,     +1, 1.486294207756040e+02, 0,_
 2142, 62,  151, 0, 2.5,   -1, 1.496234643427731e+02, 0,_
 2143, 62,  152, 0, 0,     +1, 1.506146755550588e+02, 0,_
 2144, 62,  153, 0, 1.5,   +1, 1.516084296881498e+02, 0,_
 2145, 62,  153, 1, 5.5,   -1, 1.516085343854895e+02, 0.09837,_
 2146, 62,  154, 0, 0,     +1, 1.525999504393072e+02, 0,_
 2147, 62,  155, 0, 1.5,   -1, 1.535937699637873e+02, 0,_
 2148, 62,  156, 0, 0,     +1, 1.545860632856844e+02, 0,_
 2149, 62,  157, 0, 1.5,   -1, 1.555803282256765e+02, 0,_
 2150, 62,  158, 0, 0,     +1, 1.565732568883988e+02, 0,_
 2151, 62,  159, 0, 2.5,   -1, 1.575679045729417e+02, 0,_
 2152, 62,  160, 0, 0,     +1, 1.585614139815497e+02, 0,_
 2153, 62,  161, 0, 3.5,   +1, 1.595566155011583e+02, 0,_
 2154, 62,  162, 0, 0,     +1, 1.605503051212491e+02, 0,_
 2155, 62,  163, 0, 0.5,   -1, 1.615457695061521e+02, 0,_
 2156, 62,  164, 0, 0,     +1, 1.625399648868695e+02, 0,_
 2157, 62,  165, 0, 2.5,   -1, 1.635359448047229e+02, 0,_
 2158, 63,  130, 0, 2,     +1, 1.288473882273829e+02, 0,_
 2159, 63,  131, 0, 1.5,   +1, 1.298328512730484e+02, 0,_
 2160, 63,  132, 0,-77.777,+0, 1.308211418186650e+02, 0,_
 2161, 63,  133, 0, 5.5,   -1, 1.318071917787664e+02, 0,_
 2162, 63,  134, 0,-77.777,+0, 1.327957361252200e+02, 0,_
 2163, 63,  135, 0, 5.5,   -1, 1.337826545600609e+02, 0,_
 2164, 63,  136, 0, 3,     +1, 1.347718334086073e+02, 0,_
 2165, 63,  136, 1, 3,     +1, 1.347718334086073e+02, 0,_
 2166, 63,  137, 0, 5.5,   -1, 1.357590896384827e+02, 0,_
 2167, 63,  138, 0, 6,     -1, 1.367487922268675e+02, 0,_
 2168, 63,  139, 0, 5.5,   -1, 1.377363187030811e+02, 0,_
 2169, 63,  140, 0, 1,     +1, 1.387260381940068e+02, 0,_
 2170, 63,  140, 1, 5,     -1, 1.387262617015939e+02, 0.21,_
 2171, 63,  141, 0, 2.5,   +1, 1.397143189326004e+02, 0,_
 2172, 63,  141, 1, 5.5,   -1, 1.397144215864421e+02, 0.09645,_
 2173, 63,  142, 0, 1,     +1, 1.407042513426002e+02, 0,_
 2174, 63,  142, 1, 8,     -1, 1.407047409306480e+02, 0.46,_
 2175, 63,  143, 0, 2.5,   +1, 1.416925447087768e+02, 0,_
 2176, 63,  144, 0, 1,     +1, 1.426824877714720e+02, 0,_
 2177, 63,  145, 0, 2.5,   +1, 1.436713723210938e+02, 0,_
 2178, 63,  146, 0, 4,     -1, 1.446637120221196e+02, 0,_
 2179, 63,  147, 0, 2.5,   +1, 1.456546668489636e+02, 0,_
 2180, 63,  148, 0, 5,     -1, 1.466474019013035e+02, 0,_
 2181, 63,  149, 0, 2.5,   +1, 1.476386605208096e+02, 0,_
 2182, 63,  150, 0, 5,     -1, 1.486318249773979e+02, 0,_
 2183, 63,  150, 1, 0,     -1, 1.486318697853475e+02, 0.0421,_
 2184, 63,  151, 0, 2.5,   +1, 1.496233828429621e+02, 0,_
 2185, 63,  152, 0, 3,     -1, 1.506166704643378e+02, 0,_
 2186, 63,  152, 1, 0,     -1, 1.506167189974138e+02, 0.0456,_
 2187, 63,  152, 2, 8,     -1, 1.506168277711062e+02, 0.1478,_
 2188, 63,  153, 0, 2.5,   +1, 1.516075702094603e+02, 0,_
 2189, 63,  154, 0, 3,     -1, 1.526007136174330e+02, 0,_
 2190, 63,  154, 1, 8,     -1, 1.526008807159623e+02, 0.157,_
 2191, 63,  155, 0, 2.5,   +1, 1.535920380219178e+02, 0,_
 2192, 63,  156, 0, 0,     +1, 1.545852947192478e+02, 0,_
 2193, 63,  157, 0, 2.5,   +1, 1.555773679938362e+02, 0,_
 2194, 63,  158, 0, 1,     -1, 1.565711229805308e+02, 0,_
 2195, 63,  159, 0, 2.5,   +1, 1.575638223550163e+02, 0,_
 2196, 63,  160, 0, 1,     -1, 1.585579595439606e+02, 0,_
 2197, 63,  161, 0, 2.5,   +1, 1.595511666380981e+02, 0,_
 2198, 63,  162, 0,-77.777,+0, 1.605458629155872e+02, 0,_
 2199, 63,  163, 0, 2.5,   +1, 1.615396336498306e+02, 0,_
 2200, 63,  164, 0,-77.777,+0, 1.625343465691199e+02, 0,_
 2201, 63,  165, 0, 2.5,   +1, 1.635285845804467e+02, 0,_
 2202, 63,  166, 0,-77.777,+0, 1.645237356799414e+02, 0,_
 2203, 63,  167, 0, 2.5,   +1, 1.655185487087898e+02, 0,_
 2204, 64,  134, 0, 0,     +1, 1.328049165773749e+02, 0,_
 2205, 64,  135, 0, 1.5,   -1, 1.337930395747826e+02, 0,_
 2206, 64,  136, 0, 0,     +1, 1.347794474337207e+02, 0,_
 2207, 64,  137, 0, 3.5,   +1, 1.357685965399815e+02, 0,_
 2208, 64,  138, 0, 0,     +1, 1.367551243594716e+02, 0,_
 2209, 64,  139, 0, 4.5,   -1, 1.377445847683216e+02, 0,_
 2210, 64,  139, 1, 0.5,   +1, 1.377448508487824e+02, 0.25,_
 2211, 64,  140, 0, 0,     +1, 1.387315765674670e+02, 0,_
 2212, 64,  141, 0, 0.5,   +1, 1.397214513854965e+02, 0,_
 2213, 64,  141, 1, 5.5,   -1, 1.397218534862888e+02, 0.3778,_
 2214, 64,  142, 0, 0,     +1, 1.407088853532877e+02, 0,_
 2215, 64,  143, 0, 0.5,   +1, 1.416989412839911e+02, 0,_
 2216, 64,  143, 1, 5.5,   -1, 1.416991036995043e+02, 0.1526,_
 2217, 64,  144, 0, 0,     +1, 1.426865956599640e+02, 0,_
 2218, 64,  145, 0, 0.5,   +1, 1.436767633106234e+02, 0,_
 2219, 64,  145, 1, 5.5,   -1, 1.436775605941160e+02, 0.7491,_
 2220, 64,  146, 0, 0,     +1, 1.446648101459689e+02, 0,_
 2221, 64,  147, 0, 3.5,   -1, 1.456569953834086e+02, 0,_
 2222, 64,  148, 0, 0,     +1, 1.466474338336127e+02, 0,_
 2223, 64,  149, 0, 3.5,   -1, 1.476400591468329e+02, 0,_
 2224, 64,  150, 0, 0,     +1, 1.486307907756688e+02, 0,_
 2225, 64,  151, 0, 3.5,   -1, 1.496238768117640e+02, 0,_
 2226, 64,  152, 0, 0,     +1, 1.506147348235027e+02, 0,_
 2227, 64,  153, 0, 1.5,   -1, 1.516080860556961e+02, 0,_
 2228, 64,  154, 0, 0,     +1, 1.525986192083117e+02, 0,_
 2229, 64,  155, 0, 1.5,   -1, 1.535917700379761e+02, 0,_
 2230, 64,  155, 1, 5.5,   -1, 1.535918988741352e+02, 0.12105,_
 2231, 64,  156, 0, 0,     +1, 1.545826846126022e+02, 0,_
 2232, 64,  157, 0, 1.5,   -1, 1.555759156572109e+02, 0,_
 2233, 64,  158, 0, 0,     +1, 1.565674677180723e+02, 0,_
 2234, 64,  159, 0, 1.5,   -1, 1.575611422330476e+02, 0,_
 2235, 64,  160, 0, 0,     +1, 1.585532113080253e+02, 0,_
 2236, 64,  161, 0, 2.5,   -1, 1.595472134283280e+02, 0,_
 2237, 64,  162, 0, 0,     +1, 1.605399271911398e+02, 0,_
 2238, 64,  163, 0, 2.5,   -1, 1.615344940391247e+02, 0,_
 2239, 64,  164, 0, 0,     +1, 1.625275425255844e+02, 0,_
 2240, 64,  165, 0, 0.5,   -1, 1.635224864204893e+02, 0,_
 2241, 64,  166, 0, 0,     +1, 1.645159432011144e+02, 0,_
 2242, 64,  167, 0, 2.5,   -1, 1.655113084450654e+02, 0,_
 2243, 64,  168, 0, 0,     +1, 1.665053253720693e+02, 0,_
 2244, 64,  169, 0, 3.5,   -1, 1.675012160630659e+02, 0,_
 2245, 65,  136, 0,-77.777,+0, 1.347932409143720e+02, 0,_
 2246, 65,  137, 0, 5.5,   -1, 1.357795020447011e+02, 0,_
 2247, 65,  138, 0,-77.777,+0, 1.367680364770596e+02, 0,_
 2248, 65,  139, 0, 5.5,   -1, 1.377546971454253e+02, 0,_
 2249, 65,  140, 0,-77.777,+0, 1.387436034049328e+02, 0,_
 2250, 65,  141, 0, 2.5,   -1, 1.397306933050416e+02, 0,_
 2251, 65,  141, 1, 5.5,   -1, 1.397306933050416e+02, 0,_
 2252, 65,  142, 0, 1,     +1, 1.407199543007894e+02, 0,_
 2253, 65,  142, 1, 5,     -1, 1.407202525237698e+02, 0.2802,_
 2254, 65,  143, 0, 5.5,   -1, 1.417072558916159e+02, 0,_
 2255, 65,  143, 1, 2.5,   +1, 1.417072558916159e+02, 0,_
 2256, 65,  144, 0, 1,     +1, 1.426965910507443e+02, 0,_
 2257, 65,  144, 1, 6,     -1, 1.426970134800838e+02, 0.3969,_
 2258, 65,  145, 0, 1.5,   +1, 1.436837217513205e+02, 0,_
 2259, 65,  145, 1, 5.5,   -1, 1.436837217513205e+02, 0,_
 2260, 65,  146, 0, 1,     +1, 1.446736676167303e+02, 0,_
 2261, 65,  146, 1, 5,     -1, 1.446738272650068e+02, 0.15,_
 2262, 65,  146, 2, 10,    +1, 1.446746574360444e+02, 0.93,_
 2263, 65,  147, 0, 0.5,   +1, 1.456619064623232e+02, 0,_
 2264, 65,  147, 1, 5.5,   -1, 1.456619603170085e+02, 0.0506,_
 2265, 65,  148, 0, 2,     -1, 1.466535347893449e+02, 0,_
 2266, 65,  148, 1, 9,     +1, 1.466536306847429e+02, 0.0901,_
 2267, 65,  149, 0, 0.5,   +1, 1.476439315150879e+02, 0,_
 2268, 65,  149, 1, 5.5,   -1, 1.476439740879617e+02, 0.04,_
 2269, 65,  150, 0, 2,     -1, 1.486357486144134e+02, 0,_
 2270, 65,  150, 1, 9,     +1, 1.486362275592428e+02, 0.45,_
 2271, 65,  151, 0, 0.5,   +1, 1.496266070465098e+02, 0,_
 2272, 65,  151, 1, 5.5,   -1, 1.496267129891060e+02, 0.09954,_
 2273, 65,  152, 0, 2,     -1, 1.506189814676883e+02, 0,_
 2274, 65,  152, 1, 8,     +1, 1.506195154805299e+02, 0.50174,_
 2275, 65,  153, 0, 2.5,   +1, 1.516097562030102e+02, 0,_
 2276, 65,  154, 0, 0,     +1, 1.526023971805008e+02, 0,_
 2277, 65,  154, 1, 3,     -1, 1.526024099523629e+02, 0.012,_
 2278, 65,  154, 2, 7,     -1, 1.526026100448694e+02, 0.2,_
 2279, 65,  155, 0, 1.5,   +1, 1.535926426022799e+02, 0,_
 2280, 65,  156, 0, 3,     -1, 1.545852859403165e+02, 0,_
 2281, 65,  156, 1, 7,     -1, 1.545853497996271e+02, 0.06,_
 2282, 65,  156, 2, 0,     +1, 1.545853800263675e+02, 0.0884,_
 2283, 65,  157, 0, 1.5,   +1, 1.555759795614857e+02, 0,_
 2284, 65,  158, 0, 3,     -1, 1.565687649972574e+02, 0,_
 2285, 65,  158, 1, 0,     +1, 1.565688823919567e+02, 0.1103,_
 2286, 65,  159, 0, 1.5,   +1, 1.575601088531970e+02, 0,_
 2287, 65,  160, 0, 3,     -1, 1.585533235762307e+02, 0,_
 2288, 65,  161, 0, 1.5,   +1, 1.595451318639084e+02, 0,_
 2289, 65,  162, 0, 1,     -1, 1.605384418703197e+02, 0,_
 2290, 65,  163, 0, 1.5,   +1, 1.615310007372910e+02, 0,_
 2291, 65,  164, 0, 5,     +1, 1.625250903326299e+02, 0,_
 2292, 65,  165, 0, 1.5,   +1, 1.635181093474587e+02, 0,_
 2293, 65,  166, 0,-77.777,+0, 1.645123722620012e+02, 0,_
 2294, 65,  167, 0, 1.5,   +1, 1.655058656068009e+02, 0,_
 2295, 65,  168, 0, 4,     -1, 1.665006855755159e+02, 0,_
 2296, 65,  169, 0, 1.5,   +1, 1.674946430179485e+02, 0,_
 2297, 65,  170, 0,-77.777,+0, 1.684898892927571e+02, 0,_
 2298, 65,  171, 0, 1.5,   +1, 1.694841639862362e+02, 0,_
 2299, 66,  138, 0, 0,     +1, 1.367773358983569e+02, 0,_
 2300, 66,  139, 0, 3.5,   +1, 1.377658604166201e+02, 0,_
 2301, 66,  140, 0, 0,     +1, 1.387517477855602e+02, 0,_
 2302, 66,  141, 0, 4.5,   -1, 1.397404408434419e+02, 0,_
 2303, 66,  142, 0, 0,     +1, 1.407268080545897e+02, 0,_
 2304, 66,  143, 0, 0.5,   +1, 1.417160368057343e+02, 0,_
 2305, 66,  143, 1, 5.5,   -1, 1.417163674905309e+02, 0.3107,_
 2306, 66,  144, 0, 0,     +1, 1.427027620931811e+02, 0,_
 2307, 66,  145, 0, 0.5,   +1, 1.436923915175261e+02, 0,_
 2308, 66,  145, 1, 5.5,   -1, 1.436925173203680e+02, 0.1182,_
 2309, 66,  146, 0, 0,     +1, 1.446792113418191e+02, 0,_
 2310, 66,  146, 1, 10,    +1, 1.446823358714537e+02, 2.9357,_
 2311, 66,  147, 0, 0.5,   +1, 1.456688741826135e+02, 0,_
 2312, 66,  147, 1, 5.5,   -1, 1.456696729561567e+02, 0.7505,_
 2313, 66,  148, 0, 0,     +1, 1.466563845454481e+02, 0,_
 2314, 66,  149, 0, 3.5,   -1, 1.476479682322733e+02, 0,_
 2315, 66,  149, 1, 13.5,  -1, 1.476508004991298e+02, 2.6611,_
 2316, 66,  150, 0, 0,     +1, 1.486376602661124e+02, 0,_
 2317, 66,  151, 0, 3.5,   -1, 1.496296628204855e+02, 0,_
 2318, 66,  152, 0, 0,     +1, 1.506196190431506e+02, 0,_
 2319, 66,  153, 0, 3.5,   -1, 1.516120662010714e+02, 0,_
 2320, 66,  154, 0, 0,     +1, 1.526021442937432e+02, 0,_
 2321, 66,  155, 0, 1.5,   -1, 1.535948718242591e+02, 0,_
 2322, 66,  156, 0, 0,     +1, 1.545848195892096e+02, 0,_
 2323, 66,  157, 0, 1.5,   -1, 1.555774045530537e+02, 0,_
 2324, 66,  157, 1, 5.5,   -1, 1.555776168852614e+02, 0.1995,_
 2325, 66,  158, 0, 0,     +1, 1.565677680665778e+02, 0,_
 2326, 66,  159, 0, 1.5,   -1, 1.575604975749557e+02, 0,_
 2327, 66,  160, 0, 0,     +1, 1.585513689786542e+02, 0,_
 2328, 66,  161, 0, 2.5,   +1, 1.595444994308873e+02, 0,_
 2329, 66,  162, 0, 0,     +1, 1.605357751898812e+02, 0,_
 2330, 66,  163, 0, 2.5,   -1, 1.615291008179323e+02, 0,_
 2331, 66,  164, 0, 0,     +1, 1.625209501202219e+02, 0,_
 2332, 66,  165, 0, 3.5,   +1, 1.635148665004550e+02, 0,_
 2333, 66,  165, 1, 0.5,   -1, 1.635149816175056e+02, 0.10816,_
 2334, 66,  166, 0, 0,     +1, 1.645073699486256e+02, 0,_
 2335, 66,  167, 0, 0.5,   -1, 1.655016043532045e+02, 0,_
 2336, 66,  168, 0, 0,     +1, 1.664944731219034e+02, 0,_
 2337, 66,  169, 0, 2.5,   -1, 1.674890355769527e+02, 0,_
 2338, 66,  170, 0, 0,     +1, 1.684825032918334e+02, 0,_
 2339, 66,  171, 0, 3.5,   -1, 1.694776107693092e+02, 0,_
 2340, 66,  172, 0, 0,     +1, 1.704713401875523e+02, 0,_
 2341, 66,  173, 0, 4.5,   +1, 1.714670821671209e+02, 0,_
 2342, 67,  140, 0, 8,     +1, 1.387661916308566e+02, 0,_
 2343, 67,  141, 0, 3.5,   -1, 1.397521672352441e+02, 0,_
 2344, 67,  142, 0,-77.777,+0, 1.407405053685175e+02, 0,_
 2345, 67,  143, 0, 5.5,   -1, 1.417268091294560e+02, 0,_
 2346, 67,  144, 0,-77.777,+0, 1.427154919896994e+02, 0,_
 2347, 67,  145, 0, 5.5,   -1, 1.437021007875190e+02, 0,_
 2348, 67,  145, 1, 2.5,   +1, 1.437022072197033e+02, 0.1,_
 2349, 67,  146, 0, 10,    +1, 1.446912559532748e+02, 0,_
 2350, 67,  147, 0, 5.5,   -1, 1.456778559364724e+02, 0,_
 2351, 67,  148, 0, 1,     +1, 1.466668876925625e+02, 0,_
 2352, 67,  148, 1, 5,     -1, 1.466671537730233e+02, 0.25,_
 2353, 67,  148, 2, 10,    +1, 1.466678928381111e+02, 0.9444,_
 2354, 67,  149, 0, 5.5,   -1, 1.476544066677717e+02, 0,_
 2355, 67,  149, 1, 0.5,   +1, 1.476544586066776e+02, 0.0488,_
 2356, 67,  150, 0, 2,     -1, 1.486454976339795e+02, 0,_
 2357, 67,  150, 1, 9,     +1, 1.486454976339795e+02, 0,_
 2358, 67,  151, 0, 5.5,   -1, 1.496351224373648e+02, 0,_
 2359, 67,  151, 1, 0.5,   +1, 1.496351660745604e+02, 0.041,_
 2360, 67,  152, 0, 2,     -1, 1.506265510796380e+02, 0,_
 2361, 67,  152, 1, 9,     +1, 1.506267213711328e+02, 0.16,_
 2362, 67,  153, 0, 5.5,   -1, 1.516164627453851e+02, 0,_
 2363, 67,  153, 1, 0.5,   +1, 1.516165358642957e+02, 0.0687,_
 2364, 67,  154, 0, 2,     -1, 1.526082690363640e+02, 0,_
 2365, 67,  154, 1, 8,     +1, 1.526085244736063e+02, 0.24,_
 2366, 67,  155, 0, 2.5,   +1, 1.535981882626000e+02, 0,_
 2367, 67,  156, 0, 4,     -1, 1.545901944147983e+02, 0,_
 2368, 67,  156, 1, 1,     -1, 1.545902501852629e+02, 0.0524,_
 2369, 67,  156, 2, 9,     +1, 1.545903008469826e+02, 0.1,_
 2370, 67,  157, 0, 3.5,   -1, 1.555801629834862e+02, 0,_
 2371, 67,  158, 0, 5,     +1, 1.565722592458872e+02, 0,_
 2372, 67,  158, 1, 2,     -1, 1.565723307683150e+02, 0.0672,_
 2373, 67,  158, 2, 9,     +1, 1.565724508238189e+02, 0.18,_
 2374, 67,  159, 0, 3.5,   -1, 1.575624533731292e+02, 0,_
 2375, 67,  159, 1, 0.5,   +1, 1.575626724957102e+02, 0.20588,_
 2376, 67,  160, 0, 5,     +1, 1.585548705974223e+02, 0,_
 2377, 67,  160, 1, 2,     -1, 1.585549344354465e+02, 0.05998,_
 2378, 67,  160, 2, 9,     +1, 1.585550834617909e+02, 0.2,_
 2379, 67,  161, 0, 3.5,   -1, 1.595454131842910e+02, 0,_
 2380, 67,  161, 1, 0.5,   +1, 1.595456379052049e+02, 0.21114,_
 2381, 67,  162, 0, 1,     +1, 1.605380527718254e+02, 0,_
 2382, 67,  162, 1, 6,     -1, 1.605381654515789e+02, 0.10587,_
 2383, 67,  163, 0, 3.5,   -1, 1.615291038338001e+02, 0,_
 2384, 67,  163, 1, 0.5,   +1, 1.615294208739907e+02, 0.29788,_
 2385, 67,  164, 0, 1,     +1, 1.625220000338090e+02, 0,_
 2386, 67,  164, 1, 6,     -1, 1.625221490388670e+02, 0.14,_
 2387, 67,  165, 0, 3.5,   -1, 1.635134973569681e+02, 0,_
 2388, 67,  166, 0, 0,     -1, 1.645068521136223e+02, 0,_
 2389, 67,  166, 1, 7,     -1, 1.645068584782669e+02, 0.00598,_
 2390, 67,  167, 0, 3.5,   -1, 1.654991031965167e+02, 0,_
 2391, 67,  168, 0, 3,     +1, 1.664928749301009e+02, 0,_
 2392, 67,  168, 1, 6,     +1, 1.664929377250896e+02, 0.059,_
 2393, 67,  169, 0, 3.5,   -1, 1.674856297471810e+02, 0,_
 2394, 67,  170, 0, 6,     +1, 1.684797623310280e+02, 0,_
 2395, 67,  170, 1, 1,     +1, 1.684798900496492e+02, 0.12,_
 2396, 67,  171, 0, 3.5,   -1, 1.694730021922416e+02, 0,_
 2397, 67,  172, 0,-77.777,+0, 1.704676422300428e+02, 0,_
 2398, 67,  173, 0, 3.5,   -1, 1.714613220778099e+02, 0,_
 2399, 67,  174, 0,-77.777,+0, 1.724566278371897e+02, 0,_
 2400, 67,  175, 0, 3.5,   -1, 1.734506844205743e+02, 0,_
 2401, 68,  143, 0, 4.5,   -1, 1.417382906431068e+02, 0,_
 2402, 68,  144, 0, 0,     +1, 1.427240084810190e+02, 0,_
 2403, 68,  145, 0, 0.5,   +1, 1.437126162776819e+02, 0,_
 2404, 68,  146, 0, 0,     +1, 1.446986170232233e+02, 0,_
 2405, 68,  147, 0, 0.5,   +1, 1.456875937223772e+02, 0,_
 2406, 68,  147, 1, 5.5,   -1, 1.456877001545616e+02, 0.1,_
 2407, 68,  148, 0, 0,     +1, 1.466738187366490e+02, 0,_
 2408, 68,  149, 0, 0.5,   +1, 1.476628200941405e+02, 0,_
 2409, 68,  149, 1, 5.5,   -1, 1.476636096080836e+02, 0.7418,_
 2410, 68,  150, 0, 0,     +1, 1.486498768583689e+02, 0,_
 2411, 68,  151, 0, 3.5,   -1, 1.496408234287543e+02, 0,_
 2412, 68,  151, 1, 13.5,  -1, 1.496435752328795e+02, 2.5855,_
 2413, 68,  152, 0, 0,     +1, 1.506298551541108e+02, 0,_
 2414, 68,  153, 0, 3.5,   -1, 1.516212984910559e+02, 0,_
 2415, 68,  154, 0, 0,     +1, 1.526104341775975e+02, 0,_
 2416, 68,  155, 0, 3.5,   -1, 1.536022649881166e+02, 0,_
 2417, 68,  156, 0, 0,     +1, 1.545915431818454e+02, 0,_
 2418, 68,  157, 0, 1.5,   -1, 1.555838021067891e+02, 0,_
 2419, 68,  157, 1, 4.5,   +1, 1.555839675024035e+02, 0.1554,_
 2420, 68,  158, 0, 0,     +1, 1.565731998773943e+02, 0,_
 2421, 68,  159, 0, 1.5,   -1, 1.575653999483033e+02, 0,_
 2422, 68,  160, 0, 0,     +1, 1.585552095861138e+02, 0,_
 2423, 68,  161, 0, 1.5,   -1, 1.595475372117717e+02, 0,_
 2424, 68,  162, 0, 0,     +1, 1.605377409487461e+02, 0,_
 2425, 68,  163, 0, 2.5,   -1, 1.615303923152171e+02, 0,_
 2426, 68,  164, 0, 0,     +1, 1.625209768089661e+02, 0,_
 2427, 68,  165, 0, 2.5,   -1, 1.635138990275265e+02, 0,_
 2428, 68,  166, 0, 0,     +1, 1.645048781022645e+02, 0,_
 2429, 68,  167, 0, 3.5,   +1, 1.654980276411137e+02, 0,_
 2430, 68,  167, 1, 0.5,   -1, 1.654982488082570e+02, 0.207801,_
 2431, 68,  168, 0, 0,     +1, 1.664897564673184e+02, 0,_
 2432, 68,  169, 0, 0.5,   -1, 1.674833670760360e+02, 0,_
 2433, 68,  170, 0, 0,     +1, 1.684756434051738e+02, 0,_
 2434, 68,  171, 0, 2.5,   -1, 1.694695963614784e+02, 0,_
 2435, 68,  172, 0, 0,     +1, 1.704623206847845e+02, 0,_
 2436, 68,  173, 0, 3.5,   -1, 1.714567417658276e+02, 0,_
 2437, 68,  174, 0, 0,     +1, 1.724499655652155e+02, 0,_
 2438, 68,  175, 0, 4.5,   +1, 1.734448846748825e+02, 0,_
 2439, 68,  176, 0, 0,     +1, 1.744384455535072e+02, 0,_
 2440, 68,  177, 0, 0.5,   -1, 1.754338702820294e+02, 0,_
 2441, 69,  145, 0, 5.5,   -1, 1.437250237678243e+02, 0,_
 2442, 69,  146, 0, 6,     -1, 1.447127373131002e+02, 0,_
 2443, 69,  146, 1, 10,    +1, 1.447128128799511e+02, 0.071,_
 2444, 69,  147, 0, 5.5,   -1, 1.456989110903365e+02, 0,_
 2445, 69,  148, 0, 10,    +1, 1.466873504851870e+02, 0,_
 2446, 69,  149, 0, 5.5,   -1, 1.476733131724998e+02, 0,_
 2447, 69,  150, 0, 1,     +1, 1.486619467458102e+02, 0,_
 2448, 69,  150, 1, 6,     -1, 1.486620957508682e+02, 0.14,_
 2449, 69,  150, 2, 10,    +1, 1.486628088465031e+02, 0.81,_
 2450, 69,  151, 0, 5.5,   -1, 1.496487989554866e+02, 0,_
 2451, 69,  151, 1, 0.5,   +1, 1.496488990017398e+02, 0.094,_
 2452, 69,  152, 0, 2,     -1, 1.506392000126978e+02, 0,_
 2453, 69,  152, 1, 9,     +1, 1.506393064448822e+02, 0.1,_
 2454, 69,  153, 0, 5.5,   -1, 1.516282115549393e+02, 0,_
 2455, 69,  153, 1, 0.5,   +1, 1.516282575336429e+02, 0.0432,_
 2456, 69,  154, 0, 2,     -1, 1.526191380829899e+02, 0,_
 2457, 69,  154, 1, 9,     +1, 1.526192125855189e+02, 0.07,_
 2458, 69,  155, 0, 5.5,   -1, 1.536082073916899e+02, 0,_
 2459, 69,  155, 1, 0.5,   +1, 1.536082510288855e+02, 0.041,_
 2460, 69,  156, 0, 2,     -1, 1.545993948547606e+02, 0,_
 2461, 69,  157, 0, 0.5,   +1, 1.555888090669012e+02, 0,_
 2462, 69,  158, 0, 2,     -1, 1.565802250558147e+02, 0,_
 2463, 69,  159, 0, 2.5,   +1, 1.575696472706130e+02, 0,_
 2464, 69,  160, 0, 1,     -1, 1.585613424216106e+02, 0,_
 2465, 69,  160, 1, 5,     +1, 1.585614169241396e+02, 0.07,_
 2466, 69,  161, 0, 3.5,   +1, 1.595510525605702e+02, 0,_
 2467, 69,  161, 1, 0.5,   +1, 1.595510604365519e+02, 0.0074,_
 2468, 69,  162, 0, 1,     -1, 1.605429100707392e+02, 0,_
 2469, 69,  162, 1, 5,     +1, 1.605430484325788e+02, 0.13,_
 2470, 69,  163, 0, 0.5,   +1, 1.615329881961275e+02, 0,_
 2471, 69,  164, 0, 1,     +1, 1.625252754505982e+02, 0,_
 2472, 69,  164, 1, 6,     -1, 1.625252860938166e+02, 0.01,_
 2473, 69,  165, 0, 0.5,   +1, 1.635155934167862e+02, 0,_
 2474, 69,  166, 0, 2,     +1, 1.645081111571164e+02, 0,_
 2475, 69,  166, 1, 6,     -1, 1.645082410043812e+02, 0.122,_
 2476, 69,  167, 0, 0.5,   +1, 1.654988232631160e+02, 0,_
 2477, 69,  168, 0, 3,     +1, 1.664915426660567e+02, 0,_
 2478, 69,  169, 0, 0.5,   +1, 1.674829923202632e+02, 0,_
 2479, 69,  170, 0, 1,     -1, 1.684759763541985e+02, 0,_
 2480, 69,  171, 0, 0.5,   +1, 1.694680090930259e+02, 0,_
 2481, 69,  172, 0, 2,     -1, 1.704613726226630e+02, 0,_
 2482, 69,  173, 0, 0.5,   +1, 1.714539723941996e+02, 0,_
 2483, 69,  174, 0, 4,     -1, 1.724479272906927e+02, 0,_
 2484, 69,  175, 0, 0.5,   +1, 1.734409907285993e+02, 0,_
 2485, 69,  176, 0, 4,     +1, 1.744355285401821e+02, 0,_
 2486, 69,  177, 0, 3.5,   -1, 1.754289628049056e+02, 0,_
 2487, 69,  178, 0,-77.777,+0, 1.764239413991438e+02, 0,_
 2488, 69,  179, 0, 0.5,   +1, 1.774180277248141e+02, 0,_
 2489, 70,  148, 0, 0,     +1, 1.466963276696394e+02, 0,_
 2490, 70,  149, 0,-77.777,+0, 1.476846846396938e+02, 0,_
 2491, 70,  150, 0, 0,     +1, 1.486703043280635e+02, 0,_
 2492, 70,  151, 0, 0.5,   +1, 1.496586230872122e+02, 0,_
 2493, 70,  151, 1, 5.5,   -1, 1.496594213285944e+02, 0.75,_
 2494, 70,  152, 0, 0,     +1, 1.506450004523763e+02, 0,_
 2495, 70,  153, 0, 3.5,   -1, 1.516354119203789e+02, 0,_
 2496, 70,  154, 0, 0,     +1, 1.526239222624774e+02, 0,_
 2497, 70,  155, 0, 3.5,   -1, 1.536147245599753e+02, 0,_
 2498, 70,  156, 0, 0,     +1, 1.546031932380888e+02, 0,_
 2499, 70,  157, 0, 3.5,   -1, 1.555944365353608e+02, 0,_
 2500, 70,  158, 0, 0,     +1, 1.565830912296596e+02, 0,_
 2501, 70,  159, 0, 2.5,   -1, 1.575746834198042e+02, 0,_
 2502, 70,  160, 0, 0,     +1, 1.585636193492244e+02, 0,_
 2503, 70,  161, 0, 1.5,   -1, 1.595553729705812e+02, 0,_
 2504, 70,  162, 0, 0,     +1, 1.605446677396857e+02, 0,_
 2505, 70,  163, 0, 1.5,   -1, 1.615366384261913e+02, 0,_
 2506, 70,  164, 0, 0,     +1, 1.625262190959903e+02, 0,_
 2507, 70,  165, 0, 2.5,   -1, 1.635183970951492e+02, 0,_
 2508, 70,  166, 0, 0,     +1, 1.645084226153569e+02, 0,_
 2509, 70,  167, 0, 2.5,   -1, 1.655009019534394e+02, 0,_
 2510, 70,  168, 0, 0,     +1, 1.664912563846609e+02, 0,_
 2511, 70,  169, 0, 3.5,   +1, 1.674839477079097e+02, 0,_
 2512, 70,  169, 1, 0.5,   -1, 1.674839734634340e+02, 0.024199,_
 2513, 70,  170, 0, 0,     +1, 1.684749460199580e+02, 0,_
 2514, 70,  171, 0, 0.5,   -1, 1.694679063740770e+02, 0,_
 2515, 70,  171, 1, 3.5,   +1, 1.694680077847908e+02, 0.095282,_
 2516, 70,  172, 0, 0,     +1, 1.704593705613557e+02, 0,_
 2517, 70,  173, 0, 2.5,   -1, 1.714525939215491e+02, 0,_
 2518, 70,  174, 0, 0,     +1, 1.724446491792523e+02, 0,_
 2519, 70,  175, 0, 3.5,   -1, 1.734384523206901e+02, 0,_
 2520, 70,  175, 1, 0.5,   -1, 1.734390003027558e+02, 0.514865,_
 2521, 70,  176, 0, 0,     +1, 1.744311435329010e+02, 0,_
 2522, 70,  176, 1, 8,     -1, 1.744322608579718e+02, 1.0498,_
 2523, 70,  177, 0, 4.5,   +1, 1.754252190918639e+02, 0,_
 2524, 70,  177, 1, 0.5,   -1, 1.754255719145549e+02, 0.3315,_
 2525, 70,  178, 0, 0,     +1, 1.764180025686104e+02, 0,_
 2526, 70,  179, 0, 0.5,   -1, 1.774127732543582e+02, 0,_
 2527, 70,  180, 0, 0,     +1, 1.784062449061261e+02, 0,_
 2528, 70,  181, 0, 1.5,   -1, 1.794013920399827e+02, 0,_
 2529, 71,  150, 0,-77.777,+0, 1.486852032303296e+02, 0,_
 2530, 71,  151, 0, 5.5,   -1, 1.496707921850042e+02, 0,_
 2531, 71,  152, 0,-77.777,+0, 1.506586752613080e+02, 0,_
 2532, 71,  153, 0, 5.5,   -1, 1.516448154932119e+02, 0,_
 2533, 71,  153, 1, 0.5,   +1, 1.516449006389593e+02, 0.08,_
 2534, 71,  154, 0, 2,     -1, 1.526347963385238e+02, 0,_
 2535, 71,  154, 1, 9,     +1, 1.526348580691907e+02, 0.058,_
 2536, 71,  155, 0, 5.5,   -1, 1.536231939672740e+02, 0,_
 2537, 71,  155, 1, 0.5,   +1, 1.536232152537109e+02, 0.02,_
 2538, 71,  155, 2, 12.5,  -1, 1.536250895244764e+02, 1.781,_
 2539, 71,  156, 0, 2,     -1, 1.546133747293243e+02, 0,_
 2540, 71,  156, 1, 9,     +1, 1.546136088801297e+02, 0.22,_
 2541, 71,  157, 0,-77.777,+0, 1.556018669663021e+02, 0,_
 2542, 71,  157, 1, 5.5,   -1, 1.556018893170609e+02, 0.021,_
 2543, 71,  158, 0, 2,     -1, 1.565924551837853e+02, 0,_
 2544, 71,  159, 0, 0.5,   +1, 1.575812077153327e+02, 0,_
 2545, 71,  159, 1, 5.5,   -1, 1.575813141475170e+02, 0.1,_
 2546, 71,  160, 0, 2,     -1, 1.585720197970512e+02, 0,_
 2547, 71,  160, 1,-77.777,+0, 1.585720197970512e+02, 0,_
 2548, 71,  161, 0, 0.5,   +1, 1.595609894581889e+02, 0,_
 2549, 71,  161, 1, 4.5,   -1, 1.595611661356148e+02, 0.166,_
 2550, 71,  162, 0, 1,     -1, 1.605521122387282e+02, 0,_
 2551, 71,  162, 1, 4,     -1, 1.605522399573494e+02, 0.12,_
 2552, 71,  162, 2,-77.777,+0, 1.605524315352811e+02, 0.3,_
 2553, 71,  163, 0, 0.5,   +1, 1.615414360551403e+02, 0,_
 2554, 71,  164, 0, 1,     -1, 1.625330042006298e+02, 0,_
 2555, 71,  165, 0, 0.5,   +1, 1.635224980774825e+02, 0,_
 2556, 71,  166, 0, 6,     -1, 1.645143559544730e+02, 0,_
 2557, 71,  166, 1, 3,     -1, 1.645143925352147e+02, 0.03437,_
 2558, 71,  166, 2, 0,     -1, 1.645144016138801e+02, 0.0429,_
 2559, 71,  167, 0, 3.5,   +1, 1.655041901247121e+02, 0,_
 2560, 71,  167, 1, 0.5,   -1, 1.655041901247121e+02, 0,_
 2561, 71,  168, 0, 6,     -1, 1.664960607879111e+02, 0,_
 2562, 71,  168, 1, 3,     +1, 1.664962949387166e+02, 0.22,_
 2563, 71,  169, 0, 3.5,   +1, 1.674863881982662e+02, 0,_
 2564, 71,  169, 1, 0.5,   -1, 1.674864190635996e+02, 0.029,_
 2565, 71,  170, 0, 0,     +1, 1.684786261211906e+02, 0,_
 2566, 71,  170, 1, 4,     -1, 1.684787250073331e+02, 0.09291,_
 2567, 71,  171, 0, 3.5,   +1, 1.694694798827567e+02, 0,_
 2568, 71,  171, 1, 0.5,   -1, 1.694695556624719e+02, 0.0712,_
 2569, 71,  172, 0, 4,     -1, 1.704620520851774e+02, 0,_
 2570, 71,  172, 1, 1,     -1, 1.704620966376898e+02, 0.04186,_
 2571, 71,  173, 0, 3.5,   +1, 1.714533073467796e+02, 0,_
 2572, 71,  174, 0, 1,     -1, 1.724461118949440e+02, 0,_
 2573, 71,  174, 1, 6,     -1, 1.724462937130444e+02, 0.17083,_
 2574, 71,  175, 0, 3.5,   +1, 1.734379520534635e+02, 0,_
 2575, 71,  176, 0, 7,     -1, 1.744312596279473e+02, 0,_
 2576, 71,  176, 1, 1,     -1, 1.744313947968213e+02, 0.127,_
 2577, 71,  177, 0, 3.5,   +1, 1.754237317991303e+02, 0,_
 2578, 71,  177, 1, 11.5,  -1, 1.754247643775743e+02, 0.970175,_
 2579, 71,  177, 2, 19.5,  -1, 1.754278826543180e+02, 3.9,_
 2580, 71,  178, 0, 1,     +1, 1.764173189441588e+02, 0,_
 2581, 71,  178, 1, 9,     -1, 1.764174506007708e+02, 0.1237,_
 2582, 71,  179, 0, 3.5,   +1, 1.774100895900833e+02, 0,_
 2583, 71,  179, 1, 0.5,   +1, 1.774107200943430e+02, 0.5924,_
 2584, 71,  180, 0, 5,     +1, 1.784040349313713e+02, 0,_
 2585, 71,  180, 1, 3,     -1, 1.784040497254450e+02, 0.0139,_
 2586, 71,  180, 2, 9,     -1, 1.784046990682014e+02, 0.624,_
 2587, 71,  181, 0, 3.5,   +1, 1.793974442472742e+02, 0,_
 2588, 71,  182, 0,-77.777,+0, 1.803919588618571e+02, 0,_
 2589, 71,  183, 0, 3.5,   +1, 1.813856714261383e+02, 0,_
 2590, 71,  184, 0, 3,     +1, 1.823805974756720e+02, 0,_
 2591, 72,  153, 0, 0.5,   +1, 1.516566003246396e+02, 0,_
 2592, 72,  153, 1, 5.5,   -1, 1.516573985660219e+02, 0.75,_
 2593, 72,  154, 0, 0,     +1, 1.526422943687233e+02, 0,_
 2594, 72,  155, 0, 3.5,   -1, 1.536321077193625e+02, 0,_
 2595, 72,  156, 0, 0,     +1, 1.546196357610118e+02, 0,_
 2596, 72,  157, 0, 3.5,   -1, 1.556098894075243e+02, 0,_
 2597, 72,  158, 0, 0,     +1, 1.565978936558824e+02, 0,_
 2598, 72,  159, 0, 3.5,   -1, 1.575885047104840e+02, 0,_
 2599, 72,  160, 0, 0,     +1, 1.585766293685026e+02, 0,_
 2600, 72,  161, 0, 1.5,   -1, 1.595676389915421e+02, 0,_
 2601, 72,  162, 0, 0,     +1, 1.605560105799298e+02, 0,_
 2602, 72,  163, 0, 1.5,   -1, 1.615473193350154e+02, 0,_
 2603, 72,  164, 0, 0,     +1, 1.625360097022116e+02, 0,_
 2604, 72,  165, 0, 2.5,   -1, 1.635276139905266e+02, 0,_
 2605, 72,  166, 0, 0,     +1, 1.645166570159688e+02, 0,_
 2606, 72,  167, 0, 2.5,   -1, 1.655084829279335e+02, 0,_
 2607, 72,  168, 0, 0,     +1, 1.664978779037553e+02, 0,_
 2608, 72,  169, 0, 2.5,   -1, 1.674899724877000e+02, 0,_
 2609, 72,  170, 0, 0,     +1, 1.684797461819583e+02, 0,_
 2610, 72,  171, 0, 3.5,   +1, 1.694720311165308e+02, 0,_
 2611, 72,  171, 1, 0.5,   -1, 1.694720544571089e+02, 0.02193,_
 2612, 72,  172, 0, 0,     +1, 1.704624073062171e+02, 0,_
 2613, 72,  173, 0, 0.5,   -1, 1.714548709760635e+02, 0,_
 2614, 72,  174, 0, 0,     +1, 1.724458199664796e+02, 0,_
 2615, 72,  175, 0, 2.5,   -1, 1.734386799651699e+02, 0,_
 2616, 72,  176, 0, 0,     +1, 1.744299887361180e+02, 0,_
 2617, 72,  177, 0, 3.5,   -1, 1.754232030328456e+02, 0,_
 2618, 72,  177, 1, 11.5,  +1, 1.754246030950140e+02, 1.31545,_
 2619, 72,  177, 2, 18.5,  -1, 1.754261192746955e+02, 2.74,_
 2620, 72,  178, 0, 0,     +1, 1.764150865813942e+02, 0,_
 2621, 72,  178, 1, 8,     -1, 1.764163078055633e+02, 1.14742,_
 2622, 72,  178, 2, 16,    +1, 1.764176899658383e+02, 2.44605,_
 2623, 72,  179, 0, 4.5,   +1, 1.774085952940328e+02, 0,_
 2624, 72,  179, 1, 0.5,   -1, 1.774089944541038e+02, 0.375037,_
 2625, 72,  179, 2, 12.5,  -1, 1.774097722211268e+02, 1.1058,_
 2626, 72,  180, 0, 0,     +1, 1.784007323410385e+02, 0,_
 2627, 72,  180, 1, 8,     -1, 1.784019472431359e+02, 1.14148,_
 2628, 72,  181, 0, 0.5,   -1, 1.793946712401475e+02, 0,_
 2629, 72,  182, 0, 0,     +1, 1.803875211304264e+02, 0,_
 2630, 72,  182, 1, 8,     -1, 1.803887694522296e+02, 1.17288,_
 2631, 72,  183, 0, 1.5,   -1, 1.813818753230520e+02, 0,_
 2632, 72,  184, 0, 0,     +1, 1.823751829788313e+02, 0,_
 2633, 72,  184, 1, 8,     -1, 1.823765372219444e+02, 1.2724,_
 2634, 72,  185, 0, 1.5,   -1, 1.833699765889415e+02, 0,_
 2635, 72,  186, 0, 0,     +1, 1.843634036272809e+02, 0,_
 2636, 72,  187, 0, 1.5,   -1, 1.853586528763181e+02, 0,_
 2637, 72,  188, 0, 0,     +1, 1.863521245280860e+02, 0,_
 2638, 73,  155, 0, 5.5,   -1, 1.536430082670346e+02, 0,_
 2639, 73,  156, 0, 2,     -1, 1.546323606122470e+02, 0,_
 2640, 73,  156, 1, 9,     +1, 1.546324670444313e+02, 0.1,_
 2641, 73,  157, 0, 0.5,   +1, 1.556197978031106e+02, 0,_
 2642, 73,  157, 1, 5.5,   -1, 1.556198212181911e+02, 0.022,_
 2643, 73,  157, 2, 12.5,  -1, 1.556214932678065e+02, 1.593,_
 2644, 73,  158, 0, 2,     -1, 1.566095325835539e+02, 0,_
 2645, 73,  158, 1, 9,     +1, 1.566096815886120e+02, 0.14,_
 2646, 73,  159, 0, 0.5,   +1, 1.575974593334300e+02, 0,_
 2647, 73,  159, 1, 5.5,   -1, 1.575975274500279e+02, 0.064,_
 2648, 73,  160, 0, 2,     -1, 1.585873952490539e+02, 0,_
 2649, 73,  160, 1, 9,     +1, 1.585877251888252e+02, 0.31,_
 2650, 73,  161, 0, 0.5,   +1, 1.595756593755895e+02, 0,_
 2651, 73,  161, 1, 5.5,   -1, 1.595757125916817e+02, 0.05,_
 2652, 73,  162, 0, 3,     +1, 1.605660032998529e+02, 0,_
 2653, 73,  163, 0, 0.5,   +1, 1.615544812149288e+02, 0,_
 2654, 73,  164, 0, 3,     +1, 1.625450944397258e+02, 0,_
 2655, 73,  165, 0, 2.5,   -1, 1.635337739182712e+02, 0,_
 2656, 73,  166, 0, 2,     +1, 1.645249174400892e+02, 0,_
 2657, 73,  167, 0, 1.5,   +1, 1.655139287404267e+02, 0,_
 2658, 73,  168, 0,-77.777,+0, 1.665052926555551e+02, 0,_
 2659, 73,  169, 0, 2.5,   +1, 1.674946836657388e+02, 0,_
 2660, 73,  170, 0, 3,     +1, 1.684862557768665e+02, 0,_
 2661, 73,  171, 0, 2.5,   -1, 1.694759808920584e+02, 0,_
 2662, 73,  172, 0, 3,     +1, 1.704678058126136e+02, 0,_
 2663, 73,  173, 0, 2.5,   -1, 1.714580801686796e+02, 0,_
 2664, 73,  174, 0, 3,     +1, 1.724501876409480e+02, 0,_
 2665, 73,  175, 0, 3.5,   +1, 1.734408863202885e+02, 0,_
 2666, 73,  176, 0, 1,     -1, 1.744334062189172e+02, 0,_
 2667, 73,  176, 1, 0,     +1, 1.744335158440670e+02, 0.103,_
 2668, 73,  176, 2, 20,    -1, 1.744364076065145e+02, 2.82,_
 2669, 73,  177, 0, 3.5,   +1, 1.754244440326864e+02, 0,_
 2670, 73,  178, 0, 1,     +1, 1.764170421802943e+02, 0,_
 2671, 73,  178, 1, 7,     -1, 1.764170421802943e+02, 0,_
 2672, 73,  178, 2, 15,    -1, 1.764186073719967e+02, 1.4706,_
 2673, 73,  179, 0, 3.5,   +1, 1.774087076693104e+02, 0,_
 2674, 73,  179, 1, 12.5,  +1, 1.774101097004742e+02, 1.3173,_
 2675, 73,  179, 2, 18.5,  +1, 1.774115167339507e+02, 2.6393,_
 2676, 73,  180, 0, 1,     +1, 1.784016332576717e+02, 0,_
 2677, 73,  180, 1, 9,     -1, 1.784017130818100e+02, 0.075,_
 2678, 73,  181, 0, 3.5,   +1, 1.793935691556172e+02, 0,_
 2679, 73,  182, 0, 3,     -1, 1.803871162358042e+02, 0,_
 2680, 73,  182, 1, 5,     +1, 1.803871335448703e+02, 0.016263,_
 2681, 73,  182, 2, 10,    -1, 1.803876692276328e+02, 0.519572,_
 2682, 73,  183, 0, 3.5,   +1, 1.813797360357960e+02, 0,_
 2683, 73,  184, 0, 5,     -1, 1.823737567876578e+02, 0,_
 2684, 73,  185, 0, 3.5,   +1, 1.833667043387142e+02, 0,_
 2685, 73,  185, 1, 10.5,  -1, 1.833680964716849e+02, 1.308,_
 2686, 73,  186, 0,-77.777,+0, 1.843610798734126e+02, 0,_
 2687, 73,  187, 0, 3.5,   +1, 1.853543114940302e+02, 0,_
 2688, 73,  188, 0,-77.777,+0, 1.863492157325544e+02, 0,_
 2689, 73,  189, 0, 3.5,   +1, 1.873425228103420e+02, 0,_
 2690, 73,  190, 0,-77.777,+0, 1.883374617481994e+02, 0,_
 2691, 74,  158, 0, 0,     +1, 1.566175511037514e+02, 0,_
 2692, 74,  159, 0, 3.5,   -1, 1.576071919491342e+02, 0,_
 2693, 74,  160, 0, 0,     +1, 1.585943104038199e+02, 0,_
 2694, 74,  161, 0, 3.5,   -1, 1.595844115080981e+02, 0,_
 2695, 74,  162, 0, 0,     +1, 1.605721561310882e+02, 0,_
 2696, 74,  163, 0, 1.5,   -1, 1.615625981979542e+02, 0,_
 2697, 74,  164, 0, 0,     +1, 1.625504661165979e+02, 0,_
 2698, 74,  165, 0, 1.5,   -1, 1.635412101549068e+02, 0,_
 2699, 74,  166, 0, 0,     +1, 1.645293979627376e+02, 0,_
 2700, 74,  167, 0, 1.5,   -1, 1.655205839466252e+02, 0,_
 2701, 74,  168, 0, 0,     +1, 1.665090186322805e+02, 0,_
 2702, 74,  169, 0, 2.5,   -1, 1.675004017956247e+02, 0,_
 2703, 74,  170, 0, 0,     +1, 1.684892857226414e+02, 0,_
 2704, 74,  171, 0, 2.5,   -1, 1.694809131544202e+02, 0,_
 2705, 74,  172, 0, 0,     +1, 1.704701822212330e+02, 0,_
 2706, 74,  173, 0, 2.5,   -1, 1.714619853307787e+02, 0,_
 2707, 74,  174, 0, 0,     +1, 1.724517986814179e+02, 0,_
 2708, 74,  175, 0, 0.5,   -1, 1.734438407206580e+02, 0,_
 2709, 74,  176, 0, 0,     +1, 1.744341765441142e+02, 0,_
 2710, 74,  177, 0, 0.5,   -1, 1.754265863962862e+02, 0,_
 2711, 74,  178, 0, 0,     +1, 1.764172453448902e+02, 0,_
 2712, 74,  179, 0, 3.5,   -1, 1.774098381874658e+02, 0,_
 2713, 74,  179, 1, 0.5,   -1, 1.774100743881551e+02, 0.221926,_
 2714, 74,  180, 0, 0,     +1, 1.784008847861148e+02, 0,_
 2715, 74,  180, 1, 8,     -1, 1.784025121448560e+02, 1.52901,_
 2716, 74,  181, 0, 4.5,   +1, 1.793937868017319e+02, 0,_
 2717, 74,  182, 0, 0,     +1, 1.803851832925944e+02, 0,_
 2718, 74,  183, 0, 0.5,   -1, 1.813785942492800e+02, 0,_
 2719, 74,  183, 1, 5.5,   +1, 1.813789236568905e+02, 0.3095,_
 2720, 74,  184, 0, 0,     +1, 1.823707064406578e+02, 0,_
 2721, 74,  185, 0, 1.5,   -1, 1.833645826132864e+02, 0,_
 2722, 74,  185, 1, 5.5,   +1, 1.833647927423479e+02, 0.19743,_
 2723, 74,  186, 0, 0,     +1, 1.843569279534495e+02, 0,_
 2724, 74,  186, 1, 16,    +1, 1.843606986328750e+02, 3.5428,_
 2725, 74,  187, 0, 1.5,   -1, 1.853511095615060e+02, 0,_
 2726, 74,  188, 0, 0,     +1, 1.863438347532868e+02, 0,_
 2727, 74,  189, 0, 1.5,   -1, 1.873384907478242e+02, 0,_
 2728, 74,  190, 0, 0,     +1, 1.883312149422471e+02, 0,_
 2729, 74,  190, 1, 10,    -1, 1.883337490925553e+02, 2.381,_
 2730, 74,  191, 0, 1.5,   -1, 1.893260368283481e+02, 0,_
 2731, 74,  192, 0, 0,     +1, 1.903190712685177e+02, 0,_
 2732, 75,  160, 0, 2,     -1, 1.586077075655414e+02, 0,_
 2733, 75,  161, 0, 0.5,   +1, 1.595947520293519e+02, 0,_
 2734, 75,  161, 1, 5.5,   -1, 1.595948837923960e+02, 0.1238,_
 2735, 75,  162, 0, 2,     -1, 1.605843937461836e+02, 0,_
 2736, 75,  162, 1, 9,     +1, 1.605845778738624e+02, 0.173,_
 2737, 75,  163, 0, 0.5,   +1, 1.615720769949760e+02, 0,_
 2738, 75,  163, 1, 5.5,   -1, 1.615721993919879e+02, 0.115,_
 2739, 75,  164, 0,-77.777,+0, 1.625619217564430e+02, 0,_
 2740, 75,  164, 1, 2,     -1, 1.625621559072484e+02, 0.22,_
 2741, 75,  165, 0, 0.5,   +1, 1.635499389218758e+02, 0,_
 2742, 75,  165, 1, 5.5,   -1, 1.635499889450024e+02, 0.047,_
 2743, 75,  166, 0, 2,     -1, 1.645400353843746e+02, 0,_
 2744, 75,  166, 1, 9,     +1, 1.645403014648353e+02, 0.25,_
 2745, 75,  167, 0, 4.5,   -1, 1.655283180581995e+02, 0,_
 2746, 75,  167, 1, 0.5,   +1, 1.655284564200391e+02, 0.13,_
 2747, 75,  168, 0,-77.777,+0, 1.665187020720897e+02, 0,_
 2748, 75,  169, 0, 4.5,   -1, 1.675073290852433e+02, 0,_
 2749, 75,  169, 1, 0.5,   +1, 1.675074993767382e+02, 0.16,_
 2750, 75,  170, 0, 5,     +1, 1.684982022278741e+02, 0,_
 2751, 75,  171, 0, 4.5,   -1, 1.694871243350628e+02, 0,_
 2752, 75,  172, 0,-77.777,+0, 1.704782285792183e+02, 0,_
 2753, 75,  172, 1,-77.777,+0, 1.704782285792183e+02, 0,_
 2754, 75,  173, 0, 2.5,   -1, 1.714674916192526e+02, 0,_
 2755, 75,  174, 0,-77.777,+0, 1.724587742388004e+02, 0,_
 2756, 75,  175, 0, 2.5,   -1, 1.734484646546591e+02, 0,_
 2757, 75,  176, 0, 3,     +1, 1.744401140957293e+02, 0,_
 2758, 75,  177, 0, 2.5,   -1, 1.754302397403673e+02, 0,_
 2759, 75,  178, 0, 3,     +1, 1.764223045820263e+02, 0,_
 2760, 75,  179, 0, 2.5,   +1, 1.774127234013305e+02, 0,_
 2761, 75,  180, 0, 1,     -1, 1.784049278859947e+02, 0,_
 2762, 75,  181, 0, 2.5,   +1, 1.793956136323980e+02, 0,_
 2763, 75,  182, 0, 7,     +1, 1.803881633942643e+02, 0,_
 2764, 75,  182, 1, 2,     +1, 1.803882272535749e+02, 0.06,_
 2765, 75,  183, 0, 2.5,   +1, 1.813791860117084e+02, 0,_
 2766, 75,  183, 1, 12.5,  +1, 1.813812163120562e+02, 1.9076,_
 2767, 75,  184, 0, 3,     -1, 1.823722877457820e+02, 0,_
 2768, 75,  184, 1, 8,     +1, 1.823724878489317e+02, 0.18801,_
 2769, 75,  185, 0, 2.5,   +1, 1.833641236412406e+02, 0,_
 2770, 75,  186, 0, 1,     -1, 1.843575467952375e+02, 0,_
 2771, 75,  186, 1, 8,     +1, 1.843577053791921e+02, 0.149,_
 2772, 75,  187, 0, 2.5,   +1, 1.853497126307930e+02, 0,_
 2773, 75,  188, 0, 1,     -1, 1.863434633048562e+02, 0,_
 2774, 75,  188, 1, 6,     -1, 1.863436464416514e+02, 0.172069,_
 2775, 75,  189, 0, 2.5,   +1, 1.873359773432632e+02, 0,_
 2776, 75,  190, 0, 2,     -1, 1.883298807965076e+02, 0,_
 2777, 75,  190, 1, 6,     -1, 1.883301043040947e+02, 0.21,_
 2778, 75,  191, 0, 1.5,   +1, 1.893226585379500e+02, 0,_
 2779, 75,  192, 0, 1,     -1, 1.903170071538972e+02, 0,_
 2780, 75,  193, 0, 2.5,   +1, 1.913098611575341e+02, 0,_
 2781, 75,  194, 0,-77.777,+0, 1.923044580591071e+02, 0,_
 2782, 76,  162, 0, 0,     +1, 1.605929734041694e+02, 0,_
 2783, 76,  163, 0, 3.5,   -1, 1.615825180828287e+02, 0,_
 2784, 76,  164, 0, 0,     +1, 1.625694255784569e+02, 0,_
 2785, 76,  165, 0, 3.5,   -1, 1.635593737944988e+02, 0,_
 2786, 76,  166, 0, 0,     +1, 1.645469129914879e+02, 0,_
 2787, 76,  167, 0, 1.5,   -1, 1.655371831806597e+02, 0,_
 2788, 76,  168, 0, 0,     +1, 1.665248747900085e+02, 0,_
 2789, 76,  169, 0, 1.5,   -1, 1.675155100399594e+02, 0,_
 2790, 76,  170, 0, 0,     +1, 1.685035099439610e+02, 0,_
 2791, 76,  171, 0, 2.5,   -1, 1.694945196036827e+02, 0,_
 2792, 76,  172, 0, 0,     +1, 1.704827979946186e+02, 0,_
 2793, 76,  173, 0, 2.5,   -1, 1.714740005945297e+02, 0,_
 2794, 76,  174, 0, 0,     +1, 1.724626884742795e+02, 0,_
 2795, 76,  175, 0, 2.5,   -1, 1.734539809613262e+02, 0,_
 2796, 76,  176, 0, 0,     +1, 1.744432697522313e+02, 0,_
 2797, 76,  177, 0, 0.5,   -1, 1.754348298494584e+02, 0,_
 2798, 76,  178, 0, 0,     +1, 1.764245494306024e+02, 0,_
 2799, 76,  179, 0, 0.5,   -1, 1.774165174799586e+02, 0,_
 2800, 76,  180, 0, 0,     +1, 1.784065026032026e+02, 0,_
 2801, 76,  181, 0, 0.5,   -1, 1.793987719310065e+02, 0,_
 2802, 76,  181, 1, 3.5,   -1, 1.793988239763446e+02, 0.0489,_
 2803, 76,  182, 0, 0,     +1, 1.803890541836493e+02, 0,_
 2804, 76,  183, 0, 4.5,   +1, 1.813814695540066e+02, 0,_
 2805, 76,  183, 1, 0.5,   -1, 1.813816512443885e+02, 0.17071,_
 2806, 76,  184, 0, 0,     +1, 1.823722527311805e+02, 0,_
 2807, 76,  185, 0, 0.5,   -1, 1.833652019557362e+02, 0,_
 2808, 76,  186, 0, 0,     +1, 1.843564049304002e+02, 0,_
 2809, 76,  187, 0, 0.5,   -1, 1.853497100055406e+02, 0,_
 2810, 76,  188, 0, 0,     +1, 1.863412064929414e+02, 0,_
 2811, 76,  189, 0, 1.5,   -1, 1.873349048245476e+02, 0,_
 2812, 76,  189, 1, 4.5,   -1, 1.873349376184322e+02, 0.030812,_
 2813, 76,  190, 0, 0,     +1, 1.883266112627425e+02, 0,_
 2814, 76,  190, 1, 10,    -1, 1.883284263572135e+02, 1.7054,_
 2815, 76,  191, 0, 4.5,   -1, 1.893204821184419e+02, 0,_
 2816, 76,  191, 1, 1.5,   -1, 1.893205612848292e+02, 0.074382,_
 2817, 76,  192, 0, 0,     +1, 1.903124376294419e+02, 0,_
 2818, 76,  192, 1, 10,    -1, 1.903145826636843e+02, 2.0154,_
 2819, 76,  193, 0, 1.5,   -1, 1.913064950773356e+02, 0,_
 2820, 76,  194, 0, 0,     +1, 1.922989254754785e+02, 0,_
 2821, 76,  195, 0, 0.5,   -1, 1.932934465570257e+02, 0,_
 2822, 76,  196, 0, 0,     +1, 1.942861699692264e+02, 0,_
 2823, 77,  164, 0, 2,     -1, 1.625833450018252e+02, 0,_
 2824, 77,  164, 1, 9,     +1, 1.625836323687228e+02, 0.27,_
 2825, 77,  165, 0, 0.5,   +1, 1.635702327029709e+02, 0,_
 2826, 77,  166, 0, 2,     -1, 1.645597674675350e+02, 0,_
 2827, 77,  166, 1, 9,     +1, 1.645599505308920e+02, 0.172,_
 2828, 77,  167, 0, 0.5,   +1, 1.655472192618609e+02, 0,_
 2829, 77,  167, 1, 5.5,   -1, 1.655474058374799e+02, 0.1753,_
 2830, 77,  168, 0,-77.777,+0, 1.665369324801386e+02, 0,_
 2831, 77,  168, 1,-77.777,+0, 1.665370921284150e+02, 0.15,_
 2832, 77,  169, 0, 0.5,   +1, 1.675246939164427e+02, 0,_
 2833, 77,  169, 1, 5.5,   -1, 1.675248578220066e+02, 0.154,_
 2834, 77,  170, 0,-77.777,+0, 1.685147558263369e+02, 0,_
 2835, 77,  170, 1,-77.777,+0, 1.685151503704441e+02, 0.3707,_
 2836, 77,  171, 0, 0.5,   +1, 1.695029170148221e+02, 0,_
 2837, 77,  171, 1, 5.5,   -1, 1.695031085927538e+02, 0.18,_
 2838, 77,  172, 0, 3,     +1, 1.704932969698817e+02, 0,_
 2839, 77,  172, 1, 7,     +1, 1.704934449106179e+02, 0.139,_
 2840, 77,  173, 0,-77.777,+0, 1.714816315935655e+02, 0,_
 2841, 77,  173, 1, 5.5,   -1, 1.714818976740262e+02, 0.25,_
 2842, 77,  174, 0, 3,     +1, 1.724724077813023e+02, 0,_
 2843, 77,  174, 1, 7,     +1, 1.724726131954180e+02, 0.193,_
 2844, 77,  175, 0, 2.5,   -1, 1.734611234879344e+02, 0,_
 2845, 77,  176, 0,-77.777,+0, 1.744520180678132e+02, 0,_
 2846, 77,  177, 0, 2.5,   -1, 1.754411189727346e+02, 0,_
 2847, 77,  178, 0,-77.777,+0, 1.764323108783113e+02, 0,_
 2848, 77,  179, 0, 2.5,   -1, 1.774217728694510e+02, 0,_
 2849, 77,  180, 0,-77.777,+0, 1.784132932785722e+02, 0,_
 2850, 77,  181, 0, 2.5,   -1, 1.794031217462237e+02, 0,_
 2851, 77,  182, 0, 3,     +1, 1.803949690746169e+02, 0,_
 2852, 77,  183, 0, 2.5,   -1, 1.813851528872343e+02, 0,_
 2853, 77,  184, 0, 5,     -1, 1.823771929753804e+02, 0,_
 2854, 77,  185, 0, 2.5,   -1, 1.833678311787402e+02, 0,_
 2855, 77,  186, 0, 5,     +1, 1.843604787253103e+02, 0,_
 2856, 77,  186, 1, 2,     -1, 1.843604795767678e+02, 0.0008,_
 2857, 77,  187, 0, 1.5,   +1, 1.853514869683078e+02, 0,_
 2858, 77,  187, 1, 4.5,   -1, 1.853516850918188e+02, 0.18615,_
 2859, 77,  188, 0, 1,     -1, 1.863441784263883e+02, 0,_
 2860, 77,  188, 1, 7,     +1, 1.863452108185760e+02, 0.97,_
 2861, 77,  189, 0, 1.5,   +1, 1.873354765347270e+02, 0,_
 2862, 77,  189, 1, 5.5,   -1, 1.873358726433873e+02, 0.37217,_
 2863, 77,  189, 2, 12.5,  +1, 1.873379598104511e+02, 2.3332,_
 2864, 77,  190, 0, 4,     -1, 1.883286911893535e+02, 0,_
 2865, 77,  190, 1, 1,     -1, 1.883287189681536e+02, 0.0261,_
 2866, 77,  190, 2, 11,    -1, 1.883290918000952e+02, 0.3764,_
 2867, 77,  191, 0, 1.5,   +1, 1.893201483782723e+02, 0,_
 2868, 77,  191, 1, 5.5,   -1, 1.893203306327447e+02, 0.17124,_
 2869, 77,  191, 2, 13.5,  -1, 1.893223674893150e+02, 2.085,_
 2870, 77,  192, 0, 4,     +1, 1.903135515811441e+02, 0,_
 2871, 77,  192, 1, 1,     -1, 1.903136119494791e+02, 0.05672,_
 2872, 77,  192, 2, 11,    -1, 1.903137305362188e+02, 0.16814,_
 2873, 77,  193, 0, 1.5,   +1, 1.913052796796542e+02, 0,_
 2874, 77,  193, 1, 5.5,   -1, 1.913053650808389e+02, 0.08024,_
 2875, 77,  194, 0, 1,     -1, 1.922988226623457e+02, 0,_
 2876, 77,  194, 1, 4,     +1, 1.922989791942877e+02, 0.147072,_
 2877, 77,  194, 2, 10,    -1, 1.922991121578870e+02, 0.272,_
 2878, 77,  195, 0, 1.5,   +1, 1.932911256346229e+02, 0,_
 2879, 77,  195, 1, 5.5,   -1, 1.932912565462096e+02, 0.123,_
 2880, 77,  196, 0, 0,     -1, 1.942849370711842e+02, 0,_
 2881, 77,  196, 1,-77.777,+0, 1.942851605787712e+02, 0.21,_
 2882, 77,  197, 0, 1.5,   +1, 1.952775933253040e+02, 0,_
 2883, 77,  197, 1, 5.5,   -1, 1.952777157223160e+02, 0.115,_
 2884, 77,  198, 0,-77.777,+0, 1.962716030814428e+02, 0,_
 2885, 77,  199, 0, 1.5,   +1, 1.972645265977582e+02, 0,_
 2886, 78,  166, 0, 0,     +1, 1.645689469282804e+02, 0,_
 2887, 78,  167, 0, 3.5,   -1, 1.655583518181974e+02, 0,_
 2888, 78,  168, 0, 0,     +1, 1.665450838720141e+02, 0,_
 2889, 78,  169, 0, 1.5,   -1, 1.675348912702306e+02, 0,_
 2890, 78,  170, 0, 0,     +1, 1.685222708046822e+02, 0,_
 2891, 78,  171, 0, 1.5,   -1, 1.695124345263856e+02, 0,_
 2892, 78,  172, 0, 0,     +1, 1.704999728757196e+02, 0,_
 2893, 78,  173, 0, 2.5,   -1, 1.714904926324098e+02, 0,_
 2894, 78,  174, 0, 0,     +1, 1.724783097968345e+02, 0,_
 2895, 78,  175, 0, 2.5,   -1, 1.734692985873859e+02, 0,_
 2896, 78,  176, 0, 0,     +1, 1.744572805637291e+02, 0,_
 2897, 78,  177, 0, 2.5,   -1, 1.754482254249246e+02, 0,_
 2898, 78,  178, 0, 0,     +1, 1.764368388914586e+02, 0,_
 2899, 78,  179, 0, 0.5,   -1, 1.774279603782085e+02, 0,_
 2900, 78,  180, 0, 0,     +1, 1.784170627335620e+02, 0,_
 2901, 78,  181, 0, 0.5,   -1, 1.794085301161536e+02, 0,_
 2902, 78,  182, 0, 0,     +1, 1.803980377597188e+02, 0,_
 2903, 78,  183, 0, 0.5,   -1, 1.813898687100267e+02, 0,_
 2904, 78,  183, 1, 3.5,   -1, 1.813899054291303e+02, 0.0345,_
 2905, 78,  184, 0, 0,     +1, 1.823796160189122e+02, 0,_
 2906, 78,  184, 1, 8,     -1, 1.823815737325102e+02, 1.8394,_
 2907, 78,  185, 0, 4.5,   +1, 1.833717132003498e+02, 0,_
 2908, 78,  185, 1, 0.5,   -1, 1.833718232512283e+02, 0.1034,_
 2909, 78,  186, 0, 0,     +1, 1.843618707554860e+02, 0,_
 2910, 78,  187, 0, 1.5,   -1, 1.853545355287879e+02, 0,_
 2911, 78,  188, 0, 0,     +1, 1.863447361081230e+02, 0,_
 2912, 78,  189, 0, 1.5,   -1, 1.873375841454574e+02, 0,_
 2913, 78,  190, 0, 0,     +1, 1.883281027193961e+02, 0,_
 2914, 78,  191, 0, 1.5,   -1, 1.893212238950103e+02, 0,_
 2915, 78,  192, 0, 0,     +1, 1.903120052311368e+02, 0,_
 2916, 78,  193, 0, 0.5,   -1, 1.913053399494218e+02, 0,_
 2917, 78,  193, 1, 6.5,   +1, 1.913054993635474e+02, 0.14978,_
 2918, 78,  194, 0, 0,     +1, 1.922964509668871e+02, 0,_
 2919, 78,  195, 0, 0.5,   -1, 1.932899531798449e+02, 0,_
 2920, 78,  195, 1, 6.5,   +1, 1.932902291584988e+02, 0.2593,_
 2921, 78,  196, 0, 0,     +1, 1.942815216445683e+02, 0,_
 2922, 78,  197, 0, 0.5,   -1, 1.952752990252211e+02, 0,_
 2923, 78,  197, 1, 6.5,   +1, 1.952757243175864e+02, 0.39959,_
 2924, 78,  198, 0, 0,     +1, 1.962672574698019e+02, 0,_
 2925, 78,  199, 0, 2.5,   -1, 1.972613440968606e+02, 0,_
 2926, 78,  199, 1, 6.5,   +1, 1.972617953693221e+02, 0.424,_
 2927, 78,  200, 0, 0,     +1, 1.982535939226478e+02, 0,_
 2928, 78,  201, 0, 2.5,   -1, 1.992480457492829e+02, 0,_
 2929, 78,  202, 0, 0,     +1, 2.002405713058858e+02, 0,_
 2930, 79,  169, 0, 0.5,   +1, 1.675463053680340e+02, 0,_
 2931, 79,  170, 0, 2,     -1, 1.685356249967322e+02, 0,_
 2932, 79,  170, 1, 9,     +1, 1.685359166209172e+02, 0.274,_
 2933, 79,  171, 0, 0.5,   +1, 1.695229791976963e+02, 0,_
 2934, 79,  171, 1, 5.5,   -1, 1.695232452781571e+02, 0.25,_
 2935, 79,  172, 0,-77.777,+0, 1.705125200752915e+02, 0,_
 2936, 79,  173, 0, 0.5,   +1, 1.715001891062800e+02, 0,_
 2937, 79,  173, 1, 5.5,   -1, 1.715004168711544e+02, 0.214,_
 2938, 79,  174, 0,-77.777,+0, 1.724901057538599e+02, 0,_
 2939, 79,  174, 1,-77.777,+0, 1.724905953419077e+02, 0.46,_
 2940, 79,  175, 0, 0.5,   +1, 1.734781425829091e+02, 0,_
 2941, 79,  175, 1, 5.5,   -1, 1.734783554472777e+02, 0.2,_
 2942, 79,  176, 0, 5,     -1, 1.744683632462183e+02, 0,_
 2943, 79,  176, 1, 7,     +1, 1.744685228944948e+02, 0.15,_
 2944, 79,  177, 0,-77.777,+0, 1.754565541075904e+02, 0,_
 2945, 79,  177, 1, 5.5,   -1, 1.754567776151775e+02, 0.21,_
 2946, 79,  178, 0,-77.777,+0, 1.764471561899358e+02, 0,_
 2947, 79,  179, 0, 2.5,   -1, 1.774357081930451e+02, 0,_
 2948, 79,  180, 0,-77.777,+0, 1.784264398020529e+02, 0,_
 2949, 79,  181, 0, 1.5,   -1, 1.794154592517767e+02, 0,_
 2950, 79,  182, 0, 2,     +1, 1.804064115049216e+02, 0,_
 2951, 79,  183, 0, 2.5,   -1, 1.813958086955522e+02, 0,_
 2952, 79,  184, 0, 5,     +1, 1.823870828048407e+02, 0,_
 2953, 79,  184, 1, 2,     +1, 1.823871556683141e+02, 0.06846,_
 2954, 79,  185, 0, 2.5,   -1, 1.833768538718639e+02, 0,_
 2955, 79,  186, 0, 3,     -1, 1.843684158993652e+02, 0,_
 2956, 79,  187, 0, 0.5,   +1, 1.853584279800255e+02, 0,_
 2957, 79,  187, 1, 4.5,   -1, 1.853585562414508e+02, 0.12051,_
 2958, 79,  188, 0, 1,     -1, 1.863505362593013e+02, 0,_
 2959, 79,  189, 0, 0.5,   +1, 1.873406572611685e+02, 0,_
 2960, 79,  189, 1, 5.5,   -1, 1.873409203934577e+02, 0.24723,_
 2961, 79,  190, 0, 1,     -1, 1.883328633429934e+02, 0,_
 2962, 79,  190, 1, 11,    -1, 1.883330762073620e+02, 0.2,_
 2963, 79,  191, 0, 1.5,   +1, 1.893232464616408e+02, 0,_
 2964, 79,  191, 1, 5.5,   -1, 1.893235297841154e+02, 0.2662,_
 2965, 79,  192, 0, 1,     -1, 1.903157477505214e+02, 0,_
 2966, 79,  192, 1, 5,     +1, 1.903158918703421e+02, 0.13541,_
 2967, 79,  192, 2, 11,    -1, 1.903162071118288e+02, 0.4316,_
 2968, 79,  193, 0, 1.5,   +1, 1.913064838684596e+02, 0,_
 2969, 79,  193, 1, 5.5,   -1, 1.913067927240152e+02, 0.29019,_
 2970, 79,  194, 0, 1,     -1, 1.922991630023284e+02, 0,_
 2971, 79,  194, 1, 5,     +1, 1.922992773104943e+02, 0.1074,_
 2972, 79,  194, 2, 11,    -1, 1.922996694066613e+02, 0.4758,_
 2973, 79,  195, 0, 1.5,   +1, 1.932901945860802e+02, 0,_
 2974, 79,  195, 1, 5.5,   -1, 1.932905336577329e+02, 0.31858,_
 2975, 79,  196, 0, 2,     -1, 1.942831243036622e+02, 0,_
 2976, 79,  196, 1, 5,     +1, 1.942832144091494e+02, 0.08466,_
 2977, 79,  196, 2, 12,    -1, 1.942837582776112e+02, 0.59566,_
 2978, 79,  197, 0, 1.5,   +1, 1.952745327261382e+02, 0,_
 2979, 79,  197, 1, 5.5,   -1, 1.952749681934202e+02, 0.40915,_
 2980, 79,  198, 0, 2,     -1, 1.962676014789912e+02, 0,_
 2981, 79,  198, 1, 12,    -1, 1.962684657083277e+02, 0.812,_
 2982, 79,  199, 0, 1.5,   +1, 1.972595293653564e+02, 0,_
 2983, 79,  200, 0, 1,     -1, 1.982529117644908e+02, 0,_
 2984, 79,  200, 1, 12,    -1, 1.982539335134601e+02, 0.96,_
 2985, 79,  201, 0, 1.5,   +1, 1.992452146524983e+02, 0,_
 2986, 79,  202, 0, 1,     -1, 2.002388036227117e+02, 0,_
 2987, 79,  203, 0, 1.5,   +1, 2.012315004859569e+02, 0,_
 2988, 79,  204, 0, 2,     -1, 2.022255635154862e+02, 0,_
 2989, 79,  205, 0, 1.5,   +1, 2.032189746912734e+02, 0,_
 2990, 80,  171, 0, 1.5,   -1, 1.695347318202116e+02, 0,_
 2991, 80,  172, 0, 0,     +1, 1.705213105892282e+02, 0,_
 2992, 80,  173, 0, 1.5,   -1, 1.715109629438841e+02, 0,_
 2993, 80,  174, 0, 0,     +1, 1.724981883022584e+02, 0,_
 2994, 80,  175, 0, 3.5,   -1, 1.734881806052902e+02, 0,_
 2995, 80,  176, 0, 0,     +1, 1.744755325329523e+02, 0,_
 2996, 80,  177, 0, 6.5,   +1, 1.754658802939705e+02, 0,_
 2997, 80,  178, 0, 0,     +1, 1.764535291815004e+02, 0,_
 2998, 80,  179, 0, 2.5,   -1, 1.774442870886370e+02, 0,_
 2999, 80,  180, 0, 0,     +1, 1.784321605978390e+02, 0,_
 3000, 80,  181, 0, 0.5,   -1, 1.794231330132793e+02, 0,_
 3001, 80,  182, 0, 0,     +1, 1.804114391983810e+02, 0,_
 3002, 80,  183, 0, 0.5,   -1, 1.814026063157455e+02, 0,_
 3003, 80,  183, 1, 6.5,   +1, 1.814028617529878e+02, 0.24,_
 3004, 80,  184, 0, 0,     +1, 1.823913078918177e+02, 0,_
 3005, 80,  185, 0, 0.5,   -1, 1.833828933423604e+02, 0,_
 3006, 80,  185, 1, 6.5,   +1, 1.833830038189677e+02, 0.1038,_
 3007, 80,  186, 0, 0,     +1, 1.843717959257214e+02, 0,_
 3008, 80,  187, 0, 1.5,   -1, 1.853636537025795e+02, 0,_
 3009, 80,  187, 1, 6.5,   +1, 1.853636856322348e+02, 0.03,_
 3010, 80,  188, 0, 0,     +1, 1.863528451935801e+02, 0,_
 3011, 80,  189, 0, 1.5,   -1, 1.873448672440215e+02, 0,_
 3012, 80,  189, 1, 6.5,   +1, 1.873449736762058e+02, 0.1,_
 3013, 80,  190, 0, 0,     +1, 1.883344202713403e+02, 0,_
 3014, 80,  191, 0, 1.5,   -1, 1.893266586869953e+02, 0,_
 3015, 80,  191, 1, 6.5,   +1, 1.893267949201912e+02, 0.128,_
 3016, 80,  192, 0, 0,     +1, 1.903165572344116e+02, 0,_
 3017, 80,  193, 0, 1.5,   -1, 1.913089771940037e+02, 0,_
 3018, 80,  193, 1, 6.5,   +1, 1.913091270079463e+02, 0.14076,_
 3019, 80,  194, 0, 0,     +1, 1.922991927931930e+02, 0,_
 3020, 80,  195, 0, 0.5,   -1, 1.932918481580186e+02, 0,_
 3021, 80,  195, 1, 6.5,   +1, 1.932920355531654e+02, 0.17607,_
 3022, 80,  196, 0, 0,     +1, 1.942823928645207e+02, 0,_
 3023, 80,  197, 0, 0.5,   -1, 1.952751707963138e+02, 0,_
 3024, 80,  197, 1, 6.5,   +1, 1.952754889540423e+02, 0.29893,_
 3025, 80,  198, 0, 0,     +1, 1.962661396010406e+02, 0,_
 3026, 80,  199, 0, 0.5,   -1, 1.972590479438333e+02, 0,_
 3027, 80,  199, 1, 6.5,   +1, 1.972596146952147e+02, 0.5325,_
 3028, 80,  200, 0, 0,     +1, 1.982505030141101e+02, 0,_
 3029, 80,  201, 0, 1.5,   -1, 1.992438716623944e+02, 0,_
 3030, 80,  202, 0, 0,     +1, 2.002356188038986e+02, 0,_
 3031, 80,  203, 0, 2.5,   -1, 2.012292379189099e+02, 0,_
 3032, 80,  204, 0, 0,     +1, 2.022212638090803e+02, 0,_
 3033, 80,  205, 0, 0.5,   -1, 2.032152302614427e+02, 0,_
 3034, 80,  205, 1, 6.5,   +1, 2.032168867719591e+02, 1.5564,_
 3035, 80,  206, 0, 0,     +1, 2.042080680366972e+02, 0,_
 3036, 80,  207, 0, 4.5,   +1, 2.052042226845300e+02, 0,_
 3037, 80,  208, 0, 0,     +1, 2.061990614900259e+02, 0,_
 3038, 80,  209, 0, 4.5,   +1, 2.071953893926208e+02, 0,_
 3039, 80,  210, 0, 0,     +1, 2.081902886740974e+02, 0,_
 3040, 81,  176, 0,-77.777,+0, 1.744886945174645e+02, 0,_
 3041, 81,  177, 0, 0.5,   +1, 1.754759296382483e+02, 0,_
 3042, 81,  178, 0,-77.777,+0, 1.764657957348482e+02, 0,_
 3043, 81,  179, 0, 0.5,   +1, 1.774535037331879e+02, 0,_
 3044, 81,  179, 1, 4.5,   -1, 1.774544190499728e+02, 0.86,_
 3045, 81,  180, 0,-77.777,+0, 1.784437231790462e+02, 0,_
 3046, 81,  181, 0, 0.5,   +1, 1.794315011391728e+02, 0,_
 3047, 81,  181, 1, 4.5,   -1, 1.794324058127394e+02, 0.85,_
 3048, 81,  182, 0, 2,     -1, 1.804223474274939e+02, 0,_
 3049, 81,  182, 1, 7,     +1, 1.804224538596783e+02, 0.1,_
 3050, 81,  183, 0, 0.5,   +1, 1.814102879718420e+02, 0,_
 3051, 81,  183, 1, 4.5,   -1, 1.814109712664652e+02, 0.642,_
 3052, 81,  184, 0, 2,     -1, 1.824013824684592e+02, 0,_
 3053, 81,  184, 1, 7,     +1, 1.824014889006435e+02, 0.1,_
 3054, 81,  185, 0, 0.5,   +1, 1.833897325958050e+02, 0,_
 3055, 81,  185, 1, 4.5,   -1, 1.833902145207356e+02, 0.4528,_
 3056, 81,  186, 0, 2,     -1, 1.843810049542643e+02, 0,_
 3057, 81,  186, 1, 10,    -1, 1.843814030106336e+02, 0.374,_
 3058, 81,  186, 2, 10,    -1, 1.843815584016226e+02, 0.52,_
 3059, 81,  187, 0, 0.5,   +1, 1.853696919665307e+02, 0,_
 3060, 81,  187, 1, 4.5,   -1, 1.853700463857044e+02, 0.333,_
 3061, 81,  188, 0, 7,     +1, 1.863612166317729e+02, 0,_
 3062, 81,  188, 1, 7,     +1, 1.863613230639572e+02, 0.1,_
 3063, 81,  188, 2, 9,     -1, 1.863614082097046e+02, 0.18,_
 3064, 81,  189, 0, 0.5,   +1, 1.873501998167279e+02, 0,_
 3065, 81,  189, 1, 4.5,   -1, 1.873504739860346e+02, 0.2576,_
 3066, 81,  190, 0, 2,     -1, 1.883418691097823e+02, 0,_
 3067, 81,  190, 1, 7,     +1, 1.883420074716219e+02, 0.13,_
 3068, 81,  191, 0, 0.5,   +1, 1.893312447977318e+02, 0,_
 3069, 81,  191, 1, 4.5,   -1, 1.893315609013192e+02, 0.297,_
 3070, 81,  192, 0, 2,     -1, 1.903230914341212e+02, 0,_
 3071, 81,  192, 1, 7,     +1, 1.903232617256160e+02, 0.16,_
 3072, 81,  193, 0, 0.5,   +1, 1.913127927525104e+02, 0,_
 3073, 81,  193, 1, 4.5,   -1, 1.913131854872705e+02, 0.369,_
 3074, 81,  194, 0, 2,     -1, 1.923047767090323e+02, 0,_
 3075, 81,  194, 1, 7,     +1, 1.923050960055852e+02, 0.3,_
 3076, 81,  195, 0, 0.5,   +1, 1.932948901444621e+02, 0,_
 3077, 81,  195, 1, 4.5,   -1, 1.932954038181132e+02, 0.48263,_
 3078, 81,  196, 0, 2,     -1, 1.942870006861343e+02, 0,_
 3079, 81,  196, 1, 7,     +1, 1.942874202418049e+02, 0.3942,_
 3080, 81,  197, 0, 0.5,   +1, 1.952775107934357e+02, 0,_
 3081, 81,  197, 1, 4.5,   -1, 1.952781581139807e+02, 0.6082,_
 3082, 81,  198, 0, 2,     -1, 1.962697855036018e+02, 0,_
 3083, 81,  198, 1, 7,     +1, 1.962703639625235e+02, 0.5435,_
 3084, 81,  198, 2, 10,    -1, 1.962705755497059e+02, 0.7423,_
 3085, 81,  199, 0, 0.5,   +1, 1.972606302443327e+02, 0,_
 3086, 81,  199, 1, 4.5,   -1, 1.972614282728506e+02, 0.7498,_
 3087, 81,  200, 0, 2,     -1, 1.982531170318663e+02, 0,_
 3088, 81,  200, 1, 7,     +1, 1.982539191048072e+02, 0.7536,_
 3089, 81,  201, 0, 0.5,   +1, 1.992443843499995e+02, 0,_
 3090, 81,  201, 1, 4.5,   -1, 1.992453629939342e+02, 0.9195,_
 3091, 81,  202, 0, 2,     -1, 2.002370717185158e+02, 0,_
 3092, 81,  203, 0, 0.5,   +1, 2.012287141532948e+02, 0,_
 3093, 81,  204, 0, 2,     -1, 2.022216299366160e+02, 0,_
 3094, 81,  205, 0, 0.5,   +1, 2.032135985124107e+02, 0,_
 3095, 81,  206, 0, 0,     -1, 2.042066763654117e+02, 0,_
 3096, 81,  206, 1, 12,    -1, 2.042094893680428e+02, 2.643,_
 3097, 81,  207, 0, 0.5,   +1, 2.051993832042195e+02, 0,_
 3098, 81,  207, 1, 5.5,   -1, 2.052008104598110e+02, 1.341,_
 3099, 81,  208, 0, 5,     +1, 2.061953526190804e+02, 0,_
 3100, 81,  209, 0, 0.5,   +1, 2.071900672584653e+02, 0,_
 3101, 81,  210, 0, 5,     +1, 2.081861574408854e+02, 0,_
 3102, 81,  211, 0, 0.5,   +1, 2.091809397657810e+02, 0,_
 3103, 81,  212, 0, 5,     +1, 2.101771675360144e+02, 0,_
 3104, 82,  178, 0, 0,     +1, 1.764746987539372e+02, 0,_
 3105, 82,  179, 0, 2.5,   -1, 1.774644866144463e+02, 0,_
 3106, 82,  180, 0, 0,     +1, 1.784516473398598e+02, 0,_
 3107, 82,  181, 0, 2.5,   -1, 1.794418052489292e+02, 0,_
 3108, 82,  182, 0, 0,     +1, 1.804292685168374e+02, 0,_
 3109, 82,  183, 0, 1.5,   -1, 1.814198796824768e+02, 0,_
 3110, 82,  183, 1, 6.5,   +1, 1.814199797287300e+02, 0.094,_
 3111, 82,  184, 0, 0,     +1, 1.824075892958226e+02, 0,_
 3112, 82,  185, 0, 1.5,   -1, 1.833984776189159e+02, 0,_
 3113, 82,  185, 1, 6.5,   +1, 1.833985414782265e+02, 0.06,_
 3114, 82,  186, 0, 0,     +1, 1.843865443112027e+02, 0,_
 3115, 82,  187, 0, 1.5,   -1, 1.853776292833486e+02, 0,_
 3116, 82,  187, 1, 6.5,   +1, 1.853776409908889e+02, 0.011,_
 3117, 82,  188, 0, 0,     +1, 1.863660286421084e+02, 0,_
 3118, 82,  189, 0, 1.5,   -1, 1.873574074749759e+02, 0,_
 3119, 82,  189, 1, 6.5,   +1, 1.873574500478496e+02, 0.04,_
 3120, 82,  190, 0, 0,     +1, 1.883460789092245e+02, 0,_
 3121, 82,  191, 0, 1.5,   -1, 1.893376858902078e+02, 0,_
 3122, 82,  191, 1, 6.5,   +1, 1.893376858902078e+02, 0,_
 3123, 82,  192, 0, 0,     +1, 1.903266209660243e+02, 0,_
 3124, 82,  193, 0, 1.5,   -1, 1.913184152708621e+02, 0,_
 3125, 82,  193, 1, 6.5,   +1, 1.913185536327017e+02, 0.13,_
 3126, 82,  194, 0, 0,     +1, 1.923076818313916e+02, 0,_
 3127, 82,  195, 0, 1.5,   -1, 1.932996237749524e+02, 0,_
 3128, 82,  195, 1, 6.5,   +1, 1.932998397258543e+02, 0.2029,_
 3129, 82,  196, 0, 0,     +1, 1.942892871481336e+02, 0,_
 3130, 82,  197, 0, 1.5,   -1, 1.952813383589032e+02, 0,_
 3131, 82,  197, 1, 6.5,   +1, 1.952816782075109e+02, 0.31931,_
 3132, 82,  198, 0, 0,     +1, 1.962713407515096e+02, 0,_
 3133, 82,  199, 0, 1.5,   -1, 1.972636397095698e+02, 0,_
 3134, 82,  199, 1, 6.5,   +1, 1.972640968358013e+02, 0.4295,_
 3135, 82,  200, 0, 0,     +1, 1.982539644193253e+02, 0,_
 3136, 82,  201, 0, 2.5,   -1, 1.992464169943077e+02, 0,_
 3137, 82,  201, 1, 6.5,   +1, 1.992470865591791e+02, 0.6291,_
 3138, 82,  202, 0, 0,     +1, 2.002371138682915e+02, 0,_
 3139, 82,  202, 1, 9,     -1, 2.002394232763993e+02, 2.16984,_
 3140, 82,  203, 0, 2.5,   -1, 2.012297516762458e+02, 0,_
 3141, 82,  203, 1, 6.5,   +1, 2.012306299546306e+02, 0.8252,_
 3142, 82,  203, 2, 14.5,  -1, 2.012328908935218e+02, 2.9495,_
 3143, 82,  204, 0, 0,     +1, 2.022208170630966e+02, 0,_
 3144, 82,  204, 1, 9,     -1, 2.022231432981328e+02, 2.18565,_
 3145, 82,  205, 0, 2.5,   -1, 2.032136524054322e+02, 0,_
 3146, 82,  205, 1, 6.5,   +1, 2.032147314681327e+02, 1.01385,_
 3147, 82,  206, 0, 0,     +1, 2.042050455939095e+02, 0,_
 3148, 82,  207, 0, 0.5,   -1, 2.051978744266501e+02, 0,_
 3149, 82,  207, 1, 6.5,   +1, 2.051996128580188e+02, 1.63337,_
 3150, 82,  208, 0, 0,     +1, 2.061900326422320e+02, 0,_
 3151, 82,  209, 0, 4.5,   +1, 2.071858420178198e+02, 0,_
 3152, 82,  210, 0, 0,     +1, 2.081803233240170e+02, 0,_
 3153, 82,  211, 0, 4.5,   +1, 2.091762408375982e+02, 0,_
 3154, 82,  212, 0, 0,     +1, 2.101707838253301e+02, 0,_
 3155, 82,  213, 0, 4.5,   +1, 2.111668181636348e+02, 0,_
 3156, 82,  214, 0, 0,     +1, 2.121614427463531e+02, 0,_
 3157, 82,  215, 0, 2.5,   +1, 2.131576683374684e+02, 0,_
 3158, 83,  184, 0, 3,     +1, 1.824204831199221e+02, 0,_
 3159, 83,  184, 1, 10,    -1, 1.824206427681986e+02, 0.15,_
 3160, 83,  185, 0, 4.5,   -1, 1.834083818109259e+02, 0,_
 3161, 83,  186, 0, 3,     +1, 1.843988221309284e+02, 0,_
 3162, 83,  186, 1, 10,    -1, 1.843990882113891e+02, 0.25,_
 3163, 83,  187, 0, 4.5,   -1, 1.853867863778952e+02, 0,_
 3164, 83,  187, 1, 0.5,   +1, 1.853868949387232e+02, 0.102,_
 3165, 83,  188, 0, 3,     +1, 1.863773322889600e+02, 0,_
 3166, 83,  188, 1, 10,    -1, 1.863775557965470e+02, 0.21,_
 3167, 83,  189, 0, 4.5,   -1, 1.873656872335647e+02, 0,_
 3168, 83,  189, 1, 0.5,   +1, 1.873658841331057e+02, 0.185,_
 3169, 83,  190, 0, 3,     +1, 1.883565274286829e+02, 0,_
 3170, 83,  190, 1, 10,    -1, 1.883567935091437e+02, 0.25,_
 3171, 83,  191, 0, 4.5,   -1, 1.893451273852794e+02, 0,_
 3172, 83,  191, 1, 0.5,   +1, 1.893453870798091e+02, 0.244,_
 3173, 83,  192, 0, 3,     +1, 1.903362227305431e+02, 0,_
 3174, 83,  192, 1, 10,    -1, 1.903362440169799e+02, 0.02,_
 3175, 83,  193, 0, 4.5,   -1, 1.913251310680449e+02, 0,_
 3176, 83,  193, 1, 0.5,   +1, 1.913254588791725e+02, 0.308,_
 3177, 83,  194, 0, 3,     +1, 1.923163870573416e+02, 0,_
 3178, 83,  194, 1,-77.777,+0, 1.923165041327443e+02, 0.11,_
 3179, 83,  194, 2, 10,    -1, 1.923166318513655e+02, 0.23,_
 3180, 83,  195, 0, 4.5,   -1, 1.933056713918609e+02, 0,_
 3181, 83,  195, 1, 0.5,   +1, 1.933060960562763e+02, 0.399,_
 3182, 83,  196, 0, 3,     +1, 1.942970985063720e+02, 0,_
 3183, 83,  196, 1, 7,     +1, 1.942972762481198e+02, 0.167,_
 3184, 83,  196, 2, 10,    -1, 1.942973858732697e+02, 0.27,_
 3185, 83,  197, 0, 4.5,   -1, 1.952867219227754e+02, 0,_
 3186, 83,  197, 1, 0.5,   +1, 1.952871050786389e+02, 0.36,_
 3187, 83,  198, 0,-77.777,+0, 1.962784695837781e+02, 0,_
 3188, 83,  198, 1, 7,     +1, 1.962787675938942e+02, 0.28,_
 3189, 83,  198, 2, 10,    -1, 1.962790336743549e+02, 0.53,_
 3190, 83,  199, 0, 4.5,   -1, 1.972683591668695e+02, 0,_
 3191, 83,  199, 1, 0.5,   +1, 1.972690690695388e+02, 0.667,_
 3192, 83,  200, 0, 7,     +1, 1.982602229506779e+02, 0,_
 3193, 83,  200, 1, 2,     +1, 1.982603293828622e+02, 0.1,_
 3194, 83,  200, 2, 10,    -1, 1.982606786932911e+02, 0.4282,_
 3195, 83,  201, 0, 4.5,   -1, 1.992505195331539e+02, 0,_
 3196, 83,  201, 1, 0.5,   +1, 1.992514203219458e+02, 0.84635,_
 3197, 83,  202, 0, 5,     +1, 2.002426474165615e+02, 0,_
 3198, 83,  203, 0, 4.5,   -1, 2.012332232057350e+02, 0,_
 3199, 83,  203, 1, 0.5,   +1, 2.012343919801237e+02, 1.09814,_
 3200, 83,  204, 0, 6,     +1, 2.022255681919650e+02, 0,_
 3201, 83,  204, 1, 10,    -1, 2.022264255032095e+02, 0.8055,_
 3202, 83,  204, 2, 17,    +1, 2.022285838414749e+02, 2.8334,_
 3203, 83,  205, 0, 4.5,   -1, 2.032165321784415e+02, 0,_
 3204, 83,  206, 0, 6,     +1, 2.042090445760658e+02, 0,_
 3205, 83,  207, 0, 4.5,   -1, 2.052004260530224e+02, 0,_
 3206, 83,  208, 0, 5,     +1, 2.061930961601075e+02, 0,_
 3207, 83,  208, 1, 10,    -1, 2.061947683161550e+02, 1.5711,_
 3208, 83,  209, 0, 4.5,   -1, 2.071851565780973e+02, 0,_
 3209, 83,  210, 0, 1,     -1, 2.081802557644153e+02, 0,_
 3210, 83,  210, 1, 9,     -1, 2.081805378097037e+02, 0.265,_
 3211, 83,  211, 0, 4.5,   -1, 2.091747867788944e+02, 0,_
 3212, 83,  212, 0, 1,     -1, 2.101701781147612e+02, 0,_
 3213, 83,  212, 1, 9,     -1, 2.101704441952219e+02, 0.25,_
 3214, 83,  212, 2, 17,    -1, 2.101743608996042e+02, 3.93,_
 3215, 83,  213, 0, 4.5,   -1, 2.111646596083348e+02, 0,_
 3216, 83,  214, 0, 1,     -1, 2.121603592844592e+02, 0,_
 3217, 83,  215, 0, 4.5,   -1, 2.131547809157346e+02, 0,_
 3218, 83,  215, 1, 12.5,  -1, 2.131562150894181e+02, 1.3475,_
 3219, 83,  216, 0, 1,     -1, 2.141507081302579e+02, 0,_
 3220, 83,  217, 0, 4.5,   -1, 2.151451573227180e+02, 0,_
 3221, 83,  218, 0, 1,     -1, 2.161413414709325e+02, 0,_
 3222, 84,  188, 0, 0,     +1, 1.863844104284769e+02, 0,_
 3223, 84,  189, 0, 1.5,   -1, 1.873748858027372e+02, 0,_
 3224, 84,  190, 0, 0,     +1, 1.883629514014993e+02, 0,_
 3225, 84,  191, 0, 1.5,   -1, 1.893538235467721e+02, 0,_
 3226, 84,  191, 1, 6.5,   +1, 1.893538597337147e+02, 0.034,_
 3227, 84,  192, 0, 0,     +1, 1.903420380512784e+02, 0,_
 3228, 84,  193, 0, 1.5,   -1, 1.913331765347532e+02, 0,_
 3229, 84,  193, 1, 6.5,   +1, 1.913332829669375e+02, 0.1,_
 3230, 84,  194, 0, 0,     +1, 1.923217343762732e+02, 0,_
 3231, 84,  195, 0, 1.5,   -1, 1.933130889741349e+02, 0,_
 3232, 84,  195, 1, 6.5,   +1, 1.933131741198824e+02, 0.08,_
 3233, 84,  196, 0, 0,     +1, 1.943019262592993e+02, 0,_
 3234, 84,  197, 0, 1.5,   -1, 1.952934582312297e+02, 0,_
 3235, 84,  197, 1, 6.5,   +1, 1.952937030252536e+02, 0.23,_
 3236, 84,  198, 0, 0,     +1, 1.962826163246178e+02, 0,_
 3237, 84,  199, 0, 1.5,   -1, 1.972743077508897e+02, 0,_
 3238, 84,  199, 1, 6.5,   +1, 1.972746398193048e+02, 0.312,_
 3239, 84,  200, 0, 0,     +1, 1.982638725046004e+02, 0,_
 3240, 84,  201, 0, 1.5,   -1, 1.992557296529049e+02, 0,_
 3241, 84,  201, 1, 6.5,   +1, 1.992561810317985e+02, 0.4241,_
 3242, 84,  202, 0, 0,     +1, 2.002456273764599e+02, 0,_
 3243, 84,  203, 0, 2.5,   -1, 2.012377081936919e+02, 0,_
 3244, 84,  203, 1, 6.5,   +1, 2.012383909455110e+02, 0.64149,_
 3245, 84,  204, 0, 0,     +1, 2.022280210838631e+02, 0,_
 3246, 84,  205, 0, 2.5,   -1, 2.032203031839958e+02, 0,_
 3247, 84,  205, 1, 9.5,   -1, 2.032218583817160e+02, 1.46121,_
 3248, 84,  206, 0, 0,     +1, 2.042110025077525e+02, 0,_
 3249, 84,  207, 0, 2.5,   -1, 2.052035220078346e+02, 0,_
 3250, 84,  207, 1, 9.5,   -1, 2.052049939117274e+02, 1.38295,_
 3251, 84,  208, 0, 0,     +1, 2.061945868781610e+02, 0,_
 3252, 84,  209, 0, 0.5,   -1, 2.071871708813293e+02, 0,_
 3253, 84,  210, 0, 0,     +1, 2.081790199179211e+02, 0,_
 3254, 84,  211, 0, 4.5,   +1, 2.091741764543056e+02, 0,_
 3255, 84,  211, 1, 12.5,  +1, 2.091757335571619e+02, 1.463,_
 3256, 84,  212, 0, 0,     +1, 2.101677817589823e+02, 0,_
 3257, 84,  212, 1, 18,    +1, 2.101708917074075e+02, 2.922,_
 3258, 84,  213, 0, 4.5,   +1, 2.111631461969173e+02, 0,_
 3259, 84,  214, 0, 0,     +1, 2.121568797047247e+02, 0,_
 3260, 84,  215, 0, 4.5,   +1, 2.131524702425235e+02, 0,_
 3261, 84,  216, 0, 0,     +1, 2.141463533807954e+02, 0,_
 3262, 84,  217, 0, 2.5,   +1, 2.151421277893694e+02, 0,_
 3263, 84,  218, 0, 0,     +1, 2.161361697851544e+02, 0,_
 3264, 84,  219, 0, 3.5,   +1, 2.171321819218344e+02, 0,_
 3265, 84,  220, 0, 0,     +1, 2.181263396289901e+02, 0,_
 3266, 85,  193, 0, 4.5,   -1, 1.913419657023557e+02, 0,_
 3267, 85,  194, 0, 3,     +1, 1.923326803870116e+02, 0,_
 3268, 85,  194, 1, 10,    -1, 1.923329464674723e+02, 0.25,_
 3269, 85,  195, 0, 4.5,   -1, 1.933211628823995e+02, 0,_
 3270, 85,  195, 1, 0.5,   +1, 1.933211628823995e+02, 0,_
 3271, 85,  196, 0, 3,     +1, 1.943120994365746e+02, 0,_
 3272, 85,  196, 1, 10,    -1, 1.943123123009432e+02, 0.2,_
 3273, 85,  197, 0, 4.5,   -1, 1.953009114001484e+02, 0,_
 3274, 85,  197, 1, 0.5,   +1, 1.953009646162406e+02, 0.05,_
 3275, 85,  198, 0, 3,     +1, 1.962919385493255e+02, 0,_
 3276, 85,  198, 1, 10,    -1, 1.962922897755336e+02, 0.33,_
 3277, 85,  199, 0, 4.5,   -1, 1.972811035637434e+02, 0,_
 3278, 85,  200, 0, 3,     +1, 1.982723379819518e+02, 0,_
 3279, 85,  200, 1, 7,     +1, 1.982724582503200e+02, 0.113,_
 3280, 85,  200, 2, 10,    -1, 1.982727041086658e+02, 0.344,_
 3281, 85,  201, 0, 4.5,   -1, 1.992618300772415e+02, 0,_
 3282, 85,  202, 0,-77.777,+0, 2.002534510836953e+02, 0,_
 3283, 85,  202, 1, 7,     +1, 2.002536533048454e+02, 0.19,_
 3284, 85,  202, 2, 10,    -1, 2.002540683903642e+02, 0.58,_
 3285, 85,  203, 0, 4.5,   -1, 2.012431876764352e+02, 0,_
 3286, 85,  204, 0, 7,     +1, 2.022349027884720e+02, 0,_
 3287, 85,  204, 1, 10,    -1, 2.022355278646904e+02, 0.5873,_
 3288, 85,  205, 0, 4.5,   -1, 2.032251452647735e+02, 0,_
 3289, 85,  206, 0, 5,     +1, 2.042171318911612e+02, 0,_
 3290, 85,  207, 0, 4.5,   -1, 2.052076924027140e+02, 0,_
 3291, 85,  208, 0, 6,     +1, 2.061999081953952e+02, 0,_
 3292, 85,  209, 0, 4.5,   -1, 2.071908784237860e+02, 0,_
 3293, 85,  210, 0, 5,     +1, 2.081832569414687e+02, 0,_
 3294, 85,  211, 0, 4.5,   -1, 2.091750122739983e+02, 0,_
 3295, 85,  212, 0, 1,     -1, 2.101696350275660e+02, 0,_
 3296, 85,  212, 1, 9,     -1, 2.101698722649048e+02, 0.2229,_
 3297, 85,  213, 0, 4.5,   -1, 2.111632249455669e+02, 0,_
 3298, 85,  214, 0, 1,     -1, 2.121580400434870e+02, 0,_
 3299, 85,  215, 0, 4.5,   -1, 2.131517102636762e+02, 0,_
 3300, 85,  216, 0, 1,     -1, 2.141468581321673e+02, 0,_
 3301, 85,  217, 0, 4.5,   -1, 2.151405431392295e+02, 0,_
 3302, 85,  218, 0, 1,     -1, 2.161358944043063e+02, 0,_
 3303, 85,  219, 0, 3.5,   -1, 2.171297496443143e+02, 0,_
 3304, 85,  220, 0, 3,     +0, 2.181253948157175e+02, 0,_
 3305, 85,  221, 0, 1.5,   -1, 2.191193661378834e+02, 0,_
 3306, 85,  222, 0,-77.777,+0, 2.201152141982707e+02, 0,_
 3307, 85,  223, 0, 1.5,   -1, 2.211092578933316e+02, 0,_
 3308, 86,  195, 0, 1.5,   -1, 1.933302315174403e+02, 0,_
 3309, 86,  195, 1, 6.5,   +1, 1.933302847335324e+02, 0.05,_
 3310, 86,  196, 0, 0,     +1, 1.943183636814204e+02, 0,_
 3311, 86,  197, 0, 1.5,   -1, 1.953092829345079e+02, 0,_
 3312, 86,  197, 1, 6.5,   +1, 1.953094957988765e+02, 0.2,_
 3313, 86,  198, 0, 0,     +1, 1.962977754560203e+02, 0,_
 3314, 86,  199, 0, 1.5,   -1, 1.972888985746303e+02, 0,_
 3315, 86,  199, 1, 6.5,   +1, 1.972890582229068e+02, 0.15,_
 3316, 86,  200, 0, 0,     +1, 1.982776416332597e+02, 0,_
 3317, 86,  201, 0, 1.5,   -1, 1.992689792482763e+02, 0,_
 3318, 86,  201, 1, 6.5,   +1, 1.992692772583924e+02, 0.28,_
 3319, 86,  202, 0, 0,     +1, 2.002580448015171e+02, 0,_
 3320, 86,  203, 0,-77.777,+0, 2.012495830361813e+02, 0,_
 3321, 86,  203, 1, 6.5,   +1, 2.012499693850104e+02, 0.363,_
 3322, 86,  204, 0, 0,     +1, 2.022390592203397e+02, 0,_
 3323, 86,  205, 0, 2.5,   -1, 2.032307458987515e+02, 0,_
 3324, 86,  206, 0, 0,     +1, 2.042206406976484e+02, 0,_
 3325, 86,  207, 0, 2.5,   -1, 2.052125804650652e+02, 0,_
 3326, 86,  208, 0, 0,     +1, 2.062029034943816e+02, 0,_
 3327, 86,  209, 0, 2.5,   -1, 2.071950735176509e+02, 0,_
 3328, 86,  210, 0, 0,     +1, 2.081857766246263e+02, 0,_
 3329, 86,  211, 0, 0.5,   -1, 2.091780901435180e+02, 0,_
 3330, 86,  212, 0, 0,     +1, 2.101696016220222e+02, 0,_
 3331, 86,  213, 0, 4.5,   +1, 2.111641653470671e+02, 0,_
 3332, 86,  214, 0, 0,     +1, 2.121570396765820e+02, 0,_
 3333, 86,  215, 0, 4.5,   +1, 2.131518030675386e+02, 0,_
 3334, 86,  216, 0, 0,     +1, 2.141447254447245e+02, 0,_
 3335, 86,  217, 0, 4.5,   +1, 2.151397596550539e+02, 0,_
 3336, 86,  218, 0, 0,     +1, 2.161328282889378e+02, 0,_
 3337, 86,  219, 0, 2.5,   +1, 2.171280821985912e+02, 0,_
 3338, 86,  220, 0, 0,     +1, 2.181213890592600e+02, 0,_
 3339, 86,  221, 0, 3.5,   +1, 2.191169061623643e+02, 0,_
 3340, 86,  222, 0, 0,     +1, 2.201103387297946e+02, 0,_
 3341, 86,  223, 0, 3.5,   -1, 2.211060241980292e+02, 0,_
 3342, 86,  224, 0, 0,     +1, 2.220996212819381e+02, 0,_
 3343, 86,  225, 0, 3.5,   -1, 2.230953828616729e+02, 0,_
 3344, 86,  226, 0, 0,     +1, 2.240891477803081e+02, 0,_
 3345, 86,  227, 0, 2.5,   +1, 2.250849621466513e+02, 0,_
 3346, 86,  228, 0, 0,     +1, 2.260788809459237e+02, 0,_
 3347, 87,  199, 0, 1.5,   -1, 1.972977014147616e+02, 0,_
 3348, 87,  200, 0, 3,     +1, 1.982884309447836e+02, 0,_
 3349, 87,  200, 1, 10,    -1, 1.982886438091521e+02, 0.2,_
 3350, 87,  201, 0, 4.5,   -1, 1.992771329144454e+02, 0,_
 3351, 87,  202, 0, 3,     +1, 2.002680184249099e+02, 0,_
 3352, 87,  202, 1, 10,    -1, 2.002683696511181e+02, 0.33,_
 3353, 87,  203, 0, 4.5,   -1, 2.012570653426259e+02, 0,_
 3354, 87,  204, 0, 3,     +1, 2.022481884463647e+02, 0,_
 3355, 87,  204, 1, 7,     +1, 2.022482459197442e+02, 0.054,_
 3356, 87,  204, 2, 10,    -1, 2.022485396725729e+02, 0.33,_
 3357, 87,  205, 0, 4.5,   -1, 2.032375575305355e+02, 0,_
 3358, 87,  206, 0,-77.777,+0, 2.042290387819548e+02, 0,_
 3359, 87,  206, 1, 7,     +1, 2.042292410031050e+02, 0.19,_
 3360, 87,  206, 2, 10,    -1, 2.042298050936818e+02, 0.72,_
 3361, 87,  207, 0, 4.5,   -1, 2.052187433382875e+02, 0,_
 3362, 87,  208, 0, 7,     +1, 2.062103427567990e+02, 0,_
 3363, 87,  209, 0, 4.5,   -1, 2.072005776339465e+02, 0,_
 3364, 87,  210, 0, 6,     +1, 2.081924515896166e+02, 0,_
 3365, 87,  211, 0, 4.5,   -1, 2.091830021543576e+02, 0,_
 3366, 87,  212, 0, 5,     +1, 2.101750761110358e+02, 0,_
 3367, 87,  213, 0, 4.5,   -1, 2.111664463791164e+02, 0,_
 3368, 87,  214, 0, 1,     -1, 2.121606168992487e+02, 0,_
 3369, 87,  214, 1, 9,     -1, 2.121607467465136e+02, 0.122,_
 3370, 87,  215, 0, 4.5,   -1, 2.131533853154932e+02, 0,_
 3371, 87,  216, 0, 1,     -1, 2.141476183588669e+02, 0,_
 3372, 87,  217, 0, 4.5,   -1, 2.151404579444352e+02, 0,_
 3373, 87,  218, 0, 1,     -1, 2.161347885256517e+02, 0,_
 3374, 87,  218, 1,-77.777,+0, 2.161348800573302e+02, 0.086,_
 3375, 87,  219, 0, 4.5,   -1, 2.171278569503482e+02, 0,_
 3376, 87,  220, 0, 1,     +1, 2.181223152776556e+02, 0,_
 3377, 87,  221, 0, 2.5,   -1, 2.191156352229473e+02, 0,_
 3378, 87,  222, 0, 2,     -1, 2.201103450093825e+02, 0,_
 3379, 87,  223, 0, 1.5,   -1, 2.211038877382731e+02, 0,_
 3380, 87,  224, 0, 1,     -1, 2.220988800010744e+02, 0,_
 3381, 87,  225, 0, 1.5,   -1, 2.230924947905659e+02, 0,_
 3382, 87,  226, 0, 1,     -1, 2.240878422258278e+02, 0,_
 3383, 87,  227, 0, 0.5,   +1, 2.250815527101318e+02, 0,_
 3384, 87,  228, 0, 2,     -1, 2.260769021113587e+02, 0,_
 3385, 87,  229, 0, 0.5,   +1, 2.270707425853134e+02, 0,_
 3386, 87,  230, 0,-77.777,+0, 2.280662162260157e+02, 0,_
 3387, 87,  231, 0, 0.5,   +1, 2.290603863912234e+02, 0,_
 3388, 87,  232, 0,-77.777,+0, 2.300560449605348e+02, 0,_
 3389, 88,  202, 0, 0,     +1, 2.002743816064773e+02, 0,_
 3390, 88,  203, 0, 1.5,   -1, 2.012653514174848e+02, 0,_
 3391, 88,  203, 1, 6.5,   +1, 2.012655855682902e+02, 0.22,_
 3392, 88,  204, 0, 0,     +1, 2.022539884438746e+02, 0,_
 3393, 88,  205, 0, 1.5,   -1, 2.032451661594068e+02, 0,_
 3394, 88,  205, 1, 6.5,   +1, 2.032454748127413e+02, 0.29,_
 3395, 88,  206, 0, 0,     +1, 2.042341559937454e+02, 0,_
 3396, 88,  207, 0,-77.777,+0, 2.052255431058738e+02, 0,_
 3397, 88,  207, 1, 6.5,   +1, 2.052261391261059e+02, 0.56,_
 3398, 88,  208, 0, 0,     +1, 2.062150191472692e+02, 0,_
 3399, 88,  209, 0, 2.5,   -1, 2.072065674149979e+02, 0,_
 3400, 88,  210, 0, 0,     +1, 2.081964704653962e+02, 0,_
 3401, 88,  211, 0, 2.5,   -1, 2.091882942527704e+02, 0,_
 3402, 88,  212, 0, 0,     +1, 2.101786064668002e+02, 0,_
 3403, 88,  213, 0, 0.5,   -1, 2.111705955360211e+02, 0,_
 3404, 88,  213, 1, 8.5,   -1, 2.111724793856832e+02, 1.77,_
 3405, 88,  214, 0, 0,     +1, 2.121617359715812e+02, 0,_
 3406, 88,  215, 0, 4.5,   +1, 2.131557435059712e+02, 0,_
 3407, 88,  216, 0, 0,     +1, 2.141479590785595e+02, 0,_
 3408, 88,  217, 0, 4.5,   +1, 2.151421343227582e+02, 0,_
 3409, 88,  218, 0, 0,     +1, 2.161343543388438e+02, 0,_
 3410, 88,  219, 0, 3.5,   +1, 2.171286834121265e+02, 0,_
 3411, 88,  220, 0, 0,     +1, 2.181210252397257e+02, 0,_
 3412, 88,  221, 0, 2.5,   +1, 2.191153015809273e+02, 0,_
 3413, 88,  222, 0, 0,     +1, 2.201081547230294e+02, 0,_
 3414, 88,  223, 0, 1.5,   +1, 2.211026647414377e+02, 0,_
 3415, 88,  224, 0, 0,     +1, 2.220957693079683e+02, 0,_
 3416, 88,  225, 0, 0.5,   +1, 2.230905497402631e+02, 0,_
 3417, 88,  226, 0, 0,     +1, 2.240837416965686e+02, 0,_
 3418, 88,  227, 0, 1.5,   +1, 2.250788868664430e+02, 0,_
 3419, 88,  228, 0, 0,     +1, 2.260721723146490e+02, 0,_
 3420, 88,  229, 0, 2.5,   +1, 2.270674364843995e+02, 0,_
 3421, 88,  230, 0, 0,     +1, 2.280609260539116e+02, 0,_
 3422, 88,  231, 0, 2.5,   +1, 2.290562737558626e+02, 0,_
 3423, 88,  232, 0, 0,     +1, 2.300501104287531e+02, 0,_
 3424, 88,  233, 0, 0.5,   +1, 2.310456038649293e+02, 0,_
 3425, 88,  234, 0, 0,     +1, 2.320397769696662e+02, 0,_
 3426, 89,  206, 0, 3,     +1, 2.042447075890603e+02, 0,_
 3427, 89,  206, 1,-77.777,+0, 2.042447927348077e+02, 0.08,_
 3428, 89,  206, 2, 10,    -1, 2.042450055991763e+02, 0.28,_
 3429, 89,  207, 0, 4.5,   -1, 2.052336338125813e+02, 0,_
 3430, 89,  208, 0, 3,     +1, 2.062246250568711e+02, 0,_
 3431, 89,  208, 1, 10,    -1, 2.062251678610111e+02, 0.51,_
 3432, 89,  209, 0, 4.5,   -1, 2.072140023330587e+02, 0,_
 3433, 89,  210, 0, 7,     +1, 2.082053542620461e+02, 0,_
 3434, 89,  211, 0, 4.5,   -1, 2.091950741862177e+02, 0,_
 3435, 89,  212, 0, 6,     +1, 2.101865636207313e+02, 0,_
 3436, 89,  213, 0, 4.5,   -1, 2.111767783256693e+02, 0,_
 3437, 89,  214, 0, 5,     +1, 2.121684956079015e+02, 0,_
 3438, 89,  215, 0, 4.5,   -1, 2.131594653088625e+02, 0,_
 3439, 89,  216, 0, 1,     -1, 2.141531245700109e+02, 0,_
 3440, 89,  216, 1, 9,     -1, 2.141531639499191e+02, 0.037,_
 3441, 89,  217, 0, 4.5,   -1, 2.151451293421671e+02, 0,_
 3442, 89,  218, 0, 1,     -1, 2.161388164430862e+02, 0,_
 3443, 89,  219, 0, 4.5,   -1, 2.171309985238780e+02, 0,_
 3444, 89,  220, 0, 3,     -1, 2.181247220947878e+02, 0,_
 3445, 89,  221, 0, 4.5,   -1, 2.191169611756785e+02, 0,_
 3446, 89,  222, 0, 1,     -1, 2.201106040319736e+02, 0,_
 3447, 89,  222, 1,-77.777,+0, 2.201108168963422e+02, 0.2,_
 3448, 89,  223, 0, 2.5,   -1, 2.211032954295781e+02, 0,_
 3449, 89,  224, 0, 0,     -1, 2.220972681070009e+02, 0,_
 3450, 89,  225, 0, 1.5,   -1, 2.230901710941993e+02, 0,_
 3451, 89,  226, 0, 1,     -1, 2.240844243950438e+02, 0,_
 3452, 89,  227, 0, 1.5,   -1, 2.250774733068181e+02, 0,_
 3453, 89,  228, 0, 3,     +1, 2.260721238446376e+02, 0,_
 3454, 89,  229, 0, 1.5,   +1, 2.270654440417473e+02, 0,_
 3455, 89,  230, 0, 1,     +1, 2.280602045258912e+02, 0,_
 3456, 89,  231, 0, 0.5,   +1, 2.290536622979258e+02, 0,_
 3457, 89,  232, 0, 1,     +1, 2.300486815399543e+02, 0,_
 3458, 89,  233, 0, 0.5,   +1, 2.310423831987308e+02, 0,_
 3459, 89,  234, 0,-77.777,+0, 2.320375531350064e+02, 0,_
 3460, 89,  235, 0, 0.5,   +1, 2.330316404520862e+02, 0,_
 3461, 89,  236, 0,-77.777,+0, 2.340271623387413e+02, 0,_
 3462, 90,  209, 0, 2.5,   -1, 2.072220097296414e+02, 0,_
 3463, 90,  210, 0, 0,     +1, 2.082109629700632e+02, 0,_
 3464, 90,  211, 0, 2.5,   -1, 2.092022136126884e+02, 0,_
 3465, 90,  212, 0, 0,     +1, 2.101917080308506e+02, 0,_
 3466, 90,  213, 0, 2.5,   -1, 2.111831274252558e+02, 0,_
 3467, 90,  214, 0, 0,     +1, 2.121730200727941e+02, 0,_
 3468, 90,  215, 0, 0.5,   -1, 2.131646708760609e+02, 0,_
 3469, 90,  216, 0, 0,     +1, 2.141554170528401e+02, 0,_
 3470, 90,  217, 0, 4.5,   +1, 2.151488567118228e+02, 0,_
 3471, 90,  218, 0, 0,     +1, 2.161404375453714e+02, 0,_
 3472, 90,  219, 0, 4.5,   +1, 2.171340870907065e+02, 0,_
 3473, 90,  220, 0, 0,     +1, 2.181257070363520e+02, 0,_
 3474, 90,  221, 0, 3.5,   +1, 2.191195339200650e+02, 0,_
 3475, 90,  222, 0, 0,     +1, 2.201112230809662e+02, 0,_
 3476, 90,  223, 0, 2.5,   +1, 2.211049557173245e+02, 0,_
 3477, 90,  224, 0, 0,     +1, 2.220970122420491e+02, 0,_
 3478, 90,  225, 0, 1.5,   +1, 2.230908871496392e+02, 0,_
 3479, 90,  226, 0, 0,     +1, 2.240832412637766e+02, 0,_
 3480, 90,  227, 0, 0.5,   +1, 2.250774256715735e+02, 0,_
 3481, 90,  228, 0, 0,     +1, 2.260698634983479e+02, 0,_
 3482, 90,  229, 0, 2.5,   +1, 2.270642686573541e+02, 0,_
 3483, 90,  229, 1, 1.5,   +1, 2.270642686610793e+02, 3.5e-06,_
 3484, 90,  230, 0, 0,     +1, 2.280570373273995e+02, 0,_
 3485, 90,  231, 0, 2.5,   +1, 2.290515901062918e+02, 0,_
 3486, 90,  232, 0, 0,     +1, 2.300447354217365e+02, 0,_
 3487, 90,  233, 0, 0.5,   +1, 2.310396411662118e+02, 0,_
 3488, 90,  234, 0, 0,     +1, 2.320330529883979e+02, 0,_
 3489, 90,  235, 0, 2.5,   +1, 2.330280862489571e+02, 0,_
 3490, 90,  236, 0, 0,     +1, 2.340218771345904e+02, 0,_
 3491, 90,  237, 0, 2.5,   +1, 2.350172245331700e+02, 0,_
 3492, 90,  238, 0, 0,     +1, 2.360113693520020e+02, 0,_
 3493, 91,  212, 0, 7,     +1, 2.102018005182965e+02, 0,_
 3494, 91,  213, 0, 4.5,   -1, 2.111911551159913e+02, 0,_
 3495, 91,  214, 0,-77.777,+0, 2.121823761333172e+02, 0,_
 3496, 91,  215, 0, 4.5,   -1, 2.131720597748747e+02, 0,_
 3497, 91,  216, 0,-77.777,+0, 2.141634004057591e+02, 0,_
 3498, 91,  217, 0, 4.5,   -1, 2.151540321153866e+02, 0,_
 3499, 91,  217, 1, 14.5,  +1, 2.151560053680835e+02, 1.854,_
 3500, 91,  218, 0,-77.777,+0, 2.161471608990775e+02, 0,_
 3501, 91,  219, 0, 4.5,   -1, 2.171384175407216e+02, 0,_
 3502, 91,  220, 0, 1,     -1, 2.181316129362268e+02, 0,_
 3503, 91,  221, 0, 4.5,   -1, 2.191231908431344e+02, 0,_
 3504, 91,  222, 0,-77.777,+0, 2.201164931165515e+02, 0,_
 3505, 91,  223, 0, 4.5,   -1, 2.211080793374194e+02, 0,_
 3506, 91,  224, 0, 5,     -1, 2.221011296183302e+02, 0,_
 3507, 91,  225, 0, 2.5,   -1, 2.230930483599340e+02, 0,_
 3508, 91,  226, 0,-77.777,+0, 2.240862592987576e+02, 0,_
 3509, 91,  227, 0, 2.5,   -1, 2.250785180650757e+02, 0,_
 3510, 91,  228, 0, 3,     +1, 2.260721545594959e+02, 0,_
 3511, 91,  229, 0, 2.5,   +1, 2.270646000072353e+02, 0,_
 3512, 91,  230, 0, 2,     -1, 2.280584326678916e+02, 0,_
 3513, 91,  231, 0, 1.5,   -1, 2.290511734386816e+02, 0,_
 3514, 91,  232, 0, 2,     -1, 2.300452674229904e+02, 0,_
 3515, 91,  233, 0, 1.5,   -1, 2.310383190195017e+02, 0,_
 3516, 91,  234, 0, 4,     +1, 2.320327612711037e+02, 0,_
 3517, 91,  234, 1, 0,     -1, 2.320328506741386e+02, 0.084,_
 3518, 91,  235, 0, 1.5,   -1, 2.330262461928880e+02, 0,_
 3519, 91,  236, 0, 1,     -1, 2.340208966305751e+02, 0,_
 3520, 91,  237, 0, 0.5,   +1, 2.350146409199610e+02, 0,_
 3521, 91,  238, 0, 3,     -1, 2.360096333939325e+02, 0,_
 3522, 91,  239, 0,-77.777,+0, 2.370036433810697e+02, 0,_
 3523, 91,  240, 0,-77.777,+0, 2.379988549565451e+02, 0,_
 3524, 92,  217, 0, 0.5,   -1, 2.151603169656878e+02, 0,_
 3525, 92,  218, 0, 0,     +1, 2.161505782638990e+02, 0,_
 3526, 92,  219, 0, 4.5,   +1, 2.171434692788361e+02, 0,_
 3527, 92,  220, 0, 0,     +1, 2.181345028949775e+02, 0,_
 3528, 92,  221, 0, 4.5,   +1, 2.191276010817878e+02, 0,_
 3529, 92,  222, 0, 0,     +1, 2.201187475352036e+02, 0,_
 3530, 92,  223, 0, 4.5,   +1, 2.211118218449070e+02, 0,_
 3531, 92,  224, 0, 0,     +1, 2.221031092291690e+02, 0,_
 3532, 92,  225, 0, 2.5,   +1, 2.230962830426803e+02, 0,_
 3533, 92,  226, 0, 0,     +1, 2.240876382274565e+02, 0,_
 3534, 92,  227, 0, 1.5,   +1, 2.250808747545597e+02, 0,_
 3535, 92,  228, 0, 0,     +1, 2.260724724083622e+02, 0,_
 3536, 92,  229, 0, 1.5,   +1, 2.270659981494507e+02, 0,_
 3537, 92,  230, 0, 0,     +1, 2.280578381324908e+02, 0,_
 3538, 92,  231, 0, 2.5,   +1, 2.290515795963595e+02, 0,_
 3539, 92,  232, 0, 0,     +1, 2.300438443141090e+02, 0,_
 3540, 92,  233, 0, 2.5,   +1, 2.310377120409200e+02, 0,_
 3541, 92,  234, 0, 0,     +1, 2.320304262587889e+02, 0,_
 3542, 92,  235, 0, 3.5,   -1, 2.330247880178520e+02, 0,_
 3543, 92,  235, 1, 0.5,   +1, 2.330247880995919e+02, 7.68e-05,_
 3544, 92,  236, 0, 0,     +1, 2.340178214775175e+02, 0,_
 3545, 92,  237, 0, 0.5,   +1, 2.350123660118483e+02, 0,_
 3546, 92,  238, 0, 0,     +1, 2.360058164633150e+02, 0,_
 3547, 92,  239, 0, 2.5,   +1, 2.370007009252021e+02, 0,_
 3548, 92,  240, 0, 0,     +1, 2.379943910608258e+02, 0,_
 3549, 92,  241, 0, 3.5,   +1, 2.389895060482287e+02, 0,_
 3550, 92,  242, 0, 0,     +1, 2.399834942243565e+02, 0,_
 3551, 93,  225, 0, 4.5,   -1, 2.231007614794031e+02, 0,_
 3552, 93,  226, 0,-77.777,+0, 2.240934372305825e+02, 0,_
 3553, 93,  227, 0, 2.5,   -1, 2.250846175683930e+02, 0,_
 3554, 93,  228, 0,-77.777,+0, 2.260771271861050e+02, 0,_
 3555, 93,  229, 0, 2.5,   +1, 2.270687325213484e+02, 0,_
 3556, 93,  230, 0,-77.777,+0, 2.280616923500203e+02, 0,_
 3557, 93,  231, 0, 2.5,   +1, 2.290535150636979e+02, 0,_
 3558, 93,  232, 0, 4,     +1, 2.300467710938093e+02, 0,_
 3559, 93,  233, 0, 2.5,   +1, 2.310388076693915e+02, 0,_
 3560, 93,  234, 0, 0,     +1, 2.320323525179157e+02, 0,_
 3561, 93,  235, 0, 2.5,   +1, 2.330249202728733e+02, 0,_
 3562, 93,  236, 0, 6,     -1, 2.340188150592157e+02, 0,_
 3563, 93,  236, 1, 1,     +1, 2.340188789185263e+02, 0.06,_
 3564, 93,  237, 0, 2.5,   +1, 2.350118141239108e+02, 0,_
 3565, 93,  238, 0, 2,     +1, 2.360059727838110e+02, 0,_
 3566, 93,  239, 0, 2.5,   +1, 2.369993581115692e+02, 0,_
 3567, 93,  240, 0, 5,     +1, 2.379939661476626e+02, 0,_
 3568, 93,  240, 1, 1,     -1, 2.379939874340994e+02, 0.02,_
 3569, 93,  241, 0, 2.5,   +1, 2.389874446074396e+02, 0,_
 3570, 93,  242, 0, 1,     +1, 2.399822139329735e+02, 0,_
 3571, 93,  242, 1, 6,     +1, 2.399822139329735e+02, 0,_
 3572, 93,  243, 0, 2.5,   -1, 2.409762401643557e+02, 0,_
 3573, 93,  244, 0, 7,     -1, 2.419711900077179e+02, 0,_
 3574, 94,  228, 0, 0,     +1, 2.260797791322151e+02, 0,_
 3575, 94,  229, 0, 1.5,   +1, 2.270725810194365e+02, 0,_
 3576, 94,  230, 0, 0,     +1, 2.280634996766869e+02, 0,_
 3577, 94,  231, 0, 1.5,   +1, 2.290563722266217e+02, 0,_
 3578, 94,  232, 0, 0,     +1, 2.300478393633437e+02, 0,_
 3579, 94,  233, 0, 2.5,   +1, 2.310410461293247e+02, 0,_
 3580, 94,  234, 0, 0,     +1, 2.320327730321949e+02, 0,_
 3581, 94,  235, 0, 2.5,   +1, 2.330261328569345e+02, 0,_
 3582, 94,  236, 0, 0,     +1, 2.340183078184146e+02, 0,_
 3583, 94,  237, 0, 3.5,   -1, 2.350120483414357e+02, 0,_
 3584, 94,  237, 1, 0.5,   +1, 2.350122032470940e+02, 0.145544,_
 3585, 94,  238, 0, 0,     +1, 2.360045982728330e+02, 0,_
 3586, 94,  239, 0, 0.5,   +1, 2.369985888471804e+02, 0,_
 3587, 94,  240, 0, 0,     +1, 2.379916343346262e+02, 0,_
 3588, 94,  241, 0, 2.5,   +1, 2.389860556674874e+02, 0,_
 3589, 94,  242, 0, 0,     +1, 2.399793402630812e+02, 0,_
 3590, 94,  243, 0, 3.5,   +1, 2.409739828428565e+02, 0,_
 3591, 94,  244, 0, 0,     +1, 2.419675757400430e+02, 0,_
 3592, 94,  245, 0, 4.5,   -1, 2.429625743141573e+02, 0,_
 3593, 94,  246, 0, 0,     +1, 2.439563430328652e+02, 0,_
 3594, 94,  247, 0, 0.5,   +1, 2.449517041039735e+02, 0,_
 3595, 95,  231, 0,-77.777,+0, 2.290607369962602e+02, 0,_
 3596, 95,  232, 0,-77.777,+0, 2.300531359429275e+02, 0,_
 3597, 95,  233, 0,-77.777,+0, 2.310444641673132e+02, 0,_
 3598, 95,  234, 0,-77.777,+0, 2.320371486399222e+02, 0,_
 3599, 95,  235, 0, 2.5,   -1, 2.330287330157770e+02, 0,_
 3600, 95,  236, 0,-77.777,+0, 2.340216491104008e+02, 0,_
 3601, 95,  237, 0, 2.5,   -1, 2.350136217509745e+02, 0,_
 3602, 95,  238, 0, 1,     +1, 2.360070018034426e+02, 0,_
 3603, 95,  239, 0, 2.5,   -1, 2.369994425836259e+02, 0,_
 3604, 95,  240, 0, 3,     -1, 2.379931081957437e+02, 0,_
 3605, 95,  241, 0, 2.5,   -1, 2.389860335501324e+02, 0,_
 3606, 95,  242, 0, 1,     -1, 2.399801397188642e+02, 0,_
 3607, 95,  242, 1, 5,     -1, 2.399801914768354e+02, 0.04863,_
 3608, 95,  242, 2,-77.777,+0, 2.399824812269188e+02, 2.2,_
 3609, 95,  243, 0, 2.5,   -1, 2.409733660086728e+02, 0,_
 3610, 95,  244, 0, 6,     -1, 2.419676536142694e+02, 0,_
 3611, 95,  244, 1, 0,     +1, 2.419677483389134e+02, 0.089,_
 3612, 95,  245, 0, 2.5,   +1, 2.429612144195298e+02, 0,_
 3613, 95,  246, 0, 7,     +1, 2.439559165195670e+02, 0,_
 3614, 95,  246, 1, 2,     +1, 2.439559218411762e+02, 0.005,_
 3615, 95,  247, 0, 2.5,   +0, 2.449496241268006e+02, 0,_
 3616, 95,  248, 0,-77.777,+0, 2.459446622056100e+02, 0,_
 3617, 95,  249, 0,-77.777,+0, 2.469387762907469e+02, 0,_
 3618, 96,  233, 0, 1.5,   +1, 2.310487542005365e+02, 0,_
 3619, 96,  234, 0, 0,     +1, 2.320395577244080e+02, 0,_
 3620, 96,  235, 0, 2.5,   +1, 2.330323612068072e+02, 0,_
 3621, 96,  236, 0, 0,     +1, 2.340235798863894e+02, 0,_
 3622, 96,  237, 0, 2.5,   +1, 2.350164709855963e+02, 0,_
 3623, 96,  238, 0, 0,     +1, 2.360080913506081e+02, 0,_
 3624, 96,  239, 0, 3.5,   -1, 2.370013121737846e+02, 0,_
 3625, 96,  240, 0, 0,     +1, 2.379933361059212e+02, 0,_
 3626, 96,  241, 0, 0.5,   +1, 2.389868503476506e+02, 0,_
 3627, 96,  242, 0, 0,     +1, 2.399794326802939e+02, 0,_
 3628, 96,  243, 0, 2.5,   +1, 2.409733734075620e+02, 0,_
 3629, 96,  244, 0, 0,     +1, 2.419661345072043e+02, 0,_
 3630, 96,  244, 1, 6,     +1, 2.419672416041421e+02, 1.04019,_
 3631, 96,  245, 0, 3.5,   +1, 2.429602609046559e+02, 0,_
 3632, 96,  246, 0, 0,     +1, 2.439533865237676e+02, 0,_
 3633, 96,  247, 0, 4.5,   -1, 2.449478997939992e+02, 0,_
 3634, 96,  248, 0, 0,     +1, 2.459412885391459e+02, 0,_
 3635, 96,  249, 0, 0.5,   +1, 2.469362719962479e+02, 0,_
 3636, 96,  250, 0, 0,     +1, 2.479300644185659e+02, 0,_
 3637, 96,  251, 0, 0.5,   +1, 2.489253676795938e+02, 0,_
 3638, 96,  252, 0, 0,     +1, 2.499193399574785e+02, 0,_
 3639, 97,  235, 0,-77.777,+0, 2.330373311427307e+02, 0,_
 3640, 97,  236, 0,-77.777,+0, 2.340296329312651e+02, 0,_
 3641, 97,  237, 0, 3.5,   +1, 2.350206657156138e+02, 0,_
 3642, 97,  238, 0,-77.777,+0, 2.360131687602807e+02, 0,_
 3643, 97,  239, 0, 3.5,   +1, 2.370046149623993e+02, 0,_
 3644, 97,  240, 0,-77.777,+0, 2.379975294420169e+02, 0,_
 3645, 97,  241, 0, 3.5,   +1, 2.389893305687437e+02, 0,_
 3646, 97,  242, 0, 2,     -1, 2.399825513939030e+02, 0,_
 3647, 97,  243, 0, 1.5,   -1, 2.409749780802086e+02, 0,_
 3648, 97,  244, 0, 4,     -1, 2.419685419915550e+02, 0,_
 3649, 97,  245, 0, 1.5,   -1, 2.429611222134874e+02, 0,_
 3650, 97,  246, 0, 2,     -1, 2.439548233587138e+02, 0,_
 3651, 97,  247, 0, 1.5,   -1, 2.449478534099134e+02, 0,_
 3652, 97,  248, 0, 6,     +1, 2.459420200992393e+02, 0,_
 3653, 97,  248, 1, 1,     -1, 2.459420520288946e+02, 0.03,_
 3654, 97,  249, 0, 3.5,   +1, 2.469353095120921e+02, 0,_
 3655, 97,  250, 0, 2,     -1, 2.479300222549105e+02, 0,_
 3656, 97,  251, 0, 1.5,   -1, 2.489238563422050e+02, 0,_
 3657, 97,  252, 0,-77.777,+0, 2.499187847681473e+02, 0,_
 3658, 97,  253, 0,-77.777,+0, 2.509127422105799e+02, 0,_
 3659, 97,  254, 0,-77.777,+0, 2.519078397739606e+02, 0,_
 3660, 98,  237, 0, 2.5,   +1, 2.350257218972258e+02, 0,_
 3661, 98,  238, 0, 0,     +1, 2.360164275233728e+02, 0,_
 3662, 98,  239, 0, 2.5,   +1, 2.370088919030684e+02, 0,_
 3663, 98,  240, 0, 0,     +1, 2.380000058263551e+02, 0,_
 3664, 98,  241, 0, 3.5,   -1, 2.389928371842158e+02, 0,_
 3665, 98,  242, 0, 0,     +1, 2.399843106828127e+02, 0,_
 3666, 98,  243, 0, 0.5,   +1, 2.409774258901416e+02, 0,_
 3667, 98,  244, 0, 0,     +1, 2.419693554470318e+02, 0,_
 3668, 98,  245, 0, 2.5,   +1, 2.429627946618630e+02, 0,_
 3669, 98,  246, 0, 0,     +1, 2.439549546163771e+02, 0,_
 3670, 98,  247, 0, 3.5,   +1, 2.449485072663028e+02, 0,_
 3671, 98,  248, 0, 0,     +1, 2.459411238432223e+02, 0,_
 3672, 98,  249, 0, 4.5,   -1, 2.469351779609715e+02, 0,_
 3673, 98,  250, 0, 0,     +1, 2.479281282007305e+02, 0,_
 3674, 98,  251, 0, 0.5,   +1, 2.489226930381368e+02, 0,_
 3675, 98,  252, 0, 0,     +1, 2.499161243435029e+02, 0,_
 3676, 98,  253, 0, 3.5,   +1, 2.509110109498088e+02, 0,_
 3677, 98,  254, 0, 0,     +1, 2.519045915098953e+02, 0,_
 3678, 98,  255, 0, 3.5,   +1, 2.528996924539823e+02, 0,_
 3679, 98,  256, 0, 0,     +1, 2.538934763997490e+02, 0,_
 3680, 99,  240, 0,-77.777,+0, 2.380066127360388e+02, 0,_
 3681, 99,  241, 0, 1.5,   -1, 2.389976653485780e+02, 0,_
 3682, 99,  242, 0,-77.777,+0, 2.399900732179310e+02, 0,_
 3683, 99,  243, 0, 1.5,   -1, 2.409814252361452e+02, 0,_
 3684, 99,  244, 0,-77.777,+0, 2.419741949699729e+02, 0,_
 3685, 99,  245, 0, 1.5,   -1, 2.429659673458235e+02, 0,_
 3686, 99,  246, 0, 4,     -1, 2.439590097172693e+02, 0,_
 3687, 99,  247, 0, 3.5,   +1, 2.449511409159503e+02, 0,_
 3688, 99,  248, 0,-77.777,+0, 2.459443816367159e+02, 0,_
 3689, 99,  249, 0, 3.5,   +1, 2.469367230816310e+02, 0,_
 3690, 99,  250, 0, 6,     +1, 2.479303156853603e+02, 0,_
 3691, 99,  250, 1, 1,     -1, 2.479305285497289e+02, 0.2,_
 3692, 99,  251, 0, 1.5,   -1, 2.489230945639495e+02, 0,_
 3693, 99,  252, 0, 5,     -1, 2.499174653894526e+02, 0,_
 3694, 99,  253, 0, 3.5,   +1, 2.509107012007582e+02, 0,_
 3695, 99,  254, 0, 7,     +1, 2.519052824598719e+02, 0,_
 3696, 99,  254, 1, 2,     +1, 2.519053715436101e+02, 0.0837,_
 3697, 99,  255, 0, 3.5,   +1, 2.528989256512633e+02, 0,_
 3698, 99,  256, 0,-77.777,+0, 2.538936320510436e+02, 0,_
 3699, 99,  256, 1, 8,     +1, 2.538936320510436e+02, 0,_
 3700, 99,  257, 0, 3.5,   +1, 2.548874011256675e+02, 0,_
 3701, 99,  258, 0,-77.777,+0, 2.558823212267440e+02, 0,_
 3702, 100, 242, 0, 0,     +1, 2.399939030329066e+02, 0,_
 3703, 100, 243, 0, 3.5,   -1, 2.409863634469641e+02, 0,_
 3704, 100, 244, 0, 0,     +1, 2.419773248498274e+02, 0,_
 3705, 100, 245, 0, 0.5,   +1, 2.429700341076745e+02, 0,_
 3706, 100, 246, 0, 0,     +1, 2.439614454270491e+02, 0,_
 3707, 100, 247, 0, 2.5,   +1, 2.449544344457915e+02, 0,_
 3708, 100, 248, 0, 0,     +1, 2.459460834189164e+02, 0,_
 3709, 100, 249, 0, 3.5,   +1, 2.469392185565509e+02, 0,_
 3710, 100, 250, 0, 0,     +1, 2.479312167060915e+02, 0,_
 3711, 100, 250, 1,-77.777,+0, 2.479328131888560e+02, 1.5,_
 3712, 100, 251, 0, 4.5,   -1, 2.489246289337642e+02, 0,_
 3713, 100, 252, 0, 0,     +1, 2.499169555898235e+02, 0,_
 3714, 100, 253, 0, 0.5,   +1, 2.509110579644311e+02, 0,_
 3715, 100, 254, 0, 0,     +1, 2.519041246908430e+02, 0,_
 3716, 100, 255, 0, 3.5,   +1, 2.528986174022154e+02, 0,_
 3717, 100, 256, 0, 0,     +1, 2.538918226077178e+02, 0,_
 3718, 100, 257, 0, 4.5,   +1, 2.548865349480239e+02, 0,_
 3719, 100, 258, 0, 0,     +1, 2.558798992132868e+02, 0,_
 3720, 100, 259, 0, 1.5,   +1, 2.568747975033539e+02, 0,_
 3721, 101, 245, 0, 0.5,   -1, 2.429754462122440e+02, 0,_
 3722, 101, 245, 1, 3.5,   +1, 2.429755526444283e+02, 0.1,_
 3723, 101, 246, 0,-77.777,+0, 2.439677529578259e+02, 0,_
 3724, 101, 247, 0, 0.5,   -1, 2.449589721271644e+02, 0,_
 3725, 101, 247, 1, 3.5,   +1, 2.449590253432565e+02, 0.05,_
 3726, 101, 248, 0,-77.777,+0, 2.459516714709162e+02, 0,_
 3727, 101, 249, 0, 3.5,   -1, 2.469431702177393e+02, 0,_
 3728, 101, 249, 1, 0.5,   -1, 2.469432766499236e+02, 0.1,_
 3729, 101, 250, 0,-77.777,+0, 2.479360678433952e+02, 0,_
 3730, 101, 251, 0, 3.5,   -1, 2.489278355506984e+02, 0,_
 3731, 101, 252, 0,-77.777,+0, 2.499208885391486e+02, 0,_
 3732, 101, 253, 0, 3.5,   -1, 2.509130029512837e+02, 0,_
 3733, 101, 254, 0, 0,     -1, 2.519068384503454e+02, 0,_
 3734, 101, 254, 1, 3,     -1, 2.519068916664375e+02, 0.05,_
 3735, 101, 255, 0, 3.5,   -1, 2.528997279335548e+02, 0,_
 3736, 101, 256, 0, 1,     -1, 2.538939185683949e+02, 0,_
 3737, 101, 257, 0, 3.5,   -1, 2.548869638912668e+02, 0,_
 3738, 101, 258, 0, 8,     -1, 2.558812404168706e+02, 0,_
 3739, 101, 258, 1, 1,     -1, 2.558812404168706e+02, 0,_
 3740, 101, 259, 0, 3.5,   -1, 2.568747122421352e+02, 0,_
 3741, 101, 260, 0,-77.777,+0, 2.578692377622228e+02, 0,_
 3742, 101, 261, 0, 3.5,   -1, 2.588628035978950e+02, 0,_
 3743, 102, 249, 0, 2.5,   +1, 2.469480132532444e+02, 0,_
 3744, 102, 250, 0, 0,     +1, 2.479391897919735e+02, 0,_
 3745, 102, 251, 0, 3.5,   +1, 2.489319674570774e+02, 0,_
 3746, 102, 251, 1, 4.5,   -1, 2.489320845324802e+02, 0.11,_
 3747, 102, 252, 0, 0,     +1, 2.499234009106610e+02, 0,_
 3748, 102, 253, 0, 4.5,   -1, 2.509163934042938e+02, 0,_
 3749, 102, 254, 0, 0,     +1, 2.519081909897057e+02, 0,_
 3750, 102, 254, 1,-77.777,+0, 2.519087231506272e+02, 0.5,_
 3751, 102, 255, 0, 0.5,   +1, 2.529018184365226e+02, 0,_
 3752, 102, 256, 0, 0,     +1, 2.538943080594874e+02, 0,_
 3753, 102, 257, 0, 1.5,   +1, 2.548882987666836e+02, 0,_
 3754, 102, 258, 0, 0,     +1, 2.558810175232253e+02, 0,_
 3755, 102, 259, 0, 4.5,   +1, 2.568751955572503e+02, 0,_
 3756, 102, 260, 0, 0,     +1, 2.578682364386076e+02, 0,_
 3757, 102, 261, 0, 1.5,   +1, 2.588626727318384e+02, 0,_
 3758, 102, 262, 0, 0,     +1, 2.598558340724265e+02, 0,_
 3759, 103, 251, 0,-77.777,+0, 2.489371604601430e+02, 0,_
 3760, 103, 252, 0,-77.777,+0, 2.499296436766195e+02, 0,_
 3761, 103, 253, 0, 3.5,   -1, 2.509208806913294e+02, 0,_
 3762, 103, 253, 1, 0.5,   -1, 2.509209126209846e+02, 0.03,_
 3763, 103, 254, 0,-77.777,+0, 2.519136702533475e+02, 0,_
 3764, 103, 255, 0, 3.5,   -1, 2.529051604780144e+02, 0,_
 3765, 103, 256, 0,-77.777,+0, 2.538984850293947e+02, 0,_
 3766, 103, 257, 0, 4.5,   +1, 2.548908720503969e+02, 0,_
 3767, 103, 258, 0,-77.777,+0, 2.558845350442021e+02, 0,_
 3768, 103, 259, 0, 4.5,   +1, 2.568770827022974e+02, 0,_
 3769, 103, 260, 0,-77.777,+0, 2.578710728612442e+02, 0,_
 3770, 103, 261, 0,-77.777,+0, 2.588638465607100e+02, 0,_
 3771, 103, 262, 0,-77.777,+0, 2.598579636200754e+02, 0,_
 3772, 103, 263, 0,-77.777,+0, 2.608511051324732e+02, 0,_
 3773, 104, 253, 0, 3.5,   +1, 2.509261837408517e+02, 0,_
 3774, 104, 254, 0, 0,     +1, 2.519172115681528e+02, 0,_
 3775, 104, 255, 0, 4.5,   -1, 2.529098246592764e+02, 0,_
 3776, 104, 255, 1, 2.5,   +1, 2.529098246592764e+02, 0,_
 3777, 104, 256, 0, 0,     +1, 2.539011197061425e+02, 0,_
 3778, 104, 257, 0, 0.5,   +1, 2.548942793742228e+02, 0,_
 3779, 104, 257, 1, 5.5,   -1, 2.548944007069129e+02, 0.114,_
 3780, 104, 258, 0, 0,     +1, 2.558861940312193e+02, 0,_
 3781, 104, 259, 0, 3.5,   +1, 2.568797545509537e+02, 0,_
 3782, 104, 260, 0, 0,     +1, 2.578719998291454e+02, 0,_
 3783, 104, 261, 0, 1.5,   +1, 2.588657203147887e+02, 0,_
 3784, 104, 261, 1, 4.5,   +1, 2.588657948173177e+02, 0.07,_
 3785, 104, 262, 0, 0,     +1, 2.598582729398456e+02, 0,_
 3786, 104, 262, 1,-77.777,+0, 2.598589115329514e+02, 0.6,_
 3787, 104, 263, 0, 1.5,   +1, 2.608521976657642e+02, 0,_
 3788, 104, 264, 0, 0,     +1, 2.618450130044298e+02, 0,_
 3789, 104, 265, 0, 1.5,   +1, 2.628392034280997e+02, 0,_
 3790, 105, 255, 0,-77.777,+0, 2.529154271144737e+02, 0,_
 3791, 105, 256, 0,-77.777,+0, 2.539077992930840e+02, 0,_
 3792, 105, 257, 0, 4.5,   +1, 2.548988985018705e+02, 0,_
 3793, 105, 257, 1, 0.5,   -1, 2.548990049340549e+02, 0.1,_
 3794, 105, 258, 0,-77.777,+0, 2.558920013492970e+02, 0,_
 3795, 105, 258, 1,-77.777,+0, 2.558920652086076e+02, 0.06,_
 3796, 105, 259, 0,-77.777,+0, 2.568836169486032e+02, 0,_
 3797, 105, 260, 0,-77.777,+0, 2.578768160965934e+02, 0,_
 3798, 105, 261, 0,-77.777,+0, 2.588689027492619e+02, 0,_
 3799, 105, 262, 0,-77.777,+0, 2.598623823323059e+02, 0,_
 3800, 105, 263, 0,-77.777,+0, 2.608547039490306e+02, 0,_
 3801, 105, 264, 0,-77.777,+0, 2.618485097058067e+02, 0,_
 3802, 105, 265, 0,-77.777,+0, 2.628411118914256e+02, 0,_
 3803, 106, 258, 0, 0,     +1, 2.558956695645209e+02, 0,_
 3804, 106, 259, 0, 0.5,   +1, 2.568884363241201e+02, 0,_
 3805, 106, 260, 0, 0,     +1, 2.578798760900081e+02, 0,_
 3806, 106, 261, 0, 3.5,   +1, 2.588728368486221e+02, 0,_
 3807, 106, 262, 0, 0,     +1, 2.598646302998563e+02, 0,_
 3808, 106, 263, 0, 4.5,   +1, 2.608579815489036e+02, 0,_
 3809, 106, 263, 1, 1.5,   +1, 2.608580879810879e+02, 0.1,_
 3810, 106, 264, 0, 0,     +1, 2.618500206139151e+02, 0,_
 3811, 106, 265, 0, 1.5,   +1, 2.628435725698541e+02, 0,_
 3812, 106, 266, 0, 0,     +1, 2.638358575044267e+02, 0,_
 3813, 106, 269, 0,-77.777,+0, 2.668166263929289e+02, 0,_
 3814, 107, 260, 0,-77.777,+0, 2.578870880906298e+02, 0,_
 3815, 107, 261, 0,-77.777,+0, 2.588782953630541e+02, 0,_
 3816, 107, 262, 0,-77.777,+0, 2.598712029028051e+02, 0,_
 3817, 107, 262, 1,-77.777,+0, 2.598715221993580e+02, 0.3,_
 3818, 107, 263, 0,-77.777,+0, 2.608625638437049e+02, 0,_
 3819, 107, 264, 0,-77.777,+0, 2.618556359574362e+02, 0,_
 3820, 107, 265, 0,-77.777,+0, 2.628474261786582e+02, 0,_
 3821, 107, 266, 0,-77.777,+0, 2.638406331240843e+02, 0,_
 3822, 107, 267, 0,-77.777,+0, 2.648327465448098e+02, 0,_
 3823, 107, 269, 0,-77.777,+0, 2.668184525692646e+02, 0,_
 3824, 108, 263, 0, 3.5,   +1, 2.608680800462740e+02, 0,_
 3825, 108, 264, 0, 0,     +1, 2.618593670329807e+02, 0,_
 3826, 108, 265, 0, 4.5,   +1, 2.628521996162236e+02, 0,_
 3827, 108, 265, 1, 1.5,   +1, 2.628526253449608e+02, 0.4,_
 3828, 108, 266, 0, 0,     +1, 2.638438604119070e+02, 0,_
 3829, 108, 267, 0, 1.5,   +1, 2.648368836967366e+02, 0,_
 3830, 108, 268, 0, 0,     +1, 2.658284815845118e+02, 0,_
 3831, 108, 269, 0,-77.777,+0, 2.668217371090043e+02, 0,_
 3832, 108, 273, 0, 1.5,   +1, 2.707951726247442e+02, 0,_
 3833, 109, 265, 0,-77.777,+0, 2.628583495287492e+02, 0,_
 3834, 109, 266, 0,-77.777,+0, 2.638511252110341e+02, 0,_
 3835, 109, 266, 1,-77.777,+0, 2.638524343269010e+02, 1.23,_
 3836, 109, 267, 0,-77.777,+0, 2.648423523116488e+02, 0,_
 3837, 109, 268, 0,-77.777,+0, 2.658352092895142e+02, 0,_
 3838, 109, 269, 0,-77.777,+0, 2.668268517907178e+02, 0,_
 3839, 109, 270, 0,-77.777,+0, 2.678196879489834e+02, 0,_
 3840, 109, 271, 0,-77.777,+0, 2.688115128695385e+02, 0,_
 3841, 109, 273, 0,-77.777,+0, 2.707981765955897e+02, 0,_
 3842, 110, 267, 0, 4.5,   +1, 2.648488331556808e+02, 0,_
 3843, 110, 268, 0, 0,     +1, 2.658399958146767e+02, 0,_
 3844, 110, 269, 0, 1.5,   +1, 2.668326684111910e+02, 0,_
 3845, 110, 270, 0, 0,     +1, 2.678239114427653e+02, 0,_
 3846, 110, 270, 1, 10,    -1, 2.678251247696664e+02, 1.14,_
 3847, 110, 271, 0, 5.5,   -1, 2.688166721646805e+02, 0,_
 3848, 110, 271, 1, 4.5,   +1, 2.688167030300139e+02, 0.029,_
 3849, 110, 272, 0, 0,     +1, 2.698081530661323e+02, 0,_
 3850, 110, 273, 0, 6.5,   -1, 2.708020539982223e+02, 0,_
 3851, 110, 273, 1, 1.5,   +1, 2.708022647339472e+02, 0.198,_
 3852, 111, 272, 0,-77.777,+0, 2.698153457421997e+02, 0,_
 9999
