' Nuclear properties from JEFF-3.1.1

' This version of NucProp allows for an extended table of isotopes.
' Note: In this version of NucProp, the MAT numbers from JEFF are not preserved!

' Modification: List of nuclides extended (12/April/2016) KHS

' Note: The original JEFF numbers are still the numbers in the table, but they 
' are not used any more. This cured the problem that GEFY tables go beyond the
' limits of the JEFF numbers. 

Dim Shared As Integer N_MAT_MAX  ' Number of lines in table NuclideData
#DEFINE ISOSOURCE "JEFF-3.1.1"
' Print "Spins and energies for calculation of isomeric yields provided by "+ISOSOURCE+"."
' Print

' Data until MAT 3852 from JEFF decay file

Declare Function I_MAT_ENDF(IZ As Integer,IA As Integer) As Integer
Declare Function N_ISO_MAT(IMAT As Integer) As Integer
Declare Function R_AWR_ENDF(IZ As Integer,IA AS Integer) As Single
Declare Function ISO_for_MAT(IMAT As Integer) As Integer
Declare Function ISO_for_ZA(IZ As Integer,IA As Integer) As Integer
Declare Function NStates_for_ZA(IZ As Integer, IA As Integer) As Integer


Type NucProp
  I_Z As Integer    ' Atomic number
  I_A As Integer    ' Mass number
  I_ISO As Integer  ' Nmbr of isomeric state, 0 for ground state
  R_SPI As Single   ' Spin
  I_PAR As Integer  ' Parity
  R_AWR As Double   ' Atomic mass of nucleus in units of neutron mass
  R_EXC As Single   ' Excitation energy of isomeric state in MeV
End Type

Dim Shared As Integer N_ISO_TOT  ' Total number of nuclides with isomers
N_ISO_TOT = 0

'Print "NucProp 2 **********"

Dim As Integer I_MAT_original
Dim As Single R_Dummy_NuclideData
Dim As Integer I_NuclideData

' Determine N_MAT_MAX (number of states in table NuclideData)
N_MAT_MAX = 0
Do 
  Read I_MAT_original
  If I_MAT_original = 9999 Then
    Exit Do
  End If
  N_MAT_MAX = N_MAT_MAX + 1
  For I_NuclideData = 1 To 7
    Read R_Dummy_NuclideData
  Next I_NuclideData
Loop 

Redim Shared NucTab(N_MAT_MAX) As NucProp
Redim Shared MAT_for_ISO_prov(N_MAT_MAX) As Integer

Scope
Dim As Integer I_MAT
Restore NuclideData
Do Until I_MAT = N_MAT_MAX 
  ' Read nuclide properties into UDT NucTab
  Read I_MAT_original
  I_MAT = I_MAT + 1
  If I_MAT > N_MAT_MAX Then Print "<E> NucPropx: N_MAT_MAX too small!"
  Read NucTab(I_MAT).I_Z, _
       NucTab(I_MAT).I_A, _
       NucTab(I_MAT).I_ISO, _
       NucTab(I_MAT).R_SPI, _
       NucTab(I_MAT).I_PAR, _
       NucTab(I_MAT).R_AWR, _
       NucTab(I_MAT).R_EXC 
  If NucTab(I_MAT).I_ISO < 0 or NucTab(I_MAT).I_ISO > 9 Then
    Print "<E> Error in NucProp"
    Print "GEF stopped."
    Sleep
  End If     
  If NucTab(I_MAT).R_SPI < 0 Or NucTab(I_MAT).R_SPI > 1.E3 Then _
          NucTab(I_MAT).R_SPI = 0     
  If NucTab(I_MAT).I_ISO = 1 Then  ' At least one isomeric state is in NucTab
    ' Establish table of MAT numbers of nuclides with isomers
    N_ISO_TOT = N_ISO_TOT + 1
    MAT_for_ISO_prov(N_ISO_TOT) = I_MAT - 1
  End If
'  PRINT I_MAT,NucTab(I_MAT).I_Z,NucTab(I_MAT).I_A     
  If NucTab(I_MAT).I_Z = 111 Then Exit Do
Loop

If I_MAT < N_MAT_MAX Then 
  Print "<E> Nucprop: N_MAT_MAX too large, should be ";I_MAT
End If  
End Scope

Redim Shared MAT_for_ISO(N_ISO_TOT) As Integer  
   ' List of MAT numbers of nuclides with isomers
   ' MAT number refers to ground state !
   
'Scope  


   
Dim As Integer I1,J1
Dim As Single R1
For I1 = 1 To N_ISO_TOT
  MAT_for_ISO(I1) = MAT_for_ISO_prov(I1)
Next
' Erase MAT_for_ISO_prov



Type Isoprop
   I_MAT As Integer
   I_Z As Integer
   I_A As Integer
   N_STATES As Integer
   I_ISO(10) As Integer
   R_SPI(10) As Single
   I_PAR(10) As Integer
   R_EXC(10) As Single
   R_Lim(10) As Single
   R_Prob(10) As Single
End Type


Redim Shared Isotab(N_ISO_TOT) As Isoprop



Scope

Dim Inmbr As Integer

For I1 = 1 To N_ISO_TOT
  Isotab(I1).I_MAT = MAT_for_ISO(I1)
  Isotab(I1).I_Z = NucTab(MAT_for_ISO(I1)).I_Z
  Isotab(I1).I_A = NucTab(MAT_for_ISO(I1)).I_A
  Isotab(I1).N_STATES = N_ISO_MAT(Isotab(I1).I_MAT) + 1  ' Number of states
  

  ' Sorting the spin in ascending order
  Inmbr = 0 
  For R1 = 0 To 50.0 Step 0.5
    For J1 = 1 To Isotab(I1).N_STATES
      If NucTab(MAT_for_ISO(I1)+J1-1).R_SPI = R1 Then
        Inmbr = Inmbr + 1
        Isotab(I1).I_ISO(Inmbr) = NucTab(MAT_for_ISO(I1)+J1-1).I_ISO
        Isotab(I1).R_SPI(Inmbr) = NucTab(MAT_for_ISO(I1)+J1-1).R_SPI
        Isotab(I1).I_PAR(Inmbr) = NucTab(MAT_for_ISO(I1)+J1-1).I_PAR
        Isotab(I1).R_EXC(Inmbr) = NucTab(MAT_for_ISO(I1)+J1-1).R_EXC 
  '     Print Inmbr,Mat_for_ISO(I1)+J1-1,NucTab(MAT_for_ISO(I1)+J1-1).R_SPI,NucTab(MAT_for_ISO(I1)+J1-1).R_EXC
      End If
    Next
  Next

  
  Dim As Single R_J_diff,R_E_diff,R_J_1
  ' Setting the limits of the angular-momentum distribution
  For J1 = 1 To Isotab(I1).N_STATES - 1
    R_J_diff = Isotab(I1).R_SPI(J1+1) - Isotab(I1).R_SPI(J1)
    R_E_diff = Isotab(I1).R_EXC(J1+1) - Isotab(I1).R_EXC(J1)
'   R_J_1 = R_J_diff * (0.5 + 0.5 * (R_E_diff / (Abs(R_E_diff) + 0.2 ) ) )
    R_J_1 = R_J_diff * (0.5 + 0.5 * (R_E_diff/R_J_diff / (Abs(R_E_diff/R_J_diff) + 0.05 ) ) )
    Isotab(I1).R_lim(J1) = Isotab(I1).R_SPI(J1) + R_J_1
    
    If Isotab(I1).R_SPI(J1+1) = Isotab(I1).R_SPI(J1) Then
      Isotab(I1).R_lim(J1) = 1.E3
    End If
  Next 
  Isotab(I1).R_lim(Isotab(I1).N_STATES) = 1.E3 
Next  

End Scope



Function I_MAT_ENDF(IZ As Integer,IA As Integer) As Integer
  ' Returns ENDF MAT number of the g.s. of the nucleus, specified by Z and A.
  Dim As Integer I,IMAT
  Static As Integer I_message = 0
  For I = LBound(NucTab) To UBound(NucTab)
    IMAT = I
    If IZ = NucTab(IMAT).I_Z And IA = NucTab(IMAT).I_A Then Exit For
  Next
  If I > UBound(NucTab) Then
    If I_message = 0 Then
      Print "<W> NucProb.bas: Missing MAT number in NucTab for Z,A = ";IZ;",";IA
      Print "    Information on nuclear properties is not available."
      Print "    Please extend table!"
      Print "    (Note that this message is shown only once for the first case"
      Print "     encountered in one GEF session!)"
    End If  
    IMAT = 0
    I_message = 1
  End If
  I_MAT_ENDF = IMAT 
End Function

Function N_ISO_MAT(IMAT As Integer) As Integer
  ' Returns the number of isomers in a nuclide specified by MAT number
  ' (= number of states - 1 !)
  Dim As Integer I,I_first,I_last,IZ,IA
  I_first = IMAT
  IZ = NucTab(I_first).I_Z
  IA = NucTab(I_first).I_A
  I_last = I_first
  For I = I_first + 1 To N_MAT_MAX
    If NucTab(I).I_Z <> IZ Or NucTab(I).I_A <> IA Then Exit For
    I_last = I
  Next
  N_ISO_MAT = I_last - I_first
End Function

Function R_AWR_ENDF(IZ As Integer,IA AS Integer) As Single
  ' Returns ENDF AWR value
  Dim As Integer IMAT
  If IMAT > UBound(NucTab) Then IMAT = UBound(NucTab)
  IMAT = I_MAT_ENDF(IZ,IA)
  R_AWR_ENDF = NucTab(IMAT).R_AWR
End Function

Function ISO_for_MAT(IMAT As Integer) As Integer
  Dim As Integer I
  For I = 1 To N_ISO_TOT
    If IMAT = MAT_for_ISO(I) Then
      ISO_for_MAT = I
      Exit For
    End If
  Next
End Function

Function ISO_for_ZA(IZ As Integer,IA As Integer) As Integer
  ' Returns number of element in ISO array for nucleus Z,A
  Dim As Integer I
  Dim As Integer Ires = 0
  For I = 1 To N_ISO_TOT
    If Isotab(I).I_Z = IZ And Isotab(I).I_A = IA Then
      Ires = I
      Exit For
    End If    
  Next
  ISO_for_ZA = Ires
End Function

Function NStates_for_ZA(IZ As Integer, IA As Integer) As Integer
  ' Returns number of states
  Dim As Integer I
  Dim As Integer Ires = 1
  For I = 1 To N_ISO_TOT
    If Isotab(I).I_Z = IZ And Isotab(I).I_A = IA Then
      Ires = Isotab(I).N_STATES
      Exit For
    End If
  Next
  NStates_for_ZA = Ires
 End Function


NuclideData:
' Data extracted from JEFF 3.1.1
' MAT no., Z, A, Isomer, Spin, Parity, AWR, EXC
' The MAT no. are taken from JEFF 3.1.1. This causes a problem, because
' the GEFY tables cover nuclides beyond this range. 
' Therefore, the table is extended by repeating the last MAT no.
' The MAT no. in the GEFY tables are determined by counting. They differ
' from the original MAT no. of JEFF 3.1.1!

' The following table can be modified just by changing the values for specific states or by
' adding new lines for additional states.
' No further adjustments of GEF are necessary.

Data _
 1,    0,   1,   0, 0.5,   +1, 1.000000e+00, 0,_
 2,    1,   1,   0, 0.5,   +1, 0.999167e+00, 0,_
 3,    1,   2,   0, 1,     +1, 1.996800e+00, 0,_
 4,    1,   3,   0, 0.5,   +1, 2.990140e+00, 0,_
 5,    1,   4,   0, 2,     -1, 3.993200e+00, 0,_
 6,    1,   5,   0, 0.5,   +1, 4.992050e+00, 0,_
 7,    1,   6,   0, 2,     -1, 5.993010e+00, 0,_
 8,    1,   7,   0, 0.5,   +1, 6.992170e+00, 0,_
 9,    2,   3,   0, 0.5,   +1, 2.990120e+00, 0,_
 10,   2,   4,   0, 0,     +1, 3.968220e+00, 0,_
 11,   2,   5,   0, 1.5,   -1, 4.969170e+00, 0,_
 12,   2,   6,   0, 0,     +1, 5.967180e+00, 0,_
 13,   2,   7,   0, 1.5,   -1, 6.967650e+00, 0,_
 14,   2,   8,   0, 0,     +1, 7.964910e+00, 0,_
 15,   2,   9,   0, 0.5,   -1, 8.966260e+00, 0,_
 16,   2,   10,  0, 0,     +1, 9.966040e+00, 0,_
 17,   3,   4,   0, 2,     -1, 3.992590e+00, 0,_
 18,   3,   5,   0, 1.5,   -1, 4.969480e+00, 0,_
 19,   3,   6,   0, 1,     +1, 5.963450e+00, 0,_
 20,   3,   7,   0, 1.5,   -1, 6.955730e+00, 0,_
 21,   3,   8,   0, 2,     +1, 7.953570e+00, 0,_
 22,   3,   9,   0, 1.5,   -1, 8.949250e+00, 0,_
 23,   3,   10,  0,-77.777,+0, 9.949270e+00, 0,_
 24,   3,   11,  0, 1.5,   -1, 1.094890e+01, 0,_
 25,   3,   12,  0,-77.777,+0, 1.195020e+01, 0,_
 26,   4,   5,   0, 0.5,   +1, 4.997490e+00, 0,_
 27,   4,   6,   0, 0,     +1, 5.968010e+00, 0,_
 28,   4,   7,   0, 1.5,   -1, 6.956650e+00, 0,_
 29,   4,   8,   0, 0,     +1, 7.936540e+00, 0,_
 30,   4,   9,   0, 1.5,   -1, 8.934760e+00, 0,_
 31,   4,   10,  0, 0,     +1, 9.927510e+00, 0,_
 32,   4,   11,  0, 0.5,   +1, 1.092700e+01, 0,_
 33,   4,   12,  0, 0,     +1, 1.192360e+01, 0,_
 34,   4,   13,  0, 0.5,   +1, 1.292370e+01, 0,_
 35,   4,   14,  0, 0,     +1, 1.392230e+01, 0,_
 36,   4,   15,  0,-77.777,+0, 1.492410e+01, 0,_
 37,   4,   16,  0, 0,     +1, 1.592390e+01, 0,_
 38,   5,   6,   0, 2,     -1, 5.994860e+00, 0,_
 39,   5,   7,   0, 1.5,   -1, 6.969530e+00, 0,_
 40,   5,   8,   0, 2,     +1, 7.955670e+00, 0,_
 41,   5,   9,   0, 1.5,   -1, 8.935900e+00, 0,_
 42,   5,   10,  0, 3,     +1, 9.926920e+00, 0,_
 43,   5,   11,  0, 1.5,   -1, 1.091470e+01, 0,_
 44,   5,   12,  0, 1,     +1, 1.191110e+01, 0,_
 45,   5,   13,  0, 1.5,   -1, 1.290600e+01, 0,_
 46,   5,   14,  0, 2,     -1, 1.390490e+01, 0,_
 47,   5,   15,  0, 1.5,   -1, 1.490200e+01, 0,_
 48,   5,   16,  0, 0,     -1, 1.590200e+01, 0,_
 49,   5,   17,  0, 1.5,   -1, 1.690050e+01, 0,_
 50,   5,   18,  0, 4,     -1, 1.790110e+01, 0,_
 51,   5,   19,  0, 1.5,   -1, 1.890000e+01, 0,_
 52,   6,   8,   0, 0,     +1, 7.968630e+00, 0,_
 53,   6,   9,   0, 1.5,   -1, 8.953460e+00, 0,_
 54,   6,   10,  0, 0,     +1, 9.930800e+00, 0,_
 55,   6,   11,  0, 1.5,   -1, 1.091680e+01, 0,_
 56,   6,   12,  0, 0,     +1, 1.189690e+01, 0,_
 57,   6,   13,  0, 0.5,   -1, 1.289160e+01, 0,_
 58,   6,   14,  0, 0,     +1, 1.388290e+01, 0,_
 59,   6,   15,  0, 0.5,   +1, 1.488170e+01, 0,_
 60,   6,   16,  0, 0,     +1, 1.587710e+01, 0,_
 61,   6,   17,  0, 1.5,   +1, 1.687640e+01, 0,_
 62,   6,   18,  0, 0,     +1, 1.787190e+01, 0,_
 63,   6,   19,  0, 0.5,   +1, 1.887130e+01, 0,_
 64,   6,   20,  0, 0,     +1, 1.986820e+01, 0,_
 65,   6,   21,  0, 0.5,   +1, 2.086850e+01, 0,_
 66,   6,   22,  0, 0,     +1, 2.186770e+01, 0,_
 67,   7,   10,  0, 2,     -1, 9.955390e+00, 0,_
 68,   7,   11,  0, 0.5,   +1, 1.093140e+01, 0,_
 69,   7,   12,  0, 1,     +1, 1.191540e+01, 0,_
 70,   7,   13,  0, 0.5,   -1, 1.289400e+01, 0,_
 71,   7,   14,  0, 1,     +1, 1.388280e+01, 0,_
 72,   7,   15,  0, 0.5,   -1, 1.487130e+01, 0,_
 73,   7,   16,  0, 2,     -1, 1.586860e+01, 0,_
 74,   7,   17,  0, 0.5,   -1, 1.686230e+01, 0,_
 75,   7,   18,  0, 1,     -1, 1.785930e+01, 0,_
 76,   7,   19,  0, 0.5,   -1, 1.885370e+01, 0,_
 77,   7,   20,  0,-77.777,+0, 1.985140e+01, 0,_
 78,   7,   21,  0, 0.5,   -1, 2.084650e+01, 0,_
 79,   7,   22,  0,-77.777,+0, 2.184510e+01, 0,_
 80,   7,   23,  0, 0.5,   -1, 2.284330e+01, 0,_
 81,   7,   24,  0,-77.777,+0, 2.384440e+01, 0,_
 82,   7,   25,  0, 0.5,   -1, 2.484540e+01, 0,_
 83,   8,   12,  0, 0,     +1, 1.193100e+01, 0,_
 84,   8,   13,  0, 1.5,   -1, 1.291290e+01, 0,_
 85,   8,   14,  0, 0,     +1, 1.388830e+01, 0,_
 86,   8,   15,  0, 0.5,   -1, 1.487420e+01, 0,_
 87,   8,   16,  0, 0,     +1, 1.585750e+01, 0,_
 88,   8,   17,  0, 2.5,   +1, 1.685310e+01, 0,_
 89,   8,   18,  0, 0,     +1, 1.784450e+01, 0,_
 90,   8,   19,  0, 2.5,   +1, 1.884030e+01, 0,_
 91,   8,   20,  0, 0,     +1, 1.983220e+01, 0,_
 92,   8,   21,  0, 2.5,   +1, 2.082820e+01, 0,_
 93,   8,   22,  0, 0,     +1, 2.182090e+01, 0,_
 94,   8,   23,  0, 0.5,   +1, 2.281800e+01, 0,_
 95,   8,   24,  0, 0,     +1, 2.381410e+01, 0,_
 96,   8,   25,  0, 1.5,   +1, 2.481440e+01, 0,_
 97,   8,   26,  0, 0,     +1, 2.581470e+01, 0,_
 98,   8,   27,  0, 1.5,   +1, 2.681590e+01, 0,_
 99,   8,   28,  0, 0,     +1, 2.781680e+01, 0,_
 100,  9,   14,  0, 2,     -1, 1.391450e+01, 0,_
 101,  9,   15,  0, 0.5,   +1, 1.488900e+01, 0,_
 102,  9,   16,  0, 0,     -1, 1.587390e+01, 0,_
 103,  9,   17,  0, 2.5,   +1, 1.685600e+01, 0,_
 104,  9,   18,  0, 1,     +1, 1.784630e+01, 0,_
 105,  9,   19,  0, 0.5,   +1, 1.883520e+01, 0,_
 106,  9,   20,  0, 2,     +1, 1.982820e+01, 0,_
 107,  9,   21,  0, 2.5,   +1, 2.081960e+01, 0,_
 108,  9,   22,  0,-77.777,+0, 2.181400e+01, 0,_
 109,  9,   23,  0,-77.777,+0, 2.280600e+01, 0,_
 110,  9,   24,  0,-77.777,+0, 2.380190e+01, 0,_
 111,  9,   25,  0, 2.5,   +1, 2.479720e+01, 0,_
 112,  9,   26,  0, 1,     +1, 2.579610e+01, 0,_
 113,  9,   27,  0, 2.5,   +1, 2.679460e+01, 0,_
 114,  9,   28,  0,-77.777,+0, 2.779480e+01, 0,_
 115,  9,   29,  0, 2.5,   +1, 2.879380e+01, 0,_
 116,  9,   30,  0,-77.777,+0, 2.979430e+01, 0,_
 117,  9,   31,  0, 2.5,   +1, 3.079360e+01, 0,_
 118,  10,  16,  0, 0,     +1, 1.588810e+01, 0,_
 119,  10,  17,  0, 0.5,   -1, 1.687150e+01, 0,_
 120,  10,  18,  0, 0,     +1, 1.785100e+01, 0,_
 121,  10,  19,  0, 0.5,   +1, 1.883860e+01, 0,_
 122,  10,  20,  0, 0,     +1, 1.982070e+01, 0,_
 123,  10,  21,  0, 1.5,   +1, 2.081350e+01, 0,_
 124,  10,  22,  0, 0,     +1, 2.180250e+01, 0,_
 125,  10,  23,  0, 2.5,   +1, 2.279690e+01, 0,_
 126,  10,  24,  0, 0,     +1, 2.378750e+01, 0,_
 127,  10,  25,  0, 1.5,   +1, 2.478300e+01, 0,_
 128,  10,  26,  0, 0,     +1, 2.577710e+01, 0,_
 129,  10,  27,  0, 1.5,   +1, 2.677560e+01, 0,_
 130,  10,  28,  0, 0,     +1, 2.777140e+01, 0,_
 131,  10,  29,  0, 1.5,   +1, 2.877010e+01, 0,_
 132,  10,  30,  0, 0,     +1, 2.976690e+01, 0,_
 133,  10,  31,  0, 3.5,   -1, 3.076650e+01, 0,_
 134,  10,  32,  0, 0,     +1, 3.176480e+01, 0,_
 135,  10,  33,  0, 3.5,   -1, 3.276550e+01, 0,_
 136,  10,  34,  0, 0,     +1, 3.376450e+01, 0,_
 137,  11,  18,  0, 1,     -1, 1.787110e+01, 0,_
 138,  11,  19,  0, 2.5,   +1, 1.885050e+01, 0,_
 139,  11,  20,  0, 2,     +1, 1.983550e+01, 0,_
 140,  11,  21,  0, 1.5,   +1, 2.081730e+01, 0,_
 141,  11,  22,  0, 3,     +1, 2.180550e+01, 0,_
 142,  11,  23,  0, 1.5,   +1, 2.279230e+01, 0,_
 143,  11,  24,  0, 4,     +1, 2.378490e+01, 0,_
 144,  11,  24,  1, 1,     +1, 2.378490e+01, 0.4723,_
 145,  11,  25,  0, 2.5,   +1, 2.477530e+01, 0,_
 146,  11,  26,  0, 3,     +1, 2.576930e+01, 0,_
 147,  11,  27,  0, 2.5,   +1, 2.676220e+01, 0,_
 148,  11,  28,  0, 1,     +1, 2.775840e+01, 0,_
 149,  11,  29,  0, 1.5,   +1, 2.875370e+01, 0,_
 150,  11,  30,  0, 2,     +1, 2.975120e+01, 0,_
 151,  11,  31,  0, 1.5,   +1, 3.074720e+01, 0,_
 152,  11,  32,  0,-77.777,+0, 3.174540e+01, 0,_
 153,  11,  33,  0, 1.5,   +1, 3.274300e+01, 0,_
 154,  11,  34,  0, 1,     +1, 3.374280e+01, 0,_
 155,  11,  35,  0, 1.5,   +1, 3.474150e+01, 0,_
 156,  11,  36,  0,-77.777,+0, 3.574180e+01, 0,_
 157,  11,  37,  0, 1.5,   +1, 3.674100e+01, 0,_
 158,  12,  20,  0, 0,     +1, 1.984690e+01, 0,_
 159,  12,  21,  0, 2.5,   +1, 2.083120e+01, 0,_
 160,  12,  22,  0, 0,     +1, 2.181060e+01, 0,_
 161,  12,  23,  0, 1.5,   +1, 2.279660e+01, 0,_
 162,  12,  24,  0, 0,     +1, 2.377900e+01, 0,_
 163,  12,  25,  0, 2.5,   +1, 2.477120e+01, 0,_
 164,  12,  26,  0, 0,     +1, 2.575940e+01, 0,_
 165,  12,  27,  0, 0.5,   +1, 2.675250e+01, 0,_
 166,  12,  28,  0, 0,     +1, 2.774350e+01, 0,_
 167,  12,  29,  0, 1.5,   +1, 2.873960e+01, 0,_
 168,  12,  30,  0, 0,     +1, 2.973280e+01, 0,_
 169,  12,  31,  0, 1.5,   +1, 3.073030e+01, 0,_
 170,  12,  32,  0, 0,     +1, 3.172410e+01, 0,_
 171,  12,  33,  0, 3.5,   -1, 3.272170e+01, 0,_
 172,  12,  34,  0, 0,     +1, 3.371730e+01, 0,_
 173,  12,  35,  0, 3.5,   -1, 3.471650e+01, 0,_
 174,  12,  36,  0, 0,     +1, 3.571350e+01, 0,_
 175,  12,  37,  0, 3.5,   -1, 3.671330e+01, 0,_
 176,  12,  38,  0, 0,     +1, 3.771080e+01, 0,_
 177,  12,  39,  0, 3.5,   -1, 3.871130e+01, 0,_
 178,  12,  40,  0, 0,     +1, 3.970990e+01, 0,_
 179,  13,  21,  0, 0.5,   +1, 2.084740e+01, 0,_
 180,  13,  22,  0, 3,     +1, 2.183040e+01, 0,_
 181,  13,  23,  0, 2.5,   +1, 2.280960e+01, 0,_
 182,  13,  24,  0, 4,     +1, 2.379380e+01, 0,_
 183,  13,  24,  1, 1,     +1, 2.379380e+01, 0.4258,_
 184,  13,  25,  0, 2.5,   +1, 2.477570e+01, 0,_
 185,  13,  26,  0, 5,     +1, 2.576360e+01, 0,_
 186,  13,  26,  1, 0,     +1, 2.576370e+01, 0.22844,_
 187,  13,  27,  0, 2.5,   +1, 2.674980e+01, 0,_
 188,  13,  28,  0, 3,     +1, 2.774150e+01, 0,_
 189,  13,  29,  0, 2.5,   +1, 2.873150e+01, 0,_
 190,  13,  30,  0, 3,     +1, 2.972540e+01, 0,_
 191,  13,  31,  0,-77.777,+0, 3.071780e+01, 0,_
 192,  13,  32,  0, 1,     +1, 3.171330e+01, 0,_
 193,  13,  33,  0, 2.5,   +1, 3.270740e+01, 0,_
 194,  13,  34,  0, 4,     -1, 3.370480e+01, 0,_
 195,  13,  35,  0, 2.5,   +1, 3.469920e+01, 0,_
 196,  13,  36,  0,-77.777,+0, 3.569690e+01, 0,_
 197,  13,  37,  0, 1.5,   +1, 3.669270e+01, 0,_
 198,  13,  38,  0,-77.777,+0, 3.769060e+01, 0,_
 199,  13,  39,  0, 1.5,   +1, 3.868770e+01, 0,_
 200,  13,  40,  0,-77.777,+0, 3.968760e+01, 0,_
 201,  13,  41,  0, 1.5,   +1, 4.068580e+01, 0,_
 202,  13,  42,  0,-77.777,+0, 4.168570e+01, 0,_
 203,  14,  22,  0, 0,     +1, 2.184520e+01, 0,_
 204,  14,  23,  0, 1.5,   +1, 2.282770e+01, 0,_
 205,  14,  24,  0, 0,     +1, 2.380530e+01, 0,_
 206,  14,  25,  0, 2.5,   +1, 2.478930e+01, 0,_
 207,  14,  26,  0, 0,     +1, 2.576900e+01, 0,_
 208,  14,  27,  0, 2.5,   +1, 2.675490e+01, 0,_
 209,  14,  28,  0, 0,     +1, 2.773660e+01, 0,_
 210,  14,  29,  0, 0.5,   +1, 2.872760e+01, 0,_
 211,  14,  30,  0, 0,     +1, 2.971630e+01, 0,_
 212,  14,  31,  0, 1.5,   +1, 3.070930e+01, 0,_
 213,  14,  32,  0, 0,     +1, 3.169950e+01, 0,_
 214,  14,  33,  0, 1.5,   +1, 3.269470e+01, 0,_
 215,  14,  34,  0, 0,     +1, 3.368670e+01, 0,_
 216,  14,  35,  0, 3.5,   -1, 3.468400e+01, 0,_
 217,  14,  36,  0, 0,     +1, 3.567750e+01, 0,_
 218,  14,  37,  0, 3.5,   -1, 3.667510e+01, 0,_
 219,  14,  38,  0, 0,     +1, 3.766920e+01, 0,_
 220,  14,  39,  0, 3.5,   -1, 3.866700e+01, 0,_
 221,  14,  40,  0, 0,     +1, 3.966220e+01, 0,_
 222,  14,  41,  0, 3.5,   -1, 4.066220e+01, 0,_
 223,  14,  42,  0, 0,     +1, 4.165880e+01, 0,_
 224,  14,  43,  0, 1.5,   -1, 4.265900e+01, 0,_
 225,  14,  44,  0, 0,     +1, 4.365700e+01, 0,_
 226,  15,  24,  0, 1,     +1, 2.382790e+01, 0,_
 227,  15,  25,  0, 0.5,   +1, 2.480530e+01, 0,_
 228,  15,  26,  0, 3,     +1, 2.578830e+01, 0,_
 229,  15,  27,  0, 0.5,   +1, 2.676730e+01, 0,_
 230,  15,  28,  0, 3,     +1, 2.775180e+01, 0,_
 231,  15,  29,  0, 0.5,   +1, 2.873280e+01, 0,_
 232,  15,  30,  0, 1,     +1, 2.972080e+01, 0,_
 233,  15,  31,  0, 0.5,   +1, 3.070770e+01, 0,_
 234,  15,  32,  0, 1,     +1, 3.169920e+01, 0,_
 235,  15,  33,  0, 0.5,   +1, 3.268850e+01, 0,_
 236,  15,  34,  0, 1,     +1, 3.368180e+01, 0,_
 237,  15,  35,  0, 0.5,   +1, 3.467290e+01, 0,_
 238,  15,  36,  0, 4,     -1, 3.566920e+01, 0,_
 239,  15,  37,  0, 0.5,   +1, 3.666190e+01, 0,_
 240,  15,  38,  0,-77.777,+0, 3.765780e+01, 0,_
 241,  15,  39,  0, 0.5,   +1, 3.865130e+01, 0,_
 242,  15,  40,  0,-77.777,+0, 3.964770e+01, 0,_
 243,  15,  41,  0, 0.5,   +1, 4.064220e+01, 0,_
 244,  15,  42,  0,-77.777,+0, 4.164020e+01, 0,_
 245,  15,  43,  0, 0.5,   +1, 4.263680e+01, 0,_
 246,  15,  44,  0,-77.777,+0, 4.363490e+01, 0,_
 247,  15,  45,  0, 0.5,   +1, 4.463250e+01, 0,_
 248,  15,  46,  0,-77.777,+0, 4.563200e+01, 0,_
 249,  16,  26,  0, 0,     +1, 2.580430e+01, 0,_
 250,  16,  27,  0, 2.5,   +1, 2.678670e+01, 0,_
 251,  16,  28,  0, 0,     +1, 2.776380e+01, 0,_
 252,  16,  29,  0, 2.5,   +1, 2.874750e+01, 0,_
 253,  16,  30,  0, 0,     +1, 2.972730e+01, 0,_
 254,  16,  31,  0, 0.5,   +1, 3.071340e+01, 0,_
 255,  16,  32,  0, 0,     +1, 3.169740e+01, 0,_
 256,  16,  33,  0, 1.5,   +1, 3.268820e+01, 0,_
 257,  16,  34,  0, 0,     +1, 3.367610e+01, 0,_
 258,  16,  35,  0, 1.5,   +1, 3.466860e+01, 0,_
 259,  16,  36,  0, 0,     +1, 3.565810e+01, 0,_
 260,  16,  37,  0, 3.5,   -1, 3.665350e+01, 0,_
 261,  16,  38,  0, 0,     +1, 3.764500e+01, 0,_
 262,  16,  39,  0,-77.777,+0, 3.864030e+01, 0,_
 263,  16,  40,  0, 0,     +1, 3.963200e+01, 0,_
 264,  16,  41,  0, 3.5,   -1, 4.062750e+01, 0,_
 265,  16,  42,  0, 0,     +1, 4.162040e+01, 0,_
 266,  16,  43,  0, 1.5,   -1, 4.261790e+01, 0,_
 267,  16,  44,  0, 0,     +1, 4.361230e+01, 0,_
 268,  16,  45,  0, 1.5,   -1, 4.461000e+01, 0,_
 269,  16,  46,  0, 0,     +1, 4.560560e+01, 0,_
 270,  16,  47,  0, 1.5,   -1, 4.660480e+01, 0,_
 271,  16,  48,  0, 0,     +1, 4.760170e+01, 0,_
 272,  16,  49,  0, 1.5,   -1, 4.860250e+01, 0,_
 273,  17,  28,  0, 1,     +1, 2.778770e+01, 0,_
 274,  17,  29,  0, 1.5,   +1, 2.876490e+01, 0,_
 275,  17,  30,  0, 3,     +1, 2.974700e+01, 0,_
 276,  17,  31,  0, 1.5,   +1, 3.072620e+01, 0,_
 277,  17,  32,  0, 1,     +1, 3.171090e+01, 0,_
 278,  17,  33,  0, 1.5,   +1, 3.269420e+01, 0,_
 279,  17,  34,  0, 0,     +1, 3.368190e+01, 0,_
 280,  17,  34,  1, 3,     +1, 3.368190e+01, 0.14636,_
 281,  17,  35,  0, 1.5,   +1, 3.466850e+01, 0,_
 282,  17,  36,  0, 2,     +1, 3.565930e+01, 0,_
 283,  17,  37,  0, 1.5,   +1, 3.664830e+01, 0,_
 284,  17,  38,  0, 2,     -1, 3.764180e+01, 0,_
 285,  17,  38,  1, 5,     -1, 3.764180e+01, 0.6713,_
 286,  17,  39,  0, 1.5,   +1, 3.863330e+01, 0,_
 287,  17,  40,  0, 2,     -1, 3.962700e+01, 0,_
 288,  17,  41,  0,-77.777,+0, 4.061870e+01, 0,_
 289,  17,  42,  0,-77.777,+0, 4.161270e+01, 0,_
 290,  17,  43,  0, 1.5,   +1, 4.260490e+01, 0,_
 291,  17,  44,  0,-77.777,+0, 4.360050e+01, 0,_
 292,  17,  45,  0, 1.5,   +1, 4.459390e+01, 0,_
 293,  17,  46,  0,-77.777,+0, 4.558920e+01, 0,_
 294,  17,  47,  0, 1.5,   +1, 4.658510e+01, 0,_
 295,  17,  48,  0,-77.777,+0, 4.758270e+01, 0,_
 296,  17,  49,  0, 1.5,   +1, 4.857940e+01, 0,_
 297,  17,  50,  0,-77.777,+0, 4.957820e+01, 0,_
 298,  17,  51,  0, 1.5,   +1, 5.057630e+01, 0,_
 299,  18,  30,  0, 0,     +1, 2.976370e+01, 0,_
 300,  18,  31,  0, 2.5,   +1, 3.074570e+01, 0,_
 301,  18,  32,  0, 0,     +1, 3.172280e+01, 0,_
 302,  18,  33,  0, 0.5,   +1, 3.270650e+01, 0,_
 303,  18,  34,  0, 0,     +1, 3.368840e+01, 0,_
 304,  18,  35,  0, 1.5,   +1, 3.467480e+01, 0,_
 305,  18,  36,  0, 0,     +1, 3.565860e+01, 0,_
 306,  18,  37,  0, 1.5,   +1, 3.664920e+01, 0,_
 307,  18,  38,  0, 0,     +1, 3.763660e+01, 0,_
 308,  18,  39,  0, 3.5,   -1, 3.862960e+01, 0,_
 309,  18,  40,  0, 0,     +1, 3.961910e+01, 0,_
 310,  18,  41,  0, 3.5,   -1, 4.061260e+01, 0,_
 311,  18,  42,  0, 0,     +1, 4.160260e+01, 0,_
 312,  18,  43,  0, 2.5,   -1, 4.259650e+01, 0,_
 313,  18,  44,  0, 0,     +1, 4.358720e+01, 0,_
 314,  18,  45,  0,-77.777,+0, 4.458170e+01, 0,_
 315,  18,  46,  0, 0,     +1, 4.557320e+01, 0,_
 316,  18,  47,  0, 1.5,   -1, 4.656870e+01, 0,_
 317,  18,  48,  0, 0,     +1, 4.756240e+01, 0,_
 318,  18,  49,  0, 1.5,   -1, 4.855970e+01, 0,_
 319,  18,  50,  0, 0,     +1, 4.955500e+01, 0,_
 320,  18,  51,  0, 1.5,   -1, 5.055360e+01, 0,_
 321,  18,  52,  0, 0,     +1, 5.155010e+01, 0,_
 322,  18,  53,  0, 2.5,   -1, 5.254960e+01, 0,_
 323,  19,  32,  0, 1,     +1, 3.174680e+01, 0,_
 324,  19,  33,  0, 1.5,   +1, 3.272370e+01, 0,_
 325,  19,  34,  0, 1,     +1, 3.370630e+01, 0,_
 326,  19,  35,  0, 1.5,   +1, 3.468740e+01, 0,_
 327,  19,  36,  0, 2,     +1, 3.567220e+01, 0,_
 328,  19,  37,  0, 1.5,   +1, 3.665580e+01, 0,_
 329,  19,  38,  0, 3,     +1, 3.764290e+01, 0,_
 330,  19,  38,  1, 0,     +1, 3.764290e+01, 0.1304,_
 331,  19,  39,  0, 1.5,   +1, 3.862900e+01, 0,_
 332,  19,  40,  0, 4,     -1, 3.962070e+01, 0,_
 333,  19,  41,  0, 1.5,   +1, 4.060990e+01, 0,_
 334,  19,  42,  0, 2,     -1, 4.160190e+01, 0,_
 335,  19,  43,  0, 1.5,   +1, 4.259170e+01, 0,_
 336,  19,  44,  0, 2,     -1, 4.358390e+01, 0,_
 337,  19,  45,  0, 1.5,   +1, 4.457450e+01, 0,_
 338,  19,  46,  0, 2,     -1, 4.556710e+01, 0,_
 339,  19,  47,  0, 0.5,   +1, 4.655830e+01, 0,_
 340,  19,  48,  0, 2,     -1, 4.755350e+01, 0,_
 341,  19,  49,  0, 1.5,   +1, 4.854680e+01, 0,_
 342,  19,  50,  0,-77.777,+0, 4.954350e+01, 0,_
 343,  19,  51,  0, 1.5,   +1, 5.053850e+01, 0,_
 344,  19,  52,  0, 2,     -1, 5.153610e+01, 0,_
 345,  19,  53,  0, 1.5,   +1, 5.253190e+01, 0,_
 346,  19,  54,  0, 2,     -1, 5.353040e+01, 0,_
 347,  19,  55,  0, 1.5,   +1, 5.452720e+01, 0,_
 348,  20,  34,  0, 0,     +1, 3.372190e+01, 0,_
 349,  20,  35,  0, 0.5,   +1, 3.470420e+01, 0,_
 350,  20,  36,  0, 0,     +1, 3.568390e+01, 0,_
 351,  20,  37,  0, 1.5,   +1, 3.666810e+01, 0,_
 352,  20,  38,  0, 0,     +1, 3.765010e+01, 0,_
 353,  20,  39,  0, 1.5,   +1, 3.863590e+01, 0,_
 354,  20,  40,  0, 0,     +1, 3.961930e+01, 0,_
 355,  20,  41,  0, 3.5,   -1, 4.061040e+01, 0,_
 356,  20,  42,  0, 0,     +1, 4.159820e+01, 0,_
 357,  20,  43,  0, 3.5,   -1, 4.258970e+01, 0,_
 358,  20,  44,  0, 0,     +1, 4.357790e+01, 0,_
 359,  20,  45,  0, 3.5,   -1, 4.457000e+01, 0,_
 360,  20,  46,  0, 0,     +1, 4.555890e+01, 0,_
 361,  20,  47,  0, 3.5,   -1, 4.655120e+01, 0,_
 362,  20,  48,  0, 0,     +1, 4.754060e+01, 0,_
 363,  20,  49,  0, 1.5,   -1, 4.853510e+01, 0,_
 364,  20,  50,  0, 0,     +1, 4.952840e+01, 0,_
 365,  20,  51,  0, 1.5,   -1, 5.052370e+01, 0,_
 366,  20,  52,  0, 0,     +1, 5.151870e+01, 0,_
 367,  20,  53,  0, 1.5,   -1, 5.251500e+01, 0,_
 368,  20,  54,  0, 0,     +1, 5.351070e+01, 0,_
 369,  20,  55,  0, 2.5,   -1, 5.450820e+01, 0,_
 370,  20,  56,  0, 0,     +1, 5.550460e+01, 0,_
 371,  20,  57,  0, 2.5,   -1, 5.650280e+01, 0,_
 372,  21,  36,  0,-77.777,+0, 3.570550e+01, 0,_
 373,  21,  37,  0, 3.5,   -1, 3.668520e+01, 0,_
 374,  21,  38,  0, 2,     -1, 3.766830e+01, 0,_
 375,  21,  39,  0, 3.5,   -1, 3.864990e+01, 0,_
 376,  21,  40,  0, 4,     -1, 3.963450e+01, 0,_
 377,  21,  41,  0, 3.5,   -1, 4.061730e+01, 0,_
 378,  21,  42,  0, 0,     +1, 4.160500e+01, 0,_
 379,  21,  42,  1, 7,     +1, 4.160500e+01, 0.61628,_
 380,  21,  43,  0, 3.5,   -1, 4.259210e+01, 0,_
 381,  21,  44,  0, 2,     +1, 4.358180e+01, 0,_
 382,  21,  44,  1, 6,     +1, 4.358180e+01, 0.2711,_
 383,  21,  45,  0, 3.5,   -1, 4.456970e+01, 0,_
 384,  21,  45,  1, 1.5,   +1, 4.456970e+01, 0.0124,_
 385,  21,  46,  0, 4,     +1, 4.556040e+01, 0,_
 386,  21,  46,  1, 1,     -1, 4.556040e+01, 0.14253,_
 387,  21,  47,  0, 3.5,   -1, 4.654910e+01, 0,_
 388,  21,  48,  0, 6,     +1, 4.754030e+01, 0,_
 389,  21,  49,  0, 3.5,   -1, 4.852950e+01, 0,_
 390,  21,  50,  0, 5,     +1, 4.952310e+01, 0,_
 391,  21,  50,  1, 2,     +1, 4.952310e+01, 0.256895,_
 392,  21,  51,  0, 3.5,   -1, 5.051590e+01, 0,_
 393,  21,  52,  0, 3,     +1, 5.151030e+01, 0,_
 394,  21,  53,  0, 3.5,   -1, 5.250470e+01, 0,_
 395,  21,  54,  0, 3,     +1, 5.349970e+01, 0,_
 396,  21,  55,  0, 3.5,   -1, 5.449600e+01, 0,_
 397,  21,  56,  0, 3,     +1, 5.549200e+01, 0,_
 398,  21,  57,  0, 3.5,   -1, 5.648830e+01, 0,_
 399,  21,  58,  0, 3,     +1, 5.748560e+01, 0,_
 400,  21,  59,  0, 3.5,   -1, 5.848250e+01, 0,_
 401,  21,  60,  0, 3,     +1, 5.948030e+01, 0,_
 402,  22,  38,  0, 0,     +1, 3.768320e+01, 0,_
 403,  22,  39,  0, 1.5,   +1, 3.866660e+01, 0,_
 404,  22,  40,  0, 0,     +1, 3.964700e+01, 0,_
 405,  22,  41,  0, 1.5,   +1, 4.063110e+01, 0,_
 406,  22,  42,  0, 0,     +1, 4.161250e+01, 0,_
 407,  22,  43,  0, 3.5,   -1, 4.259940e+01, 0,_
 408,  22,  44,  0, 0,     +1, 4.358210e+01, 0,_
 409,  22,  45,  0, 3.5,   -1, 4.457190e+01, 0,_
 410,  22,  46,  0, 0,     +1, 4.555790e+01, 0,_
 411,  22,  47,  0, 2.5,   -1, 4.654840e+01, 0,_
 412,  22,  48,  0, 0,     +1, 4.753610e+01, 0,_
 413,  22,  49,  0, 3.5,   -1, 4.852740e+01, 0,_
 414,  22,  50,  0, 0,     +1, 4.951570e+01, 0,_
 415,  22,  51,  0, 1.5,   -1, 5.050900e+01, 0,_
 416,  22,  52,  0, 0,     +1, 5.150060e+01, 0,_
 417,  22,  53,  0, 1.5,   -1, 5.249490e+01, 0,_
 418,  22,  54,  0, 0,     +1, 5.348760e+01, 0,_
 419,  22,  55,  0, 1.5,   -1, 5.448320e+01, 0,_
 420,  22,  56,  0, 0,     +1, 5.547750e+01, 0,_
 421,  22,  57,  0, 2.5,   -1, 5.647460e+01, 0,_
 422,  22,  58,  0, 0,     +1, 5.746900e+01, 0,_
 423,  22,  59,  0, 2.5,   -1, 5.846630e+01, 0,_
 424,  22,  60,  0, 0,     +1, 5.946150e+01, 0,_
 425,  22,  61,  0, 0.5,   -1, 6.045930e+01, 0,_
 426,  22,  62,  0, 0,     +1, 6.145500e+01, 0,_
 427,  22,  63,  0, 0.5,   -1, 6.245330e+01, 0,_
 428,  23,  40,  0, 2,     -1, 3.966740e+01, 0,_
 429,  23,  41,  0, 3.5,   -1, 4.064760e+01, 0,_
 430,  23,  42,  0, 2,     -1, 4.163050e+01, 0,_
 431,  23,  43,  0, 3.5,   -1, 4.261140e+01, 0,_
 432,  23,  44,  0, 2,     +1, 4.359630e+01, 0,_
 433,  23,  44,  1, 6,     +1, 4.359630e+01, 0.27,_
 434,  23,  45,  0, 3.5,   -1, 4.457950e+01, 0,_
 435,  23,  46,  0, 0,     +1, 4.556540e+01, 0,_
 436,  23,  46,  1, 3,     +1, 4.556540e+01, 0.8011,_
 437,  23,  47,  0, 1.5,   -1, 4.655150e+01, 0,_
 438,  23,  48,  0, 4,     +1, 4.754030e+01, 0,_
 439,  23,  49,  0, 3.5,   -1, 4.852800e+01, 0,_
 440,  23,  50,  0, 6,     +1, 4.951810e+01, 0,_
 441,  23,  51,  0, 3.5,   -1, 5.050630e+01, 0,_
 442,  23,  52,  0, 3,     +1, 5.149850e+01, 0,_
 443,  23,  53,  0, 3.5,   -1, 5.248950e+01, 0,_
 444,  23,  54,  0, 3,     +1, 5.348300e+01, 0,_
 445,  23,  55,  0, 3.5,   -1, 5.447520e+01, 0,_
 446,  23,  56,  0, 1,     +1, 5.546990e+01, 0,_
 447,  23,  57,  0, 1.5,   -1, 5.646330e+01, 0,_
 448,  23,  58,  0, 3,     +1, 5.745900e+01, 0,_
 449,  23,  59,  0, 3.5,   -1, 5.845370e+01, 0,_
 450,  23,  60,  0, 3,     +1, 5.944990e+01, 0,_
 451,  23,  61,  0, 3.5,   -1, 6.044470e+01, 0,_
 452,  23,  62,  0, 3,     +1, 6.144140e+01, 0,_
 453,  23,  63,  0, 3.5,   -1, 6.243650e+01, 0,_
 454,  23,  64,  0,-77.777,+0, 6.343380e+01, 0,_
 455,  23,  65,  0, 2.5,   -1, 6.442960e+01, 0,_
 456,  24,  42,  0, 0,     +1, 4.164560e+01, 0,_
 457,  24,  43,  0, 1.5,   +1, 4.262830e+01, 0,_
 458,  24,  44,  0, 0,     +1, 4.360770e+01, 0,_
 459,  24,  45,  0, 3.5,   -1, 4.459320e+01, 0,_
 460,  24,  45,  1, 1.5,   +1, 4.459320e+01, 0.05,_
 461,  24,  46,  0, 0,     +1, 4.557350e+01, 0,_
 462,  24,  47,  0, 1.5,   -1, 4.655950e+01, 0,_
 463,  24,  48,  0, 0,     +1, 4.754210e+01, 0,_
 464,  24,  49,  0, 2.5,   -1, 4.853080e+01, 0,_
 465,  24,  50,  0, 0,     +1, 4.951700e+01, 0,_
 466,  24,  51,  0, 3.5,   -1, 5.050710e+01, 0,_
 467,  24,  52,  0, 0,     +1, 5.149430e+01, 0,_
 468,  24,  53,  0, 1.5,   -1, 5.248590e+01, 0,_
 469,  24,  54,  0, 0,     +1, 5.347550e+01, 0,_
 470,  24,  55,  0, 1.5,   -1, 5.446890e+01, 0,_
 471,  24,  56,  0, 0,     +1, 5.546010e+01, 0,_
 472,  24,  57,  0, 1.5,   -1, 5.645440e+01, 0,_
 473,  24,  58,  0, 0,     +1, 5.744660e+01, 0,_
 474,  24,  59,  0, 2.5,   -1, 5.844220e+01, 0,_
 475,  24,  60,  0, 0,     +1, 5.943510e+01, 0,_
 476,  24,  61,  0, 2.5,   -1, 6.043110e+01, 0,_
 477,  24,  62,  0, 0,     +1, 6.142440e+01, 0,_
 478,  24,  63,  0, 0.5,   -1, 6.242100e+01, 0,_
 479,  24,  64,  0, 0,     +1, 6.341490e+01, 0,_
 480,  24,  65,  0, 0.5,   -1, 6.441200e+01, 0,_
 481,  24,  66,  0, 0,     +1, 6.540660e+01, 0,_
 482,  24,  67,  0, 0.5,   -1, 6.640420e+01, 0,_
 483,  25,  44,  0, 2,     -1, 4.362880e+01, 0,_
 484,  25,  45,  0, 3.5,   -1, 4.460800e+01, 0,_
 485,  25,  46,  0, 4,     +1, 4.559170e+01, 0,_
 486,  25,  46,  1, 1,     -1, 4.559170e+01, 0.15,_
 487,  25,  47,  0, 2.5,   -1, 4.657260e+01, 0,_
 488,  25,  48,  0, 4,     +1, 4.755650e+01, 0,_
 489,  25,  49,  0, 2.5,   -1, 4.853900e+01, 0,_
 490,  25,  50,  0, 0,     +1, 4.952510e+01, 0,_
 491,  25,  50,  1, 5,     +1, 4.952510e+01, 0.229,_
 492,  25,  51,  0, 2.5,   -1, 5.051050e+01, 0,_
 493,  25,  52,  0, 6,     +1, 5.149930e+01, 0,_
 494,  25,  52,  1, 2,     +1, 5.149930e+01, 0.377738,_
 495,  25,  53,  0, 3.5,   -1, 5.248650e+01, 0,_
 496,  25,  54,  0, 3,     +1, 5.347700e+01, 0,_
 497,  25,  55,  0, 2.5,   -1, 5.446610e+01, 0,_
 498,  25,  56,  0, 3,     +1, 5.545840e+01, 0,_
 499,  25,  57,  0, 2.5,   -1, 5.644920e+01, 0,_
 500,  25,  58,  0, 3,     +1, 5.744230e+01, 0,_
 501,  25,  58,  1, 0,     +1, 5.744230e+01, 0.073,_
 502,  25,  59,  0, 2.5,   -1, 5.843410e+01, 0,_
 503,  25,  60,  0, 0,     +1, 5.942800e+01, 0,_
 504,  25,  60,  1, 3,     +1, 5.942800e+01, 0.2719,_
 505,  25,  61,  0, 2.5,   -1, 6.042110e+01, 0,_
 506,  25,  62,  0, 3,     +1, 6.141630e+01, 0,_
 507,  25,  62,  1, 1,     +1, 6.141630e+01, 0.001,_
 508,  25,  63,  0, 2.5,   -1, 6.240950e+01, 0,_
 509,  25,  64,  0, 1,     +1, 6.340480e+01, 0,_
 510,  25,  65,  0, 2.5,   -1, 6.439830e+01, 0,_
 511,  25,  66,  0,-77.777,+0, 6.539440e+01, 0,_
 512,  25,  67,  0, 2.5,   -1, 6.638890e+01, 0,_
 513,  25,  68,  0,-77.777,+0, 6.738540e+01, 0,_
 514,  25,  69,  0, 2.5,   -1, 6.838030e+01, 0,_
 515,  26,  45,  0, 1.5,   +1, 4.462790e+01, 0,_
 516,  26,  46,  0, 0,     +1, 4.560560e+01, 0,_
 517,  26,  47,  0, 3.5,   -1, 4.658920e+01, 0,_
 518,  26,  48,  0, 0,     +1, 4.756830e+01, 0,_
 519,  26,  49,  0, 3.5,   -1, 4.855290e+01, 0,_
 520,  26,  50,  0, 0,     +1, 4.953380e+01, 0,_
 521,  26,  51,  0, 2.5,   -1, 5.051910e+01, 0,_
 522,  26,  52,  0, 0,     +1, 5.150180e+01, 0,_
 523,  26,  52,  1, 12,    +1, 5.150180e+01, 6.82,_
 524,  26,  53,  0, 3.5,   -1, 5.249050e+01, 0,_
 525,  26,  53,  1, 9.5,   -1, 5.249050e+01, 3.0404,_
 526,  26,  54,  0, 0,     +1, 5.347620e+01, 0,_
 527,  26,  55,  0, 1.5,   -1, 5.446630e+01, 0,_
 528,  26,  56,  0, 0,     +1, 5.545440e+01, 0,_
 529,  26,  57,  0, 0.5,   -1, 5.644630e+01, 0,_
 530,  26,  58,  0, 0,     +1, 5.743560e+01, 0,_
 531,  26,  59,  0, 1.5,   -1, 5.842860e+01, 0,_
 532,  26,  60,  0, 0,     +1, 5.941920e+01, 0,_
 533,  26,  61,  0,-77.777,+0, 6.041330e+01, 0,_
 534,  26,  62,  0, 0,     +1, 6.140470e+01, 0,_
 535,  26,  63,  0, 2.5,   -1, 6.239940e+01, 0,_
 536,  26,  64,  0, 0,     +1, 6.339190e+01, 0,_
 537,  26,  65,  0, 0.5,   -1, 6.438750e+01, 0,_
 538,  26,  66,  0, 0,     +1, 6.538030e+01, 0,_
 539,  26,  67,  0, 0.5,   -1, 6.637580e+01, 0,_
 540,  26,  68,  0, 0,     +1, 6.736990e+01, 0,_
 541,  26,  69,  0, 0.5,   -1, 6.836640e+01, 0,_
 542,  26,  70,  0, 0,     +1, 6.936050e+01, 0,_
 543,  26,  71,  0, 3.5,   +1, 7.035710e+01, 0,_
 544,  26,  72,  0, 0,     +1, 7.135140e+01, 0,_
 545,  27,  47,  0, 3.5,   -1, 4.660760e+01, 0,_
 546,  27,  48,  0, 6,     +1, 4.758940e+01, 0,_
 547,  27,  49,  0, 3.5,   -1, 4.856890e+01, 0,_
 548,  27,  50,  0, 6,     +1, 4.955220e+01, 0,_
 549,  27,  51,  0, 3.5,   -1, 5.053290e+01, 0,_
 550,  27,  52,  0, 6,     +1, 5.151720e+01, 0,_
 551,  27,  52,  1, 2,     +1, 5.151720e+01, 0.38,_
 552,  27,  53,  0, 3.5,   -1, 5.249930e+01, 0,_
 553,  27,  53,  1, 9.5,   -1, 5.249930e+01, 3.195,_
 554,  27,  54,  0, 0,     +1, 5.348500e+01, 0,_
 555,  27,  54,  1, 7,     +1, 5.348500e+01, 0.1974,_
 556,  27,  55,  0, 3.5,   -1, 5.447000e+01, 0,_
 557,  27,  56,  0, 4,     +1, 5.545930e+01, 0,_
 558,  27,  57,  0, 3.5,   -1, 5.644720e+01, 0,_
 559,  27,  58,  0, 2,     +1, 5.743810e+01, 0,_
 560,  27,  58,  1, 5,     +1, 5.743810e+01, 0.02488,_
 561,  27,  59,  0, 3.5,   -1, 5.842690e+01, 0,_
 562,  27,  60,  0, 5,     +1, 5.941900e+01, 0,_
 563,  27,  60,  1, 2,     +1, 5.941900e+01, 0.0586,_
 564,  27,  61,  0, 3.5,   -1, 6.040900e+01, 0,_
 565,  27,  62,  0, 2,     +1, 6.140200e+01, 0,_
 566,  27,  62,  1, 5,     +1, 6.140200e+01, 0.022,_
 567,  27,  63,  0, 3.5,   -1, 6.239300e+01, 0,_
 568,  27,  64,  0, 1,     +1, 6.338660e+01, 0,_
 569,  27,  65,  0, 3.5,   -1, 6.437860e+01, 0,_
 570,  27,  66,  0, 3,     +1, 6.537330e+01, 0,_
 571,  27,  67,  0, 3.5,   -1, 6.636580e+01, 0,_
 572,  27,  68,  0, 7,     -1, 6.736120e+01, 0,_
 573,  27,  68,  1, 3,     +1, 6.736120e+01, 0.15,_
 574,  27,  69,  0, 3.5,   -1, 6.835400e+01, 0,_
 575,  27,  70,  0, 6,     -1, 6.935010e+01, 0,_
 576,  27,  70,  1, 3,     +1, 6.935010e+01, 0.2,_
 577,  27,  71,  0, 3.5,   -1, 7.034340e+01, 0,_
 578,  27,  72,  0,-77.777,+0, 7.133970e+01, 0,_
 579,  27,  73,  0, 3.5,   -1, 7.233350e+01, 0,_
 580,  27,  74,  0,-77.777,+0, 7.333000e+01, 0,_
 581,  27,  75,  0, 3.5,   -1, 7.432430e+01, 0,_
 582,  28,  48,  0, 0,     +1, 4.760720e+01, 0,_
 583,  28,  49,  0, 3.5,   -1, 4.858860e+01, 0,_
 584,  28,  50,  0, 0,     +1, 4.956640e+01, 0,_
 585,  28,  51,  0, 3.5,   -1, 5.054970e+01, 0,_
 586,  28,  52,  0, 0,     +1, 5.152920e+01, 0,_
 587,  28,  53,  0, 3.5,   -1, 5.251340e+01, 0,_
 588,  28,  54,  0, 0,     +1, 5.349440e+01, 0,_
 589,  28,  55,  0, 3.5,   -1, 5.447930e+01, 0,_
 590,  28,  56,  0, 0,     +1, 5.546160e+01, 0,_
 591,  28,  57,  0, 1.5,   -1, 5.645070e+01, 0,_
 592,  28,  58,  0, 0,     +1, 5.743770e+01, 0,_
 593,  28,  59,  0, 1.5,   -1, 5.842810e+01, 0,_
 594,  28,  60,  0, 0,     +1, 5.941600e+01, 0,_
 595,  28,  61,  0, 1.5,   -1, 6.040760e+01, 0,_
 596,  28,  62,  0, 0,     +1, 6.139640e+01, 0,_
 597,  28,  63,  0, 0.5,   -1, 6.238910e+01, 0,_
 598,  28,  64,  0, 0,     +1, 6.337880e+01, 0,_
 599,  28,  65,  0, 2.5,   -1, 6.437230e+01, 0,_
 600,  28,  66,  0, 0,     +1, 6.536280e+01, 0,_
 601,  28,  67,  0, 0.5,   -1, 6.635660e+01, 0,_
 602,  28,  68,  0, 0,     +1, 6.734830e+01, 0,_
 603,  28,  69,  0, 4.5,   +1, 6.834340e+01, 0,_
 604,  28,  69,  1, 0.5,   -1, 6.834340e+01, 0.321,_
 605,  28,  70,  0, 0,     +1, 6.933570e+01, 0,_
 606,  28,  71,  0, 0.5,   -1, 7.033130e+01, 0,_
 607,  28,  72,  0, 0,     +1, 7.132410e+01, 0,_
 608,  28,  73,  0, 4.5,   +1, 7.231980e+01, 0,_
 609,  28,  74,  0, 0,     +1, 7.331280e+01, 0,_
 610,  28,  75,  0, 3.5,   +1, 7.430900e+01, 0,_
 611,  28,  76,  0, 0,     +1, 7.530280e+01, 0,_
 612,  28,  77,  0, 4.5,   +1, 7.629940e+01, 0,_
 613,  28,  78,  0, 0,     +1, 7.729340e+01, 0,_
 614,  29,  52,  0, 3,     +1, 5.155050e+01, 0,_
 615,  29,  53,  0, 1.5,   -1, 5.253040e+01, 0,_
 616,  29,  54,  0, 3,     +1, 5.351300e+01, 0,_
 617,  29,  55,  0, 1.5,   -1, 5.449390e+01, 0,_
 618,  29,  56,  0, 4,     +1, 5.547790e+01, 0,_
 619,  29,  57,  0, 1.5,   -1, 5.646000e+01, 0,_
 620,  29,  58,  0, 1,     +1, 5.744680e+01, 0,_
 621,  29,  59,  0, 1.5,   -1, 5.843320e+01, 0,_
 622,  29,  60,  0, 2,     +1, 5.942250e+01, 0,_
 623,  29,  61,  0, 1.5,   -1, 6.041000e+01, 0,_
 624,  29,  62,  0, 1,     +1, 6.140060e+01, 0,_
 625,  29,  63,  0, 1.5,   -1, 6.238900e+01, 0,_
 626,  29,  64,  0, 1,     +1, 6.338060e+01, 0,_
 627,  29,  65,  0, 1.5,   -1, 6.437000e+01, 0,_
 628,  29,  66,  0, 1,     +1, 6.536250e+01, 0,_
 629,  29,  67,  0, 1.5,   -1, 6.635280e+01, 0,_
 630,  29,  68,  0, 1,     +1, 6.734610e+01, 0,_
 631,  29,  68,  1, 6,     -1, 6.734610e+01, 0.7216,_
 632,  29,  69,  0, 1.5,   -1, 6.833730e+01, 0,_
 633,  29,  70,  0, 6,     -1, 6.933160e+01, 0,_
 634,  29,  70,  1, 3,     -1, 6.933160e+01, 0.1011,_
 635,  29,  70,  2, 1,     +1, 6.933160e+01, 0.2426,_
 636,  29,  71,  0, 1.5,   -1, 7.032330e+01, 0,_
 637,  29,  72,  0, 1,     +1, 7.131790e+01, 0,_
 638,  29,  73,  0, 1.5,   -1, 7.231010e+01, 0,_
 639,  29,  74,  0, 1,     +1, 7.330470e+01, 0,_
 640,  29,  75,  0, 1.5,   -1, 7.429810e+01, 0,_
 641,  29,  76,  0,-77.777,+0, 7.529290e+01, 0,_
 642,  29,  76,  1,-77.777,+0, 7.529290e+01, 0,_
 643,  29,  77,  0, 1.5,   -1, 7.628680e+01, 0,_
 644,  29,  78,  0,-77.777,+0, 7.728230e+01, 0,_
 645,  29,  79,  0, 1.5,   -1, 7.827630e+01, 0,_
 646,  29,  80,  0,-77.777,+0, 7.927400e+01, 0,_
 647,  30,  54,  0, 0,     +1, 5.352910e+01, 0,_
 648,  30,  55,  0, 2.5,   -1, 5.451160e+01, 0,_
 649,  30,  56,  0, 0,     +1, 5.549150e+01, 0,_
 650,  30,  57,  0, 3.5,   -1, 5.647540e+01, 0,_
 651,  30,  58,  0, 0,     +1, 5.745670e+01, 0,_
 652,  30,  59,  0, 1.5,   -1, 5.844290e+01, 0,_
 653,  30,  60,  0, 0,     +1, 5.942690e+01, 0,_
 654,  30,  61,  0, 1.5,   -1, 6.041600e+01, 0,_
 655,  30,  61,  1, 0.5,   -1, 6.041600e+01, 0.0884,_
 656,  30,  61,  2, 1.5,   -1, 6.041600e+01, 0.4181,_
 657,  30,  61,  3, 2.5,   -1, 6.041600e+01, 0.75602,_
 658,  30,  62,  0, 0,     +1, 6.140230e+01, 0,_
 659,  30,  63,  0, 1.5,   -1, 6.239260e+01, 0,_
 660,  30,  64,  0, 0,     +1, 6.338000e+01, 0,_
 661,  30,  65,  0, 2.5,   -1, 6.437150e+01, 0,_
 662,  30,  66,  0, 0,     +1, 6.535970e+01, 0,_
 663,  30,  67,  0, 2.5,   -1, 6.635220e+01, 0,_
 664,  30,  68,  0, 0,     +1, 6.734130e+01, 0,_
 665,  30,  69,  0, 0.5,   -1, 6.833440e+01, 0,_
 666,  30,  69,  1, 4.5,   +1, 6.833440e+01, 0.438636,_
 667,  30,  70,  0, 0,     +1, 6.932460e+01, 0,_
 668,  30,  71,  0, 0.5,   -1, 7.031840e+01, 0,_
 669,  30,  71,  1, 4.5,   +1, 7.031840e+01, 0.1577,_
 670,  30,  72,  0, 0,     +1, 7.130900e+01, 0,_
 671,  30,  73,  0, 0.5,   -1, 7.230330e+01, 0,_
 672,  30,  73,  1, 2.5,   +1, 7.230330e+01, 0.1955,_
 673,  30,  73,  2, 3.5,   +1, 7.230330e+01, 0.2376,_
 674,  30,  74,  0, 0,     +1, 7.329440e+01, 0,_
 675,  30,  75,  0, 3.5,   +1, 7.428920e+01, 0,_
 676,  30,  76,  0, 0,     +1, 7.528100e+01, 0,_
 677,  30,  77,  0, 3.5,   +1, 7.627600e+01, 0,_
 678,  30,  77,  1, 0.5,   -1, 7.627600e+01, 0.77239,_
 679,  30,  78,  0, 0,     +1, 7.726890e+01, 0,_
 680,  30,  79,  0, 4.5,   +1, 7.826450e+01, 0,_
 681,  30,  80,  0, 0,     +1, 7.925760e+01, 0,_
 682,  30,  81,  0, 2.5,   +1, 8.025510e+01, 0,_
 683,  30,  82,  0, 0,     +1, 8.125040e+01, 0,_
 684,  30,  83,  0, 2.5,   +1, 8.224840e+01, 0,_
 685,  31,  56,  0, 3,     +1, 5.551390e+01, 0,_
 686,  31,  57,  0, 0.5,   -1, 5.649340e+01, 0,_
 687,  31,  58,  0, 2,     +1, 5.747620e+01, 0,_
 688,  31,  59,  0, 1.5,   -1, 5.845680e+01, 0,_
 689,  31,  60,  0, 2,     +1, 5.944200e+01, 0,_
 690,  31,  61,  0, 1.5,   -1, 6.042590e+01, 0,_
 691,  31,  62,  0, 0,     +1, 6.141200e+01, 0,_
 692,  31,  63,  0, 1.5,   -1, 6.239860e+01, 0,_
 693,  31,  64,  0, 0,     +1, 6.338760e+01, 0,_
 694,  31,  65,  0, 1.5,   -1, 6.437490e+01, 0,_
 695,  31,  66,  0, 0,     +1, 6.536520e+01, 0,_
 696,  31,  67,  0, 1.5,   -1, 6.635330e+01, 0,_
 697,  31,  68,  0, 1,     +1, 6.734440e+01, 0,_
 698,  31,  69,  0, 1.5,   -1, 6.833350e+01, 0,_
 699,  31,  70,  0, 1,     +1, 6.932530e+01, 0,_
 700,  31,  71,  0, 1.5,   -1, 7.031540e+01, 0,_
 701,  31,  72,  0, 3,     -1, 7.130850e+01, 0,_
 702,  31,  72,  1, 0,     +1, 7.130850e+01, 0.11966,_
 703,  31,  73,  0, 1.5,   -1, 7.229870e+01, 0,_
 704,  31,  74,  0, 3,     -1, 7.329190e+01, 0,_
 705,  31,  74,  1,-77.777,+0, 7.329190e+01, 0.059571,_
 706,  31,  75,  0, 1.5,   -1, 7.428280e+01, 0,_
 707,  31,  76,  0,-77.777,+0, 7.527660e+01, 0,_
 708,  31,  77,  0, 1.5,   -1, 7.626840e+01, 0,_
 709,  31,  78,  0, 3,     +1, 7.726210e+01, 0,_
 710,  31,  79,  0, 1.5,   -1, 7.825480e+01, 0,_
 711,  31,  80,  0, 3,     +0, 7.924980e+01, 0,_
 712,  31,  81,  0, 2.5,   -1, 8.024250e+01, 0,_
 713,  31,  82,  0,-77.777,+0, 8.123910e+01, 0,_
 714,  31,  83,  0, 1.5,   -1, 8.223440e+01, 0,_
 715,  31,  84,  0,-77.777,+0, 8.323150e+01, 0,_
 716,  31,  85,  0, 1.5,   -1, 8.422720e+01, 0,_
 717,  31,  86,  0,-77.777,+0, 8.522470e+01, 0,_
 718,  32,  58,  0, 0,     +1, 5.749280e+01, 0,_
 719,  32,  59,  0, 3.5,   -1, 5.847510e+01, 0,_
 720,  32,  60,  0, 0,     +1, 5.945500e+01, 0,_
 721,  32,  61,  0, 1.5,   -1, 6.044010e+01, 0,_
 722,  32,  62,  0, 0,     +1, 6.142240e+01, 0,_
 723,  32,  63,  0, 1.5,   -1, 6.240890e+01, 0,_
 724,  32,  64,  0, 0,     +1, 6.339240e+01, 0,_
 725,  32,  65,  0, 1.5,   -1, 6.438160e+01, 0,_
 726,  32,  66,  0, 0,     +1, 6.536740e+01, 0,_
 727,  32,  67,  0, 0.5,   -1, 6.635770e+01, 0,_
 728,  32,  68,  0, 0,     -1, 6.734460e+01, 0,_
 729,  32,  69,  0, 2.5,   -1, 6.833580e+01, 0,_
 730,  32,  70,  0, 0,     +1, 6.932360e+01, 0,_
 731,  32,  71,  0, 0.5,   -1, 7.031570e+01, 0,_
 732,  32,  71,  1, 4.5,   +1, 7.031570e+01, 0.198367,_
 733,  32,  72,  0, 0,     +1, 7.130420e+01, 0,_
 734,  32,  73,  0, 4.5,   +1, 7.229700e+01, 0,_
 735,  32,  73,  1, 0.5,   -1, 7.229700e+01, 0.066726,_
 736,  32,  74,  0, 0,     +1, 7.328620e+01, 0,_
 737,  32,  75,  0, 0.5,   -1, 7.427920e+01, 0,_
 738,  32,  75,  1, 3.5,   +1, 7.427920e+01, 0.13969,_
 739,  32,  76,  0, 0,     +1, 7.526920e+01, 0,_
 740,  32,  77,  0, 3.5,   +1, 7.626270e+01, 0,_
 741,  32,  77,  1, 0.5,   -1, 7.626270e+01, 0.1597,_
 742,  32,  78,  0, 0,     +1, 7.725350e+01, 0,_
 743,  32,  79,  0, 0.5,   -1, 7.824740e+01, 0,_
 744,  32,  79,  1, 3.5,   +1, 7.824740e+01, 0.18602,_
 745,  32,  80,  0, 0,     +1, 7.923880e+01, 0,_
 746,  32,  81,  0, 4.5,   +1, 8.023360e+01, 0,_
 747,  32,  81,  1, 0.5,   +1, 8.023360e+01, 0.67913,_
 748,  32,  82,  0, 0,     +1, 8.122570e+01, 0,_
 749,  32,  83,  0, 2.5,   +1, 8.222220e+01, 0,_
 750,  32,  84,  0, 0,     +1, 8.321640e+01, 0,_
 751,  32,  85,  0, 2.5,   +1, 8.421330e+01, 0,_
 752,  32,  86,  0, 0,     +1, 8.520820e+01, 0,_
 753,  32,  87,  0, 2.5,   +1, 8.620550e+01, 0,_
 754,  32,  88,  0, 0,     +1, 8.720130e+01, 0,_
 755,  32,  89,  0, 1.5,   +1, 8.819960e+01, 0,_
 756,  33,  60,  0, 5,     +1, 5.947780e+01, 0,_
 757,  33,  60,  1, 2,     +1, 5.947780e+01, 0.06,_
 758,  33,  61,  0, 1.5,   -1, 6.045680e+01, 0,_
 759,  33,  62,  0, 1,     +1, 6.144080e+01, 0,_
 760,  33,  63,  0, 1.5,   -1, 6.242280e+01, 0,_
 761,  33,  64,  0, 0,     +1, 6.340810e+01, 0,_
 762,  33,  65,  0, 1.5,   -1, 6.439160e+01, 0,_
 763,  33,  66,  0, 0,     +1, 6.537820e+01, 0,_
 764,  33,  67,  0, 2.5,   -1, 6.636410e+01, 0,_
 765,  33,  68,  0, 3,     +1, 6.735320e+01, 0,_
 766,  33,  69,  0, 2.5,   -1, 6.834010e+01, 0,_
 767,  33,  70,  0, 4,     +1, 6.933020e+01, 0,_
 768,  33,  71,  0, 2.5,   -1, 7.031780e+01, 0,_
 769,  33,  72,  0, 2,     -1, 7.130890e+01, 0,_
 770,  33,  73,  0, 1.5,   -1, 7.229740e+01, 0,_
 771,  33,  74,  0, 2,     -1, 7.328890e+01, 0,_
 772,  33,  75,  0, 1.5,   -1, 7.427800e+01, 0,_
 773,  33,  75,  1, 4.5,   +1, 7.427800e+01, 0.303924,_
 774,  33,  76,  0, 2,     -1, 7.527020e+01, 0,_
 775,  33,  77,  0, 1.5,   -1, 7.625990e+01, 0,_
 776,  33,  78,  0, 2,     -1, 7.725240e+01, 0,_
 777,  33,  79,  0, 1.5,   -1, 7.824300e+01, 0,_
 778,  33,  80,  0, 1,     +1, 7.923600e+01, 0,_
 779,  33,  81,  0, 1.5,   -1, 8.022700e+01, 0,_
 780,  33,  82,  0, 1,     +1, 8.122070e+01, 0,_
 781,  33,  82,  1, 5,     -1, 8.122070e+01, 0.245,_
 782,  33,  83,  0, 1.5,   -1, 8.221260e+01, 0,_
 783,  33,  84,  0, 3,     +1, 8.320810e+01, 0,_
 784,  33,  84,  1,-77.777,+0, 8.320810e+01, 0,_
 785,  33,  85,  0, 1.5,   -1, 8.420220e+01, 0,_
 786,  33,  86,  0,-77.777,+0, 8.519830e+01, 0,_
 787,  33,  87,  0, 1.5,   -1, 8.619300e+01, 0,_
 788,  33,  88,  0,-77.777,+0, 8.718940e+01, 0,_
 789,  33,  89,  0, 1.5,   -1, 8.818530e+01, 0,_
 790,  33,  90,  0,-77.777,+0, 8.918270e+01, 0,_
 791,  33,  91,  0, 1.5,   -1, 9.017900e+01, 0,_
 792,  33,  92,  0,-77.777,+0, 9.117680e+01, 0,_
 793,  34,  65,  0, 1.5,   -1, 6.440660e+01, 0,_
 794,  34,  66,  0, 0,     +1, 6.538860e+01, 0,_
 795,  34,  67,  0, 2.5,   -1, 6.637500e+01, 0,_
 796,  34,  68,  0, 0,     +1, 6.735820e+01, 0,_
 797,  34,  69,  0, 1.5,   -1, 6.834730e+01, 0,_
 798,  34,  70,  0, 0,     +1, 6.933260e+01, 0,_
 799,  34,  71,  0, 2.5,   -1, 7.032290e+01, 0,_
 800,  34,  72,  0, 0,     +1, 7.130920e+01, 0,_
 801,  34,  73,  0, 4.5,   +1, 7.230030e+01, 0,_
 802,  34,  73,  1, 1.5,   -1, 7.230030e+01, 0.02571,_
 803,  34,  74,  0, 0,     +1, 7.328740e+01, 0,_
 804,  34,  75,  0, 2.5,   +1, 7.427890e+01, 0,_
 805,  34,  76,  0, 0,     +1, 7.526700e+01, 0,_
 806,  34,  77,  0, 0.5,   -1, 7.625910e+01, 0,_
 807,  34,  77,  1, 3.5,   +1, 7.625910e+01, 0.161922,_
 808,  34,  78,  0, 0,     +1, 7.724800e+01, 0,_
 809,  34,  79,  0, 3.5,   +1, 7.824060e+01, 0,_
 810,  34,  79,  1, 0.5,   -1, 7.824060e+01, 0.09577,_
 811,  34,  80,  0, 0,     +1, 7.923000e+01, 0,_
 812,  34,  81,  0, 0.5,   -1, 8.022290e+01, 0,_
 813,  34,  81,  1, 3.5,   +1, 8.022290e+01, 0.10299,_
 814,  34,  82,  0, 0,     +1, 8.121300e+01, 0,_
 815,  34,  83,  0, 4.5,   +1, 8.220680e+01, 0,_
 816,  34,  83,  1, 0.5,   -1, 8.220680e+01, 0.22866,_
 817,  34,  84,  0, 0,     +1, 8.319760e+01, 0,_
 818,  34,  85,  0, 2.5,   +1, 8.419270e+01, 0,_
 819,  34,  86,  0, 0,     +1, 8.518610e+01, 0,_
 820,  34,  87,  0, 2.5,   +1, 8.618180e+01, 0,_
 821,  34,  88,  0, 0,     +1, 8.717600e+01, 0,_
 822,  34,  89,  0, 2.5,   +1, 8.817240e+01, 0,_
 823,  34,  90,  0, 0,     +1, 8.916730e+01, 0,_
 824,  34,  91,  0, 0.5,   +1, 9.016470e+01, 0,_
 825,  34,  92,  0, 0,     +1, 9.116000e+01, 0,_
 826,  34,  93,  0, 0.5,   +1, 9.215770e+01, 0,_
 827,  34,  94,  0, 0,     +1, 9.315330e+01, 0,_
 828,  35,  67,  0, 0.5,   -1, 6.638950e+01, 0,_
 829,  35,  68,  0, 3,     +1, 6.737470e+01, 0,_
 830,  35,  69,  0, 0.5,   -1, 6.835780e+01, 0,_
 831,  35,  70,  0, 0,     +1, 6.934390e+01, 0,_
 832,  35,  70,  1, 9,     +1, 6.934390e+01, 2.2923,_
 833,  35,  71,  0, 2.5,   -1, 7.032930e+01, 0,_
 834,  35,  72,  0, 1,     +1, 7.131850e+01, 0,_
 835,  35,  72,  1, 1,     -1, 7.131850e+01, 0.1011,_
 836,  35,  73,  0, 0.5,   -1, 7.230520e+01, 0,_
 837,  35,  74,  0, 0,     -1, 7.329480e+01, 0,_
 838,  35,  74,  1, 4,     +1, 7.329480e+01, 0.0138,_
 839,  35,  75,  0, 1.5,   -1, 7.428210e+01, 0,_
 840,  35,  76,  0, 1,     -1, 7.527230e+01, 0,_
 841,  35,  76,  1, 4,     +1, 7.527230e+01, 0.10258,_
 842,  35,  77,  0, 1.5,   -1, 7.626060e+01, 0,_
 843,  35,  77,  1, 4.5,   +1, 7.626060e+01, 0.10587,_
 844,  35,  78,  0, 1,     +1, 7.725180e+01, 0,_
 845,  35,  79,  0, 1.5,   -1, 7.824040e+01, 0,_
 846,  35,  79,  1, 4.5,   +1, 7.824040e+01, 0.2072,_
 847,  35,  80,  0, 1,     +1, 7.923200e+01, 0,_
 848,  35,  80,  1, 5,     -1, 7.923200e+01, 0.08585,_
 849,  35,  81,  0, 1.5,   -1, 8.022120e+01, 0,_
 850,  35,  82,  0, 5,     -1, 8.121310e+01, 0,_
 851,  35,  82,  1, 2,     -1, 8.121310e+01, 0.045949,_
 852,  35,  83,  0, 1.5,   -1, 8.220290e+01, 0,_
 853,  35,  84,  0, 2,     -1, 8.319560e+01, 0,_
 854,  35,  84,  1, 6,     -1, 8.319560e+01, 0.32,_   ' spin from ENSDF
 855,  35,  85,  0, 1.5,   -1, 8.418610e+01, 0,_
 856,  35,  86,  0, 2,     -1, 8.518070e+01, 0,_
 857,  35,  87,  0, 1.5,   -1, 8.617400e+01, 0,_
 858,  35,  88,  0, 2,     -1, 8.716880e+01, 0,_
 859,  35,  89,  0, 1.5,   -1, 8.816250e+01, 0,_
 860,  35,  90,  0, 1,     -1, 8.915810e+01, 0,_
 861,  35,  91,  0, 1.5,   -1, 9.015280e+01, 0,_
 862,  35,  92,  0, 2,     -1, 9.114950e+01, 0,_
 863,  35,  93,  0, 1.5,   -1, 9.214460e+01, 0,_
 864,  35,  94,  0,-77.777,+0, 9.314160e+01, 0,_
 865,  35,  95,  0, 1.5,   -1, 9.413720e+01, 0,_
 866,  35,  96,  0,-77.777,+0, 9.513420e+01, 0,_
 867,  35,  97,  0, 1.5,   -1, 9.612980e+01, 0,_
 868,  36,  69,  0, 2.5,   -1, 6.837270e+01, 0,_
 869,  36,  70,  0, 0,     +1, 6.935430e+01, 0,_
 870,  36,  71,  0, 2.5,   -1, 7.034010e+01, 0,_
 871,  36,  72,  0, 0,     +1, 7.132410e+01, 0,_
 872,  36,  73,  0, 1.5,   -1, 7.231270e+01, 0,_
 873,  36,  74,  0, 0,     +1, 7.329800e+01, 0,_
 874,  36,  75,  0, 2.5,   +1, 7.428720e+01, 0,_
 875,  36,  76,  0, 0,     +1, 7.527370e+01, 0,_
 876,  36,  77,  0, 2.5,   +1, 7.626380e+01, 0,_
 877,  36,  78,  0, 0,     +1, 7.725100e+01, 0,_
 878,  36,  79,  0, 0.5,   -1, 7.824210e+01, 0,_
 879,  36,  79,  1, 3.5,   +1, 7.824210e+01, 0.12977,_
 880,  36,  80,  0, 0,     +1, 7.922990e+01, 0,_
 881,  36,  81,  0, 3.5,   +1, 8.022150e+01, 0,_
 882,  36,  81,  1, 0.5,   -1, 8.022150e+01, 0.19053,_
 883,  36,  82,  0, 0,     +1, 8.120980e+01, 0,_
 884,  36,  83,  0, 4.5,   +1, 8.220190e+01, 0,_
 885,  36,  83,  1, 0.5,   -1, 8.220190e+01, 0.041543,_
 886,  36,  84,  0, 0,     +1, 8.319070e+01, 0,_
 887,  36,  85,  0, 4.5,   +1, 8.418310e+01, 0,_
 888,  36,  85,  1, 0.5,   -1, 8.418310e+01, 0.30487,_
 889,  36,  86,  0, 0,     +1, 8.517260e+01, 0,_
 890,  36,  87,  0, 2.5,   +1, 8.616670e+01, 0,_
 891,  36,  88,  0, 0,     +1, 8.715920e+01, 0,_
 892,  36,  89,  0, 1.5,   +1, 8.815380e+01, 0,_
 893,  36,  90,  0, 0,     +1, 8.914710e+01, 0,_
 894,  36,  91,  0, 2.5,   +1, 9.014240e+01, 0,_
 895,  36,  92,  0, 0,     +1, 9.113650e+01, 0,_
 896,  36,  93,  0, 0.5,   +1, 9.213290e+01, 0,_
 897,  36,  94,  0, 0,     +1, 9.312740e+01, 0,_
 898,  36,  95,  0, 0.5,   +1, 9.412430e+01, 0,_
 899,  36,  96,  0, 0,     +1, 9.511890e+01, 0,_
 900,  36,  97,  0, 1.5,   +1, 9.611570e+01, 0,_
 901,  36,  98,  0, 0,     +1, 9.711050e+01, 0,_
 902,  36,  99,  0, 1.5,   +1, 9.810750e+01, 0,_
 903,  36,  100, 0, 0,     +1, 9.910240e+01, 0,_
 904,  37,  71,  0, 2.5,   -1, 7.035570e+01, 0,_
 905,  37,  72,  0, 3,     +1, 7.134090e+01, 0,_
 906,  37,  73,  0, 1.5,   -1, 7.232390e+01, 0,_
 907,  37,  74,  0, 0,     +1, 7.330900e+01, 0,_
 908,  37,  75,  0, 1.5,   -1, 7.429480e+01, 0,_
 909,  37,  76,  0, 1,     -1, 7.528270e+01, 0,_
 910,  37,  77,  0, 1.5,   -1, 7.626950e+01, 0,_
 911,  37,  78,  0, 0,     +1, 7.725870e+01, 0,_
 912,  37,  78,  1, 4,     -1, 7.725870e+01, 0.1033,_
 913,  37,  79,  0, 2.5,   +1, 7.824600e+01, 0,_
 914,  37,  80,  0, 1,     +1, 7.923590e+01, 0,_
 915,  37,  81,  0, 1.5,   -1, 8.022390e+01, 0,_
 916,  37,  81,  1, 4.5,   +1, 8.022390e+01, 0.0862,_
 917,  37,  82,  0, 1,     +1, 8.121450e+01, 0,_
 918,  37,  82,  1, 5,     -1, 8.121450e+01, 0.0689,_
 919,  37,  83,  0, 2.5,   -1, 8.220280e+01, 0,_
 920,  37,  83,  1, 4.5,   +1, 8.220280e+01, 0.04211,_
 921,  37,  84,  0, 2,     -1, 8.319350e+01, 0,_
 922,  37,  84,  1, 6,     -1, 8.319350e+01, 0.4636,_
 923,  37,  85,  0, 2.5,   -1, 8.418240e+01, 0,_
 924,  37,  86,  0, 2,     -1, 8.517310e+01, 0,_
 925,  37,  86,  1, 6,     -1, 8.517320e+01, 0.556,_
 926,  37,  87,  0, 1.5,   -1, 8.616260e+01, 0,_
 927,  37,  88,  0, 2,     -1, 8.715610e+01, 0,_
 928,  37,  89,  0, 1.5,   -1, 8.814850e+01, 0,_
 929,  37,  90,  0, 0,     -1, 8.914240e+01, 0,_
 930,  37,  90,  1, 3,     -1, 8.914240e+01, 0.1069,_
 931,  37,  91,  0, 1.5,   -1, 9.013550e+01, 0,_
 932,  37,  92,  0, 0,     -1, 9.113010e+01, 0,_
 933,  37,  93,  0, 2.5,   -1, 9.212380e+01, 0,_
 934,  37,  94,  0, 3,     -1, 9.311960e+01, 0,_
 935,  37,  95,  0, 2.5,   -1, 9.411380e+01, 0,_
 936,  37,  96,  0, 2,     +1, 9.511020e+01, 0,_
 937,  37,  96,  1, 1,     -1, 9.511020e+01, 0,_
 938,  37,  97,  0, 1.5,   +1, 9.610460e+01, 0,_
 939,  37,  98,  0,-77.777,+0, 9.710040e+01, 0,_
 940,  37,  98,  1,-77.777,+0, 9.710040e+01, 0.39,_
 941,  37,  99,  0, 2.5,   +1, 9.809540e+01, 0,_
 942,  37,  100, 0, 3,     +1, 9.909120e+01, 0,_
 943,  37,  101, 0, 1.5,   +1, 1.000860e+02, 0,_
 944,  37,  102, 0,-77.777,+0, 1.010830e+02, 0,_
 945,  38,  73,  0, 0.5,   -1, 7.233920e+01, 0,_
 946,  38,  74,  0, 0,     +1, 7.332100e+01, 0,_
 947,  38,  75,  0, 1.5,   -1, 7.430610e+01, 0,_
 948,  38,  76,  0, 0,     +1, 7.528940e+01, 0,_
 949,  38,  77,  0, 2.5,   +1, 7.627700e+01, 0,_
 950,  38,  78,  0, 0,     +1, 7.726270e+01, 0,_
 951,  38,  79,  0, 1.5,   -1, 7.825170e+01, 0,_
 952,  38,  80,  0, 0,     +1, 7.923790e+01, 0,_
 953,  38,  81,  0, 0.5,   -1, 8.022800e+01, 0,_
 954,  38,  82,  0, 0,     +1, 8.121470e+01, 0,_
 955,  38,  83,  0, 3.5,   +1, 8.220530e+01, 0,_
 956,  38,  83,  1, 0.5,   -1, 8.220530e+01, 0.25915,_
 957,  38,  84,  0, 0,     +1, 8.319260e+01, 0,_
 958,  38,  85,  0, 4.5,   +1, 8.418350e+01, 0,_
 959,  38,  85,  1, 0.5,   -1, 8.418350e+01, 0.23866,_
 960,  38,  86,  0, 0,     +1, 8.517130e+01, 0,_
 961,  38,  87,  0, 4.5,   +1, 8.616230e+01, 0,_
 962,  38,  87,  1, 0.5,   -1, 8.616230e+01, 0.388533,_
 963,  38,  88,  0, 0,     +1, 8.715050e+01, 0,_
 964,  38,  89,  0, 2.5,   +1, 8.814370e+01, 0,_
 965,  38,  90,  0, 0,     +1, 8.913540e+01, 0,_
 966,  38,  91,  0, 2.5,   +1, 9.012920e+01, 0,_
 967,  38,  92,  0, 0,     +1, 9.112150e+01, 0,_
 968,  38,  93,  0, 2.5,   +1, 9.211580e+01, 0,_
 969,  38,  94,  0, 0,     +1, 9.310860e+01, 0,_
 970,  38,  95,  0, 0.5,   +1, 9.410390e+01, 0,_
 971,  38,  96,  0, 0,     +1, 9.509770e+01, 0,_
 972,  38,  97,  0, 0.5,   +1, 9.609350e+01, 0,_
 973,  38,  98,  0, 0,     +1, 9.708720e+01, 0,_
 974,  38,  99,  0, 1.5,   +1, 9.808330e+01, 0,_
 975,  38,  100, 0, 0,     +1, 9.907690e+01, 0,_
 976,  38,  101, 0, 2.5,   -1, 1.000730e+02, 0,_
 977,  38,  102, 0, 0,     +1, 1.010670e+02, 0,_
 978,  38,  103, 0,-77.777,+0, 1.020650e+02, 0,_
 979,  38,  104, 0, 0,     +1, 1.030590e+02, 0,_
 980,  38,  105, 0,-77.777,+0, 1.040570e+02, 0,_
 981,  39,  76,  0,-77.777,+0, 7.530590e+01, 0,_
 982,  39,  77,  0, 2.5,   +1, 7.628860e+01, 0,_
 983,  39,  78,  0, 0,     +1, 7.727400e+01, 0,_
 984,  39,  78,  1, 5,     +1, 7.727400e+01, 0,_
 985,  39,  79,  0, 2.5,   +1, 7.825920e+01, 0,_
 986,  39,  80,  0, 4,     -1, 7.924760e+01, 0,_
 987,  39,  80,  1, 1,     -1, 7.924760e+01, 0.2285,_
 988,  39,  81,  0, 2.5,   +1, 8.023390e+01, 0,_
 989,  39,  82,  0, 1,     +1, 8.122300e+01, 0,_
 990,  39,  83,  0, 4.5,   +1, 8.221000e+01, 0,_
 991,  39,  83,  1, 1.5,   -1, 8.221000e+01, 0.06198,_
 992,  39,  84,  0, 1,     +1, 8.319950e+01, 0,_
 993,  39,  84,  1, 5,     -1, 8.319950e+01, 0.5,_
 994,  39,  85,  0, 0.5,   -1, 8.418700e+01, 0,_
 995,  39,  85,  1, 4.5,   +1, 8.418700e+01, 0.0198,_
 996,  39,  86,  0, 4,     -1, 8.517680e+01, 0,_
 997,  39,  86,  1, 8,     +1, 8.517680e+01, 0.2183,_
 998,  39,  87,  0, 0.5,   -1, 8.616430e+01, 0,_
 999,  39,  87,  1, 4.5,   +1, 8.616430e+01, 0.38082,_
 1000, 39,  88,  0, 4,     -1, 8.715430e+01, 0,_
 1001, 39,  88,  1, 8,     +1, 8.715430e+01, 0.67455,_
 1002, 39,  89,  0, 0.5,   -1, 8.814210e+01, 0,_
 1003, 39,  89,  1, 4.5,   +1, 8.814210e+01, 0.90896,_
 1004, 39,  90,  0, 2,     -1, 8.913480e+01, 0,_
 1005, 39,  90,  1, 7,     +1, 8.913480e+01, 0.68204,_
 1006, 39,  91,  0, 0.5,   -1, 9.012640e+01, 0,_
 1007, 39,  91,  1, 4.5,   +1, 9.012640e+01, 0.55557,_
 1008, 39,  92,  0, 2,     -1, 9.111940e+01, 0,_
 1009, 39,  93,  0, 0.5,   -1, 9.211140e+01, 0,_
 1010, 39,  93,  1, 3.5,   +1, 9.211140e+01, 0.758721,_
 1011, 39,  94,  0, 2,     -1, 9.310480e+01, 0,_
 1012, 39,  95,  0, 0.5,   -1, 9.409750e+01, 0,_
 1013, 39,  96,  0, 0,     -1, 9.509190e+01, 0,_
 1014, 39,  96,  1, 8,     +1, 9.509190e+01, 1.6,_
 1015, 39,  97,  0, 0.5,   -1, 9.608560e+01, 0,_
 1016, 39,  97,  1, 4.5,   +1, 9.608560e+01, 0.6675,_
 1017, 39,  97,  2, 13.5,  -1, 9.608560e+01, 3.5233,_
 1018, 39,  98,  0, 0,     -1, 9.708100e+01, 0,_
 1019, 39,  98,  1, 5,     +1, 9.708100e+01, 0.41,_
 1020, 39,  99,  0, 2.5,   +1, 9.807480e+01, 0,_
 1021, 39,  100, 0,-77.777,+0, 9.906930e+01, 0,_
 1022, 39,  100, 1,-77.777,+0, 9.906930e+01, 0.2,_
 1023, 39,  101, 0, 2.5,   +1, 1.000630e+02, 0,_
 1024, 39,  102, 0,-77.777,+0, 1.010580e+02, 0,_
 1025, 39,  102, 1,-77.777,+0, 1.010580e+02, 0.2,_
 1026, 39,  103, 0, 2.5,   +1, 1.020520e+02, 0,_
 1027, 39,  104, 0, 0,     -1, 1.030480e+02, 0,_
 1028, 39,  105, 0, 2.5,   +1, 1.040420e+02, 0,_
 1029, 39,  106, 0,-77.777,+0, 1.050400e+02, 0,_
 1030, 39,  107, 0, 2.5,   +1, 1.060350e+02, 0,_
 1031, 39,  108, 0,-77.777,+0, 1.070320e+02, 0,_
 1032, 40,  78,  0, 0,     +1, 7.728560e+01, 0,_
 1033, 40,  79,  0, 2.5,   +1, 7.827090e+01, 0,_
 1034, 40,  80,  0, 0,     +1, 7.925370e+01, 0,_
 1035, 40,  81,  0, 1.5,   -1, 8.024190e+01, 0,_
 1036, 40,  82,  0, 0,     +1, 8.122730e+01, 0,_
 1037, 40,  83,  0, 0.5,   -1, 8.221620e+01, 0,_
 1038, 40,  84,  0, 0,     +1, 8.320230e+01, 0,_
 1039, 40,  85,  0, 3.5,   +1, 8.419190e+01, 0,_
 1040, 40,  85,  1, 0.5,   -1, 8.419200e+01, 0.2922,_
 1041, 40,  86,  0, 0,     +1, 8.517840e+01, 0,_
 1042, 40,  87,  0, 4.5,   +1, 8.616820e+01, 0,_
 1043, 40,  87,  1, 0.5,   -1, 8.616820e+01, 0.3363,_
 1044, 40,  88,  0, 0,     +1, 8.715500e+01, 0,_
 1045, 40,  89,  0, 4.5,   +1, 8.814510e+01, 0,_
 1046, 40,  89,  1, 0.5,   -1, 8.814510e+01, 0.5878,_
 1047, 40,  90,  0, 0,     +1, 8.913240e+01, 0,_
 1048, 40,  90,  1, 5,     -1, 8.913240e+01, 2.31899,_
 1049, 40,  91,  0, 2.5,   +1, 9.012470e+01, 0,_
 1050, 40,  92,  0, 0,     +1, 9.111550e+01, 0,_
 1051, 40,  93,  0, 2.5,   +1, 9.210840e+01, 0,_
 1052, 40,  94,  0, 0,     +1, 9.309960e+01, 0,_
 1053, 40,  95,  0, 2.5,   +1, 9.409270e+01, 0,_
 1054, 40,  96,  0, 0,     +1, 9.508440e+01, 0,_
 1055, 40,  97,  0, 0.5,   +1, 9.607840e+01, 0,_
 1056, 40,  98,  0, 0,     +1, 9.707160e+01, 0,_
 1057, 40,  99,  0, 0.5,   +1, 9.806680e+01, 0,_
 1058, 40,  100, 0, 0,     +1, 9.905940e+01, 0,_
 1059, 40,  101, 0, 1.5,   +1, 1.000540e+02, 0,_
 1060, 40,  102, 0, 0,     +1, 1.010470e+02, 0,_
 1061, 40,  103, 0, 2.5,   -1, 1.020420e+02, 0,_
 1062, 40,  104, 0, 0,     +1, 1.030360e+02, 0,_
 1063, 40,  105, 0, 2.5,   +1, 1.040310e+02, 0,_
 1064, 40,  106, 0, 0,     +1, 1.050250e+02, 0,_
 1065, 40,  107, 0, 2.5,   +1, 1.060220e+02, 0,_
 1066, 40,  108, 0, 0,     +1, 1.070170e+02, 0,_
 1067, 40,  109, 0,-77.777,+0, 1.080130e+02, 0,_
 1068, 40,  110, 0, 0,     +1, 1.090080e+02, 0,_
 1069, 41,  81,  0, 1.5,   -1, 8.025360e+01, 0,_
 1070, 41,  82,  0, 0,     +1, 8.123920e+01, 0,_
 1071, 41,  83,  0, 2.5,   +1, 8.222420e+01, 0,_
 1072, 41,  84,  0, 3,     +1, 8.321250e+01, 0,_
 1073, 41,  85,  0, 4.5,   +1, 8.419830e+01, 0,_
 1074, 41,  85,  1, 0.5,   -1, 8.419830e+01, 0.759,_
 1075, 41,  86,  0, 6,     +1, 8.518690e+01, 0,_
 1076, 41,  86,  1,-77.777,+0, 8.518690e+01, 0.25,_
 1077, 41,  87,  0, 0.5,   -1, 8.617370e+01, 0,_
 1078, 41,  87,  1, 4.5,   +1, 8.617370e+01, 0.00384,_
 1079, 41,  88,  0, 8,     +1, 8.716310e+01, 0,_
 1080, 41,  88,  1, 4,     -1, 8.716310e+01, 0.39,_
 1081, 41,  89,  0, 4.5,   +1, 8.814960e+01, 0,_
 1082, 41,  89,  1, 0.5,   -1, 8.814960e+01, 0,_
 1083, 41,  90,  0, 8,     +1, 8.913890e+01, 0,_
 1084, 41,  90,  1, 4,     -1, 8.913890e+01, 0.12467,_
 1085, 41,  90,  2, 1,     +1, 8.913890e+01, 0.38201,_
 1086, 41,  91,  0, 4.5,   +1, 9.012610e+01, 0,_
 1087, 41,  91,  1, 0.5,   -1, 9.012610e+01, 0.10449,_
 1088, 41,  92,  0, 7,     +1, 9.111770e+01, 0,_
 1089, 41,  92,  1, 2,     +1, 9.111770e+01, 0.1355,_
 1090, 41,  93,  0, 4.5,   +1, 9.210830e+01, 0,_
 1091, 41,  93,  1, 0.5,   -1, 9.210830e+01, 0.0308,_
 1092, 41,  94,  0, 6,     +1, 9.310060e+01, 0,_
 1093, 41,  94,  1, 3,     +1, 9.310060e+01, 0.04095,_
 1094, 41,  95,  0, 4.5,   +1, 9.409150e+01, 0,_
 1095, 41,  95,  1, 0.5,   -1, 9.409150e+01, 0.23569,_
 1096, 41,  96,  0, 6,     +1, 9.508420e+01, 0,_
 1097, 41,  97,  0, 4.5,   +1, 9.607560e+01, 0,_
 1098, 41,  97,  1, 0.5,   -1, 9.607560e+01, 0.7434,_
 1099, 41,  98,  0, 1,     +1, 9.706920e+01, 0,_
 1100, 41,  98,  1, 5,     +1, 9.706920e+01, 0.084,_
 1101, 41,  99,  0, 4.5,   +1, 9.806190e+01, 0,_
 1102, 41,  99,  1, 0.5,   -1, 9.806190e+01, 0.36529,_
 1103, 41,  100, 0, 1,     +1, 9.905590e+01, 0,_
 1104, 41,  100, 1, 4,     +1, 9.905590e+01, 0.48,_
 1105, 41,  101, 0, 2.5,   +1, 1.000480e+02, 0,_
 1106, 41,  102, 0, 1,     +1, 1.010430e+02, 0,_
 1107, 41,  102, 1,-77.777,+0, 1.010430e+02, 0.13,_
 1108, 41,  103, 0, 2.5,   +1, 1.020350e+02, 0,_
 1109, 41,  104, 0, 1,     +1, 1.030300e+02, 0,_
 1110, 41,  104, 1,-77.777,+0, 1.030300e+02, 0.22,_
 1111, 41,  105, 0, 2.5,   +1, 1.040230e+02, 0,_
 1112, 41,  106, 0, 2,     +1, 1.050180e+02, 0,_
 1113, 41,  107, 0, 2.5,   +1, 1.060120e+02, 0,_
 1114, 41,  108, 0, 2,     +1, 1.070080e+02, 0,_
 1115, 41,  109, 0, 2.5,   +1, 1.080020e+02, 0,_
 1116, 41,  110, 0, 2,     +1, 1.089980e+02, 0,_
 1117, 41,  111, 0, 2.5,   +1, 1.099930e+02, 0,_
 1118, 41,  112, 0, 2,     +1, 1.109890e+02, 0,_
 1119, 41,  113, 0, 2.5,   +1, 1.119840e+02, 0,_
 1120, 42,  83,  0, 1.5,   -1, 8.223620e+01, 0,_
 1121, 42,  84,  0, 0,     +1, 8.321900e+01, 0,_
 1122, 42,  85,  0, 0.5,   -1, 8.420690e+01, 0,_
 1123, 42,  86,  0, 0,     +1, 8.519250e+01, 0,_
 1124, 42,  87,  0, 3.5,   +1, 8.618060e+01, 0,_
 1125, 42,  88,  0, 0,     +1, 8.716670e+01, 0,_
 1126, 42,  89,  0, 4.5,   +1, 8.815560e+01, 0,_
 1127, 42,  89,  1, 0.5,   -1, 8.815560e+01, 0.3875,_
 1128, 42,  90,  0, 0,     +1, 8.914150e+01, 0,_
 1129, 42,  91,  0, 4.5,   +1, 9.013080e+01, 0,_
 1130, 42,  91,  1, 0.5,   -1, 9.013080e+01, 0.6529,_
 1131, 42,  92,  0, 0,     +1, 9.111730e+01, 0,_
 1132, 42,  93,  0, 2.5,   +1, 9.210870e+01, 0,_
 1133, 42,  93,  1, 10.5,  +1, 9.210870e+01, 2.42493,_
 1134, 42,  94,  0, 0,     +1, 9.309840e+01, 0,_
 1135, 42,  95,  0, 2.5,   +1, 9.409060e+01, 0,_
 1136, 42,  96,  0, 0,     +1, 9.508080e+01, 0,_
 1137, 42,  97,  0, 2.5,   +1, 9.607360e+01, 0,_
 1138, 42,  98,  0, 0,     +1, 9.706430e+01, 0,_
 1139, 42,  99,  0, 0.5,   +1, 9.805800e+01, 0,_
 1140, 42,  100, 0, 0,     +1, 9.904920e+01, 0,_
 1141, 42,  101, 0, 0.5,   +1, 1.000430e+02, 0,_
 1142, 42,  102, 0, 0,     +1, 1.010350e+02, 0,_
 1143, 42,  103, 0, 1.5,   +1, 1.020290e+02, 0,_
 1144, 42,  104, 0, 0,     +1, 1.030210e+02, 0,_
 1145, 42,  105, 0, 2.5,   -1, 1.040160e+02, 0,_
 1146, 42,  106, 0, 0,     +1, 1.050080e+02, 0,_
 1147, 42,  107, 0, 3.5,   -1, 1.060030e+02, 0,_
 1148, 42,  108, 0, 0,     +1, 1.069960e+02, 0,_
 1149, 42,  109, 0, 2.5,   +1, 1.079920e+02, 0,_
 1150, 42,  110, 0, 0,     +1, 1.089850e+02, 0,_
 1151, 42,  111, 0, 2.5,   +1, 1.099820e+02, 0,_
 1152, 42,  112, 0, 0,     +1, 1.109760e+02, 0,_
 1153, 42,  113, 0,-77.777,+0, 1.119720e+02, 0,_
 1154, 42,  114, 0, 0,     +1, 1.129660e+02, 0,_
 1155, 42,  115, 0,-77.777,+0, 1.139630e+02, 0,_
 1156, 43,  85,  0, 0.5,   -1, 8.421910e+01, 0,_
 1157, 43,  86,  0, 0,     +1, 8.520460e+01, 0,_
 1158, 43,  87,  0, 0.5,   -1, 8.618970e+01, 0,_
 1159, 43,  87,  1, 4.5,   +1, 8.618970e+01, 0.02,_
 1160, 43,  88,  0,-77.777,+0, 8.717730e+01, 0,_
 1161, 43,  88,  1,-77.777,+0, 8.717730e+01, 0,_
 1162, 43,  89,  0, 4.5,   +1, 8.816320e+01, 0,_
 1163, 43,  89,  1, 0.5,   -1, 8.816320e+01, 0.0626,_
 1164, 43,  90,  0, 1,     +1, 8.915110e+01, 0,_
 1165, 43,  90,  1, 8,     +1, 8.915110e+01, 0.31,_
 1166, 43,  91,  0, 4.5,   +1, 9.013740e+01, 0,_
 1167, 43,  91,  1, 0.5,   -1, 9.013740e+01, 0.1393,_
 1168, 43,  92,  0, 8,     +1, 9.112570e+01, 0,_
 1169, 43,  93,  0, 4.5,   +1, 9.211210e+01, 0,_
 1170, 43,  93,  1, 0.5,   -1, 9.211210e+01, 0.39183,_
 1171, 43,  94,  0, 7,     +1, 9.310290e+01, 0,_
 1172, 43,  94,  1, 2,     +1, 9.310290e+01, 0.0755,_
 1173, 43,  95,  0, 4.5,   +1, 9.409230e+01, 0,_
 1174, 43,  95,  1, 0.5,   -1, 9.409230e+01, 0.0389,_
 1175, 43,  96,  0, 7,     +1, 9.508400e+01, 0,_
 1176, 43,  96,  1, 4,     +1, 9.508400e+01, 0.03428,_
 1177, 43,  97,  0, 4.5,   +1, 9.607390e+01, 0,_
 1178, 43,  97,  1, 0.5,   -1, 9.607390e+01, 0.0966,_
 1179, 43,  98,  0, 6,     +1, 9.706610e+01, 0,_
 1180, 43,  99,  0, 4.5,   +1, 9.805660e+01, 0,_
 1181, 43,  99,  1, 0.5,   -1, 9.805660e+01, 0.142683,_
 1182, 43,  100, 0, 1,     +1, 9.904940e+01, 0,_
 1183, 43,  101, 0, 4.5,   +1, 1.000400e+02, 0,_
 1184, 43,  102, 0, 1,     +1, 1.010340e+02, 0,_
 1185, 43,  102, 1,-77.777,+0, 1.010340e+02, 0.02,_
 1186, 43,  103, 0, 2.5,   +1, 1.020250e+02, 0,_
 1187, 43,  104, 0, 3,     +1, 1.030190e+02, 0,_
 1188, 43,  105, 0, 1.5,   -1, 1.040100e+02, 0,_
 1189, 43,  106, 0,-77.777,+0, 1.050040e+02, 0,_
 1190, 43,  107, 0, 1.5,   -1, 1.059970e+02, 0,_
 1191, 43,  108, 0, 2,     +1, 1.069910e+02, 0,_
 1192, 43,  109, 0, 1.5,   -1, 1.079840e+02, 0,_
 1193, 43,  110, 0, 2,     +1, 1.089800e+02, 0,_
 1194, 43,  111, 0, 1.5,   -1, 1.099730e+02, 0,_
 1195, 43,  112, 0, 2,     +1, 1.109680e+02, 0,_
 1196, 43,  113, 0, 2.5,   +1, 1.119620e+02, 0,_
 1197, 43,  114, 0, 2,     +1, 1.129580e+02, 0,_
 1198, 43,  115, 0, 2.5,   +1, 1.139520e+02, 0,_
 1199, 43,  116, 0, 2,     +1, 1.149480e+02, 0,_
 1200, 43,  117, 0, 1.5,   -1, 1.159420e+02, 0,_
 1201, 43,  118, 0, 2,     +1, 1.169380e+02, 0,_
 1202, 44,  87,  0, 0.5,   -1, 8.620220e+01, 0,_
 1203, 44,  88,  0, 0,     +1, 8.718480e+01, 0,_
 1204, 44,  89,  0, 3.5,   +1, 8.817210e+01, 0,_
 1205, 44,  90,  0, 0,     +1, 8.915730e+01, 0,_
 1206, 44,  91,  0, 4.5,   +1, 9.014520e+01, 0,_
 1207, 44,  91,  1, 0.5,   -1, 9.014520e+01, 0.4,_
 1208, 44,  92,  0, 0,     +1, 9.113050e+01, 0,_
 1209, 44,  93,  0, 4.5,   +1, 9.211880e+01, 0,_
 1210, 44,  93,  1, 0.5,   -1, 9.211880e+01, 0.7344,_
 1211, 44,  94,  0, 0,     +1, 9.310460e+01, 0,_
 1212, 44,  95,  0, 2.5,   +1, 9.409510e+01, 0,_
 1213, 44,  96,  0, 0,     +1, 9.508370e+01, 0,_
 1214, 44,  97,  0, 2.5,   +1, 9.607510e+01, 0,_
 1215, 44,  98,  0, 0,     +1, 9.706420e+01, 0,_
 1216, 44,  99,  0, 2.5,   +1, 9.805630e+01, 0,_
 1217, 44,  100, 0, 0,     +1, 9.904600e+01, 0,_
 1218, 44,  101, 0, 2.5,   +1, 1.000390e+02, 0,_
 1219, 44,  102, 0, 0,     +1, 1.010290e+02, 0,_
 1220, 44,  103, 0, 1.5,   +1, 1.020220e+02, 0,_
 1221, 44,  103, 1, 5.5,   -1, 1.020220e+02, 0.238,_
 1222, 44,  104, 0, 0,     +1, 1.030130e+02, 0,_
 1223, 44,  105, 0, 1.5,   +1, 1.040070e+02, 0,_
 1224, 44,  106, 0, 0,     +1, 1.049980e+02, 0,_
 1225, 44,  107, 0, 2.5,   +1, 1.059910e+02, 0,_
 1226, 44,  108, 0, 0,     +1, 1.069830e+02, 0,_
 1227, 44,  109, 0, 2.5,   +1, 1.079780e+02, 0,_
 1228, 44,  110, 0, 0,     +1, 1.089700e+02, 0,_
 1229, 44,  111, 0, 2.5,   +1, 1.099650e+02, 0,_
 1230, 44,  112, 0, 0,     +1, 1.109580e+02, 0,_
 1231, 44,  113, 0, 2.5,   +1, 1.119520e+02, 0,_
 1232, 44,  113, 1, 5.5,   -1, 1.119520e+02, 0.13,_
 1233, 44,  114, 0, 0,     +1, 1.129460e+02, 0,_
 1234, 44,  115, 0, 2.5,   +1, 1.139410e+02, 0,_
 1235, 44,  116, 0, 0,     +1, 1.149350e+02, 0,_
 1236, 44,  117, 0, 2.5,   +1, 1.159310e+02, 0,_
 1237, 44,  118, 0, 0,     +1, 1.169240e+02, 0,_
 1238, 44,  119, 0, 2.5,   +1, 1.179200e+02, 0,_
 1239, 44,  120, 0, 0,     +1, 1.189150e+02, 0,_
 1240, 45,  89,  0, 3.5,   +1, 8.818470e+01, 0,_
 1241, 45,  90,  0, 0,     +1, 8.917020e+01, 0,_
 1242, 45,  90,  1, 9,     +1, 8.917020e+01, 0,_
 1243, 45,  91,  0, 3.5,   +1, 9.015540e+01, 0,_
 1244, 45,  92,  0, 6,     +1, 9.114220e+01, 0,_
 1245, 45,  93,  0, 4.5,   +1, 9.212750e+01, 0,_
 1246, 45,  94,  0, 3,     +1, 9.311490e+01, 0,_
 1247, 45,  94,  1, 8,     +1, 9.311490e+01, 0.3,_
 1248, 45,  95,  0, 4.5,   +1, 9.410050e+01, 0,_
 1249, 45,  95,  1, 0.5,   -1, 9.410050e+01, 0.5433,_
 1250, 45,  96,  0, 6,     +1, 9.509050e+01, 0,_
 1251, 45,  96,  1, 3,     +1, 9.509050e+01, 0.052,_
 1252, 45,  97,  0, 4.5,   +1, 9.607880e+01, 0,_
 1253, 45,  97,  1, 0.5,   -1, 9.607880e+01, 0.25876,_
 1254, 45,  98,  0, 2,     +1, 9.706960e+01, 0,_
 1255, 45,  98,  1, 5,     +1, 9.706960e+01, 0.06,_
 1256, 45,  99,  0, 0.5,   -1, 9.805850e+01, 0,_
 1257, 45,  99,  1, 4.5,   +1, 9.805850e+01, 0.0643,_
 1258, 45,  100, 0, 1,     -1, 9.904990e+01, 0,_
 1259, 45,  100, 1, 5,     +1, 9.904990e+01, 0.1076,_
 1260, 45,  101, 0, 0.5,   -1, 1.000400e+02, 0,_
 1261, 45,  101, 1, 4.5,   +1, 1.000400e+02, 0.15732,_
 1262, 45,  102, 0, 6,     +1, 1.010310e+02, 0,_
 1263, 45,  102, 1, 2,     -1, 1.010310e+02, 0.07,_
 1264, 45,  103, 0, 0.5,   -1, 1.020210e+02, 0,_
 1265, 45,  103, 1, 3.5,   +1, 1.020210e+02, 0.039756,_
 1266, 45,  104, 0, 1,     +1, 1.030140e+02, 0,_
 1267, 45,  104, 1, 5,     +1, 1.030140e+02, 0.128956,_
 1268, 45,  105, 0, 3.5,   +1, 1.040040e+02, 0,_
 1269, 45,  105, 1, 0.5,   -1, 1.040040e+02, 0.12957,_
 1270, 45,  106, 0, 1,     +1, 1.049970e+02, 0,_
 1271, 45,  106, 1, 6,     +1, 1.049980e+02, 0.137,_
 1272, 45,  107, 0, 3.5,   +1, 1.059880e+02, 0,_
 1273, 45,  108, 0, 1,     +1, 1.069820e+02, 0,_
 1274, 45,  108, 1, 5,     +1, 1.069820e+02, 0,_
 1275, 45,  109, 0, 3.5,   +1, 1.079730e+02, 0,_
 1276, 45,  110, 0, 4,     +1, 1.089670e+02, 0,_
 1277, 45,  110, 1, 1,     +1, 1.089670e+02, 0.1,_
 1278, 45,  111, 0, 3.5,   +1, 1.099590e+02, 0,_
 1279, 45,  112, 0, 1,     +1, 1.109530e+02, 0,_
 1280, 45,  112, 1,-77.777,+0, 1.109530e+02, 0.36,_
 1281, 45,  113, 0, 3.5,   +1, 1.119460e+02, 0,_
 1282, 45,  114, 0, 1,     +1, 1.129400e+02, 0,_
 1283, 45,  114, 1,-77.777,+0, 1.129400e+02, 0.2,_
 1284, 45,  115, 0, 3.5,   +1, 1.139330e+02, 0,_
 1285, 45,  116, 0, 1,     +1, 1.149280e+02, 0,_
 1286, 45,  116, 1, 6,     -1, 1.149280e+02, 0.15,_
 1287, 45,  117, 0, 3.5,   +1, 1.159220e+02, 0,_
 1288, 45,  118, 0, 1,     +1, 1.169170e+02, 0,_
 1289, 45,  119, 0, 3.5,   +1, 1.179100e+02, 0,_
 1290, 45,  120, 0, 3.5,   +1, 1.189060e+02, 0,_
 1291, 45,  121, 0, 3.5,   +1, 1.198990e+02, 0,_
 1292, 45,  122, 0,-77.777,+0, 1.208960e+02, 0,_
 1293, 46,  91,  0, 3.5,   +1, 9.016780e+01, 0,_
 1294, 46,  92,  0, 0,     +1, 9.115060e+01, 0,_
 1295, 46,  93,  0, 4.5,   +1, 9.213750e+01, 0,_
 1296, 46,  94,  0, 0,     +1, 9.312190e+01, 0,_
 1297, 46,  95,  0, 4.5,   +1, 9.410920e+01, 0,_
 1298, 46,  95,  1, 10.5,  +1, 9.410920e+01, 1.87,_
 1299, 46,  96,  0, 0,     +1, 9.509420e+01, 0,_
 1300, 46,  97,  0, 2.5,   +1, 9.608390e+01, 0,_
 1301, 46,  98,  0, 0,     +1, 9.707160e+01, 0,_
 1302, 46,  99,  0, 2.5,   +1, 9.806210e+01, 0,_
 1303, 46,  100, 0, 0,     +1, 9.905020e+01, 0,_
 1304, 46,  101, 0, 2.5,   +1, 1.000410e+02, 0,_
 1305, 46,  102, 0, 0,     +1, 1.010300e+02, 0,_
 1306, 46,  103, 0, 2.5,   +1, 1.020220e+02, 0,_
 1307, 46,  104, 0, 0,     +1, 1.030110e+02, 0,_
 1308, 46,  105, 0, 2.5,   +1, 1.040040e+02, 0,_
 1309, 46,  106, 0, 0,     +1, 1.049940e+02, 0,_
 1310, 46,  107, 0, 2.5,   +1, 1.059870e+02, 0,_
 1311, 46,  107, 1, 5.5,   -1, 1.059870e+02, 0.2149,_
 1312, 46,  108, 0, 0,     +1, 1.069770e+02, 0,_
 1313, 46,  109, 0, 2.5,   +1, 1.079700e+02, 0,_
 1314, 46,  109, 1, 5.5,   -1, 1.079700e+02, 0.1889,_
 1315, 46,  110, 0, 0,     +1, 1.089610e+02, 0,_
 1316, 46,  111, 0, 2.5,   +1, 1.099550e+02, 0,_
 1317, 46,  111, 1, 5.5,   -1, 1.099550e+02, 0.17218,_
 1318, 46,  112, 0, 0,     +1, 1.109460e+02, 0,_
 1319, 46,  113, 0, 2.5,   +1, 1.119410e+02, 0,_
 1320, 46,  113, 1, 4.5,   -1, 1.119410e+02, 0.0811,_
 1321, 46,  114, 0, 0,     +1, 1.129320e+02, 0,_
 1322, 46,  115, 0, 2.5,   +1, 1.139270e+02, 0,_
 1323, 46,  115, 1, 5.5,   -1, 1.139270e+02, 0.08918,_
 1324, 46,  116, 0, 0,     +1, 1.149180e+02, 0,_
 1325, 46,  117, 0, 2.5,   +1, 1.159130e+02, 0,_
 1326, 46,  117, 1, 5.5,   -1, 1.159130e+02, 0.2032,_
 1327, 46,  118, 0, 0,     +1, 1.169060e+02, 0,_
 1328, 46,  119, 0,-77.777,+0, 1.179020e+02, 0,_
 1329, 46,  120, 0, 0,     +1, 1.188940e+02, 0,_
 1330, 46,  121, 0, 2.5,   +1, 1.198890e+02, 0,_
 1331, 46,  122, 0, 0,     +1, 1.208830e+02, 0,_
 1332, 46,  123, 0,-77.777,+0, 1.218790e+02, 0,_
 1333, 46,  124, 0, 0,     +1, 1.228720e+02, 0,_
 1334, 47,  93,  0, 4.5,   +1, 9.215130e+01, 0,_
 1335, 47,  94,  0, 0,     +1, 9.313580e+01, 0,_
 1336, 47,  94,  1, 7,     +1, 9.313580e+01, 1.35,_
 1337, 47,  94,  2, 21,    +1, 9.313580e+01, 6.5,_
 1338, 47,  95,  0, 4.5,   +1, 9.411990e+01, 0,_
 1339, 47,  95,  1, 0.5,   -1, 9.411990e+01, 0.3442,_
 1340, 47,  95,  2, 11.5,  +1, 9.411990e+01, 2.531,_
 1341, 47,  95,  3, 18.5,  +1, 9.411990e+01, 4.859,_
 1342, 47,  96,  0,-77.777,+0, 9.510660e+01, 0,_
 1343, 47,  96,  1, 2,     +1, 9.510660e+01, 0,_
 1344, 47,  97,  0, 4.5,   +1, 9.609130e+01, 0,_
 1345, 47,  98,  0, 6,     +1, 9.708040e+01, 0,_
 1346, 47,  99,  0, 4.5,   +1, 9.806780e+01, 0,_
 1347, 47,  99,  1, 0.5,   -1, 9.806780e+01, 0.5061,_
 1348, 47,  100, 0, 5,     +1, 9.905780e+01, 0,_
 1349, 47,  100, 1, 2,     +1, 9.905780e+01, 0.01552,_
 1350, 47,  101, 0, 4.5,   +1, 1.000460e+02, 0,_
 1351, 47,  101, 1, 0.5,   -1, 1.000460e+02, 0.2741,_
 1352, 47,  102, 0, 5,     +1, 1.010360e+02, 0,_
 1353, 47,  102, 1, 2,     +1, 1.010360e+02, 0.0093,_
 1354, 47,  103, 0, 3.5,   +1, 1.020250e+02, 0,_
 1355, 47,  103, 1, 0.5,   -1, 1.020250e+02, 0.13445,_
 1356, 47,  104, 0, 5,     +1, 1.030160e+02, 0,_
 1357, 47,  104, 1, 2,     +1, 1.030160e+02, 0.0069,_
 1358, 47,  105, 0, 0.5,   -1, 1.040050e+02, 0,_
 1359, 47,  105, 1, 3.5,   +1, 1.040050e+02, 0.025465,_
 1360, 47,  106, 0, 1,     +1, 1.049970e+02, 0,_
 1361, 47,  106, 1, 6,     +1, 1.049970e+02, 0.08966,_
 1362, 47,  107, 0, 0.5,   -1, 1.059870e+02, 0,_
 1363, 47,  107, 1, 3.5,   +1, 1.059870e+02, 0.093124,_
 1364, 47,  108, 0, 1,     +1, 1.069790e+02, 0,_
 1365, 47,  108, 1, 6,     +1, 1.069790e+02, 0.109472,_
 1366, 47,  109, 0, 0.5,   -1, 1.079690e+02, 0,_
 1367, 47,  109, 1, 3.5,   +1, 1.079690e+02, 0.088034,_
 1368, 47,  110, 0, 1,     +1, 1.089620e+02, 0,_
 1369, 47,  110, 1, 6,     +1, 1.089620e+02, 0.11759,_
 1370, 47,  111, 0, 0.5,   -1, 1.099530e+02, 0,_
 1371, 47,  111, 1, 3.5,   +1, 1.099530e+02, 0.05982,_
 1372, 47,  112, 0, 2,     -1, 1.109460e+02, 0,_
 1373, 47,  113, 0, 0.5,   -1, 1.119370e+02, 0,_
 1374, 47,  113, 1, 3.5,   +1, 1.119370e+02, 0.0432,_
 1375, 47,  114, 0, 1,     +1, 1.129300e+02, 0,_
 1376, 47,  114, 1, 4,     +1, 1.129300e+02, 0.199,_
 1377, 47,  115, 0, 0.5,   -1, 1.139220e+02, 0,_
 1378, 47,  115, 1, 3.5,   +1, 1.139220e+02, 0.0411,_
 1379, 47,  116, 0, 2,     -1, 1.149160e+02, 0,_
 1380, 47,  116, 1, 5,     +1, 1.149160e+02, 0.0819,_
 1381, 47,  117, 0, 0.5,   -1, 1.159070e+02, 0,_
 1382, 47,  117, 1, 3.5,   +1, 1.159070e+02, 0.0286,_
 1383, 47,  118, 0, 1,     -1, 1.169020e+02, 0,_
 1384, 47,  118, 1, 4,     +1, 1.169020e+02, 0.12749,_
 1385, 47,  119, 0, 0.5,   -1, 1.178940e+02, 0,_
 1386, 47,  119, 1, 3.5,   +1, 1.178940e+02, 0.02,_
 1387, 47,  120, 0, 3,     +1, 1.188890e+02, 0,_
 1388, 47,  120, 1, 6,     -1, 1.188890e+02, 0.203,_
 1389, 47,  121, 0, 3.5,   +1, 1.198810e+02, 0,_
 1390, 47,  122, 0, 3,     +1, 1.208760e+02, 0,_
 1391, 47,  122, 1, 8,     -1, 1.208760e+02, 0.08,_
 1392, 47,  123, 0, 3.5,   +1, 1.218690e+02, 0,_
 1393, 47,  124, 0, 3,     +1, 1.228640e+02, 0,_
 1394, 47,  124, 1, 8,     -1, 1.228640e+02, 0,_
 1395, 47,  125, 0, 3.5,   +1, 1.238570e+02, 0,_
 1396, 47,  126, 0, 3,     +1, 1.248530e+02, 0,_
 1397, 47,  127, 0, 3.5,   +1, 1.258460e+02, 0,_
 1398, 47,  128, 0,-77.777,+0, 1.268420e+02, 0,_
 1399, 47,  129, 0, 3.5,   +1, 1.278360e+02, 0,_
 1400, 47,  130, 0,-77.777,+0, 1.288340e+02, 0,_
 1401, 48,  95,  0, 4.5,   +1, 9.413420e+01, 0,_
 1402, 48,  96,  0, 0,     +1, 9.511560e+01, 0,_
 1403, 48,  97,  0, 4.5,   +1, 9.610220e+01, 0,_
 1404, 48,  98,  0, 0,     +1, 9.708610e+01, 0,_
 1405, 48,  99,  0, 2.5,   +1, 9.807520e+01, 0,_
 1406, 48,  100, 0, 0,     +1, 9.906190e+01, 0,_
 1407, 48,  101, 0, 2.5,   +1, 1.000520e+02, 0,_
 1408, 48,  102, 0, 0,     +1, 1.010390e+02, 0,_
 1409, 48,  103, 0, 2.5,   +1, 1.020290e+02, 0,_
 1410, 48,  104, 0, 0,     +1, 1.030170e+02, 0,_
 1411, 48,  105, 0, 2.5,   +1, 1.040080e+02, 0,_
 1412, 48,  106, 0, 0,     +1, 1.049970e+02, 0,_
 1413, 48,  107, 0, 2.5,   +1, 1.059880e+02, 0,_
 1414, 48,  108, 0, 0,     +1, 1.069770e+02, 0,_
 1415, 48,  109, 0, 2.5,   +1, 1.079690e+02, 0,_
 1416, 48,  110, 0, 0,     +1, 1.089590e+02, 0,_
 1417, 48,  111, 0, 0.5,   +1, 1.099510e+02, 0,_
 1418, 48,  111, 1, 5.5,   -1, 1.099510e+02, 0.39622,_
 1419, 48,  112, 0, 0,     +1, 1.109410e+02, 0,_
 1420, 48,  113, 0, 0.5,   +1, 1.119340e+02, 0,_
 1421, 48,  113, 1, 5.5,   -1, 1.119340e+02, 0.2637,_
 1422, 48,  114, 0, 0,     +1, 1.129250e+02, 0,_
 1423, 48,  115, 0, 0.5,   +1, 1.139180e+02, 0,_
 1424, 48,  115, 1, 5.5,   -1, 1.139180e+02, 0.181,_
 1425, 48,  116, 0, 0,     +1, 1.149090e+02, 0,_
 1426, 48,  117, 0, 0.5,   +1, 1.159030e+02, 0,_
 1427, 48,  117, 1, 5.5,   -1, 1.159030e+02, 0.1364,_
 1428, 48,  118, 0, 0,     +1, 1.168940e+02, 0,_
 1429, 48,  119, 0, 1.5,   +1, 1.178880e+02, 0,_
 1430, 48,  119, 1, 5.5,   -1, 1.178880e+02, 0.14654,_
 1431, 48,  120, 0, 0,     +1, 1.188800e+02, 0,_
 1432, 48,  121, 0, 1.5,   +1, 1.198740e+02, 0,_
 1433, 48,  121, 1, 5.5,   -1, 1.198740e+02, 0.21489,_
 1434, 48,  122, 0, 0,     +1, 1.208660e+02, 0,_
 1435, 48,  123, 0, 1.5,   +1, 1.218610e+02, 0,_
 1436, 48,  123, 1, 5.5,   -1, 1.218610e+02, 0.316522,_
 1437, 48,  124, 0, 0,     +1, 1.228530e+02, 0,_
 1438, 48,  125, 0, 1.5,   +1, 1.238480e+02, 0,_
 1439, 48,  125, 1, 5.5,   -1, 1.238480e+02, 0.05,_
 1440, 48,  126, 0, 0,     +1, 1.248410e+02, 0,_
 1441, 48,  127, 0, 1.5,   +1, 1.258360e+02, 0,_
 1442, 48,  128, 0, 0,     +1, 1.268290e+02, 0,_
 1443, 48,  129, 0, 1.5,   +1, 1.278250e+02, 0,_
 1444, 48,  129, 1, 5.5,   -1, 1.278250e+02, 0,_
 1445, 48,  130, 0, 0,     +1, 1.288180e+02, 0,_
 1446, 48,  131, 0, 3.5,   -1, 1.298160e+02, 0,_
 1447, 48,  132, 0, 0,     +1, 1.308120e+02, 0,_
 1448, 49,  97,  0, 4.5,   +1, 9.611670e+01, 0,_
 1449, 49,  98,  0, 0,     +1, 9.710080e+01, 0,_
 1450, 49,  98,  1,-77.777,+0, 9.710080e+01, 0,_
 1451, 49,  99,  0, 4.5,   +1, 9.808430e+01, 0,_
 1452, 49,  99,  1, 0.5,   -1, 9.808430e+01, 0.4,_
 1453, 49,  100, 0, 6,     +1, 9.907270e+01, 0,_
 1454, 49,  101, 0, 4.5,   +1, 1.000590e+02, 0,_
 1455, 49,  101, 1, 0.5,   -1, 1.000590e+02, 0.55,_
 1456, 49,  102, 0, 6,     +1, 1.010490e+02, 0,_
 1457, 49,  103, 0, 4.5,   +1, 1.020360e+02, 0,_
 1458, 49,  103, 1, 0.5,   -1, 1.020360e+02, 0.6317,_
 1459, 49,  104, 0,-77.777,+0, 1.030260e+02, 0,_
 1460, 49,  104, 1, 3,     +1, 1.030260e+02, 0.09348,_
 1461, 49,  105, 0, 4.5,   +1, 1.040130e+02, 0,_
 1462, 49,  105, 1, 0.5,   -1, 1.040130e+02, 0.6741,_
 1463, 49,  106, 0, 7,     +1, 1.050040e+02, 0,_
 1464, 49,  106, 1, 3,     +1, 1.050040e+02, 0.0286,_
 1465, 49,  107, 0, 4.5,   +1, 1.059920e+02, 0,_
 1466, 49,  107, 1, 0.5,   -1, 1.059920e+02, 0.6785,_
 1467, 49,  108, 0, 7,     +1, 1.069830e+02, 0,_
 1468, 49,  108, 1, 2,     +1, 1.069830e+02, 0.02975,_
 1469, 49,  109, 0, 4.5,   +1, 1.079720e+02, 0,_
 1470, 49,  109, 1, 0.5,   -1, 1.079720e+02, 0.6501,_
 1471, 49,  109, 2, 9.5,   +1, 1.079720e+02, 2.101,_
 1472, 49,  110, 0, 7,     +1, 1.089630e+02, 0,_
 1473, 49,  110, 1, 2,     +1, 1.089630e+02, 0.0621,_
 1474, 49,  111, 0, 4.5,   +1, 1.099520e+02, 0,_
 1475, 49,  111, 1, 0.5,   -1, 1.099520e+02, 0.5372,_
 1476, 49,  112, 0, 1,     +1, 1.109440e+02, 0,_
 1477, 49,  112, 1, 4,     +1, 1.109440e+02, 0.15658,_
 1478, 49,  113, 0, 4.5,   +1, 1.119340e+02, 0,_
 1479, 49,  113, 1, 0.5,   -1, 1.119340e+02, 0.391699,_
 1480, 49,  114, 0, 1,     +1, 1.129260e+02, 0,_
 1481, 49,  114, 1, 5,     +1, 1.129260e+02, 0.19034,_
 1482, 49,  114, 2, 8,     -1, 1.129260e+02, 0.50198,_
 1483, 49,  115, 0, 4.5,   +1, 1.139170e+02, 0,_
 1484, 49,  115, 1, 0.5,   -1, 1.139170e+02, 0.326241,_
 1485, 49,  116, 0, 1,     +1, 1.149100e+02, 0,_
 1486, 49,  116, 1, 5,     +1, 1.149100e+02, 0.127267,_
 1487, 49,  116, 2, 8,     -1, 1.149100e+02, 0.28966,_
 1488, 49,  117, 0, 4.5,   +1, 1.159000e+02, 0,_
 1489, 49,  117, 1, 0.5,   -1, 1.159000e+02, 0.3153,_
 1490, 49,  118, 0, 1,     +1, 1.168930e+02, 0,_
 1491, 49,  118, 1, 5,     +1, 1.168930e+02, 0.1,_
 1492, 49,  118, 2, 8,     -1, 1.168930e+02, 0.24,_
 1493, 49,  119, 0, 4.5,   +1, 1.178840e+02, 0,_
 1494, 49,  119, 1, 0.5,   -1, 1.178840e+02, 0.31137,_
 1495, 49,  120, 0, 1,     +1, 1.188780e+02, 0,_
 1496, 49,  120, 1, 5,     +1, 1.188780e+02, 0.07,_
 1497, 49,  120, 2, 8,     -1, 1.188780e+02, 0.3,_
 1498, 49,  121, 0, 4.5,   +1, 1.198690e+02, 0,_
 1499, 49,  121, 1, 0.5,   -1, 1.198690e+02, 0.3136,_
 1500, 49,  122, 0, 1,     +1, 1.208630e+02, 0,_
 1501, 49,  122, 1, 8,     -1, 1.208630e+02, 0.2,_
 1502, 49,  122, 2, 8,     -1, 1.208630e+02, 0.2,_
 1503, 49,  123, 0, 4.5,   +1, 1.218550e+02, 0,_
 1504, 49,  123, 1, 0.5,   -1, 1.218550e+02, 0.32721,_
 1505, 49,  124, 0, 3,     +1, 1.228490e+02, 0,_
 1506, 49,  124, 1, 8,     -1, 1.228490e+02, 0.05,_
 1507, 49,  125, 0, 4.5,   +1, 1.238410e+02, 0,_
 1508, 49,  125, 1, 0.5,   -1, 1.238410e+02, 0.36012,_
 1509, 49,  126, 0, 3,     +1, 1.248350e+02, 0,_
 1510, 49,  126, 1, 8,     -1, 1.248350e+02, 0.1,_
 1511, 49,  127, 0, 4.5,   +1, 1.258270e+02, 0,_
 1512, 49,  127, 1, 0.5,   -1, 1.258270e+02, 0.46,_
 1513, 49,  128, 0, 3,     +1, 1.268210e+02, 0,_
 1514, 49,  128, 1, 1,     -1, 1.268210e+02, 0.24787,_
 1515, 49,  128, 2, 8,     -1, 1.268210e+02, 0.32,_
 1516, 49,  129, 0, 4.5,   +1, 1.278140e+02, 0,_
 1517, 49,  129, 1, 0.5,   -1, 1.278140e+02, 0.38,_
 1518, 49,  130, 0, 1,     -1, 1.288090e+02, 0,_
 1519, 49,  130, 1, 10,    -1, 1.288090e+02, 0.05,_
 1520, 49,  130, 2, 5,     +1, 1.288090e+02, 0.4,_
 1521, 49,  131, 0, 4.5,   +1, 1.298020e+02, 0,_
 1522, 49,  131, 1, 0.5,   -1, 1.298020e+02, 0.35,_
 1523, 49,  131, 2,-77.777,+0, 1.298020e+02, 4.1,_
 1524, 49,  132, 0, 7,     -1, 1.308000e+02, 0,_
 1525, 49,  133, 0, 4.5,   +1, 1.317960e+02, 0,_
 1526, 49,  133, 1, 0.5,   -1, 1.317960e+02, 0.33,_
 1527, 49,  134, 0,-77.777,+0, 1.327940e+02, 0,_
 1528, 49,  135, 0, 4.5,   +1, 1.337900e+02, 0,_
 1529, 50,  99,  0, 4.5,   +1, 9.809930e+01, 0,_
 1530, 50,  100, 0, 0,     +1, 9.908050e+01, 0,_
 1531, 50,  101, 0, 2.5,   +1, 1.000690e+02, 0,_
 1532, 50,  102, 0, 0,     +1, 1.010550e+02, 0,_
 1533, 50,  103, 0, 2.5,   +1, 1.020440e+02, 0,_
 1534, 50,  104, 0, 0,     +1, 1.030300e+02, 0,_
 1535, 50,  105, 0, 2.5,   +1, 1.040200e+02, 0,_
 1536, 50,  106, 0, 0,     +1, 1.050070e+02, 0,_
 1537, 50,  107, 0, 2.5,   +1, 1.059970e+02, 0,_
 1538, 50,  108, 0, 0,     +1, 1.069850e+02, 0,_
 1539, 50,  109, 0, 2.5,   +1, 1.079760e+02, 0,_
 1540, 50,  110, 0, 0,     +1, 1.089640e+02, 0,_
 1541, 50,  111, 0, 3.5,   +1, 1.099550e+02, 0,_
 1542, 50,  112, 0, 0,     +1, 1.109440e+02, 0,_
 1543, 50,  113, 0, 0.5,   +1, 1.119350e+02, 0,_
 1544, 50,  113, 1, 3.5,   +1, 1.119350e+02, 0.07739,_
 1545, 50,  114, 0, 0,     +1, 1.129240e+02, 0,_
 1546, 50,  115, 0, 0.5,   +1, 1.139160e+02, 0,_
 1547, 50,  116, 0, 0,     +1, 1.149060e+02, 0,_
 1548, 50,  117, 0, 0.5,   +1, 1.158990e+02, 0,_
 1549, 50,  117, 1, 5.5,   -1, 1.158990e+02, 0.31458,_
 1550, 50,  118, 0, 0,     +1, 1.168890e+02, 0,_
 1551, 50,  119, 0, 0.5,   +1, 1.178820e+02, 0,_
 1552, 50,  119, 1, 5.5,   -1, 1.178820e+02, 0.08953,_
 1553, 50,  120, 0, 0,     +1, 1.188720e+02, 0,_
 1554, 50,  121, 0, 1.5,   +1, 1.198660e+02, 0,_
 1555, 50,  121, 1, 5.5,   -1, 1.198660e+02, 0.0063,_
 1556, 50,  122, 0, 0,     +1, 1.208560e+02, 0,_
 1557, 50,  123, 0, 5.5,   -1, 1.218500e+02, 0,_
 1558, 50,  123, 1, 1.5,   +1, 1.218500e+02, 0.0246,_
 1559, 50,  124, 0, 0,     +1, 1.228410e+02, 0,_
 1560, 50,  125, 0, 5.5,   -1, 1.238350e+02, 0,_
 1561, 50,  125, 1, 1.5,   +1, 1.238350e+02, 0.0275,_
 1562, 50,  126, 0, 0,     +1, 1.248260e+02, 0,_
 1563, 50,  127, 0, 5.5,   -1, 1.258200e+02, 0,_
 1564, 50,  127, 1, 1.5,   +1, 1.258200e+02, 0.0047,_
 1565, 50,  128, 0, 0,     +1, 1.268120e+02, 0,_
 1566, 50,  128, 1, 7,     -1, 1.268120e+02, 2.0915,_
 1567, 50,  129, 0, 1.5,   +1, 1.278060e+02, 0,_
 1568, 50,  129, 1, 5.5,   -1, 1.278060e+02, 0.0352,_
 1569, 50,  130, 0, 0,     +1, 1.287980e+02, 0,_
 1570, 50,  130, 1, 7,     -1, 1.287980e+02, 1.9469,_
 1571, 50,  131, 0, 1.5,   +1, 1.297920e+02, 0,_
 1572, 50,  131, 1, 5.5,   -1, 1.297920e+02, 0.08,_
 1573, 50,  132, 0, 0,     +1, 1.307850e+02, 0,_
 1574, 50,  133, 0, 3.5,   -1, 1.317820e+02, 0,_
 1575, 50,  134, 0, 0,     +1, 1.327780e+02, 0,_
 1576, 50,  135, 0, 3.5,   -1, 1.337760e+02, 0,_
 1577, 50,  136, 0, 0,     +1, 1.347720e+02, 0,_
 1578, 50,  137, 0, 2.5,   -1, 1.357700e+02, 0,_
 1579, 51,  103, 0, 2.5,   +1, 1.020550e+02, 0,_
 1580, 51,  104, 0,-77.777,+0, 1.030440e+02, 0,_
 1581, 51,  105, 0, 2.5,   +1, 1.040300e+02, 0,_
 1582, 51,  106, 0, 4,     +1, 1.050190e+02, 0,_
 1583, 51,  107, 0, 2.5,   +1, 1.060060e+02, 0,_
 1584, 51,  108, 0, 4,     +1, 1.069950e+02, 0,_
 1585, 51,  109, 0, 2.5,   +1, 1.079820e+02, 0,_
 1586, 51,  110, 0, 4,     +1, 1.089730e+02, 0,_
 1587, 51,  111, 0, 2.5,   +1, 1.099600e+02, 0,_
 1588, 51,  112, 0, 3,     +1, 1.109510e+02, 0,_
 1589, 51,  113, 0, 2.5,   +1, 1.119390e+02, 0,_
 1590, 51,  114, 0, 3,     +1, 1.129310e+02, 0,_
 1591, 51,  115, 0, 2.5,   +1, 1.139190e+02, 0,_
 1592, 51,  116, 0, 3,     +1, 1.149110e+02, 0,_
 1593, 51,  116, 1, 8,     -1, 1.149110e+02, 0.383,_
 1594, 51,  117, 0, 2.5,   +1, 1.159010e+02, 0,_
 1595, 51,  118, 0, 1,     +1, 1.168930e+02, 0,_
 1596, 51,  118, 1, 8,     -1, 1.168930e+02, 0.25,_
 1597, 51,  119, 0, 2.5,   +1, 1.178830e+02, 0,_
 1598, 51,  119, 1, 13.5,  +1, 1.178830e+02, 2.852,_
 1599, 51,  120, 0, 1,     +1, 1.188750e+02, 0,_
 1600, 51,  120, 1, 8,     -1, 1.188750e+02, 0.2,_
 1601, 51,  121, 0, 2.5,   +1, 1.198650e+02, 0,_
 1602, 51,  122, 0, 2,     -1, 1.208580e+02, 0,_
 1603, 51,  122, 1, 8,     -1, 1.208580e+02, 0.16356,_
 1604, 51,  123, 0, 3.5,   +1, 1.218480e+02, 0,_
 1605, 51,  124, 0, 3,     -1, 1.228420e+02, 0,_
 1606, 51,  124, 1, 5,     +1, 1.228420e+02, 0.01086,_
 1607, 51,  124, 2, 8,     -1, 1.228420e+02, 0.03685,_
 1608, 51,  125, 0, 3.5,   +1, 1.238320e+02, 0,_
 1609, 51,  126, 0, 8,     -1, 1.248260e+02, 0,_
 1610, 51,  126, 1, 5,     +1, 1.248260e+02, 0.0177,_
 1611, 51,  126, 2, 3,     -1, 1.248260e+02, 0.0404,_
 1612, 51,  127, 0, 3.5,   +1, 1.258170e+02, 0,_
 1613, 51,  128, 0, 8,     -1, 1.268100e+02, 0,_
 1614, 51,  128, 1, 5,     +1, 1.268100e+02, 0.01,_
 1615, 51,  129, 0, 3.5,   +1, 1.278020e+02, 0,_
 1616, 51,  129, 1, 9.5,   -1, 1.278020e+02, 1.8513,_
 1617, 51,  130, 0, 8,     -1, 1.287960e+02, 0,_
 1618, 51,  130, 1, 4,     +0, 1.287960e+02, 0.0048,_  ' spin is (4,5)+ from NNDC
 1619, 51,  131, 0, 3.5,   +1, 1.297870e+02, 0,_
 1620, 51,  132, 0, 4,     +1, 1.307810e+02, 0,_
 1621, 51,  132, 1, 8,     -1, 1.307810e+02, 0.2,_
 1622, 51,  133, 0, 3.5,   +1, 1.317730e+02, 0,_
 1623, 51,  134, 0, 0,     -1, 1.327700e+02, 0,_
 1624, 51,  134, 1, 7,     -1, 1.327700e+02, 0.08,_
 1625, 51,  135, 0, 3.5,   +1, 1.337660e+02, 0,_
 1626, 51,  136, 0, 1,     -1, 1.347630e+02, 0,_
 1627, 51,  137, 0, 3.5,   +1, 1.357590e+02, 0,_
 1628, 51,  138, 0, 2,     -1, 1.367560e+02, 0,_
 1629, 51,  139, 0, 3.5,   +1, 1.377520e+02, 0,_
 1630, 52,  105, 0, 2.5,   +1, 1.040420e+02, 0,_
 1631, 52,  106, 0, 0,     +1, 1.050270e+02, 0,_
 1632, 52,  107, 0, 2.5,   +1, 1.060160e+02, 0,_
 1633, 52,  108, 0, 0,     +1, 1.070020e+02, 0,_
 1634, 52,  109, 0, 2.5,   +1, 1.079920e+02, 0,_
 1635, 52,  110, 0, 0,     +1, 1.089780e+02, 0,_
 1636, 52,  111, 0, 2.5,   +1, 1.099680e+02, 0,_
 1637, 52,  112, 0, 0,     +1, 1.109560e+02, 0,_
 1638, 52,  113, 0, 3.5,   +1, 1.119460e+02, 0,_
 1639, 52,  114, 0, 0,     +1, 1.129340e+02, 0,_
 1640, 52,  115, 0, 3.5,   +1, 1.139250e+02, 0,_
 1641, 52,  115, 1, 0.5,   +1, 1.139250e+02, 0.02,_
 1642, 52,  116, 0, 0,     +1, 1.149130e+02, 0,_
 1643, 52,  117, 0, 0.5,   +1, 1.159040e+02, 0,_
 1644, 52,  117, 1, 5.5,   -1, 1.159040e+02, 0.2961,_
 1645, 52,  118, 0, 0,     +1, 1.168930e+02, 0,_
 1646, 52,  119, 0, 0.5,   +1, 1.178850e+02, 0,_
 1647, 52,  119, 1, 5.5,   -1, 1.178850e+02, 0.26096,_
 1648, 52,  120, 0, 0,     +1, 1.188740e+02, 0,_
 1649, 52,  121, 0, 0.5,   +1, 1.198660e+02, 0,_
 1650, 52,  121, 1, 5.5,   -1, 1.198660e+02, 0.293991,_
 1651, 52,  122, 0, 0,     +1, 1.208560e+02, 0,_
 1652, 52,  123, 0, 0.5,   +1, 1.218480e+02, 0,_
 1653, 52,  123, 1, 5.5,   -1, 1.218480e+02, 0.2474,_
 1654, 52,  124, 0, 0,     +1, 1.228380e+02, 0,_
 1655, 52,  125, 0, 0.5,   +1, 1.238310e+02, 0,_
 1656, 52,  125, 1, 5.5,   -1, 1.238310e+02, 0.144766,_
 1657, 52,  126, 0, 0,     +1, 1.248220e+02, 0,_
 1658, 52,  127, 0, 1.5,   +1, 1.258150e+02, 0,_
 1659, 52,  127, 1, 5.5,   -1, 1.258150e+02, 0.08826,_
 1660, 52,  128, 0, 0,     +1, 1.268060e+02, 0,_
 1661, 52,  129, 0, 1.5,   +1, 1.277990e+02, 0,_
 1662, 52,  129, 1, 5.5,   -1, 1.277990e+02, 0.1055,_
 1663, 52,  130, 0, 0,     +1, 1.287900e+02, 0,_
 1664, 52,  131, 0, 1.5,   +1, 1.297840e+02, 0,_
 1665, 52,  131, 1, 5.5,   -1, 1.297840e+02, 0.18232,_
 1666, 52,  132, 0, 0,     +1, 1.307750e+02, 0,_
 1667, 52,  133, 0, 1.5,   +1, 1.317690e+02, 0,_
 1668, 52,  133, 1, 5.5,   -1, 1.317690e+02, 0.33427,_
 1669, 52,  134, 0, 0,     +1, 1.327610e+02, 0,_
 1670, 52,  135, 0, 3.5,   -1, 1.337570e+02, 0,_
 1671, 52,  136, 0, 0,     +1, 1.347520e+02, 0,_
 1672, 52,  137, 0, 1.5,   -1, 1.357490e+02, 0,_
 1673, 52,  138, 0, 0,     +1, 1.367440e+02, 0,_
 1674, 52,  139, 0, 2.5,   -1, 1.377410e+02, 0,_
 1675, 52,  140, 0, 0,     +1, 1.387370e+02, 0,_
 1676, 52,  141, 0, 2.5,   -1, 1.397340e+02, 0,_
 1677, 52,  142, 0, 0,     +1, 1.407300e+02, 0,_
 1678, 53,  108, 0, 1,     +1, 1.070160e+02, 0,_
 1679, 53,  109, 0, 2.5,   +1, 1.080020e+02, 0,_
 1680, 53,  110, 0, 1,     +1, 1.089910e+02, 0,_
 1681, 53,  111, 0, 2.5,   +1, 1.099770e+02, 0,_
 1682, 53,  112, 0, 1,     +1, 1.109660e+02, 0,_
 1683, 53,  113, 0, 2.5,   +1, 1.119540e+02, 0,_
 1684, 53,  114, 0, 1,     +1, 1.129430e+02, 0,_
 1685, 53,  114, 1,-77.777,+0, 1.129430e+02, 0.2659,_
 1686, 53,  115, 0, 2.5,   +1, 1.139310e+02, 0,_
 1687, 53,  116, 0, 1,     +1, 1.149210e+02, 0,_
 1688, 53,  117, 0, 2.5,   +1, 1.159090e+02, 0,_
 1689, 53,  118, 0, 2,     -1, 1.169000e+02, 0,_
 1690, 53,  118, 1, 7,     -1, 1.169000e+02, 0.19011,_
 1691, 53,  119, 0, 2.5,   +1, 1.178890e+02, 0,_
 1692, 53,  120, 0, 2,     -1, 1.188800e+02, 0,_
 1693, 53,  120, 1, 7,     -1, 1.188800e+02, 0.32,_
 1694, 53,  121, 0, 2.5,   +1, 1.198690e+02, 0,_
 1695, 53,  122, 0, 1,     +1, 1.208600e+02, 0,_
 1696, 53,  123, 0, 2.5,   +1, 1.218500e+02, 0,_
 1697, 53,  124, 0, 2,     -1, 1.228420e+02, 0,_
 1698, 53,  125, 0, 2.5,   +1, 1.238320e+02, 0,_
 1699, 53,  126, 0, 2,     -1, 1.248240e+02, 0,_
 1700, 53,  127, 0, 2.5,   +1, 1.258140e+02, 0,_
 1701, 53,  128, 0, 1,     +1, 1.268070e+02, 0,_
 1702, 53,  129, 0, 3.5,   +1, 1.277980e+02, 0,_
 1703, 53,  130, 0, 5,     +1, 1.287910e+02, 0,_
 1704, 53,  130, 1, 2,     +1, 1.287910e+02, 0.0399525,_
 1705, 53,  131, 0, 3.5,   +1, 1.297820e+02, 0,_
 1706, 53,  132, 0, 4,     +1, 1.307750e+02, 0,_
 1707, 53,  132, 1, 8,     -1, 1.307750e+02, 0.12,_
 1708, 53,  133, 0, 3.5,   +1, 1.317660e+02, 0,_
 1709, 53,  133, 1, 9.5,   -1, 1.317660e+02, 1.632,_
 1710, 53,  134, 0, 4,     +1, 1.327590e+02, 0,_
 1711, 53,  134, 1, 8,     -1, 1.327600e+02, 0.3165,_
 1712, 53,  135, 0, 3.5,   +1, 1.337510e+02, 0,_
 1713, 53,  136, 0, 1,     -1, 1.347470e+02, 0,_
 1714, 53,  136, 1, 6,     -1, 1.347470e+02, 0.65,_
 1715, 53,  137, 0, 3.5,   +1, 1.357420e+02, 0,_
 1716, 53,  138, 0, 2,     -1, 1.367380e+02, 0,_
 1717, 53,  139, 0, 3.5,   +1, 1.377330e+02, 0,_
 1718, 53,  140, 0, 3,     -1, 1.387290e+02, 0,_
 1719, 53,  141, 0, 3.5,   +1, 1.397240e+02, 0,_
 1720, 53,  142, 0, 2,     -1, 1.407210e+02, 0,_
 1721, 53,  143, 0, 3.5,   +1, 1.417170e+02, 0,_
 1722, 53,  144, 0, 1,     -1, 1.427130e+02, 0,_
 1723, 54,  110, 0, 0,     +1, 1.090000e+02, 0,_
 1724, 54,  111, 0, 2.5,   +1, 1.099890e+02, 0,_
 1725, 54,  112, 0, 0,     +1, 1.109740e+02, 0,_
 1726, 54,  113, 0, 2.5,   +1, 1.119630e+02, 0,_
 1727, 54,  114, 0, 0,     +1, 1.129490e+02, 0,_
 1728, 54,  115, 0, 2.5,   +1, 1.139390e+02, 0,_
 1729, 54,  116, 0, 0,     +1, 1.149260e+02, 0,_
 1730, 54,  117, 0, 2.5,   +1, 1.159160e+02, 0,_
 1731, 54,  118, 0, 0,     +1, 1.169030e+02, 0,_
 1732, 54,  119, 0, 2.5,   +1, 1.178940e+02, 0,_
 1733, 54,  120, 0, 0,     +1, 1.188820e+02, 0,_
 1734, 54,  121, 0, 2.5,   +1, 1.198730e+02, 0,_
 1735, 54,  122, 0, 0,     +1, 1.208610e+02, 0,_
 1736, 54,  123, 0, 0.5,   +1, 1.218530e+02, 0,_
 1737, 54,  124, 0, 0,     +1, 1.228410e+02, 0,_
 1738, 54,  125, 0, 0.5,   +1, 1.238330e+02, 0,_
 1739, 54,  125, 1, 4.5,   -1, 1.238330e+02, 0.2526,_
 1740, 54,  126, 0, 0,     +1, 1.248230e+02, 0,_
 1741, 54,  127, 0, 0.5,   +1, 1.258150e+02, 0,_
 1742, 54,  127, 1, 4.5,   -1, 1.258150e+02, 0.2971,_
 1743, 54,  128, 0, 0,     +1, 1.268050e+02, 0,_
 1744, 54,  129, 0, 0.5,   +1, 1.277970e+02, 0,_
 1745, 54,  129, 1, 5.5,   -1, 1.277970e+02, 0.23614,_
 1746, 54,  130, 0, 0,     +1, 1.287880e+02, 0,_
 1747, 54,  131, 0, 1.5,   +1, 1.297810e+02, 0,_
 1748, 54,  131, 1, 5.5,   -1, 1.297810e+02, 0.16393,_
 1749, 54,  132, 0, 0,     +1, 1.307710e+02, 0,_
 1750, 54,  132, 1, 10,    +1, 1.307710e+02, 2.7523,_
 1751, 54,  133, 0, 1.5,   +1, 1.317640e+02, 0,_
 1752, 54,  133, 1, 5.5,   -1, 1.317640e+02, 0.23322,_
 1753, 54,  134, 0, 0,     +1, 1.327550e+02, 0,_
 1754, 54,  134, 1, 7,     -1, 1.327550e+02, 1.9655,_
 1755, 54,  135, 0, 1.5,   +1, 1.337480e+02, 0,_
 1756, 54,  135, 1, 5.5,   -1, 1.337480e+02, 0.52655,_
 1757, 54,  136, 0, 0,     +1, 1.347400e+02, 0,_
 1758, 54,  137, 0, 3.5,   -1, 1.357350e+02, 0,_
 1759, 54,  138, 0, 0,     +1, 1.367290e+02, 0,_
 1760, 54,  139, 0, 1.5,   -1, 1.377250e+02, 0,_
 1761, 54,  140, 0, 0,     +1, 1.387200e+02, 0,_
 1762, 54,  141, 0, 2.5,   -1, 1.397160e+02, 0,_
 1763, 54,  142, 0, 0,     +1, 1.407100e+02, 0,_
 1764, 54,  143, 0, 2.5,   -1, 1.417070e+02, 0,_
 1765, 54,  144, 0, 0,     +1, 1.427020e+02, 0,_
 1766, 54,  145, 0, 1.5,   -1, 1.436990e+02, 0,_
 1767, 54,  146, 0, 0,     +1, 1.446940e+02, 0,_
 1768, 54,  147, 0, 1.5,   -1, 1.456910e+02, 0,_
 1769, 55,  112, 0, 1,     +1, 1.109890e+02, 0,_
 1770, 55,  113, 0, 2.5,   +1, 1.119740e+02, 0,_
 1771, 55,  114, 0, 1,     +1, 1.129630e+02, 0,_
 1772, 55,  115, 0, 4.5,   +1, 1.139490e+02, 0,_
 1773, 55,  116, 0, 1,     +1, 1.149370e+02, 0,_
 1774, 55,  116, 1,-77.777,+0, 1.149370e+02, 0.1,_
 1775, 55,  117, 0, 4.5,   +1, 1.159240e+02, 0,_
 1776, 55,  117, 1, 1.5,   +1, 1.159240e+02, 0.15,_
 1777, 55,  118, 0,-77.777,+0, 1.169140e+02, 0,_
 1778, 55,  118, 1, 7,     -1, 1.169140e+02, 0.1,_
 1779, 55,  119, 0, 4.5,   +1, 1.179010e+02, 0,_
 1780, 55,  119, 1, 1.5,   +1, 1.179010e+02, 0.05,_
 1781, 55,  120, 0, 2,     +1, 1.188910e+02, 0,_
 1782, 55,  120, 1, 7,     -1, 1.188910e+02, 0.1,_
 1783, 55,  121, 0, 1.5,   +1, 1.198780e+02, 0,_
 1784, 55,  121, 1, 4.5,   +1, 1.198780e+02, 0.0685,_
 1785, 55,  122, 0, 1,     +1, 1.208690e+02, 0,_
 1786, 55,  122, 1, 8,     -1, 1.208690e+02, 0.126,_
 1787, 55,  122, 2, 5,     -1, 1.208690e+02, 0.127,_
 1788, 55,  123, 0, 0.5,   +1, 1.218570e+02, 0,_
 1789, 55,  123, 1, 5.5,   -1, 1.218570e+02, 0.15627,_
 1790, 55,  124, 0, 1,     +1, 1.228480e+02, 0,_
 1791, 55,  124, 1, 7,     +1, 1.228480e+02, 0.46255,_
 1792, 55,  125, 0, 0.5,   +1, 1.238370e+02, 0,_
 1793, 55,  125, 1, 5.5,   -1, 1.238370e+02, 0.2666,_
 1794, 55,  126, 0, 1,     +1, 1.248280e+02, 0,_
 1795, 55,  127, 0, 0.5,   +1, 1.258170e+02, 0,_
 1796, 55,  128, 0, 1,     +1, 1.268090e+02, 0,_
 1797, 55,  129, 0, 0.5,   +1, 1.277990e+02, 0,_
 1798, 55,  130, 0, 1,     +1, 1.287910e+02, 0,_
 1799, 55,  130, 1, 5,     -1, 1.287910e+02, 0.16325,_
 1800, 55,  131, 0, 2.5,   +1, 1.297810e+02, 0,_
 1801, 55,  132, 0, 2,     +1, 1.307730e+02, 0,_
 1802, 55,  133, 0, 3.5,   +1, 1.317640e+02, 0,_
 1803, 55,  134, 0, 4,     +1, 1.327560e+02, 0,_
 1804, 55,  134, 1, 8,     -1, 1.327560e+02, 0.138747,_
 1805, 55,  135, 0, 3.5,   +1, 1.337470e+02, 0,_
 1806, 55,  135, 1, 9.5,   -1, 1.337470e+02, 1.6333,_
 1807, 55,  136, 0, 5,     +1, 1.347400e+02, 0,_
 1808, 55,  136, 1, 8,     -1, 1.347400e+02, 0.6,_
 1809, 55,  137, 0, 3.5,   +1, 1.357310e+02, 0,_
 1810, 55,  138, 0, 3,     -1, 1.367260e+02, 0,_
 1811, 55,  138, 1, 6,     -1, 1.367260e+02, 0.0799,_
 1812, 55,  139, 0, 3.5,   +1, 1.377200e+02, 0,_
 1813, 55,  140, 0, 1,     -1, 1.387150e+02, 0,_
 1814, 55,  141, 0, 3.5,   +1, 1.397090e+02, 0,_
 1815, 55,  142, 0, 0,     -1, 1.407050e+02, 0,_
 1816, 55,  143, 0, 1.5,   +1, 1.417000e+02, 0,_
 1817, 55,  144, 0, 1,     -1, 1.426960e+02, 0,_
 1818, 55,  144, 1,-77.777,+0, 1.426960e+02, 0.3,_
 1819, 55,  145, 0, 1.5,   +1, 1.436900e+02, 0,_
 1820, 55,  146, 0, 1,     -1, 1.446870e+02, 0,_
 1821, 55,  147, 0, 1.5,   +1, 1.456820e+02, 0,_
 1822, 55,  148, 0,-77.777,+0, 1.466780e+02, 0,_
 1823, 55,  149, 0, 1.5,   +1, 1.476730e+02, 0,_
 1824, 55,  150, 0,-77.777,+0, 1.486700e+02, 0,_
 1825, 55,  151, 0, 1.5,   +1, 1.496650e+02, 0,_
 1826, 56,  114, 0, 0,     +1, 1.129720e+02, 0,_
 1827, 56,  115, 0, 2.5,   +1, 1.139600e+02, 0,_
 1828, 56,  116, 0, 0,     +1, 1.149450e+02, 0,_
 1829, 56,  117, 0, 1.5,   +1, 1.159340e+02, 0,_
 1830, 56,  118, 0, 0,     +1, 1.169200e+02, 0,_
 1831, 56,  119, 0, 2.5,   +1, 1.179090e+02, 0,_
 1832, 56,  120, 0, 0,     +1, 1.188960e+02, 0,_
 1833, 56,  121, 0, 2.5,   +1, 1.198850e+02, 0,_
 1834, 56,  122, 0, 0,     +1, 1.208730e+02, 0,_
 1835, 56,  123, 0, 2.5,   +1, 1.218630e+02, 0,_
 1836, 56,  124, 0, 0,     +1, 1.228510e+02, 0,_
 1837, 56,  125, 0, 0.5,   +1, 1.238410e+02, 0,_
 1838, 56,  126, 0, 0,     +1, 1.248300e+02, 0,_
 1839, 56,  127, 0, 0.5,   +1, 1.258210e+02, 0,_
 1840, 56,  127, 1, 3.5,   -1, 1.258210e+02, 0.08033,_
 1841, 56,  128, 0, 0,     +1, 1.268100e+02, 0,_
 1842, 56,  129, 0, 0.5,   +1, 1.278010e+02, 0,_
 1843, 56,  129, 1, 3.5,   +1, 1.278010e+02, 0.00842,_
 1844, 56,  130, 0, 0,     +1, 1.287900e+02, 0,_
 1845, 56,  130, 1, 8,     -1, 1.287900e+02, 2.47512,_
 1846, 56,  131, 0, 0.5,   +1, 1.297820e+02, 0,_
 1847, 56,  131, 1, 4.5,   -1, 1.297820e+02, 0.1875,_
 1848, 56,  132, 0, 0,     +1, 1.307720e+02, 0,_
 1849, 56,  133, 0, 0.5,   +1, 1.317640e+02, 0,_
 1850, 56,  133, 1, 5.5,   -1, 1.317640e+02, 0.28825,_
 1851, 56,  134, 0, 0,     +1, 1.327540e+02, 0,_
 1852, 56,  135, 0, 1.5,   +1, 1.337470e+02, 0,_
 1853, 56,  135, 1, 5.5,   -1, 1.337470e+02, 0.268218,_
 1854, 56,  136, 0, 0,     +1, 1.347370e+02, 0,_
 1855, 56,  136, 1, 7,     -1, 1.347370e+02, 2.03052,_
 1856, 56,  137, 0, 1.5,   +1, 1.357300e+02, 0,_
 1857, 56,  137, 1, 5.5,   -1, 1.357300e+02, 0.661659,_
 1858, 56,  138, 0, 0,     +1, 1.367210e+02, 0,_
 1859, 56,  139, 0, 3.5,   -1, 1.377160e+02, 0,_
 1860, 56,  140, 0, 0,     +1, 1.387090e+02, 0,_
 1861, 56,  141, 0, 1.5,   -1, 1.397040e+02, 0,_
 1862, 56,  142, 0, 0,     +1, 1.406970e+02, 0,_
 1863, 56,  143, 0, 2.5,   -1, 1.416930e+02, 0,_
 1864, 56,  144, 0, 0,     +1, 1.426870e+02, 0,_
 1865, 56,  145, 0, 2.5,   -1, 1.436830e+02, 0,_
 1866, 56,  146, 0, 0,     +1, 1.446770e+02, 0,_
 1867, 56,  147, 0, 1.5,   +1, 1.456730e+02, 0,_
 1868, 56,  148, 0, 0,     +1, 1.466670e+02, 0,_
 1869, 56,  149, 0, 1.5,   -1, 1.476630e+02, 0,_
 1870, 56,  150, 0, 0,     +1, 1.486580e+02, 0,_
 1871, 56,  151, 0, 1.5,   -1, 1.496540e+02, 0,_
 1872, 56,  152, 0, 0,     +1, 1.506490e+02, 0,_
 1873, 56,  153, 0, 2.5,   -1, 1.516460e+02, 0,_
 1874, 57,  117, 0,-77.777,+0, 1.159450e+02, 0,_
 1875, 57,  117, 1, 4.5,   +1, 1.159450e+02, 0.138,_
 1876, 57,  118, 0,-77.777,+0, 1.169340e+02, 0,_
 1877, 57,  119, 0, 5.5,   -1, 1.179190e+02, 0,_
 1878, 57,  120, 0,-77.777,+0, 1.189080e+02, 0,_
 1879, 57,  121, 0, 5.5,   -1, 1.198940e+02, 0,_
 1880, 57,  122, 0,-77.777,+0, 1.208830e+02, 0,_
 1881, 57,  123, 0, 5.5,   -1, 1.218700e+02, 0,_
 1882, 57,  124, 0,-77.777,+0, 1.228600e+02, 0,_
 1883, 57,  124, 1, 0,     +1, 1.228600e+02, 0.1,_
 1884, 57,  125, 0, 5.5,   -1, 1.238480e+02, 0,_
 1885, 57,  125, 1, 1.5,   +1, 1.238480e+02, 0.107,_
 1886, 57,  126, 0, 5,     +1, 1.248380e+02, 0,_
 1887, 57,  126, 1,-77.777,+0, 1.248380e+02, 0.1,_
 1888, 57,  127, 0, 5.5,   -1, 1.258260e+02, 0,_
 1889, 57,  127, 1, 1.5,   +1, 1.258260e+02, 0.0148,_
 1890, 57,  128, 0, 5,     +1, 1.268170e+02, 0,_
 1891, 57,  128, 1,-77.777,+0, 1.268170e+02, 0.1,_
 1892, 57,  129, 0, 1.5,   +1, 1.278050e+02, 0,_
 1893, 57,  129, 1, 5.5,   -1, 1.278050e+02, 0.1721,_
 1894, 57,  130, 0, 3,     +1, 1.287960e+02, 0,_
 1895, 57,  131, 0, 1.5,   +1, 1.297850e+02, 0,_
 1896, 57,  132, 0, 2,     -1, 1.307770e+02, 0,_
 1897, 57,  132, 1, 6,     -1, 1.307770e+02, 0.18818,_
 1898, 57,  133, 0, 2.5,   +1, 1.317660e+02, 0,_
 1899, 57,  134, 0, 1,     +1, 1.327580e+02, 0,_
 1900, 57,  135, 0, 2.5,   +1, 1.337480e+02, 0,_
 1901, 57,  136, 0, 1,     +1, 1.347400e+02, 0,_
 1902, 57,  136, 1,-77.777,+0, 1.347400e+02, 0.255,_
 1903, 57,  137, 0, 3.5,   +1, 1.357300e+02, 0,_
 1904, 57,  138, 0, 5,     +1, 1.367220e+02, 0,_
 1905, 57,  139, 0, 3.5,   +1, 1.377130e+02, 0,_
 1906, 57,  140, 0, 3,     -1, 1.387080e+02, 0,_
 1907, 57,  141, 0, 3.5,   +1, 1.397000e+02, 0,_
 1908, 57,  142, 0, 2,     -1, 1.406950e+02, 0,_
 1909, 57,  143, 0, 3.5,   +1, 1.416880e+02, 0,_
 1910, 57,  144, 0, 3,     -1, 1.426830e+02, 0,_
 1911, 57,  145, 0, 2.5,   +1, 1.436770e+02, 0,_
 1912, 57,  146, 0, 2,     -1, 1.446720e+02, 0,_
 1913, 57,  146, 1, 6,     -1, 1.446720e+02, 0.13,_
 1914, 57,  147, 0, 2.5,   +1, 1.456660e+02, 0,_
 1915, 57,  148, 0, 2,     -1, 1.466610e+02, 0,_
 1916, 57,  149, 0, 2.5,   +1, 1.476550e+02, 0,_
 1917, 57,  150, 0, 3,     +1, 1.486510e+02, 0,_
 1918, 57,  151, 0, 2.5,   +1, 1.496450e+02, 0,_
 1919, 57,  152, 0, 2,     -1, 1.506390e+02, 0,_
 1920, 57,  153, 0, 2.5,   +1, 1.516360e+02, 0,_
 1921, 57,  154, 0,-77.777,+0, 1.526320e+02, 0,_
 1922, 57,  155, 0, 2.5,   +1, 1.536270e+02, 0,_
 1923, 58,  119, 0, 2.5,   +1, 1.179310e+02, 0,_
 1924, 58,  120, 0, 0,     +1, 1.189160e+02, 0,_
 1925, 58,  121, 0, 2.5,   +1, 1.199040e+02, 0,_
 1926, 58,  122, 0, 0,     +1, 1.208900e+02, 0,_
 1927, 58,  123, 0, 2.5,   +1, 1.218790e+02, 0,_
 1928, 58,  124, 0, 0,     +1, 1.228660e+02, 0,_
 1929, 58,  125, 0, 3.5,   -1, 1.238550e+02, 0,_
 1930, 58,  126, 0, 0,     +1, 1.248420e+02, 0,_
 1931, 58,  127, 0, 2.5,   +1, 1.258320e+02, 0,_
 1932, 58,  127, 1, 0.5,   +1, 1.258320e+02, 0,_
 1933, 58,  128, 0, 0,     +1, 1.268200e+02, 0,_
 1934, 58,  129, 0, 2.5,   +1, 1.278110e+02, 0,_
 1935, 58,  130, 0, 0,     +1, 1.287990e+02, 0,_
 1936, 58,  131, 0, 3.5,   +1, 1.297900e+02, 0,_
 1937, 58,  131, 1, 0.5,   +1, 1.297900e+02, 0.0618,_
 1938, 58,  132, 0, 0,     +1, 1.307780e+02, 0,_
 1939, 58,  132, 1, 8,     -1, 1.307780e+02, 2.3408,_
 1940, 58,  133, 0, 0.5,   +1, 1.317700e+02, 0,_
 1941, 58,  133, 1, 4.5,   -1, 1.317700e+02, 0.0371,_
 1942, 58,  134, 0, 0,     +1, 1.327590e+02, 0,_
 1943, 58,  135, 0, 0.5,   +1, 1.337500e+02, 0,_
 1944, 58,  135, 1, 5.5,   -1, 1.337500e+02, 0.4458,_
 1945, 58,  136, 0, 0,     +1, 1.347400e+02, 0,_
 1946, 58,  137, 0, 1.5,   +1, 1.357320e+02, 0,_
 1947, 58,  137, 1, 5.5,   -1, 1.357320e+02, 0.25429,_
 1948, 58,  138, 0, 0,     +1, 1.367210e+02, 0,_
 1949, 58,  138, 1, 7,     -1, 1.367210e+02, 2.12917,_
 1950, 58,  139, 0, 1.5,   +1, 1.377130e+02, 0,_
 1951, 58,  139, 1, 5.5,   -1, 1.377130e+02, 0.75424,_
 1952, 58,  140, 0, 0,     +1, 1.387040e+02, 0,_
 1953, 58,  141, 0, 3.5,   -1, 1.396980e+02, 0,_
 1954, 58,  142, 0, 0,     +1, 1.406900e+02, 0,_
 1955, 58,  143, 0, 1.5,   -1, 1.416850e+02, 0,_
 1956, 58,  144, 0, 0,     +1, 1.426770e+02, 0,_
 1957, 58,  145, 0, 1.5,   -1, 1.436720e+02, 0,_
 1958, 58,  146, 0, 0,     +1, 1.446650e+02, 0,_
 1959, 58,  147, 0, 2.5,   -1, 1.456600e+02, 0,_
 1960, 58,  148, 0, 0,     +1, 1.466540e+02, 0,_
 1961, 58,  149, 0, 1.5,   -1, 1.476490e+02, 0,_
 1962, 58,  150, 0, 0,     +1, 1.486420e+02, 0,_
 1963, 58,  151, 0, 1.5,   -1, 1.496370e+02, 0,_
 1964, 58,  152, 0, 0,     +1, 1.506310e+02, 0,_
 1965, 58,  153, 0, 1.5,   -1, 1.516250e+02, 0,_
 1966, 58,  154, 0, 0,     +1, 1.526180e+02, 0,_
 1967, 58,  155, 0, 2.5,   -1, 1.536170e+02, 0,_
 1968, 58,  156, 0, 0,     +1, 1.546120e+02, 0,_
 1969, 58,  157, 0, 3.5,   +1, 1.556080e+02, 0,_
 1970, 59,  121, 0, 1.5,   -1, 1.199160e+02, 0,_
 1971, 59,  122, 0,-77.777,+0, 1.209040e+02, 0,_
 1972, 59,  123, 0, 1.5,   +1, 1.218900e+02, 0,_
 1973, 59,  124, 0,-77.777,+0, 1.228780e+02, 0,_
 1974, 59,  125, 0, 1.5,   +1, 1.238650e+02, 0,_
 1975, 59,  126, 0,-77.777,+0, 1.248530e+02, 0,_
 1976, 59,  127, 0, 1.5,   +1, 1.258400e+02, 0,_
 1977, 59,  127, 1, 5.5,   -1, 1.258400e+02, 0.6,_
 1978, 59,  128, 0, 4,     -1, 1.268300e+02, 0,_
 1979, 59,  129, 0, 1.5,   +1, 1.278180e+02, 0,_
 1980, 59,  129, 1, 5.5,   -1, 1.278180e+02, 0.3827,_
 1981, 59,  130, 0,-77.777,+0, 1.288070e+02, 0,_
 1982, 59,  130, 1, 2,     +1, 1.288070e+02, 0.1,_
 1983, 59,  131, 0, 1.5,   +1, 1.297960e+02, 0,_
 1984, 59,  131, 1, 5.5,   -1, 1.297960e+02, 0.1524,_
 1985, 59,  132, 0, 2,     +1, 1.307860e+02, 0,_
 1986, 59,  132, 1, 5,     +1, 1.307860e+02, 0,_
 1987, 59,  133, 0, 1.5,   +1, 1.317750e+02, 0,_
 1988, 59,  134, 0, 2,     -1, 1.327650e+02, 0,_
 1989, 59,  134, 1, 2,     -1, 1.327650e+02, 0,_
 1990, 59,  135, 0, 1.5,   +1, 1.337540e+02, 0,_
 1991, 59,  136, 0, 2,     +1, 1.347450e+02, 0,_
 1992, 59,  137, 0, 2.5,   +1, 1.357350e+02, 0,_
 1993, 59,  138, 0, 1,     +1, 1.367260e+02, 0,_
 1994, 59,  138, 1, 7,     -1, 1.367260e+02, 0.364,_
 1995, 59,  139, 0, 2.5,   +1, 1.377160e+02, 0,_
 1996, 59,  140, 0, 1,     +1, 1.387070e+02, 0,_
 1997, 59,  141, 0, 2.5,   +1, 1.396970e+02, 0,_
 1998, 59,  142, 0, 2,     -1, 1.406910e+02, 0,_
 1999, 59,  142, 1, 5,     -1, 1.406910e+02, 0.003694,_
 2000, 59,  143, 0, 3.5,   +1, 1.416830e+02, 0,_
 2001, 59,  144, 0, 0,     -1, 1.426770e+02, 0,_
 2002, 59,  144, 1, 3,     -1, 1.426770e+02, 0.05903,_
 2003, 59,  145, 0, 3.5,   +1, 1.436700e+02, 0,_
 2004, 59,  146, 0, 2,     -1, 1.446640e+02, 0,_
 2005, 59,  147, 0, 1.5,   +1, 1.456570e+02, 0,_
 2006, 59,  148, 0, 1,     -1, 1.466510e+02, 0,_
 2007, 59,  148, 1, 4,     +0, 1.466510e+02, 0.09,_
 2008, 59,  149, 0, 2.5,   +1, 1.476440e+02, 0,_
 2009, 59,  150, 0, 1,     -1, 1.486390e+02, 0,_
 2010, 59,  151, 0, 1.5,   -1, 1.496320e+02, 0,_
 2011, 59,  152, 0, 4,     -1, 1.506260e+02, 0,_
 2012, 59,  153, 0, 2.5,   -1, 1.516200e+02, 0,_
 2013, 59,  154, 0,-77.777,+0, 1.526150e+02, 0,_
 2014, 59,  155, 0, 2.5,   -1, 1.536090e+02, 0,_
 2015, 59,  156, 0, 2,     -1, 1.546030e+02, 0,_
 2016, 59,  157, 0, 2.5,   -1, 1.555970e+02, 0,_
 2017, 59,  158, 0,-77.777,+0, 1.565950e+02, 0,_
 2018, 59,  159, 0, 2.5,   -1, 1.575900e+02, 0,_
 2019, 60,  124, 0, 0,     +1, 1.228870e+02, 0,_
 2020, 60,  125, 0, 2.5,   +1, 1.238760e+02, 0,_
 2021, 60,  126, 0, 0,     +1, 1.248610e+02, 0,_
 2022, 60,  127, 0, 2.5,   +1, 1.258500e+02, 0,_
 2023, 60,  128, 0, 0,     +1, 1.268360e+02, 0,_
 2024, 60,  129, 0, 2.5,   +1, 1.278260e+02, 0,_
 2025, 60,  130, 0, 0,     +1, 1.288120e+02, 0,_
 2026, 60,  131, 0, 2.5,   +1, 1.298030e+02, 0,_
 2027, 60,  132, 0, 0,     +1, 1.307900e+02, 0,_
 2028, 60,  133, 0, 3.5,   +1, 1.317800e+02, 0,_
 2029, 60,  133, 1, 0.5,   +1, 1.317800e+02, 0.12797,_
 2030, 60,  134, 0, 0,     +1, 1.327680e+02, 0,_
 2031, 60,  135, 0, 4.5,   -1, 1.337590e+02, 0,_
 2032, 60,  135, 1, 0.5,   +1, 1.337590e+02, 0.065,_
 2033, 60,  136, 0, 0,     +1, 1.347470e+02, 0,_
 2034, 60,  137, 0, 0.5,   +1, 1.357380e+02, 0,_
 2035, 60,  137, 1, 5.5,   -1, 1.357380e+02, 0.51943,_
 2036, 60,  138, 0, 0,     +1, 1.367270e+02, 0,_
 2037, 60,  139, 0, 1.5,   +1, 1.377190e+02, 0,_
 2038, 60,  139, 1, 5.5,   -1, 1.377190e+02, 0.23115,_
 2039, 60,  140, 0, 0,     +1, 1.387070e+02, 0,_
 2040, 60,  141, 0, 1.5,   +1, 1.396990e+02, 0,_
 2041, 60,  141, 1, 5.5,   -1, 1.396990e+02, 0.75651,_
 2042, 60,  142, 0, 0,     +1, 1.406890e+02, 0,_
 2043, 60,  143, 0, 3.5,   -1, 1.416820e+02, 0,_
 2044, 60,  144, 0, 0,     +1, 1.426740e+02, 0,_
 2045, 60,  145, 0, 3.5,   -1, 1.436680e+02, 0,_
 2046, 60,  146, 0, 0,     +1, 1.446600e+02, 0,_
 2047, 60,  147, 0, 2.5,   -1, 1.456540e+02, 0,_
 2048, 60,  148, 0, 0,     +1, 1.466460e+02, 0,_
 2049, 60,  149, 0, 2.5,   -1, 1.476410e+02, 0,_
 2050, 60,  150, 0, 0,     +1, 1.486330e+02, 0,_
 2051, 60,  151, 0, 1.5,   +1, 1.496270e+02, 0,_
 2052, 60,  152, 0, 0,     +1, 1.506200e+02, 0,_
 2053, 60,  153, 0, 1.5,   -1, 1.516140e+02, 0,_
 2054, 60,  154, 0, 0,     +1, 1.526070e+02, 0,_
 2055, 60,  155, 0, 1.5,   -1, 1.536020e+02, 0,_
 2056, 60,  156, 0, 0,     +1, 1.545950e+02, 0,_
 2057, 60,  157, 0, 2.5,   -1, 1.555890e+02, 0,_
 2058, 60,  158, 0, 0,     +1, 1.565830e+02, 0,_
 2059, 60,  159, 0, 3.5,   +1, 1.575790e+02, 0,_
 2060, 60,  160, 0, 0,     +1, 1.585730e+02, 0,_
 2061, 60,  161, 0, 0.5,   -1, 1.595710e+02, 0,_
 2062, 61,  126, 0,-77.777,+0, 1.248750e+02, 0,_
 2063, 61,  127, 0, 2.5,   +1, 1.258610e+02, 0,_
 2064, 61,  128, 0, 6,     +1, 1.268490e+02, 0,_
 2065, 61,  129, 0, 2.5,   +1, 1.278350e+02, 0,_
 2066, 61,  130, 0,-77.777,+0, 1.288240e+02, 0,_
 2067, 61,  131, 0, 2.5,   +1, 1.298110e+02, 0,_
 2068, 61,  132, 0, 3,     +1, 1.308000e+02, 0,_
 2069, 61,  133, 0, 1.5,   +1, 1.317880e+02, 0,_
 2070, 61,  133, 1, 5.5,   -1, 1.317880e+02, 0.1304,_
 2071, 61,  134, 0, 2,     +1, 1.327780e+02, 0,_
 2072, 61,  134, 1, 2,     +1, 1.327780e+02, 0,_
 2073, 61,  135, 0,-77.777,+0, 1.337660e+02, 0,_
 2074, 61,  135, 1, 5.5,   -1, 1.337660e+02, 0.05,_
 2075, 61,  136, 0, 5,     -1, 1.347560e+02, 0,_
 2076, 61,  136, 1, 2,     +1, 1.347560e+02, 0.28,_
 2077, 61,  137, 0, 2.5,   +1, 1.357440e+02, 0,_
 2078, 61,  137, 1, 5.5,   -1, 1.357440e+02, 0.18,_
 2079, 61,  138, 0, 1,     +1, 1.367350e+02, 0,_
 2080, 61,  138, 1, 5,     -1, 1.367350e+02, 0.03,_
 2081, 61,  139, 0, 2.5,   +1, 1.377230e+02, 0,_
 2082, 61,  139, 1, 5.5,   -1, 1.377230e+02, 0.1887,_
 2083, 61,  140, 0, 1,     +1, 1.387140e+02, 0,_
 2084, 61,  140, 1, 7,     -1, 1.387140e+02, 0.4,_
 2085, 61,  141, 0, 2.5,   +1, 1.397030e+02, 0,_
 2086, 61,  142, 0, 1,     +1, 1.406940e+02, 0,_
 2087, 61,  142, 1, 8,     -1, 1.406940e+02, 0.88317,_
 2088, 61,  143, 0, 2.5,   +1, 1.416830e+02, 0,_
 2089, 61,  144, 0, 5,     -1, 1.426760e+02, 0,_
 2090, 61,  145, 0, 2.5,   +1, 1.436680e+02, 0,_
 2091, 61,  146, 0, 3,     -1, 1.446610e+02, 0,_
 2092, 61,  147, 0, 3.5,   +1, 1.456530e+02, 0,_
 2093, 61,  148, 0, 1,     -1, 1.466470e+02, 0,_
 2094, 61,  148, 1, 6,     -1, 1.466470e+02, 0.1379,_
 2095, 61,  149, 0, 3.5,   +1, 1.476390e+02, 0,_
 2096, 61,  150, 0, 1,     -1, 1.486330e+02, 0,_
 2097, 61,  151, 0, 2.5,   +1, 1.496250e+02, 0,_
 2098, 61,  152, 0, 1,     +1, 1.506180e+02, 0,_
 2099, 61,  152, 1, 4,     -1, 1.506180e+02, 0.15,_
 2100, 61,  152, 2, 8,     -1, 1.506180e+02, 0.32,_
 2101, 61,  153, 0, 2.5,   -1, 1.516100e+02, 0,_
 2102, 61,  154, 0,-77.777,+0, 1.526040e+02, 0,_
 2103, 61,  154, 1,-77.777,+0, 1.526040e+02, 0.05,_
 2104, 61,  155, 0, 2.5,   -1, 1.535970e+02, 0,_
 2105, 61,  156, 0, 4,     -1, 1.545920e+02, 0,_
 2106, 61,  157, 0, 2.5,   -1, 1.555850e+02, 0,_
 2107, 61,  158, 0,-77.777,+0, 1.565800e+02, 0,_
 2108, 61,  159, 0, 2.5,   -1, 1.575730e+02, 0,_
 2109, 61,  160, 0, 1,     -1, 1.585680e+02, 0,_
 2110, 61,  161, 0, 2.5,   -1, 1.595630e+02, 0,_
 2111, 61,  162, 0,-77.777,+0, 1.605590e+02, 0,_
 2112, 61,  163, 0, 2.5,   -1, 1.615540e+02, 0,_
 2113, 62,  128, 0, 0,     +1, 1.268590e+02, 0,_
 2114, 62,  129, 0, 2.5,   +1, 1.278470e+02, 0,_
 2115, 62,  130, 0, 0,     +1, 1.288330e+02, 0,_
 2116, 62,  131, 0, 2.5,   +1, 1.298210e+02, 0,_
 2117, 62,  132, 0, 0,     +1, 1.308070e+02, 0,_
 2118, 62,  133, 0, 2.5,   +1, 1.317970e+02, 0,_
 2119, 62,  134, 0, 0,     +1, 1.327830e+02, 0,_
 2120, 62,  135, 0, 3.5,   +1, 1.337730e+02, 0,_
 2121, 62,  135, 1,-77.777,+0, 1.337730e+02, 0,_
 2122, 62,  136, 0, 0,     +1, 1.347610e+02, 0,_
 2123, 62,  137, 0, 4.5,   -1, 1.357510e+02, 0,_
 2124, 62,  137, 1, 0.5,   +1, 1.357510e+02, 0.18,_
 2125, 62,  138, 0, 0,     +1, 1.367380e+02, 0,_
 2126, 62,  139, 0, 0.5,   +1, 1.377290e+02, 0,_
 2127, 62,  139, 1, 5.5,   -1, 1.377290e+02, 0.4574,_
 2128, 62,  140, 0, 0,     +1, 1.387170e+02, 0,_
 2129, 62,  141, 0, 0.5,   +1, 1.397080e+02, 0,_
 2130, 62,  141, 1, 5.5,   -1, 1.397080e+02, 0.176,_
 2131, 62,  142, 0, 0,     +1, 1.406960e+02, 0,_
 2132, 62,  143, 0, 1.5,   +1, 1.416870e+02, 0,_
 2133, 62,  143, 1, 5.5,   -1, 1.416870e+02, 0.75399,_
 2134, 62,  143, 2, 11.5,  -1, 1.416870e+02, 2.7938,_
 2135, 62,  144, 0, 0,     +1, 1.426760e+02, 0,_
 2136, 62,  145, 0, 3.5,   -1, 1.436690e+02, 0,_
 2137, 62,  146, 0, 0,     +1, 1.446600e+02, 0,_
 2138, 62,  147, 0, 3.5,   -1, 1.456530e+02, 0,_
 2139, 62,  148, 0, 0,     +1, 1.466440e+02, 0,_
 2140, 62,  149, 0, 3.5,   -1, 1.476380e+02, 0,_
 2141, 62,  150, 0, 0,     +1, 1.486290e+02, 0,_
 2142, 62,  151, 0, 2.5,   -1, 1.496230e+02, 0,_
 2143, 62,  152, 0, 0,     +1, 1.506150e+02, 0,_
 2144, 62,  153, 0, 1.5,   +1, 1.516080e+02, 0,_
 2145, 62,  153, 1, 5.5,   -1, 1.516080e+02, 0.09837,_
 2146, 62,  154, 0, 0,     +1, 1.526000e+02, 0,_
 2147, 62,  155, 0, 1.5,   -1, 1.535940e+02, 0,_
 2148, 62,  156, 0, 0,     +1, 1.545860e+02, 0,_
 2149, 62,  157, 0, 1.5,   -1, 1.555800e+02, 0,_
 2150, 62,  158, 0, 0,     +1, 1.565730e+02, 0,_
 2151, 62,  159, 0, 2.5,   -1, 1.575680e+02, 0,_
 2152, 62,  160, 0, 0,     +1, 1.585610e+02, 0,_
 2153, 62,  161, 0, 3.5,   +1, 1.595560e+02, 0,_
 2154, 62,  162, 0, 0,     +1, 1.605500e+02, 0,_
 2155, 62,  163, 0, 0.5,   -1, 1.615460e+02, 0,_
 2156, 62,  164, 0, 0,     +1, 1.625400e+02, 0,_
 2157, 62,  165, 0, 2.5,   -1, 1.635360e+02, 0,_
 2158, 63,  130, 0, 2,     +1, 1.288470e+02, 0,_
 2159, 63,  131, 0, 1.5,   +1, 1.298330e+02, 0,_
 2160, 63,  132, 0,-77.777,+0, 1.308210e+02, 0,_
 2161, 63,  133, 0, 5.5,   -1, 1.318070e+02, 0,_
 2162, 63,  134, 0,-77.777,+0, 1.327960e+02, 0,_
 2163, 63,  135, 0, 5.5,   -1, 1.337830e+02, 0,_
 2164, 63,  136, 0, 3,     +1, 1.347720e+02, 0,_
 2165, 63,  136, 1, 3,     +1, 1.347720e+02, 0,_
 2166, 63,  137, 0, 5.5,   -1, 1.357590e+02, 0,_
 2167, 63,  138, 0, 6,     -1, 1.367490e+02, 0,_
 2168, 63,  139, 0, 5.5,   -1, 1.377360e+02, 0,_
 2169, 63,  140, 0, 1,     +1, 1.387260e+02, 0,_
 2170, 63,  140, 1, 5,     -1, 1.387260e+02, 0.21,_
 2171, 63,  141, 0, 2.5,   +1, 1.397140e+02, 0,_
 2172, 63,  141, 1, 5.5,   -1, 1.397140e+02, 0.09645,_
 2173, 63,  142, 0, 1,     +1, 1.407040e+02, 0,_
 2174, 63,  142, 1, 8,     -1, 1.407040e+02, 0.46,_
 2175, 63,  143, 0, 2.5,   +1, 1.416930e+02, 0,_
 2176, 63,  144, 0, 1,     +1, 1.426820e+02, 0,_
 2177, 63,  145, 0, 2.5,   +1, 1.436710e+02, 0,_
 2178, 63,  146, 0, 4,     -1, 1.446640e+02, 0,_
 2179, 63,  147, 0, 2.5,   +1, 1.456550e+02, 0,_
 2180, 63,  148, 0, 5,     -1, 1.466470e+02, 0,_
 2181, 63,  149, 0, 2.5,   +1, 1.476390e+02, 0,_
 2182, 63,  150, 0, 5,     -1, 1.486320e+02, 0,_
 2183, 63,  150, 1, 0,     -1, 1.486320e+02, 0.0421,_
 2184, 63,  151, 0, 2.5,   +1, 1.496230e+02, 0,_
 2185, 63,  152, 0, 3,     -1, 1.506170e+02, 0,_
 2186, 63,  152, 1, 0,     -1, 1.506170e+02, 0.0456,_
 2187, 63,  152, 2, 8,     -1, 1.506170e+02, 0.1478,_
 2188, 63,  153, 0, 2.5,   +1, 1.516080e+02, 0,_
 2189, 63,  154, 0, 3,     -1, 1.526010e+02, 0,_
 2190, 63,  154, 1, 8,     -1, 1.526010e+02, 0.157,_
 2191, 63,  155, 0, 2.5,   +1, 1.535920e+02, 0,_
 2192, 63,  156, 0, 0,     +1, 1.545850e+02, 0,_
 2193, 63,  157, 0, 2.5,   +1, 1.555770e+02, 0,_
 2194, 63,  158, 0, 1,     -1, 1.565710e+02, 0,_
 2195, 63,  159, 0, 2.5,   +1, 1.575640e+02, 0,_
 2196, 63,  160, 0, 1,     -1, 1.585580e+02, 0,_
 2197, 63,  161, 0, 2.5,   +1, 1.595510e+02, 0,_
 2198, 63,  162, 0,-77.777,+0, 1.605460e+02, 0,_
 2199, 63,  163, 0, 2.5,   +1, 1.615390e+02, 0,_
 2200, 63,  164, 0,-77.777,+0, 1.625350e+02, 0,_
 2201, 63,  165, 0, 2.5,   +1, 1.635290e+02, 0,_
 2202, 63,  166, 0,-77.777,+0, 1.645240e+02, 0,_
 2203, 63,  167, 0, 2.5,   +1, 1.655190e+02, 0,_
 2204, 64,  134, 0, 0,     +1, 1.328050e+02, 0,_
 2205, 64,  135, 0, 1.5,   -1, 1.337930e+02, 0,_
 2206, 64,  136, 0, 0,     +1, 1.347790e+02, 0,_
 2207, 64,  137, 0, 3.5,   +1, 1.357690e+02, 0,_
 2208, 64,  138, 0, 0,     +1, 1.367550e+02, 0,_
 2209, 64,  139, 0, 4.5,   -1, 1.377450e+02, 0,_
 2210, 64,  139, 1, 0.5,   +1, 1.377450e+02, 0.25,_
 2211, 64,  140, 0, 0,     +1, 1.387320e+02, 0,_
 2212, 64,  141, 0, 0.5,   +1, 1.397210e+02, 0,_
 2213, 64,  141, 1, 5.5,   -1, 1.397210e+02, 0.3778,_
 2214, 64,  142, 0, 0,     +1, 1.407090e+02, 0,_
 2215, 64,  143, 0, 0.5,   +1, 1.416990e+02, 0,_
 2216, 64,  143, 1, 5.5,   -1, 1.416990e+02, 0.1526,_
 2217, 64,  144, 0, 0,     +1, 1.426870e+02, 0,_
 2218, 64,  145, 0, 0.5,   +1, 1.436770e+02, 0,_
 2219, 64,  145, 1, 5.5,   -1, 1.436770e+02, 0.7491,_
 2220, 64,  146, 0, 0,     +1, 1.446650e+02, 0,_
 2221, 64,  147, 0, 3.5,   -1, 1.456570e+02, 0,_
 2222, 64,  148, 0, 0,     +1, 1.466470e+02, 0,_
 2223, 64,  149, 0, 3.5,   -1, 1.476400e+02, 0,_
 2224, 64,  150, 0, 0,     +1, 1.486310e+02, 0,_
 2225, 64,  151, 0, 3.5,   -1, 1.496240e+02, 0,_
 2226, 64,  152, 0, 0,     +1, 1.506150e+02, 0,_
 2227, 64,  153, 0, 1.5,   -1, 1.516080e+02, 0,_
 2228, 64,  154, 0, 0,     +1, 1.525990e+02, 0,_
 2229, 64,  155, 0, 1.5,   -1, 1.535920e+02, 0,_
 2230, 64,  155, 1, 5.5,   -1, 1.535920e+02, 0.12105,_
 2231, 64,  156, 0, 0,     +1, 1.545830e+02, 0,_
 2232, 64,  157, 0, 1.5,   -1, 1.555760e+02, 0,_
 2233, 64,  158, 0, 0,     +1, 1.565670e+02, 0,_
 2234, 64,  159, 0, 1.5,   -1, 1.575610e+02, 0,_
 2235, 64,  160, 0, 0,     +1, 1.585530e+02, 0,_
 2236, 64,  161, 0, 2.5,   -1, 1.595470e+02, 0,_
 2237, 64,  162, 0, 0,     +1, 1.605400e+02, 0,_
 2238, 64,  163, 0, 2.5,   -1, 1.615340e+02, 0,_
 2239, 64,  164, 0, 0,     +1, 1.625280e+02, 0,_
 2240, 64,  165, 0, 0.5,   -1, 1.635220e+02, 0,_
 2241, 64,  166, 0, 0,     +1, 1.645160e+02, 0,_
 2242, 64,  167, 0, 2.5,   -1, 1.655110e+02, 0,_
 2243, 64,  168, 0, 0,     +1, 1.665060e+02, 0,_
 2244, 64,  169, 0, 3.5,   -1, 1.675010e+02, 0,_
 2245, 65,  136, 0,-77.777,+0, 1.347930e+02, 0,_
 2246, 65,  137, 0, 5.5,   -1, 1.357790e+02, 0,_
 2247, 65,  138, 0,-77.777,+0, 1.367680e+02, 0,_
 2248, 65,  139, 0, 5.5,   -1, 1.377550e+02, 0,_
 2249, 65,  140, 0,-77.777,+0, 1.387440e+02, 0,_
 2250, 65,  141, 0, 2.5,   -1, 1.397310e+02, 0,_
 2251, 65,  141, 1, 5.5,   -1, 1.397310e+02, 0,_
 2252, 65,  142, 0, 1,     +1, 1.407190e+02, 0,_
 2253, 65,  142, 1, 5,     -1, 1.407190e+02, 0.2802,_
 2254, 65,  143, 0, 5.5,   -1, 1.417070e+02, 0,_
 2255, 65,  143, 1, 2.5,   +1, 1.417070e+02, 0,_
 2256, 65,  144, 0, 1,     +1, 1.426970e+02, 0,_
 2257, 65,  144, 1, 6,     -1, 1.426970e+02, 0.3969,_
 2258, 65,  145, 0, 1.5,   +1, 1.436840e+02, 0,_
 2259, 65,  145, 1, 5.5,   -1, 1.436840e+02, 0,_
 2260, 65,  146, 0, 1,     +1, 1.446740e+02, 0,_
 2261, 65,  146, 1, 5,     -1, 1.446740e+02, 0.15,_
 2262, 65,  146, 2, 10,    +1, 1.446740e+02, 0.93,_
 2263, 65,  147, 0, 0.5,   +1, 1.456620e+02, 0,_
 2264, 65,  147, 1, 5.5,   -1, 1.456620e+02, 0.0506,_
 2265, 65,  148, 0, 2,     -1, 1.466540e+02, 0,_
 2266, 65,  148, 1, 9,     +1, 1.466540e+02, 0.0901,_
 2267, 65,  149, 0, 0.5,   +1, 1.476440e+02, 0,_
 2268, 65,  149, 1, 5.5,   -1, 1.476440e+02, 0.04,_
 2269, 65,  150, 0, 2,     -1, 1.486360e+02, 0,_
 2270, 65,  150, 1, 9,     +1, 1.486360e+02, 0.45,_
 2271, 65,  151, 0, 0.5,   +1, 1.496270e+02, 0,_
 2272, 65,  151, 1, 5.5,   -1, 1.496270e+02, 0.09954,_
 2273, 65,  152, 0, 2,     -1, 1.506190e+02, 0,_
 2274, 65,  152, 1, 8,     +1, 1.506190e+02, 0.50174,_
 2275, 65,  153, 0, 2.5,   +1, 1.516100e+02, 0,_
 2276, 65,  154, 0, 0,     +1, 1.526020e+02, 0,_
 2277, 65,  154, 1, 3,     -1, 1.526020e+02, 0.012,_
 2278, 65,  154, 2, 7,     -1, 1.526020e+02, 0.2,_
 2279, 65,  155, 0, 1.5,   +1, 1.535930e+02, 0,_
 2280, 65,  156, 0, 3,     -1, 1.545850e+02, 0,_
 2281, 65,  156, 1, 7,     -1, 1.545850e+02, 0.06,_
 2282, 65,  156, 2, 0,     +1, 1.545850e+02, 0.0884,_
 2283, 65,  157, 0, 1.5,   +1, 1.555760e+02, 0,_
 2284, 65,  158, 0, 3,     -1, 1.565690e+02, 0,_
 2285, 65,  158, 1, 0,     +1, 1.565690e+02, 0.1103,_
 2286, 65,  159, 0, 1.5,   +1, 1.575600e+02, 0,_
 2287, 65,  160, 0, 3,     -1, 1.585530e+02, 0,_
 2288, 65,  161, 0, 1.5,   +1, 1.595450e+02, 0,_
 2289, 65,  162, 0, 1,     -1, 1.605380e+02, 0,_
 2290, 65,  163, 0, 1.5,   +1, 1.615310e+02, 0,_
 2291, 65,  164, 0, 5,     +1, 1.625250e+02, 0,_
 2292, 65,  165, 0, 1.5,   +1, 1.635180e+02, 0,_
 2293, 65,  166, 0,-77.777,+0, 1.645130e+02, 0,_
 2294, 65,  167, 0, 1.5,   +1, 1.655060e+02, 0,_
 2295, 65,  168, 0, 4,     -1, 1.665010e+02, 0,_
 2296, 65,  169, 0, 1.5,   +1, 1.674950e+02, 0,_
 2297, 65,  170, 0,-77.777,+0, 1.684900e+02, 0,_
 2298, 65,  171, 0, 1.5,   +1, 1.694850e+02, 0,_
 2299, 66,  138, 0, 0,     +1, 1.367770e+02, 0,_
 2300, 66,  139, 0, 3.5,   +1, 1.377660e+02, 0,_
 2301, 66,  140, 0, 0,     +1, 1.387520e+02, 0,_
 2302, 66,  141, 0, 4.5,   -1, 1.397410e+02, 0,_
 2303, 66,  142, 0, 0,     +1, 1.407270e+02, 0,_
 2304, 66,  143, 0, 0.5,   +1, 1.417160e+02, 0,_
 2305, 66,  143, 1, 5.5,   -1, 1.417160e+02, 0.3107,_
 2306, 66,  144, 0, 0,     +1, 1.427030e+02, 0,_
 2307, 66,  145, 0, 0.5,   +1, 1.436920e+02, 0,_
 2308, 66,  145, 1, 5.5,   -1, 1.436920e+02, 0.1182,_
 2309, 66,  146, 0, 0,     +1, 1.446790e+02, 0,_
 2310, 66,  146, 1, 10,    +1, 1.446790e+02, 2.9357,_
 2311, 66,  147, 0, 0.5,   +1, 1.456690e+02, 0,_
 2312, 66,  147, 1, 5.5,   -1, 1.456690e+02, 0.7505,_
 2313, 66,  148, 0, 0,     +1, 1.466560e+02, 0,_
 2314, 66,  149, 0, 3.5,   -1, 1.476480e+02, 0,_
 2315, 66,  149, 1, 13.5,  -1, 1.476480e+02, 2.6611,_
 2316, 66,  150, 0, 0,     +1, 1.486380e+02, 0,_
 2317, 66,  151, 0, 3.5,   -1, 1.496300e+02, 0,_
 2318, 66,  152, 0, 0,     +1, 1.506200e+02, 0,_
 2319, 66,  153, 0, 3.5,   -1, 1.516120e+02, 0,_
 2320, 66,  154, 0, 0,     +1, 1.526020e+02, 0,_
 2321, 66,  155, 0, 1.5,   -1, 1.535950e+02, 0,_
 2322, 66,  156, 0, 0,     +1, 1.545850e+02, 0,_
 2323, 66,  157, 0, 1.5,   -1, 1.555770e+02, 0,_
 2324, 66,  157, 1, 5.5,   -1, 1.555770e+02, 0.1995,_
 2325, 66,  158, 0, 0,     +1, 1.565680e+02, 0,_
 2326, 66,  159, 0, 1.5,   -1, 1.575610e+02, 0,_
 2327, 66,  160, 0, 0,     +1, 1.585510e+02, 0,_
 2328, 66,  161, 0, 2.5,   +1, 1.595440e+02, 0,_
 2329, 66,  162, 0, 0,     +1, 1.605360e+02, 0,_
 2330, 66,  163, 0, 2.5,   -1, 1.615290e+02, 0,_
 2331, 66,  164, 0, 0,     +1, 1.625210e+02, 0,_
 2332, 66,  165, 0, 3.5,   +1, 1.635150e+02, 0,_
 2333, 66,  165, 1, 0.5,   -1, 1.635150e+02, 0.10816,_
 2334, 66,  166, 0, 0,     +1, 1.645070e+02, 0,_
 2335, 66,  167, 0, 0.5,   -1, 1.655020e+02, 0,_
 2336, 66,  168, 0, 0,     +1, 1.664940e+02, 0,_
 2337, 66,  169, 0, 2.5,   -1, 1.674890e+02, 0,_
 2338, 66,  170, 0, 0,     +1, 1.684830e+02, 0,_
 2339, 66,  171, 0, 3.5,   -1, 1.694780e+02, 0,_
 2340, 66,  172, 0, 0,     +1, 1.704720e+02, 0,_
 2341, 66,  173, 0, 4.5,   +1, 1.714670e+02, 0,_
 2342, 67,  140, 0, 8,     +1, 1.387660e+02, 0,_
 2343, 67,  141, 0, 3.5,   -1, 1.397520e+02, 0,_
 2344, 67,  142, 0,-77.777,+0, 1.407400e+02, 0,_
 2345, 67,  143, 0, 5.5,   -1, 1.417270e+02, 0,_
 2346, 67,  144, 0,-77.777,+0, 1.427150e+02, 0,_
 2347, 67,  145, 0, 5.5,   -1, 1.437020e+02, 0,_
 2348, 67,  145, 1, 2.5,   +1, 1.437020e+02, 0.1,_
 2349, 67,  146, 0, 10,    +1, 1.446910e+02, 0,_
 2350, 67,  147, 0, 5.5,   -1, 1.456780e+02, 0,_
 2351, 67,  148, 0, 1,     +1, 1.466670e+02, 0,_
 2352, 67,  148, 1, 10,    +1, 1.466670e+02, 0.6944,_
 2353, 67,  148, 2, 10,    +1, 1.466670e+02, 0.69,_
 2354, 67,  149, 0, 5.5,   -1, 1.476540e+02, 0,_
 2355, 67,  149, 1, 0.5,   +1, 1.476540e+02, 0.0488,_
 2356, 67,  150, 0, 2,     -1, 1.486450e+02, 0,_
 2357, 67,  150, 1, 9,     +1, 1.486450e+02, 0,_
 2358, 67,  151, 0, 5.5,   -1, 1.496350e+02, 0,_
 2359, 67,  151, 1, 0.5,   +1, 1.496350e+02, 0.041,_
 2360, 67,  152, 0, 2,     -1, 1.506270e+02, 0,_
 2361, 67,  152, 1, 9,     +1, 1.506270e+02, 0.16,_
 2362, 67,  153, 0, 5.5,   -1, 1.516160e+02, 0,_
 2363, 67,  153, 1, 0.5,   +1, 1.516160e+02, 0.0687,_
 2364, 67,  154, 0, 2,     -1, 1.526080e+02, 0,_
 2365, 67,  154, 1, 8,     +1, 1.526080e+02, 0.24,_
 2366, 67,  155, 0, 2.5,   +1, 1.535980e+02, 0,_
 2367, 67,  156, 0, 4,     -1, 1.545900e+02, 0,_
 2368, 67,  156, 1, 1,     -1, 1.545900e+02, 0.0524,_
 2369, 67,  156, 2, 9,     +1, 1.545900e+02, 0.1,_
 2370, 67,  157, 0, 3.5,   -1, 1.555800e+02, 0,_
 2371, 67,  158, 0, 5,     +1, 1.565720e+02, 0,_
 2372, 67,  158, 1, 2,     -1, 1.565720e+02, 0.0672,_
 2373, 67,  158, 2, 9,     +1, 1.565720e+02, 0.18,_
 2374, 67,  159, 0, 3.5,   -1, 1.575620e+02, 0,_
 2375, 67,  159, 1, 0.5,   +1, 1.575620e+02, 0.20588,_
 2376, 67,  160, 0, 5,     +1, 1.585550e+02, 0,_
 2377, 67,  160, 1, 2,     -1, 1.585550e+02, 0.05998,_
 2378, 67,  160, 2, 9,     +1, 1.585550e+02, 0.2,_
 2379, 67,  161, 0, 3.5,   -1, 1.595450e+02, 0,_
 2380, 67,  161, 1, 0.5,   +1, 1.595450e+02, 0.21114,_
 2381, 67,  162, 0, 1,     +1, 1.605380e+02, 0,_
 2382, 67,  162, 1, 6,     -1, 1.605380e+02, 0.106,_
 2383, 67,  163, 0, 3.5,   -1, 1.615290e+02, 0,_
 2384, 67,  163, 1, 0.5,   +1, 1.615290e+02, 0.29788,_
 2385, 67,  164, 0, 1,     +1, 1.625220e+02, 0,_
 2386, 67,  164, 1, 6,     -1, 1.625220e+02, 0.14,_
 2387, 67,  165, 0, 3.5,   -1, 1.635130e+02, 0,_
 2388, 67,  166, 0, 0,     -1, 1.645070e+02, 0,_
 2389, 67,  166, 1, 7,     +1, 1.645070e+02, 0.00598,_
 2390, 67,  167, 0, 3.5,   -1, 1.654990e+02, 0,_
 2391, 67,  168, 0, 3,     +1, 1.664930e+02, 0,_
 2392, 67,  168, 1, 6,     +1, 1.664930e+02, 0.059,_
 2393, 67,  169, 0, 3.5,   -1, 1.674860e+02, 0,_
 2394, 67,  170, 0, 6,     +1, 1.684800e+02, 0,_
 2395, 67,  170, 1, 1,     +1, 1.684800e+02, 0.12,_
 2396, 67,  171, 0, 3.5,   -1, 1.694730e+02, 0,_
 2397, 67,  172, 0,-77.777,+0, 1.704680e+02, 0,_
 2398, 67,  173, 0, 3.5,   -1, 1.714620e+02, 0,_
 2399, 67,  174, 0,-77.777,+0, 1.724570e+02, 0,_
 2400, 67,  175, 0, 3.5,   -1, 1.734510e+02, 0,_
 2401, 68,  143, 0, 4.5,   -1, 1.417380e+02, 0,_
 2402, 68,  144, 0, 0,     +1, 1.427240e+02, 0,_
 2403, 68,  145, 0, 0.5,   +1, 1.437120e+02, 0,_
 2404, 68,  146, 0, 0,     +1, 1.446980e+02, 0,_
 2405, 68,  147, 0, 0.5,   +1, 1.456870e+02, 0,_
 2406, 68,  147, 1, 5.5,   -1, 1.456870e+02, 0.1,_
 2407, 68,  148, 0, 0,     +1, 1.466740e+02, 0,_
 2408, 68,  149, 0, 0.5,   +1, 1.476630e+02, 0,_
 2409, 68,  149, 1, 5.5,   -1, 1.476630e+02, 0.7418,_
 2410, 68,  150, 0, 0,     +1, 1.486500e+02, 0,_
 2411, 68,  151, 0, 3.5,   -1, 1.496410e+02, 0,_
 2412, 68,  151, 1, 13.5,  -1, 1.496410e+02, 2.5855,_
 2413, 68,  152, 0, 0,     +1, 1.506300e+02, 0,_
 2414, 68,  153, 0, 3.5,   -1, 1.516210e+02, 0,_
 2415, 68,  154, 0, 0,     +1, 1.526100e+02, 0,_
 2416, 68,  155, 0, 3.5,   -1, 1.536020e+02, 0,_
 2417, 68,  156, 0, 0,     +1, 1.545920e+02, 0,_
 2418, 68,  157, 0, 1.5,   -1, 1.555840e+02, 0,_
 2419, 68,  157, 1, 4.5,   +1, 1.555840e+02, 0.1554,_
 2420, 68,  158, 0, 0,     +1, 1.565730e+02, 0,_
 2421, 68,  159, 0, 1.5,   -1, 1.575650e+02, 0,_
 2422, 68,  160, 0, 0,     +1, 1.585550e+02, 0,_
 2423, 68,  161, 0, 1.5,   -1, 1.595480e+02, 0,_
 2424, 68,  162, 0, 0,     +1, 1.605380e+02, 0,_
 2425, 68,  163, 0, 2.5,   -1, 1.615300e+02, 0,_
 2426, 68,  164, 0, 0,     +1, 1.625210e+02, 0,_
 2427, 68,  165, 0, 2.5,   -1, 1.635140e+02, 0,_
 2428, 68,  166, 0, 0,     +1, 1.645050e+02, 0,_
 2429, 68,  167, 0, 3.5,   +1, 1.654980e+02, 0,_
 2430, 68,  167, 1, 0.5,   -1, 1.654980e+02, 0.207801,_
 2431, 68,  168, 0, 0,     +1, 1.664900e+02, 0,_
 2432, 68,  169, 0, 0.5,   -1, 1.674830e+02, 0,_
 2433, 68,  170, 0, 0,     +1, 1.684760e+02, 0,_
 2434, 68,  171, 0, 2.5,   -1, 1.694700e+02, 0,_
 2435, 68,  172, 0, 0,     +1, 1.704620e+02, 0,_
 2436, 68,  173, 0, 3.5,   -1, 1.714570e+02, 0,_
 2437, 68,  174, 0, 0,     +1, 1.724500e+02, 0,_
 2438, 68,  175, 0, 4.5,   +1, 1.734450e+02, 0,_
 2439, 68,  176, 0, 0,     +1, 1.744390e+02, 0,_
 2440, 68,  177, 0, 0.5,   -1, 1.754340e+02, 0,_
 2441, 69,  145, 0, 5.5,   -1, 1.437250e+02, 0,_
 2442, 69,  146, 0, 6,     -1, 1.447120e+02, 0,_
 2443, 69,  146, 1, 10,    +1, 1.447120e+02, 0.071,_
 2444, 69,  147, 0, 5.5,   -1, 1.456980e+02, 0,_
 2445, 69,  148, 0, 10,    +1, 1.466870e+02, 0,_
 2446, 69,  149, 0, 5.5,   -1, 1.476730e+02, 0,_
 2447, 69,  150, 0, 1,     +1, 1.486620e+02, 0,_
 2448, 69,  150, 1, 6,     -1, 1.486620e+02, 0.14,_
 2449, 69,  150, 2, 10,    +1, 1.486620e+02, 0.81,_
 2450, 69,  151, 0, 5.5,   -1, 1.496490e+02, 0,_
 2451, 69,  151, 1, 0.5,   +1, 1.496490e+02, 0.05,_
 2452, 69,  152, 0, 2,     -1, 1.506390e+02, 0,_
 2453, 69,  152, 1, 9,     +1, 1.506390e+02, 0.1,_
 2454, 69,  153, 0, 5.5,   -1, 1.516280e+02, 0,_
 2455, 69,  153, 1, 0.5,   +1, 1.516280e+02, 0.0432,_
 2456, 69,  154, 0, 2,     -1, 1.526190e+02, 0,_
 2457, 69,  154, 1, 9,     +1, 1.526190e+02, 0.07,_
 2458, 69,  155, 0, 5.5,   -1, 1.536080e+02, 0,_
 2459, 69,  155, 1, 0.5,   +1, 1.536080e+02, 0.041,_
 2460, 69,  156, 0, 2,     -1, 1.545990e+02, 0,_
 2461, 69,  157, 0, 0.5,   +1, 1.555890e+02, 0,_
 2462, 69,  158, 0, 2,     -1, 1.565800e+02, 0,_
 2463, 69,  159, 0, 2.5,   +1, 1.575700e+02, 0,_
 2464, 69,  160, 0, 1,     -1, 1.585610e+02, 0,_
 2465, 69,  160, 1, 5,     +1, 1.585610e+02, 0.07,_
 2466, 69,  161, 0, 3.5,   +1, 1.595510e+02, 0,_
 2467, 69,  161, 1, 0.5,   +1, 1.595510e+02, 0.0074,_
 2468, 69,  162, 0, 1,     -1, 1.605430e+02, 0,_
 2469, 69,  162, 1, 5,     +1, 1.605430e+02, 0.13,_
 2470, 69,  163, 0, 0.5,   +1, 1.615330e+02, 0,_
 2471, 69,  164, 0, 1,     +1, 1.625250e+02, 0,_
 2472, 69,  164, 1, 6,     -1, 1.625250e+02, 0.01,_
 2473, 69,  165, 0, 0.5,   +1, 1.635160e+02, 0,_
 2474, 69,  166, 0, 2,     +1, 1.645080e+02, 0,_
 2475, 69,  166, 1, 6,     -1, 1.645080e+02, 0.122,_
 2476, 69,  167, 0, 0.5,   +1, 1.654990e+02, 0,_
 2477, 69,  168, 0, 3,     +1, 1.664920e+02, 0,_
 2478, 69,  169, 0, 0.5,   +1, 1.674830e+02, 0,_
 2479, 69,  170, 0, 1,     -1, 1.684760e+02, 0,_
 2480, 69,  171, 0, 0.5,   +1, 1.694680e+02, 0,_
 2481, 69,  172, 0, 2,     -1, 1.704610e+02, 0,_
 2482, 69,  173, 0, 0.5,   +1, 1.714540e+02, 0,_
 2483, 69,  174, 0, 4,     -1, 1.724480e+02, 0,_
 2484, 69,  175, 0, 0.5,   +1, 1.734410e+02, 0,_
 2485, 69,  176, 0, 4,     +1, 1.744360e+02, 0,_
 2486, 69,  177, 0, 3.5,   -1, 1.754290e+02, 0,_
 2487, 69,  178, 0,-77.777,+0, 1.764240e+02, 0,_
 2488, 69,  179, 0, 0.5,   +1, 1.774180e+02, 0,_
 2489, 70,  148, 0, 0,     +1, 1.466960e+02, 0,_
 2490, 70,  149, 0,-77.777,+0, 1.476840e+02, 0,_
 2491, 70,  150, 0, 0,     +1, 1.486700e+02, 0,_
 2492, 70,  151, 0, 0.5,   +1, 1.496590e+02, 0,_
 2493, 70,  151, 1, 5.5,   -1, 1.496590e+02, 0.75,_
 2494, 70,  152, 0, 0,     +1, 1.506450e+02, 0,_
 2495, 70,  153, 0, 3.5,   -1, 1.516360e+02, 0,_
 2496, 70,  154, 0, 0,     +1, 1.526240e+02, 0,_
 2497, 70,  155, 0, 3.5,   -1, 1.536150e+02, 0,_
 2498, 70,  156, 0, 0,     +1, 1.546030e+02, 0,_
 2499, 70,  157, 0, 3.5,   -1, 1.555940e+02, 0,_
 2500, 70,  158, 0, 0,     +1, 1.565830e+02, 0,_
 2501, 70,  159, 0, 2.5,   -1, 1.575750e+02, 0,_
 2502, 70,  160, 0, 0,     +1, 1.585640e+02, 0,_
 2503, 70,  161, 0, 1.5,   -1, 1.595550e+02, 0,_
 2504, 70,  162, 0, 0,     +1, 1.605450e+02, 0,_
 2505, 70,  163, 0, 1.5,   -1, 1.615370e+02, 0,_
 2506, 70,  164, 0, 0,     +1, 1.625260e+02, 0,_
 2507, 70,  165, 0, 2.5,   -1, 1.635180e+02, 0,_
 2508, 70,  166, 0, 0,     +1, 1.645080e+02, 0,_
 2509, 70,  167, 0, 2.5,   -1, 1.655010e+02, 0,_
 2510, 70,  168, 0, 0,     +1, 1.664910e+02, 0,_
 2511, 70,  169, 0, 3.5,   +1, 1.674840e+02, 0,_
 2512, 70,  169, 1, 0.5,   -1, 1.674840e+02, 0.024199,_
 2513, 70,  170, 0, 0,     +1, 1.684750e+02, 0,_
 2514, 70,  171, 0, 0.5,   -1, 1.694680e+02, 0,_
 2515, 70,  171, 1, 3.5,   +1, 1.694680e+02, 0.095282,_
 2516, 70,  172, 0, 0,     +1, 1.704590e+02, 0,_
 2517, 70,  173, 0, 2.5,   -1, 1.714530e+02, 0,_
 2518, 70,  174, 0, 0,     +1, 1.724450e+02, 0,_
 2519, 70,  175, 0, 3.5,   -1, 1.734380e+02, 0,_
 2520, 70,  175, 1, 0.5,   -1, 1.734380e+02, 0.514865,_
 2521, 70,  176, 0, 0,     +1, 1.744310e+02, 0,_
 2522, 70,  176, 1, 8,     -1, 1.744310e+02, 1.05,_
 2523, 70,  177, 0, 4.5,   +1, 1.754250e+02, 0,_
 2524, 70,  177, 1, 0.5,   -1, 1.754250e+02, 0.3315,_
 2525, 70,  178, 0, 0,     +1, 1.764180e+02, 0,_
 2526, 70,  179, 0, 0.5,   -1, 1.774130e+02, 0,_
 2527, 70,  180, 0, 0,     +1, 1.784060e+02, 0,_
 2528, 70,  181, 0, 1.5,   -1, 1.794020e+02, 0,_
 2529, 71,  150, 0,-77.777,+0, 1.486850e+02, 0,_
 2530, 71,  151, 0, 5.5,   -1, 1.496710e+02, 0,_
 2531, 71,  152, 0,-77.777,+0, 1.506590e+02, 0,_
 2532, 71,  153, 0, 5.5,   -1, 1.516450e+02, 0,_
 2533, 71,  153, 1, 0.5,   +1, 1.516450e+02, 0.08,_
 2534, 71,  154, 0, 2,     -1, 1.526350e+02, 0,_
 2535, 71,  154, 1, 9,     +1, 1.526350e+02, 0.058,_
 2536, 71,  155, 0, 5.5,   -1, 1.536230e+02, 0,_
 2537, 71,  155, 1, 0.5,   +1, 1.536230e+02, 0.02,_
 2538, 71,  155, 2, 12.5,  -1, 1.536230e+02, 1.781,_
 2539, 71,  156, 0, 2,     -1, 1.546130e+02, 0,_
 2540, 71,  156, 1, 9,     +1, 1.546130e+02, 0.22,_
 2541, 71,  157, 0,-77.777,+0, 1.556020e+02, 0,_
 2542, 71,  157, 1, 5.5,   -1, 1.556020e+02, 0.021,_
 2543, 71,  158, 0, 2,     -1, 1.565920e+02, 0,_
 2544, 71,  159, 0, 0.5,   +1, 1.575810e+02, 0,_
 2545, 71,  159, 1, 5.5,   -1, 1.575810e+02, 0.1,_
 2546, 71,  160, 0, 2,     -1, 1.585720e+02, 0,_
 2547, 71,  160, 1,-77.777,+0, 1.585720e+02, 0,_
 2548, 71,  161, 0, 0.5,   +1, 1.595610e+02, 0,_
 2549, 71,  161, 1, 4.5,   -1, 1.595610e+02, 0.166,_
 2550, 71,  162, 0, 1,     -1, 1.605520e+02, 0,_
 2551, 71,  162, 1, 4,     -1, 1.605520e+02, 0.12,_
 2552, 71,  162, 2,-77.777,+0, 1.605520e+02, 0.3,_
 2553, 71,  163, 0, 0.5,   +1, 1.615410e+02, 0,_
 2554, 71,  164, 0, 1,     -1, 1.625330e+02, 0,_
 2555, 71,  165, 0, 0.5,   +1, 1.635220e+02, 0,_
 2556, 71,  166, 0, 6,     -1, 1.645140e+02, 0,_
 2557, 71,  166, 1, 3,     -1, 1.645140e+02, 0.03437,_
 2558, 71,  166, 2, 0,     -1, 1.645140e+02, 0.0429,_
 2559, 71,  167, 0, 3.5,   +1, 1.655040e+02, 0,_
 2560, 71,  167, 1, 0.5,   -1, 1.655040e+02, 0,_
 2561, 71,  168, 0, 6,     -1, 1.664960e+02, 0,_
 2562, 71,  168, 1, 3,     +1, 1.664960e+02, 0.22,_
 2563, 71,  169, 0, 3.5,   +1, 1.674860e+02, 0,_
 2564, 71,  169, 1, 0.5,   -1, 1.674860e+02, 0.029,_
 2565, 71,  170, 0, 0,     +1, 1.684790e+02, 0,_
 2566, 71,  170, 1, 4,     -1, 1.684790e+02, 0.09291,_
 2567, 71,  171, 0, 3.5,   +1, 1.694690e+02, 0,_
 2568, 71,  171, 1, 0.5,   -1, 1.694690e+02, 0.0712,_
 2569, 71,  172, 0, 4,     -1, 1.704620e+02, 0,_
 2570, 71,  172, 1, 1,     -1, 1.704620e+02, 0.04186,_
 2571, 71,  173, 0, 3.5,   +1, 1.714530e+02, 0,_
 2572, 71,  174, 0, 1,     -1, 1.724460e+02, 0,_
 2573, 71,  174, 1, 6,     -1, 1.724460e+02, 0.17083,_
 2574, 71,  175, 0, 3.5,   +1, 1.734380e+02, 0,_
 2575, 71,  176, 0, 7,     -1, 1.744310e+02, 0,_
 2576, 71,  176, 1, 1,     -1, 1.744310e+02, 0.127,_
 2577, 71,  177, 0, 3.5,   +1, 1.754240e+02, 0,_
 2578, 71,  177, 1, 11.5,  -1, 1.754240e+02, 0.970175,_
 2579, 71,  177, 2, 19.5,  -1, 1.754240e+02, 3.9,_
 2580, 71,  178, 0, 1,     +1, 1.764170e+02, 0,_
 2581, 71,  178, 1, 9,     -1, 1.764170e+02, 0.1237,_
 2582, 71,  179, 0, 3.5,   +1, 1.774100e+02, 0,_
 2583, 71,  179, 1, 0.5,   +1, 1.774100e+02, 0.5924,_
 2584, 71,  180, 0, 3,     +1, 1.784040e+02, 0,_
 2585, 71,  180, 1, 3,     -1, 1.784040e+02, 0.0139,_
 2586, 71,  180, 2, 9,     -1, 1.784040e+02, 0.624,_
 2587, 71,  181, 0, 3.5,   +1, 1.793980e+02, 0,_
 2588, 71,  182, 0,-77.777,+0, 1.803920e+02, 0,_
 2589, 71,  183, 0, 3.5,   +1, 1.813860e+02, 0,_
 2590, 71,  184, 0, 3,     +1, 1.823810e+02, 0,_
 2591, 72,  153, 0, 0.5,   +1, 1.516570e+02, 0,_
 2592, 72,  153, 1, 5.5,   -1, 1.516570e+02, 0.75,_
 2593, 72,  154, 0, 0,     +1, 1.526420e+02, 0,_
 2594, 72,  155, 0, 3.5,   -1, 1.536320e+02, 0,_
 2595, 72,  156, 0, 0,     +1, 1.546200e+02, 0,_
 2596, 72,  157, 0, 3.5,   -1, 1.556100e+02, 0,_
 2597, 72,  158, 0, 0,     +1, 1.565980e+02, 0,_
 2598, 72,  159, 0, 3.5,   -1, 1.575890e+02, 0,_
 2599, 72,  160, 0, 0,     +1, 1.585770e+02, 0,_
 2600, 72,  161, 0, 1.5,   -1, 1.595680e+02, 0,_
 2601, 72,  162, 0, 0,     +1, 1.605560e+02, 0,_
 2602, 72,  163, 0, 1.5,   -1, 1.615470e+02, 0,_
 2603, 72,  164, 0, 0,     +1, 1.625360e+02, 0,_
 2604, 72,  165, 0, 2.5,   -1, 1.635280e+02, 0,_
 2605, 72,  166, 0, 0,     +1, 1.645170e+02, 0,_
 2606, 72,  167, 0, 2.5,   -1, 1.655080e+02, 0,_
 2607, 72,  168, 0, 0,     +1, 1.664980e+02, 0,_
 2608, 72,  169, 0, 2.5,   -1, 1.674900e+02, 0,_
 2609, 72,  170, 0, 0,     +1, 1.684800e+02, 0,_
 2610, 72,  171, 0, 3.5,   +1, 1.694720e+02, 0,_
 2611, 72,  171, 1, 0.5,   -1, 1.694720e+02, 0.02193,_
 2612, 72,  172, 0, 0,     +1, 1.704620e+02, 0,_
 2613, 72,  173, 0, 0.5,   -1, 1.714550e+02, 0,_
 2614, 72,  174, 0, 0,     +1, 1.724460e+02, 0,_
 2615, 72,  175, 0, 2.5,   -1, 1.734390e+02, 0,_
 2616, 72,  176, 0, 0,     +1, 1.744300e+02, 0,_
 2617, 72,  177, 0, 3.5,   -1, 1.754230e+02, 0,_
 2618, 72,  177, 1, 11.5,  +1, 1.754230e+02, 1.31545,_
 2619, 72,  177, 2, 18.5,  -1, 1.754230e+02, 2.74,_
 2620, 72,  178, 0, 0,     +1, 1.764150e+02, 0,_
 2621, 72,  178, 1, 8,     -1, 1.764150e+02, 1.14742,_
 2622, 72,  178, 2, 16,    +1, 1.764150e+02, 2.44605,_
 2623, 72,  179, 0, 4.5,   +1, 1.774090e+02, 0,_
 2624, 72,  179, 1, 0.5,   -1, 1.774090e+02, 0.375037,_
 2625, 72,  179, 2, 12.5,  -1, 1.774090e+02, 1.1058,_
 2626, 72,  180, 0, 0,     +1, 1.784010e+02, 0,_
 2627, 72,  180, 1, 8,     -1, 1.784010e+02, 1.14148,_
 2628, 72,  181, 0, 0.5,   -1, 1.793950e+02, 0,_
 2629, 72,  182, 0, 0,     +1, 1.803870e+02, 0,_
 2630, 72,  182, 1, 8,     -1, 1.803880e+02, 1.17288,_
 2631, 72,  183, 0, 1.5,   -1, 1.813820e+02, 0,_
 2632, 72,  184, 0, 0,     +1, 1.823750e+02, 0,_
 2633, 72,  184, 1, 8,     -1, 1.823750e+02, 1.2724,_
 2634, 72,  185, 0, 1.5,   -1, 1.833700e+02, 0,_
 2635, 72,  186, 0, 0,     +1, 1.843630e+02, 0,_
 2636, 72,  187, 0, 1.5,   -1, 1.853580e+02, 0,_
 2637, 72,  188, 0, 0,     +1, 1.863520e+02, 0,_
 2638, 73,  155, 0, 5.5,   -1, 1.536430e+02, 0,_
 2639, 73,  156, 0, 2,     -1, 1.546320e+02, 0,_
 2640, 73,  156, 1, 9,     +1, 1.546320e+02, 0.1,_
 2641, 73,  157, 0, 0.5,   +1, 1.556200e+02, 0,_
 2642, 73,  157, 1, 5.5,   -1, 1.556200e+02, 0.022,_
 2643, 73,  157, 2, 12.5,  -1, 1.556200e+02, 1.593,_
 2644, 73,  158, 0, 2,     -1, 1.566100e+02, 0,_
 2645, 73,  158, 1, 9,     +1, 1.566100e+02, 0.14,_
 2646, 73,  159, 0, 0.5,   +1, 1.575970e+02, 0,_
 2647, 73,  159, 1, 5.5,   -1, 1.575970e+02, 0.064,_
 2648, 73,  160, 0, 2,     -1, 1.585870e+02, 0,_
 2649, 73,  160, 1, 9,     +1, 1.585870e+02, 0.31,_
 2650, 73,  161, 0, 0.5,   +1, 1.595760e+02, 0,_
 2651, 73,  161, 1, 5.5,   -1, 1.595760e+02, 0.05,_
 2652, 73,  162, 0, 3,     +1, 1.605660e+02, 0,_
 2653, 73,  163, 0, 0.5,   +1, 1.615540e+02, 0,_
 2654, 73,  164, 0, 3,     +1, 1.625450e+02, 0,_
 2655, 73,  165, 0, 2.5,   -1, 1.635340e+02, 0,_
 2656, 73,  166, 0, 2,     +1, 1.645250e+02, 0,_
 2657, 73,  167, 0, 1.5,   +1, 1.655140e+02, 0,_
 2658, 73,  168, 0,-77.777,+0, 1.665050e+02, 0,_
 2659, 73,  169, 0, 2.5,   +1, 1.674950e+02, 0,_
 2660, 73,  170, 0, 3,     +1, 1.684860e+02, 0,_
 2661, 73,  171, 0, 2.5,   -1, 1.694760e+02, 0,_
 2662, 73,  172, 0, 3,     +1, 1.704680e+02, 0,_
 2663, 73,  173, 0, 2.5,   -1, 1.714580e+02, 0,_
 2664, 73,  174, 0, 3,     +1, 1.724500e+02, 0,_
 2665, 73,  175, 0, 3.5,   +1, 1.734410e+02, 0,_
 2666, 73,  176, 0, 1,     -1, 1.744330e+02, 0,_
 2667, 73,  176, 1, 0,     +1, 1.744330e+02, 0.103,_
 2668, 73,  176, 2, 20,    -1, 1.744330e+02, 2.82,_
 2669, 73,  177, 0, 3.5,   +1, 1.754240e+02, 0,_
 2670, 73,  178, 0, 1,     +1, 1.764170e+02, 0,_
 2671, 73,  178, 1, 7,     -1, 1.764170e+02, 0,_
 2672, 73,  178, 2, 15,    -1, 1.764170e+02, 1.4706,_
 2673, 73,  179, 0, 3.5,   +1, 1.774090e+02, 0,_
 2674, 73,  179, 1, 12.5,  +1, 1.774090e+02, 1.3173,_
 2675, 73,  179, 2, 18.5,  +1, 1.774090e+02, 2.6393,_
 2676, 73,  180, 0, 1,     +1, 1.784020e+02, 0,_
 2677, 73,  180, 1, 9,     -1, 1.784020e+02, 0.075,_
 2678, 73,  181, 0, 3.5,   +1, 1.793940e+02, 0,_
 2679, 73,  182, 0, 3,     -1, 1.803870e+02, 0,_
 2680, 73,  182, 1, 5,     +1, 1.803870e+02, 0.016263,_
 2681, 73,  182, 2, 10,    -1, 1.803870e+02, 0.519572,_
 2682, 73,  183, 0, 3.5,   +1, 1.813800e+02, 0,_
 2683, 73,  184, 0, 5,     -1, 1.823740e+02, 0,_
 2684, 73,  185, 0, 3.5,   +1, 1.833670e+02, 0,_
 2685, 73,  185, 1, 10.5,  -1, 1.833670e+02, 1.308,_
 2686, 73,  186, 0,-77.777,+0, 1.843610e+02, 0,_
 2687, 73,  187, 0, 3.5,   +1, 1.853540e+02, 0,_
 2688, 73,  188, 0,-77.777,+0, 1.863490e+02, 0,_
 2689, 73,  189, 0, 3.5,   +1, 1.873430e+02, 0,_
 2690, 73,  190, 0,-77.777,+0, 1.883370e+02, 0,_
 2691, 74,  158, 0, 0,     +1, 1.566170e+02, 0,_
 2692, 74,  159, 0, 3.5,   -1, 1.576070e+02, 0,_
 2693, 74,  160, 0, 0,     +1, 1.585940e+02, 0,_
 2694, 74,  161, 0, 3.5,   -1, 1.595850e+02, 0,_
 2695, 74,  162, 0, 0,     +1, 1.605720e+02, 0,_
 2696, 74,  163, 0, 1.5,   -1, 1.615630e+02, 0,_
 2697, 74,  164, 0, 0,     +1, 1.625500e+02, 0,_
 2698, 74,  165, 0, 1.5,   -1, 1.635410e+02, 0,_
 2699, 74,  166, 0, 0,     +1, 1.645290e+02, 0,_
 2700, 74,  167, 0, 1.5,   -1, 1.655210e+02, 0,_
 2701, 74,  168, 0, 0,     +1, 1.665090e+02, 0,_
 2702, 74,  169, 0, 2.5,   -1, 1.675000e+02, 0,_
 2703, 74,  170, 0, 0,     +1, 1.684890e+02, 0,_
 2704, 74,  171, 0, 2.5,   -1, 1.694810e+02, 0,_
 2705, 74,  172, 0, 0,     +1, 1.704700e+02, 0,_
 2706, 74,  173, 0, 2.5,   -1, 1.714620e+02, 0,_
 2707, 74,  174, 0, 0,     +1, 1.724520e+02, 0,_
 2708, 74,  175, 0, 0.5,   -1, 1.734440e+02, 0,_
 2709, 74,  176, 0, 0,     +1, 1.744340e+02, 0,_
 2710, 74,  177, 0, 0.5,   -1, 1.754270e+02, 0,_
 2711, 74,  178, 0, 0,     +1, 1.764170e+02, 0,_
 2712, 74,  179, 0, 3.5,   -1, 1.774100e+02, 0,_
 2713, 74,  179, 1, 0.5,   -1, 1.774100e+02, 0.221926,_
 2714, 74,  180, 0, 0,     +1, 1.784010e+02, 0,_
 2715, 74,  180, 1, 8,     -1, 1.784010e+02, 1.52901,_
 2716, 74,  181, 0, 4.5,   +1, 1.793940e+02, 0,_
 2717, 74,  182, 0, 0,     +1, 1.803850e+02, 0,_
 2718, 74,  183, 0, 0.5,   -1, 1.813790e+02, 0,_
 2719, 74,  183, 1, 5.5,   +1, 1.813790e+02, 0.3095,_
 2720, 74,  184, 0, 0,     +1, 1.823710e+02, 0,_
 2721, 74,  185, 0, 1.5,   -1, 1.833650e+02, 0,_
 2722, 74,  185, 1, 5.5,   +1, 1.833650e+02, 0.19743,_
 2723, 74,  186, 0, 0,     +1, 1.843570e+02, 0,_
 2724, 74,  186, 1, 16,    +1, 1.843570e+02, 3.5428,_
 2725, 74,  187, 0, 1.5,   -1, 1.853510e+02, 0,_
 2726, 74,  188, 0, 0,     +1, 1.863440e+02, 0,_
 2727, 74,  189, 0, 1.5,   -1, 1.873390e+02, 0,_
 2728, 74,  190, 0, 0,     +1, 1.883310e+02, 0,_
 2729, 74,  190, 1, 10,    -1, 1.883310e+02, 2.381,_
 2730, 74,  191, 0, 1.5,   -1, 1.893260e+02, 0,_
 2731, 74,  192, 0, 0,     +1, 1.903190e+02, 0,_
 2732, 75,  160, 0, 2,     -1, 1.586080e+02, 0,_
 2733, 75,  161, 0, 0.5,   +1, 1.595950e+02, 0,_
 2734, 75,  161, 1, 5.5,   -1, 1.595950e+02, 0.1238,_
 2735, 75,  162, 0, 2,     -1, 1.605850e+02, 0,_
 2736, 75,  162, 1, 9,     +1, 1.605850e+02, 0.173,_
 2737, 75,  163, 0, 0.5,   +1, 1.615720e+02, 0,_
 2738, 75,  163, 1, 5.5,   -1, 1.615720e+02, 0.115,_
 2739, 75,  164, 0,-77.777,+0, 1.625620e+02, 0,_
 2740, 75,  164, 1, 2,     -1, 1.625620e+02, 0.22,_
 2741, 75,  165, 0, 0.5,   +1, 1.635500e+02, 0,_
 2742, 75,  165, 1, 5.5,   -1, 1.635500e+02, 0.047,_
 2743, 75,  166, 0, 2,     -1, 1.645400e+02, 0,_
 2744, 75,  166, 1, 9,     +1, 1.645400e+02, 0.25,_
 2745, 75,  167, 0, 4.5,   -1, 1.655280e+02, 0,_
 2746, 75,  167, 1, 0.5,   +1, 1.655280e+02, 0.13,_
 2747, 75,  168, 0,-77.777,+0, 1.665190e+02, 0,_
 2748, 75,  169, 0, 4.5,   -1, 1.675070e+02, 0,_
 2749, 75,  169, 1, 0.5,   +1, 1.675070e+02, 0.16,_
 2750, 75,  170, 0, 5,     +1, 1.684980e+02, 0,_
 2751, 75,  171, 0, 4.5,   -1, 1.694870e+02, 0,_
 2752, 75,  172, 0,-77.777,+0, 1.704780e+02, 0,_
 2753, 75,  172, 1,-77.777,+0, 1.704780e+02, 0,_
 2754, 75,  173, 0, 2.5,   -1, 1.714670e+02, 0,_
 2755, 75,  174, 0,-77.777,+0, 1.724590e+02, 0,_
 2756, 75,  175, 0, 2.5,   -1, 1.734480e+02, 0,_
 2757, 75,  176, 0, 3,     +1, 1.744400e+02, 0,_
 2758, 75,  177, 0, 2.5,   -1, 1.754300e+02, 0,_
 2759, 75,  178, 0, 3,     +1, 1.764220e+02, 0,_
 2760, 75,  179, 0, 2.5,   +1, 1.774130e+02, 0,_
 2761, 75,  180, 0, 1,     -1, 1.784050e+02, 0,_
 2762, 75,  181, 0, 2.5,   +1, 1.793960e+02, 0,_
 2763, 75,  182, 0, 7,     +1, 1.803880e+02, 0,_
 2764, 75,  182, 1, 2,     +1, 1.803880e+02, 0.06,_
 2765, 75,  183, 0, 2.5,   +1, 1.813790e+02, 0,_
 2766, 75,  183, 1, 12.5,  +1, 1.813790e+02, 1.9076,_
 2767, 75,  184, 0, 3,     -1, 1.823720e+02, 0,_
 2768, 75,  184, 1, 8,     +1, 1.823720e+02, 0.18801,_
 2769, 75,  185, 0, 2.5,   +1, 1.833640e+02, 0,_
 2770, 75,  186, 0, 1,     -1, 1.843580e+02, 0,_
 2771, 75,  186, 1, 8,     +1, 1.843580e+02, 0.149,_
 2772, 75,  187, 0, 2.5,   +1, 1.853500e+02, 0,_
 2773, 75,  188, 0, 1,     -1, 1.863430e+02, 0,_
 2774, 75,  188, 1, 6,     -1, 1.863430e+02, 0.172069,_
 2775, 75,  189, 0, 2.5,   +1, 1.873360e+02, 0,_
 2776, 75,  190, 0, 2,     -1, 1.883300e+02, 0,_
 2777, 75,  190, 1, 6,     -1, 1.883300e+02, 0.21,_
 2778, 75,  191, 0, 1.5,   +1, 1.893230e+02, 0,_
 2779, 75,  192, 0, 1,     -1, 1.903170e+02, 0,_
 2780, 75,  193, 0, 2.5,   +1, 1.913100e+02, 0,_
 2781, 75,  194, 0,-77.777,+0, 1.923040e+02, 0,_
 2782, 76,  162, 0, 0,     +1, 1.605930e+02, 0,_
 2783, 76,  163, 0, 3.5,   -1, 1.615830e+02, 0,_
 2784, 76,  164, 0, 0,     +1, 1.625690e+02, 0,_
 2785, 76,  165, 0, 3.5,   -1, 1.635600e+02, 0,_
 2786, 76,  166, 0, 0,     +1, 1.645470e+02, 0,_
 2787, 76,  167, 0, 1.5,   -1, 1.655370e+02, 0,_
 2788, 76,  168, 0, 0,     +1, 1.665250e+02, 0,_
 2789, 76,  169, 0, 1.5,   -1, 1.675160e+02, 0,_
 2790, 76,  170, 0, 0,     +1, 1.685040e+02, 0,_
 2791, 76,  171, 0, 2.5,   -1, 1.694950e+02, 0,_
 2792, 76,  172, 0, 0,     +1, 1.704830e+02, 0,_
 2793, 76,  173, 0, 2.5,   -1, 1.714740e+02, 0,_
 2794, 76,  174, 0, 0,     +1, 1.724630e+02, 0,_
 2795, 76,  175, 0, 2.5,   -1, 1.734540e+02, 0,_
 2796, 76,  176, 0, 0,     +1, 1.744430e+02, 0,_
 2797, 76,  177, 0, 0.5,   -1, 1.754350e+02, 0,_
 2798, 76,  178, 0, 0,     +1, 1.764250e+02, 0,_
 2799, 76,  179, 0, 0.5,   -1, 1.774170e+02, 0,_
 2800, 76,  180, 0, 0,     +1, 1.784060e+02, 0,_
 2801, 76,  181, 0, 0.5,   -1, 1.793990e+02, 0,_
 2802, 76,  181, 1, 3.5,   -1, 1.793990e+02, 0.0489,_
 2803, 76,  182, 0, 0,     +1, 1.803890e+02, 0,_
 2804, 76,  183, 0, 4.5,   +1, 1.813810e+02, 0,_
 2805, 76,  183, 1, 0.5,   -1, 1.813810e+02, 0.17071,_
 2806, 76,  184, 0, 0,     +1, 1.823720e+02, 0,_
 2807, 76,  185, 0, 0.5,   -1, 1.833650e+02, 0,_
 2808, 76,  186, 0, 0,     +1, 1.843560e+02, 0,_
 2809, 76,  187, 0, 0.5,   -1, 1.853500e+02, 0,_
 2810, 76,  188, 0, 0,     +1, 1.863410e+02, 0,_
 2811, 76,  189, 0, 1.5,   -1, 1.873350e+02, 0,_
 2812, 76,  189, 1, 4.5,   -1, 1.873350e+02, 0.030812,_
 2813, 76,  190, 0, 0,     +1, 1.883270e+02, 0,_
 2814, 76,  190, 1, 10,    -1, 1.883270e+02, 1.7054,_
 2815, 76,  191, 0, 4.5,   -1, 1.893210e+02, 0,_
 2816, 76,  191, 1, 1.5,   -1, 1.893210e+02, 0.074382,_
 2817, 76,  192, 0, 0,     +1, 1.903120e+02, 0,_
 2818, 76,  192, 1, 10,    -1, 1.903120e+02, 2.0154,_
 2819, 76,  193, 0, 1.5,   -1, 1.913060e+02, 0,_
 2820, 76,  194, 0, 0,     +1, 1.922990e+02, 0,_
 2821, 76,  195, 0, 0.5,   -1, 1.932930e+02, 0,_
 2822, 76,  196, 0, 0,     +1, 1.942860e+02, 0,_
 2823, 77,  164, 0, 2,     -1, 1.625830e+02, 0,_
 2824, 77,  164, 1, 9,     +1, 1.625830e+02, 0.27,_
 2825, 77,  165, 0, 0.5,   +1, 1.635700e+02, 0,_
 2826, 77,  166, 0, 2,     -1, 1.645600e+02, 0,_
 2827, 77,  166, 1, 9,     +1, 1.645600e+02, 0.172,_
 2828, 77,  167, 0, 0.5,   +1, 1.655470e+02, 0,_
 2829, 77,  167, 1, 5.5,   -1, 1.655470e+02, 0.1753,_
 2830, 77,  168, 0,-77.777,+0, 1.665370e+02, 0,_
 2831, 77,  168, 1,-77.777,+0, 1.665370e+02, 0.15,_
 2832, 77,  169, 0, 0.5,   +1, 1.675250e+02, 0,_
 2833, 77,  169, 1, 5.5,   -1, 1.675250e+02, 0.154,_
 2834, 77,  170, 0,-77.777,+0, 1.685150e+02, 0,_
 2835, 77,  170, 1,-77.777,+0, 1.685150e+02, 0.3707,_
 2836, 77,  171, 0, 0.5,   +1, 1.695030e+02, 0,_
 2837, 77,  171, 1, 5.5,   -1, 1.695030e+02, 0.18,_
 2838, 77,  172, 0, 3,     +1, 1.704930e+02, 0,_
 2839, 77,  172, 1, 7,     +1, 1.704930e+02, 0.139,_
 2840, 77,  173, 0,-77.777,+0, 1.714820e+02, 0,_
 2841, 77,  173, 1, 5.5,   -1, 1.714820e+02, 0.25,_
 2842, 77,  174, 0, 3,     +1, 1.724720e+02, 0,_
 2843, 77,  174, 1, 7,     +1, 1.724720e+02, 0.193,_
 2844, 77,  175, 0, 2.5,   -1, 1.734610e+02, 0,_
 2845, 77,  176, 0,-77.777,+0, 1.744520e+02, 0,_
 2846, 77,  177, 0, 2.5,   -1, 1.754410e+02, 0,_
 2847, 77,  178, 0,-77.777,+0, 1.764320e+02, 0,_
 2848, 77,  179, 0, 2.5,   -1, 1.774220e+02, 0,_
 2849, 77,  180, 0,-77.777,+0, 1.784130e+02, 0,_
 2850, 77,  181, 0, 2.5,   -1, 1.794030e+02, 0,_
 2851, 77,  182, 0, 3,     +1, 1.803950e+02, 0,_
 2852, 77,  183, 0, 2.5,   -1, 1.813850e+02, 0,_
 2853, 77,  184, 0, 5,     -1, 1.823770e+02, 0,_
 2854, 77,  185, 0, 2.5,   -1, 1.833680e+02, 0,_
 2855, 77,  186, 0, 5,     +1, 1.843600e+02, 0,_
 2856, 77,  186, 1, 2,     -1, 1.843600e+02, 0.0008,_
 2857, 77,  187, 0, 1.5,   +1, 1.853510e+02, 0,_
 2858, 77,  187, 1, 4.5,   -1, 1.853510e+02, 0.18615,_
 2859, 77,  188, 0, 1,     -1, 1.863440e+02, 0,_
 2860, 77,  188, 1, 7,     +1, 1.863440e+02, 0.97,_
 2861, 77,  189, 0, 1.5,   +1, 1.873350e+02, 0,_
 2862, 77,  189, 1, 5.5,   -1, 1.873350e+02, 0.37217,_
 2863, 77,  189, 2, 12.5,  +1, 1.873350e+02, 2.3332,_
 2864, 77,  190, 0, 4,     -1, 1.883290e+02, 0,_
 2865, 77,  190, 1, 1,     -1, 1.883290e+02, 0.0261,_
 2866, 77,  190, 2, 11,    -1, 1.883290e+02, 0.3764,_
 2867, 77,  191, 0, 1.5,   +1, 1.893200e+02, 0,_
 2868, 77,  191, 1, 5.5,   -1, 1.893200e+02, 0.17124,_
 2869, 77,  191, 2, 13.5,  -1, 1.893200e+02, 2.085,_
 2870, 77,  192, 0, 4,     +1, 1.903140e+02, 0,_
 2871, 77,  192, 1, 1,     -1, 1.903140e+02, 0.05672,_
 2872, 77,  192, 2, 11,    -1, 1.903140e+02, 0.16814,_
 2873, 77,  193, 0, 1.5,   +1, 1.913050e+02, 0,_
 2874, 77,  193, 1, 5.5,   -1, 1.913050e+02, 0.080236,_
 2875, 77,  194, 0, 1,     -1, 1.922990e+02, 0,_
 2876, 77,  194, 1, 4,     +1, 1.922990e+02, 0.147078,_
 2877, 77,  194, 2,-77.777,+0, 1.922990e+02, 0.37,_
 2878, 77,  195, 0, 1.5,   +1, 1.932910e+02, 0,_
 2879, 77,  195, 1, 5.5,   -1, 1.932910e+02, 0.123,_
 2880, 77,  196, 0, 0,     -1, 1.942850e+02, 0,_
 2881, 77,  196, 1,-77.777,+0, 1.942850e+02, 0.21,_
 2882, 77,  197, 0, 1.5,   +1, 1.952780e+02, 0,_
 2883, 77,  197, 1, 5.5,   -1, 1.952780e+02, 0.115,_
 2884, 77,  198, 0,-77.777,+0, 1.962720e+02, 0,_
 2885, 77,  199, 0, 1.5,   +1, 1.972650e+02, 0,_
 2886, 78,  166, 0, 0,     +1, 1.645690e+02, 0,_
 2887, 78,  167, 0, 3.5,   -1, 1.655580e+02, 0,_
 2888, 78,  168, 0, 0,     +1, 1.665450e+02, 0,_
 2889, 78,  169, 0, 1.5,   -1, 1.675350e+02, 0,_
 2890, 78,  170, 0, 0,     +1, 1.685220e+02, 0,_
 2891, 78,  171, 0, 1.5,   -1, 1.695120e+02, 0,_
 2892, 78,  172, 0, 0,     +1, 1.705000e+02, 0,_
 2893, 78,  173, 0, 2.5,   -1, 1.714900e+02, 0,_
 2894, 78,  174, 0, 0,     +1, 1.724780e+02, 0,_
 2895, 78,  175, 0, 2.5,   -1, 1.734690e+02, 0,_
 2896, 78,  176, 0, 0,     +1, 1.744570e+02, 0,_
 2897, 78,  177, 0, 2.5,   -1, 1.754480e+02, 0,_
 2898, 78,  178, 0, 0,     +1, 1.764370e+02, 0,_
 2899, 78,  179, 0, 0.5,   -1, 1.774280e+02, 0,_
 2900, 78,  180, 0, 0,     +1, 1.784170e+02, 0,_
 2901, 78,  181, 0, 0.5,   -1, 1.794090e+02, 0,_
 2902, 78,  182, 0, 0,     +1, 1.803980e+02, 0,_
 2903, 78,  183, 0, 0.5,   -1, 1.813900e+02, 0,_
 2904, 78,  183, 1, 3.5,   -1, 1.813900e+02, 0.0345,_
 2905, 78,  184, 0, 0,     +1, 1.823800e+02, 0,_
 2906, 78,  184, 1, 8,     -1, 1.823800e+02, 1.8394,_
 2907, 78,  185, 0, 4.5,   +1, 1.833720e+02, 0,_
 2908, 78,  185, 1, 0.5,   -1, 1.833720e+02, 0.1034,_
 2909, 78,  186, 0, 0,     +1, 1.843620e+02, 0,_
 2910, 78,  187, 0, 1.5,   -1, 1.853540e+02, 0,_
 2911, 78,  188, 0, 0,     +1, 1.863450e+02, 0,_
 2912, 78,  189, 0, 1.5,   -1, 1.873380e+02, 0,_
 2913, 78,  190, 0, 0,     +1, 1.883280e+02, 0,_
 2914, 78,  191, 0, 1.5,   -1, 1.893210e+02, 0,_
 2915, 78,  192, 0, 0,     +1, 1.903120e+02, 0,_
 2916, 78,  193, 0, 0.5,   -1, 1.913050e+02, 0,_
 2917, 78,  193, 1, 6.5,   +1, 1.913050e+02, 0.14978,_
 2918, 78,  194, 0, 0,     +1, 1.922960e+02, 0,_
 2919, 78,  195, 0, 0.5,   -1, 1.932900e+02, 0,_
 2920, 78,  195, 1, 6.5,   +1, 1.932900e+02, 0.2593,_
 2921, 78,  196, 0, 0,     +1, 1.942820e+02, 0,_
 2922, 78,  197, 0, 0.5,   -1, 1.952750e+02, 0,_
 2923, 78,  197, 1, 6.5,   +1, 1.952750e+02, 0.39959,_
 2924, 78,  198, 0, 0,     +1, 1.962670e+02, 0,_
 2925, 78,  199, 0, 2.5,   -1, 1.972610e+02, 0,_
 2926, 78,  199, 1, 6.5,   +1, 1.972610e+02, 0.424,_
 2927, 78,  200, 0, 0,     +1, 1.982540e+02, 0,_
 2928, 78,  201, 0, 2.5,   -1, 1.992480e+02, 0,_
 2929, 78,  202, 0, 0,     +1, 2.002410e+02, 0,_
 2930, 79,  169, 0, 0.5,   +1, 1.675460e+02, 0,_
 2931, 79,  170, 0, 2,     -1, 1.685360e+02, 0,_
 2932, 79,  170, 1, 9,     +1, 1.685360e+02, 0.274,_
 2933, 79,  171, 0, 0.5,   +1, 1.695230e+02, 0,_
 2934, 79,  171, 1, 5.5,   -1, 1.695230e+02, 0.25,_
 2935, 79,  172, 0,-77.777,+0, 1.705130e+02, 0,_
 2936, 79,  173, 0, 0.5,   +1, 1.715000e+02, 0,_
 2937, 79,  173, 1, 5.5,   -1, 1.715000e+02, 0.214,_
 2938, 79,  174, 0,-77.777,+0, 1.724900e+02, 0,_
 2939, 79,  174, 1,-77.777,+0, 1.724900e+02, 0.46,_
 2940, 79,  175, 0, 0.5,   +1, 1.734780e+02, 0,_
 2941, 79,  175, 1, 5.5,   -1, 1.734780e+02, 0.2,_
 2942, 79,  176, 0, 5,     -1, 1.744680e+02, 0,_
 2943, 79,  176, 1, 7,     +1, 1.744680e+02, 0.15,_
 2944, 79,  177, 0,-77.777,+0, 1.754570e+02, 0,_
 2945, 79,  177, 1, 5.5,   -1, 1.754570e+02, 0.21,_
 2946, 79,  178, 0,-77.777,+0, 1.764470e+02, 0,_
 2947, 79,  179, 0, 2.5,   -1, 1.774360e+02, 0,_
 2948, 79,  180, 0,-77.777,+0, 1.784260e+02, 0,_
 2949, 79,  181, 0, 1.5,   -1, 1.794150e+02, 0,_
 2950, 79,  182, 0, 2,     +1, 1.804060e+02, 0,_
 2951, 79,  183, 0, 2.5,   -1, 1.813960e+02, 0,_
 2952, 79,  184, 0, 5,     +1, 1.823870e+02, 0,_
 2953, 79,  184, 1, 2,     +1, 1.823870e+02, 0.06846,_
 2954, 79,  185, 0, 2.5,   -1, 1.833770e+02, 0,_
 2955, 79,  186, 0, 3,     -1, 1.843680e+02, 0,_
 2956, 79,  187, 0, 0.5,   +1, 1.853580e+02, 0,_
 2957, 79,  187, 1, 4.5,   -1, 1.853580e+02, 0.12051,_
 2958, 79,  188, 0, 1,     -1, 1.863510e+02, 0,_
 2959, 79,  189, 0, 0.5,   +1, 1.873410e+02, 0,_
 2960, 79,  189, 1, 5.5,   -1, 1.873410e+02, 0.24723,_
 2961, 79,  190, 0, 1,     -1, 1.883330e+02, 0,_
 2962, 79,  190, 1, 11,    -1, 1.883330e+02, 0.2,_
 2963, 79,  191, 0, 1.5,   +1, 1.893230e+02, 0,_
 2964, 79,  191, 1, 5.5,   -1, 1.893230e+02, 0.2662,_
 2965, 79,  192, 0, 1,     -1, 1.903160e+02, 0,_
 2966, 79,  192, 1, 5,     +1, 1.903160e+02, 0.13541,_
 2967, 79,  192, 2, 11,    -1, 1.903160e+02, 0.4316,_
 2968, 79,  193, 0, 1.5,   +1, 1.913060e+02, 0,_
 2969, 79,  193, 1, 5.5,   -1, 1.913060e+02, 0.29019,_
 2970, 79,  194, 0, 1,     -1, 1.922990e+02, 0,_
 2971, 79,  194, 1, 5,     +1, 1.922990e+02, 0.1074,_
 2972, 79,  194, 2, 11,    -1, 1.922990e+02, 0.4758,_
 2973, 79,  195, 0, 1.5,   +1, 1.932800e+02, 0,_
 2974, 79,  195, 1, 5.5,   -1, 1.932800e+02, 0.31858,_
 2975, 79,  196, 0, 2,     -1, 1.942830e+02, 0,_
 2976, 79,  196, 1, 5,     +1, 1.942830e+02, 0.08466,_
 2977, 79,  196, 2, 12,    -1, 1.942830e+02, 0.59566,_
 2978, 79,  197, 0, 1.5,   +1, 1.952750e+02, 0,_
 2979, 79,  197, 1, 5.5,   -1, 1.952750e+02, 0.40915,_
 2980, 79,  198, 0, 2,     -1, 1.962680e+02, 0,_
 2981, 79,  198, 1, 12,    -1, 1.962680e+02, 0.812,_
 2982, 79,  199, 0, 1.5,   +1, 1.972600e+02, 0,_
 2983, 79,  200, 0, 1,     -1, 1.982530e+02, 0,_
 2984, 79,  200, 1, 12,    -1, 1.982530e+02, 0.96,_
 2985, 79,  201, 0, 1.5,   +1, 1.992450e+02, 0,_
 2986, 79,  202, 0, 1,     -1, 2.002390e+02, 0,_
 2987, 79,  203, 0, 1.5,   +1, 2.012310e+02, 0,_
 2988, 79,  204, 0, 2,     -1, 2.022250e+02, 0,_
 2989, 79,  205, 0, 1.5,   +1, 2.032190e+02, 0,_
 2990, 80,  171, 0, 1.5,   -1, 1.695350e+02, 0,_
 2991, 80,  172, 0, 0,     +1, 1.705210e+02, 0,_
 2992, 80,  173, 0, 1.5,   -1, 1.715110e+02, 0,_
 2993, 80,  174, 0, 0,     +1, 1.724980e+02, 0,_
 2994, 80,  175, 0, 2.5,   -1, 1.734880e+02, 0,_
 2995, 80,  176, 0, 0,     +1, 1.744760e+02, 0,_
 2996, 80,  177, 0, 6.5,   +1, 1.754660e+02, 0,_
 2997, 80,  178, 0, 0,     +1, 1.764540e+02, 0,_
 2998, 80,  179, 0, 2.5,   -1, 1.774440e+02, 0,_
 2999, 80,  180, 0, 0,     +1, 1.784320e+02, 0,_
 3000, 80,  181, 0, 0.5,   -1, 1.794230e+02, 0,_
 3001, 80,  182, 0, 0,     +1, 1.804110e+02, 0,_
 3002, 80,  183, 0, 0.5,   -1, 1.814030e+02, 0,_
 3003, 80,  183, 1, 6.5,   +1, 1.814030e+02, 0.24,_
 3004, 80,  184, 0, 0,     +1, 1.823910e+02, 0,_
 3005, 80,  185, 0, 0.5,   -1, 1.833830e+02, 0,_
 3006, 80,  185, 1, 6.5,   +1, 1.833830e+02, 0.1038,_
 3007, 80,  186, 0, 0,     +1, 1.843720e+02, 0,_
 3008, 80,  187, 0, 1.5,   -1, 1.853640e+02, 0,_
 3009, 80,  187, 1, 6.5,   +1, 1.853640e+02, 0.03,_
 3010, 80,  188, 0, 0,     +1, 1.863530e+02, 0,_
 3011, 80,  189, 0, 1.5,   -1, 1.873450e+02, 0,_
 3012, 80,  189, 1, 6.5,   +1, 1.873450e+02, 0.1,_
 3013, 80,  190, 0, 0,     +1, 1.883340e+02, 0,_
 3014, 80,  191, 0, 1.5,   -1, 1.893270e+02, 0,_
 3015, 80,  191, 1, 6.5,   +1, 1.893270e+02, 0.128,_
 3016, 80,  192, 0, 0,     +1, 1.903170e+02, 0,_
 3017, 80,  193, 0, 1.5,   -1, 1.913090e+02, 0,_
 3018, 80,  193, 1, 6.5,   +1, 1.913090e+02, 0.14076,_
 3019, 80,  194, 0, 0,     +1, 1.922990e+02, 0,_
 3020, 80,  195, 0, 0.5,   -1, 1.932920e+02, 0,_
 3021, 80,  195, 1, 6.5,   +1, 1.932920e+02, 0.17607,_
 3022, 80,  196, 0, 0,     +1, 1.942820e+02, 0,_
 3023, 80,  197, 0, 0.5,   -1, 1.952750e+02, 0,_
 3024, 80,  197, 1, 6.5,   +1, 1.952750e+02, 0.29893,_
 3025, 80,  198, 0, 0,     +1, 1.962660e+02, 0,_
 3026, 80,  199, 0, 0.5,   -1, 1.972590e+02, 0,_
 3027, 80,  199, 1, 6.5,   +1, 1.972590e+02, 0.5325,_
 3028, 80,  200, 0, 0,     +1, 1.982510e+02, 0,_
 3029, 80,  201, 0, 1.5,   -1, 1.992440e+02, 0,_
 3030, 80,  202, 0, 0,     +1, 2.002360e+02, 0,_
 3031, 80,  203, 0, 2.5,   -1, 2.012290e+02, 0,_
 3032, 80,  204, 0, 0,     +1, 2.022210e+02, 0,_
 3033, 80,  205, 0, 0.5,   -1, 2.032150e+02, 0,_
 3034, 80,  205, 1, 6.5,   +1, 2.032150e+02, 1.5564,_
 3035, 80,  206, 0, 0,     +1, 2.042080e+02, 0,_
 3036, 80,  207, 0, 4.5,   +1, 2.052050e+02, 0,_
 3037, 80,  208, 0, 0,     +1, 2.061990e+02, 0,_
 3038, 80,  209, 0, 4.5,   +1, 2.071960e+02, 0,_
 3039, 80,  210, 0, 0,     +1, 2.081910e+02, 0,_
 3040, 81,  176, 0,-77.777,+0, 1.744890e+02, 0,_
 3041, 81,  177, 0, 0.5,   +1, 1.754760e+02, 0,_
 3042, 81,  178, 0,-77.777,+0, 1.764660e+02, 0,_
 3043, 81,  179, 0, 0.5,   +1, 1.774530e+02, 0,_
 3044, 81,  179, 1, 4.5,   -1, 1.774530e+02, 0.86,_
 3045, 81,  180, 0,-77.777,+0, 1.784440e+02, 0,_
 3046, 81,  181, 0, 0.5,   +1, 1.794310e+02, 0,_
 3047, 81,  181, 1, 4.5,   -1, 1.794310e+02, 0.85,_
 3048, 81,  182, 0, 2,     -1, 1.804220e+02, 0,_
 3049, 81,  182, 1, 7,     +1, 1.804220e+02, 0.1,_
 3050, 81,  183, 0, 0.5,   +1, 1.814100e+02, 0,_
 3051, 81,  183, 1, 4.5,   -1, 1.814100e+02, 0.642,_
 3052, 81,  184, 0, 2,     -1, 1.824010e+02, 0,_
 3053, 81,  184, 1, 7,     +1, 1.824010e+02, 0.1,_
 3054, 81,  185, 0, 0.5,   +1, 1.833900e+02, 0,_
 3055, 81,  185, 1, 4.5,   -1, 1.833900e+02, 0.4528,_
 3056, 81,  186, 0, 2,     -1, 1.843810e+02, 0,_
 3057, 81,  186, 1, 10,    -1, 1.843810e+02, 0.374,_
 3058, 81,  186, 2, 10,    -1, 1.843810e+02, 0.52,_
 3059, 81,  187, 0, 0.5,   +1, 1.853700e+02, 0,_
 3060, 81,  187, 1, 4.5,   -1, 1.853700e+02, 0.333,_
 3061, 81,  188, 0, 7,     +1, 1.863610e+02, 0,_
 3062, 81,  188, 1, 7,     +1, 1.863610e+02, 0.1,_
 3063, 81,  188, 2, 9,     -1, 1.863610e+02, 0.18,_
 3064, 81,  189, 0, 0.5,   +1, 1.873500e+02, 0,_
 3065, 81,  189, 1, 4.5,   -1, 1.873500e+02, 0.2576,_
 3066, 81,  190, 0, 2,     -1, 1.883420e+02, 0,_
 3067, 81,  190, 1, 7,     +1, 1.883420e+02, 0.13,_
 3068, 81,  191, 0, 0.5,   +1, 1.893310e+02, 0,_
 3069, 81,  191, 1, 4.5,   -1, 1.893310e+02, 0.297,_
 3070, 81,  192, 0, 2,     -1, 1.903230e+02, 0,_
 3071, 81,  192, 1, 7,     +1, 1.903230e+02, 0.16,_
 3072, 81,  193, 0, 0.5,   +1, 1.913130e+02, 0,_
 3073, 81,  193, 1, 4.5,   -1, 1.913130e+02, 0.369,_
 3074, 81,  194, 0, 2,     -1, 1.923050e+02, 0,_
 3075, 81,  194, 1, 7,     +1, 1.923050e+02, 0.3,_
 3076, 81,  195, 0, 0.5,   +1, 1.932950e+02, 0,_
 3077, 81,  195, 1, 4.5,   -1, 1.932950e+02, 0.48263,_
 3078, 81,  196, 0, 2,     -1, 1.942870e+02, 0,_
 3079, 81,  196, 1, 7,     +1, 1.942870e+02, 0.3942,_
 3080, 81,  197, 0, 0.5,   +1, 1.952770e+02, 0,_
 3081, 81,  197, 1, 4.5,   -1, 1.952770e+02, 0.6083,_
 3082, 81,  198, 0, 2,     -1, 1.962700e+02, 0,_
 3083, 81,  198, 1, 7,     +1, 1.962700e+02, 0.5435,_
 3084, 81,  198, 2, 10,    -1, 1.962700e+02, 0.7423,_
 3085, 81,  199, 0, 0.5,   +1, 1.972610e+02, 0,_
 3086, 81,  199, 1, 4.5,   -1, 1.972610e+02, 0.7495,_
 3087, 81,  200, 0, 2,     -1, 1.982530e+02, 0,_
 3088, 81,  200, 1, 7,     +1, 1.982530e+02, 0.7536,_
 3089, 81,  201, 0, 0.5,   +1, 1.992440e+02, 0,_
 3090, 81,  201, 1, 4.5,   -1, 1.992440e+02, 0.9195,_
 3091, 81,  202, 0, 2,     -1, 2.002370e+02, 0,_
 3092, 81,  203, 0, 0.5,   +1, 2.012290e+02, 0,_
 3093, 81,  204, 0, 2,     -1, 2.022220e+02, 0,_
 3094, 81,  205, 0, 0.5,   +1, 2.032140e+02, 0,_
 3095, 81,  206, 0, 0,     -1, 2.042070e+02, 0,_
 3096, 81,  206, 1, 12,    -1, 2.042070e+02, 2.643,_
 3097, 81,  207, 0, 0.5,   +1, 2.051990e+02, 0,_
 3098, 81,  207, 1, 5.5,   -1, 2.051990e+02, 1.341,_
 3099, 81,  208, 0, 5,     +1, 2.061950e+02, 0,_
 3100, 81,  209, 0, 0.5,   +1, 2.071900e+02, 0,_
 3101, 81,  210, 0, 5,     +1, 2.081860e+02, 0,_
 3102, 81,  211, 0, 0.5,   +1, 2.091810e+02, 0,_
 3103, 81,  212, 0, 5,     +1, 2.101770e+02, 0,_
 3104, 82,  178, 0, 0,     +1, 1.764750e+02, 0,_
 3105, 82,  179, 0, 2.5,   -1, 1.774640e+02, 0,_
 3106, 82,  180, 0, 0,     +1, 1.784520e+02, 0,_
 3107, 82,  181, 0, 2.5,   -1, 1.794420e+02, 0,_
 3108, 82,  182, 0, 0,     +1, 1.804290e+02, 0,_
 3109, 82,  183, 0, 1.5,   -1, 1.814200e+02, 0,_
 3110, 82,  183, 1, 6.5,   +1, 1.814200e+02, 0.094,_
 3111, 82,  184, 0, 0,     +1, 1.824080e+02, 0,_
 3112, 82,  185, 0, 1.5,   -1, 1.833980e+02, 0,_
 3113, 82,  185, 1, 6.5,   +1, 1.833980e+02, 0.06,_
 3114, 82,  186, 0, 0,     +1, 1.843870e+02, 0,_
 3115, 82,  187, 0, 1.5,   -1, 1.853780e+02, 0,_
 3116, 82,  187, 1, 6.5,   +1, 1.853780e+02, 0.011,_
 3117, 82,  188, 0, 0,     +1, 1.863660e+02, 0,_
 3118, 82,  189, 0, 1.5,   -1, 1.873570e+02, 0,_
 3119, 82,  189, 1, 6.5,   +1, 1.873570e+02, 0.04,_
 3120, 82,  190, 0, 0,     +1, 1.883460e+02, 0,_
 3121, 82,  191, 0, 1.5,   -1, 1.893380e+02, 0,_
 3122, 82,  191, 1, 6.5,   +1, 1.893380e+02, 0,_
 3123, 82,  192, 0, 0,     +1, 1.903270e+02, 0,_
 3124, 82,  193, 0, 1.5,   -1, 1.913180e+02, 0,_
 3125, 82,  193, 1, 6.5,   +1, 1.913180e+02, 0.13,_
 3126, 82,  194, 0, 0,     +1, 1.923080e+02, 0,_
 3127, 82,  195, 0, 1.5,   -1, 1.933000e+02, 0,_
 3128, 82,  195, 1, 6.5,   +1, 1.933000e+02, 0.2029,_
 3129, 82,  196, 0, 0,     +1, 1.942890e+02, 0,_
 3130, 82,  197, 0, 1.5,   -1, 1.952810e+02, 0,_
 3131, 82,  197, 1, 6.5,   +1, 1.952810e+02, 0.31931,_
 3132, 82,  198, 0, 0,     +1, 1.962710e+02, 0,_
 3133, 82,  199, 0, 1.5,   -1, 1.972640e+02, 0,_
 3134, 82,  199, 1, 6.5,   +1, 1.972640e+02, 0.4295,_
 3135, 82,  200, 0, 0,     +1, 1.982540e+02, 0,_
 3136, 82,  201, 0, 2.5,   -1, 1.992460e+02, 0,_
 3137, 82,  201, 1, 6.5,   +1, 1.992460e+02, 0.6291,_
 3138, 82,  202, 0, 0,     +1, 2.002370e+02, 0,_
 3139, 82,  202, 1, 9,     -1, 2.002370e+02, 2.16984,_
 3140, 82,  203, 0, 2.5,   -1, 2.012300e+02, 0,_
 3141, 82,  203, 1, 6.5,   +1, 2.012300e+02, 0.8252,_
 3142, 82,  203, 2, 14.5,  -1, 2.012300e+02, 2.9495,_
 3143, 82,  204, 0, 0,     +1, 2.022210e+02, 0,_
 3144, 82,  204, 1, 9,     -1, 2.022210e+02, 2.18565,_
 3145, 82,  205, 0, 2.5,   -1, 2.032140e+02, 0,_
 3146, 82,  205, 1, 6.5,   +1, 2.032140e+02, 1.01384,_
 3147, 82,  206, 0, 0,     +1, 2.042050e+02, 0,_
 3148, 82,  207, 0, 0.5,   -1, 2.051980e+02, 0,_
 3149, 82,  207, 1, 6.5,   +1, 2.051980e+02, 1.63337,_
 3150, 82,  208, 0, 0,     +1, 2.061900e+02, 0,_
 3151, 82,  209, 0, 4.5,   +1, 2.071860e+02, 0,_
 3152, 82,  210, 0, 0,     +1, 2.081800e+02, 0,_
 3153, 82,  211, 0, 4.5,   +1, 2.091760e+02, 0,_
 3154, 82,  212, 0, 0,     +1, 2.101710e+02, 0,_
 3155, 82,  213, 0, 4.5,   +1, 2.111670e+02, 0,_
 3156, 82,  214, 0, 0,     +1, 2.121610e+02, 0,_
 3157, 82,  215, 0, 2.5,   +1, 2.131580e+02, 0,_
 3158, 83,  184, 0, 3,     +1, 1.824200e+02, 0,_
 3159, 83,  184, 1, 10,    -1, 1.824200e+02, 0.15,_
 3160, 83,  185, 0, 4.5,   -1, 1.834080e+02, 0,_
 3161, 83,  186, 0, 3,     +1, 1.843990e+02, 0,_
 3162, 83,  186, 1, 10,    -1, 1.843990e+02, 0.25,_
 3163, 83,  187, 0, 4.5,   -1, 1.853870e+02, 0,_
 3164, 83,  187, 1, 0.5,   +1, 1.853870e+02, 0.102,_
 3165, 83,  188, 0, 3,     +1, 1.863770e+02, 0,_
 3166, 83,  188, 1, 10,    -1, 1.863770e+02, 0.21,_
 3167, 83,  189, 0, 4.5,   -1, 1.873660e+02, 0,_
 3168, 83,  189, 1, 0.5,   +1, 1.873660e+02, 0.185,_
 3169, 83,  190, 0, 3,     +1, 1.883560e+02, 0,_
 3170, 83,  190, 1, 10,    -1, 1.883560e+02, 0.25,_
 3171, 83,  191, 0, 4.5,   -1, 1.893450e+02, 0,_
 3172, 83,  191, 1, 0.5,   +1, 1.893450e+02, 0.244,_
 3173, 83,  192, 0, 3,     +1, 1.903360e+02, 0,_
 3174, 83,  192, 1, 10,    -1, 1.903360e+02, 0.02,_
 3175, 83,  193, 0, 4.5,   -1, 1.913250e+02, 0,_
 3176, 83,  193, 1, 0.5,   +1, 1.913250e+02, 0.308,_
 3177, 83,  194, 0, 3,     +1, 1.923160e+02, 0,_
 3178, 83,  194, 1,-77.777,+0, 1.923160e+02, 0.11,_
 3179, 83,  194, 2, 10,    -1, 1.923160e+02, 0.23,_
 3180, 83,  195, 0, 4.5,   -1, 1.933060e+02, 0,_
 3181, 83,  195, 1, 0.5,   +1, 1.933060e+02, 0.399,_
 3182, 83,  196, 0, 3,     +1, 1.942970e+02, 0,_
 3183, 83,  196, 1, 7,     +1, 1.942970e+02, 0.167,_
 3184, 83,  196, 2, 10,    -1, 1.942970e+02, 0.27,_
 3185, 83,  197, 0, 4.5,   -1, 1.952870e+02, 0,_
 3186, 83,  197, 1, 0.5,   +1, 1.952870e+02, 0.36,_
 3187, 83,  198, 0,-77.777,+0, 1.962780e+02, 0,_
 3188, 83,  198, 1, 7,     +1, 1.962780e+02, 0.28,_
 3189, 83,  198, 2, 10,    -1, 1.962780e+02, 0.53,_
 3190, 83,  199, 0, 4.5,   -1, 1.972680e+02, 0,_
 3191, 83,  199, 1, 0.5,   +1, 1.972680e+02, 0.667,_
 3192, 83,  200, 0, 7,     +1, 1.982600e+02, 0,_
 3193, 83,  200, 1, 2,     +1, 1.982600e+02, 0.1,_
 3194, 83,  200, 2, 10,    -1, 1.982600e+02, 0.4282,_
 3195, 83,  201, 0, 4.5,   -1, 1.992510e+02, 0,_
 3196, 83,  201, 1, 0.5,   +1, 1.992510e+02, 0.84634,_
 3197, 83,  202, 0, 5,     +1, 2.002430e+02, 0,_
 3198, 83,  203, 0, 4.5,   -1, 2.012330e+02, 0,_
 3199, 83,  203, 1, 0.5,   +1, 2.012330e+02, 1.09814,_
 3200, 83,  204, 0, 6,     +1, 2.022260e+02, 0,_
 3201, 83,  204, 1, 10,    -1, 2.022260e+02, 0.8055,_
 3202, 83,  204, 2, 17,    +1, 2.022260e+02, 2.8334,_
 3203, 83,  205, 0, 4.5,   -1, 2.032170e+02, 0,_
 3204, 83,  206, 0, 6,     +1, 2.042090e+02, 0,_
 3205, 83,  207, 0, 4.5,   -1, 2.052000e+02, 0,_
 3206, 83,  208, 0, 5,     +1, 2.061930e+02, 0,_
 3207, 83,  208, 1, 10,    -1, 2.061930e+02, 1.5711,_
 3208, 83,  209, 0, 4.5,   -1, 2.071850e+02, 0,_
 3209, 83,  210, 0, 1,     -1, 2.081800e+02, 0,_
 3210, 83,  210, 1, 9,     -1, 2.081800e+02, 0.265,_
 3211, 83,  211, 0, 4.5,   -1, 2.091750e+02, 0,_
 3212, 83,  212, 0, 1,     -1, 2.101700e+02, 0,_
 3213, 83,  212, 1, 9,     -1, 2.101700e+02, 0.25,_
 3214, 83,  212, 2, 17,    -1, 2.101700e+02, 3.93,_
 3215, 83,  213, 0, 4.5,   -1, 2.111650e+02, 0,_
 3216, 83,  214, 0, 1,     -1, 2.121600e+02, 0,_
 3217, 83,  215, 0, 4.5,   -1, 2.131550e+02, 0,_
 3218, 83,  215, 1, 12.5,  -1, 2.131550e+02, 1.3475,_
 3219, 83,  216, 0, 1,     -1, 2.141510e+02, 0,_
 3220, 83,  217, 0, 4.5,   -1, 2.151450e+02, 0,_
 3221, 83,  218, 0, 1,     -1, 2.161410e+02, 0,_
 3222, 84,  188, 0, 0,     +1, 1.863840e+02, 0,_
 3223, 84,  189, 0, 1.5,   -1, 1.873750e+02, 0,_
 3224, 84,  190, 0, 0,     +1, 1.883630e+02, 0,_
 3225, 84,  191, 0, 1.5,   -1, 1.893540e+02, 0,_
 3226, 84,  191, 1, 6.5,   +1, 1.893540e+02, 0.034,_
 3227, 84,  192, 0, 0,     +1, 1.903420e+02, 0,_
 3228, 84,  193, 0, 1.5,   -1, 1.913330e+02, 0,_
 3229, 84,  193, 1, 6.5,   +1, 1.913330e+02, 0.1,_
 3230, 84,  194, 0, 0,     +1, 1.923220e+02, 0,_
 3231, 84,  195, 0, 1.5,   -1, 1.933130e+02, 0,_
 3232, 84,  195, 1, 6.5,   +1, 1.933130e+02, 0.08,_
 3233, 84,  196, 0, 0,     +1, 1.943020e+02, 0,_
 3234, 84,  197, 0, 1.5,   -1, 1.952930e+02, 0,_
 3235, 84,  197, 1, 6.5,   +1, 1.952930e+02, 0.23,_
 3236, 84,  198, 0, 0,     +1, 1.962830e+02, 0,_
 3237, 84,  199, 0, 1.5,   -1, 1.972740e+02, 0,_
 3238, 84,  199, 1, 6.5,   +1, 1.972740e+02, 0.312,_
 3239, 84,  200, 0, 0,     +1, 1.982640e+02, 0,_
 3240, 84,  201, 0, 1.5,   -1, 1.992560e+02, 0,_
 3241, 84,  201, 1, 6.5,   +1, 1.992560e+02, 0.4241,_
 3242, 84,  202, 0, 0,     +1, 2.002460e+02, 0,_
 3243, 84,  203, 0, 2.5,   -1, 2.012380e+02, 0,_
 3244, 84,  203, 1, 6.5,   +1, 2.012380e+02, 0.64149,_
 3245, 84,  204, 0, 0,     +1, 2.022280e+02, 0,_
 3246, 84,  205, 0, 2.5,   -1, 2.032200e+02, 0,_
 3247, 84,  205, 1, 9.5,   -1, 2.032200e+02, 1.46121,_
 3248, 84,  206, 0, 0,     +1, 2.042110e+02, 0,_
 3249, 84,  207, 0, 2.5,   -1, 2.052040e+02, 0,_
 3250, 84,  207, 1, 9.5,   -1, 2.052040e+02, 1.38295,_
 3251, 84,  208, 0, 0,     +1, 2.061950e+02, 0,_
 3252, 84,  209, 0, 0.5,   -1, 2.071870e+02, 0,_
 3253, 84,  210, 0, 0,     +1, 2.081790e+02, 0,_
 3254, 84,  211, 0, 4.5,   +1, 2.091740e+02, 0,_
 3255, 84,  211, 1, 12.5,  +1, 2.091740e+02, 1.463,_
 3256, 84,  212, 0, 0,     +1, 2.101680e+02, 0,_
 3257, 84,  212, 1, 18,    +1, 2.101680e+02, 2.922,_
 3258, 84,  213, 0, 4.5,   +1, 2.111630e+02, 0,_
 3259, 84,  214, 0, 0,     +1, 2.121570e+02, 0,_
 3260, 84,  215, 0, 4.5,   +1, 2.131520e+02, 0,_
 3261, 84,  216, 0, 0,     +1, 2.141460e+02, 0,_
 3262, 84,  217, 0, 2.5,   +1, 2.151420e+02, 0,_
 3263, 84,  218, 0, 0,     +1, 2.161360e+02, 0,_
 3264, 84,  219, 0, 3.5,   +1, 2.171320e+02, 0,_
 3265, 84,  220, 0, 0,     +1, 2.181270e+02, 0,_
 3266, 85,  193, 0, 4.5,   -1, 1.913420e+02, 0,_
 3267, 85,  194, 0, 3,     +1, 1.923320e+02, 0,_
 3268, 85,  194, 1, 10,    -1, 1.923320e+02, 0.25,_
 3269, 85,  195, 0, 4.5,   -1, 1.933210e+02, 0,_
 3270, 85,  195, 1, 0.5,   +1, 1.933210e+02, 0,_
 3271, 85,  196, 0, 3,     +1, 1.943120e+02, 0,_
 3272, 85,  196, 1, 10,    -1, 1.943120e+02, 0.2,_
 3273, 85,  197, 0, 4.5,   -1, 1.953010e+02, 0,_
 3274, 85,  197, 1, 0.5,   +1, 1.953010e+02, 0.05,_
 3275, 85,  198, 0, 3,     +1, 1.962920e+02, 0,_
 3276, 85,  198, 1, 10,    -1, 1.962920e+02, 0.33,_
 3277, 85,  199, 0, 4.5,   -1, 1.972810e+02, 0,_
 3278, 85,  200, 0, 3,     +1, 1.982720e+02, 0,_
 3279, 85,  200, 1, 7,     +1, 1.982720e+02, 0.113,_
 3280, 85,  200, 2, 10,    -1, 1.982720e+02, 0.344,_
 3281, 85,  201, 0, 4.5,   -1, 1.992620e+02, 0,_
 3282, 85,  202, 0,-77.777,+0, 2.002530e+02, 0,_
 3283, 85,  202, 1, 7,     +1, 2.002530e+02, 0.19,_
 3284, 85,  202, 2, 10,    -1, 2.002530e+02, 0.58,_
 3285, 85,  203, 0, 4.5,   -1, 2.012430e+02, 0,_
 3286, 85,  204, 0, 7,     +1, 2.022350e+02, 0,_
 3287, 85,  204, 1, 10,    -1, 2.022350e+02, 0.5873,_
 3288, 85,  205, 0, 4.5,   -1, 2.032250e+02, 0,_
 3289, 85,  206, 0, 5,     +1, 2.042170e+02, 0,_
 3290, 85,  207, 0, 4.5,   -1, 2.052080e+02, 0,_
 3291, 85,  208, 0, 6,     +1, 2.062000e+02, 0,_
 3292, 85,  209, 0, 4.5,   -1, 2.071910e+02, 0,_
 3293, 85,  210, 0, 5,     +1, 2.081830e+02, 0,_
 3294, 85,  211, 0, 4.5,   -1, 2.091750e+02, 0,_
 3295, 85,  212, 0, 1,     -1, 2.101700e+02, 0,_
 3296, 85,  212, 1, 9,     -1, 2.101700e+02, 0.222,_
 3297, 85,  213, 0, 4.5,   -1, 2.111630e+02, 0,_
 3298, 85,  214, 0, 1,     -1, 2.121580e+02, 0,_
 3299, 85,  215, 0, 4.5,   -1, 2.131520e+02, 0,_
 3300, 85,  216, 0, 1,     -1, 2.141470e+02, 0,_
 3301, 85,  217, 0, 4.5,   -1, 2.151410e+02, 0,_
 3302, 85,  218, 0, 1,     -1, 2.161360e+02, 0,_
 3303, 85,  219, 0, 2.5,   -1, 2.171300e+02, 0,_
 3304, 85,  220, 0, 3,     +0, 2.181250e+02, 0,_
 3305, 85,  221, 0, 1.5,   -1, 2.191190e+02, 0,_
 3306, 85,  222, 0,-77.777,+0, 2.201150e+02, 0,_
 3307, 85,  223, 0, 1.5,   -1, 2.211090e+02, 0,_
 3308, 86,  195, 0, 1.5,   -1, 1.933300e+02, 0,_
 3309, 86,  195, 1, 6.5,   +1, 1.933300e+02, 0.05,_
 3310, 86,  196, 0, 0,     +1, 1.943180e+02, 0,_
 3311, 86,  197, 0, 1.5,   -1, 1.953090e+02, 0,_
 3312, 86,  197, 1, 6.5,   +1, 1.953090e+02, 0.2,_
 3313, 86,  198, 0, 0,     +1, 1.962980e+02, 0,_
 3314, 86,  199, 0, 1.5,   -1, 1.972890e+02, 0,_
 3315, 86,  199, 1, 6.5,   +1, 1.972890e+02, 0.15,_
 3316, 86,  200, 0, 0,     +1, 1.982780e+02, 0,_
 3317, 86,  201, 0, 1.5,   -1, 1.992690e+02, 0,_
 3318, 86,  201, 1, 6.5,   +1, 1.992690e+02, 0.28,_
 3319, 86,  202, 0, 0,     +1, 2.002580e+02, 0,_
 3320, 86,  203, 0,-77.777,+0, 2.012500e+02, 0,_
 3321, 86,  203, 1, 6.5,   +1, 2.012500e+02, 0.363,_
 3322, 86,  204, 0, 0,     +1, 2.022390e+02, 0,_
 3323, 86,  205, 0, 2.5,   -1, 2.032310e+02, 0,_
 3324, 86,  206, 0, 0,     +1, 2.042210e+02, 0,_
 3325, 86,  207, 0, 2.5,   -1, 2.052130e+02, 0,_
 3326, 86,  208, 0, 0,     +1, 2.062030e+02, 0,_
 3327, 86,  209, 0, 2.5,   -1, 2.071950e+02, 0,_
 3328, 86,  210, 0, 0,     +1, 2.081860e+02, 0,_
 3329, 86,  211, 0, 0.5,   -1, 2.091780e+02, 0,_
 3330, 86,  212, 0, 0,     +1, 2.101700e+02, 0,_
 3331, 86,  213, 0, 4.5,   +1, 2.111640e+02, 0,_
 3332, 86,  214, 0, 0,     +1, 2.121570e+02, 0,_
 3333, 86,  215, 0, 4.5,   +1, 2.131520e+02, 0,_
 3334, 86,  216, 0, 0,     +1, 2.141450e+02, 0,_
 3335, 86,  217, 0, 4.5,   +1, 2.151400e+02, 0,_
 3336, 86,  218, 0, 0,     +1, 2.161330e+02, 0,_
 3337, 86,  219, 0, 2.5,   +1, 2.171280e+02, 0,_
 3338, 86,  220, 0, 0,     +1, 2.181210e+02, 0,_
 3339, 86,  221, 0, 3.5,   +1, 2.191170e+02, 0,_
 3340, 86,  222, 0, 0,     +1, 2.201100e+02, 0,_
 3341, 86,  223, 0, 3.5,   -1, 2.211040e+02, 0,_
 3342, 86,  224, 0, 0,     +1, 2.221000e+02, 0,_
 3343, 86,  225, 0, 3.5,   -1, 2.230950e+02, 0,_
 3344, 86,  226, 0, 0,     +1, 2.240890e+02, 0,_
 3345, 86,  227, 0, 2.5,   +1, 2.250850e+02, 0,_
 3346, 86,  228, 0, 0,     +1, 2.260790e+02, 0,_
 3347, 87,  199, 0, 1.5,   -1, 1.972980e+02, 0,_
 3348, 87,  200, 0, 3,     +1, 1.982880e+02, 0,_
 3349, 87,  200, 1, 10,    -1, 1.982880e+02, 0.2,_
 3350, 87,  201, 0, 4.5,   -1, 1.992770e+02, 0,_
 3351, 87,  202, 0, 3,     +1, 2.002680e+02, 0,_
 3352, 87,  202, 1, 10,    -1, 2.002680e+02, 0.33,_
 3353, 87,  203, 0, 4.5,   -1, 2.012570e+02, 0,_
 3354, 87,  204, 0, 3,     +1, 2.022480e+02, 0,_
 3355, 87,  204, 1, 7,     +1, 2.022480e+02, 0.054,_
 3356, 87,  204, 2, 10,    -1, 2.022480e+02, 0.33,_
 3357, 87,  205, 0, 4.5,   -1, 2.032380e+02, 0,_
 3358, 87,  206, 0,-77.777,+0, 2.042290e+02, 0,_
 3359, 87,  206, 1, 7,     +1, 2.042290e+02, 0.19,_
 3360, 87,  206, 2, 10,    -1, 2.042290e+02, 0.72,_
 3361, 87,  207, 0, 4.5,   -1, 2.052190e+02, 0,_
 3362, 87,  208, 0, 7,     +1, 2.062100e+02, 0,_
 3363, 87,  209, 0, 4.5,   -1, 2.072010e+02, 0,_
 3364, 87,  210, 0, 6,     +1, 2.081920e+02, 0,_
 3365, 87,  211, 0, 4.5,   -1, 2.091830e+02, 0,_
 3366, 87,  212, 0, 5,     +1, 2.101750e+02, 0,_
 3367, 87,  213, 0, 4.5,   -1, 2.111660e+02, 0,_
 3368, 87,  214, 0, 1,     -1, 2.121610e+02, 0,_
 3369, 87,  214, 1, 9,     -1, 2.121610e+02, 0.122,_
 3370, 87,  215, 0, 4.5,   -1, 2.131530e+02, 0,_
 3371, 87,  216, 0, 1,     -1, 2.141480e+02, 0,_
 3372, 87,  217, 0, 4.5,   -1, 2.151400e+02, 0,_
 3373, 87,  218, 0, 1,     -1, 2.161350e+02, 0,_
 3374, 87,  218, 1,-77.777,+0, 2.161350e+02, 0.086,_
 3375, 87,  219, 0, 4.5,   -1, 2.171280e+02, 0,_
 3376, 87,  220, 0, 1,     +1, 2.181220e+02, 0,_
 3377, 87,  221, 0, 2.5,   -1, 2.191160e+02, 0,_
 3378, 87,  222, 0, 2,     -1, 2.201100e+02, 0,_
 3379, 87,  223, 0, 1.5,   -1, 2.211040e+02, 0,_
 3380, 87,  224, 0, 1,     -1, 2.220990e+02, 0,_
 3381, 87,  225, 0, 1.5,   -1, 2.230920e+02, 0,_
 3382, 87,  226, 0, 1,     -1, 2.240880e+02, 0,_
 3383, 87,  227, 0, 0.5,   +1, 2.250810e+02, 0,_
 3384, 87,  228, 0, 2,     -1, 2.260770e+02, 0,_
 3385, 87,  229, 0, 0.5,   +1, 2.270710e+02, 0,_
 3386, 87,  230, 0,-77.777,+0, 2.280660e+02, 0,_
 3387, 87,  231, 0, 0.5,   +1, 2.290610e+02, 0,_
 3388, 87,  232, 0,-77.777,+0, 2.300560e+02, 0,_
 3389, 88,  202, 0, 0,     +1, 2.002750e+02, 0,_
 3390, 88,  203, 0, 1.5,   -1, 2.012650e+02, 0,_
 3391, 88,  203, 1, 1.5,   -1, 2.012650e+02, 0.19,_
 3392, 88,  204, 0, 0,     +1, 2.022540e+02, 0,_
 3393, 88,  205, 0, 1.5,   -1, 2.032450e+02, 0,_
 3394, 88,  205, 1, 6.5,   +1, 2.032450e+02, 0.29,_
 3395, 88,  206, 0, 0,     +1, 2.042340e+02, 0,_
 3396, 88,  207, 0,-77.777,+0, 2.052260e+02, 0,_
 3397, 88,  207, 1, 6.5,   +1, 2.052260e+02, 0.56,_
 3398, 88,  208, 0, 0,     +1, 2.062150e+02, 0,_
 3399, 88,  209, 0, 2.5,   -1, 2.072070e+02, 0,_
 3400, 88,  210, 0, 0,     +1, 2.081960e+02, 0,_
 3401, 88,  211, 0, 2.5,   -1, 2.091880e+02, 0,_
 3402, 88,  212, 0, 0,     +1, 2.101790e+02, 0,_
 3403, 88,  213, 0, 0.5,   -1, 2.111710e+02, 0,_
 3404, 88,  213, 1, 6.5,   +1, 2.111710e+02, 1.77,_
 3405, 88,  214, 0, 0,     +1, 2.121620e+02, 0,_
 3406, 88,  215, 0, 4.5,   +1, 2.131560e+02, 0,_
 3407, 88,  216, 0, 0,     +1, 2.141480e+02, 0,_
 3408, 88,  217, 0, 4.5,   +1, 2.151420e+02, 0,_
 3409, 88,  218, 0, 0,     +1, 2.161340e+02, 0,_
 3410, 88,  219, 0, 3.5,   +1, 2.171290e+02, 0,_
 3411, 88,  220, 0, 0,     +1, 2.181210e+02, 0,_
 3412, 88,  221, 0, 2.5,   +1, 2.191150e+02, 0,_
 3413, 88,  222, 0, 0,     +1, 2.201080e+02, 0,_
 3414, 88,  223, 0, 0.5,   +1, 2.211030e+02, 0,_
 3415, 88,  224, 0, 0,     +1, 2.220960e+02, 0,_
 3416, 88,  225, 0, 1.5,   +1, 2.230910e+02, 0,_
 3417, 88,  226, 0, 0,     +1, 2.240840e+02, 0,_
 3418, 88,  227, 0, 1.5,   +1, 2.250790e+02, 0,_
 3419, 88,  228, 0, 0,     +1, 2.260720e+02, 0,_
 3420, 88,  229, 0, 2.5,   +1, 2.270670e+02, 0,_
 3421, 88,  230, 0, 0,     +1, 2.280610e+02, 0,_
 3422, 88,  231, 0, 2.5,   +1, 2.290560e+02, 0,_
 3423, 88,  232, 0, 0,     +1, 2.300500e+02, 0,_
 3424, 88,  233, 0, 0.5,   +1, 2.310460e+02, 0,_
 3425, 88,  234, 0, 0,     +1, 2.320400e+02, 0,_
 3426, 89,  206, 0, 3,     +1, 2.042450e+02, 0,_
 3427, 89,  206, 1,-77.777,+0, 2.042450e+02, 0.08,_
 3428, 89,  206, 2, 10,    -1, 2.042450e+02, 0.28,_
 3429, 89,  207, 0, 4.5,   -1, 2.052340e+02, 0,_
 3430, 89,  208, 0, 3,     +1, 2.062250e+02, 0,_
 3431, 89,  208, 1, 10,    -1, 2.062250e+02, 0.51,_
 3432, 89,  209, 0, 4.5,   -1, 2.072140e+02, 0,_
 3433, 89,  210, 0, 7,     +1, 2.082050e+02, 0,_
 3434, 89,  211, 0, 4.5,   -1, 2.091950e+02, 0,_
 3435, 89,  212, 0, 6,     +1, 2.101870e+02, 0,_
 3436, 89,  213, 0, 4.5,   -1, 2.111770e+02, 0,_
 3437, 89,  214, 0, 5,     +1, 2.121680e+02, 0,_
 3438, 89,  215, 0, 4.5,   -1, 2.131590e+02, 0,_
 3439, 89,  216, 0, 1,     -1, 2.141530e+02, 0,_
 3440, 89,  216, 1, 9,     -1, 2.141530e+02, 0.037,_
 3441, 89,  217, 0, 4.5,   -1, 2.151450e+02, 0,_
 3442, 89,  218, 0, 1,     -1, 2.161390e+02, 0,_
 3443, 89,  219, 0, 4.5,   -1, 2.171310e+02, 0,_
 3444, 89,  220, 0, 3,     -1, 2.181250e+02, 0,_
 3445, 89,  221, 0, 4.5,   -1, 2.191170e+02, 0,_
 3446, 89,  222, 0, 1,     -1, 2.201110e+02, 0,_
 3447, 89,  222, 1,-77.777,+0, 2.201110e+02, 0.2,_
 3448, 89,  223, 0, 2.5,   -1, 2.211030e+02, 0,_
 3449, 89,  224, 0, 0,     -1, 2.220970e+02, 0,_
 3450, 89,  225, 0, 1.5,   -1, 2.230900e+02, 0,_
 3451, 89,  226, 0, 1,     -1, 2.240840e+02, 0,_
 3452, 89,  227, 0, 1.5,   -1, 2.250770e+02, 0,_
 3453, 89,  228, 0, 3,     +1, 2.260720e+02, 0,_
 3454, 89,  229, 0, 1.5,   +1, 2.270660e+02, 0,_
 3455, 89,  230, 0, 1,     +1, 2.280600e+02, 0,_
 3456, 89,  231, 0, 0.5,   +1, 2.290540e+02, 0,_
 3457, 89,  232, 0, 1,     +1, 2.300490e+02, 0,_
 3458, 89,  233, 0, 0.5,   +1, 2.310430e+02, 0,_
 3459, 89,  234, 0,-77.777,+0, 2.320380e+02, 0,_
 3460, 89,  235, 0, 0.5,   +1, 2.330320e+02, 0,_
 3461, 89,  236, 0,-77.777,+0, 2.340270e+02, 0,_
 3462, 90,  209, 0, 2.5,   -1, 2.072220e+02, 0,_
 3463, 90,  210, 0, 0,     +1, 2.082110e+02, 0,_
 3464, 90,  211, 0, 2.5,   -1, 2.092020e+02, 0,_
 3465, 90,  212, 0, 0,     +1, 2.101920e+02, 0,_
 3466, 90,  213, 0, 2.5,   -1, 2.111830e+02, 0,_
 3467, 90,  214, 0, 0,     +1, 2.121730e+02, 0,_
 3468, 90,  215, 0, 0.5,   -1, 2.131650e+02, 0,_
 3469, 90,  216, 0, 0,     +1, 2.141550e+02, 0,_
 3470, 90,  217, 0, 4.5,   +1, 2.151490e+02, 0,_
 3471, 90,  218, 0, 0,     +1, 2.161400e+02, 0,_
 3472, 90,  219, 0, 4.5,   +1, 2.171340e+02, 0,_
 3473, 90,  220, 0, 0,     +1, 2.181260e+02, 0,_
 3474, 90,  221, 0, 3.5,   +1, 2.191200e+02, 0,_
 3475, 90,  222, 0, 0,     +1, 2.201110e+02, 0,_
 3476, 90,  223, 0, 2.5,   +1, 2.211050e+02, 0,_
 3477, 90,  224, 0, 0,     +1, 2.220970e+02, 0,_
 3478, 90,  225, 0, 1.5,   +1, 2.230910e+02, 0,_
 3479, 90,  226, 0, 0,     +1, 2.240830e+02, 0,_
 3480, 90,  227, 0, 0.5,   +1, 2.250770e+02, 0,_
 3481, 90,  228, 0, 0,     +1, 2.260700e+02, 0,_
 3482, 90,  229, 0, 2.5,   +1, 2.270640e+02, 0,_
 3483, 90,  229, 1, 1.5,   +1, 2.270640e+02, 3.5e-06,_
 3484, 90,  230, 0, 0,     +1, 2.280570e+02, 0,_
 3485, 90,  231, 0, 2.5,   +1, 2.290520e+02, 0,_
 3486, 90,  232, 0, 0,     +1, 2.300450e+02, 0,_
 3487, 90,  233, 0, 1.5,   +1, 2.310400e+02, 0,_
 3488, 90,  234, 0, 0,     +1, 2.320330e+02, 0,_
 3489, 90,  235, 0, 2.5,   +1, 2.330280e+02, 0,_
 3490, 90,  236, 0, 0,     +1, 2.340220e+02, 0,_
 3491, 90,  237, 0, 2.5,   +1, 2.350170e+02, 0,_
 3492, 90,  238, 0, 0,     +1, 2.360110e+02, 0,_
 3493, 91,  212, 0, 7,     +1, 2.102020e+02, 0,_
 3494, 91,  213, 0, 4.5,   -1, 2.111910e+02, 0,_
 3495, 91,  214, 0,-77.777,+0, 2.121820e+02, 0,_
 3496, 91,  215, 0, 4.5,   -1, 2.131720e+02, 0,_
 3497, 91,  216, 0,-77.777,+0, 2.141630e+02, 0,_
 3498, 91,  217, 0, 4.5,   -1, 2.151540e+02, 0,_
 3499, 91,  217, 1, 14.5,  +1, 2.151540e+02, 1.854,_
 3500, 91,  218, 0,-77.777,+0, 2.161470e+02, 0,_
 3501, 91,  219, 0, 4.5,   -1, 2.171380e+02, 0,_
 3502, 91,  220, 0, 1,     -1, 2.181320e+02, 0,_
 3503, 91,  221, 0, 4.5,   -1, 2.191230e+02, 0,_
 3504, 91,  222, 0,-77.777,+0, 2.201160e+02, 0,_
 3505, 91,  223, 0, 4.5,   -1, 2.211080e+02, 0,_
 3506, 91,  224, 0, 5,     -1, 2.221010e+02, 0,_
 3507, 91,  225, 0, 2.5,   -1, 2.230930e+02, 0,_
 3508, 91,  226, 0,-77.777,+0, 2.240860e+02, 0,_
 3509, 91,  227, 0, 2.5,   -1, 2.250790e+02, 0,_
 3510, 91,  228, 0, 3,     +1, 2.260720e+02, 0,_
 3511, 91,  229, 0, 2.5,   +1, 2.270650e+02, 0,_
 3512, 91,  230, 0, 2,     -1, 2.280580e+02, 0,_
 3513, 91,  231, 0, 1.5,   -1, 2.290510e+02, 0,_
 3514, 91,  232, 0, 2,     -1, 2.300450e+02, 0,_
 3515, 91,  233, 0, 1.5,   -1, 2.310380e+02, 0,_
 3516, 91,  234, 0, 4,     +1, 2.320330e+02, 0,_
 3517, 91,  234, 1, 0,     -1, 2.320330e+02, 0.079,_
 3518, 91,  235, 0, 1.5,   -1, 2.330260e+02, 0,_
 3519, 91,  236, 0, 1,     -1, 2.340210e+02, 0,_
 3520, 91,  237, 0, 0.5,   +1, 2.350150e+02, 0,_
 3521, 91,  238, 0, 3,     -1, 2.360100e+02, 0,_
 3522, 91,  239, 0,-77.777,+0, 2.370040e+02, 0,_
 3523, 91,  240, 0,-77.777,+0, 2.379990e+02, 0,_
 3524, 92,  217, 0, 0.5,   -1, 2.151600e+02, 0,_
 3525, 92,  218, 0, 0,     +1, 2.161510e+02, 0,_
 3526, 92,  219, 0, 4.5,   +1, 2.171430e+02, 0,_
 3527, 92,  220, 0, 0,     +1, 2.181350e+02, 0,_
 3528, 92,  221, 0, 4.5,   +1, 2.191280e+02, 0,_
 3529, 92,  222, 0, 0,     +1, 2.201190e+02, 0,_
 3530, 92,  223, 0, 3.5,   +1, 2.211120e+02, 0,_
 3531, 92,  224, 0, 0,     +1, 2.221030e+02, 0,_
 3532, 92,  225, 0, 2.5,   +1, 2.230960e+02, 0,_
 3533, 92,  226, 0, 0,     +1, 2.240880e+02, 0,_
 3534, 92,  227, 0, 1.5,   +1, 2.250810e+02, 0,_
 3535, 92,  228, 0, 0,     +1, 2.260720e+02, 0,_
 3536, 92,  229, 0, 1.5,   +1, 2.270660e+02, 0,_
 3537, 92,  230, 0, 0,     +1, 2.280580e+02, 0,_
 3538, 92,  231, 0, 2.5,   +1, 2.290520e+02, 0,_
 3539, 92,  232, 0, 0,     +1, 2.300440e+02, 0,_
 3540, 92,  233, 0, 2.5,   +1, 2.310380e+02, 0,_
 3541, 92,  234, 0, 0,     +1, 2.320300e+02, 0,_
 3542, 92,  235, 0, 3.5,   -1, 2.330250e+02, 0,_
 3543, 92,  235, 1, 0.5,   +1, 2.330250e+02, 7.68e-05,_
 3544, 92,  236, 0, 0,     +1, 2.340180e+02, 0,_
 3545, 92,  237, 0, 0.5,   +1, 2.350120e+02, 0,_
 3546, 92,  238, 0, 0,     +1, 2.360060e+02, 0,_
 3547, 92,  239, 0, 2.5,   +1, 2.370010e+02, 0,_
 3548, 92,  240, 0, 0,     +1, 2.379940e+02, 0,_
 3549, 92,  241, 0, 3.5,   +1, 2.389900e+02, 0,_
 3550, 92,  242, 0, 0,     +1, 2.399830e+02, 0,_
 3551, 93,  225, 0, 4.5,   -1, 2.231010e+02, 0,_
 3552, 93,  226, 0,-77.777,+0, 2.240930e+02, 0,_
 3553, 93,  227, 0, 2.5,   -1, 2.250850e+02, 0,_
 3554, 93,  228, 0,-77.777,+0, 2.260770e+02, 0,_
 3555, 93,  229, 0, 2.5,   +1, 2.270690e+02, 0,_
 3556, 93,  230, 0,-77.777,+0, 2.280620e+02, 0,_
 3557, 93,  231, 0, 2.5,   +1, 2.290540e+02, 0,_
 3558, 93,  232, 0, 4,     +1, 2.300470e+02, 0,_
 3559, 93,  233, 0, 2.5,   +1, 2.310390e+02, 0,_
 3560, 93,  234, 0, 0,     +1, 2.320320e+02, 0,_
 3561, 93,  235, 0, 2.5,   +1, 2.330250e+02, 0,_
 3562, 93,  236, 0, 6,     -1, 2.340190e+02, 0,_
 3563, 93,  236, 1, 1,     -1, 2.340190e+02, 5e-06,_
 3564, 93,  237, 0, 2.5,   +1, 2.350120e+02, 0,_
 3565, 93,  238, 0, 2,     +1, 2.360060e+02, 0,_
 3566, 93,  239, 0, 2.5,   +1, 2.369990e+02, 0,_
 3567, 93,  240, 0, 5,     +1, 2.379940e+02, 0,_
 3568, 93,  240, 1, 1,     -1, 2.379940e+02, 0.02,_
 3569, 93,  241, 0, 2.5,   +1, 2.389880e+02, 0,_
 3570, 93,  242, 0, 1,     +1, 2.399820e+02, 0,_
 3571, 93,  242, 1, 6,     +1, 2.399820e+02, 0,_
 3572, 93,  243, 0, 2.5,   -1, 2.409760e+02, 0,_
 3573, 93,  244, 0, 7,     -1, 2.419710e+02, 0,_
 3574, 94,  228, 0, 0,     +1, 2.260800e+02, 0,_
 3575, 94,  229, 0, 1.5,   +1, 2.270730e+02, 0,_
 3576, 94,  230, 0, 0,     +1, 2.280630e+02, 0,_
 3577, 94,  231, 0, 1.5,   +1, 2.290560e+02, 0,_
 3578, 94,  232, 0, 0,     +1, 2.300480e+02, 0,_
 3579, 94,  233, 0, 2.5,   +1, 2.310410e+02, 0,_
 3580, 94,  234, 0, 0,     +1, 2.320330e+02, 0,_
 3581, 94,  235, 0, 2.5,   +1, 2.330260e+02, 0,_
 3582, 94,  236, 0, 0,     +1, 2.340180e+02, 0,_
 3583, 94,  237, 0, 3.5,   -1, 2.350120e+02, 0,_
 3584, 94,  237, 1, 0.5,   +1, 2.350120e+02, 0.145544,_
 3585, 94,  238, 0, 0,     +1, 2.360050e+02, 0,_
 3586, 94,  239, 0, 0.5,   +1, 2.369990e+02, 0,_
 3587, 94,  240, 0, 0,     +1, 2.379920e+02, 0,_
 3588, 94,  241, 0, 2.5,   +1, 2.389860e+02, 0,_
 3589, 94,  242, 0, 0,     +1, 2.399790e+02, 0,_
 3590, 94,  243, 0, 3.5,   +1, 2.409740e+02, 0,_
 3591, 94,  244, 0, 0,     +1, 2.419680e+02, 0,_
 3592, 94,  245, 0, 4.5,   -1, 2.429630e+02, 0,_
 3593, 94,  246, 0, 0,     +1, 2.439560e+02, 0,_
 3594, 94,  247, 0, 0.5,   +1, 2.449520e+02, 0,_
 3595, 95,  231, 0,-77.777,+0, 2.290610e+02, 0,_
 3596, 95,  232, 0,-77.777,+0, 2.300530e+02, 0,_
 3597, 95,  233, 0,-77.777,+0, 2.310440e+02, 0,_
 3598, 95,  234, 0,-77.777,+0, 2.320370e+02, 0,_
 3599, 95,  235, 0, 2.5,   -1, 2.330290e+02, 0,_
 3600, 95,  236, 0,-77.777,+0, 2.340220e+02, 0,_
 3601, 95,  237, 0, 2.5,   -1, 2.350140e+02, 0,_
 3602, 95,  238, 0, 1,     +1, 2.360070e+02, 0,_
 3603, 95,  239, 0, 2.5,   -1, 2.369990e+02, 0,_
 3604, 95,  240, 0, 3,     -1, 2.379930e+02, 0,_
 3605, 95,  241, 0, 2.5,   -1, 2.389860e+02, 0,_
 3606, 95,  242, 0, 1,     -1, 2.399800e+02, 0,_
 3607, 95,  242, 1, 5,     -1, 2.399800e+02, 0,_
 3608, 95,  242, 2,-77.777,+0, 2.399800e+02, 2.2,_
 3609, 95,  243, 0, 2.5,   -1, 2.409730e+02, 0,_
 3610, 95,  244, 0, 6,     -1, 2.419680e+02, 0,_
 3611, 95,  244, 1, 1,     -1, 2.419680e+02, 0.069,_
 3612, 95,  245, 0, 2.5,   +1, 2.429610e+02, 0,_
 3613, 95,  246, 0, 7,     +1, 2.439560e+02, 0,_
 3614, 95,  246, 1, 2,     +1, 2.439560e+02, 0.005,_
 3615, 95,  247, 0, 2.5,   +0, 2.449500e+02, 0,_
 3616, 95,  248, 0,-77.777,+0, 2.459450e+02, 0,_
 3617, 95,  249, 0,-77.777,+0, 2.469390e+02, 0,_
 3618, 96,  233, 0, 1.5,   +1, 2.310490e+02, 0,_
 3619, 96,  234, 0, 0,     +1, 2.320400e+02, 0,_
 3620, 96,  235, 0, 2.5,   +1, 2.330320e+02, 0,_
 3621, 96,  236, 0, 0,     +1, 2.340240e+02, 0,_
 3622, 96,  237, 0, 2.5,   +1, 2.350170e+02, 0,_
 3623, 96,  238, 0, 0,     +1, 2.360080e+02, 0,_
 3624, 96,  239, 0, 3.5,   -1, 2.370010e+02, 0,_
 3625, 96,  240, 0, 0,     +1, 2.379930e+02, 0,_
 3626, 96,  241, 0, 0.5,   +1, 2.389870e+02, 0,_
 3627, 96,  242, 0, 0,     +1, 2.399790e+02, 0,_
 3628, 96,  243, 0, 2.5,   +1, 2.409730e+02, 0,_
 3629, 96,  244, 0, 0,     +1, 2.419660e+02, 0,_
 3630, 96,  244, 1, 6,     +1, 2.419660e+02, 1.04019,_
 3631, 96,  245, 0, 3.5,   +1, 2.429600e+02, 0,_
 3632, 96,  246, 0, 0,     +1, 2.439530e+02, 0,_
 3633, 96,  247, 0, 4.5,   -1, 2.449480e+02, 0,_
 3634, 96,  248, 0, 0,     +1, 2.459410e+02, 0,_
 3635, 96,  249, 0, 0.5,   +1, 2.469360e+02, 0,_
 3636, 96,  250, 0, 0,     +1, 2.479300e+02, 0,_
 3637, 96,  251, 0, 0.5,   +1, 2.489250e+02, 0,_
 3638, 96,  252, 0, 0,     +1, 2.499190e+02, 0,_
 3639, 97,  235, 0,-77.777,+0, 2.330370e+02, 0,_
 3640, 97,  236, 0,-77.777,+0, 2.340290e+02, 0,_
 3641, 97,  237, 0, 3.5,   +1, 2.350210e+02, 0,_
 3642, 97,  238, 0,-77.777,+0, 2.360130e+02, 0,_
 3643, 97,  239, 0, 3.5,   +1, 2.370050e+02, 0,_
 3644, 97,  240, 0,-77.777,+0, 2.379980e+02, 0,_
 3645, 97,  241, 0, 3.5,   +1, 2.389890e+02, 0,_
 3646, 97,  242, 0, 2,     -1, 2.399830e+02, 0,_
 3647, 97,  243, 0, 1.5,   -1, 2.409750e+02, 0,_
 3648, 97,  244, 0, 4,     -1, 2.419690e+02, 0,_
 3649, 97,  245, 0, 1.5,   -1, 2.429610e+02, 0,_
 3650, 97,  246, 0, 2,     -1, 2.439550e+02, 0,_
 3651, 97,  247, 0, 1.5,   -1, 2.449480e+02, 0,_
 3652, 97,  248, 0, 6,     +1, 2.459420e+02, 0,_
 3653, 97,  248, 1, 1,     -1, 2.459420e+02, 0.03,_
 3654, 97,  249, 0, 3.5,   +1, 2.469350e+02, 0,_
 3655, 97,  250, 0, 2,     -1, 2.479300e+02, 0,_
 3656, 97,  251, 0, 1.5,   -1, 2.489240e+02, 0,_
 3657, 97,  252, 0,-77.777,+0, 2.499190e+02, 0,_
 3658, 97,  253, 0,-77.777,+0, 2.509130e+02, 0,_
 3659, 97,  254, 0,-77.777,+0, 2.519080e+02, 0,_
 3660, 98,  237, 0, 2.5,   +1, 2.350260e+02, 0,_
 3661, 98,  238, 0, 0,     +1, 2.360160e+02, 0,_
 3662, 98,  239, 0, 2.5,   +1, 2.370090e+02, 0,_
 3663, 98,  240, 0, 0,     +1, 2.380000e+02, 0,_
 3664, 98,  241, 0, 3.5,   -1, 2.389930e+02, 0,_
 3665, 98,  242, 0, 0,     +1, 2.399840e+02, 0,_
 3666, 98,  243, 0, 0.5,   +1, 2.409770e+02, 0,_
 3667, 98,  244, 0, 0,     +1, 2.419690e+02, 0,_
 3668, 98,  245, 0, 2.5,   +1, 2.429630e+02, 0,_
 3669, 98,  246, 0, 0,     +1, 2.439550e+02, 0,_
 3670, 98,  247, 0, 3.5,   +1, 2.449490e+02, 0,_
 3671, 98,  248, 0, 0,     +1, 2.459410e+02, 0,_
 3672, 98,  249, 0, 4.5,   -1, 2.469350e+02, 0,_
 3673, 98,  250, 0, 0,     +1, 2.479280e+02, 0,_
 3674, 98,  251, 0, 0.5,   +1, 2.489230e+02, 0,_
 3675, 98,  252, 0, 0,     +1, 2.499160e+02, 0,_
 3676, 98,  253, 0, 3.5,   +1, 2.509110e+02, 0,_
 3677, 98,  254, 0, 0,     +1, 2.519050e+02, 0,_
 3678, 98,  255, 0, 3.5,   +1, 2.529000e+02, 0,_
 3679, 98,  256, 0, 0,     +1, 2.538930e+02, 0,_
 3680, 99,  240, 0,-77.777,+0, 2.380070e+02, 0,_
 3681, 99,  241, 0, 1.5,   -1, 2.389980e+02, 0,_
 3682, 99,  242, 0,-77.777,+0, 2.399900e+02, 0,_
 3683, 99,  243, 0, 1.5,   -1, 2.409810e+02, 0,_
 3684, 99,  244, 0,-77.777,+0, 2.419740e+02, 0,_
 3685, 99,  245, 0, 1.5,   -1, 2.429660e+02, 0,_
 3686, 99,  246, 0, 4,     -1, 2.439590e+02, 0,_
 3687, 99,  247, 0, 3.5,   +1, 2.449510e+02, 0,_
 3688, 99,  248, 0,-77.777,+0, 2.459440e+02, 0,_
 3689, 99,  249, 0, 3.5,   +1, 2.469370e+02, 0,_
 3690, 99,  250, 0, 6,     +1, 2.479300e+02, 0,_
 3691, 99,  250, 1, 1,     -1, 2.479300e+02, 0.2,_
 3692, 99,  251, 0, 1.5,   -1, 2.489230e+02, 0,_
 3693, 99,  252, 0, 5,     -1, 2.499170e+02, 0,_
 3694, 99,  253, 0, 3.5,   +1, 2.509110e+02, 0,_
 3695, 99,  254, 0, 7,     +1, 2.519050e+02, 0,_
 3696, 99,  254, 1, 2,     +1, 2.519050e+02, 0.0837,_
 3697, 99,  255, 0, 3.5,   +1, 2.528990e+02, 0,_
 3698, 99,  256, 0,-77.777,+0, 2.538940e+02, 0,_
 3699, 99,  256, 1, 8,     +1, 2.538940e+02, 0,_
 3700, 99,  257, 0, 3.5,   +1, 2.548870e+02, 0,_
 3701, 99,  258, 0,-77.777,+0, 2.558820e+02, 0,_
 3701, 99,  259, 0,-77.777,+0, 259,          0,_
 3701, 99,  260, 0,-77.777,+0, 260,          0,_
 3702, 100, 242, 0, 0,     +1, 2.399940e+02, 0,_
 3703, 100, 243, 0, 3.5,   -1, 2.409860e+02, 0,_
 3704, 100, 244, 0, 0,     +1, 2.419770e+02, 0,_
 3705, 100, 245, 0, 0.5,   +1, 2.429700e+02, 0,_
 3706, 100, 246, 0, 0,     +1, 2.439610e+02, 0,_
 3707, 100, 247, 0, 2.5,   +1, 2.449540e+02, 0,_
 3708, 100, 248, 0, 0,     +1, 2.459460e+02, 0,_
 3709, 100, 249, 0, 3.5,   +1, 2.469390e+02, 0,_
 3710, 100, 250, 0, 0,     +1, 2.479310e+02, 0,_
 3711, 100, 250, 1,-77.777,+0, 2.479310e+02, 1.5,_
 3712, 100, 251, 0, 4.5,   -1, 2.489250e+02, 0,_
 3713, 100, 252, 0, 0,     +1, 2.499170e+02, 0,_
 3714, 100, 253, 0, 0.5,   +1, 2.509110e+02, 0,_
 3715, 100, 254, 0, 0,     +1, 2.519040e+02, 0,_
 3716, 100, 255, 0, 3.5,   +1, 2.528990e+02, 0,_
 3717, 100, 256, 0, 0,     +1, 2.538920e+02, 0,_
 3718, 100, 257, 0, 4.5,   +1, 2.548870e+02, 0,_
 3719, 100, 258, 0, 0,     +1, 2.558800e+02, 0,_
 3720, 100, 259, 0, 1.5,   +1, 2.568750e+02, 0,_
 3720, 100, 260, 0,-77.777,+0, 260,          0,_
 3720, 100, 261, 0,-77.777,+0, 261,          0,_
 3721, 101, 245, 0, 0.5,   -1, 2.429750e+02, 0,_
 3722, 101, 245, 1, 3.5,   +1, 2.429750e+02, 0.1,_
 3723, 101, 246, 0,-77.777,+0, 2.439680e+02, 0,_
 3724, 101, 247, 0, 0.5,   -1, 2.449590e+02, 0,_
 3725, 101, 247, 1, 3.5,   +1, 2.449590e+02, 0.05,_
 3726, 101, 248, 0,-77.777,+0, 2.459520e+02, 0,_
 3727, 101, 249, 0, 3.5,   -1, 2.469430e+02, 0,_
 3728, 101, 249, 1, 0.5,   -1, 2.469430e+02, 0.1,_
 3729, 101, 250, 0,-77.777,+0, 2.479360e+02, 0,_
 3730, 101, 251, 0, 3.5,   -1, 2.489280e+02, 0,_
 3731, 101, 252, 0,-77.777,+0, 2.499210e+02, 0,_
 3732, 101, 253, 0, 3.5,   -1, 2.509130e+02, 0,_
 3733, 101, 254, 0, 0,     -1, 2.519070e+02, 0,_
 3734, 101, 254, 1, 3,     -1, 2.519070e+02, 0.05,_
 3735, 101, 255, 0, 3.5,   -1, 2.529000e+02, 0,_
 3736, 101, 256, 0, 1,     -1, 2.538940e+02, 0,_
 3737, 101, 257, 0, 3.5,   -1, 2.548870e+02, 0,_
 3738, 101, 258, 0, 8,     -1, 2.558810e+02, 0,_
 3739, 101, 258, 1, 1,     -1, 2.558810e+02, 0,_
 3740, 101, 259, 0, 3.5,   -1, 2.568750e+02, 0,_
 3741, 101, 260, 0,-77.777,+0, 2.578690e+02, 0,_
 3742, 101, 261, 0, 3.5,   -1, 2.588630e+02, 0,_
 3742, 101, 262, 0,-77.777,+0, 262,          0,_
 3742, 101, 263, 0,-77.777,+0, 223,          0,_
 3743, 102, 249, 0, 2.5,   +1, 2.469480e+02, 0,_
 3744, 102, 250, 0, 0,     +1, 2.479390e+02, 0,_
 3745, 102, 251, 0, 3.5,   +1, 2.489320e+02, 0,_
 3746, 102, 251, 1, 4.5,   -1, 2.489320e+02, 0.11,_
 3747, 102, 252, 0, 0,     +1, 2.499230e+02, 0,_
 3748, 102, 253, 0, 4.5,   -1, 2.509170e+02, 0,_
 3749, 102, 254, 0, 0,     +1, 2.519080e+02, 0,_
 3750, 102, 254, 1,-77.777,+0, 2.519080e+02, 0.5,_
 3751, 102, 255, 0, 0.5,   +1, 2.529020e+02, 0,_
 3752, 102, 256, 0, 0,     +1, 2.538940e+02, 0,_
 3753, 102, 257, 0, 3.5,   +1, 2.548880e+02, 0,_
 3754, 102, 258, 0, 0,     +1, 2.558810e+02, 0,_
 3755, 102, 259, 0, 4.5,   +1, 2.568750e+02, 0,_
 3756, 102, 260, 0, 0,     +1, 2.578680e+02, 0,_
 3757, 102, 261, 0, 1.5,   +1, 2.588630e+02, 0,_
 3758, 102, 262, 0, 0,     +1, 2.598560e+02, 0,_
 3758, 102, 263, 0,-77.777,+0, 263,          0,_
 3759, 102, 264, 0,-77.777,+0, 264,          0,_
 3759, 103, 251, 0,-77.777,+0, 2.489370e+02, 0,_
 3760, 103, 252, 0,-77.777,+0, 2.499300e+02, 0,_
 3761, 103, 253, 0, 3.5,   -1, 2.509210e+02, 0,_
 3762, 103, 253, 1, 0.5,   -1, 2.509210e+02, 0.03,_
 3763, 103, 254, 0,-77.777,+0, 2.519140e+02, 0,_
 3764, 103, 255, 0, 3.5,   -1, 2.529050e+02, 0,_
 3765, 103, 256, 0,-77.777,+0, 2.538990e+02, 0,_
 3766, 103, 257, 0, 4.5,   +1, 2.548910e+02, 0,_
 3767, 103, 258, 0,-77.777,+0, 2.558850e+02, 0,_
 3768, 103, 259, 0, 4.5,   +1, 2.568770e+02, 0,_
 3769, 103, 260, 0,-77.777,+0, 2.578710e+02, 0,_
 3770, 103, 261, 0,-77.777,+0, 2.588640e+02, 0,_
 3771, 103, 262, 0,-77.777,+0, 2.598580e+02, 0,_
 3772, 103, 263, 0,-77.777,+0, 2.608510e+02, 0,_
 3772, 103, 264, 0,-77.777,+0, 264,          0,_
 3772, 103, 265, 0,-77.777,+0, 265,          0,_
 3773, 104, 253, 0, 3.5,   +1, 2.509260e+02, 0,_
 3774, 104, 254, 0, 0,     +1, 2.519170e+02, 0,_
 3775, 104, 255, 0, 4.5,   -1, 2.529100e+02, 0,_
 3776, 104, 255, 1, 2.5,   +1, 2.529100e+02, 0,_
 3777, 104, 256, 0, 0,     +1, 2.539010e+02, 0,_
 3778, 104, 257, 0, 0.5,   +1, 2.548940e+02, 0,_
 3779, 104, 257, 1, 5.5,   -1, 2.548940e+02, 0.114,_
 3780, 104, 258, 0, 0,     +1, 2.558860e+02, 0,_
 3781, 104, 259, 0, 3.5,   +1, 2.568800e+02, 0,_
 3782, 104, 260, 0, 0,     +1, 2.578720e+02, 0,_
 3783, 104, 261, 0, 1.5,   +1, 2.588660e+02, 0,_
 3784, 104, 261, 1, 4.5,   +1, 2.588660e+02, 0.07,_
 3785, 104, 262, 0, 0,     +1, 2.598580e+02, 0,_
 3786, 104, 262, 1,-77.777,+0, 2.598580e+02, 0.6,_
 3787, 104, 263, 0, 1.5,   +1, 2.608520e+02, 0,_
 3788, 104, 264, 0, 0,     +1, 2.618450e+02, 0,_
 3789, 104, 265, 0, 1.5,   +1, 2.628390e+02, 0,_
 3789, 104, 266, 0,-77.777,+0, 266,          0,_
 3789, 104, 267, 0,-77.777,+0, 267,          0,_
 3790, 105, 255, 0,-77.777,+0, 2.529160e+02, 0,_
 3791, 105, 256, 0,-77.777,+0, 2.539080e+02, 0,_
 3792, 105, 257, 0, 4.5,   +1, 2.548990e+02, 0,_
 3793, 105, 257, 1, 0.5,   -1, 2.548990e+02, 0.1,_
 3794, 105, 258, 0,-77.777,+0, 2.558920e+02, 0,_
 3795, 105, 258, 1,-77.777,+0, 2.558920e+02, 0.06,_
 3796, 105, 259, 0,-77.777,+0, 2.568840e+02, 0,_
 3797, 105, 260, 0,-77.777,+0, 2.578770e+02, 0,_
 3798, 105, 261, 0,-77.777,+0, 2.588690e+02, 0,_
 3799, 105, 262, 0,-77.777,+0, 2.598620e+02, 0,_
 3800, 105, 263, 0,-77.777,+0, 2.608550e+02, 0,_
 3801, 105, 264, 0,-77.777,+0, 2.618490e+02, 0,_
 3802, 105, 265, 0,-77.777,+0, 2.628410e+02, 0,_
 3802, 105, 266, 0,-77.777,+0, 266,          0,_
 3802, 105, 267, 0,-77.777,+0, 267,          0,_
 3802, 105, 268, 0,-77.777,+0, 268,          0,_
 3802, 105, 269, 0,-77.777,+0, 269,          0,_
 3803, 106, 258, 0, 0,     +1, 2.558960e+02, 0,_
 3804, 106, 259, 0, 0.5,   +1, 2.568890e+02, 0,_
 3805, 106, 260, 0, 0,     +1, 2.578800e+02, 0,_
 3806, 106, 261, 0, 3.5,   +1, 2.588730e+02, 0,_
 3807, 106, 262, 0, 0,     +1, 2.598650e+02, 0,_
 3808, 106, 263, 0, 4.5,   +1, 2.608580e+02, 0,_
 3809, 106, 263, 1, 1.5,   +1, 2.608580e+02, 0.1,_
 3810, 106, 264, 0, 0,     +1, 2.618500e+02, 0,_
 3811, 106, 265, 0, 1.5,   +1, 2.628440e+02, 0,_
 3812, 106, 266, 0, 0,     +1, 2.638360e+02, 0,_
 3813, 106, 269, 0,-77.777,+0, 2.668170e+02, 0,_
 3813, 106, 270, 0,-77.777,+0, 270,          0,_
 3813, 106, 271, 0,-77.777,+0, 270,          0,_
 3814, 107, 260, 0,-77.777,+0, 2.578870e+02, 0,_
 3815, 107, 261, 0,-77.777,+0, 2.588790e+02, 0,_
 3816, 107, 262, 0,-77.777,+0, 2.598710e+02, 0,_
 3817, 107, 262, 1,-77.777,+0, 2.598710e+02, 0.3,_
 3818, 107, 263, 0,-77.777,+0, 2.608630e+02, 0,_
 3819, 107, 264, 0,-77.777,+0, 2.618560e+02, 0,_
 3820, 107, 265, 0,-77.777,+0, 2.628480e+02, 0,_
 3821, 107, 266, 0,-77.777,+0, 2.638410e+02, 0,_
 3822, 107, 267, 0,-77.777,+0, 2.648330e+02, 0,_
 3823, 107, 269, 0,-77.777,+0, 2.668190e+02, 0,_
 3823, 107, 270, 0,-77.777,+0, 270,          0,_
 3823, 107, 271, 0,-77.777,+0, 271,          0,_
 3824, 108, 263, 0, 3.5,   +1, 2.608680e+02, 0,_
 3825, 108, 264, 0, 0,     +1, 2.618590e+02, 0,_
 3826, 108, 265, 0, 4.5,   +1, 2.628520e+02, 0,_
 3827, 108, 265, 1, 1.5,   +1, 2.628520e+02, 0.4,_
 3828, 108, 266, 0, 0,     +1, 2.638440e+02, 0,_
 3829, 108, 267, 0, 1.5,   +1, 2.648370e+02, 0,_
 3830, 108, 268, 0, 0,     +1, 2.658290e+02, 0,_
 3831, 108, 269, 0,-77.777,+0, 2.668220e+02, 0,_
 3832, 108, 273, 0, 1.5,   +1, 2.707960e+02, 0,_
 3832, 108, 274, 0,-77.777,+0, 274,          0,_
 3832, 108, 275, 0,-77.777,+0, 275,          0,_
 3833, 109, 265, 0,-77.777,+0, 2.628590e+02, 0,_
 3834, 109, 266, 0,-77.777,+0, 2.638510e+02, 0,_
 3835, 109, 266, 1,-77.777,+0, 2.638510e+02, 1.23,_
 3836, 109, 267, 0,-77.777,+0, 2.648420e+02, 0,_
 3837, 109, 268, 0,-77.777,+0, 2.658350e+02, 0,_
 3838, 109, 269, 0,-77.777,+0, 2.668270e+02, 0,_
 3839, 109, 270, 0,-77.777,+0, 2.678200e+02, 0,_
 3840, 109, 271, 0,-77.777,+0, 2.688120e+02, 0,_
 3841, 109, 273, 0,-77.777,+0, 2.707980e+02, 0,_
 3841, 109, 274, 0,-77.777,+0, 274,          0,_
 3841, 109, 275, 0,-77.777,+0, 275,          0,_
 3842, 110, 267, 0, 4.5,   +1, 2.648490e+02, 0,_
 3843, 110, 268, 0, 0,     +1, 2.658400e+02, 0,_
 3844, 110, 269, 0, 1.5,   +1, 2.668330e+02, 0,_
 3845, 110, 270, 0, 0,     +1, 2.678240e+02, 0,_
 3846, 110, 270, 1, 10,    -1, 2.678240e+02, 1.14,_
 3847, 110, 271, 0, 5.5,   -1, 2.688170e+02, 0,_
 3848, 110, 271, 1, 4.5,   +1, 2.688170e+02, 0.029,_
 3849, 110, 272, 0, 0,     +1, 2.698080e+02, 0,_
 3850, 110, 273, 0, 6.5,   -1, 2.708020e+02, 0,_
 3851, 110, 273, 1, 1.5,   +1, 2.708020e+02, 0.198,_
 3851, 110, 274, 0,-77.777,+0, 274,          0,_
 3851, 110, 275, 0,-77.777,+0, 275,          0,_
 3851, 111, 270, 0,-77.777,+0, 270,          0,_
 3851, 111, 271, 0,-77.777,+0, 271,          0,_
 3852, 111, 272, 0,-77.777,+0, 2.698160e+02, 0,_
 3852, 111, 273, 0,-77.777,+0, 273,          0,_
 3852, 111, 274, 0,-77.777,+0, 274,          0,_
 3852, 111, 275, 0,-77.777,+0, 275,          0,_
 3852, 111, 276, 0,-77.777,+0, 276,          0,_
 3852, 111, 277, 0,-77.777,+0, 277,          0,_
 3852, 111, 278, 0,-77.777,+0, 278,          0,_
 3852, 111, 279, 0,-77.777,+0, 279,          0,_
 9999
